/*------------------------------------------------------------------------
 * Name..........r8class.cpp
 * Description...Reflection Class
 * Language......C++
 * Copyright.....(c) 1998-2001 WRQ Inc.
 *------------------------------------------------------------------------
 */


#include "r8class.h"
// Array to map errors in arguments
static WORD argErr[] = {
	rcRteInvalidArg1,
	rcRteInvalidArg2,
	rcRteInvalidArg3,
	rcRteInvalidArg4,
	rcRteInvalidArg5,
	rcRteInvalidArg6,
	rcRteInvalidArg7,
	rcRteInvalidArg8,
	rcRteInvalidArg9,
	rcRteInvalidArg10
};

//----------------------------------------------------------------------
// Method:		CReflectionIBM
// Description:	Constructor, Initializes COM
//----------------------------------------------------------------------
CReflectionIBM::CReflectionIBM() : m_pDisp(NULL), m_pStream(NULL)
{
	CoInitialize(NULL);
}

//----------------------------------------------------------------------
// Method:		CReflectionIBM
// Description:	Destructor, Uninitializes COM and releases the IDispatch pointer.
//----------------------------------------------------------------------
CReflectionIBM::~CReflectionIBM()
{
	if ( m_pDisp != NULL )
		m_pDisp->Release();
	CoUninitialize();
}

//----------------------------------------------------------------------
// Method:		GetDispatch()
// Description: Returns the IDispatch pointer to Reflection that was
//              set by CreateSession().  Note: caller must call Release()
//              on the returned pointer when done with it!
//----------------------------------------------------------------------
IDispatch* CReflectionIBM::GetDispatch()
{
	if ( m_pDisp != NULL )
		m_pDisp->AddRef();

	return m_pDisp;
}

//----------------------------------------------------------------------
// Method:		InterfaceToStream()
// Description: Marshals the contained IDispatch pointer to a stream.
//              Used by the creating thread to allow its CReflectionIBM
//              object to be used on another thread.
//----------------------------------------------------------------------
BOOL CReflectionIBM::InterfaceToStream()
{
	if ( m_pDisp != NULL )
	{
		HRESULT hr = CoMarshalInterThreadInterfaceInStream(
						IID_IDispatch,	//Reference to the identifier of the interface
						m_pDisp,		//Pointer to the interface to be marshaled
						&m_pStream);	//IStream interface pointer for the marshaled interface

		if (S_OK != hr)
		{
			m_pStream = NULL;
			return FALSE;
		}
	}
	m_pDisp = NULL;
	return TRUE;
}

//----------------------------------------------------------------------
// Method:		StreamToInterface()
// Description: Unmarshals the IDispatch pointer to Reflection for use
//              by this thread.
//----------------------------------------------------------------------
BOOL CReflectionIBM::StreamToInterface()
{
	if ( m_pStream != NULL )
	{
		HRESULT hr = CoGetInterfaceAndReleaseStream(
						m_pStream,		//Pointer to the stream from which the object is to be marshaled
						IID_IDispatch,	//Reference to the identifier of the interface
						(LPVOID*) &m_pDisp);		//interface pointer requested

		if (S_OK != hr)
		{
			m_pDisp = NULL;
			return FALSE;
		}
	}
	m_pStream = NULL;
	return TRUE;
}

//----------------------------------------------------------------------
// Method:		IsBound()
// Description: First checks to see if IDispatch has been initialized and then
//				checks to make sure the session hasn't went away.
//----------------------------------------------------------------------
BOOL CReflectionIBM::IsBound()
{
	HRESULT hr;
	DISPID dispid;
	if ( !m_pDisp )
		return FALSE;
	LPOLESTR str = L"Connect";
	hr = m_pDisp->GetIDsOfNames(IID_NULL, &str, 1, LOCALE_USER_DEFAULT, &dispid);
	if ( HRESULT_CODE(hr) == RPC_S_SERVER_UNAVAILABLE )
		return FALSE;
	return TRUE;
}

//----------------------------------------------------------------------
// Method:		GetNumSessions(pdwNumWindows)
// Description: If already bound to a RIBM instance, get the Numeric property
//				Else we will Grab the first Reflection session we find and
//				retrieve the Number of Windows property.
//----------------------------------------------------------------------
WORD CReflectionIBM::GetNumSessions(
	LPDWORD pdwNumWindows)	// the total number of windows opened [out]
{
	// If The object has a IDispatch pointer, we are already bound to a session
	// and can simply get the property NumberSessions
	if ( IsBound() ) {
		return get_NumberSessions(pdwNumWindows);
	}
	// Else the object hasn't bound yet.
	IRunningObjectTable* pROT = NULL;
	IEnumMoniker* pEnumMoniker = NULL;
	IMoniker* pMoniker = NULL;
	HRESULT hr;
	WORD uError = rcRteNoError;
	
	*pdwNumWindows = 0;

	hr = GetRunningObjectTable(NULL, &pROT);
	if ( FAILED(hr) ) {
		uError = rcRteOleInitFailed;
		goto NumError;
	}

	hr = pROT->EnumRunning(&pEnumMoniker);
	if ( FAILED(hr) ) {
		uError = rcRteOleInitFailed;
		goto NumError;
	}

	// Cycle through the Monikers to find a RIBM object
	while ( pEnumMoniker->Next(1, &pMoniker, NULL) == S_OK ) {
		MKSYS mType;
		IDispatch* pDisp = NULL;

		pMoniker->IsSystemMoniker((DWORD *)&mType);
		if ( mType != MKSYS_FILEMONIKER ) {
			// The only monikers we are interested in are File Monikers
			pMoniker->Release();
			continue;
		}
		// Try to bind to the interface IID_DRIBMAPP which is Unique to RIBM
		// The IID_RIBMAPP interface is an IDispatch pointer
		hr = BindMoniker(pMoniker, 0, IID_DRIBMAPP, (void **)&pDisp);
		if ( FAILED(hr) ) {
			// Must not be a reflection object
			pMoniker->Release();
			continue;
		}
		
		// Temporarily place the pointer in the member so we can call the helper method
		m_pDisp = pDisp;
		uError = get_NumberSessions(pdwNumWindows);
		m_pDisp = NULL;
		
		pDisp->Release();
		pMoniker->Release();
		pMoniker = NULL;
		break;
	}
	if ( !*pdwNumWindows && uError == rcRteNoError )
		uError = rcRteNoSessionsActive;
NumError:
	if (pEnumMoniker)
		pEnumMoniker->Release();
	if (pROT)
		pROT->Release();
	return uError;
}

//----------------------------------------------------------------------
// Method:		GetSessionSettingsFile(UINT uIndex, LPSTR pName, LPDWORD pdwLen)
// Description: We will Grab nth Reflection session we find and retrieve
//				the String property containing the Settings File Name
//				Can Only Be called before the C++ object is bound to a session
//----------------------------------------------------------------------
WORD CReflectionIBM::GetSessionSettingsFile(UINT uIndex, LPSTR pName, LPDWORD pdwLen)
{
	IRunningObjectTable* pROT = NULL;
	IEnumMoniker* pEnumMoniker = NULL;
	IMoniker* pMoniker = NULL;
	IBindCtx *pBc = NULL;
	WORD uError = rcRteNoError;
	HRESULT hr;
	
	// Check to see if the Dispatch has already been initialized
	if ( IsBound() ) {
		return rcRteApiInvalidEvent;
	}

	// Null Terminate the String Passed in
	*pName = NULL;
	
	// A Bind Context is needed for the IMoniker::GetDisplayName call 
	hr = CreateBindCtx(0, &pBc);
	if ( FAILED(hr) ) {
		return rcRteOleInitFailed;
	}
	
	hr = GetRunningObjectTable(NULL, &pROT);
	if ( FAILED(hr) ) {
		return rcRteOleInitFailed;
		goto NumError;
	}
	
	hr = pROT->EnumRunning(&pEnumMoniker);
	if ( FAILED(hr) ) {
		return rcRteOleInitFailed;
		goto NumError;
	}
	
	UINT count = 0;
	while ( pEnumMoniker->Next(1, &pMoniker, NULL) == S_OK ) {
		MKSYS mType;
		IDispatch* pDisp = NULL;

		LPOLESTR pwDisplayName;
		char pDisplayName[255];

		pMoniker->IsSystemMoniker((DWORD *)&mType);
		if ( mType != MKSYS_FILEMONIKER ) {
			// The only monikers we are interested in are File Monikers
			pMoniker->Release();
			continue;
		}
		// Try to bind to the interface IID_DRIBMAPP which is Unique to RIBM
		// The IID_RIBMAPP interface pointer is the same as the IDispatch pointer
		hr = BindMoniker(pMoniker, 0, IID_DRIBMAPP, (void **)&pDisp);
		if ( FAILED(hr) ) {
			// Must not be a reflection object
			pMoniker->Release();
			continue;
		}
		
		// There are two Monikers in the ROT from every ReflectionIBM Instance, one's display name is
		// its settings file, the other's is its non unique OLE Name(can be changed by user).
		// If settings file and display name are not equal, the Moniker is not unique.  Skip it and
		// go on.
		pMoniker->GetDisplayName(pBc, NULL, &pwDisplayName);
		pMoniker->Release();
		
		// Convert to a multi byte string.
		WideCharToMultiByte(CP_ACP, NULL, pwDisplayName, -1, pDisplayName, 255, NULL, NULL);
		
		// Free the Wide string returned
		IMalloc * pIMalloc = NULL;
		CoGetMalloc(1, &pIMalloc);
		pIMalloc->Free(pwDisplayName);
		pIMalloc->Release();
		
		// If the Display Name has an _ for the first character, it is an internally used name.
		//		the names are not guarantied to exist in future releases.
		if ( *pDisplayName == '_' ) {
			pDisp->Release();
			continue;
		}
		
		char buffer[255];
		DWORD len = 255;
		
		// use member pointer to take advantage of the wrapped getstring call.
		m_pDisp = pDisp;
		uError = get_SettingsFile(buffer, &len);
		m_pDisp = NULL;
		
		if ( _stricmp(buffer, pDisplayName) != 0 &&
			!(len == 0 && _stricmp(pDisplayName, "Untitled") == 0) ) {
			// the display name is not the settings file name meaning we can't count this instance
			// as unique.  Unless the length of the settings file returned is 0 and the display name is Untitled
			pDisp->Release();
			continue;
		}
		
		// We have found an Unique Reflection Instance
		if ( count == uIndex ) {
			if ( len + 1 <= *pdwLen ) {
				strncpy(pName, buffer, len + 1);
				*pdwLen = len;
			}
			else {
				strncpy(pName, buffer, *pdwLen);
				uError = rcRteStringTooLong;
			}
			pDisp->Release();
			break;
		}
		else {
			pDisp->Release();
			++count;
		}
	}
	// If we didn't bail for an error and the count is greater than the index, return an error.
	if ( uError == rcRteNoError && count > uIndex ) {
		uError = rcRteIndexTooLarge;
	}

NumError:
	if (pEnumMoniker)
		pEnumMoniker->Release();
	if (pROT)
		pROT->Release();
	if (pBc)
		pBc->Release();
	return uError;
}

//----------------------------------------------------------------------
// Method:		GetErrorString(wErrorID, pErrorStirng, pdwLength)
// Description:	Get string corresponding to error code
//----------------------------------------------------------------------
WORD CReflectionIBM::GetErrorString(
	WORD wErrorId,			// error code
	LPSTR pErrorString,		// pointer to buffer to return error code string in, should be at least 128 bytes long [out]
	LPDWORD pdwLength)		// length of buffer [in], length of error string [out]
{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], (DWORD) wErrorId);

	WORD uErr = InvokeMethodHelper(OLESTR("GetErrorString"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], pErrorString, pdwLength);
	}
	
	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}


//-----------------------------------------------------------------------
// Member function for GetDisplayText for CReflectionIBM class.
// Description:	Get text from specified location on screen
// Syntax:	Row, Column, Chars
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetDisplayText(
	LONG Row,
	LONG Column,
	LPSTR Chars,
	LPDWORD lpdwArg3)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[2], Row);
	CopyToVariant(&pvArgs[1], Column);
	CopyToVariant(&pvArgs[0], lpdwArg3);

	WORD uErr = InvokeMethodHelper(OLESTR("GetDisplayText"), pvArgs, 3, &pvArgs[3]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[3], Chars, lpdwArg3);
	}

	return uErr;
}



//----------------------------------------------------------------------
// Method:		Create(lpLanchArgs);
// Description:	Creates a RIBM instance with the specified command line args.
// Notes:		Does not return until instance is ready for commands.  As with
//				all OLE objects, the session is invisible after creation.  Change
//				the Visible property to rcYes to make the session visible.
//				If lpLaunchArgs == 0, then untitled RIBM instance is launched,  But if
//				lpLaunchArgs contains an invalid settings file, the RIBM instance is
//				closed and an error message returned.
//----------------------------------------------------------------------
WORD CReflectionIBM::CreateSession(
	LPSTR lpLaunchArgs)
{
	if ( IsBound() )
		return rcRteApiInvalidEvent;

	// Do equivalent of CreateObject
	WORD uError = rcRteNoError;
    LPDISPATCH pdisp = NULL;       // IDispatch of automation object
    LPOLESTR pszProgID = OLESTR("ReflectionIBM.Application");
	CLSID clsid;                   // CLSID of automation object
    HRESULT hr;
    LPUNKNOWN punk = NULL;         // IUnknown of automation object
    
	// Retrieve CLSID from the ProgID that the user specified
    hr = CLSIDFromProgID(pszProgID, &clsid);
    if (FAILED(hr)) {
		uError = rcRteOleInitFailed;
        goto launchError;
	}
    
    // Create an instance of the automation object and ask for the IDispatch interface
    hr = CoCreateInstance(clsid, NULL, CLSCTX_SERVER, 
                          IID_IUnknown, (void **)&punk);
    if (FAILED(hr)) {
		uError = rcRteCantStartNewSession;
        goto launchError;
	}

    hr = punk->QueryInterface(IID_IDispatch, (void **)&pdisp);
    if (FAILED(hr)) {
		uError = rcRteCantStartNewSession;
        goto launchError;
	}

	// Switch the dispatch pointer to the member variable
    m_pDisp = pdisp;
	pdisp = NULL;
    
	punk->Release();
	punk = NULL;

	// Check args, if there are args try to open the settings file.
	if (lpLaunchArgs && *lpLaunchArgs) {
		// strip leading spaces
		while (*lpLaunchArgs == ' ') {
			lpLaunchArgs++;
		}
		uError = this->OpenSettings(rcSettings, lpLaunchArgs);
		if ( uError != rcRteNoError ) {
			m_pDisp->Release();
			m_pDisp = NULL;
		}
	}
	return uError;
launchError:
    if (punk) punk->Release();
    if (pdisp) pdisp->Release();
	return uError;
}

//----------------------------------------------------------------------
// Method:		GetSession(pName)
// Description:	Binds to a running Reflection session of the given Name.
// Notes:		The name of a config file is passed in. Active instances of
//				Reflection are scanned and the first instance found with a
//				setup file of that name is set to bound to the reflection Class.
//				If pName is NULL, the first session found is bound to.
//				The settings file passed in needs to include the full path.
//				
//				The OLE Name can also be passed in.  The first session with
//				the supplied OLE Name will be found.
//----------------------------------------------------------------------
WORD CReflectionIBM::GetSession(LPSTR pName)
{
	IRunningObjectTable* pROT = NULL;
	IEnumMoniker* pEnumMoniker = NULL;
	IMoniker* pMoniker = NULL;
	IBindCtx *pBc = NULL;
	WORD uError = rcRteNoError;
	HRESULT hr;
	
	if ( IsBound() )
		return rcRteApiInvalidEvent;
	
	// A Bind Context is needed for the IMoniker::GetDisplayName call 
	hr = CreateBindCtx(0, &pBc);
	if ( FAILED(hr) ) {
		return rcRteOleInitFailed;
	}
	hr = GetRunningObjectTable(NULL, &pROT);
	if ( FAILED(hr) ) {
		return rcRteOleInitFailed;
		goto NumError;
	}
	hr = pROT->EnumRunning(&pEnumMoniker);
	if ( FAILED(hr) ) {
		return rcRteOleInitFailed;
		goto NumError;
	}
	while ( pEnumMoniker->Next(1, &pMoniker, NULL) == S_OK ) {
		MKSYS mType;
		IDispatch* pDisp = NULL;

		LPOLESTR pwDisplayName;
		char pDisplayName[255];

		pMoniker->IsSystemMoniker((DWORD *)&mType);
		if (mType != MKSYS_FILEMONIKER ) {
			// The only monikers we are interested in are File Monikers
			pMoniker->Release();
			continue;
		}
		// Try to bind to the interface IID_DRIBMAPP which is Unique to RIBM
		// The IID_RIBMAPP interface pointer is the same as the IDispatch pointer
		hr = BindMoniker(pMoniker, 0, IID_DRIBMAPP, (void **)&pDisp);
		if ( FAILED(hr) ) {
			// Must not be a reflection object
			pMoniker->Release();
			continue;
		}
		
		// If the name passed in is null we can stop now because we found a Reflection Session
		if ( !pName || !*pName ) {
			m_pDisp = pDisp;
			break;
		}

		// There are two Monikers in the ROT from every ReflectionIBM session, one's display name is
		// its settings file, the other's is its non unique Ole Name(can be changed by user).
		pMoniker->GetDisplayName(pBc, NULL, &pwDisplayName);
		pMoniker->Release();
		
		// Convert to a multi byte string.
		WideCharToMultiByte(CP_ACP, NULL, pwDisplayName, -1, pDisplayName, 255, NULL, NULL);
		
		// Free the Wide string that was returned
		IMalloc * pIMalloc = NULL;
		CoGetMalloc(1, &pIMalloc);
		pIMalloc->Free(pwDisplayName);
		pIMalloc->Release();
		
		// If the Display Name has an '_' for the first character, it is an internally used name.
		//		the names are not guarantied to exist in future releases.
		if ( *pDisplayName == '_' ) {
			pDisp->Release();
			continue;
		}
		
		char buffer[255];
		DWORD len = 255;
		
		// use member IDispatch pointer to take advantage of the wrapped getstring call.
		m_pDisp = pDisp;
		uError = get_SettingsFile(buffer, &len);
		m_pDisp = NULL;
		
		// This will compare the name passed in with both the actual settings file name and
		// the moniker display Name.  A match to either will bind the object to the session.
		if ( _stricmp(pName, pDisplayName) == 0 ||
			 _stricmp(pName, buffer) == 0 ) {
			// We found a session we want.
			m_pDisp = pDisp;
			// Don't Release pDisp, because we are now bound to it.
			break;
		}
		else {
			pDisp->Release();
		}
	}

	// If we didn't bail for an error and the m_pDisp is still null, return an error.
	if ( uError == rcRteNoError && !m_pDisp ) {
		uError = rcRteNoSessionFound;
	}

NumError:
	if (pEnumMoniker)
		pEnumMoniker->Release();
	if (pROT)
		pROT->Release();
	if (pBc)
		pBc->Release();
	return uError;
}

//----------------------------------------------------------------------
// Method:		CopyToVariant(VARIANTARG *pvArg, LPSTR pInputStr)
// Description:	Allocates a BSTR and copies the passed LPSTR into the BSTR
//				and then places the BSTR in the passed in Variant
// Notes:		Strings are never passed as [out] arguments, they
//				are always returned in the return value.
//				So the only strings we have to worry about being passed in
//				are constant strings.
//----------------------------------------------------------------------
void CReflectionIBM::CopyToVariant(VARIANTARG *pvArg, LPSTR pInputStr)
{
	UINT uStringLen = strlen(pInputStr);
	LPOLESTR olestr = NULL;
	BSTR pBstr = NULL;
	if ( uStringLen ) {
		olestr = new OLECHAR[uStringLen + 1];
		mbstowcs(olestr, pInputStr, uStringLen + 1);
		pBstr = SysAllocString(olestr);
		delete[] olestr;
	}
    V_VT(pvArg) = VT_BSTR;
    V_BSTR(pvArg) = pBstr;
}

//----------------------------------------------------------------------
// Method:		CleanUpStringVariant(VARIANTARG *pvArg)
// Description:	Deallocates the BSTR in the Variant that was passed to Invoke
//----------------------------------------------------------------------
void CReflectionIBM::CleanUpStringVariant(VARIANTARG *pvArg)
{
	if ( pvArg->vt == VT_BSTR ) {
		SysFreeString(V_BSTR(pvArg));
	}
}

//----------------------------------------------------------------------
// Method:		CopyToVariant(VARIANTARG *pvArg, LONG lArg)
// Description:	Copies the long value into the Variant
// Notes:		Signed and Unsigned are passed the same
//----------------------------------------------------------------------
void CReflectionIBM::CopyToVariant(VARIANTARG *pvArg, LONG lArg)
{
	pvArg->vt = VT_I4;
	V_I4(pvArg) = lArg;
}

//----------------------------------------------------------------------
// Method:		CopyToVariant(VARIANTARG *pvArg, DWORD dwArg)
// Description:	Copies the DWORD (unsigned long) value into the Variant
// Notes:		Signed and Unsigned are passed the same
//----------------------------------------------------------------------
void CReflectionIBM::CopyToVariant(VARIANTARG *pvArg, DWORD dwArg)
{
	// Signed and Unsigned are treated the same
	pvArg->vt = VT_I4;
	V_I4(pvArg) = dwArg;
}

//----------------------------------------------------------------------
// Method:		CopyToVariant(VARIANTARG *pvArg, LPDWORD pdwArg)
// Description:	Copies the value of the DWORD pointed to into the Variant
// Notes:		The Reflection object doesn't have any [out] arguments,
//				Even though the argument was passed to this stub by
//				reference it will be passed to Invoke by value.
//				The return value will then determine if the referenced
//				memory is to be updated.
//----------------------------------------------------------------------
void CReflectionIBM::CopyToVariant(VARIANTARG *pvArg, LPDWORD pdwArg)
{
	pvArg->vt = VT_I4;
	V_I4(pvArg) = *pdwArg;
}

//----------------------------------------------------------------------
// Method:		InitVariants(VARIANTARG *pvArgs, UINT uNum)
// Description:	Initializes the variant array passed in
//----------------------------------------------------------------------
void CReflectionIBM::InitVariants(VARIANTARG *pvArgs, UINT uNum)
{
	// Initialize the Variants
	for ( UINT i = 0; i < uNum; ++i ) {
		VariantInit(&pvArgs[i]);
	}
}

//----------------------------------------------------------------------
// Method:		GetNumericReturn(VARIANTARG *pvRet, LPDWORD pdwRet)
// Description:	Copies the DWORD in the Variant to the address passed in
//				If the return is equal to rcVbTrue it is changed to rcYes
//----------------------------------------------------------------------
void CReflectionIBM::GetNumericReturn(VARIANTARG *pvRet, LPDWORD pdwRet)
{
	if ( pvRet->vt == VT_I4  ) {
		*pdwRet = V_I4(pvRet);
		// The Ole Object returns VbTrue (-1) through the invoke.  To make this
		// work better in C++ we will change the value to rcYes (1).
		if ( *pdwRet == rcVbTrue )
			*pdwRet = rcYes;
	}
}

//----------------------------------------------------------------------
// Method:		GetStringReturn(VARIANTARG *pvRet, LPSTR pszArg, LPDWORD pdwLen)
// Description:	Copies the BSTR in the Variant to the LPSTR passed in and the length
//				of the string returned to the address of the DWORD.  The DWORD initially
//				contains the size of the LPSTR buffer.
//----------------------------------------------------------------------
void CReflectionIBM::GetStringReturn(VARIANTARG *pvRet, LPSTR pszArg, LPDWORD pdwLen)
{
	if ( pvRet->vt == VT_BSTR ) {
		// copy returned argument string to buffer
		UINT uLen = SysStringLen(V_BSTR(pvRet));
		WideCharToMultiByte(CP_ACP, NULL, V_BSTR(pvRet), uLen + 1, pszArg, *pdwLen, NULL, NULL);
		
		if (*pdwLen > uLen)
			*pdwLen = uLen;
		// else the string was truncated at the max length.
		SysFreeString(V_BSTR(pvRet));
	}
}

///////////////////////////////////////////////////////////////////////////////
//	InvokeMethodHelper: Helper method to pack arguments for IDispatch	
//	Parameters:
//		LPOLESTR pbszCmdName,	// Wide string containing name of Command.
//		VARIANTARG *pvArgs,		// Array of input VARIANTS
//		UINT uNumArgs,			// Number of Variants Inputted
//		VARIANTARG *pvRet		// Pointer to return Variant
///////////////////////////////////////////////////////////////////////////////
WORD CReflectionIBM::InvokeMethodHelper(
		LPOLESTR pwszCmdName,	// Wide string containing name of Command.
		VARIANTARG *pvArgs,		// Array of input VARIANTS
		UINT uNumArgs,			// Number of Variants Inputted
		VARIANTARG *pvRet)		// Pointer to return Variant
{
	HRESULT hr;
	WORD uError = rcRteNoError;
	DISPID dispid;

	// if the IDispatch pointer has not been initialized return an error.
	if ( !IsBound() ) {  
		return rcRteNoStartup;
	}

	// First we need to look up the command name and get the Dispatch ID
	// Get DISPID of property/method
	hr = m_pDisp->GetIDsOfNames(IID_NULL, &pwszCmdName, 1, LOCALE_USER_DEFAULT, &dispid);

	if ( FAILED(hr) ) {
		return rcRteUndefinedError;
	}

	// Build and submit command
	DISPPARAMS dispParams;
    memset(&dispParams, 0, sizeof dispParams);

	dispParams.cArgs = uNumArgs;
	dispParams.rgvarg = pvArgs;
	dispParams.cNamedArgs = 0;
	dispParams.rgdispidNamedArgs = NULL;

	EXCEPINFO excepInfo;
	excepInfo.wCode = rcRteNoError;

	hr = m_pDisp->Invoke(dispid, IID_NULL, LOCALE_USER_DEFAULT,
				  DISPATCH_METHOD, &dispParams, pvRet, &excepInfo, NULL);

	if ( excepInfo.wCode != rcRteNoError ) {
		uError = excepInfo.wCode;
	}
	else if ( FAILED(hr) ) {
		uError = rcRteUndefinedError;
	}
	
	return uError;
}

///////////////////////////////////////////////////////////////////////////////
//	InvokePutHelper: Helper method to pack arguments for IDispatch	
//	Parameters:
//		LPOLESTR pbszPrpName,	// Wide string containing name of Property.
//		VARIANTARG *pvArgs,		// Pointer to input Variant
///////////////////////////////////////////////////////////////////////////////
WORD CReflectionIBM::InvokePutHelper(
		LPOLESTR pwszPrpName,	// Wide string containing name of Property.
		VARIANTARG *pvArgs)		// Pointer to input Variant
{
	HRESULT hr;
	WORD uError = rcRteNoError;
	DISPID dispid;
	// if the IDispatch pointer has not been initialized return an error.
	if ( !IsBound() ) {  
		return rcRteNoStartup;
	}

	// First we need to look up the command name and get the Dispatch ID
	// Get DISPID of property/method
	hr = m_pDisp->GetIDsOfNames(IID_NULL, &pwszPrpName, 1, LOCALE_USER_DEFAULT, &dispid);

	if ( FAILED(hr) ) {
		return rcRteUndefinedError;
	}

	// Build and submit command
	DISPPARAMS dispParams;
    memset(&dispParams, 0, sizeof dispParams);

	dispParams.cArgs = 1;
	dispParams.rgvarg = pvArgs;
	
	// To signal that this is a put, we need this
	DISPID dispidNamed = DISPID_PROPERTYPUT;
	dispParams.cNamedArgs = 1;
	dispParams.rgdispidNamedArgs = &dispidNamed;

	EXCEPINFO excepInfo;
	excepInfo.wCode = rcRteNoError;

	hr = m_pDisp->Invoke(dispid, IID_NULL, LOCALE_USER_DEFAULT,
				  DISPATCH_PROPERTYPUT, &dispParams, NULL, &excepInfo, NULL);
	
	if ( excepInfo.wCode != rcRteNoError ) {
		uError = excepInfo.wCode;
	}
	else if ( FAILED(hr) ) {
		uError = rcRteUndefinedError;
	}
	
	return uError;
}

///////////////////////////////////////////////////////////////////////////////
//	InvokeGetHelper: Helper method to pack arguments for IDispatch	
//	Parameters:
//		LPOLESTR pbszPrpName,	// Wide string containing name of Property.
//		VARIANTARG *pvRet		// Pointer to return Variant
///////////////////////////////////////////////////////////////////////////////
WORD CReflectionIBM::InvokeGetHelper(
		LPOLESTR pwszPrpName,	// Wide string containing name of Command.
		VARIANTARG *pvRet)		// Pointer to return Variant
{
	HRESULT hr;
	WORD uError = rcRteNoError;
	DISPID dispid;

	// if the IDispatch pointer has not been initialized return an error.
	if ( !IsBound() ) {  
		return rcRteNoStartup;
	}

	// First we need to look up the command name and get the Dispatch ID
	// Get DISPID of property/method
	hr = m_pDisp->GetIDsOfNames(IID_NULL, &pwszPrpName, 1, LOCALE_USER_DEFAULT, &dispid);

	if ( FAILED(hr) ) {
		return rcRteUndefinedError;
	}

	// Build and submit command
	DISPPARAMS dispParams;
    memset(&dispParams, 0, sizeof dispParams);

	dispParams.cArgs = 0;
	dispParams.rgvarg = NULL;
	dispParams.cNamedArgs = 0;
	dispParams.rgdispidNamedArgs = NULL;

	EXCEPINFO excepInfo;
	excepInfo.wCode = rcRteNoError;

	hr = m_pDisp->Invoke(dispid, IID_NULL, LOCALE_USER_DEFAULT,
				  DISPATCH_PROPERTYGET, &dispParams, pvRet, &excepInfo, NULL);

	if ( excepInfo.wCode != rcRteNoError ) {
		uError = excepInfo.wCode;
	}
	else if ( FAILED(hr) ) {
		uError = rcRteUndefinedError;
	}
	
	return uError;
}

//***********************************************************************
//***** Standard Method Implementations *********************************
//***********************************************************************
//-----------------------------------------------------------------------
// Member function for AbortTransfer for CReflectionIBM class.
// Description:	Abort the current IND$FILE transfer
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::AbortTransfer(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("AbortTransfer"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for AboutReflectionDlg for CReflectionIBM class.
// Description:	Dialog about Reflection
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::AboutReflectionDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("AboutReflectionDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ActivateCommandLine for CReflectionIBM class.
// Description:	Enable the command line
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ActivateCommandLine(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ActivateCommandLine"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for AddButton for CReflectionIBM class.
// Description:	Add a button to the toolbar
// Syntax:	nIndex
//-----------------------------------------------------------------------
WORD CReflectionIBM::AddButton(
	DWORD nIndex)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nIndex);

	WORD uErr = InvokeMethodHelper(OLESTR("AddButton"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for AddHotListItem for CReflectionIBM class.
// Description:	Add an item to a hotlist
// Syntax:	sName, sSelection, sCommands, nOption
//-----------------------------------------------------------------------
WORD CReflectionIBM::AddHotListItem(
	LPSTR sName,
	LPSTR sSelection,
	LPSTR sCommands,
	DWORD nOption)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[3], sName);
	CopyToVariant(&pvArgs[2], sSelection);
	CopyToVariant(&pvArgs[1], sCommands);
	CopyToVariant(&pvArgs[0], nOption);

	WORD uErr = InvokeMethodHelper(OLESTR("AddHotListItem"), pvArgs, 4, NULL);

	CleanUpStringVariant(&pvArgs[3]);
	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for AddReferenceDlg for CReflectionIBM class.
// Description:	Dialog to reference a VBA project
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::AddReferenceDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("AddReferenceDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for AppendSelection for CReflectionIBM class.
// Description:	Append the current selection
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::AppendSelection(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("AppendSelection"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for AppendSQLStatement for CReflectionIBM class.
// Description:	Append a portion of an SQL statement to an existing statement
// Syntax:	nType, sStatement
//-----------------------------------------------------------------------
WORD CReflectionIBM::AppendSQLStatement(
	DWORD nType,
	LPSTR sStatement)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nType);
	CopyToVariant(&pvArgs[0], sStatement);

	WORD uErr = InvokeMethodHelper(OLESTR("AppendSQLStatement"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for CancelSelection for CReflectionIBM class.
// Description:	Cancel the current selection
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::CancelSelection(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("CancelSelection"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for CancelTransfer for CReflectionIBM class.
// Description:	Cancel the current IND$FILE transfer
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::CancelTransfer(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("CancelTransfer"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ClearHotlists for CReflectionIBM class.
// Description:	Remove all hotlists
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ClearHotlists(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ClearHotlists"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ClearHotspots for CReflectionIBM class.
// Description:	Remove all hotspots
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ClearHotspots(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ClearHotspots"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ClearIBMPrinterFonts for CReflectionIBM class.
// Description:	Remove all IBM printer font mappings
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ClearIBMPrinterFonts(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ClearIBMPrinterFonts"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ClearMenu for CReflectionIBM class.
// Description:	Clear all popups and items from the main menu
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ClearMenu(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ClearMenu"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ClearNWSAA3270Config for CReflectionIBM class.
// Description:	Remove all NetWare SAA for 3270 configurations
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ClearNWSAA3270Config(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ClearNWSAA3270Config"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ClearSelection for CReflectionIBM class.
// Description:	Clear the current selection without putting it on the clipboard
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ClearSelection(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ClearSelection"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ClosePrinter for CReflectionIBM class.
// Description:	Close and flush screen printing
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ClosePrinter(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ClosePrinter"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for Connect for CReflectionIBM class.
// Description:	Connect to host
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::Connect(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("Connect"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for CopyPasteSettingsDlg for CReflectionIBM class.
// Description:	Copy Paste Settings Dialog
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::CopyPasteSettingsDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("CopyPasteSettingsDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for CopySelection for CReflectionIBM class.
// Description:	Copy the current selection with the selected format
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::CopySelection(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("CopySelection"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for CopyTable for CReflectionIBM class.
// Description:	Copy the current selection and preserve tabs
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::CopyTable(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("CopyTable"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for CutSelection for CReflectionIBM class.
// Description:	Cut the current selection
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::CutSelection(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("CutSelection"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DDESetupDlg for CReflectionIBM class.
// Description:	Dialog to change DDE settings
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::DDESetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("DDESetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DefineButton for CReflectionIBM class.
// Description:	Define a button on the toolbar
// Syntax:	nIndex, nSeparator, sPicture, sText, sHelp, sCmds
//-----------------------------------------------------------------------
WORD CReflectionIBM::DefineButton(
	DWORD nIndex,
	DWORD nSeparator,
	LPSTR sPicture,
	LPSTR sText,
	LPSTR sHelp,
	LPSTR sCmds)

{
	VARIANTARG pvArgs[6];
	InitVariants(pvArgs, 6);

	CopyToVariant(&pvArgs[5], nIndex);
	CopyToVariant(&pvArgs[4], nSeparator);
	CopyToVariant(&pvArgs[3], sPicture);
	CopyToVariant(&pvArgs[2], sText);
	CopyToVariant(&pvArgs[1], sHelp);
	CopyToVariant(&pvArgs[0], sCmds);

	WORD uErr = InvokeMethodHelper(OLESTR("DefineButton"), pvArgs, 6, NULL);

	CleanUpStringVariant(&pvArgs[3]);
	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DefineHotspot for CReflectionIBM class.
// Description:	Map a display string to Reflection commands
// Syntax:	sString, nFlags, sCommands
//-----------------------------------------------------------------------
WORD CReflectionIBM::DefineHotspot(
	LPSTR sString,
	DWORD nFlags,
	LPSTR sCommands)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], sString);
	CopyToVariant(&pvArgs[1], nFlags);
	CopyToVariant(&pvArgs[0], sCommands);

	WORD uErr = InvokeMethodHelper(OLESTR("DefineHotspot"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DefineHotspotEx for CReflectionIBM class.
// Description:	Map a display string to Reflection commands
// Syntax:	sString, nFlags, nRow, nColumn, sTip, sHelp, sCommands
//-----------------------------------------------------------------------
WORD CReflectionIBM::DefineHotspotEx(
	LPSTR sString,
	DWORD nFlags,
	DWORD nRow,
	DWORD nColumn,
	LPSTR sTip,
	LPSTR sHelp,
	LPSTR sCommands)

{
	VARIANTARG pvArgs[7];
	InitVariants(pvArgs, 7);

	CopyToVariant(&pvArgs[6], sString);
	CopyToVariant(&pvArgs[5], nFlags);
	CopyToVariant(&pvArgs[4], nRow);
	CopyToVariant(&pvArgs[3], nColumn);
	CopyToVariant(&pvArgs[2], sTip);
	CopyToVariant(&pvArgs[1], sHelp);
	CopyToVariant(&pvArgs[0], sCommands);

	WORD uErr = InvokeMethodHelper(OLESTR("DefineHotspotEx"), pvArgs, 7, NULL);

	CleanUpStringVariant(&pvArgs[6]);
	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DefineToolbarButton for CReflectionIBM class.
// Description:	Define a button on a toolbar
// Syntax:	sToolbarName, nIndex, nSeparator, sSmallPictureFile, uSmallPicIndex, sLargePictureFile, sLargePicIndex, sLabel, sTip, sHelp, sCommands
//-----------------------------------------------------------------------
WORD CReflectionIBM::DefineToolbarButton(
	LPSTR sToolbarName,
	DWORD nIndex,
	DWORD nSeparator,
	LPSTR sSmallPictureFile,
	DWORD uSmallPicIndex,
	LPSTR sLargePictureFile,
	DWORD sLargePicIndex,
	LPSTR sLabel,
	LPSTR sTip,
	LPSTR sHelp,
	LPSTR sCommands)

{
	VARIANTARG pvArgs[11];
	InitVariants(pvArgs, 11);

	CopyToVariant(&pvArgs[10], sToolbarName);
	CopyToVariant(&pvArgs[9], nIndex);
	CopyToVariant(&pvArgs[8], nSeparator);
	CopyToVariant(&pvArgs[7], sSmallPictureFile);
	CopyToVariant(&pvArgs[6], uSmallPicIndex);
	CopyToVariant(&pvArgs[5], sLargePictureFile);
	CopyToVariant(&pvArgs[4], sLargePicIndex);
	CopyToVariant(&pvArgs[3], sLabel);
	CopyToVariant(&pvArgs[2], sTip);
	CopyToVariant(&pvArgs[1], sHelp);
	CopyToVariant(&pvArgs[0], sCommands);

	WORD uErr = InvokeMethodHelper(OLESTR("DefineToolbarButton"), pvArgs, 11, NULL);

	CleanUpStringVariant(&pvArgs[10]);
	CleanUpStringVariant(&pvArgs[7]);
	CleanUpStringVariant(&pvArgs[5]);
	CleanUpStringVariant(&pvArgs[3]);
	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DefineTransferTemplate for CReflectionIBM class.
// Description:	Define a file transfer filename template
// Syntax:	nHostSystem, sPCFilename, sHostFilename, nTransferType, nExistsAction, nRecfm, nAsciiCRLF, nBinaryCRLF, nJISCIICRLF, nLRECL, sAddParm, nBlockSize, nAllocUnits, nAvgBlkSize, nPrimAlloc, nSecAlloc
//-----------------------------------------------------------------------
WORD CReflectionIBM::DefineTransferTemplate(
	DWORD nHostSystem,
	LPSTR sPCFilename,
	LPSTR sHostFilename,
	DWORD nTransferType,
	DWORD nExistsAction,
	DWORD nRecfm,
	DWORD nAsciiCRLF,
	DWORD nBinaryCRLF,
	DWORD nJISCIICRLF,
	DWORD nLRECL,
	LPSTR sAddParm,
	DWORD nBlockSize,
	DWORD nAllocUnits,
	DWORD nAvgBlkSize,
	DWORD nPrimAlloc,
	DWORD nSecAlloc)

{
	VARIANTARG pvArgs[16];
	InitVariants(pvArgs, 16);

	CopyToVariant(&pvArgs[15], nHostSystem);
	CopyToVariant(&pvArgs[14], sPCFilename);
	CopyToVariant(&pvArgs[13], sHostFilename);
	CopyToVariant(&pvArgs[12], nTransferType);
	CopyToVariant(&pvArgs[11], nExistsAction);
	CopyToVariant(&pvArgs[10], nRecfm);
	CopyToVariant(&pvArgs[9], nAsciiCRLF);
	CopyToVariant(&pvArgs[8], nBinaryCRLF);
	CopyToVariant(&pvArgs[7], nJISCIICRLF);
	CopyToVariant(&pvArgs[6], nLRECL);
	CopyToVariant(&pvArgs[5], sAddParm);
	CopyToVariant(&pvArgs[4], nBlockSize);
	CopyToVariant(&pvArgs[3], nAllocUnits);
	CopyToVariant(&pvArgs[2], nAvgBlkSize);
	CopyToVariant(&pvArgs[1], nPrimAlloc);
	CopyToVariant(&pvArgs[0], nSecAlloc);

	WORD uErr = InvokeMethodHelper(OLESTR("DefineTransferTemplate"), pvArgs, 16, NULL);

	CleanUpStringVariant(&pvArgs[14]);
	CleanUpStringVariant(&pvArgs[13]);
	CleanUpStringVariant(&pvArgs[5]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DeletePopupItem for CReflectionIBM class.
// Description:	Delete a menu item from the menu
// Syntax:	nPopup, nItem
//-----------------------------------------------------------------------
WORD CReflectionIBM::DeletePopupItem(
	DWORD nPopup,
	DWORD nItem)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nPopup);
	CopyToVariant(&pvArgs[0], nItem);

	WORD uErr = InvokeMethodHelper(OLESTR("DeletePopupItem"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for Disconnect for CReflectionIBM class.
// Description:	Disconnect from host
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::Disconnect(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("Disconnect"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DisplayFontsDlg for CReflectionIBM class.
// Description:	Dialog to set up fonts
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::DisplayFontsDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("DisplayFontsDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DisplayOptionsDlg for CReflectionIBM class.
// Description:	Dialog to set up display options
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::DisplayOptionsDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("DisplayOptionsDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DisplaySetupDlg for CReflectionIBM class.
// Description:	Dialog to set up display colors and fonts
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::DisplaySetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("DisplaySetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for EditMacro for CReflectionIBM class.
// Description:	Open the Visual Basic editor and edit the (optional) given macro
// Syntax:	sMacro
//-----------------------------------------------------------------------
WORD CReflectionIBM::EditMacro(
	LPSTR sMacro)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sMacro);

	WORD uErr = InvokeMethodHelper(OLESTR("EditMacro"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for EditScript for CReflectionIBM class.
// Description:	Edit a Reflection Basic script
// Syntax:	sScriptFile
//-----------------------------------------------------------------------
WORD CReflectionIBM::EditScript(
	LPSTR sScriptFile)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sScriptFile);

	WORD uErr = InvokeMethodHelper(OLESTR("EditScript"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for EditScriptDlg for CReflectionIBM class.
// Description:	Dialog to edit a Reflection Basic script
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::EditScriptDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("EditScriptDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for EnableInternalTrace for CReflectionIBM class.
// Description:	Enable or disable internal trace nodes
// Syntax:	sTraceNode, nEnableOrDisable
//-----------------------------------------------------------------------
WORD CReflectionIBM::EnableInternalTrace(
	LPSTR sTraceNode,
	DWORD nEnableOrDisable)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sTraceNode);
	CopyToVariant(&pvArgs[0], nEnableOrDisable);

	WORD uErr = InvokeMethodHelper(OLESTR("EnableInternalTrace"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for EnableOnEvent for CReflectionIBM class.
// Description:	Enable or disable event commands
// Syntax:	nOnEventNumber, nEnableOrDisable
//-----------------------------------------------------------------------
WORD CReflectionIBM::EnableOnEvent(
	DWORD nOnEventNumber,
	DWORD nEnableOrDisable)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nOnEventNumber);
	CopyToVariant(&pvArgs[0], nEnableOrDisable);

	WORD uErr = InvokeMethodHelper(OLESTR("EnableOnEvent"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for Exit for CReflectionIBM class.
// Description:	Exit Reflection
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::Exit(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("Exit"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ExitAll for CReflectionIBM class.
// Description:	Close all Reflection sessions
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ExitAll(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ExitAll"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ExtendSelection for CReflectionIBM class.
// Description:	Extend the selection
// Syntax:	nRow, nColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::ExtendSelection(
	LONG nRow,
	LONG nColumn)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nColumn);

	WORD uErr = InvokeMethodHelper(OLESTR("ExtendSelection"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ExtendSelectionLine for CReflectionIBM class.
// Description:	Extend the selection to the end of the line
// Syntax:	nDirection
//-----------------------------------------------------------------------
WORD CReflectionIBM::ExtendSelectionLine(
	DWORD nDirection)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nDirection);

	WORD uErr = InvokeMethodHelper(OLESTR("ExtendSelectionLine"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ExtendSelectionPage for CReflectionIBM class.
// Description:	Extend the selection to the end of the page
// Syntax:	nDirection
//-----------------------------------------------------------------------
WORD CReflectionIBM::ExtendSelectionPage(
	DWORD nDirection)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nDirection);

	WORD uErr = InvokeMethodHelper(OLESTR("ExtendSelectionPage"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ExtendSelectionRect for CReflectionIBM class.
// Description:	Extend the selection in a rectangular fashion
// Syntax:	nRow, nColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::ExtendSelectionRect(
	LONG nRow,
	LONG nColumn)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nColumn);

	WORD uErr = InvokeMethodHelper(OLESTR("ExtendSelectionRect"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ExtendSelectionWord for CReflectionIBM class.
// Description:	Extend the selection to the end of the word
// Syntax:	nDirection
//-----------------------------------------------------------------------
WORD CReflectionIBM::ExtendSelectionWord(
	DWORD nDirection)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nDirection);

	WORD uErr = InvokeMethodHelper(OLESTR("ExtendSelectionWord"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for FontDefault for CReflectionIBM class.
// Description:	Set the terminal font size to its default value
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::FontDefault(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("FontDefault"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for FontGrow for CReflectionIBM class.
// Description:	Increase the terminal font size
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::FontGrow(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("FontGrow"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for FontShrink for CReflectionIBM class.
// Description:	Decrease the terminal font size
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::FontShrink(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("FontShrink"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for FTPClient for CReflectionIBM class.
// Description:	Start FTP Client
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::FTPClient(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("FTPClient"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GraphicsMouse for CReflectionIBM class.
// Description:	Simulate a graphics mouse click at the given position
// Syntax:	nType, nXCoordinate, nYCoordinate
//-----------------------------------------------------------------------
WORD CReflectionIBM::GraphicsMouse(
	DWORD nType,
	LONG nXCoordinate,
	LONG nYCoordinate)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nType);
	CopyToVariant(&pvArgs[1], nXCoordinate);
	CopyToVariant(&pvArgs[0], nYCoordinate);

	WORD uErr = InvokeMethodHelper(OLESTR("GraphicsMouse"), pvArgs, 3, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for Help for CReflectionIBM class.
// Description:	Start WinHelp on the given topic
// Syntax:	Topic
//-----------------------------------------------------------------------
WORD CReflectionIBM::Help(
	DWORD Topic)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], Topic);

	WORD uErr = InvokeMethodHelper(OLESTR("Help"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for HelpCursor for CReflectionIBM class.
// Description:	Enter mouse cursor help mode
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::HelpCursor(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("HelpCursor"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for HotlistSetupDlg for CReflectionIBM class.
// Description:	Dialog to configure hotlists
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::HotlistSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("HotlistSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for HotspotSetupDlg for CReflectionIBM class.
// Description:	Dialog to map display strings to Reflection commands
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::HotspotSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("HotspotSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ImportVBAProjectItemDlg for CReflectionIBM class.
// Description:	Run the Import Visual Basic File dialog
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ImportVBAProjectItemDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ImportVBAProjectItemDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for IndReceiveFile for CReflectionIBM class.
// Description:	Receive a file from the host using IND$FILE
// Syntax:	sLocalFile, sHostFile, nType, nExistsOption, nShowStat
//-----------------------------------------------------------------------
WORD CReflectionIBM::IndReceiveFile(
	LPSTR sLocalFile,
	LPSTR sHostFile,
	DWORD nType,
	DWORD nExistsOption,
	DWORD nShowStat)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[4], sLocalFile);
	CopyToVariant(&pvArgs[3], sHostFile);
	CopyToVariant(&pvArgs[2], nType);
	CopyToVariant(&pvArgs[1], nExistsOption);
	CopyToVariant(&pvArgs[0], nShowStat);

	WORD uErr = InvokeMethodHelper(OLESTR("IndReceiveFile"), pvArgs, 5, NULL);

	CleanUpStringVariant(&pvArgs[4]);
	CleanUpStringVariant(&pvArgs[3]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for IndSendFile for CReflectionIBM class.
// Description:	Send a file to the host using IND$FILE
// Syntax:	sLocalFile, sHostFile, nType, nExistsOption, nShowStat
//-----------------------------------------------------------------------
WORD CReflectionIBM::IndSendFile(
	LPSTR sLocalFile,
	LPSTR sHostFile,
	DWORD nType,
	DWORD nExistsOption,
	DWORD nShowStat)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[4], sLocalFile);
	CopyToVariant(&pvArgs[3], sHostFile);
	CopyToVariant(&pvArgs[2], nType);
	CopyToVariant(&pvArgs[1], nExistsOption);
	CopyToVariant(&pvArgs[0], nShowStat);

	WORD uErr = InvokeMethodHelper(OLESTR("IndSendFile"), pvArgs, 5, NULL);

	CleanUpStringVariant(&pvArgs[4]);
	CleanUpStringVariant(&pvArgs[3]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for InsertPopupItem for CReflectionIBM class.
// Description:	Insert an item in the main menu or a popup menu
// Syntax:	nItemType, nPopup, nItem, sName, sDescription, sCommands
//-----------------------------------------------------------------------
WORD CReflectionIBM::InsertPopupItem(
	DWORD nItemType,
	DWORD nPopup,
	DWORD nItem,
	LPSTR sName,
	LPSTR sDescription,
	LPSTR sCommands)

{
	VARIANTARG pvArgs[6];
	InitVariants(pvArgs, 6);

	CopyToVariant(&pvArgs[5], nItemType);
	CopyToVariant(&pvArgs[4], nPopup);
	CopyToVariant(&pvArgs[3], nItem);
	CopyToVariant(&pvArgs[2], sName);
	CopyToVariant(&pvArgs[1], sDescription);
	CopyToVariant(&pvArgs[0], sCommands);

	WORD uErr = InvokeMethodHelper(OLESTR("InsertPopupItem"), pvArgs, 6, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for InternalTraceDlg for CReflectionIBM class.
// Description:	Set internal trace options
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::InternalTraceDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("InternalTraceDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for IssuePrinterCommand for CReflectionIBM class.
// Description:	Issue a command to the printer
// Syntax:	nPrinterCommand
//-----------------------------------------------------------------------
WORD CReflectionIBM::IssuePrinterCommand(
	DWORD nPrinterCommand)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nPrinterCommand);

	WORD uErr = InvokeMethodHelper(OLESTR("IssuePrinterCommand"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for KeyboardMapSetupDlg for CReflectionIBM class.
// Description:	Dialog to change the keyboard map
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::KeyboardMapSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("KeyboardMapSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for LightPen for CReflectionIBM class.
// Description:	Simulate a light pen at the given coordinates
// Syntax:	nRow, nColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::LightPen(
	LONG nRow,
	LONG nColumn)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nColumn);

	WORD uErr = InvokeMethodHelper(OLESTR("LightPen"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for MacrosDlg for CReflectionIBM class.
// Description:	Run VBA Macro dialog
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::MacrosDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("MacrosDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for MenuSetupDlg for CReflectionIBM class.
// Description:	Dialog to configure menus and items
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::MenuSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("MenuSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for MouseMapSetupDlg for CReflectionIBM class.
// Description:	Dialog to change the mouse map
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::MouseMapSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("MouseMapSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for MoveCursor for CReflectionIBM class.
// Description:	Move the cursor
// Syntax:	nRow, nColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::MoveCursor(
	LONG nRow,
	LONG nColumn)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nColumn);

	WORD uErr = InvokeMethodHelper(OLESTR("MoveCursor"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for MoveGraphicsCursor for CReflectionIBM class.
// Description:	Move the graphics cursor
// Syntax:	nXCoordinate, nYCoordinate
//-----------------------------------------------------------------------
WORD CReflectionIBM::MoveGraphicsCursor(
	LONG nXCoordinate,
	LONG nYCoordinate)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nXCoordinate);
	CopyToVariant(&pvArgs[0], nYCoordinate);

	WORD uErr = InvokeMethodHelper(OLESTR("MoveGraphicsCursor"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for NewHotlist for CReflectionIBM class.
// Description:	Create a new hotlist
// Syntax:	sName, nOptions
//-----------------------------------------------------------------------
WORD CReflectionIBM::NewHotlist(
	LPSTR sName,
	DWORD nOptions)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sName);
	CopyToVariant(&pvArgs[0], nOptions);

	WORD uErr = InvokeMethodHelper(OLESTR("NewHotlist"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for NewSession for CReflectionIBM class.
// Description:	Begin a new Reflection session
// Syntax:	sFilename
//-----------------------------------------------------------------------
WORD CReflectionIBM::NewSession(
	LPSTR sFilename)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sFilename);

	WORD uErr = InvokeMethodHelper(OLESTR("NewSession"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for NewSessionDlg for CReflectionIBM class.
// Description:	Dialog to choose a new Reflection session
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::NewSessionDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("NewSessionDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for NewToolbar for CReflectionIBM class.
// Description:	Create a new toolbar
// Syntax:	sName
//-----------------------------------------------------------------------
WORD CReflectionIBM::NewToolbar(
	LPSTR sName)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sName);

	WORD uErr = InvokeMethodHelper(OLESTR("NewToolbar"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for OnEvent for CReflectionIBM class.
// Description:	Define commands to be executed when an event occurs
// Syntax:	nEvNumber, nEvent, sCommands, nEnable, nAfterEvent, sString, nRow, nColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::OnEvent(
	DWORD nEvNumber,
	DWORD nEvent,
	LPSTR sCommands,
	DWORD nEnable,
	DWORD nAfterEvent,
	LPSTR sString,
	DWORD nRow,
	DWORD nColumn)

{
	VARIANTARG pvArgs[8];
	InitVariants(pvArgs, 8);

	CopyToVariant(&pvArgs[7], nEvNumber);
	CopyToVariant(&pvArgs[6], nEvent);
	CopyToVariant(&pvArgs[5], sCommands);
	CopyToVariant(&pvArgs[4], nEnable);
	CopyToVariant(&pvArgs[3], nAfterEvent);
	CopyToVariant(&pvArgs[2], sString);
	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nColumn);

	WORD uErr = InvokeMethodHelper(OLESTR("OnEvent"), pvArgs, 8, NULL);

	CleanUpStringVariant(&pvArgs[5]);
	CleanUpStringVariant(&pvArgs[2]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for OnEventSetupDlg for CReflectionIBM class.
// Description:	Dialog to configure asynchronous events
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::OnEventSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("OnEventSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for OpenLayoutDlg for CReflectionIBM class.
// Description:	Dialog to open layout files
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::OpenLayoutDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("OpenLayoutDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for OpenSettings for CReflectionIBM class.
// Description:	Open a settings file
// Syntax:	nSettingsType, sFilename
//-----------------------------------------------------------------------
WORD CReflectionIBM::OpenSettings(
	DWORD nSettingsType,
	LPSTR sFilename)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nSettingsType);
	CopyToVariant(&pvArgs[0], sFilename);

	WORD uErr = InvokeMethodHelper(OLESTR("OpenSettings"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for OpenSettingsDlg for CReflectionIBM class.
// Description:	Dialog to open settings files
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::OpenSettingsDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("OpenSettingsDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for OpenTemplateDlg for CReflectionIBM class.
// Description:	Dialog to open a file that contains a settings template
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::OpenTemplateDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("OpenTemplateDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PageSetupDlg for CReflectionIBM class.
// Description:	Dialog to change Windows page settings
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::PageSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("PageSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for Paste for CReflectionIBM class.
// Description:	Paste from the clipboard
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::Paste(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("Paste"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PasteNext for CReflectionIBM class.
// Description:	Continue previous paste operation
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::PasteNext(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("PasteNext"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PasteRange for CReflectionIBM class.
// Description:	Paste range from the clipboard
// Syntax:	n1stTop, n1stLeft, n1stBottom, n1stRight, nRemTop, nRemLeft, nRemBottom, nRemRight, nKey, nWrapText, nUseFieldDelim
//-----------------------------------------------------------------------
WORD CReflectionIBM::PasteRange(
	LONG n1stTop,
	LONG n1stLeft,
	LONG n1stBottom,
	LONG n1stRight,
	LONG nRemTop,
	LONG nRemLeft,
	LONG nRemBottom,
	LONG nRemRight,
	DWORD nKey,
	DWORD nWrapText,
	DWORD nUseFieldDelim)

{
	VARIANTARG pvArgs[11];
	InitVariants(pvArgs, 11);

	CopyToVariant(&pvArgs[10], n1stTop);
	CopyToVariant(&pvArgs[9], n1stLeft);
	CopyToVariant(&pvArgs[8], n1stBottom);
	CopyToVariant(&pvArgs[7], n1stRight);
	CopyToVariant(&pvArgs[6], nRemTop);
	CopyToVariant(&pvArgs[5], nRemLeft);
	CopyToVariant(&pvArgs[4], nRemBottom);
	CopyToVariant(&pvArgs[3], nRemRight);
	CopyToVariant(&pvArgs[2], nKey);
	CopyToVariant(&pvArgs[1], nWrapText);
	CopyToVariant(&pvArgs[0], nUseFieldDelim);

	WORD uErr = InvokeMethodHelper(OLESTR("PasteRange"), pvArgs, 11, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PasteRangeDlg for CReflectionIBM class.
// Description:	Dialog to insert clipboard contents across multiple screens
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::PasteRangeDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("PasteRangeDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PCSReceiveFile for CReflectionIBM class.
// Description:	Receive a file from the host using PC Support
// Syntax:	sTransferFile, nShowStat
//-----------------------------------------------------------------------
WORD CReflectionIBM::PCSReceiveFile(
	LPSTR sTransferFile,
	DWORD nShowStat)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sTransferFile);
	CopyToVariant(&pvArgs[0], nShowStat);

	WORD uErr = InvokeMethodHelper(OLESTR("PCSReceiveFile"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PCSSendFile for CReflectionIBM class.
// Description:	Send a file to the host using PC Support
// Syntax:	sTransferFile, nShowStat
//-----------------------------------------------------------------------
WORD CReflectionIBM::PCSSendFile(
	LPSTR sTransferFile,
	DWORD nShowStat)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sTransferFile);
	CopyToVariant(&pvArgs[0], nShowStat);

	WORD uErr = InvokeMethodHelper(OLESTR("PCSSendFile"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PlayBackTrace for CReflectionIBM class.
// Description:	Play back a recorded trace of host data or commands
// Syntax:	nType, sFilename
//-----------------------------------------------------------------------
WORD CReflectionIBM::PlayBackTrace(
	DWORD nType,
	LPSTR sFilename)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nType);
	CopyToVariant(&pvArgs[0], sFilename);

	WORD uErr = InvokeMethodHelper(OLESTR("PlayBackTrace"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PlayBackTraceDlg for CReflectionIBM class.
// Description:	Dialog to play back a trace of host data or commands
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::PlayBackTraceDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("PlayBackTraceDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PlayNextTraceRecord for CReflectionIBM class.
// Description:	Play next trace record
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::PlayNextTraceRecord(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("PlayNextTraceRecord"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PlayPreviousTraceRecord for CReflectionIBM class.
// Description:	Play previous trace record
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::PlayPreviousTraceRecord(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("PlayPreviousTraceRecord"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PreferencesSetupDlg for CReflectionIBM class.
// Description:	Dialog to change preferences
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::PreferencesSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("PreferencesSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for Print for CReflectionIBM class.
// Description:	Print the screen/selection/partition
// Syntax:	nItem, nCopies
//-----------------------------------------------------------------------
WORD CReflectionIBM::Print(
	DWORD nItem,
	DWORD nCopies)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nItem);
	CopyToVariant(&pvArgs[0], nCopies);

	WORD uErr = InvokeMethodHelper(OLESTR("Print"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PrintControl for CReflectionIBM class.
// Description:	Send a print control command to the printer session
// Syntax:	nPrintControl
//-----------------------------------------------------------------------
WORD CReflectionIBM::PrintControl(
	DWORD nPrintControl)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nPrintControl);

	WORD uErr = InvokeMethodHelper(OLESTR("PrintControl"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PrintDlg for CReflectionIBM class.
// Description:	Dialog to print screen/selection/partition
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::PrintDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("PrintDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PrintHostFile for CReflectionIBM class.
// Description:	Print a file from the host using IND$FILE
// Syntax:	HostFile, Type, CarriageControl
//-----------------------------------------------------------------------
WORD CReflectionIBM::PrintHostFile(
	LPSTR HostFile,
	DWORD Type,
	DWORD CarriageControl)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], HostFile);
	CopyToVariant(&pvArgs[1], Type);
	CopyToVariant(&pvArgs[0], CarriageControl);

	WORD uErr = InvokeMethodHelper(OLESTR("PrintHostFile"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PrintSetupDlg for CReflectionIBM class.
// Description:	Dialog to change Windows printer settings
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::PrintSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("PrintSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PrintString for CReflectionIBM class.
// Description:	Print a line of text
// Syntax:	sString
//-----------------------------------------------------------------------
WORD CReflectionIBM::PrintString(
	LPSTR sString)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sString);

	WORD uErr = InvokeMethodHelper(OLESTR("PrintString"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PrintTransparentChar for CReflectionIBM class.
// Description:	Print a character in transparent mode
// Syntax:	nASCIICode
//-----------------------------------------------------------------------
WORD CReflectionIBM::PrintTransparentChar(
	DWORD nASCIICode)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nASCIICode);

	WORD uErr = InvokeMethodHelper(OLESTR("PrintTransparentChar"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RaiseControlEvent for CReflectionIBM class.
// Description:	Proxy an event notification out to an OLE or COM container
// Syntax:	nIntData, sStrData
//-----------------------------------------------------------------------
WORD CReflectionIBM::RaiseControlEvent(
	DWORD nIntData,
	LPSTR sStrData)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nIntData);
	CopyToVariant(&pvArgs[0], sStrData);

	WORD uErr = InvokeMethodHelper(OLESTR("RaiseControlEvent"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ReadQuickText for CReflectionIBM class.
// Description:	Determine and copy display window text to quick text
// Syntax:	nRow, nColumn, nReadOption
//-----------------------------------------------------------------------
WORD CReflectionIBM::ReadQuickText(
	LONG nRow,
	LONG nColumn,
	DWORD nReadOption)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nRow);
	CopyToVariant(&pvArgs[1], nColumn);
	CopyToVariant(&pvArgs[0], nReadOption);

	WORD uErr = InvokeMethodHelper(OLESTR("ReadQuickText"), pvArgs, 3, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RecordingSetupDlg for CReflectionIBM class.
// Description:	Dialog to configure command recording
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::RecordingSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("RecordingSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ReferencesDlg for CReflectionIBM class.
// Description:	Manage Visual Basic references
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ReferencesDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ReferencesDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RemoveAllToolbars for CReflectionIBM class.
// Description:	Remove all toolbars
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::RemoveAllToolbars(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("RemoveAllToolbars"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RemoveButton for CReflectionIBM class.
// Description:	Remove a button from the toolbar
// Syntax:	nIndex
//-----------------------------------------------------------------------
WORD CReflectionIBM::RemoveButton(
	DWORD nIndex)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nIndex);

	WORD uErr = InvokeMethodHelper(OLESTR("RemoveButton"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RemoveHotlist for CReflectionIBM class.
// Description:	Remove a hotlist
// Syntax:	sName
//-----------------------------------------------------------------------
WORD CReflectionIBM::RemoveHotlist(
	LPSTR sName)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sName);

	WORD uErr = InvokeMethodHelper(OLESTR("RemoveHotlist"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RemoveNWSAA3270Config for CReflectionIBM class.
// Description:	Remove a NetWare SAA for 3270 configuration
// Syntax:	Index
//-----------------------------------------------------------------------
WORD CReflectionIBM::RemoveNWSAA3270Config(
	DWORD Index)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], Index);

	WORD uErr = InvokeMethodHelper(OLESTR("RemoveNWSAA3270Config"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RemoveOnEvent for CReflectionIBM class.
// Description:	Remove an asynchronous event
// Syntax:	nEvent
//-----------------------------------------------------------------------
WORD CReflectionIBM::RemoveOnEvent(
	DWORD nEvent)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nEvent);

	WORD uErr = InvokeMethodHelper(OLESTR("RemoveOnEvent"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RemoveToolbar for CReflectionIBM class.
// Description:	Remove a toolbar
// Syntax:	sName
//-----------------------------------------------------------------------
WORD CReflectionIBM::RemoveToolbar(
	LPSTR sName)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sName);

	WORD uErr = InvokeMethodHelper(OLESTR("RemoveToolbar"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RemoveToolbarButton for CReflectionIBM class.
// Description:	Remove a button from a toolbar
// Syntax:	sToolbarName,nIndex
//-----------------------------------------------------------------------
WORD CReflectionIBM::RemoveToolbarButton(
	LPSTR sToolbarName,
	DWORD nIndex)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sToolbarName);
	CopyToVariant(&pvArgs[0], nIndex);

	WORD uErr = InvokeMethodHelper(OLESTR("RemoveToolbarButton"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ResetCommands for CReflectionIBM class.
// Description:	Reset Reflection and clear all commands
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ResetCommands(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ResetCommands"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RestoreDefaults for CReflectionIBM class.
// Description:	Reset entire or partial settings to defaults
// Syntax:	nSettingsType
//-----------------------------------------------------------------------
WORD CReflectionIBM::RestoreDefaults(
	DWORD nSettingsType)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nSettingsType);

	WORD uErr = InvokeMethodHelper(OLESTR("RestoreDefaults"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RunExternalMacro for CReflectionIBM class.
// Description:	Run an external VBA Macro
// Syntax:	sFileName, sMacroName, sMacroData
//-----------------------------------------------------------------------
WORD CReflectionIBM::RunExternalMacro(
	LPSTR sFileName,
	LPSTR sMacroName,
	LPSTR sMacroData)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], sFileName);
	CopyToVariant(&pvArgs[1], sMacroName);
	CopyToVariant(&pvArgs[0], sMacroData);

	WORD uErr = InvokeMethodHelper(OLESTR("RunExternalMacro"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RunMacro for CReflectionIBM class.
// Description:	Run a VBA Macro
// Syntax:	sMacroName, sMacroData
//-----------------------------------------------------------------------
WORD CReflectionIBM::RunMacro(
	LPSTR sMacroName,
	LPSTR sMacroData)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sMacroName);
	CopyToVariant(&pvArgs[0], sMacroData);

	WORD uErr = InvokeMethodHelper(OLESTR("RunMacro"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RunScript for CReflectionIBM class.
// Description:	Run a Reflection Basic script
// Syntax:	sScriptFile, sArguments
//-----------------------------------------------------------------------
WORD CReflectionIBM::RunScript(
	LPSTR sScriptFile,
	LPSTR sArguments)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sScriptFile);
	CopyToVariant(&pvArgs[0], sArguments);

	WORD uErr = InvokeMethodHelper(OLESTR("RunScript"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RunScriptDlg for CReflectionIBM class.
// Description:	Dialog to run a Reflection Basic script
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::RunScriptDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("RunScriptDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SaveDisplay for CReflectionIBM class.
// Description:	Save display image to file.
// Syntax:	nLayeringOption, nExistOption, sFileName nColorType, nBackground
//-----------------------------------------------------------------------
WORD CReflectionIBM::SaveDisplay(
	DWORD nLayeringOption,
	DWORD nExistOption,
	LPSTR sFileName,
	DWORD nColorType,
	DWORD nBackground)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[4], nLayeringOption);
	CopyToVariant(&pvArgs[3], nExistOption);
	CopyToVariant(&pvArgs[2], sFileName);
	CopyToVariant(&pvArgs[1], nColorType);
	CopyToVariant(&pvArgs[0], nBackground);

	WORD uErr = InvokeMethodHelper(OLESTR("SaveDisplay"), pvArgs, 5, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SaveDisplayDlg for CReflectionIBM class.
// Description:	Dialog to save screen into a bitmap format.
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::SaveDisplayDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("SaveDisplayDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SaveLayoutAsDlg for CReflectionIBM class.
// Description:	Dialog to save layout files
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::SaveLayoutAsDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("SaveLayoutAsDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SaveSettings for CReflectionIBM class.
// Description:	Save information to the specified file
// Syntax:	nType, nExistOption, sFileName
//-----------------------------------------------------------------------
WORD CReflectionIBM::SaveSettings(
	DWORD nType,
	DWORD nExistOption,
	LPSTR sFileName)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nType);
	CopyToVariant(&pvArgs[1], nExistOption);
	CopyToVariant(&pvArgs[0], sFileName);

	WORD uErr = InvokeMethodHelper(OLESTR("SaveSettings"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SaveSettingsAsDlg for CReflectionIBM class.
// Description:	Dialog to save setting files
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::SaveSettingsAsDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("SaveSettingsAsDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SaveSettingsDlg for CReflectionIBM class.
// Description:	Save the current settings
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::SaveSettingsDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("SaveSettingsDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SaveXML for CReflectionIBM class.
// Description:	Save settings as XML
// Syntax:	sFileName, nXMLType, nExistOption, nOption
//-----------------------------------------------------------------------
WORD CReflectionIBM::SaveXML(
	LPSTR sFileName,
	DWORD nXMLType,
	DWORD nExistOption,
	DWORD nOption)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[3], sFileName);
	CopyToVariant(&pvArgs[2], nXMLType);
	CopyToVariant(&pvArgs[1], nExistOption);
	CopyToVariant(&pvArgs[0], nOption);

	WORD uErr = InvokeMethodHelper(OLESTR("SaveXML"), pvArgs, 4, NULL);

	CleanUpStringVariant(&pvArgs[3]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SelectAll for CReflectionIBM class.
// Description:	Select the entire display page
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::SelectAll(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("SelectAll"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SelectLine for CReflectionIBM class.
// Description:	Select a line on the display
// Syntax:	nLineNumber
//-----------------------------------------------------------------------
WORD CReflectionIBM::SelectLine(
	DWORD nLineNumber)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nLineNumber);

	WORD uErr = InvokeMethodHelper(OLESTR("SelectLine"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SelectWord for CReflectionIBM class.
// Description:	Select the word at the given coordinates
// Syntax:	nRow, nColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::SelectWord(
	LONG nRow,
	LONG nColumn)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nColumn);

	WORD uErr = InvokeMethodHelper(OLESTR("SelectWord"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SendMail for CReflectionIBM class.
// Description:	Send Mail via MAPI
// Syntax:	sRecipient, sSubject, nContentType, sContent
//-----------------------------------------------------------------------
WORD CReflectionIBM::SendMail(
	LPSTR sRecipient,
	LPSTR sSubject,
	DWORD nContentType,
	LPSTR sContent)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[3], sRecipient);
	CopyToVariant(&pvArgs[2], sSubject);
	CopyToVariant(&pvArgs[1], nContentType);
	CopyToVariant(&pvArgs[0], sContent);

	WORD uErr = InvokeMethodHelper(OLESTR("SendMail"), pvArgs, 4, NULL);

	CleanUpStringVariant(&pvArgs[3]);
	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SendMailDlg for CReflectionIBM class.
// Description:	Dialog to send screen contents via electronic mail
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::SendMailDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("SendMailDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SessionSetupDlg for CReflectionIBM class.
// Description:	Dialog to select terminal and connection types
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::SessionSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("SessionSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetAPLKeyMap for CReflectionIBM class.
// Description:	Map an APL keystroke to Reflection commands
// Syntax:	nControls, sKey, sCommands
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetAPLKeyMap(
	DWORD nControls,
	LPSTR sKey,
	LPSTR sCommands)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nControls);
	CopyToVariant(&pvArgs[1], sKey);
	CopyToVariant(&pvArgs[0], sCommands);

	WORD uErr = InvokeMethodHelper(OLESTR("SetAPLKeyMap"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetColorMap for CReflectionIBM class.
// Description:	Map colors to terminal attributes
// Syntax:	nAttribute, nForeground, nBackground
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetColorMap(
	DWORD nAttribute,
	DWORD nForeground,
	DWORD nBackground)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nAttribute);
	CopyToVariant(&pvArgs[1], nForeground);
	CopyToVariant(&pvArgs[0], nBackground);

	WORD uErr = InvokeMethodHelper(OLESTR("SetColorMap"), pvArgs, 3, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetColorRGB for CReflectionIBM class.
// Description:	Assign RGB values to colors in palette
// Syntax:	nColor, nRed, nGreen, nBlue
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetColorRGB(
	DWORD nColor,
	DWORD nRed,
	DWORD nGreen,
	DWORD nBlue)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[3], nColor);
	CopyToVariant(&pvArgs[2], nRed);
	CopyToVariant(&pvArgs[1], nGreen);
	CopyToVariant(&pvArgs[0], nBlue);

	WORD uErr = InvokeMethodHelper(OLESTR("SetColorRGB"), pvArgs, 4, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetCommandProfile for CReflectionIBM class.
// Description:	Enable or disable execution of a command
// Syntax:	sCommandName, nEnableFlag
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetCommandProfile(
	LPSTR sCommandName,
	DWORD nEnableFlag)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sCommandName);
	CopyToVariant(&pvArgs[0], nEnableFlag);

	WORD uErr = InvokeMethodHelper(OLESTR("SetCommandProfile"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetCommandRecording for CReflectionIBM class.
// Description:	Enable or disable recording of a command
// Syntax:	sCommandName, nEnableFlag
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetCommandRecording(
	LPSTR sCommandName,
	DWORD nEnableFlag)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sCommandName);
	CopyToVariant(&pvArgs[0], nEnableFlag);

	WORD uErr = InvokeMethodHelper(OLESTR("SetCommandRecording"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetDisplayAttribute for CReflectionIBM class.
// Description:	Set how host attribute is displayed by terminal
// Syntax:	nHostAttrIndex, nAttribute, nEnable
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetDisplayAttribute(
	DWORD nHostAttrIndex,
	DWORD nAttribute,
	DWORD nEnable)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nHostAttrIndex);
	CopyToVariant(&pvArgs[1], nAttribute);
	CopyToVariant(&pvArgs[0], nEnable);

	WORD uErr = InvokeMethodHelper(OLESTR("SetDisplayAttribute"), pvArgs, 3, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetEncryptedString for CReflectionIBM class.
// Description:	Assign a new value to an encrypted string setting
// Syntax:	nEncryptedSetting, sString
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetEncryptedString(
	DWORD nEncryptedSetting,
	LPSTR sString)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nEncryptedSetting);
	CopyToVariant(&pvArgs[0], sString);

	WORD uErr = InvokeMethodHelper(OLESTR("SetEncryptedString"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetIBMPrinterFont for CReflectionIBM class.
// Description:	Map a font to an index for IBM printing
// Syntax:	nFontID, sFontDescription
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetIBMPrinterFont(
	DWORD nFontID,
	LPSTR sFontDescription)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nFontID);
	CopyToVariant(&pvArgs[0], sFontDescription);

	WORD uErr = InvokeMethodHelper(OLESTR("SetIBMPrinterFont"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetKeyMap for CReflectionIBM class.
// Description:	Map a keystroke to Reflection commands
// Syntax:	nControls, sKey, sCommands
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetKeyMap(
	DWORD nControls,
	LPSTR sKey,
	LPSTR sCommands)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nControls);
	CopyToVariant(&pvArgs[1], sKey);
	CopyToVariant(&pvArgs[0], sCommands);

	WORD uErr = InvokeMethodHelper(OLESTR("SetKeyMap"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetMouseMap for CReflectionIBM class.
// Description:	Map a mouse chord to Reflection commands
// Syntax:	nControls, sChord, sCommands
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetMouseMap(
	DWORD nControls,
	LPSTR sChord,
	LPSTR sCommands)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nControls);
	CopyToVariant(&pvArgs[1], sChord);
	CopyToVariant(&pvArgs[0], sCommands);

	WORD uErr = InvokeMethodHelper(OLESTR("SetMouseMap"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetMousePos for CReflectionIBM class.
// Description:	Set the position for Mouse Row and Column
// Syntax:	nRow, nColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetMousePos(
	LONG nRow,
	LONG nColumn)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nColumn);

	WORD uErr = InvokeMethodHelper(OLESTR("SetMousePos"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetNumeric for CReflectionIBM class.
// Description:	Assign a new value to a numeric setting
// Syntax:	nSetting, nValue
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetNumeric(
	DWORD nSetting,
	DWORD nValue)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nSetting);
	CopyToVariant(&pvArgs[0], nValue);

	WORD uErr = InvokeMethodHelper(OLESTR("SetNumeric"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetNumericProfile for CReflectionIBM class.
// Description:	Enable or disable modification of a numeric setting
// Syntax:	nSetting, nEnableFlag
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetNumericProfile(
	DWORD nSetting,
	DWORD nEnableFlag)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nSetting);
	CopyToVariant(&pvArgs[0], nEnableFlag);

	WORD uErr = InvokeMethodHelper(OLESTR("SetNumericProfile"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetRecentlyUsedFile for CReflectionIBM class.
// Description:	Set the most recently used file in the file menu list
// Syntax:	sTitle, sDescription, sCommands
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetRecentlyUsedFile(
	LPSTR sTitle,
	LPSTR sDescription,
	LPSTR sCommands)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], sTitle);
	CopyToVariant(&pvArgs[1], sDescription);
	CopyToVariant(&pvArgs[0], sCommands);

	WORD uErr = InvokeMethodHelper(OLESTR("SetRecentlyUsedFile"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetScanCodeName for CReflectionIBM class.
// Description:	Map a keyboard scan code to a keystroke
// Syntax:	nScanType, nScanCode, sKey
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetScanCodeName(
	DWORD nScanType,
	DWORD nScanCode,
	LPSTR sKey)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nScanType);
	CopyToVariant(&pvArgs[1], nScanCode);
	CopyToVariant(&pvArgs[0], sKey);

	WORD uErr = InvokeMethodHelper(OLESTR("SetScanCodeName"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetSelectionStartPos for CReflectionIBM class.
// Description:	Move the selection starting position
// Syntax:	nRow, nColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetSelectionStartPos(
	LONG nRow,
	LONG nColumn)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nColumn);

	WORD uErr = InvokeMethodHelper(OLESTR("SetSelectionStartPos"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetSQLStatement for CReflectionIBM class.
// Description:	Set the current SQL statement
// Syntax:	nType, sStatement
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetSQLStatement(
	DWORD nType,
	LPSTR sStatement)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nType);
	CopyToVariant(&pvArgs[0], sStatement);

	WORD uErr = InvokeMethodHelper(OLESTR("SetSQLStatement"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetString for CReflectionIBM class.
// Description:	Assign a new value to a string setting
// Syntax:	nSetting, sString
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetString(
	DWORD nSetting,
	LPSTR sString)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nSetting);
	CopyToVariant(&pvArgs[0], sString);

	WORD uErr = InvokeMethodHelper(OLESTR("SetString"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetStringProfile for CReflectionIBM class.
// Description:	Enable or disable modification of a string setting
// Syntax:	nSetting, nEnableFlag
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetStringProfile(
	DWORD nSetting,
	DWORD nEnableFlag)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nSetting);
	CopyToVariant(&pvArgs[0], nEnableFlag);

	WORD uErr = InvokeMethodHelper(OLESTR("SetStringProfile"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetTerminalKbdPos for CReflectionIBM class.
// Description:	Change the size and position of the terminal keyboard
// Syntax:	nLeft, nTop, nWidth, nHeight
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetTerminalKbdPos(
	LONG nLeft,
	LONG nTop,
	LONG nWidth,
	LONG nHeight)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[3], nLeft);
	CopyToVariant(&pvArgs[2], nTop);
	CopyToVariant(&pvArgs[1], nWidth);
	CopyToVariant(&pvArgs[0], nHeight);

	WORD uErr = InvokeMethodHelper(OLESTR("SetTerminalKbdPos"), pvArgs, 4, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetToolbarMode for CReflectionIBM class.
// Description:	Set a toolbar's docking position
// Syntax:	sToolbarName, nMode
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetToolbarMode(
	LPSTR sToolbarName,
	DWORD nMode)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sToolbarName);
	CopyToVariant(&pvArgs[0], nMode);

	WORD uErr = InvokeMethodHelper(OLESTR("SetToolbarMode"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetToolbarPos for CReflectionIBM class.
// Description:	Change the size and position of the toolbar
// Syntax:	nLeft, nTop, nWidth, nHeight
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetToolbarPos(
	LONG nLeft,
	LONG nTop,
	LONG nWidth,
	LONG nHeight)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[3], nLeft);
	CopyToVariant(&pvArgs[2], nTop);
	CopyToVariant(&pvArgs[1], nWidth);
	CopyToVariant(&pvArgs[0], nHeight);

	WORD uErr = InvokeMethodHelper(OLESTR("SetToolbarPos"), pvArgs, 4, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetToolbarPosition for CReflectionIBM class.
// Description:	Change the size and position of a toolbar
// Syntax:	sToolbarName, nLeft, nTop, nWidth, nHeight
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetToolbarPosition(
	LPSTR sToolbarName,
	LONG nLeft,
	LONG nTop,
	LONG nWidth,
	LONG nHeight)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[4], sToolbarName);
	CopyToVariant(&pvArgs[3], nLeft);
	CopyToVariant(&pvArgs[2], nTop);
	CopyToVariant(&pvArgs[1], nWidth);
	CopyToVariant(&pvArgs[0], nHeight);

	WORD uErr = InvokeMethodHelper(OLESTR("SetToolbarPosition"), pvArgs, 5, NULL);

	CleanUpStringVariant(&pvArgs[4]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetupNWSAA3270Config for CReflectionIBM class.
// Description:	Set up a NetWare SAA for 3270 configuration
// Syntax:	sServer, sService, sLU
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetupNWSAA3270Config(
	LPSTR sServer,
	LPSTR sService,
	LPSTR sLU)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], sServer);
	CopyToVariant(&pvArgs[1], sService);
	CopyToVariant(&pvArgs[0], sLU);

	WORD uErr = InvokeMethodHelper(OLESTR("SetupNWSAA3270Config"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetupSession for CReflectionIBM class.
// Description:	Set terminal type, model and connection type
// Syntax:	nSessionType, nTerminalModel, nTransportType
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetupSession(
	DWORD nSessionType,
	DWORD nTerminalModel,
	DWORD nTransportType)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nSessionType);
	CopyToVariant(&pvArgs[1], nTerminalModel);
	CopyToVariant(&pvArgs[0], nTransportType);

	WORD uErr = InvokeMethodHelper(OLESTR("SetupSession"), pvArgs, 3, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetWindowPos for CReflectionIBM class.
// Description:	Change the size and position of the terminal window
// Syntax:	nLeft, nTop, nWidth, nHeight
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetWindowPos(
	LONG nLeft,
	LONG nTop,
	LONG nWidth,
	LONG nHeight)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[3], nLeft);
	CopyToVariant(&pvArgs[2], nTop);
	CopyToVariant(&pvArgs[1], nWidth);
	CopyToVariant(&pvArgs[0], nHeight);

	WORD uErr = InvokeMethodHelper(OLESTR("SetWindowPos"), pvArgs, 4, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for Shell for CReflectionIBM class.
// Description:	Shell an application
// Syntax:	sApplicationName, sArguments, nShowHow
//-----------------------------------------------------------------------
WORD CReflectionIBM::Shell(
	LPSTR sApplicationName,
	LPSTR sArguments,
	DWORD nShowHow)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], sApplicationName);
	CopyToVariant(&pvArgs[1], sArguments);
	CopyToVariant(&pvArgs[0], nShowHow);

	WORD uErr = InvokeMethodHelper(OLESTR("Shell"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ShowDBCSHostCodeInputWindow for CReflectionIBM class.
// Description:	Display a DBCS Host Code Input Window on the screen
// Syntax:	nHostCode
//-----------------------------------------------------------------------
WORD CReflectionIBM::ShowDBCSHostCodeInputWindow(
	DWORD nHostCode)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nHostCode);

	WORD uErr = InvokeMethodHelper(OLESTR("ShowDBCSHostCodeInputWindow"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ShowDisplayAttribute for CReflectionIBM class.
// Description:	Enable or disable display of terminal attribute
// Syntax:	nAttribute, nEnable
//-----------------------------------------------------------------------
WORD CReflectionIBM::ShowDisplayAttribute(
	DWORD nAttribute,
	DWORD nEnable)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nAttribute);
	CopyToVariant(&pvArgs[0], nEnable);

	WORD uErr = InvokeMethodHelper(OLESTR("ShowDisplayAttribute"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ShowHotlist for CReflectionIBM class.
// Description:	Display a hotlist on the screen
// Syntax:	sName, nRow, nColumn, nWidth, nHeight
//-----------------------------------------------------------------------
WORD CReflectionIBM::ShowHotlist(
	LPSTR sName,
	DWORD nRow,
	DWORD nColumn,
	DWORD nWidth,
	DWORD nHeight)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[4], sName);
	CopyToVariant(&pvArgs[3], nRow);
	CopyToVariant(&pvArgs[2], nColumn);
	CopyToVariant(&pvArgs[1], nWidth);
	CopyToVariant(&pvArgs[0], nHeight);

	WORD uErr = InvokeMethodHelper(OLESTR("ShowHotlist"), pvArgs, 5, NULL);

	CleanUpStringVariant(&pvArgs[4]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ShowToolbar for CReflectionIBM class.
// Description:	Change the toolbar visiblity
// Syntax:	sToolbarName, nYesNo
//-----------------------------------------------------------------------
WORD CReflectionIBM::ShowToolbar(
	LPSTR sToolbarName,
	DWORD nYesNo)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sToolbarName);
	CopyToVariant(&pvArgs[0], nYesNo);

	WORD uErr = InvokeMethodHelper(OLESTR("ShowToolbar"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ShowToolbarButtonsFixedWidth for CReflectionIBM class.
// Description:	Change the toolbar buttons width
// Syntax:	sToolbarName, nYesNo
//-----------------------------------------------------------------------
WORD CReflectionIBM::ShowToolbarButtonsFixedWidth(
	LPSTR sToolbarName,
	DWORD nYesNo)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sToolbarName);
	CopyToVariant(&pvArgs[0], nYesNo);

	WORD uErr = InvokeMethodHelper(OLESTR("ShowToolbarButtonsFixedWidth"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ShowToolbarColor for CReflectionIBM class.
// Description:	Change the toolbar color
// Syntax:	sToolbarName, nYesNo
//-----------------------------------------------------------------------
WORD CReflectionIBM::ShowToolbarColor(
	LPSTR sToolbarName,
	DWORD nYesNo)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sToolbarName);
	CopyToVariant(&pvArgs[0], nYesNo);

	WORD uErr = InvokeMethodHelper(OLESTR("ShowToolbarColor"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ShowToolbarLabels for CReflectionIBM class.
// Description:	Change the toolbar label visibility
// Syntax:	sToolbarName, nYesNo
//-----------------------------------------------------------------------
WORD CReflectionIBM::ShowToolbarLabels(
	LPSTR sToolbarName,
	DWORD nYesNo)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sToolbarName);
	CopyToVariant(&pvArgs[0], nYesNo);

	WORD uErr = InvokeMethodHelper(OLESTR("ShowToolbarLabels"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ShowToolbarLarge for CReflectionIBM class.
// Description:	Change the toolbar size
// Syntax:	sToolbarName, nYesNo
//-----------------------------------------------------------------------
WORD CReflectionIBM::ShowToolbarLarge(
	LPSTR sToolbarName,
	DWORD nYesNo)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sToolbarName);
	CopyToVariant(&pvArgs[0], nYesNo);

	WORD uErr = InvokeMethodHelper(OLESTR("ShowToolbarLarge"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ShowToolbarTips for CReflectionIBM class.
// Description:	Change the toolbar tooltip visibility
// Syntax:	sToolbarName, nYesNo
//-----------------------------------------------------------------------
WORD CReflectionIBM::ShowToolbarTips(
	LPSTR sToolbarName,
	DWORD nYesNo)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sToolbarName);
	CopyToVariant(&pvArgs[0], nYesNo);

	WORD uErr = InvokeMethodHelper(OLESTR("ShowToolbarTips"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StartInternalTrace for CReflectionIBM class.
// Description:	Begin recording internal trace
// Syntax:	sFileName
//-----------------------------------------------------------------------
WORD CReflectionIBM::StartInternalTrace(
	LPSTR sFileName)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sFileName);

	WORD uErr = InvokeMethodHelper(OLESTR("StartInternalTrace"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StartRecording for CReflectionIBM class.
// Description:	Begin recording Reflection commands
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::StartRecording(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("StartRecording"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StartTrace for CReflectionIBM class.
// Description:	Start trace of host data or commands
// Syntax:	nType, nExistOption, sFileName
//-----------------------------------------------------------------------
WORD CReflectionIBM::StartTrace(
	DWORD nType,
	DWORD nExistOption,
	LPSTR sFileName)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nType);
	CopyToVariant(&pvArgs[1], nExistOption);
	CopyToVariant(&pvArgs[0], sFileName);

	WORD uErr = InvokeMethodHelper(OLESTR("StartTrace"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StartTraceDlg for CReflectionIBM class.
// Description:	Dialog to begin trace of host data or commands
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::StartTraceDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("StartTraceDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StopInternalTrace for CReflectionIBM class.
// Description:	Stop recording internal trace file
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::StopInternalTrace(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("StopInternalTrace"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StopMacro for CReflectionIBM class.
// Description:	Stop the currently running Visual Basic Macro
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::StopMacro(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("StopMacro"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StopRecording for CReflectionIBM class.
// Description:	Stop recording and save recorded data
// Syntax:	nDestination, nExistsOption, sFile
//-----------------------------------------------------------------------
WORD CReflectionIBM::StopRecording(
	DWORD nDestination,
	DWORD nExistsOption,
	LPSTR sFile)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nDestination);
	CopyToVariant(&pvArgs[1], nExistsOption);
	CopyToVariant(&pvArgs[0], sFile);

	WORD uErr = InvokeMethodHelper(OLESTR("StopRecording"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StopRecordingDlg for CReflectionIBM class.
// Description:	Dialog to stop recording and save recorded data
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::StopRecordingDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("StopRecordingDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StopRecordingMacro for CReflectionIBM class.
// Description:	Stop recording and save recorded data
// Syntax:	nDestination, sName, sDescription
//-----------------------------------------------------------------------
WORD CReflectionIBM::StopRecordingMacro(
	DWORD nDestination,
	LPSTR sName,
	LPSTR sDescription)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nDestination);
	CopyToVariant(&pvArgs[1], sName);
	CopyToVariant(&pvArgs[0], sDescription);

	WORD uErr = InvokeMethodHelper(OLESTR("StopRecordingMacro"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StopScript for CReflectionIBM class.
// Description:	Stop the currently running Reflection Basic script
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::StopScript(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("StopScript"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for StopTrace for CReflectionIBM class.
// Description:	Stop recording or playing a trace file
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::StopTrace(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("StopTrace"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SwitchToWindow for CReflectionIBM class.
// Description:	Switch to the specified Reflection session
// Syntax:	nSession
//-----------------------------------------------------------------------
WORD CReflectionIBM::SwitchToWindow(
	DWORD nSession)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nSession);

	WORD uErr = InvokeMethodHelper(OLESTR("SwitchToWindow"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for TerminalMouse for CReflectionIBM class.
// Description:	Simulate a terminal mouse click at the given position
// Syntax:	nType, nRow, nCol
//-----------------------------------------------------------------------
WORD CReflectionIBM::TerminalMouse(
	DWORD nType,
	LONG nRow,
	LONG nCol)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], nType);
	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nCol);

	WORD uErr = InvokeMethodHelper(OLESTR("TerminalMouse"), pvArgs, 3, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for TerminalSetupDlg for CReflectionIBM class.
// Description:	Dialog to change terminal settings
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::TerminalSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("TerminalSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for TetherToolbar for CReflectionIBM class.
// Description:	Change the toolbar tether
// Syntax:	sToolbarName, nYesNo
//-----------------------------------------------------------------------
WORD CReflectionIBM::TetherToolbar(
	LPSTR sToolbarName,
	DWORD nYesNo)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sToolbarName);
	CopyToVariant(&pvArgs[0], nYesNo);

	WORD uErr = InvokeMethodHelper(OLESTR("TetherToolbar"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for Toggle for CReflectionIBM class.
// Description:	Toggle the value of a numeric setting
// Syntax:	nSetting
//-----------------------------------------------------------------------
WORD CReflectionIBM::Toggle(
	DWORD nSetting)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nSetting);

	WORD uErr = InvokeMethodHelper(OLESTR("Toggle"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ToolbarSetupDlg for CReflectionIBM class.
// Description:	Dialog to configure the toolbar
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ToolbarSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ToolbarSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for TransferDlg for CReflectionIBM class.
// Description:	Dialog to transfer files
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::TransferDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("TransferDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for TransferSetupDlg for CReflectionIBM class.
// Description:	Dialog to change file transfer settings
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::TransferSetupDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("TransferSetupDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for TransformXML for CReflectionIBM class.
// Description:	Transform XML file using XSL stylesheet file
// Syntax:	sInputFile, sTransformFile, sOutputFile
//-----------------------------------------------------------------------
WORD CReflectionIBM::TransformXML(
	LPSTR sInputFile,
	LPSTR sTransformFile,
	LPSTR sOutputFile)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], sInputFile);
	CopyToVariant(&pvArgs[1], sTransformFile);
	CopyToVariant(&pvArgs[0], sOutputFile);

	WORD uErr = InvokeMethodHelper(OLESTR("TransformXML"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for TransmitANSI for CReflectionIBM class.
// Description:	Transmit an ANSI string to the host
// Syntax:	sString
//-----------------------------------------------------------------------
WORD CReflectionIBM::TransmitANSI(
	LPSTR sString)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sString);

	WORD uErr = InvokeMethodHelper(OLESTR("TransmitANSI"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for TransmitTerminalKey for CReflectionIBM class.
// Description:	Send a terminal function key to the host
// Syntax:	nKey
//-----------------------------------------------------------------------
WORD CReflectionIBM::TransmitTerminalKey(
	DWORD nKey)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nKey);

	WORD uErr = InvokeMethodHelper(OLESTR("TransmitTerminalKey"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ViewSettingsDlg for CReflectionIBM class.
// Description:	Dialog to view/modify settings
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ViewSettingsDlg(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ViewSettingsDlg"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for Wait for CReflectionIBM class.
// Description:	Wait for the specified amount of time
// Syntax:	sDuration
//-----------------------------------------------------------------------
WORD CReflectionIBM::Wait(
	LPSTR sDuration)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sDuration);

	WORD uErr = InvokeMethodHelper(OLESTR("Wait"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitForCommString for CReflectionIBM class.
// Description:	Wait until the given string is received from the host
// Syntax:	sString, sTimeOut
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitForCommString(
	LPSTR sString,
	LPSTR sTimeOut)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sString);
	CopyToVariant(&pvArgs[0], sTimeOut);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitForCommString"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitForDisplayString for CReflectionIBM class.
// Description:	Wait until the given string appears on the display
// Syntax:	sString, sTimeOut, nRow, nCol
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitForDisplayString(
	LPSTR sString,
	LPSTR sTimeOut,
	DWORD nRow,
	DWORD nCol)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[3], sString);
	CopyToVariant(&pvArgs[2], sTimeOut);
	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nCol);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitForDisplayString"), pvArgs, 4, NULL);

	CleanUpStringVariant(&pvArgs[3]);
	CleanUpStringVariant(&pvArgs[2]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitForEvent for CReflectionIBM class.
// Description:	Wait until the specified event occurs
// Syntax:	nEvent, sTimeOut, sDuration, nRow, nCol
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitForEvent(
	DWORD nEvent,
	LPSTR sTimeOut,
	LPSTR sDuration,
	LONG nRow,
	LONG nCol)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[4], nEvent);
	CopyToVariant(&pvArgs[3], sTimeOut);
	CopyToVariant(&pvArgs[2], sDuration);
	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nCol);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitForEvent"), pvArgs, 5, NULL);

	CleanUpStringVariant(&pvArgs[3]);
	CleanUpStringVariant(&pvArgs[2]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitForPrinterStatusString for CReflectionIBM class.
// Description:	Wait until the given string appears on the status box of the printer session
// Syntax:	sString, sTimeOut
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitForPrinterStatusString(
	LPSTR sString,
	LPSTR sTimeOut)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sString);
	CopyToVariant(&pvArgs[0], sTimeOut);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitForPrinterStatusString"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitForPrinterString for CReflectionIBM class.
// Description:	Wait until the given string is sent to the printer
// Syntax:	sString, sTimeOut
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitForPrinterString(
	LPSTR sString,
	LPSTR sTimeOut)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], sString);
	CopyToVariant(&pvArgs[0], sTimeOut);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitForPrinterString"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitForPrinterStringReplace for CReflectionIBM class.
// Description:	Wait until the given string, which will be replaced, is sent to the printer
// Syntax:	sString, sTimeOut, nCol
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitForPrinterStringReplace(
	LPSTR sString,
	LPSTR sTimeOut,
	DWORD nCol)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], sString);
	CopyToVariant(&pvArgs[1], sTimeOut);
	CopyToVariant(&pvArgs[0], nCol);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitForPrinterStringReplace"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitForTerminalKey for CReflectionIBM class.
// Description:	Wait until the given terminal key is sent to the host
// Syntax:	nTerminalKey, sTimeOut
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitForTerminalKey(
	DWORD nTerminalKey,
	LPSTR sTimeOut)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], nTerminalKey);
	CopyToVariant(&pvArgs[0], sTimeOut);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitForTerminalKey"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitForTimeOfDay for CReflectionIBM class.
// Description:	Wait until the specified time of day
// Syntax:	sTimeOfDay
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitForTimeOfDay(
	LPSTR sTimeOfDay)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], sTimeOfDay);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitForTimeOfDay"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WriteQuickText for CReflectionIBM class.
// Description:	Write quick text to display window at command prompt
// Syntax:	nTerminalKey
//-----------------------------------------------------------------------
WORD CReflectionIBM::WriteQuickText(
	DWORD nTerminalKey)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], nTerminalKey);

	WORD uErr = InvokeMethodHelper(OLESTR("WriteQuickText"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for CheckEvent for CReflectionIBM class.
// Description:	Check for the occurrence of a defined event
// Syntax:	Event
//-----------------------------------------------------------------------
WORD CReflectionIBM::CheckEvent(
	LPDWORD Event)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("CheckEvent"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[0], Event);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ClearEvents for CReflectionIBM class.
// Description:	Delete all defined events
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::ClearEvents(void)
{

	WORD uErr = InvokeMethodHelper(OLESTR("ClearEvents"), NULL, 0, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for CommitLoginProperties for CReflectionIBM class.
// Description:	Commit changes to single signon database
// Syntax:	Options
//-----------------------------------------------------------------------
WORD CReflectionIBM::CommitLoginProperties(
	DWORD Options)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], Options);

	WORD uErr = InvokeMethodHelper(OLESTR("CommitLoginProperties"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for CompileScript for CReflectionIBM class.
// Description:	Compile a script
// Syntax:	ScriptName
//-----------------------------------------------------------------------
WORD CReflectionIBM::CompileScript(
	LPSTR ScriptName,
	LPDWORD lpdwArg1)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], ScriptName);

	WORD uErr = InvokeMethodHelper(OLESTR("CompileScript"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], lpdwArg1);
	}

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for DefineEvent for CReflectionIBM class.
// Description:	Define an event
// Syntax:	Event, EventType, Duration, String, Row, Column, Key
//-----------------------------------------------------------------------
WORD CReflectionIBM::DefineEvent(
	DWORD Event,
	DWORD EventType,
	LPSTR Duration,
	LPSTR String,
	DWORD Row,
	DWORD Column,
	DWORD Key)

{
	VARIANTARG pvArgs[7];
	InitVariants(pvArgs, 7);

	CopyToVariant(&pvArgs[6], Event);
	CopyToVariant(&pvArgs[5], EventType);
	CopyToVariant(&pvArgs[4], Duration);
	CopyToVariant(&pvArgs[3], String);
	CopyToVariant(&pvArgs[2], Row);
	CopyToVariant(&pvArgs[1], Column);
	CopyToVariant(&pvArgs[0], Key);

	WORD uErr = InvokeMethodHelper(OLESTR("DefineEvent"), pvArgs, 7, NULL);

	CleanUpStringVariant(&pvArgs[4]);
	CleanUpStringVariant(&pvArgs[3]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for FindField for CReflectionIBM class.
// Description:	Locate a field
// Syntax:	Row, Column, FindType, FindAttribute
//-----------------------------------------------------------------------
WORD CReflectionIBM::FindField(
	LONG Row,
	LONG Column,
	DWORD FindType,
	DWORD FindAttribute,
	LPDWORD lpdwArg4,
	LPDWORD lpdwArg5,
	LPDWORD lpdwArg6,
	LPDWORD lpdwArg7,
	LPDWORD lpdwArg8)

{
	VARIANTARG pvArgs[4];
	InitVariants(pvArgs, 4);

	CopyToVariant(&pvArgs[3], Row);
	CopyToVariant(&pvArgs[2], Column);
	CopyToVariant(&pvArgs[1], FindType);
	CopyToVariant(&pvArgs[0], FindAttribute);

	WORD uErr = InvokeMethodHelper(OLESTR("FindField"), pvArgs, 4, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetBufferedTermKey for CReflectionIBM class.
// Description:	Get key from top of terminal key buffer
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetBufferedTermKey(
	LPDWORD lpdwArg0)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetBufferedTermKey"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[0], lpdwArg0);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetButtonCommands for CReflectionIBM class.
// Description:	Get toolbar button commands
// Syntax:	Index
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetButtonCommands(
	DWORD Index,
	LPSTR lpsArg1,
	LPDWORD lpdwArg2)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], Index);

	WORD uErr = InvokeMethodHelper(OLESTR("GetButtonCommands"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], lpsArg1, lpdwArg2);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetButtonHelpText for CReflectionIBM class.
// Description:	Get toolbar button help text
// Syntax:	Index
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetButtonHelpText(
	DWORD Index,
	LPSTR lpsArg1,
	LPDWORD lpdwArg2)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], Index);

	WORD uErr = InvokeMethodHelper(OLESTR("GetButtonHelpText"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], lpsArg1, lpdwArg2);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetButtonLabel for CReflectionIBM class.
// Description:	Get toolbar button label
// Syntax:	Index
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetButtonLabel(
	DWORD Index,
	LPSTR lpsArg1,
	LPDWORD lpdwArg2)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], Index);

	WORD uErr = InvokeMethodHelper(OLESTR("GetButtonLabel"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], lpsArg1, lpdwArg2);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetButtonPictureFile for CReflectionIBM class.
// Description:	Get toolbar button picture file name
// Syntax:	Index
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetButtonPictureFile(
	DWORD Index,
	LPSTR lpsArg1,
	LPDWORD lpdwArg2)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], Index);

	WORD uErr = InvokeMethodHelper(OLESTR("GetButtonPictureFile"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], lpsArg1, lpdwArg2);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetButtonSeparator for CReflectionIBM class.
// Description:	Determine toolbar button spacing
// Syntax:	Index
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetButtonSeparator(
	DWORD Index,
	LPDWORD lpdwArg1)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], Index);

	WORD uErr = InvokeMethodHelper(OLESTR("GetButtonSeparator"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], lpdwArg1);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetCommandProfile for CReflectionIBM class.
// Description:	Determine if a command is executable
// Syntax:	Command
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetCommandProfile(
	LPSTR Command,
	LPDWORD lpdwArg1)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], Command);

	WORD uErr = InvokeMethodHelper(OLESTR("GetCommandProfile"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], lpdwArg1);
	}

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetCommandRecording for CReflectionIBM class.
// Description:	Determine if a command is recordable
// Syntax:	Command
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetCommandRecording(
	LPSTR Command,
	LPDWORD lpdwArg1)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], Command);

	WORD uErr = InvokeMethodHelper(OLESTR("GetCommandRecording"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], lpdwArg1);
	}

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetCurrentHotspot for CReflectionIBM class.
// Description:	Get the last selected hotspot
// Syntax:	nRow, nColumn, sName, nLength
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetCurrentHotspot(
	LPDWORD nRow,
	LPDWORD nColumn,
	LPSTR sName,
	LPDWORD nLength)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetCurrentHotspot"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[0], sName, nLength);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetCurrentHotspotEx for CReflectionIBM class.
// Description:	Get the last selected hotspot
// Syntax:	nRow, nColumn, sName, nLength, nFlags, nDefRow, nDefColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetCurrentHotspotEx(
	LPDWORD nRow,
	LPDWORD nColumn,
	LPSTR sName,
	LPDWORD nLength,
	LPDWORD nFlags,
	LPDWORD nDefRow,
	LPDWORD nDefColumn)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetCurrentHotspotEx"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[0], sName, nLength);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetFieldAttributes for CReflectionIBM class.
// Description:	Get display attributes for a screen location
// Syntax:	Row, Column
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetFieldAttributes(
	LONG Row,
	LONG Column,
	LPDWORD lpdwArg2)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[1], Row);
	CopyToVariant(&pvArgs[0], Column);

	WORD uErr = InvokeMethodHelper(OLESTR("GetFieldAttributes"), pvArgs, 2, &pvArgs[2]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[2], lpdwArg2);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetFieldColor for CReflectionIBM class.
// Description:	Get the host display color of a field
// Syntax:	Row, Column
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetFieldColor(
	LONG Row,
	LONG Column,
	LPDWORD lpdwArg2)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[1], Row);
	CopyToVariant(&pvArgs[0], Column);

	WORD uErr = InvokeMethodHelper(OLESTR("GetFieldColor"), pvArgs, 2, &pvArgs[2]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[2], lpdwArg2);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotlistItem for CReflectionIBM class.
// Description:	Get the name of the indexed item of a hotlist
// Syntax:	sName, nIndex, sItem, nLength
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotlistItem(
	LPSTR sName,
	DWORD nIndex,
	LPSTR sItem,
	LPDWORD nLength)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[1], sName);
	CopyToVariant(&pvArgs[0], nIndex);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotlistItem"), pvArgs, 2, &pvArgs[2]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[2], sItem, nLength);
	}

	CleanUpStringVariant(&pvArgs[1]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotlistItemCommands for CReflectionIBM class.
// Description:	Get the command associated with a hotlist item
// Syntax:	sList, sItem, sCommands, nLength
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotlistItemCommands(
	LPSTR sList,
	LPSTR sItem,
	LPSTR sCommands,
	LPDWORD nLength)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[1], sList);
	CopyToVariant(&pvArgs[0], sItem);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotlistItemCommands"), pvArgs, 2, &pvArgs[2]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[2], sCommands, nLength);
	}

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotlistItemOptions for CReflectionIBM class.
// Description:	Get the options for a item in a hotlist
// Syntax:	sList, sItem, nOptions
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotlistItemOptions(
	LPSTR sList,
	LPSTR sItem,
	LPDWORD nOptions)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[1], sList);
	CopyToVariant(&pvArgs[0], sItem);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotlistItemOptions"), pvArgs, 2, &pvArgs[2]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[2], nOptions);
	}

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotlistName for CReflectionIBM class.
// Description:	Get the name of a hotlist
// Syntax:	nIndex, sName, nLength
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotlistName(
	DWORD nIndex,
	LPSTR sName,
	LPDWORD nLength)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], nIndex);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotlistName"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], sName, nLength);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotlistOptions for CReflectionIBM class.
// Description:	Get the options set by the rcNewHotlist command
// Syntax:	sName, nOptions
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotlistOptions(
	LPSTR sName,
	LPDWORD nOptions)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], sName);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotlistOptions"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], nOptions);
	}

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotlistShowPosition for CReflectionIBM class.
// Description:	Get the most recent position for the specified hotlist
// Syntax:	sName, nRow, nColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotlistShowPosition(
	LPSTR sName,
	LPDWORD nRow,
	LPDWORD nColumn,
	DWORD dwArg3,
	DWORD dwArg4)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[2], sName);
	CopyToVariant(&pvArgs[1], nRow);
	CopyToVariant(&pvArgs[0], nColumn);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotlistShowPosition"), pvArgs, 3, NULL);

	CleanUpStringVariant(&pvArgs[2]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotspotCommands for CReflectionIBM class.
// Description:	Get the commands activated by a hotspot
// Syntax:	Name,Commands,Size
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotspotCommands(
	LPSTR Name,
	LPSTR Commands,
	LPDWORD Size)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], Name);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotspotCommands"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], Commands, Size);
	}

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotspotCommandsEx for CReflectionIBM class.
// Description:	Get the commands activated by a hotspot
// Syntax:	Name,Flags,Row,Column,Commands,Size
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotspotCommandsEx(
	LPSTR Name,
	DWORD Flags,
	DWORD Row,
	DWORD Column,
	LPSTR Commands,
	LPDWORD Size)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[3], Name);
	CopyToVariant(&pvArgs[2], Flags);
	CopyToVariant(&pvArgs[1], Row);
	CopyToVariant(&pvArgs[0], Column);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotspotCommandsEx"), pvArgs, 4, &pvArgs[4]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[4], Commands, Size);
	}

	CleanUpStringVariant(&pvArgs[3]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotspotHelp for CReflectionIBM class.
// Description:	Get hotspot's Help message
// Syntax:	Name,Flags,Row,Column,Commands,Size
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotspotHelp(
	LPSTR Name,
	DWORD Flags,
	DWORD Row,
	DWORD Column,
	LPSTR Commands,
	LPDWORD Size)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[3], Name);
	CopyToVariant(&pvArgs[2], Flags);
	CopyToVariant(&pvArgs[1], Row);
	CopyToVariant(&pvArgs[0], Column);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotspotHelp"), pvArgs, 4, &pvArgs[4]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[4], Commands, Size);
	}

	CleanUpStringVariant(&pvArgs[3]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotspots for CReflectionIBM class.
// Description:	Enumerate the hotspots currently displayed
// Syntax:	nIndex, nRow, nColumn, sDescription, nLength
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotspots(
	DWORD nIndex,
	LPDWORD nRow,
	LPDWORD nColumn,
	LPSTR sDescription,
	LPDWORD nLength)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], nIndex);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotspots"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], sDescription, nLength);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotspotsEx for CReflectionIBM class.
// Description:	Enumerate the hotspots currently displayed
// Syntax:	nIndex, nRow, nColumn, sDescription, nLength, nFlags, nDefRow, nDefColumn
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotspotsEx(
	DWORD nIndex,
	LPDWORD nRow,
	LPDWORD nColumn,
	LPSTR sDescription,
	LPDWORD nLength,
	LPDWORD nFlags,
	LPDWORD nDefRow,
	LPDWORD nDefColumn)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], nIndex);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotspotsEx"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], sDescription, nLength);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetHotspotTip for CReflectionIBM class.
// Description:	Get hotspot's Tooltip
// Syntax:	Name,Flags,Row,Column,Commands,Size
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetHotspotTip(
	LPSTR Name,
	DWORD Flags,
	DWORD Row,
	DWORD Column,
	LPSTR Commands,
	LPDWORD Size)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[3], Name);
	CopyToVariant(&pvArgs[2], Flags);
	CopyToVariant(&pvArgs[1], Row);
	CopyToVariant(&pvArgs[0], Column);

	WORD uErr = InvokeMethodHelper(OLESTR("GetHotspotTip"), pvArgs, 4, &pvArgs[4]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[4], Commands, Size);
	}

	CleanUpStringVariant(&pvArgs[3]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetKeyMap for CReflectionIBM class.
// Description:	Get commands for a keystroke or mouse chord
// Syntax:	Modifiers, KeyOrChord
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetKeyMap(
	DWORD Modifiers,
	LPSTR KeyOrChord,
	LPSTR lpsArg2,
	LPDWORD lpdwArg3)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[1], Modifiers);
	CopyToVariant(&pvArgs[0], KeyOrChord);

	WORD uErr = InvokeMethodHelper(OLESTR("GetKeyMap"), pvArgs, 2, &pvArgs[2]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[2], lpsArg2, lpdwArg3);
	}

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetLastActiveInstance for CReflectionIBM class.
// Description:	Get window ID of last active Reflection window
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetLastActiveInstance(
	LPDWORD lpdwArg0)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetLastActiveInstance"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[0], lpdwArg0);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetLoginProperty for CReflectionIBM class.
// Description:	Return a single-signon property
// Syntax:	Property, HostUsername, Hostname, Options
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetLoginProperty(
	LPSTR Property,
	LPSTR HostUsername,
	LPSTR Hostname,
	LPSTR Options,
	LPSTR lpsArg4,
	LPDWORD lpdwArg5)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[3], Property);
	CopyToVariant(&pvArgs[2], HostUsername);
	CopyToVariant(&pvArgs[1], Hostname);
	CopyToVariant(&pvArgs[0], Options);

	WORD uErr = InvokeMethodHelper(OLESTR("GetLoginProperty"), pvArgs, 4, &pvArgs[4]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[4], lpsArg4, lpdwArg5);
	}

	CleanUpStringVariant(&pvArgs[3]);
	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetMouseMap for CReflectionIBM class.
// Description:	Get commands for a mouse chord
// Syntax:	Modifiers, Chord
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetMouseMap(
	DWORD Modifiers,
	LPSTR Chord,
	LPSTR lpsArg2,
	LPDWORD lpdwArg3)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[1], Modifiers);
	CopyToVariant(&pvArgs[0], Chord);

	WORD uErr = InvokeMethodHelper(OLESTR("GetMouseMap"), pvArgs, 2, &pvArgs[2]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[2], lpsArg2, lpdwArg3);
	}

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetMyWindow for CReflectionIBM class.
// Description:	Get window ID of current Reflection window
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetMyWindow(
	LPDWORD lpdwArg0)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetMyWindow"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[0], lpdwArg0);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetNumButtons for CReflectionIBM class.
// Description:	Get the number of toolbar buttons
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetNumButtons(
	LPDWORD lpdwArg0)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetNumButtons"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[0], lpdwArg0);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetNumeric for CReflectionIBM class.
// Description:	Get the value of a numeric setting
// Syntax:	NumericSetting
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetNumeric(
	DWORD NumericSetting,
	LPDWORD lpdwArg1)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], NumericSetting);

	WORD uErr = InvokeMethodHelper(OLESTR("GetNumeric"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], lpdwArg1);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetNumHotlists for CReflectionIBM class.
// Description:	Get the number of defined hotlists
// Syntax:	nNumberLists
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetNumHotlists(
	LPDWORD nNumberLists)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetNumHotlists"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[0], nNumberLists);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetNumHotspots for CReflectionIBM class.
// Description:	Get number of hotspots currently displayed on screen
// Syntax:	nHotspots
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetNumHotspots(
	LPDWORD nHotspots)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetNumHotspots"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[0], nHotspots);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetNumItems for CReflectionIBM class.
// Description:	Get number of items in a specified hotlist
// Syntax:	sList, nItems
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetNumItems(
	LPSTR sList,
	LPDWORD nItems)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], sList);

	WORD uErr = InvokeMethodHelper(OLESTR("GetNumItems"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], nItems);
	}

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetNumToolbars for CReflectionIBM class.
// Description:	Get the number of toolbars
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetNumToolbars(
	LPDWORD lpdwArg0)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetNumToolbars"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[0], lpdwArg0);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetPassword for CReflectionIBM class.
// Description:	Return a password from single-signon, or prompt
// Syntax:	Prompt, HostUsername, Hostname, Options
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetPassword(
	LPSTR Prompt,
	LPSTR HostUsername,
	LPSTR Hostname,
	LPSTR Options,
	LPSTR lpsArg4,
	LPDWORD lpdwArg5)

{
	VARIANTARG pvArgs[5];
	InitVariants(pvArgs, 5);

	CopyToVariant(&pvArgs[3], Prompt);
	CopyToVariant(&pvArgs[2], HostUsername);
	CopyToVariant(&pvArgs[1], Hostname);
	CopyToVariant(&pvArgs[0], Options);

	WORD uErr = InvokeMethodHelper(OLESTR("GetPassword"), pvArgs, 4, &pvArgs[4]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[4], lpsArg4, lpdwArg5);
	}

	CleanUpStringVariant(&pvArgs[3]);
	CleanUpStringVariant(&pvArgs[2]);
	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetScanCode for CReflectionIBM class.
// Description:	Get keystroke mapping of a scan code
// Syntax:	ScanType, ScanCode
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetScanCode(
	DWORD ScanType,
	DWORD ScanCode,
	LPSTR lpsArg2,
	LPDWORD lpdwArg3)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[1], ScanType);
	CopyToVariant(&pvArgs[0], ScanCode);

	WORD uErr = InvokeMethodHelper(OLESTR("GetScanCode"), pvArgs, 2, &pvArgs[2]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[2], lpsArg2, lpdwArg3);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetSetProfile for CReflectionIBM class.
// Description:	Determine if a numeric setting can be modified
// Syntax:	SettingName
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetSetProfile(
	DWORD SettingName,
	LPDWORD lpdwArg1)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], SettingName);

	WORD uErr = InvokeMethodHelper(OLESTR("GetSetProfile"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], lpdwArg1);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetSetStringProfile for CReflectionIBM class.
// Description:	Determine if a string setting can be modified
// Syntax:	SettingName
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetSetStringProfile(
	DWORD SettingName,
	LPDWORD lpdwArg1)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], SettingName);

	WORD uErr = InvokeMethodHelper(OLESTR("GetSetStringProfile"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], lpdwArg1);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetSQLStatement for CReflectionIBM class.
// Description:	Retrieve the currently defined SQL statement
// Syntax:	nType
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetSQLStatement(
	DWORD nType,
	LPSTR lpsArg1,
	LPDWORD lpdwArg2)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], nType);

	WORD uErr = InvokeMethodHelper(OLESTR("GetSQLStatement"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], lpsArg1, lpdwArg2);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetString for CReflectionIBM class.
// Description:	Get the value of a string setting
// Syntax:	StringSetting
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetString(
	DWORD StringSetting,
	LPSTR lpsArg1,
	LPDWORD lpdwArg2)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], StringSetting);

	WORD uErr = InvokeMethodHelper(OLESTR("GetString"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[1], lpsArg1, lpdwArg2);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for GetVersionNumber for CReflectionIBM class.
// Description:	Get the Reflection version number
// Syntax:	
//-----------------------------------------------------------------------
WORD CReflectionIBM::GetVersionNumber(
	LPSTR lpsArg0,
	LPDWORD lpdwArg1)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);


	WORD uErr = InvokeMethodHelper(OLESTR("GetVersionNumber"), NULL, 0, &pvArgs[0]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[0], lpsArg0, lpdwArg1);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for PasswordBox for CReflectionIBM class.
// Description:	Prompt for and return a password
// Syntax:	Prompt, Title
//-----------------------------------------------------------------------
WORD CReflectionIBM::PasswordBox(
	LPSTR Prompt,
	LPSTR Title,
	LPSTR lpsArg2,
	LPDWORD lpdwArg3)

{
	VARIANTARG pvArgs[3];
	InitVariants(pvArgs, 3);

	CopyToVariant(&pvArgs[1], Prompt);
	CopyToVariant(&pvArgs[0], Title);

	WORD uErr = InvokeMethodHelper(OLESTR("PasswordBox"), pvArgs, 2, &pvArgs[2]);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvArgs[2], lpsArg2, lpdwArg3);
	}

	CleanUpStringVariant(&pvArgs[1]);
	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for RemoveEvent for CReflectionIBM class.
// Description:	Delete a defined event
// Syntax:	Event
//-----------------------------------------------------------------------
WORD CReflectionIBM::RemoveEvent(
	DWORD Event)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], Event);

	WORD uErr = InvokeMethodHelper(OLESTR("RemoveEvent"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for ResetEvent for CReflectionIBM class.
// Description:	Reset a defined event
// Syntax:	Event
//-----------------------------------------------------------------------
WORD CReflectionIBM::ResetEvent(
	DWORD Event)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], Event);

	WORD uErr = InvokeMethodHelper(OLESTR("ResetEvent"), pvArgs, 1, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetClipboardText for CReflectionIBM class.
// Description:	Copy a string to the clipboard
// Syntax:	Text
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetClipboardText(
	LPSTR Text)

{
	VARIANTARG pvArgs[1];
	InitVariants(pvArgs, 1);

	CopyToVariant(&pvArgs[0], Text);

	WORD uErr = InvokeMethodHelper(OLESTR("SetClipboardText"), pvArgs, 1, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}
//-----------------------------------------------------------------------
// Member function for SetTermKeyBuffer for CReflectionIBM class.
// Description:	Change the size and type of the Terminal Keyboard Buffer
// Syntax:	Size, Overflow
//-----------------------------------------------------------------------
WORD CReflectionIBM::SetTermKeyBuffer(
	LONG Size,
	LONG Overflow)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], Size);
	CopyToVariant(&pvArgs[0], Overflow);

	WORD uErr = InvokeMethodHelper(OLESTR("SetTermKeyBuffer"), pvArgs, 2, NULL);

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitEvent for CReflectionIBM class.
// Description:	Wait for a defined event
// Syntax:	Timeout
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitEvent(
	LPDWORD Timeout,
	LPSTR lpsArg1)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[0], Timeout);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitEvent"), pvArgs, 1, &pvArgs[1]);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvArgs[1], Timeout);
	}

	return uErr;
}
//-----------------------------------------------------------------------
// Member function for WaitEventNumber for CReflectionIBM class.
// Description:	Wait for a particular defined event
// Syntax:	Event, Timeout
//-----------------------------------------------------------------------
WORD CReflectionIBM::WaitEventNumber(
	DWORD Event,
	LPSTR Timeout)

{
	VARIANTARG pvArgs[2];
	InitVariants(pvArgs, 2);

	CopyToVariant(&pvArgs[1], Event);
	CopyToVariant(&pvArgs[0], Timeout);

	WORD uErr = InvokeMethodHelper(OLESTR("WaitEventNumber"), pvArgs, 2, NULL);

	CleanUpStringVariant(&pvArgs[0]);
	return uErr;
}

//***********************************************************************
//***** Property Implementations*****************************************
//***********************************************************************
	//-----------------------------------------------------------
	// Property: CapsLockState
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CapsLockState(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CapsLockState"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CapsLockState(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CapsLockState"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DDEServerEnable
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DDEServerEnable(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DDEServerEnable"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DDEServerEnable(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DDEServerEnable"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DisableInterrupts
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DisableInterrupts(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DisableInterrupts"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DisableInterrupts(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DisableInterrupts"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DisableXlate
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DisableXlate(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DisableXlate"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DisableXlate(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DisableXlate"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NumLockState
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NumLockState(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NumLockState"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_NumLockState(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("NumLockState"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: MappedNumLockPreservesNumLock
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MappedNumLockPreservesNumLock(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MappedNumLockPreservesNumLock"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_MappedNumLockPreservesNumLock(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("MappedNumLockPreservesNumLock"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: KanaLockState
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KanaLockState(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KanaLockState"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_KanaLockState(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("KanaLockState"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: QueryExit
	//-----------------------------------------------------------
WORD CReflectionIBM::get_QueryExit(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("QueryExit"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_QueryExit(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("QueryExit"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SaveChanges
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SaveChanges(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SaveChanges"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SaveChanges(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SaveChanges"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SaveWindowState
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SaveWindowState(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SaveWindowState"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SaveWindowState(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SaveWindowState"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ScrollLockState
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ScrollLockState(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ScrollLockState"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ScrollLockState(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ScrollLockState"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: UseCtrlZ
	//-----------------------------------------------------------
WORD CReflectionIBM::get_UseCtrlZ(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("UseCtrlZ"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_UseCtrlZ(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("UseCtrlZ"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XlateASCII
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XlateASCII(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XlateASCII"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XlateASCII(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XlateASCII"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DDETimeout
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DDETimeout(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DDETimeout"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DDETimeout(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DDETimeout"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FileSystem
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FileSystem(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FileSystem"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_FileSystem(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("FileSystem"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SpacesPerTab
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SpacesPerTab(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SpacesPerTab"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SpacesPerTab(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SpacesPerTab"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TransferType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TransferType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TransferType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TransferType(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TransferType"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrHostSys
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrHostSys(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrHostSys"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrHostSys(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrHostSys"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndStrucFldSz
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndStrucFldSz(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndStrucFldSz"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndStrucFldSz(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndStrucFldSz"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrTimeout
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrTimeout(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrTimeout"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrTimeout(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrTimeout"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndRecFm
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndRecFm(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndRecFm"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndRecFm(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndRecFm"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndLrecl
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndLrecl(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndLrecl"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndLrecl(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndLrecl"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndBlkSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndBlkSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndBlkSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndBlkSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndBlkSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndAllocUnit
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndAllocUnit(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndAllocUnit"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndAllocUnit(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndAllocUnit"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndAvgBlk
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndAvgBlk(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndAvgBlk"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndAvgBlk(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndAvgBlk"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndPrimAlloc
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndPrimAlloc(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndPrimAlloc"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndPrimAlloc(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndPrimAlloc"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndSecAlloc
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndSecAlloc(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndSecAlloc"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndSecAlloc(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndSecAlloc"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrBinaryCRLF
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrBinaryCRLF(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrBinaryCRLF"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrBinaryCRLF(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrBinaryCRLF"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrASCIICRLF
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrASCIICRLF(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrASCIICRLF"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrASCIICRLF(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrASCIICRLF"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrJISCIICRLF
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrJISCIICRLF(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrJISCIICRLF"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrJISCIICRLF(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrJISCIICRLF"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CurrentTraceRecord
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CurrentTraceRecord(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CurrentTraceRecord"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: MaxDebugFilelength
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MaxDebugFilelength(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MaxDebugFilelength"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TraceFileMaxSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TraceFileMaxSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TraceFileMaxSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TraceFileMaxSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TraceFileMaxSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TraceFileCompressType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TraceFileCompressType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TraceFileCompressType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TraceFileCompressType(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TraceFileCompressType"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TraceBufferSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TraceBufferSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TraceBufferSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TraceBufferSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TraceBufferSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: WindowState
	//-----------------------------------------------------------
WORD CReflectionIBM::get_WindowState(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("WindowState"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_WindowState(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("WindowState"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ToolBarMode
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ToolBarMode(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ToolBarMode"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ToolBarMode(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ToolBarMode"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ToolBarVisible
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ToolBarVisible(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ToolBarVisible"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ToolBarVisible(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ToolBarVisible"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ToolBarTether
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ToolBarTether(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ToolBarTether"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ToolBarTether(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ToolBarTether"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TerminalKbdVisible
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TerminalKbdVisible(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TerminalKbdVisible"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TerminalKbdVisible(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TerminalKbdVisible"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TerminalKbdTether
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TerminalKbdTether(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TerminalKbdTether"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TerminalKbdTether(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TerminalKbdTether"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TerminalKbdTooltips
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TerminalKbdTooltips(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TerminalKbdTooltips"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TerminalKbdTooltips(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TerminalKbdTooltips"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SettingsChanged
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SettingsChanged(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SettingsChanged"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SessionType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SessionType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SessionType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TerminalModel
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TerminalModel(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TerminalModel"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TransportName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TransportName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TransportName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TransportType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TransportType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TransportType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_HostName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("HostName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AssignedDeviceName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AssignedDeviceName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AssignedDeviceName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AssignedEncryptionStrength
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AssignedEncryptionStrength(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AssignedEncryptionStrength"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrHostCmd
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrHostCmd(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrHostCmd"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_XfrHostCmd(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrHostCmd"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrAddParam
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrAddParam(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrAddParam"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_XfrAddParam(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrAddParam"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: RecordToggle
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RecordToggle(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RecordToggle"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RecordToggle(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RecordToggle"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DfltXfrType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DfltXfrType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DfltXfrType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DfltXfrType(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DfltXfrType"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DfltExistsAction
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DfltExistsAction(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DfltExistsAction"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DfltExistsAction(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DfltExistsAction"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ConnectScript
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ConnectScript(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ConnectScript"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_ConnectScript(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("ConnectScript"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: ConnectScriptArguments
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ConnectScriptArguments(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ConnectScriptArguments"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_ConnectScriptArguments(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("ConnectScriptArguments"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostsFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostsFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostsFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_HostsFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("HostsFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TerminalKbdShape
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TerminalKbdShape(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TerminalKbdShape"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TerminalKbdShape(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TerminalKbdShape"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AutoConnect
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AutoConnect(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AutoConnect"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AutoConnect(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AutoConnect"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintCompressFont
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintCompressFont(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintCompressFont"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintCompressFont(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintCompressFont"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDest
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDest(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDest"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_PrintDest(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDest"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: DfltPrintCopies
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DfltPrintCopies(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DfltPrintCopies"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DfltPrintCopies(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DfltPrintCopies"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDefaultFont
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDefaultFont(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDefaultFont"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_PrintDefaultFont(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDefaultFont"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintLineSpacing
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintLineSpacing(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintLineSpacing"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintLineSpacing(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintLineSpacing"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintCase
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintCase(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintCase"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintCase(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintCase"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintCharsPerLine
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintCharsPerLine(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintCharsPerLine"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintCharsPerLine(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintCharsPerLine"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintLinesPerPage
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintLinesPerPage(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintLinesPerPage"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintLinesPerPage(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintLinesPerPage"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintLinesPerInch
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintLinesPerInch(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintLinesPerInch"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintLinesPerInch(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintLinesPerInch"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintHostLUName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintHostLUName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintHostLUName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_PrintHostLUName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintHostLUName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: DeviceName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DeviceName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DeviceName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_DeviceName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("DeviceName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AutoSignon
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AutoSignon(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AutoSignon"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AutoSignon(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AutoSignon"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CursorShape
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CursorShape(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CursorShape"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CursorShape(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CursorShape"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CursorMovementStyle
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CursorMovementStyle(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CursorMovementStyle"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CursorMovementStyle(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CursorMovementStyle"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CursorBlinkRate
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CursorBlinkRate(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CursorBlinkRate"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CursorBlinkRate(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CursorBlinkRate"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: BlinkRate
	//-----------------------------------------------------------
WORD CReflectionIBM::get_BlinkRate(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("BlinkRate"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_BlinkRate(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("BlinkRate"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TypeAhead
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TypeAhead(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TypeAhead"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TypeAhead(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TypeAhead"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: EnableHotSpots
	//-----------------------------------------------------------
WORD CReflectionIBM::get_EnableHotSpots(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("EnableHotSpots"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_EnableHotSpots(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("EnableHotSpots"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShowHotSpots
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShowHotSpots(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShowHotSpots"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ShowHotSpots(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ShowHotSpots"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: QuickText
	//-----------------------------------------------------------
WORD CReflectionIBM::get_QuickText(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("QuickText"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_QuickText(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("QuickText"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: CmdPrompt
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CmdPrompt(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CmdPrompt"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_CmdPrompt(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("CmdPrompt"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: DfltDatasetType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DfltDatasetType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DfltDatasetType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DfltDatasetType(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DfltDatasetType"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PasteLineBreak
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PasteLineBreak(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PasteLineBreak"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PasteLineBreak(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PasteLineBreak"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HLLAPIShortName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HLLAPIShortName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HLLAPIShortName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_HLLAPIShortName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("HLLAPIShortName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: HLLAPILongName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HLLAPILongName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HLLAPILongName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_HLLAPILongName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("HLLAPILongName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: KeysDuringWait
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KeysDuringWait(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KeysDuringWait"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_KeysDuringWait(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("KeysDuringWait"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DragQuickTextOption
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DragQuickTextOption(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DragQuickTextOption"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DragQuickTextOption(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DragQuickTextOption"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FastTextPrinting
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FastTextPrinting(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FastTextPrinting"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_FastTextPrinting(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("FastTextPrinting"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintStatusLine
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintStatusLine(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintStatusLine"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintStatusLine(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintStatusLine"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintMonochrome
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintMonochrome(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintMonochrome"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintMonochrome(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintMonochrome"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintSquareAspectRatio
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintSquareAspectRatio(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintSquareAspectRatio"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintSquareAspectRatio(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintSquareAspectRatio"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintBackground
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintBackground(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintBackground"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintBackground(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintBackground"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintWhat
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintWhat(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintWhat"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintWhat(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintWhat"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: BackgndColor
	//-----------------------------------------------------------
WORD CReflectionIBM::get_BackgndColor(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("BackgndColor"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_BackgndColor(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("BackgndColor"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AutoFormfeed
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AutoFormfeed(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AutoFormfeed"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AutoFormfeed(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AutoFormfeed"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ClosePrinterManually
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ClosePrinterManually(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ClosePrinterManually"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ClosePrinterManually(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ClosePrinterManually"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFitToPage
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFitToPage(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFitToPage"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintFitToPage(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFitToPage"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFileExistsAction
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFileExistsAction(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFileExistsAction"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintFileExistsAction(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFileExistsAction"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFileName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFileName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFileName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_PrintFileName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFileName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFormFeedOnEndOfJob
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFormFeedOnEndOfJob(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFormFeedOnEndOfJob"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintFormFeedOnEndOfJob(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFormFeedOnEndOfJob"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintPaperSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintPaperSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintPaperSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintPaperSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintPaperSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintPaperSource
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintPaperSource(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintPaperSource"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintPaperSource(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintPaperSource"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintToFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintToFile(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintToFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintToFile(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintToFile"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintUsePrinterFontsOnly
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintUsePrinterFontsOnly(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintUsePrinterFontsOnly"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintUsePrinterFontsOnly(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintUsePrinterFontsOnly"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintSuppressBlankPage
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintSuppressBlankPage(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintSuppressBlankPage"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintSuppressBlankPage(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintSuppressBlankPage"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintTranslateEBCDICInTRN
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintTranslateEBCDICInTRN(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintTranslateEBCDICInTRN"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintTranslateEBCDICInTRN(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintTranslateEBCDICInTRN"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: BypassWindowsPrinting
	//-----------------------------------------------------------
WORD CReflectionIBM::get_BypassWindowsPrinting(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("BypassWindowsPrinting"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_BypassWindowsPrinting(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("BypassWindowsPrinting"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: BypassWindowsTransparent
	//-----------------------------------------------------------
WORD CReflectionIBM::get_BypassWindowsTransparent(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("BypassWindowsTransparent"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_BypassWindowsTransparent(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("BypassWindowsTransparent"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostPrintTimeout
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostPrintTimeout(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostPrintTimeout"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HostPrintTimeout(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HostPrintTimeout"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDeviceAscii899
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDeviceAscii899(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDeviceAscii899"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDeviceAscii899(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDeviceAscii899"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDeviceFormsMode
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDeviceFormsMode(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDeviceFormsMode"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDeviceFormsMode(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDeviceFormsMode"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDeviceFontId
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDeviceFontId(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDeviceFontId"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDeviceFontId(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDeviceFontId"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDeviceEnvelopeSrc
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDeviceEnvelopeSrc(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDeviceEnvelopeSrc"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDeviceEnvelopeSrc(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDeviceEnvelopeSrc"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDevicePaperSrc1
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDevicePaperSrc1(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDevicePaperSrc1"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDevicePaperSrc1(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDevicePaperSrc1"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDevicePaperSrc2
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDevicePaperSrc2(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDevicePaperSrc2"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDevicePaperSrc2(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDevicePaperSrc2"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDeviceWscstName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDeviceWscstName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDeviceWscstName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_PrintDeviceWscstName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDeviceWscstName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDeviceWscstLibrary
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDeviceWscstLibrary(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDeviceWscstLibrary"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_PrintDeviceWscstLibrary(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDeviceWscstLibrary"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: KeyClick
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KeyClick(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KeyClick"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_KeyClick(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("KeyClick"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PauseDebugPlayOnEOR
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PauseDebugPlayOnEOR(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PauseDebugPlayOnEOR"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PauseDebugPlayOnEOR(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PauseDebugPlayOnEOR"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NationalCharacterSet
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NationalCharacterSet(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NationalCharacterSet"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_NationalCharacterSet(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("NationalCharacterSet"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CEGC
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CEGC(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CEGC"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CEGC(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CEGC"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetPort
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetPort(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetPort"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetPort(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetPort"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostKbdType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostKbdType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostKbdType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HostKbdType(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HostKbdType"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: LocalKbd
	//-----------------------------------------------------------
WORD CReflectionIBM::get_LocalKbd(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("LocalKbd"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_LocalKbd(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("LocalKbd"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Host3270Kbd
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Host3270Kbd(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Host3270Kbd"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Host3270Kbd(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Host3270Kbd"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Host5250Kbd
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Host5250Kbd(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Host5250Kbd"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Host5250Kbd(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Host5250Kbd"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TerminalKbdMode
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TerminalKbdMode(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TerminalKbdMode"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TerminalKbdMode(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TerminalKbdMode"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HelpDisplayMode
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HelpDisplayMode(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HelpDisplayMode"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HelpDisplayMode(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HelpDisplayMode"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FrameTerminalWindow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FrameTerminalWindow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FrameTerminalWindow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_FrameTerminalWindow(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("FrameTerminalWindow"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrHostmessage
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrHostmessage(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrHostmessage"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_XfrHostmessage(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrHostmessage"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: InsertProtocol
	//-----------------------------------------------------------
WORD CReflectionIBM::get_InsertProtocol(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("InsertProtocol"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_InsertProtocol(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("InsertProtocol"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: InsertArena
	//-----------------------------------------------------------
WORD CReflectionIBM::get_InsertArena(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("InsertArena"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_InsertArena(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("InsertArena"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: EnableQuickText
	//-----------------------------------------------------------
WORD CReflectionIBM::get_EnableQuickText(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("EnableQuickText"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_EnableQuickText(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("EnableQuickText"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ToolbarInsideTerminalKbd
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ToolbarInsideTerminalKbd(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ToolbarInsideTerminalKbd"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ToolbarInsideTerminalKbd(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ToolbarInsideTerminalKbd"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: KbdMapShowHostMappedKeys
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KbdMapShowHostMappedKeys(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KbdMapShowHostMappedKeys"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_KbdMapShowHostMappedKeys(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("KbdMapShowHostMappedKeys"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: KbdMapShowKeyHelp
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KbdMapShowKeyHelp(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KbdMapShowKeyHelp"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_KbdMapShowKeyHelp(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("KbdMapShowKeyHelp"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TerminalKbdFixUpperRight
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TerminalKbdFixUpperRight(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TerminalKbdFixUpperRight"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TerminalKbdFixUpperRight(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TerminalKbdFixUpperRight"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ExtendHotSpots
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ExtendHotSpots(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ExtendHotSpots"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ExtendHotSpots(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ExtendHotSpots"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400UseDescriptionFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400UseDescriptionFile(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400UseDescriptionFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400UseDescriptionFile(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400UseDescriptionFile"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400SaveDescriptionFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400SaveDescriptionFile(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400SaveDescriptionFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400SaveDescriptionFile(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400SaveDescriptionFile"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400SendConversion
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400SendConversion(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400SendConversion"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400SendConversion(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400SendConversion"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400ReceiveConversion
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400ReceiveConversion(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400ReceiveConversion"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400ReceiveConversion(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400ReceiveConversion"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400HostFileType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400HostFileType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400HostFileType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400HostFileType(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400HostFileType"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400Authority
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400Authority(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400Authority"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400Authority(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400Authority"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400RecordLength
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400RecordLength(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400RecordLength"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400RecordLength(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400RecordLength"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400TimeFormat
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400TimeFormat(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400TimeFormat"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400TimeFormat(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400TimeFormat"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400DateFormat
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400DateFormat(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400DateFormat"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400DateFormat(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400DateFormat"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400TimeSeparator
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400TimeSeparator(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400TimeSeparator"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400TimeSeparator(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400TimeSeparator"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400DateSeparator
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400DateSeparator(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400DateSeparator"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400DateSeparator(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400DateSeparator"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400IgnoreDecErrors
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400IgnoreDecErrors(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400IgnoreDecErrors"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400IgnoreDecErrors(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400IgnoreDecErrors"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400ReturnMissing
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400ReturnMissing(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400ReturnMissing"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400ReturnMissing(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400ReturnMissing"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400DecSep
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400DecSep(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400DecSep"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400DecSep(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400DecSep"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400OutputDest
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400OutputDest(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400OutputDest"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400OutputDest(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400OutputDest"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400FileText
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400FileText(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400FileText"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400FileText(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400FileText"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400MemberText
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400MemberText(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400MemberText"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400MemberText(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400MemberText"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400SelectString
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400SelectString(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400SelectString"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400SelectString(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400SelectString"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400WhereString
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400WhereString(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400WhereString"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400WhereString(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400WhereString"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400GroupByString
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400GroupByString(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400GroupByString"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400GroupByString(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400GroupByString"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400OrderByString
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400OrderByString(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400OrderByString"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400OrderByString(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400OrderByString"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400JoinByString
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400JoinByString(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400JoinByString"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400JoinByString(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400JoinByString"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400HavingString
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400HavingString(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400HavingString"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400HavingString(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400HavingString"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400SendDescFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400SendDescFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400SendDescFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400SendDescFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400SendDescFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400ReceiveDescFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400ReceiveDescFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400ReceiveDescFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400ReceiveDescFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400ReceiveDescFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AssignedHostName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AssignedHostName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AssignedHostName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400LocalFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400LocalFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400LocalFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400LocalFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400LocalFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400RemoteFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400RemoteFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400RemoteFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400RemoteFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400RemoteFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrLocalFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrLocalFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrLocalFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_XfrLocalFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrLocalFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrRemoteFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrRemoteFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrRemoteFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_XfrRemoteFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrRemoteFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400SortSeq
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400SortSeq(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400SortSeq"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400SortSeq(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400SortSeq"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: XFR400Objects
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XFR400Objects(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XFR400Objects"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XFR400Objects(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XFR400Objects"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400RefFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400RefFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400RefFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Xfr400RefFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400RefFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: CommandLineEnabled
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CommandLineEnabled(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CommandLineEnabled"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CommandLineEnabled(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CommandLineEnabled"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrStartTimeout
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrStartTimeout(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrStartTimeout"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrStartTimeout(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrStartTimeout"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostcommTimeout
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostcommTimeout(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostcommTimeout"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HostcommTimeout(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HostcommTimeout"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AutoReConnect
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AutoReConnect(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AutoReConnect"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AutoReConnect(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AutoReConnect"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Caption
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Caption(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Caption"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Caption(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Caption"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: IconTitle
	//-----------------------------------------------------------
WORD CReflectionIBM::get_IconTitle(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("IconTitle"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_IconTitle(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("IconTitle"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: CommPCSBufSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CommPCSBufSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CommPCSBufSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CommPCSBufSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CommPCSBufSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: MenuVisible
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MenuVisible(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MenuVisible"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_MenuVisible(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("MenuVisible"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PreserveEntryMode
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PreserveEntryMode(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PreserveEntryMode"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PreserveEntryMode(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PreserveEntryMode"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: RestrictedCursor
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RestrictedCursor(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RestrictedCursor"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RestrictedCursor(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RestrictedCursor"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: VerticalCursorSpeed
	//-----------------------------------------------------------
WORD CReflectionIBM::get_VerticalCursorSpeed(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("VerticalCursorSpeed"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_VerticalCursorSpeed(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("VerticalCursorSpeed"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HorizontalCursorSpeed
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HorizontalCursorSpeed(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HorizontalCursorSpeed"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HorizontalCursorSpeed(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HorizontalCursorSpeed"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TypeOf5250StatusLine
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TypeOf5250StatusLine(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TypeOf5250StatusLine"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TypeOf5250StatusLine(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TypeOf5250StatusLine"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: MessageQueue
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MessageQueue(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MessageQueue"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_MessageQueue(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("MessageQueue"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: MessageLibrary
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MessageLibrary(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MessageLibrary"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_MessageLibrary(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("MessageLibrary"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: RulerCursorType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RulerCursorType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RulerCursorType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RulerCursorType(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RulerCursorType"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: RulerCursorVisible
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RulerCursorVisible(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RulerCursorVisible"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RulerCursorVisible(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RulerCursorVisible"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintCharsPerInch
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintCharsPerInch(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintCharsPerInch"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintCharsPerInch(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintCharsPerInch"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintOutputReduction
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintOutputReduction(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintOutputReduction"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintOutputReduction(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintOutputReduction"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintMarginBottom
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintMarginBottom(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintMarginBottom"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintMarginBottom(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintMarginBottom"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintMarginTop
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintMarginTop(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintMarginTop"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintMarginTop(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintMarginTop"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintMarginLeft
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintMarginLeft(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintMarginLeft"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintMarginLeft(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintMarginLeft"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintMarginRight
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintMarginRight(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintMarginRight"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintMarginRight(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintMarginRight"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400ExistsAction
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400ExistsAction(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400ExistsAction"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400ExistsAction(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400ExistsAction"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ProfileMessage
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ProfileMessage(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ProfileMessage"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_ProfileMessage(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("ProfileMessage"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostNumlockControl
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostNumlockControl(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostNumlockControl"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HostNumlockControl(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HostNumlockControl"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostTypeaheadControl
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostTypeaheadControl(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostTypeaheadControl"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HostTypeaheadControl(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HostTypeaheadControl"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintProportionalCPI
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintProportionalCPI(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintProportionalCPI"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintProportionalCPI(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintProportionalCPI"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PCSIdleInterval
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PCSIdleInterval(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PCSIdleInterval"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PCSIdleInterval(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PCSIdleInterval"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PCSUseLowerMemory
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PCSUseLowerMemory(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PCSUseLowerMemory"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PCSUseLowerMemory(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PCSUseLowerMemory"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HLLAPIStyle
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HLLAPIStyle(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HLLAPIStyle"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HLLAPIStyle(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HLLAPIStyle"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintTransparent
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintTransparent(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintTransparent"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintTransparent(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintTransparent"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintOverrideHostFmt
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintOverrideHostFmt(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintOverrideHostFmt"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintOverrideHostFmt(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintOverrideHostFmt"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintOverrideOrientation
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintOverrideOrientation(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintOverrideOrientation"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintOverrideOrientation(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintOverrideOrientation"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CloseDOSWindowOnExit
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CloseDOSWindowOnExit(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CloseDOSWindowOnExit"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CloseDOSWindowOnExit(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CloseDOSWindowOnExit"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400ShowProgress
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400ShowProgress(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400ShowProgress"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400ShowProgress(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400ShowProgress"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400Charset
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400Charset(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400Charset"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400Charset(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400Charset"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: IgnoreHostPrintReq
	//-----------------------------------------------------------
WORD CReflectionIBM::get_IgnoreHostPrintReq(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("IgnoreHostPrintReq"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_IgnoreHostPrintReq(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("IgnoreHostPrintReq"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AutoFontSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AutoFontSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AutoFontSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AutoFontSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AutoFontSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FontName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FontName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FontName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_FontName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("FontName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: FontStyle
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FontStyle(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FontStyle"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_FontStyle(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("FontStyle"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FontSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FontSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FontSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_FontSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("FontSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: VisibleTop
	//-----------------------------------------------------------
WORD CReflectionIBM::get_VisibleTop(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("VisibleTop"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_VisibleTop(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("VisibleTop"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: VisibleLeft
	//-----------------------------------------------------------
WORD CReflectionIBM::get_VisibleLeft(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("VisibleLeft"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_VisibleLeft(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("VisibleLeft"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: VisibleColumns
	//-----------------------------------------------------------
WORD CReflectionIBM::get_VisibleColumns(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("VisibleColumns"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: VisibleRows
	//-----------------------------------------------------------
WORD CReflectionIBM::get_VisibleRows(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("VisibleRows"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400Transport
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400Transport(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400Transport"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400Transport(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400Transport"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400HostTP
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400HostTP(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400HostTP"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400HostTP(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400HostTP"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DefaultPorts
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DefaultPorts(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DefaultPorts"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DefaultPorts(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DefaultPorts"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrCurrentDirectory
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrCurrentDirectory(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrCurrentDirectory"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_XfrCurrentDirectory(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrCurrentDirectory"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400DeleteTrailingSpaces
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400DeleteTrailingSpaces(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400DeleteTrailingSpaces"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400DeleteTrailingSpaces(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400DeleteTrailingSpaces"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400AssumePositiveNumerics
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400AssumePositiveNumerics(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400AssumePositiveNumerics"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400AssumePositiveNumerics(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400AssumePositiveNumerics"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrEnableTemplates
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrEnableTemplates(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrEnableTemplates"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrEnableTemplates(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrEnableTemplates"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400CompatibleFDFFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400CompatibleFDFFile(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400CompatibleFDFFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400CompatibleFDFFile(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400CompatibleFDFFile"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400CompatibleAsciiFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400CompatibleAsciiFile(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400CompatibleAsciiFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400CompatibleAsciiFile(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400CompatibleAsciiFile"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400ForceHexConversion
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400ForceHexConversion(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400ForceHexConversion"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400ForceHexConversion(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400ForceHexConversion"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrCloseDlgAfterTransfer
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrCloseDlgAfterTransfer(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrCloseDlgAfterTransfer"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrCloseDlgAfterTransfer(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrCloseDlgAfterTransfer"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NWSAANetwork
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NWSAANetwork(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NWSAANetwork"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_NWSAANetwork(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("NWSAANetwork"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: UserName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_UserName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("UserName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_UserName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("UserName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Password
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Password(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Password"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_Password(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("Password"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AssignedServer
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AssignedServer(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AssignedServer"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AssignedService
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AssignedService(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AssignedService"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AssignedLUName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AssignedLUName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AssignedLUName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: MouseShape
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MouseShape(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MouseShape"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_MouseShape(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("MouseShape"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SettingsFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SettingsFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SettingsFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetSysReq
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetSysReq(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetSysReq"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetSysReq(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetSysReq"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetAttn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetAttn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetAttn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetAttn(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetAttn"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetKeepAlive
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetKeepAlive(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetKeepAlive"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetKeepAlive(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetKeepAlive"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetKeepAliveTimeout
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetKeepAliveTimeout(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetKeepAliveTimeout"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetKeepAliveTimeout(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetKeepAliveTimeout"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetKeepAliveAverage
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetKeepAliveAverage(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetKeepAliveAverage"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TN3287ConnectType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TN3287ConnectType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TN3287ConnectType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TN3287ConnectType(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TN3287ConnectType"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetLocation
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetLocation(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetLocation"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TelnetLocation(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetLocation"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetResponseMode
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetResponseMode(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetResponseMode"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetResponseMode(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetResponseMode"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetProtocol
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetProtocol(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetProtocol"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintToComm
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintToComm(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintToComm"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintToComm(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintToComm"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShellDOSFullScreen
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShellDOSFullScreen(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShellDOSFullScreen"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ShellDOSFullScreen(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ShellDOSFullScreen"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: C370Charset
	//-----------------------------------------------------------
WORD CReflectionIBM::get_C370Charset(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("C370Charset"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_C370Charset(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("C370Charset"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Support3270Partitions
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Support3270Partitions(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Support3270Partitions"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Support3270Partitions(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Support3270Partitions"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CharacterSetIndicator
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CharacterSetIndicator(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CharacterSetIndicator"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CharacterSetIndicator(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CharacterSetIndicator"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AlternateScreenRows
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AlternateScreenRows(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AlternateScreenRows"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ReconnectDelay
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ReconnectDelay(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ReconnectDelay"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ReconnectDelay(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ReconnectDelay"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TNAPPNGateway
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TNAPPNGateway(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TNAPPNGateway"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TNAPPNGateway(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TNAPPNGateway"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NumberMouseButtons
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NumberMouseButtons(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NumberMouseButtons"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_NumberMouseButtons(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("NumberMouseButtons"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: EnterKeyRepeat
	//-----------------------------------------------------------
WORD CReflectionIBM::get_EnterKeyRepeat(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("EnterKeyRepeat"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_EnterKeyRepeat(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("EnterKeyRepeat"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: RightControlKeyRepeat
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RightControlKeyRepeat(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RightControlKeyRepeat"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RightControlKeyRepeat(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RightControlKeyRepeat"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Locale
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Locale(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Locale"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Locale(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Locale"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: LanguageOverride
	//-----------------------------------------------------------
WORD CReflectionIBM::get_LanguageOverride(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("LanguageOverride"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_LanguageOverride(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("LanguageOverride"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AsyncTransportEnabled
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AsyncTransportEnabled(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AsyncTransportEnabled"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AsyncTransportEnabled(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AsyncTransportEnabled"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: OLEServerName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_OLEServerName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("OLEServerName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_OLEServerName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("OLEServerName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostNetworkAddress
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostNetworkAddress(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostNetworkAddress"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: LocationName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_LocationName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("LocationName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_LocationName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("LocationName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: SNADomainSuffix
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SNADomainSuffix(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SNADomainSuffix"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_SNADomainSuffix(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("SNADomainSuffix"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: APPNNetID
	//-----------------------------------------------------------
WORD CReflectionIBM::get_APPNNetID(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("APPNNetID"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_APPNNetID(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("APPNNetID"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Visible
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Visible(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Visible"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Visible(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Visible"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Top
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Top(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Top"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Top(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Top"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Left
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Left(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Left"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Left(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Left"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Height
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Height(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Height"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Height(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Height"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Width
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Width(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Width"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Width(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Width"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Fullname
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Fullname(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Fullname"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Name
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Name(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Name"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: Path
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Path(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Path"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintMetricUnits
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintMetricUnits(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintMetricUnits"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintMetricUnits(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintMetricUnits"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintProportionalLPI
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintProportionalLPI(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintProportionalLPI"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintProportionalLPI(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintProportionalLPI"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintTransform
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintTransform(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintTransform"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintTransform(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintTransform"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintMfrType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintMfrType(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintMfrType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_PrintMfrType(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintMfrType"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AllFontCharacterSets
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AllFontCharacterSets(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AllFontCharacterSets"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AllFontCharacterSets(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AllFontCharacterSets"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FontCharacterSet
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FontCharacterSet(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FontCharacterSet"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_FontCharacterSet(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("FontCharacterSet"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DisplayVariableWidthFonts
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DisplayVariableWidthFonts(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DisplayVariableWidthFonts"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DisplayVariableWidthFonts(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DisplayVariableWidthFonts"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: UseOLESyntax
	//-----------------------------------------------------------
WORD CReflectionIBM::get_UseOLESyntax(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("UseOLESyntax"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_UseOLESyntax(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("UseOLESyntax"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: UseTableFormat
	//-----------------------------------------------------------
WORD CReflectionIBM::get_UseTableFormat(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("UseTableFormat"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_UseTableFormat(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("UseTableFormat"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TNAssociation
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TNAssociation(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TNAssociation"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TNAssociation(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TNAssociation"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrSpacesToUnderscores
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrSpacesToUnderscores(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrSpacesToUnderscores"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrSpacesToUnderscores(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrSpacesToUnderscores"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrUnderscoresToSpaces
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrUnderscoresToSpaces(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrUnderscoresToSpaces"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrUnderscoresToSpaces(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrUnderscoresToSpaces"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrReceive8Dot3
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrReceive8Dot3(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrReceive8Dot3"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrReceive8Dot3(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrReceive8Dot3"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr8Dot3FilenameCase
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr8Dot3FilenameCase(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr8Dot3FilenameCase"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr8Dot3FilenameCase(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr8Dot3FilenameCase"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDefaultTabH
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDefaultTabH(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDefaultTabH"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDefaultTabH(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDefaultTabH"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDefaultTabV
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDefaultTabV(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDefaultTabV"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDefaultTabV(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDefaultTabV"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Connected
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Connected(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Connected"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SubstituteDisplayChars
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SubstituteDisplayChars(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SubstituteDisplayChars"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SubstituteDisplayChars(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SubstituteDisplayChars"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DisplayMargins
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DisplayMargins(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DisplayMargins"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DisplayMargins(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DisplayMargins"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: KeyboardErrorReset
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KeyboardErrorReset(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KeyboardErrorReset"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_KeyboardErrorReset(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("KeyboardErrorReset"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AidFieldExit
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AidFieldExit(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AidFieldExit"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AidFieldExit(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AidFieldExit"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: EnableHostAlarm
	//-----------------------------------------------------------
WORD CReflectionIBM::get_EnableHostAlarm(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("EnableHostAlarm"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_EnableHostAlarm(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("EnableHostAlarm"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostCursorBlinkControl
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostCursorBlinkControl(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostCursorBlinkControl"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HostCursorBlinkControl(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HostCursorBlinkControl"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrinterOrientation
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrinterOrientation(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrinterOrientation"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrinterOrientation(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrinterOrientation"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: KeyboardLocked
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KeyboardLocked(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KeyboardLocked"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: KeyboardErrorAlarm
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KeyboardErrorAlarm(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KeyboardErrorAlarm"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_KeyboardErrorAlarm(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("KeyboardErrorAlarm"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400StartRecord
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400StartRecord(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400StartRecord"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400StartRecord(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400StartRecord"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400NumDisplayRecords
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400NumDisplayRecords(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400NumDisplayRecords"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400NumDisplayRecords(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400NumDisplayRecords"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NewWindowOnOpen
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NewWindowOnOpen(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NewWindowOnOpen"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_NewWindowOnOpen(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("NewWindowOnOpen"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NewSessionCurrentSettings
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NewSessionCurrentSettings(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NewSessionCurrentSettings"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_NewSessionCurrentSettings(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("NewSessionCurrentSettings"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShowDISOSSStatusDialog
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShowDISOSSStatusDialog(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShowDISOSSStatusDialog"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ShowDISOSSStatusDialog(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ShowDISOSSStatusDialog"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SNAEngineCfgFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SNAEngineCfgFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SNAEngineCfgFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_SNAEngineCfgFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("SNAEngineCfgFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: StartupWorkingDir
	//-----------------------------------------------------------
WORD CReflectionIBM::get_StartupWorkingDir(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("StartupWorkingDir"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: NativeOS
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NativeOS(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NativeOS"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostControlsOutlineColor
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostControlsOutlineColor(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostControlsOutlineColor"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_HostControlsOutlineColor(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("HostControlsOutlineColor"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ModeName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ModeName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ModeName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_ModeName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("ModeName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: FQRemoteLu
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FQRemoteLu(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FQRemoteLu"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_FQRemoteLu(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("FQRemoteLu"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintEmulateFormFeed
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintEmulateFormFeed(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintEmulateFormFeed"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintEmulateFormFeed(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintEmulateFormFeed"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFlushOnEndOfJob
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFlushOnEndOfJob(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFlushOnEndOfJob"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintFlushOnEndOfJob(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFlushOnEndOfJob"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: IgnoreEUICmdErr
	//-----------------------------------------------------------
WORD CReflectionIBM::get_IgnoreEUICmdErr(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("IgnoreEUICmdErr"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_IgnoreEUICmdErr(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("IgnoreEUICmdErr"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NWSAACheckModel
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NWSAACheckModel(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NWSAACheckModel"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_NWSAACheckModel(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("NWSAACheckModel"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetTerminalID
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetTerminalID(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetTerminalID"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TelnetTerminalID(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetTerminalID"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: FoundTextColumn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FoundTextColumn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FoundTextColumn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FoundFieldStartRow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FoundFieldStartRow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FoundFieldStartRow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FoundFieldStartColumn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FoundFieldStartColumn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FoundFieldStartColumn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FoundFieldEndRow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FoundFieldEndRow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FoundFieldEndRow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FoundFieldEndColumn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FoundFieldEndColumn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FoundFieldEndColumn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FoundFieldLength
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FoundFieldLength(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FoundFieldLength"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: FoundTextRow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_FoundTextRow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("FoundTextRow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CursorRow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CursorRow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CursorRow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CursorColumn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CursorColumn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CursorColumn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DisplayRows
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DisplayRows(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DisplayRows"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DisplayColumns
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DisplayColumns(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DisplayColumns"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: MouseClickRow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MouseClickRow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MouseClickRow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: MouseClickColumn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MouseClickColumn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MouseClickColumn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SelectionStartRow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SelectionStartRow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SelectionStartRow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SelectionStartColumn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SelectionStartColumn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SelectionStartColumn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SelectionEndRow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SelectionEndRow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SelectionEndRow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SelectionEndColumn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SelectionEndColumn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SelectionEndColumn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CurrentHotspotRow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CurrentHotspotRow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CurrentHotspotRow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CurrentHotspotColumn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CurrentHotspotColumn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CurrentHotspotColumn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HotlistShowRow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HotlistShowRow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HotlistShowRow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: HotlistShowColumn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HotlistShowColumn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HotlistShowColumn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShowRecentlyUsedFiles
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShowRecentlyUsedFiles(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShowRecentlyUsedFiles"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ShowRecentlyUsedFiles(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ShowRecentlyUsedFiles"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: RecentlyUsedFiles
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RecentlyUsedFiles(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RecentlyUsedFiles"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RecentlyUsedFiles(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RecentlyUsedFiles"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShowReflectionWindows
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShowReflectionWindows(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShowReflectionWindows"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ShowReflectionWindows(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ShowReflectionWindows"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: APL
	//-----------------------------------------------------------
WORD CReflectionIBM::get_APL(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("APL"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_APL(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("APL"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Recording
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Recording(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Recording"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SingleSignOn
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SingleSignOn(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SingleSignOn"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SingleSignOn(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SingleSignOn"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: GraphicsExtendedColors
	//-----------------------------------------------------------
WORD CReflectionIBM::get_GraphicsExtendedColors(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("GraphicsExtendedColors"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_GraphicsExtendedColors(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("GraphicsExtendedColors"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetUseSocks
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetUseSocks(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetUseSocks"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetUseSocks(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetUseSocks"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetUseReflectionSecurityProxy
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetUseReflectionSecurityProxy(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetUseReflectionSecurityProxy"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetUseReflectionSecurityProxy(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetUseReflectionSecurityProxy"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: RequestAuthentication
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RequestAuthentication(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RequestAuthentication"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RequestAuthentication(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RequestAuthentication"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: KerberosPrincipalRealm
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KerberosPrincipalRealm(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KerberosPrincipalRealm"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_KerberosPrincipalRealm(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("KerberosPrincipalRealm"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: KerberosFlags
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KerberosFlags(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KerberosFlags"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_KerberosFlags(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("KerberosFlags"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: KerberosRenew
	//-----------------------------------------------------------
WORD CReflectionIBM::get_KerberosRenew(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("KerberosRenew"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_KerberosRenew(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("KerberosRenew"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NWSAAUse20Query
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NWSAAUse20Query(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NWSAAUse20Query"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_NWSAAUse20Query(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("NWSAAUse20Query"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NWSAARefreshScreen
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NWSAARefreshScreen(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NWSAARefreshScreen"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_NWSAARefreshScreen(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("NWSAARefreshScreen"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: WordWrap
	//-----------------------------------------------------------
WORD CReflectionIBM::get_WordWrap(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("WordWrap"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_WordWrap(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("WordWrap"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: RectangularSelection
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RectangularSelection(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RectangularSelection"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RectangularSelection(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RectangularSelection"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: RecordPasswords
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RecordPasswords(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RecordPasswords"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RecordPasswords(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RecordPasswords"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: GraphicsCursorX
	//-----------------------------------------------------------
WORD CReflectionIBM::get_GraphicsCursorX(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("GraphicsCursorX"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_GraphicsCursorX(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("GraphicsCursorX"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: GraphicsCursorY
	//-----------------------------------------------------------
WORD CReflectionIBM::get_GraphicsCursorY(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("GraphicsCursorY"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_GraphicsCursorY(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("GraphicsCursorY"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: MacroData
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MacroData(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MacroData"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_MacroData(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("MacroData"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: UseIPV6
	//-----------------------------------------------------------
WORD CReflectionIBM::get_UseIPV6(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("UseIPV6"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_UseIPV6(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("UseIPV6"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: WinsockVersion
	//-----------------------------------------------------------
WORD CReflectionIBM::get_WinsockVersion(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("WinsockVersion"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_WinsockVersion(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("WinsockVersion"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: GenerateDeviceNames
	//-----------------------------------------------------------
WORD CReflectionIBM::get_GenerateDeviceNames(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("GenerateDeviceNames"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_GenerateDeviceNames(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("GenerateDeviceNames"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: WordWrap3270
	//-----------------------------------------------------------
WORD CReflectionIBM::get_WordWrap3270(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("WordWrap3270"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_WordWrap3270(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("WordWrap3270"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: WordWrap3270MinimumFieldLength
	//-----------------------------------------------------------
WORD CReflectionIBM::get_WordWrap3270MinimumFieldLength(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("WordWrap3270MinimumFieldLength"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_WordWrap3270MinimumFieldLength(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("WordWrap3270MinimumFieldLength"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PasteWrapText
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PasteWrapText(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PasteWrapText"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PasteWrapText(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PasteWrapText"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PasteMaskProtectedFields
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PasteMaskProtectedFields(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PasteMaskProtectedFields"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PasteMaskProtectedFields(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PasteMaskProtectedFields"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PasteUsingFieldDelimiters
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PasteUsingFieldDelimiters(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PasteUsingFieldDelimiters"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PasteUsingFieldDelimiters(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PasteUsingFieldDelimiters"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PasteReplaceTabs
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PasteReplaceTabs(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PasteReplaceTabs"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PasteReplaceTabs(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PasteReplaceTabs"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PasteTabReplacement
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PasteTabReplacement(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PasteTabReplacement"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_PasteTabReplacement(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("PasteTabReplacement"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: CutFillCharacter
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CutFillCharacter(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CutFillCharacter"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CutFillCharacter(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CutFillCharacter"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: BlanksBetweenFields
	//-----------------------------------------------------------
WORD CReflectionIBM::get_BlanksBetweenFields(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("BlanksBetweenFields"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_BlanksBetweenFields(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("BlanksBetweenFields"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CopyInputFieldsOnly
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CopyInputFieldsOnly(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CopyInputFieldsOnly"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CopyInputFieldsOnly(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CopyInputFieldsOnly"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: GraphicsEnabled
	//-----------------------------------------------------------
WORD CReflectionIBM::get_GraphicsEnabled(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("GraphicsEnabled"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_GraphicsEnabled(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("GraphicsEnabled"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: GraphicsCursorShape
	//-----------------------------------------------------------
WORD CReflectionIBM::get_GraphicsCursorShape(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("GraphicsCursorShape"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_GraphicsCursorShape(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("GraphicsCursorShape"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: GraphicsCrosshairColor
	//-----------------------------------------------------------
WORD CReflectionIBM::get_GraphicsCrosshairColor(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("GraphicsCrosshairColor"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_GraphicsCrosshairColor(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("GraphicsCrosshairColor"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SerialNumber
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SerialNumber(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SerialNumber"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: VersionString
	//-----------------------------------------------------------
WORD CReflectionIBM::get_VersionString(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("VersionString"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: SelectionMode
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SelectionMode(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SelectionMode"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: MaximumSessions
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MaximumSessions(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MaximumSessions"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: NumberSessions
	//-----------------------------------------------------------
WORD CReflectionIBM::get_NumberSessions(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("NumberSessions"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: StatusBarText
	//-----------------------------------------------------------
WORD CReflectionIBM::get_StatusBarText(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("StatusBarText"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_StatusBarText(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("StatusBarText"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: ProfileName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ProfileName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ProfileName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: StartupSettings
	//-----------------------------------------------------------
WORD CReflectionIBM::get_StartupSettings(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("StartupSettings"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetXSystem
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetXSystem(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetXSystem"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetXSystem(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetXSystem"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: BDTIgnoreScrollLock
	//-----------------------------------------------------------
WORD CReflectionIBM::get_BDTIgnoreScrollLock(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("BDTIgnoreScrollLock"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_BDTIgnoreScrollLock(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("BDTIgnoreScrollLock"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Xfr400NCS
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Xfr400NCS(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Xfr400NCS"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Xfr400NCS(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Xfr400NCS"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CommandLineSwitches
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CommandLineSwitches(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CommandLineSwitches"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: MPTNIPAddress
	//-----------------------------------------------------------
WORD CReflectionIBM::get_MPTNIPAddress(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("MPTNIPAddress"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_MPTNIPAddress(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("MPTNIPAddress"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: CMSFileListCommand
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CMSFileListCommand(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CMSFileListCommand"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_CMSFileListCommand(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("CMSFileListCommand"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TSOMemberListCommand
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TSOMemberListCommand(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TSOMemberListCommand"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TSOMemberListCommand(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TSOMemberListCommand"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TSOFileListCommand
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TSOFileListCommand(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TSOFileListCommand"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TSOFileListCommand(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TSOFileListCommand"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostFileListScript
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostFileListScript(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostFileListScript"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_HostFileListScript(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("HostFileListScript"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: HostFileListMacro
	//-----------------------------------------------------------
WORD CReflectionIBM::get_HostFileListMacro(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("HostFileListMacro"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_HostFileListMacro(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("HostFileListMacro"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrHostReturnCode
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrHostReturnCode(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrHostReturnCode"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShowSOSIChars
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShowSOSIChars(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShowSOSIChars"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ShowSOSIChars(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ShowSOSIChars"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: Preserve3270InsertState
	//-----------------------------------------------------------
WORD CReflectionIBM::get_Preserve3270InsertState(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("Preserve3270InsertState"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_Preserve3270InsertState(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("Preserve3270InsertState"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SupportDBCSFeatures
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SupportDBCSFeatures(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SupportDBCSFeatures"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SupportDBCSFeatures(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SupportDBCSFeatures"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndNOSO
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndNOSO(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndNOSO"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndNOSO(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndNOSO"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrIndSO
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrIndSO(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrIndSO"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrIndSO(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrIndSO"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: XfrApvUser
	//-----------------------------------------------------------
WORD CReflectionIBM::get_XfrApvUser(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("XfrApvUser"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_XfrApvUser(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("XfrApvUser"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AutoIME
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AutoIME(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AutoIME"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AutoIME(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AutoIME"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DBCSUnmappedChar
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DBCSUnmappedChar(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DBCSUnmappedChar"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DBCSUnmappedChar(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DBCSUnmappedChar"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintAutoWordWrap
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintAutoWordWrap(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintAutoWordWrap"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintAutoWordWrap(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintAutoWordWrap"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrinterMPL
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrinterMPL(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrinterMPL"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrinterMPL(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrinterMPL"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrinterMPP
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrinterMPP(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrinterMPP"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrinterMPP(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrinterMPP"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: GraphicsDevice
	//-----------------------------------------------------------
WORD CReflectionIBM::get_GraphicsDevice(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("GraphicsDevice"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_GraphicsDevice(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("GraphicsDevice"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CursorProgressionIndicator
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CursorProgressionIndicator(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CursorProgressionIndicator"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CursorProgressionIndicator(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CursorProgressionIndicator"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: EventDefined
	//-----------------------------------------------------------
WORD CReflectionIBM::get_EventDefined(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("EventDefined"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: IndAutoPositionCursor
	//-----------------------------------------------------------
WORD CReflectionIBM::get_IndAutoPositionCursor(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("IndAutoPositionCursor"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_IndAutoPositionCursor(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("IndAutoPositionCursor"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: CreateShortcut
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CreateShortcut(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CreateShortcut"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CreateShortcut(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CreateShortcut"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShortcutFolder
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShortcutFolder(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShortcutFolder"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_ShortcutFolder(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("ShortcutFolder"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: CopyTrailingSpaces
	//-----------------------------------------------------------
WORD CReflectionIBM::get_CopyTrailingSpaces(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("CopyTrailingSpaces"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_CopyTrailingSpaces(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("CopyTrailingSpaces"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintAutoOrientation
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintAutoOrientation(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintAutoOrientation"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintAutoOrientation(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintAutoOrientation"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: BrowserCloseOption
	//-----------------------------------------------------------
WORD CReflectionIBM::get_BrowserCloseOption(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("BrowserCloseOption"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_BrowserCloseOption(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("BrowserCloseOption"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: WebSecurity
	//-----------------------------------------------------------
WORD CReflectionIBM::get_WebSecurity(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("WebSecurity"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_WebSecurity(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("WebSecurity"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TermKeyBufferOverflow
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TermKeyBufferOverflow(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TermKeyBufferOverflow"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TermKeyBufferSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TermKeyBufferSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TermKeyBufferSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: WorkstationAddress
	//-----------------------------------------------------------
WORD CReflectionIBM::get_WorkstationAddress(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("WorkstationAddress"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_WorkstationAddress(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("WorkstationAddress"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AS400LinkName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AS400LinkName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AS400LinkName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_AS400LinkName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("AS400LinkName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: ReuseMSSNAPoolName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ReuseMSSNAPoolName(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ReuseMSSNAPoolName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ReuseMSSNAPoolName(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ReuseMSSNAPoolName"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: QueryExitAll
	//-----------------------------------------------------------
WORD CReflectionIBM::get_QueryExitAll(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("QueryExitAll"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_QueryExitAll(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("QueryExitAll"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: UseServiceLocation
	//-----------------------------------------------------------
WORD CReflectionIBM::get_UseServiceLocation(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("UseServiceLocation"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_UseServiceLocation(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("UseServiceLocation"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPScope
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPScope(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPScope"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_SLPScope(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPScope"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPScopedServicesOnly
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPScopedServicesOnly(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPScopedServicesOnly"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPScopedServicesOnly(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPScopedServicesOnly"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPDirectoryAgents
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPDirectoryAgents(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPDirectoryAgents"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_SLPDirectoryAgents(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPDirectoryAgents"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPDiscoveryRetryInterval
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPDiscoveryRetryInterval(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPDiscoveryRetryInterval"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPDiscoveryRetryInterval(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPDiscoveryRetryInterval"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPMulticastRetryInterval
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPMulticastRetryInterval(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPMulticastRetryInterval"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPMulticastRetryInterval(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPMulticastRetryInterval"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPDARequestTimeout
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPDARequestTimeout(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPDARequestTimeout"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPDARequestTimeout(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPDARequestTimeout"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPMaxMulticastReplyWait
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPMaxMulticastReplyWait(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPMaxMulticastReplyWait"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPMaxMulticastReplyWait(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPMaxMulticastReplyWait"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPMaxDiscoveryWait
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPMaxDiscoveryWait(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPMaxDiscoveryWait"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPMaxDiscoveryWait(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPMaxDiscoveryWait"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPMaxSARequestWait
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPMaxSARequestWait(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPMaxSARequestWait"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPMaxSARequestWait(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPMaxSARequestWait"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPAgentSearchMethod
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPAgentSearchMethod(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPAgentSearchMethod"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPAgentSearchMethod(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPAgentSearchMethod"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPMulticastRadius
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPMulticastRadius(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPMulticastRadius"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPMulticastRadius(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPMulticastRadius"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPPathMTU
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPPathMTU(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPPathMTU"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPPathMTU(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPPathMTU"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SLPUseLoadBalancing
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SLPUseLoadBalancing(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SLPUseLoadBalancing"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SLPUseLoadBalancing(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SLPUseLoadBalancing"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DelayAfterAID
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DelayAfterAID(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DelayAfterAID"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DelayAfterAID(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DelayAfterAID"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: RecordInitialSyncCmds
	//-----------------------------------------------------------
WORD CReflectionIBM::get_RecordInitialSyncCmds(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("RecordInitialSyncCmds"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_RecordInitialSyncCmds(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("RecordInitialSyncCmds"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetEnvironment
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetEnvironment(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetEnvironment"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TelnetEnvironment(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetEnvironment"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: ClipboardSyntax
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ClipboardSyntax(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ClipboardSyntax"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ClipboardSyntax(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ClipboardSyntax"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ClipboardPrologue
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ClipboardPrologue(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ClipboardPrologue"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ClipboardPrologue(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ClipboardPrologue"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ThaiCompositionMode
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ThaiCompositionMode(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ThaiCompositionMode"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ThaiCompositionMode(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ThaiCompositionMode"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ConnectMacro
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ConnectMacro(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ConnectMacro"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_ConnectMacro(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("ConnectMacro"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: ConnectMacroData
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ConnectMacroData(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ConnectMacroData"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_ConnectMacroData(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("ConnectMacroData"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShowBytesRemaining
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShowBytesRemaining(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShowBytesRemaining"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ShowBytesRemaining(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ShowBytesRemaining"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDBCS2To3
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDBCS2To3(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDBCS2To3"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDBCS2To3(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDBCS2To3"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDBCSSmallerFont
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDBCSSmallerFont(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDBCSSmallerFont"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDBCSSmallerFont(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDBCSSmallerFont"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintSOSIPresentation
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintSOSIPresentation(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintSOSIPresentation"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintSOSIPresentation(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintSOSIPresentation"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDBCSHorzFontSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDBCSHorzFontSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDBCSHorzFontSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDBCSHorzFontSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDBCSHorzFontSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDBCSVertFontSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDBCSVertFontSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDBCSVertFontSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDBCSVertFontSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDBCSVertFontSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintDBCSRotationAngle
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintDBCSRotationAngle(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintDBCSRotationAngle"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintDBCSRotationAngle(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintDBCSRotationAngle"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: InputFieldUnderlines
	//-----------------------------------------------------------
WORD CReflectionIBM::get_InputFieldUnderlines(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("InputFieldUnderlines"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_InputFieldUnderlines(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("InputFieldUnderlines"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: DBCSUserDefinedCharacterSource
	//-----------------------------------------------------------
WORD CReflectionIBM::get_DBCSUserDefinedCharacterSource(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("DBCSUserDefinedCharacterSource"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_DBCSUserDefinedCharacterSource(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("DBCSUserDefinedCharacterSource"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: UserDefinedCharactersViaTelnet
	//-----------------------------------------------------------
WORD CReflectionIBM::get_UserDefinedCharactersViaTelnet(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("UserDefinedCharactersViaTelnet"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_UserDefinedCharactersViaTelnet(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("UserDefinedCharactersViaTelnet"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: IGCFEAT
	//-----------------------------------------------------------
WORD CReflectionIBM::get_IGCFEAT(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("IGCFEAT"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_IGCFEAT(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("IGCFEAT"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFitForm
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFitForm(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFitForm"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintFitForm(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFitForm"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFitFormSize
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFitFormSize(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFitFormSize"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintFitFormSize(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFitFormSize"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFitUser
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFitUser(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFitUser"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintFitUser(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFitUser"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFitUserLength
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFitUserLength(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFitUserLength"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintFitUserLength(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFitUserLength"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintFitUserWidth
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintFitUserWidth(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintFitUserWidth"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintFitUserWidth(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintFitUserWidth"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintXlateANSIToASCII
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintXlateANSIToASCII(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintXlateANSIToASCII"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintXlateANSIToASCII(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintXlateANSIToASCII"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintTransverse
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintTransverse(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintTransverse"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintTransverse(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintTransverse"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShowStatusBar
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShowStatusBar(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShowStatusBar"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ShowStatusBar(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ShowStatusBar"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ShowTitleBar
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ShowTitleBar(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ShowTitleBar"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ShowTitleBar(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ShowTitleBar"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintOverstrike
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintOverstrike(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintOverstrike"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintOverstrike(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintOverstrike"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: PrintSuppressNewlines
	//-----------------------------------------------------------
WORD CReflectionIBM::get_PrintSuppressNewlines(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("PrintSuppressNewlines"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_PrintSuppressNewlines(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("PrintSuppressNewlines"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AS400HostName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AS400HostName(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AS400HostName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_AS400HostName(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("AS400HostName"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AssignedAS400
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AssignedAS400(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AssignedAS400"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

	//-----------------------------------------------------------
	// Property: UserDataDirectory
	//-----------------------------------------------------------
WORD CReflectionIBM::get_UserDataDirectory(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("UserDataDirectory"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_UserDataDirectory(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("UserDataDirectory"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetEncryption
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetEncryption(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetEncryption"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetEncryption(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetEncryption"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetEncryptionStrength
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetEncryptionStrength(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetEncryptionStrength"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetEncryptionStrength(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetEncryptionStrength"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TelnetEncryptionVerifyHostName
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TelnetEncryptionVerifyHostName(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TelnetEncryptionVerifyHostName"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_TelnetEncryptionVerifyHostName(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("TelnetEncryptionVerifyHostName"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: ColumnSeparatorStyle
	//-----------------------------------------------------------
WORD CReflectionIBM::get_ColumnSeparatorStyle(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("ColumnSeparatorStyle"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_ColumnSeparatorStyle(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("ColumnSeparatorStyle"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: WindowsTSEClientKeyboard
	//-----------------------------------------------------------
WORD CReflectionIBM::get_WindowsTSEClientKeyboard(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("WindowsTSEClientKeyboard"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_WindowsTSEClientKeyboard(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("WindowsTSEClientKeyboard"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SettingsFileUpdate
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SettingsFileUpdate(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SettingsFileUpdate"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

	//-----------------------------------------------------------
	// Property: AutoUpdateFile
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AutoUpdateFile(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AutoUpdateFile"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_AutoUpdateFile(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("AutoUpdateFile"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: AutoUpdateSave
	//-----------------------------------------------------------
WORD CReflectionIBM::get_AutoUpdateSave(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("AutoUpdateSave"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_AutoUpdateSave(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("AutoUpdateSave"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: SharedMacrosFolder
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SharedMacrosFolder(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SharedMacrosFolder"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_SharedMacrosFolder(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("SharedMacrosFolder"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: SettingsUpdateType
	//-----------------------------------------------------------
WORD CReflectionIBM::get_SettingsUpdateType(LPDWORD pdwOut)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("SettingsUpdateType"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetNumericReturn(&pvRet, pdwOut);
	}

	return uErr;
}

WORD CReflectionIBM::put_SettingsUpdateType(DWORD dwIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, dwIn);

	WORD uErr = InvokePutHelper(OLESTR("SettingsUpdateType"), &pvArg);

	return uErr;
}

	//-----------------------------------------------------------
	// Property: TransformRWebToXML
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TransformRWebToXML(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TransformRWebToXML"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TransformRWebToXML(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TransformRWebToXML"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TransformXMLtoSettings
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TransformXMLtoSettings(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TransformXMLtoSettings"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TransformXMLtoSettings(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TransformXMLtoSettings"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TransformXMLtoSettingsJ
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TransformXMLtoSettingsJ(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TransformXMLtoSettingsJ"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TransformXMLtoSettingsJ(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TransformXMLtoSettingsJ"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TransformXMLtoRWeb
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TransformXMLtoRWeb(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TransformXMLtoRWeb"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TransformXMLtoRWeb(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TransformXMLtoRWeb"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

	//-----------------------------------------------------------
	// Property: TransformSettingsToHTML
	//-----------------------------------------------------------
WORD CReflectionIBM::get_TransformSettingsToHTML(LPSTR pStrOut, LPDWORD length)
{
	VARIANTARG pvRet;
	VariantInit(&pvRet);

	WORD uErr = InvokeGetHelper(OLESTR("TransformSettingsToHTML"), &pvRet);

	if ( uErr == rcRteNoError ) {
		GetStringReturn(&pvRet, pStrOut, length);
	}
	return uErr;
}

WORD CReflectionIBM::put_TransformSettingsToHTML(LPSTR pStrIn)
{
	VARIANTARG pvArg;
	VariantInit(&pvArg);

	CopyToVariant(&pvArg, pStrIn);

	WORD uErr = InvokePutHelper(OLESTR("TransformSettingsToHTML"), &pvArg);

	CleanUpStringVariant(&pvArg);
	return uErr;
}

