require("common")
require("ControlSumm")

SB9600 = class()

--todo 1 based arrays

local enterSBEP_TEMPL = { 0x00, 0x12, 0x01, 0x06, 0x02 } -- switch to SBEP, P-COL = 01 SBEP, BAUD = 0010 9600, Group = 0, Dev addr = 1
SB9600.SBEP_ACK = 0x50
SB9600.SBEP_NAK = 0x60
SB9600.GOOD_WRITE_REPLY = 0x84 --$F4 $84 addr_msb addr addr_lsb cksum
SB9600.BAD_WRITE_REPLY = 0x85 --$F4 $85 addr_msb addr addr_lsb cksum

SB9600.VHFmin = 103000000
SB9600.VHFmax = 178000000
SB9600.UHFmin = 375000000
SB9600.UHFmax = 520000000
SB9600.MHFmin = 801003000

SB9600.SB9600_PKTLEN = 5

SB9600.SbepResp = 0
SB9600.RecvPkt = {}

SB9600.port = nil
SB9600.SbepMode = false
SB9600.sentOpcode = 0
SB9600.wfStage = {
    waiting = {
		echo = false,
		ack = false,
		data = false,
	},
    done = {
		echo = false,
		ack = false,
		data = false,
	}
}
SB9600.CtsState = false
SB9600.FLogTX = false
SB9600.FLogRX = false
SB9600.FPortName = ""
SB9600.FBaudRate = 9600

SB9600.FSentPkt = {}
SB9600.FReadBuf = {}
SB9600.FReadBufCount = 0

SB9600.TimerWaitingCts = TQueuedTimer(1000, 0, 8)
SB9600.TimerWaitingResp = TQueuedTimer(3000, 0, 8)

local SbepRespOpcodesDescr = {
	[0x11] = "Read_Data_Request",
	[0x12] = "Checksum_Request",
	[0x13] = "Configuration_Request",
	[0x14] = "Status_Request",
	[0x15] = "Erase_Flash_Request",
	[0x16] = "Zero_Flash_Request",
	[0x17] = "Write_Data_Request",
	[0x18] = "Write_Ser_Num_Request",
	[0x19] = "Radio_Key",
	[0x1B] = "Tuning_Parameters_Request",
	[0x1C] = "Button_Test_Request",
	[0x1D] = "RLAP_Request",
	[0x1F] = "Softpot_Autotune_Request",
	[0x20] = "VRIS_Call",
	[0x21] = "Channel_Frequency_Request",
	[0x22] = "Test_Mode_Request",
	[0x23] = "Radio_Status_Request",
	[0x24] = "Keypad_Request",
	[0x25] = "Frac_N_Frequency",
	[0x27] = "Channel_Steering_Request",
	[0x2D] = "Radster_Write_Request",
	[0x2E] = "Radster_Read_Request",
	[0x2F] = "Option_Board_Request",
	[0x44] = "ReadRSSIRequeest",
	[0xD6] = "Rated_Volume_SoftpotRequest",
	[0xD8] = "ReferenceOscillatorRequest",
}

local Sb9600OpcodesDescr = {
	[0x01] = "CHANNEL INFORMATION",
	[0x81] = "CHANNEL INFORMATION",
	[0x03] = "MEMORY ADDRESS/DATA FRAMED",
	[0x83] = "MEMORY ADDRESS/DATA FRAMED",
	[0x06] = "EXPANDED PROTOCOL REQUEST",
	[0x07] = "MEMORY ADDRESS/DATA",
	[0x87] = "MEMORY ADDRESS/DATA",
	[0x08] = "MEMORY ACCESS",
	[0x09] = "SHOW BUTTON",
	[0x0A] = "SET BUTTON",
	[0x0B] = "INCREMENT BUTTON",
	[0x0C] = "DECREMENT BUTTON",
	[0x0D] = "ENTER CONFIGURATION",
	[0x0E] = "EXIT CONFIGURATION",
	[0x0F] = "DELETE BUTTON",
	[0x10] = "RECALL BUTTON",
	[0x11] = "REQUEST BUTTON",
	[0x14] = "BUTTON PRESS",
	[0x15] = "RADIO READY",
	[0x16] = "OPTION STATUS",
	[0x96] = "OPTION STATUS",
	[0x17] = "DEVICE SUBROUTINE JUMP",
	[0x18] = "PTT INHIBIT",
	[0x19] = "RADIO KEYED",
	[0x1A] = "RECEIVE AUDIO ROUTING",
	[0x1B] = "TRANSMIT AUDIO ROUTING",
	[0x1C] = "ALERT TONE",
	[0x1D] = "AUDIO MUTE",
	[0x1E] = "SQUELCH DETECT",
	[0x1F] = "ACTIVE MODE UPDATE",
	[0x20] = "TRANSMIT MODE UPDATE",
	[0xA0] = "TRANSMIT MODE UPDATE",
	[0x21] = "RECEIVE MODE UPDATE",
	[0xA1] = "RECEIVE MODE UPDATE",
	[0x22] = "DISCRIMINATOR MUTE",
	[0x23] = "PL DETECT",
	[0x24] = "REPEAT/DIRECT",
	[0x25] = "SIGNALLING INFORMATION",
	[0xA5] = "SIGNALLING INFORMATION",
	[0x26] = "VOLUME MINIMUM",
	[0x27] = "VOLUME VALUE",
	[0xA7] = "VOLUME VALUE",
	[0x28] = "SQUELCH VALUE",
	[0x29] = "ACTIVE RECEIVE PL CODE",
	[0x2A] = "ACTIVE TRANSMIT PL CODE",
	[0x2B] = "RECEIVE PL MUTE CONTROL INHIBIT",
	[0x2C] = "REVERSE BURST INHIBIT",
	[0x2D] = "SCAN ON/OFF",
	[0x2E] = "TRANSMIT CONTROL",
	[0x2F] = "ACTIVE MODE WRITE",
	[0x30] = "UNQUALIFY SCAN",
	[0x31] = "TALKBACK ON/OFF",
	[0x32] = "DEVIATION VALUE",
	[0x33] = "ACTIVE POWER LEVEL",
	[0x34] = "TRANSMIT LIGHT",
	[0x35] = "EXTENDER ON/OFF",
	[0x36] = "TIME-OUT-TIMER VALUE",
	[0x37] = "ACTIVE NP LIST",
	[0x38] = "ACTIVE NP LIST BLANKING",
	[0x39] = "ACTIVE PRIORITY 1 MODE",
	[0x3A] = "ACTIVE PRIORITY 2 MODE",
	[0x3B] = "POWER UP STATUS",
	[0x3C] = "DISPLAY",
	[0x3D] = "DISPLAY MESSAGE",
	[0x3E] = "BATTERY STATUS",
	[0x3F] = "CHANNEL NUMBER",
	[0x40] = "TEST MODE",
	[0x41] = "ACTIVE DEVIATION ADJUST",
	[0x42] = "PA LIMIT VALUE",
	[0x43] = "OSCILLATOR VALUE",
	[0x44] = "PORT VALUE",
	[0x45] = "TRUNKING OPCODE",
	[0x46] = "TRUNKED MODE UPDATE",
	[0x47] = "ALARMS",
	[0x48] = "POWER CONTROL",
	[0x49] = "ACTIVE STATE UPDATE",
	[0x4A] = "VEHICULAR REPEATER STATUS UP",
	[0x4B] = "DATA OPCODE",
	[0x4C] = "DATA CHANNEL",
	[0x4D] = "SYSTEM STATUS",
	[0x4E] = "TIME UPDATE",
	[0x4F] = "METROCOM INFORMATION",
	[0x50] = "TRANSFER START OF SEQUENCE",
	[0x51] = "TRANSFER BODY OF SEQUENCE",
	[0x52] = "TRANSFER END OF SEQUENCE",
	[0x53] = "TRANSFER SEQUENCE ERROR",
	[0x54] = "PROCESSOR CRYSTAL PULL",
	[0x55] = "DEFINE BUTTON",
	[0x56] = "SOFT POT REGISTER #1",
	[0xE6] = "SOFT POT REGISTER #1",
	[0x57] = "BUTTON CONTROL",
	[0x58] = "ILLUMINATION CONTROL",
	[0x59] = "CONFIGURATION REQUEST",
	[0x5A] = "SPECIAL APPLICATIONS OPCODE",
	[0x5B] = "TEST OPCODE",
}

local SbepRespOpcodesResp = {
	[0x11] = 0x80,
	[0x12] = 0x81,
	[0x13] = 0x82,
	[0x14] = 0x83,
	[0x15] = 0x84,
	[0x16] = 0x84,
	[0x17] = 0x84, -- WRITE_DATA_REQ
	[0x18] = 0x88, -- 135
	[0x19] = 0x8C,
	[0x1B] = 0x8D,
	[0x1C] = 0x8E,
	[0x1D] = 0x8C,
	[0x1F] = 0x8A,
	[0x20] = 0x20, -- 138?
	[0x21] = 0x8C,
	[0x22] = 0x8C,
	[0x23] = 0x8B,
	[0x24] = 0,
	[0x25] = 0x8C,
	[0x27] = 0x8C,
	[0x2D] = 0x8C,
	[0x2E] = 0x2E,
	[0x2F] = 0x8F,
	[0x44] = 0x44,
	[0xD6] = 0x56,
	[0xD8] = 0x56,
}

function SB9600:ZeroWfStages()
    self.wfStage.waiting.ack = false
    self.wfStage.waiting.data = false
    self.wfStage.waiting.echo = false
    self.wfStage.done.ack = false
    self.wfStage.done.data = false
    self.wfStage.done.echo = false
end

function SB9600:GetSbepOpcode(buf, len)
	if (len < 2) then
		return false
	end

	if (bit32.band(buf[1], 0xF0) == 0xF0) then -- plus opcode byte
		return buf[2]
	end
end

function SB9600:GetSbepPktLen(buf, len)
	local b_hi, b_lo
	local pkt_len = 1
	local pkt_data_len = 0
	b_hi = bit32.band(bit32.rshift(buf[1], 4), 0xF)
	if (b_hi == 0xF) then -- plus opcode byte
		pkt_len = pkt_len + 1
	end
	b_lo = bit32.band(buf[1], 0xF)
	if (b_lo ~= 0xF) then
		pkt_len = pkt_len + b_lo
		pkt_data_len = b_lo
	else
		-- length in 2 bytes after hdr and/or opcode
		if (b_hi == 0xF) then -- plus opcode byte
			if (len < 3) then
				return false
			end
			pkt_data_len = bit32.bor(buf[4], bit32.lshift(buf[3], 8))
		else
			if (len < 2) then
				return false
			end
			pkt_data_len = bit32.bor(buf[3], bit32.lshift(buf[2], 8))
		end
		pkt_len = pkt_len + pkt_data_len
		pkt_len = pkt_len + 2
	end
	return true, pkt_len, pkt_data_len
end

function SB9600:GetPktOpcode(buf, len)
	if not(self.SbepMode) then
		if (len > 0) then
			return buf[1]
		else
			return false
		end
	else
		return self:GetSbepOpcode(buf, len)
	end
end

function SB9600:SetSentPkt(buf)
	ClearTable(self.FSentPkt)
	ArrayCopy(buf, self.FSentPkt, #buf)
	self.sentOpcode = self:GetPktOpcode(buf, #buf)
end

function SB9600:SendToMoto(buf)
	local delay, bytePS, op
	
	self.port:Purge()
	self.FReadBufCount = 0
	self:ZeroWfStages()
	self.wfStage.waiting.echo = true
	if (self.SbepMode) then
		op = self:GetPktOpcode(buf, #buf)
		if not(op) then
			return false
		end
		self.wfStage.waiting.ack = true
		self.wfStage.waiting.data = SbepRespOpcodesResp[op] ~= nil
	end
	
	self:SetSentPkt(buf)
	if (self.FLogTX) then
		mydump.dumpPkt(buf, "TX")
	end
	
	bytePS = 1 / (self.FBaudRate / 12);
	delay = 90 + math.floor(1000 * #buf * bytePS); -- near 90ms for 5 bytes
	
	-- wait for Busy inactive in SB9600
	if not(self.SbepMode) and (self.CtsState) then
		self.TimerWaitingCts:StartTimer()
		while (self.CtsState and self.TimerWaitingCts:GetEnabled()) do
			sleep(20)
			ProcessMessages()
		end
		self.TimerWaitingCts:StopTimer()
		if (self.CtsState) then
			print("waiting for low Busy state expired")
			return false
		end
	end
	
	self.port:SetDtrState(true)
	-- if not(self.port:SetTimeouts(BaudRate)) then // not works with overlapped
	-- exit;
	self.port:Write(buf)
	sleep(delay)
	if not(self.SbepMode) then
		self.port:SetDtrState(false)
	end
	
	return true
end

function SB9600:CompareStages()
	--mydump.dump(self.wfStage)
	return ((self.wfStage.waiting.echo == self.wfStage.done.echo)
		and (not(self.SbepMode) or (self.SbepMode and (self.wfStage.waiting.ack == self.wfStage.done.ack)))
		and (self.wfStage.waiting.data == self.wfStage.done.data))
end

function SB9600:WaitForResponse()
	--print(debug.traceback())
	--error('here is error')
	self.TimerWaitingResp:StartTimer()
	while (not(self:CompareStages()) and self.TimerWaitingResp:GetEnabled()) do
		sleep(20)
		ProcessMessages()
	end
	self.TimerWaitingResp:StopTimer()

	return self:CompareStages()
end

function SB9600:SendRepeatAndWait(buf)
	local rpt_count = 0
	local result = false
	--long startTick = DateTime.Now.Ticks, nextTick;
	repeat
		if not(self:SendToMoto(buf)) then
			break
		end
		result = self:WaitForResponse()
		rpt_count = rpt_count + 1
	until (result or (rpt_count >= 3))
	if not(result) then
		self.FReadBufCount = 0
	end
	
	return result
end

function SB9600:LeaveSbepMode()
	print("leaving SBEP mode")
	self.port:SetDtrState(false)
	self.SbepMode = false
end

function SB9600:EnterSbepMode(DevAddr)
	if (self.SbepMode) then
		return true
	end
	
	local switch_pkt = {}
	local result = false

	-- SB 9600
	ArrayCopy(enterSBEP_TEMPL, switch_pkt, #enterSBEP_TEMPL)
	switch_pkt[3] = bit32.band(DevAddr, 0x1F) -- group is 0, 3 high bits
	switch_pkt[5] = ControlSumm.CRC_16(switch_pkt, 4, 0x00FF) -- ks
	self:ZeroWfStages()
	if not(self:SendRepeatAndWait(switch_pkt)) then
		print("unable to enter SBEP mode!")
		return false
	end
	self:ZeroWfStages()
	self.wfStage.waiting.ack = true
	self.SbepMode = true
	self.port:SetDtrState(false)
	sleep(40) -- 10, SB9600 packet time, 5-10 more to wait for ACK
	self.port:SetDtrState(true)
	result = self:WaitForResponse()
	if (result and self.SbepResp == self.SBEP_ACK) then
		print("SBEP mode: success")
	else
		print("SBEP mode: error")
		self:LeaveSbepMode()
	end
	
	return result
end

function SB9600:EchoLengthMatch(count)
	return #self.FSentPkt <= count
end

function SB9600:EchoMatch(p, count)
    if (#self.FSentPkt > count) then
        return false
	end
    for i = 1, #self.FSentPkt do
        if (p[i] ~= self.FSentPkt[i]) then
            return false
		end
    end
    return true
end

function SB9600:MoveReadBuf(len)
	if (self.FReadBufCount >= len) then
		if (self.FReadBufCount > len) then
			ArrayCopyExt(self.FReadBuf, len + 1, self.FReadBuf, 1, self.FReadBufCount - len)
		end
		self.FReadBufCount = self.FReadBufCount - len
		for i = self.FReadBufCount + 1, #self.FReadBuf do
			self.FReadBuf[i] = nil
		end
	else
		self.FReadBufCount = 0
		ClearTable(self.FReadBuf)
	end
end

function SB9600:ExtractPackets()
	local crc
	local op
	local sig
	local continue
	ClearTable(self.RecvPkt)

	while (self.FReadBufCount > 0) do
		continue = false
		--mydump.dump(self.wfStage.waiting)
		
		if (self.wfStage.waiting.echo and not(self.wfStage.done.echo)) then
			if not(self:EchoLengthMatch(self.FReadBufCount)) then
				return sig
			end
			if (self:EchoMatch(self.FReadBuf, self.FReadBufCount)) then
				self:MoveReadBuf(#self.FSentPkt)
				self.wfStage.done.echo = true
				if (self.FReadBufCount == 0) then
					return sig
				end
			else
				self:MoveReadBuf(1)
				continue = true
			end
		end

		if not(continue) and (self.wfStage.waiting.ack and not(self.wfStage.done.ack and self.SbepMode)) then
			--mydump.dumpPkt(self.FReadBuf)
			if (self.FReadBuf[1] == self.SBEP_ACK or self.FReadBuf[1] == self.SBEP_NAK) then
				--print(string.format('set SbepResp %X', self.FReadBuf[1]))
				self.SbepResp = self.FReadBuf[1]
				self:MoveReadBuf(1)
				self.wfStage.done.ack = true
				if (self.FReadBufCount == 0) then
					return sig
				end
			else
				self:MoveReadBuf(1)
				continue = true
			end
		end

		if not(continue) and (self.wfStage.waiting.data and not(self.wfStage.done.data)) then
			if (self.SbepMode) then
				--mydump.dumpPkt(self.FReadBuf, 'before GetSbepPktLen')
				--print('self.FReadBufCount', tostring(self.FReadBufCount))
				local is_success, pkt_len, pkt_data_len = self:GetSbepPktLen(self.FReadBuf, self.FReadBufCount)
				if not(is_success) then
					return sig
				end
				--print('SbepPktLen', pkt_len, pkt_data_len)
				if (self.FReadBufCount < pkt_len) then
					return sig
				elseif (pkt_data_len == 0) then
					ClearTable(self.RecvPkt)
					ArrayCopy(self.FReadBuf, self.RecvPkt, pkt_len)
					self:MoveReadBuf(pkt_len)
					if (self.FReadBufCount == 0) then
						return sig
					end
				else
					crc = ControlSumm.PlainCS(self.FReadBuf, 1, pkt_len)
					--print("PlainCS", string.format("%X", crc))
					if (crc == 0xFF) then
						ClearTable(self.RecvPkt)
						ArrayCopy(self.FReadBuf, self.RecvPkt, pkt_len)
						self.wfStage.done.data = true
						self.FReadBufCount = 0
						-- we have packet, leave
						return sig
					else
						-- something wrong, erase and ask again
						self.FReadBufCount = 0
						return sig
					end
				end
			end
		end

		if not(continue) and (not(self.SbepMode) and not(self.wfStage.waiting.data)) then
			if (self.FReadBuf[1] == 0xFF) then
				self:MoveReadBuf(3); -- some 3 bytes packets
			else
				if (self.FReadBufCount < self.SB9600_PKTLEN) then
					return sig
				else
					crc = ControlSumm.CRC_16(self.FReadBuf, self.SB9600_PKTLEN)
					--print("CRC_16", string.format("%X", crc))
					if (crc == 0) then
						-- setlength(self.RecvPkt, self.SB9600_PKTLEN);
						-- move(self.FReadBuf[1], self.RecvPkt[1], self.SB9600_PKTLEN);
						op = self.FReadBuf[4]
						sig = Sb9600OpcodesDescr[op]
						if sig then
							print(sig)
						end
						self:MoveReadBuf(self.SB9600_PKTLEN)
					else
						-- some crap
						self.FReadBufCount = 0
					end
				end
			end
		end
	end
	
	return sig
end

function SB9600:AnalyzeBuffer()
	local sig = self:ExtractPackets()
	return sig
end

local function OnDataReadEvent(port)
	--print('OnDataReadEvent')
	--mydump.dump(port, 1, 3)
	self = port.parent
	local buf = port:Read()
	
	if buf == nil or #buf == 0 then
		return
	end
	
	local count = #buf

	if (self.FLogRX) then
		mydump.dumpPkt(buf, "RX")
	end

	if (count + self.FReadBufCount > #self.FReadBuf) then
		self.FReadBufCount = 0 -- buf full, reset
	end
	ArrayCopyExt(buf, 1, self.FReadBuf, self.FReadBufCount + 1, count)
	self.FReadBufCount = self.FReadBufCount + count

	local sig = self:AnalyzeBuffer()

	--if (self.FLogRX) then
	--	print(sig)
	--	mydump.dumpPkt(buf, "RX")
	--end
end

local function OnCtsEvent(port)
	self = port.parent
	self.CtsState = port:GetCtsState()
	print('OnCtsEvent CtsState', self.CtsState)
end

function SB9600:OpenPort(portName, baudRate, dataBits, stopBits, parity, logTX, logRX)
	--print("Opening port", portName, baudRate, dataBits, stopBits, parity, logTX, logRX)
	self.FLogTX = logTX
	self.FLogRX = logRX
	
	self:ZeroWfStages()
	
	self.FPortName = portName
	self.FBaudRate = baudRate
	self.SbepMode = false

	local is_success
	
	self.port = self.port or TComPort()
	self.port.parent = self
	is_success = self.port:SetEvent('OnCts', OnCtsEvent)
	if not(is_success) then
		print(self.port:GetError())
		return false
	end
	self.port:SetDtrState(false)
	is_success = self.port:SetEvent('OnRead', OnDataReadEvent)
	if not(is_success) then
		print(self.port:GetError())
		return false
	end
	local is_success = self.port:Open(portName, baudRate, dataBits, stopBits, parity)
	print(portName.." open "..tostring(is_success))
	if not(is_success) then
		print(self.port:GetError())
		return false
	end
	self.CtsState = self.port:GetCtsState()
	print('CtsState', self.CtsState)
	return true
end

function SB9600:ClosePort()
	self.port:Close()
	self.port = nil
	print("port closed")
end