/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013-2015 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: ConnectionManager.c
#
# --------------------------- General Description -----------------------------
# This file contains Connection Manager code.
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/
#include <avr32/io.h>
#include "board.h"
#include "gpio.h"
#include "FreeRTOS.h"
#include "task.h"
#include "ConnectionManager.h"
#include "TaskConfig.h"
#include "test_commands_manager.h"
#include "timers.h"
#include "codeplug_ui.h"
#include "config.h"
#include "codeplug_ui.h"
#include "controls.h"
#include "bluetooth_manager.h"
#include "initialization.h"
#include "TypeDef.h"
#include "osal.h"
#include "xnl.h"
#include "xcmp_config.h"
#include "debug_manager.h"
#include "xcmp_config.h"
#include "version_string.h"
#include "initialization.h"
#include "usb_standard_request.h"

/*******************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR            Date Modified Tracking Number   Description
* YewFoong          08/26/2010    CCMPD01384903     Connection Manager Framework
* YewFoong          09/02/2010    CCMPD01384903     State Machine & Single Pairing&  connection scenario
* YewFoong          09/07/2010    CCMPD01389483     Updated connection_manager_task(), added new functions
*                                                   "SUCC_PAIR_EVENT", "connection_store_pair_data" &
*                                                   "connection_retrieve_pair_data"
* YewFoong          10/04/2010    CCMPD01398520     Updated the statemachine to support Wireless 1.3 Earpiece
*                                                   Only send the connect request to BT manager if this is a
*                                                   newly paired devices
* YewFoong          10/06/2010   CCMPD01399503      Sent message to Control Manager to notify about link lost event
*                                                   before transition to connecting state
* Tan Seng Kai      10/06/2010    CCMPD01399559     RADIO_INTERFACE_MANAGER_TASK,MPP_TASK,NIBBLER_TASK Compile sw
*                                                   added to remove Radio interface and mpp task
* YewFoong          10/06/2010    CCMPD01398513     Updated connection manager task for multiple pairing/
*                                                   Connection scenario handling
*                                                   Send the inter-task message to set the Dongle to Discoverable Mode
* YewFoong          10/18/2010    CCMPD01403711     Updated StateMachine to allow Dongle to connect with SPP first if
*                                                   define SPP_CONNECT_FIRST.
* YewFoong          10/19/2010    CCMPD01404335     Removed the connection_retrieve_pair_data to allow multiple pairing
*                                                   at same pairing session
* YewFoong          10/27/2010    CCMPD01407520     Removed the StateMachine for SPP_CONNECT_FIRST. Added the handling for
*                                                   BT_HS_RING_TIMEOUT scenario.
* YewFoong          10/28/2010    CCMPD01408237     Added in conn mgr task for BT_HS_ACCEPT_CKPD_CMND_REQUEST and
*                                                   BT_STOP_SPP_CONNECT_REQUEST
* YewFoong          10/29/2010    CCMPD01408762     Added the case handling for USER_VALID_PAIRING_REQUEST_ACK, USER_VALID_PAIRING_REQUEST_ACK
*                                                   Renamed case BT_DISCOVERY_AND_PAIRING_REQUEST to USER_LONG_PTT_PRESS_DETECTED_ACK
* Guru Mahesh       11/24/2010    CCMPD01414087     Message added for MPP
* JJ Low            11/30/2010    CCMPD01446399     Add MPP command
* YewFoong          11/03/2010    CCMPD01409995     Added the handling for Device Discovery Type
* YewFoong          11/10/2010    CCMPD01412657     Added the handling for connection fail scenario while the dongle is at connecting state
* YewFoong          11/12/2010    CCMPD01413554     Added the handling for Headset Pairing fail scenario
* YewFoong          11/18/2010    CCMPD01411683     Added the case "BT_QUAL_TEST_MODE" for Bluetooth Qualification Test Mode scenario, and also
*                                                   modified function 'bt_qualification_test_mode', "connection_store_pair_data" and "connection_retrieve_pair_data"
* YewFoong          11/25/2010    CCMPD01421145     Exception handling for handling R1.4 connection when Headset is disconnected
* Tan Seng Kai      12/08/2010    CCMPD01451448     Linkey handle and store in codeplug
* Tan Seng Kai      12/10/2010    CCMPD01452429     Retry for reconnection with HS in connecting state
* YewFoong          12/09/2010    CCMPD01452089     Exception handling for R1.4 connect fail
* YewFoong          12/10/2010    CCMPD01452415     State Transition to IDLE_STATE when Device Inquiry Timer is expired (61.44s)
* Tan Seng Kai      17/12/2010    CCMPD01455488     Remove reconnection request for R1.4
* YewFoong          12/20/2010    CCMPD01455955     FATAL_ERROR_STATE handling
* Tan Seng Kai      12/21/2010    CCMPD01456401     SPP pairing info retriving after pairing
* Mahesh            24-Dec-2010   CCMPD01414087     Message added for get local bluetooth Address.
* Mahesh            06-Jan-2011   CCMPD01460340     Software catch-up to dongle R1.2 version R01.00.00
* Mahesh            07-Jan-2011   CCMPD01460638     Added  initial MPP inter task messages
* Mahesh            13-Jan-2011   CCMPD01462638     Remove compiler switch for enter into Device discovery in stardard pairing.
* Mahesh            14-Jan-2011   CCMPD01462973     (1)Initialize MPP database and structure to allow both audio / non-audio
*                                                      device (HSP & SPP) using local variable.
*                                                   (2)Use "SYSTEM_TIME" as a seed value to generate random number.(Linkkey)
*                                                   (3)added Inter task timer to update link key ( randomizer )
* YewFoong          01/14/2011    CCMPD01462474     Defined new macro for "PAIRING_MODE_TIMER" and "NEW_CONNECTION_REQUEST_TIMER"
* YeWFoong          01/17/2011    CCMPD01462978     Enhanced the statemachine and inter-task messages for MPP Pairing & Standard Pairing
* PeiSee Toh        01/17/2011    CCMPD01462570     Added pairing mode timer and new connection timer
* Mahesh            24-Jan-2011   CCMPD01465693     Added Unpair(Clear paired device history) for Long PTT press on power-up.
* Tan Seng Kai      01/25/2011    CCMPD01466195     Implement Multiple pairing state machine for MPP base pairing
* PeiSee Toh        01/26/2011    CCMPD01466788     Update the codeplug fields naming according to what define in codeplug_ui.h
* Mahesh            27-Jan-2011   CCMPD01467306     Update linkkey each pairing process. bugfix: clear codeplug for long PTT press on power-up.
* Mahesh            01-Feb-2011   CCMPD01468897     RejectPair for same BTAdd.
* Tan Seng Kai      01-02-2011    CCMPD01468916     Support R1.3 and R1.4 (standard pairing)
* PeiSee Toh        02-10-2011    CCMPD01471881     Change the parameter of codeplug field "PAIRING_BT_AUDIO_DEVICE_TYPE"  under function "connection_clear_pair_data"
* Tan Seng Kai      02/11/2011    CCMPD01471859     Implement align to new MSC diagram. SecureWireless_R2.1_R2.0_R2.3_data_connection.pdf
* Mahesh            09-Feb-2011   CCMPD01471861     (1)Change connection manager state to "connected state" , if any one device connected.
*                                                   (2)Sent new messages to control manager for (i)Audio device connected (AUDIO_DEVICE_CONNECTED_STATE_UPDATE)
*                                                     (ii)Audio device disconnected (AUDIO_DEVICE_DISCONNECTED_STATE_UPDATE) and (iii)any device connected (DEVICE_CONNECTED_STATE_UPDATE).
* Tan Seng Kai      02/17/2011    CCMPD01474548     Handing for device with same bluetooth address but different profile
* YewFoong          02/17/2011    CCMPD01474661       To support the pairing with same BT address of device, only add new node to mac list after device get connected
*                                                    Delete the node from mac list if device get disconnected
* Sevuhan.R         04/20/2011    CCMPD01498417      Removed the FATAL_ERROR_STATE_UPDATE message sending to control manager
* PeiSee Toh        05/03/2011      CCMPD01503308      Handling of pairing behavior of 1) BT Device with same BT addr & same BT Profile
*                                                   2) BT Device with different BT addr & same BT Profile
*
* Tan Seng Kai      29/04/2011    CCMPD01503017     Add semaphore to protect shared memory
* Vinayakumar       28/12/2011    CCMPD01599398     Allow MPP pairing always regardless of the connection b/w the host and remote headset device &
                                                    completely block the Wireless PTT from pairing and connection.
* Sean              08/26/2013    CCMPD01806989     MPP second touch disconnect device, PTT to drop connection, Link Lost new handler
* Toh Pei See       16/12/2011    CCMPD01592074     Added the flag to indicate first time connection
* wrn637            17/9/2013     CCMPD01815076     Remove Low power reconnection feature
* Abhishek Trivedi  10/16/2013    CCMPD01827271     XNL-XCMP Host implementation
* wrn637            10/16/2013    CCMPD01826478     Audio detection handling
* WRN637            07/11/2013    CCMPD01825398     MPP Class 1 RFPA handling
* Abhishek Trivedi  11/8/2013     CCMPD01834482     Updates for XNL-XCMP SPP connection
* Abhishek Trivedi  11/11/2013    CCMPD01834144     Change the BT Config XCMP message to send Infinite pairing instead of Immediate pairing
* Abhishek Trivedi  11/19/2013    CCMPD01836292     Add the BT Config Broadcast message to send in case of user initiated disconnects
* WRN637            11/20/2013    CCMPD01838036     Resolve Gateway Reset issue
* WRN637            12/20/2013    CCMPD01833277     WPTT state machine
* WRN637            01/06/2014    CCMPD01849552     WPTT turn to become headset fix
* WRN637            01/07/2014    CCMPD01849957     Volume tone request support
* WRN637            01/27/2014    CCMPD01856321     OneDot TwoDot disconnection
* WRN637            01/22/2014    CCMPD01854392     Move XCMP related handling to HSP_XNL_XCMP_MANAGER_TASK to prevent multiple access to XCMP decoder at the same time    
* Abhishek Trivedi  01/16/2014    CCMPD01853480     Tone requests for WPTT
* Abhishek Trivedi  03/18/2014	  CCMPD01873731	    Resolve 2 device connection issue
* Abhishek Trivedi  03/10/2014    CCMPD01870625     CSR DSP Audio Detect Integration Changes
* Aneeket Patkar    03/26/2014    CCMPD01876782     Initialization Sequence update - Create MPP State Machine after BT initialization
* Aneeket Patkar    03/27/2014    CCMPD01867590     Initialization updates, XCMP PING and tone issue 
* Aneeket Patkar    03/28/2014    CCMPD01877631     Remove Volume ramp feature as per the updated design
* Abhishek Trivedi  04/04/2014    CCMPD01880139     Reset on DSP Command failure
* Abhishek Trivedi  04/11/2014    CCMPD01882874     CSR Version check and Interchip Ping
* Abhishek Trivedi  04/15/2014    CCMPD01883363     Interchip Ping part 2
* Abhishek Trivedi  04/18/2014    CCMPD01885148     System Ping Timeout
* Abhishek Trivedi  04/24/2014    CCMPD01886810     Standard Pairing CSR Version check update
* Abhishek Trivedi  05/28/2014    CCMPD01897875     Fix Klocwork Issues
* Abhishek Trivedi  06/25/2014    CCMPD01906926     Audio Route (Mic Select) Fix
* Abhishek Trivedi  10/22/2014    CCMPD01938895     Audio Switch
* Abhishek Trivedi  06/05/2014    CCMPD01900617     Version Request Support
* Abhishek Trivedi  10/24/2014    CCMPD01941655     Audio Switch Redesign
* Abhishek Trivedi  11/02/2014    CCMPD01943851     Codeplug Read Request for Talk Permit Tone issue
* Abhishek Trivedi  11/11/2014    CCMPD01945832     Edge of Range Support
* Abhishek Trivedi  11/29/2014    CCMPD01957210     WPTT Support for Audio Switch
* Abhishek Trivedi  01/14/2015    CCMPD01958122     GW supports infinte pairing
*--------------------------- End of History Template----------------------------
* *****************************************************************************/

/*==============================================================================
                                      EXTERNAL DEFINITIONS
==============================================================================*/

/*==============================================================================
                                    EXTERN VARIABLES
==============================================================================*/
extern UINT8_T timer_flag;
extern tc_flags_t tc_flags;
extern bool codeplug_initialized = false;
extern vm_cmd_send_t vm_cmd_send_state;
extern cplg_param_t dsp_speaker_gain;
extern cplg_param_t dsp_mic_gain;
extern cplg_param_t dsp_audio_line_delay;
extern cplg_param_t dsp_aud_det_threshold;
extern UINT16_T accy_vol;
extern UINT8_T accy_step_size;

extern UINT16_T hsp_unsupported_xcmp_opcode;
extern UINT16_T spp_unsupported_xcmp_opcode;
extern UINT16_T hsp_invalid_msg_xcmp_opcode;
extern UINT16_T spp_invalid_msg_xcmp_opcode;
extern bool local_PttPress;
extern bool isHSPttPress;
extern bool isWPTTPttPress;
extern bool current_mic_spkr_used;
extern bool wrsm_oor_gw_ptt_pressed;
extern controls_button_state_t gw_ptt_current_state;
extern ver_msg_state_t ver_msg_state;
extern bool null_data_nibble_rxd;
/*==============================================================================
                                      DEFINITIONS
==============================================================================*/

#define MPP_PAIRING_MODE  (pairing_mode == PAIRING_PAIRING_MODE_MPP)   // CCMPD01446399 (JJ 30/11/2010)

/*==============================================================================
                                    GLOBAL VARIABLES
==============================================================================*/
connectionManagerStateType ConnectionManagerState;
connectionManagerStateType ConnectionManagerPrevState;
connectionManagerStateType WirelessPTTState;          /* device specific state */
connectionManagerStateType WirelessPTTPrevState;      /* device specific state */

discoveryType deviceDiscovery;
bool SoftwareReady = false;        /* true: system software initialization success */
bool isBtQualTestMode = false;
bool gw_legacy_behavior = true;
bool gw_never_repair = false;
//Initialization Parameters for the HSP and SPP device connections
xnl_init_parameters_t hsp_xnl_init_params;
xnl_init_parameters_t spp_xnl_init_params;
// Variables to store the HSP and SPP XNL Device IDs
UINT16_T device_1_id = 0;
UINT16_T device_2_id = 0;

UINT8_T version_buff[MAX_VERSION_SIZE];
/*==============================================================================
                               PRIVATE VARIABLES & CONSTANT
==============================================================================*/
static U8 StateTransitionCommand;
static BT_BdAddr localbtaddr;

static btDevInfo hspDev;
static btDevInfo sppDev;
static btDevInfo temp_hspDev ;   // Temporary database used when swapping device
static btDevInfo temp_sppDev;   // Temporary database used when swapping device


static device_status hspstatus;
static device_status sppstatus;

static bool isPairingRequest = false;    /* true: pairing request from user */

static bool ringTimeout = false; /* to indicate the audio link is not yet established */
static bool isHspPairFail = false;
static BT_U8 hspConnTimeoutCount = 0;
static BT_U8 sppConnTimeoutCount = 0;

static cplg_param_t pairing_mode = PAIRING_PAIRING_MODE_MPP;
static cplg_param_t time_connection = 0;
static cplg_param_t cp_parameter_value;
static cplg_param_t onedot_twodot_dc = 0;
cplg_param_t hw_audio_detect = 0;

#ifdef MPP_HOST
init_host_mpp_pkt               MPP_init_host_pkt;
connection_database             MPP_host_connection_db;
#endif

static BT_U16 NewConnectionTimerValue;
U32 mute_delay;
audio_transfer_state_t audio_transfer_state = GW_NOT_CONNECTED_LOCAL_AUDIO;

/*==============================================================================
                                     EXTERNAL FUNCTIONS
==============================================================================*/
extern BT_BOOL CompareBTAddress(BT_U8 *, BT_U8 * );
extern void BT_TimeoutCancel(BT_TimerId timerId );
extern void disable_system_ping_feature(UINT16_T address);
/*==============================================================================
                                      FUNCTION PROTOTYPES
==============================================================================*/
static void StateTransaction(unsigned char msg_cmd, connectionManagerStateType *ConnectionManagerState, unsigned char Target_State);
static void wirelessPTTStateTransaction(unsigned char Target_State);
void connection_store_pair_data(btDevInfo *);
void connection_retrieve_pair_data();
bool bt_qualification_test_mode();
void RunMppStateMachine(void * , BT_U8 msg_id);
void RunStandardStateMachine(void * , BT_U8 msg_id);
void new_headset_connection_request_timer();
void new_wirelessPTT_connection_request_timer();
void init_xnl_dev_params();
void connection_clear_all_pair_data(DEV_PROFILE device);
UINT8_T send_gw_version_info(UINT16_T xnl_devid);
UINT8_T send_gw_product_id(UINT16_T xnl_devid);
void bt_disconnect_reset(void);
void init_mpp_pkts();
#ifdef MPP_HOST
void delete_mac_list();
void init_connection_db();
void connection_mgr_update_link_key();
void calc_random_link_key(bt_link_key *p_bt_link_key);
void connection_mgr_update_BTAddr(U8 *local_btaddr);
void mpp_add_node_to_mac_list(btDevInfo *devinfo);
void mpp_delete_node_from_mac_list( btDevInfo *devinfo);
void reject_pair_handling(host_reject_mpp * mpp_reject_info);
#endif
void fatal_error_reset(void);
void headset_database_status_update(device_status newstatus);
void wirelessPTT_database_status_update(device_status newstatus);
void clear_pairing_database(btDevInfo* dabatase);

/*** Event handling in ANY STATE ***/
void bt_connect_success_ack_hsp_handling(btDevInfo * dev);
void bt_connect_success_ack_spp_handling(btDevInfo * dev);
void bt_pair_success_hsp_handling(host_succ_mpp * dev);
void bt_pair_success_spp_handling(host_succ_mpp * dev);
void bt_disconnected_ack_hsp_handling(btDevInfo * dev);
void bt_disconnected_ack_spp_handling(btDevInfo * dev);


/* This is a timer for dongle to wait the new connection request from newly paired device . */
TimerDescriptorType ConnMgrNewHeadsetConnReqTimer;
#define    NEW_HEADSET_CONNECTION_REQUEST_TIMER(time, message)      TaskCancelTimer(&ConnMgrNewHeadsetConnReqTimer); \
           ConnMgrNewHeadsetConnReqTimer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           ConnMgrNewHeadsetConnReqTimer.TaskToNotify = CONNECTION_MANAGER; \
           ConnMgrNewHeadsetConnReqTimer.MessageID = message; \
           ConnMgrNewHeadsetConnReqTimer.Message = NULL; \
           ConnMgrNewHeadsetConnReqTimer.Next = NULL; \
           ScheduleTimer(&ConnMgrNewHeadsetConnReqTimer); \

#define CANCEL_NEW_HEADSET_CONNECTION_REQUEST_TIMER()          TaskCancelTimer(&ConnMgrNewHeadsetConnReqTimer); \
                                                                

/* This is a timer for dongle to wait the new connection request from newly paired device . */
TimerDescriptorType ConnMgrNewWirelessPTTConnReqTimer;
#define    NEW_WIRELESSPTT_CONNECTION_REQUEST_TIMER(time, message)      TaskCancelTimer(&ConnMgrNewWirelessPTTConnReqTimer); \
           ConnMgrNewWirelessPTTConnReqTimer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           ConnMgrNewWirelessPTTConnReqTimer.TaskToNotify = CONNECTION_MANAGER; \
           ConnMgrNewWirelessPTTConnReqTimer.MessageID = message; \
           ConnMgrNewWirelessPTTConnReqTimer.Message = NULL; \
           ConnMgrNewWirelessPTTConnReqTimer.Next = NULL; \
           ScheduleTimer(&ConnMgrNewWirelessPTTConnReqTimer); \

#define CANCEL_NEW_WIRELESSPTT_CONNECTION_REQUEST_TIMER()          TaskCancelTimer(&ConnMgrNewWirelessPTTConnReqTimer); \

/* This is a timer for dongle stay in reconnection mode. */
TimerDescriptorType ConnMgrConnectingStateTMOTimer;
#define    CONNECTING_STATE_TMO_TIMER(time, message)      TaskCancelTimer(&ConnMgrConnectingStateTMOTimer);\
           ConnMgrConnectingStateTMOTimer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           ConnMgrConnectingStateTMOTimer.TaskToNotify = CONNECTION_MANAGER; \
           ConnMgrConnectingStateTMOTimer.MessageID = message; \
           ConnMgrConnectingStateTMOTimer.Message = NULL; \
           ConnMgrConnectingStateTMOTimer.Next = NULL; \
           ScheduleTimer(&ConnMgrConnectingStateTMOTimer)

#define CANCEL_CONNECTING_STATE_TMO_TIMER()          TaskCancelTimer(&ConnMgrConnectingStateTMOTimer)

/* This is a timer for watch dog that will hit the watch dog when TMO */
TimerDescriptorType ConnMgrWatchdogResetTimer;
#define    WATCHDOG_RESET_TIMER(time, message)      TaskCancelTimer(&ConnMgrWatchdogResetTimer);\
           ConnMgrWatchdogResetTimer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           ConnMgrWatchdogResetTimer.TaskToNotify = CONNECTION_MANAGER; \
           ConnMgrWatchdogResetTimer.MessageID = message; \
           ConnMgrWatchdogResetTimer.Message = NULL; \
           ConnMgrWatchdogResetTimer.Next = NULL; \
           ScheduleTimer(&ConnMgrWatchdogResetTimer)

#define CANCEL_WATCHDOG_RESET_TIMER()          TaskCancelTimer(&ConnMgrWatchdogResetTimer)

TimerDescriptorType spp_dev_disconnect_delay_timer;
#define    SEND_SPP_DISCONNECT_DELAY_TIMER(time, message)      TaskCancelTimer(&spp_dev_disconnect_delay_timer);\
           spp_dev_disconnect_delay_timer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           spp_dev_disconnect_delay_timer.TaskToNotify = CONNECTION_MANAGER; \
           spp_dev_disconnect_delay_timer.MessageID = message; \
           spp_dev_disconnect_delay_timer.Message = NULL; \
           spp_dev_disconnect_delay_timer.Next = NULL; \
           ScheduleTimer(&spp_dev_disconnect_delay_timer)


#define CANCEL_SPP_DISCONNECT_DELAY_TIMER()          TaskCancelTimer(&spp_dev_disconnect_delay_timer)

TimerDescriptorType hsp_dev_disconnect_delay_timer;
#define    SEND_HSP_DISCONNECT_DELAY_TIMER(time, message)      TaskCancelTimer(&hsp_dev_disconnect_delay_timer);\
           hsp_dev_disconnect_delay_timer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           hsp_dev_disconnect_delay_timer.TaskToNotify = CONNECTION_MANAGER; \
           hsp_dev_disconnect_delay_timer.MessageID = message; \
           hsp_dev_disconnect_delay_timer.Message = NULL; \
           hsp_dev_disconnect_delay_timer.Next = NULL; \
           ScheduleTimer(&hsp_dev_disconnect_delay_timer)


#define CANCEL_HSP_DISCONNECT_DELAY_TIMER()          TaskCancelTimer(&hsp_dev_disconnect_delay_timer)

TimerDescriptorType pending_pairing_req_delay_timer;
#define    PENDING_PAIRING_REQ_DELAY_TIMER(time, message)      TaskCancelTimer(&pending_pairing_req_delay_timer);\
           pending_pairing_req_delay_timer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           pending_pairing_req_delay_timer.TaskToNotify = CONNECTION_MANAGER; \
           pending_pairing_req_delay_timer.MessageID = message; \
           pending_pairing_req_delay_timer.Message = NULL; \
           pending_pairing_req_delay_timer.Next = NULL; \
           ScheduleTimer(&pending_pairing_req_delay_timer)

#define CANCEL_PENDING_PAIRING_REQ_DELAY_TIMER()          TaskCancelTimer(&pending_pairing_req_delay_timer)

TimerDescriptorType send_interchip_ping_msg_timer;
#define    SEND_INTERCHIP_PING_MSG_TIMER(time, message)      TaskCancelTimer(&send_interchip_ping_msg_timer);\
           send_interchip_ping_msg_timer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           send_interchip_ping_msg_timer.TaskToNotify = CONNECTION_MANAGER; \
           send_interchip_ping_msg_timer.MessageID = message; \
           send_interchip_ping_msg_timer.Message = NULL; \
           send_interchip_ping_msg_timer.Next = NULL; \
           ScheduleTimer(&send_interchip_ping_msg_timer)

#define CANCEL_SEND_INTERCHIP_PING_MSG_TIMER()          TaskCancelTimer(&send_interchip_ping_msg_timer)

TimerDescriptorType interchip_ping_timer;
#define    INTERCHIP_PING_TIMER(time, message)      TaskCancelTimer(&interchip_ping_timer);\
           interchip_ping_timer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           interchip_ping_timer.TaskToNotify = CONNECTION_MANAGER; \
           interchip_ping_timer.MessageID = message; \
           interchip_ping_timer.Message = NULL; \
           interchip_ping_timer.Next = NULL; \
           ScheduleTimer(&interchip_ping_timer)

#define CANCEL_INTERCHIP_PING_TIMER()          TaskCancelTimer(&interchip_ping_timer)

TimerDescriptorType xnl_msg_retry_timer[NUMBER_OF_XNL_TX_SUPPORTED_MSGS];

/*==============================================================================
                                      FUNCTION DEFINITIONS
==============================================================================*/

/*=============================================================================
	FUNCTION: StateTransaction

	DESCRIPTION: This function will peform a new state transition, and send
        an inter-task message to Control Manager for informing about new state
        changed.

	ARGUMENTS PASSED: unsigned char Command,

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
static void StateTransaction(unsigned char msg_cmd, connectionManagerStateType *ConnectionManagerState, unsigned char Target_State)
{
  ConnectionManagerPrevState = *ConnectionManagerState;
  *ConnectionManagerState = Target_State;
  StateTransitionCommand = msg_cmd;

  #ifdef DISPLAY_STATE_TRANSITION_DEBUG
  switch(*ConnectionManagerState)
  {
    case OFF_STATE:
      send_debug_info((char *) __FUNCTION__, "OFF_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case INIT_STATE:
      send_debug_info((char *) __FUNCTION__, "INIT_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case PAIRING_STANDBY_STATE:
      send_debug_info((char *) __FUNCTION__, "PAIRING_STANDBY_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case PAIRING_STATE:
      send_debug_info((char *) __FUNCTION__, "PAIRING_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BT_CONNECTING_STATE:
      send_debug_info((char *) __FUNCTION__, "BT_CONNECTING_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BT_CONNECTED_STATE:
      send_debug_info((char *) __FUNCTION__, "BT_CONNECTED_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case PAIRED_SUCCESS_STATE:
      send_debug_info((char *) __FUNCTION__, "PAIRED_SUCCESS_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BT_DISCONNECTED_STATE:
      send_debug_info((char *) __FUNCTION__, "BT_DISCONNECTED_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BT_QUAL_TEST_MODE:
      send_debug_info((char *) __FUNCTION__, "BT_QUAL_TEST_MODE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case IDLE_STATE:
      send_debug_info((char *) __FUNCTION__, "IDLE_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case FATAL_ERROR_STATE:
      send_debug_info((char *) __FUNCTION__, "FATAL_ERROR_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BT_DISCONNECTED_RESET_STATE:
      send_debug_info((char *) __FUNCTION__, "BT_DISCONNECTED_RESET_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
    default:
      send_debug_info((char *) __FUNCTION__, "default\0", get_time_stamp(), PRIORITY_LOW);
      break;
  }
  #endif
   
  switch(*ConnectionManagerState)
  {
    case INIT_STATE:
    case PAIRING_STANDBY_STATE:
    case BT_CONNECTING_STATE:
    case BT_CONNECTED_STATE:
    case IDLE_STATE:
    case BT_QUAL_TEST_MODE:
    {
      SEND_MESSAGE_TO_CONTROLS_MANAGER(StateTransitionCommand, NULL, 0);   /* Notify Control Manager to Update the LED Indication */
      break;
    }
  }
}
/* End of StateTransaction()   */


/*=============================================================================
	FUNCTION: wirelessPTTStateTransaction

	DESCRIPTION: This function will peform a new state transition for Wireless PTT device

	ARGUMENTS PASSED: Target_State

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
static void wirelessPTTStateTransaction(unsigned char Target_State)
{
  #ifdef DISPLAY_STATE_TRANSITION_DEBUG
  switch(Target_State)
  {
    case OFF_STATE:
      send_debug_info((char *) __FUNCTION__, "OFF_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case INIT_STATE:
      send_debug_info((char *) __FUNCTION__, "INIT_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case PAIRING_STANDBY_STATE:
      send_debug_info((char *) __FUNCTION__, "PAIRING_STANDBY_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case PAIRING_STATE:
      send_debug_info((char *) __FUNCTION__, "PAIRING_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BT_CONNECTING_STATE:
      send_debug_info((char *) __FUNCTION__, "BT_CONNECTING_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BT_CONNECTED_STATE:
      send_debug_info((char *) __FUNCTION__, "BT_CONNECTED_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case PAIRED_SUCCESS_STATE:
      send_debug_info((char *) __FUNCTION__, "PAIRED_SUCCESS_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BT_DISCONNECTED_STATE:
      send_debug_info((char *) __FUNCTION__, "BT_DISCONNECTED_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BT_QUAL_TEST_MODE:
      send_debug_info((char *) __FUNCTION__, "BT_QUAL_TEST_MODE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case IDLE_STATE:
      send_debug_info((char *) __FUNCTION__, "IDLE_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case FATAL_ERROR_STATE:
      send_debug_info((char *) __FUNCTION__, "FATAL_ERROR_STATE\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    default:
      send_debug_info((char *) __FUNCTION__, "default\0", get_time_stamp(), PRIORITY_LOW);
      break;
  }
  #endif

  WirelessPTTPrevState = WirelessPTTState;
  WirelessPTTState = Target_State;
}
/* End of wirelessPTTStateTransaction()   */


/*=============================================================================
	FUNCTION: SUCC_PAIR_EVENT

	DESCRIPTION: Get a pairing information of the paired success BT device
        from BT manager

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
static void SUCC_PAIR_EVENT (btDevInfo* pairing_msg)
{
  if( capture_time_semaphore( CONNMNG_SHARE_DB, portMAX_DELAY) == SEM_RET_SUCCESS )
  {
    if(((btDevInfo *)pairing_msg)->profile == HSP)
    {
      memcpy( &temp_hspDev, pairing_msg, sizeof(btDevInfo));
      headset_database_status_update(NEW_DEV_CONNECTING);
    }

    else
    {
      memcpy( &temp_sppDev, pairing_msg, sizeof(btDevInfo));
      wirelessPTT_database_status_update(NEW_DEV_CONNECTING);
    }
    release_semaphore( CONNMNG_SHARE_DB );
  }
}

/* End of SUCC_PAIR_EVENT() */
/*=============================================================================
	FUNCTION: check_new_mpp_audio_device

	DESCRIPTION: Check if the device pairing over MPP is a new Audio device

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: host_succ_mpp

	RETURN VALUE: TRUE/FALSE
==============================================================================*/
bool check_new_mpp_audio_device(host_succ_mpp * mpp_ptr2message)
{
  if((mpp_ptr2message)->device_type.device == AUDIO_DEVICE)
  {  
    if( hspstatus == DEV_CONNECTED)
    {
      if(ComparePairedBTAddress(hspDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]) == FALSE)
      {
        headset_database_status_update(SECOND_DEV_PAIRED_SWAP);
        SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, NULL, 0);
        SEND_HSP_DISCONNECT_DELAY_TIMER(50, HSP_DISCONNECT_REQUEST_DELAY_TIMEOUT);        
        SEND_MESSAGE_TO_MPP_MNGR(STOP_PAIR, NULL, MESSAGE_SIZE_ZERO);
        return FALSE;
      }
    }
  }
  return TRUE;
}
/*=============================================================================
  FUNCTION: bt_database_handling

  DESCRIPTION: This function will check for the paired device and decide what is the
               hanling need to be executed

  ARGUMENTS PASSED: none

  REFERENCE ARGUMENTS PASSED: host_succ_mpp

  RETURN VALUE: none
==============================================================================*/
static void bt_database_handling (host_succ_mpp * mpp_ptr2message)
{

  if((mpp_ptr2message)->device_type.device == AUDIO_DEVICE)
  {
    if( hspstatus == DEV_CONNECTED)
    {
      if(ComparePairedBTAddress(hspDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        headset_database_status_update(SECOND_DEV_PAIRED_TOUCH_ACTION);
      }
      else if(ComparePairedBTAddress(sppDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        headset_database_status_update(SPP_SWITCH_TO_HSP);
      }
      else
      {
        headset_database_status_update(SECOND_DEV_PAIRED_SWAP);
      }
    }
    else if( hspstatus == PAIRED_DEV_CONNECTING)
    {
      if(ComparePairedBTAddress(hspDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        headset_database_status_update(NEW_DEV_CONNECTING);
      }
      else if(ComparePairedBTAddress(sppDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        headset_database_status_update(SPP_SWITCH_TO_HSP);
      }
      else
      {
        headset_database_status_update(NEW_DEV_CONNECTING);
      }
    }
    else
    {
      if(ComparePairedBTAddress(sppDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        /* SPP dev plug in earpiece to become HSP device */
        headset_database_status_update(SPP_SWITCH_TO_HSP);
      }
      else
      {
        headset_database_status_update(NEW_DEV_CONNECTING);
      }
    }
  }
  else if((mpp_ptr2message)->device_type.device == NON_AUDIO_DEVICE) 
  {
    if( sppstatus == DEV_CONNECTED)
    {
      if(ComparePairedBTAddress(sppDev.addr, &(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        wirelessPTT_database_status_update(SECOND_DEV_PAIRED_TOUCH_ACTION);
      }
      else if(ComparePairedBTAddress(hspDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        wirelessPTT_database_status_update(BLUETOOTH_DATABASE_ERROR);
      }
      else 
      {
        wirelessPTT_database_status_update(SECOND_DEV_PAIRED_SWAP);
      }
    }
    else if( sppstatus == PAIRED_DEV_CONNECTING)
    {
      if(ComparePairedBTAddress(sppDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        wirelessPTT_database_status_update(NEW_DEV_CONNECTING);
      }
      else if(ComparePairedBTAddress(hspDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        wirelessPTT_database_status_update(HSP_SWITCH_TO_SPP);
      }
      else
      {
        wirelessPTT_database_status_update(NEW_DEV_CONNECTING);
      }
    }
    else
    {
      if(ComparePairedBTAddress(hspDev.addr,&(mpp_ptr2message)->mac_address.bt_mac_byte[0]))
      {
        wirelessPTT_database_status_update(HSP_SWITCH_TO_SPP);
      }
      else
      {
        wirelessPTT_database_status_update(NEW_DEV_CONNECTING);
      }
    }
  }
}

/*=============================================================================
*	FUNCTION: mpp_succ_pair_event
*
*	DESCRIPTION:  Get a pairing information of the paired success BT device
*                 from MPP manager
*
*	ARGUMENTS PASSED: none
*
*	REFERENCE ARGUMENTS PASSED: none
*
*	RETURN VALUE: none
*   CCMPD01414087 (GM_24Nov2010)
*==============================================================================*/
static void mpp_succ_pair_event (host_succ_mpp * mpp_ptr2message)
{

    if( capture_time_semaphore( CONNMNG_SHARE_DB, portMAX_DELAY) == SEM_RET_SUCCESS )
    {
      if ((mpp_ptr2message)->device_type.device == AUDIO_DEVICE)
      {        
        temp_hspDev.profile            = HSP;
        temp_hspDev.pairingType        = SEC_PAIRING_TYPE_BONDING;
        temp_hspDev.keyType            = HCI_COMBINATION_KEY;
        temp_hspDev.IsNibblerDefaultON = true;
        temp_hspDev.IsDeviceClass1     = (mpp_ptr2message)->rfpa_flag.rfpa;

        temp_hspDev.addr[5] = (mpp_ptr2message)->mac_address.bt_mac_byte[0];
        temp_hspDev.addr[4] = (mpp_ptr2message)->mac_address.bt_mac_byte[1];
        temp_hspDev.addr[3] = (mpp_ptr2message)->mac_address.bt_mac_byte[2];
        temp_hspDev.addr[2] = (mpp_ptr2message)->mac_address.bt_mac_byte[3];
        temp_hspDev.addr[1] = (mpp_ptr2message)->mac_address.bt_mac_byte[4];
        temp_hspDev.addr[0] = (mpp_ptr2message)->mac_address.bt_mac_byte[5];

        memcpy( &temp_hspDev.linkkey[0], &(mpp_ptr2message)->link_key.link_key_byte[0] , sizeof(HCI_LinkKey));
      }
      else
      {
        temp_sppDev.profile            = SPP;
        temp_sppDev.pairingType        = SEC_PAIRING_TYPE_BONDING;
        temp_sppDev.keyType            = HCI_COMBINATION_KEY;
        temp_sppDev.IsNibblerDefaultON = true;
        temp_sppDev.IsDeviceClass1     = (mpp_ptr2message)->rfpa_flag.rfpa;

        temp_sppDev.addr[5] = (mpp_ptr2message)->mac_address.bt_mac_byte[0];
        temp_sppDev.addr[4] = (mpp_ptr2message)->mac_address.bt_mac_byte[1];
        temp_sppDev.addr[3] = (mpp_ptr2message)->mac_address.bt_mac_byte[2];
        temp_sppDev.addr[2] = (mpp_ptr2message)->mac_address.bt_mac_byte[3];
        temp_sppDev.addr[1] = (mpp_ptr2message)->mac_address.bt_mac_byte[4];
        temp_sppDev.addr[0] = (mpp_ptr2message)->mac_address.bt_mac_byte[5];

        memcpy( &temp_sppDev.linkkey[0], &(mpp_ptr2message)->link_key.link_key_byte[0] , sizeof(HCI_LinkKey));

      }
      release_semaphore( CONNMNG_SHARE_DB );
    }
}
/*=============================================================================
	FUNCTION: disconnect_xnl_link

	DESCRIPTION: This function calls the xnl disconnect interface 

	ARGUMENTS PASSED: XNL Device address

	RETURN VALUE: none
==============================================================================*/
void disconnect_xnl_link(UINT16_T device_id)
{
  if(device_id != 0)
  {
    disable_system_ping_feature(device_id);
    xnl_device_disconnected(device_id);
    if(device_id == device_1_id)
    {
      device_1_id = 0x00;
    }
    else if (device_id == device_2_id)
    {
      device_2_id = 0x00;
    }
  }
}
/*=============================================================================
	FUNCTION: connection_manager_task

	DESCRIPTION: Task handling BT Connection using MPP

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void connection_manager_task(void *pvParameters)
{
  ConnectionManagerState = INIT_STATE;
  WirelessPTTState = INIT_STATE;
  ConnectionManagerPrevState = OFF_STATE;
  WirelessPTTPrevState = OFF_STATE;

  SEND_MESSAGE_TO_CONNECTION_MNGR(CONN_MGR_INIT, NULL, 0);
  semaphore_init(CONNMNG_SHARE_DB, BINARY, NULL);


  for(;;)
  {
    void *ptr2message;
    unsigned char message_id;
    ms_task_msg_t *msg;

    msg = (ms_task_msg_t *)Get_Msg_w_Time_Out(SUSPEND_INDEFFINATELY);

    if(msg != NULL)
    {
      message_id = msg->ros_msg.radio_if_msg.sub_opcode;
      ptr2message = msg->ros_msg.radio_if_msg.data;
      OS_free(msg);

      if(gw_legacy_behavior == false)
      {
        audio_transfer_state_machine(message_id);
      }
      
    switch (message_id)
    {
    case CONN_MGR_WATCHDOG_RESET:
      /* watch dog set to 3sec */      
      watchdog_reset();

      /* hitting watch dog every 500ms */
      CANCEL_WATCHDOG_RESET_TIMER();
      WATCHDOG_RESET_TIMER(500, CONN_MGR_WATCHDOG_RESET);
      break;
    
    case CONN_MGR_INIT:
      if (codeplugInit() == CODEPLUG_OK)
      {
        /* watch dog set to 3sec */      
        watchdog_enable(3000);

        /* kick start for watch dog reset every 500ms */
        WATCHDOG_RESET_TIMER(500, CONN_MGR_WATCHDOG_RESET);

        /* If BT_QUALIFICATION_TEST_MODE is set to "1", then Dongle will enter to BT Qualification Test Mode */
        if(bt_qualification_test_mode() == CP_SELECT_ON)
        {
          if(SoftwareReady)
          {
            StateTransaction(BT_QUAL_TEST_MODE_UPDATE,&ConnectionManagerState,BT_QUAL_TEST_MODE);
            pairing_mode = PAIRING_PAIRING_MODE_STANDARD;  /* If BQP is on then use standard mode */
            break;
          }
        }
        else
        {
          codeplugGetParameter(PAIRING_METHOD, &pairing_mode, 0); /* read pairing mode from codeplug */
          codeplugGetParameter(TIME_CONNECTION, &time_connection, 0); /* read pairing mode from codeplug */
          codeplugGetParameter(ENABLE_ONEDOT_TWODOT_DISCONNECTION, &onedot_twodot_dc, 0); /* read onedot twodot disconnection fuction */
          codeplugGetParameter(CSR_DSP_SPEAKER_GAIN, &dsp_speaker_gain, 0);
          codeplugGetParameter(CSR_DSP_MIC_GAIN, &dsp_mic_gain, 0);
          codeplugGetParameter(CSR_DSP_AUDIO_LINE_DELAY, &dsp_audio_line_delay, 0);
          codeplugGetParameter(CSR_DSP_AUDIO_DETECT_THRESHOLD, &dsp_aud_det_threshold, 0);
          codeplugGetParameter(ENABLE_HW_AUDIO_DETECT, &hw_audio_detect, 0);
        }
        
        //Initialize device structures for XNL
        init_xnl_dev_params();

        codeplugGetParameter(MUTE_DELAY_TIME, (cplg_param_t*) &mute_delay, 0);
      }
      else
      {
         SEND_MESSAGE_TO_CONNECTION_MNGR(CODEPLUG_INIT_FAILED_ACK, NULL, 0);
      }
      break;

    case BT_INIT_SUCCESS_ACK:
      if(isBtQualTestMode)
      {
        SoftwareReady = true;
        StateTransaction(BT_QUAL_TEST_MODE_UPDATE,&ConnectionManagerState,BT_QUAL_TEST_MODE);
        break;
      }
      break;

    case BT_CONNECT_ACL_SUCCESS_ACK:
      if( hspstatus == NEW_DEV_CONNECTING )
      {
        CANCEL_NEW_HEADSET_CONNECTION_REQUEST_TIMER();
      }
      break;
      

    /* Label 8 in the state diagram */
    case NEW_HEADSET_CONNECTION_REQUEST_TIMEOUT_ACK: /*NEW CONNECTION REQUEST TIMER expired */
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &temp_hspDev, sizeof(btDevInfo));
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_CLEAR_PAIRING_INFO, &temp_hspDev, sizeof(btDevInfo));
        clear_pairing_database(&temp_hspDev);

        headset_database_status_update(DB_EMPTY);
        /* Do a soft reset as CSR might be in a bad state */ 
        bt_disconnect_reset();
      break;

    /* Label W13 in the state diagram */ 
    case NEW_WIRELESSPTT_CONNECTION_REQUEST_TIMEOUT_ACK:
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &temp_sppDev, sizeof(btDevInfo));
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_CLEAR_PAIRING_INFO, &temp_sppDev, sizeof(btDevInfo));
        clear_pairing_database(&temp_sppDev);

        wirelessPTT_database_status_update(DB_EMPTY);
        wirelessPTTStateTransaction(PAIRING_STANDBY_STATE);
        /* Do a soft reset as CSR not able to connect - Headset should reconnect back */ 
        bt_disconnect_reset();        
      break;

    case SEND_INTERCHIP_PING_MSG:
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(HCI_PING_GET_BT_CHIP_RESPONSE, NULL, 0);     
      INTERCHIP_PING_TIMER(3000, INTERCHIP_PING_TIMEOUT);          
      break;
 
    case INTERCHIP_PING_TIMEOUT:
      //BT chip did not respond back
      fatal_error_reset();
      break;
      
    case FATAL_ERROR_ACK:
    case CODEPLUG_INIT_FAILED_ACK:
    case BT_INIT_FAILED_ACK:
      StateTransaction(FATAL_ERROR_STATE_UPDATE,&ConnectionManagerState,FATAL_ERROR_STATE); /* Update LED Indication */
      SEND_MESSAGE_TO_CONNECTION_MNGR(SYSTEM_WARM_RESET, NULL, 0);
      break;

    default:
      break;
    }

    if(MPP_PAIRING_MODE)
      RunMppStateMachine(ptr2message, message_id);
    else
      RunStandardStateMachine(ptr2message, message_id);
    }
  }
}
/* End of connection_manager_task()  */
/*****************************************************************************************
CCMPD01414087  : ONLY FOR MPP PAIRING. CONNECTION MANAGER STATE CHANGED HERE ACCORDING TO
                  MPP STATE MEACHINE.
******************************************************************************************/
/*===========================================================================
 *  Function:       RunMppStateMachine
 *  Description:    Runs state machine for MPP Pairing
 *  Returns:        None.
 *===========================================================================*/
void RunMppStateMachine(void * ptr2message, BT_U8 msg_id)
{
  UINT16_T temp_data;
  cplg_param_t code_plug_value;
  UINT8_T echo_function = 0;
  BOOL_T send_echo = FALSE;
  
  /********************************** ANY_STATE ********************************/
  switch(msg_id)
  {
    case MPP_SUCC_PAIR:
      if(((host_succ_mpp*)ptr2message)->device_type.device == AUDIO_DEVICE)
      {
        bt_pair_success_hsp_handling((host_succ_mpp *)ptr2message);
      }
      else if(((host_succ_mpp *)ptr2message)->device_type.device == NON_AUDIO_DEVICE)
      {
        bt_pair_success_spp_handling((host_succ_mpp *)ptr2message);
      }
      //Clear the OOR GW PTT Pressed flag
      wrsm_oor_gw_ptt_pressed = FALSE;
      break;

    case REJECT_PAIR:
      reject_pair_handling((host_reject_mpp *) ptr2message);
      break;

    case BT_CONNECT_SUCCESS_ACK:
      if(((btDevInfo*)ptr2message)->profile == HSP)
      {
        bt_connect_success_ack_hsp_handling(ptr2message);
      }
      else if(((btDevInfo*)ptr2message)->profile == SPP)
      {
        bt_connect_success_ack_spp_handling(ptr2message);
      }
      break;

    case BT_DISCONNECTED_ACK: /* vgxt68:  case for BT disconnected(Normal Power Off)*/
      if ( CompareBTAddress(hspDev.addr,(((btDevInfo *)ptr2message)->addr)) )
      {
        bt_disconnected_ack_hsp_handling(ptr2message);
      }
      else if ( CompareBTAddress(sppDev.addr,(((btDevInfo *)ptr2message)->addr)) )
      {
        bt_disconnected_ack_spp_handling(ptr2message);
      }
      break;

    /* Label W12b in the state diagram */ 
    case SPP_DISCONNECT_REQUEST_DELAY_TIMEOUT:
      disconnect_xnl_link(device_2_id);
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &sppDev, sizeof(btDevInfo));
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &sppDev, sizeof(btDevInfo));
      break;

    /* Label 7b in the state diagram */
    case HSP_DISCONNECT_REQUEST_DELAY_TIMEOUT:
      disconnect_xnl_link(device_1_id);
      //Unload the CSR DSP Application before SCO Disconnect
      if(hw_audio_detect == CP_SELECT_OFF)
      {           
        vm_cmd_send_state = UNLOAD_DSP_CMD;
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(DEVICE_SEND_VM_CMD, NULL, 0);     
      }
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &hspDev, sizeof(btDevInfo));    
      break;

    case HSP_SYSTEM_PING_TIMER_EXPIRED:
      if(null_data_nibble_rxd == false)
      {
      disconnect_xnl_link(device_1_id);
      //Unload the CSR DSP Application before SCO Disconnect
      if(hw_audio_detect == CP_SELECT_OFF)
      {           
        vm_cmd_send_state = UNLOAD_DSP_CMD;
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(DEVICE_SEND_VM_CMD, NULL, 0);     
      }
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &hspDev, sizeof(btDevInfo));          
      }
      break;      
    case HSP_DATA_COMM_ERROR:
      if(null_data_nibble_rxd == false)
      {
        disconnect_xnl_link(device_1_id);
        //Unload the CSR DSP Application before SCO Disconnect
        if(hw_audio_detect == CP_SELECT_OFF)
        {           
          vm_cmd_send_state = UNLOAD_DSP_CMD;
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(DEVICE_SEND_VM_CMD, NULL, 0);     
        }
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &hspDev, sizeof(btDevInfo)); 
      }
      else
      {
        //Reset the Retry Failure variables and keep going on
        xnl_reset_xcmp_retry_failures(device_1_id);
      }
      break;
  
    case HSP_NIBBLER_PTT_PRESS_TMO:
      if(gw_legacy_behavior == true)
      {
        disconnect_xnl_link(device_1_id);
        //Unload the CSR DSP Application before SCO Disconnect
        if(hw_audio_detect == CP_SELECT_OFF)
        {           
          vm_cmd_send_state = UNLOAD_DSP_CMD;
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(DEVICE_SEND_VM_CMD, NULL, 0);     
        }
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &hspDev, sizeof(btDevInfo));          
      }
      break;
      
    case SPP_DATA_COMM_ERROR:
    case SPP_SYSTEM_PING_TIMER_EXPIRED:
      disconnect_xnl_link(device_2_id);
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &sppDev, sizeof(btDevInfo));
      break;

    case XNL_XCMP_INITIALIZATION_COMPLETE:
      temp_data = *((UINT16_T*)ptr2message);
      if(device_1_id == temp_data)
      {
        SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_VOL_STEP_SIZE_MSG, NULL, 0);
        //Start timer to send message to get the hw name of wrsm
        RETRY_XCMP_MSG_TIMER(HW_VER_REQ_MSG, HSP_XCMP_HW_VER_REQ_MSG, 2500, NULL);
        // Allow connection of WPTT device only once Headset is successfully connected
        /* Label W8 in the state diagram */       
        if( (WirelessPTTState == BT_DISCONNECTED_STATE) || (WirelessPTTState == BT_CONNECTING_STATE))
        {
           wirelessPTTStateTransaction(BT_CONNECTING_STATE);
           SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &sppDev, sizeof(btDevInfo));
           SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &sppDev, sizeof(btDevInfo));
        }   
      }
      else if(device_2_id == temp_data)
      {
        SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_VOL_STEP_SIZE_MSG, NULL, 0);
      }
      xnl_free(ptr2message);
      break;   
  
    case HSP_SEND_XCMP_VOL_BCAST:
      CANCEL_RETRY_XCMP_MSG_TIMER(HSP_TONE_REQ_MSG);
      CANCEL_RETRY_XCMP_MSG_TIMER(HSP_VOL_BCAST_MSG);            
      if( device_1_id == 0) break;
      if(send_xcmp_volume_control_msg_bdst(device_1_id, accy_vol, 0x00) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_VOL_BCAST_MSG, HSP_SEND_XCMP_VOL_BCAST, 25, NULL);
      }   
      else
      {
        RETRY_XCMP_MSG_TIMER(HSP_TONE_REQ_MSG, SEND_XCMP_TONE_RQST_START, 25, NULL);
      }
      break;
      
    case SEND_XCMP_TONE_RQST_START:
      if( device_1_id == 0) break;
      CANCEL_RETRY_XCMP_MSG_TIMER(HSP_TONE_REQ_MSG);
      CANCEL_RETRY_XCMP_MSG_TIMER(HSP_VOL_BCAST_MSG);     
      if((isHSPttPress || isWPTTPttPress) != true)
      { 
        if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_START, VOL_FEEDBACK_TONE, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
        {
          RETRY_XCMP_MSG_TIMER(HSP_TONE_REQ_MSG, SEND_XCMP_TONE_RQST_START, 25, NULL);
        }
        else
        {
          /* Tone Stop not required for Momentary tone
          RETRY_XCMP_MSG_TIMER(HSP_TONE_REQ_MSG, SEND_XCMP_TONE_RQST_STOP, 150, NULL);  */
        }
      }
      break;
      
    case SEND_XCMP_TONE_RQST_STOP:
/*      if( device_1_id == 0) break;
      if( send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_STOP, VOL_FEEDBACK_TONE, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_TONE_REQ_MSG, SEND_XCMP_TONE_RQST_STOP, 25, NULL);
      }*/
      break; 
      
    case HSP_XCMP_VOL_STEP_SIZE_MSG:
      if( device_1_id == 0 ) break;
      if(send_xcmp_volume_control_msg_rqst(device_1_id, SET_INCR_SIZE) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(VOL_LVL_REQ_MSG, HSP_XCMP_VOL_STEP_SIZE_MSG, 50, NULL);
      }
      break;

    case HSP_XCMP_BT_CONFIG_REQ_RCVD_FAIL:
      if( device_1_id == 0) break;
      if(send_xcmp_bt_config_msg_rply(device_1_id, XCMP_MSG_RESULT_SUCCESS, XCMP_MSG_BT_POWER_UP_CONFIG, XCMP_MSG_BT_AUDIO_ACCESSORY_TYPE,
                                      XCMP_MSG_BT_IMMEDIATE_PAIRING, XCMP_MSG_BT_8_HR_TIMEOUT, XCMP_MSG_BT_STATE_ON, XCMP_MSG_BT_MPP_PAIRING_TYPE) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_BT_CFG_RPLY_MSG, HSP_XCMP_BT_CONFIG_REQ_RCVD_FAIL, 50, NULL);
      }
      break;

    case SEND_XCMP_VOL_BCAST_NO_TONE:
      if( device_1_id == 0) break;
      if(send_xcmp_volume_control_msg_bdst(device_1_id, accy_vol, 0x00) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_VOL_BCAST_MSG, SEND_XCMP_VOL_BCAST_NO_TONE, 25, NULL);
      }    
      else
      {
      /*  if( device_2_id != 0 ) SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_SEND_XCMP_VOL_BCAST, NULL, 0);        */
      }
      break; 
      
    case HSP_XCMP_PING_MSG:
      if( device_1_id == 0) break;
      if(send_xcmp_ping_msg_rply(device_1_id, XCMP_MSG_RESULT_SUCCESS) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_PING_MSG, HSP_XCMP_PING_MSG, 50, NULL);
      }
      break;
      
    case HSP_XCMP_RADIO_STATUS_MSG:
      if( device_1_id == 0) break;
      if(send_xcmp_radio_status_msg_rply(device_1_id, XCMP_MSG_RESULT_SUCCESS, RAD_STAT_LOW_BATT, 0x00) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_RAD_STAT_MSG, HSP_XCMP_RADIO_STATUS_MSG, 50, NULL);
      }        
      break;

    case HSP_SEND_XCMP_VOL_RPLY:
      if( device_1_id == 0)
      { 
        xnl_free(ptr2message);
        break;
      }
      
      if(send_xcmp_volume_control_msg_rply(device_1_id, XCMP_MSG_RESULT_SUCCESS, ((xcmp_msg_volume_ctrl_request_msg_t *)ptr2message)->function, 0x00, 0x00) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_VOL_LVL_REP_MSG, HSP_SEND_XCMP_VOL_RPLY, 25, ptr2message);
      }
      else
      {
        if(((xcmp_msg_volume_ctrl_request_msg_t *)ptr2message)->function == SPECIFIED_VALUE)
        {
          if( device_1_id != 0 ) SEND_MESSAGE_TO_CONNECTION_MNGR(SEND_XCMP_VOL_BCAST_NO_TONE, NULL, 0);        
        /*  if( device_2_id != 0 ) SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_SEND_XCMP_VOL_BCAST, NULL, 0);        */
        }
        xnl_free(ptr2message);
      }
      break;

    case HSP_SEND_XCMP_VOL_NEG_RPLY:
      if( device_1_id == 0)
      { 
        xnl_free(ptr2message);
        break;
      }
      if(send_xcmp_volume_control_msg_rply(device_1_id, XCMP_MSG_FUNC_NOT_AVAILABLE, ((xcmp_msg_volume_ctrl_request_msg_t *)ptr2message)->function, 0x00, 0x00) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_VOL_LVL_REP_MSG, HSP_SEND_XCMP_VOL_NEG_RPLY, 25, ptr2message);
      }
      else
      {
        xnl_free(ptr2message);
      }         
      break;
      
    case HSP_XCMP_HW_VER_REQ_MSG: 
      if( device_1_id == 0) break;
      CANCEL_RETRY_XCMP_MSG_TIMER(HW_VER_REQ_MSG);
      if(send_xcmp_version_info_msg_rqst(device_1_id, ACCY_NAME) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HW_VER_REQ_MSG, HSP_XCMP_HW_VER_REQ_MSG, 20, NULL);
      } 
      else
      {
        ver_msg_state = ACCY_HW_VER_MSG_SENT;
        //Start a timer to check if response received
        RETRY_XCMP_MSG_TIMER(HW_VER_REQ_MSG, HSP_XCMP_HW_VER_REPLY_NOT_RCVD, 2500, NULL);
      }
      break;
    case HSP_XCMP_VER_PRD_ID_REQ_RCVD:
      if( device_1_id == 0) break;    
      if(send_gw_product_id(device_1_id) != SUCC)
      {
        RETRY_XCMP_MSG_TIMER(HSP_VER_PRD_ID_REQ_RCVD, HSP_XCMP_VER_PRD_ID_REQ_RCVD, 50, NULL);
      }      
      break;      
    case HSP_XCMP_HW_VER_REPLY_NOT_RCVD:
    case HSP_SEND_SW_VER_REQ_MSG:
      if( device_1_id == 0) break;
      //Cancel the Timer for HW Version info request as a reply was received
      CANCEL_RETRY_XCMP_MSG_TIMER(HW_VER_REQ_MSG);
      //Send the Version Info Request for Software version of device
      if(send_xcmp_version_info_msg_rqst(device_1_id, HOST_SW_VER) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(VER_REQ_MSG, HSP_SEND_SW_VER_REQ_MSG, 50, NULL);
      }   
      else
      {
        ver_msg_state = ACCY_SW_VER_MSG_SENT;
      }
      break;
    case HSP_SW_VER_REPLY_RCVD_MSG:
      if( device_1_id == 0) break;       
      codeplugGetParameter(GW_LEGACY_BEHAVIOR, &code_plug_value, 0);
      //Store the variable in codeplug for disconnect resets if it was changed
      if(gw_legacy_behavior != (bool)code_plug_value)
      {
        code_plug_value = (cplg_param_t)gw_legacy_behavior;
        codeplugSetParameter(GW_LEGACY_BEHAVIOR, code_plug_value, 0);
        codeplugWriteback();
      }      
      //Only send the codeplug read requests if it is a later version(Phase 3) of WRSM
      if(gw_legacy_behavior == FALSE)
      {
        SEND_MESSAGE_TO_CONNECTION_MNGR(GW_TPT_DELAY_CPLG_READ_REQ, NULL, 0);
      }
      break;
    case HSP_XCMP_VER_REQ_RCVD:
      if( device_1_id == 0) break;    
      if(send_gw_version_info(device_1_id) != SUCC)
      {
        RETRY_XCMP_MSG_TIMER(HSP_VER_REQ_RCVD, HSP_XCMP_VER_REQ_RCVD, 50, NULL);
      }
      break;

    case GW_TPT_DELAY_CPLG_READ_REQ:
      if( device_1_id == 0) break;    
      if(send_xcmp_cplg_read_rqst(device_1_id, 0x0000, 0x0000, GW_TPT_DELAY_CPLG_FIELD, 0) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_CPLG_READ_REQ, GW_TPT_DELAY_CPLG_READ_REQ, 25, NULL);
      }
      break;
    case GW_TPT_DELAY_CPLG_READ_REPLY_RCVD:
      if( device_1_id == 0) break;    
      if(send_xcmp_cplg_read_rqst(device_1_id, 0x0000, 0x0000, GW_PTT_PRESS_NIBBLER_TMO_CPLG_FIELD, 0) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_CPLG_READ_REQ, GW_TPT_DELAY_CPLG_READ_REPLY_RCVD, 25, NULL);
      }
      break;    
    case GW_PTT_TMO_CPLG_READ_REPLY_RCVD:
      if( device_1_id == 0) break;    
      if(send_xcmp_cplg_read_rqst(device_1_id, 0x0000, 0x0000, GW_USE_NEVER_REPAIR_FIELD, 0) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_CPLG_READ_REQ, GW_PTT_TMO_CPLG_READ_REPLY_RCVD, 25, NULL);
      }
      break;  
    case GW_NEVER_REPAIR_CPLG_READ_REPLY_RCVD:
      //Store the variable in codeplug for power off reboots
      codeplugGetParameter(GW_NEVER_REPAIR, &code_plug_value, 0);
      if(gw_never_repair != (bool)code_plug_value)
      {
        code_plug_value = (cplg_param_t)gw_never_repair;
        codeplugSetParameter(GW_NEVER_REPAIR, code_plug_value, 0);
        codeplugWriteback();
      }
      if(gw_never_repair == true)
      {        
        //Send Updated BT config message to Accessory
        SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_BT_CONFIG_BCAST_INF_PAIRING, NULL, 0);
      }
      break;
    case HSP_XCMP_BT_CONFIG_BCAST_INF_PAIRING:
      if( device_1_id == 0) break;
      if(send_xcmp_bt_config_msg_bcast(device_1_id, XCMP_MSG_BT_POWER_UP_CONFIG, XCMP_MSG_BT_AUDIO_ACCESSORY_TYPE, 
                                        XCMP_MSG_BT_INFINITE_PAIRING, XCMP_MSG_BT_0_TIMEOUT, XCMP_MSG_BT_STATE_ON, 
                                        XCMP_MSG_BT_MPP_PAIRING_TYPE) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_BT_CFG_BCAST_MSG, HSP_XCMP_BT_CONFIG_BCAST_INF_PAIRING, 15, NULL);
      }
      break;    
    case HSP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT:
      if( device_1_id == 0) break;
      if(send_xcmp_bt_config_msg_bcast(device_1_id, XCMP_MSG_BT_POWER_UP_CONFIG, XCMP_MSG_BT_AUDIO_ACCESSORY_TYPE, 
                            XCMP_MSG_BT_IMMEDIATE_PAIRING, XCMP_MSG_BT_0_TIMEOUT, XCMP_MSG_BT_STATE_ON, 
                            XCMP_MSG_BT_MPP_PAIRING_TYPE) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_BT_CFG_RPLY_MSG, HSP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, 10, NULL);
      }
      break;
      
    case HSP_XCMP_UNSUPPORTED_MSG:
      if( device_1_id == 0) break;
      if(xcmp_msg_tx_unsupported_xcmp_msg_reply(device_1_id, hsp_unsupported_xcmp_opcode) != SUCC)
      {
        RETRY_XCMP_MSG_TIMER(HSP_UNSUPPORTED_REQ_MSG, HSP_XCMP_UNSUPPORTED_MSG, 50, NULL);
      }           
      break;

    case HSP_RCVD_XCMP_INVALID_MSG:
      if( device_1_id == 0)
      {
        xnl_free(ptr2message);
        break;
      }
      if(ptr2message != NULL)
      {
        echo_function = *((UINT8_T*)ptr2message);
        send_echo = TRUE;
      }
      if(xcmp_msg_send_invalid_xcmp_msg_reply(device_1_id, hsp_invalid_msg_xcmp_opcode, echo_function, send_echo) != SUCC)
      {
        RETRY_XCMP_MSG_TIMER(HSP_RCVD_INVALID_MSG, HSP_RCVD_XCMP_INVALID_MSG, 50, ptr2message);
      } 
      else
      {
        xnl_free(ptr2message);
      }           
      break;
    case HSP_XCMP_BT_CONFIG_REQ_RCVD_SUCC:
      if( device_1_id == 0) break;
      CANCEL_RETRY_XCMP_MSG_TIMER(HW_VER_REQ_MSG);
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_HW_VER_REQ_MSG, NULL, 0);
      break;      
      
      case SPP_SEND_XCMP_VOL_BCAST:
        /* Do nothing to reduce BT traffic */ /*
        if( device_2_id == 0) break;
        if(send_xcmp_volume_control_msg_bdst(device_2_id, accy_vol, 0x00) != TX_SUCCESS)
        {
          RETRY_XCMP_MSG_TIMER(SPP_VOL_BCAST_MSG, SPP_SEND_XCMP_VOL_BCAST, 25, NULL);
        }   */
        break;
  
      case SPP_XCMP_VOL_STEP_SIZE_MSG:
        if( device_2_id == 0 ) break;
        if(send_xcmp_volume_control_msg_rqst(device_2_id, SET_INCR_SIZE) != TX_SUCCESS)
        {
          RETRY_XCMP_MSG_TIMER(VOL_LVL_REQ_MSG, SPP_XCMP_VOL_STEP_SIZE_MSG, 50, NULL);
        }
        break; 
  
      case SPP_XCMP_BT_CONFIG_REQ_RCVD_FAIL:
        if( device_2_id == 0) break;
        if(send_xcmp_bt_config_msg_rply(device_2_id, XCMP_MSG_RESULT_SUCCESS, XCMP_MSG_BT_POWER_UP_CONFIG, XCMP_MSG_BT_PTT_ACCESSORY_TYPE,
                                        XCMP_MSG_BT_IMMEDIATE_PAIRING, XCMP_MSG_BT_8_HR_TIMEOUT, XCMP_MSG_BT_STATE_ON, XCMP_MSG_BT_MPP_PAIRING_TYPE) != TX_SUCCESS)
        {
          RETRY_XCMP_MSG_TIMER(SPP_BT_CFG_RPLY_MSG, SPP_XCMP_BT_CONFIG_REQ_RCVD_FAIL, 50, NULL);
        }
        break;
     case SPP_XCMP_VER_REQ_RCVD:
        if( device_2_id == 0) break;      
        if(send_gw_version_info(device_2_id) != SUCC)
        {
          RETRY_XCMP_MSG_TIMER(SPP_VER_REQ_RCVD, SPP_XCMP_VER_REQ_RCVD, 50, NULL);
        }
        break;
     case SPP_XCMP_VER_PRD_ID_REQ_RCVD:
        if( device_2_id == 0) break;    
        if(send_gw_product_id(device_2_id) != SUCC)
        {
          RETRY_XCMP_MSG_TIMER(SPP_VER_PRD_ID_REQ_RCVD, SPP_XCMP_VER_PRD_ID_REQ_RCVD, 50, NULL);
        }      
        break;               
     case SPP_XCMP_PING_MSG:
      if( device_2_id == 0) break;
      if(send_xcmp_ping_msg_rply(device_2_id, XCMP_MSG_RESULT_SUCCESS) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(SPP_PING_MSG, SPP_XCMP_PING_MSG, 50, NULL);
      }
      break;
      
    case SPP_XCMP_RADIO_STATUS_MSG:
      if( device_2_id == 0) break;
      if(send_xcmp_radio_status_msg_rply(device_2_id, XCMP_MSG_RESULT_SUCCESS, RAD_STAT_LOW_BATT, 0x00) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(SPP_RAD_STAT_MSG, SPP_XCMP_RADIO_STATUS_MSG, 50, NULL);
      }        
      break;
  
    case SPP_SEND_XCMP_VOL_RPLY:
      if( device_2_id == 0)
      { 
        xnl_free(ptr2message); 
        break;
      }
      
      if(send_xcmp_volume_control_msg_rply(device_2_id, XCMP_MSG_RESULT_SUCCESS, ((xcmp_msg_volume_ctrl_request_msg_t *)ptr2message)->function, 0x00, 0x00) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(SPP_VOL_LVL_REP_MSG, SPP_SEND_XCMP_VOL_RPLY, 25, ptr2message);
      }
      else
      {
        xnl_free(ptr2message);
      }        
      break;
    case SPP_SEND_XCMP_VOL_NEG_RPLY:
      if( device_1_id == 0)
      { 
        xnl_free(ptr2message);
        break;
      }
      
      if(send_xcmp_volume_control_msg_rply(device_1_id, XCMP_MSG_FUNC_NOT_AVAILABLE, ((xcmp_msg_volume_ctrl_request_msg_t *)ptr2message)->function, 0x00, 0x00) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(SPP_VOL_LVL_REP_MSG, SPP_SEND_XCMP_VOL_NEG_RPLY, 25, ptr2message);
      }
      else
      {
        xnl_free(ptr2message);
      }         
      break;  
    case SPP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT:
      if( device_2_id == 0) break;
      if(send_xcmp_bt_config_msg_bcast(device_2_id, XCMP_MSG_BT_POWER_UP_CONFIG, XCMP_MSG_BT_AUDIO_ACCESSORY_TYPE, 
                              XCMP_MSG_BT_IMMEDIATE_PAIRING, XCMP_MSG_BT_0_TIMEOUT, XCMP_MSG_BT_STATE_ON, 
                              XCMP_MSG_BT_MPP_PAIRING_TYPE) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(SPP_BT_CFG_RPLY_MSG, SPP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, 10, NULL);
      }   
      break;
      
    case SPP_XCMP_UNSUPPORTED_MSG:
      if( device_2_id == 0) break;
      if(xcmp_msg_tx_unsupported_xcmp_msg_reply(device_2_id, spp_unsupported_xcmp_opcode) != SUCC)
      {
        RETRY_XCMP_MSG_TIMER(SPP_UNSUPPORTED_REQ_MSG, SPP_XCMP_UNSUPPORTED_MSG, 50, NULL);
      }           
      break;
  
    case SPP_RCVD_XCMP_INVALID_MSG:
      if( device_2_id == 0)
      {
        xnl_free(ptr2message);
        break;
      }
      if(ptr2message != NULL)
      {
        echo_function = *((UINT8_T*)ptr2message);
        send_echo = TRUE;
      }
      
      if(xcmp_msg_send_invalid_xcmp_msg_reply(device_2_id, spp_invalid_msg_xcmp_opcode, echo_function, send_echo) != SUCC)
      {
        RETRY_XCMP_MSG_TIMER(SPP_RCVD_INVALID_MSG, SPP_RCVD_XCMP_INVALID_MSG, 50, ptr2message);
      }     
      else
      {
        xnl_free(ptr2message);
      }           
      break;

    case SPP_XCMP_BT_CONFIG_REQ_RCVD_SUCC:
      /* Label W12a in the state diagram */ 
      if( ConnectionManagerState != BT_CONNECTED_STATE )
      {
        SEND_SPP_DISCONNECT_DELAY_TIMER(10, SPP_DISCONNECT_REQUEST_DELAY_TIMEOUT);
        wirelessPTT_database_status_update(DEV_DISCONNECTING);
      }    
      else  //Send WPTT Connected Tone Request to the Audio Device
      {
        if(device_1_id != 0)
        {
          if((isHSPttPress || isWPTTPttPress) != true)
          {
            if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_START, BT_CONNECTING_SUCCESS_TONE, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
            {
              RETRY_XCMP_MSG_TIMER(SPP_TONE_REQ_MSG, SPP_REQ_TONE_CONNECTED, 25, NULL);
            } 
          }
          else
          {
            RETRY_XCMP_MSG_TIMER(SPP_TONE_REQ_MSG, SPP_REQ_TONE_CONNECTED, 500, NULL);
          }
        }
      }     
      break;
    case SPP_REQ_TONE_CONNECTED:
      if( device_2_id == 0) break;
      if((isHSPttPress || isWPTTPttPress) != true)     
      {
        if(device_1_id != 0)
        {
          if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_START, BT_CONNECTING_SUCCESS_TONE, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(SPP_TONE_REQ_MSG, SPP_REQ_TONE_CONNECTED, 25, NULL);
          }  
        }      
      }
      else
      {
        RETRY_XCMP_MSG_TIMER(SPP_TONE_REQ_MSG, SPP_REQ_TONE_CONNECTED, 500, NULL);
      }
      break;
      
    case SPP_REQ_TONE_DISCONNECTED:
      if(device_1_id != 0)
      {      
        CANCEL_RETRY_XCMP_MSG_TIMER(SPP_TONE_REQ_MSG);

        if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_START, BT_DISCONNECTING_SUCCESS_TONE, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
        {
          RETRY_XCMP_MSG_TIMER(SPP_TONE_REQ_MSG, SPP_REQ_TONE_DISCONNECTED, 25, NULL);
        }   
      }
      break;
    case SPP_REQ_TONE_DISCON_CLN_XNL_LINK:
      if(device_1_id != 0)
      {      
        CANCEL_RETRY_XCMP_MSG_TIMER(SPP_TONE_REQ_MSG);
        if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_START, BT_DISCONNECTING_SUCCESS_TONE, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
        {
          RETRY_XCMP_MSG_TIMER(SPP_TONE_REQ_MSG, SPP_REQ_TONE_DISCON_CLN_XNL_LINK, 25, NULL);
        }    
        else
        {
          //Cleanup the PTT XNL link as it was pending on the message
          disconnect_xnl_link(device_2_id);
        }
      }
      else
      {
        //Cleanup the PTT XNL link as it was pending on the message
        disconnect_xnl_link(device_2_id);   
        // Do a soft reset as the Audio device was disconnected and WPTT link disconnect was pending
        bt_disconnect_reset();
      }
      break;

    case SPP_XCMP_BATT_LVL_LOW:
      // Start a low battery tone if PTT device not disconnected and audio device connected
      if( device_2_id == 0) break;
      if(device_1_id != 0)
      {
        if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_START, ALERT_TONE_LOW_BATTERY, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
        {
          RETRY_XCMP_MSG_TIMER(SPP_BATT_LVL_RCVD, SPP_XCMP_BATT_LVL_LOW, 25, NULL);
        }
      }
      break;
      
    case SPP_XCMP_BATT_LVL_OKAY:
      if( device_2_id == 0) break;
      if(device_1_id != 0)
      {     
        // Stop the tone - Not required to send a message as the tone requested is a Momentary Tone so shouldnt be playing continuously
        if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_STOP, ALERT_TONE_LOW_BATTERY, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
        {
          RETRY_XCMP_MSG_TIMER(SPP_BATT_LVL_RCVD, SPP_XCMP_BATT_LVL_OKAY, 25, NULL);
        }      
      }
      break;  

    default:
      break;
  }
  

  switch(ConnectionManagerState)
  {
/********************************** INIT_STATE ********************************/
  case INIT_STATE:
      switch(msg_id)
      {
      case MPP_INIT_HOST_MPP_PKT_REQ:
        SoftwareReady = true;
        init_mpp_pkts();     /* Initialize MPP packet */
        init_connection_db(); /* initialize MPP database */

        /* send the INIT_HOST_MPP_PKT_RPLY to MPP manager */
        SEND_MESSAGE_TO_MPP_MNGR(INIT_HOST_MPP_PKT_RPLY, &MPP_init_host_pkt, sizeof(init_host_mpp_pkt));

        /* send the HOST_CONN_ALLOC_RES_DB_RPLY to MPP manager */
        SEND_MESSAGE_TO_MPP_MNGR(HOST_CONN_ALLOC_RES_DB_RPLY,&MPP_host_connection_db, sizeof(connection_database ));
        
        SEND_MESSAGE_TO_CONTROLS_MANAGER(INIT_STATE_UPDATE, NULL, 0); /* this is after completion of codeplug init, MPP init, BT mgr init */
        
        break;

      case MPP_HOST_CONN_ALLOC_RES_DB_REQ:
        SEND_MESSAGE_TO_MPP_MNGR(HOST_CONN_ALLOC_RES_DB_RPLY,&MPP_host_connection_db, sizeof(connection_database ));        
        break;

      case BT_INIT_SUCCESS_ACK:
        /*Create the MPP STATE MACHINE task once BT Manager has notified that BT Stack initialization is complete*/
        vr_create_task(MPP_STATE_MACHINE);
        /*Store the BT Address*/
        connection_mgr_update_BTAddr(ptr2message); 
        //Atmel and CSR Version has matched
        tc_flags.version_match = TRUE;
        // Start the interchip ping - if USB not enumerated
        if (!Is_device_enumerated())
        {
          SEND_INTERCHIP_PING_MSG_TIMER(3000, SEND_INTERCHIP_PING_MSG);
        }
        break;
        
      case BT_INIT_CSR_VERSION_MISMATCH:
        /* Let the Gateway controls be functional - MPP and BT not allowed */
        SEND_MESSAGE_TO_CONTROLS_MANAGER(BT_INIT_CSR_FW_MISMATCH_UPDATE, NULL, 0);
        tc_flags.version_match = FALSE;
        break;
        
      case MPP_READY_PAIR: /* MPP task is ready for MPP Pairing Process */
        SEND_MESSAGE_TO_MPP_MNGR(START_PAIR, NULL, MESSAGE_SIZE_ZERO);
        //Get the Codeplug Value if GW is supposed to be Infinitely paired
        codeplugGetParameter(GW_NEVER_REPAIR, &cp_parameter_value, 0);
        gw_never_repair = (bool)cp_parameter_value;          
                  
        /* Get the reset_cause to check if GW needs to reconnect to the devices connected prior to reset */
        if((get_reset_cause() == WATCHDOG_RESET) || (gw_never_repair == true))
        {
          connection_retrieve_pair_data();
          if(hspstatus != DB_EMPTY) /* if hsp data is present in codeplug, then dongle will enter to initate the connect request and transition to <CONNECTING_STATE> */
          {
            codeplugGetParameter(GW_LEGACY_BEHAVIOR, &cp_parameter_value, 0);
            gw_legacy_behavior = (bool)cp_parameter_value;
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &hspDev, sizeof(btDevInfo));
            StateTransaction(BT_CONNECTING_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTING_STATE);
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &hspDev, sizeof(btDevInfo));
          }
          else /* if no hsp data in codeplug, then dongle will start the device discovery & pairing */
          {
            gw_legacy_behavior = TRUE;
            null_data_nibble_rxd = false;
            StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
          }
          // Check for the SPP device if there is any previous connected device
          if(sppstatus != DB_EMPTY)
          {
            /* The WPTT connection will be allowed once the Headset is successfully connected */
            wirelessPTTStateTransaction(BT_CONNECTING_STATE);
          }
          else
          {
            wirelessPTTStateTransaction(PAIRING_STANDBY_STATE);
          }
        }
        else
        {
          /* Hard Reset - Clear the pairing info of both profiles */
          headset_database_status_update(DB_EMPTY);
          connection_clear_all_pair_data(HSP);             
          StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
          
          wirelessPTT_database_status_update(DB_EMPTY);
          connection_clear_all_pair_data(SPP);                   
          wirelessPTTStateTransaction(PAIRING_STANDBY_STATE);
        }
        break;
        
      }
      break;

/*************************** PAIRING_STANDBY_STATE ****************************/
    case PAIRING_STANDBY_STATE:
      switch(msg_id)
      {
        default:
          break;
      }
      break;

/**************************** BT_CONNECTING_STATE *****************************/
  case BT_CONNECTING_STATE:
    switch(msg_id)
    {
      case BT_DISCONNECTED_ACK:  
        break;

      /* Label 7a in the state diagram */
      case USER_PTT_PRESS_DETECTED_ACK:
        /* Switch back to local Mic done in General Case */
        if(gw_legacy_behavior == true)
        {
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_CLEAR_PAIRING_INFO, &temp_hspDev, sizeof(btDevInfo));
          headset_database_status_update(DB_EMPTY);
          connection_clear_all_pair_data(HSP);         
          StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
        }
        break;

      case TWO_DOT_BUTTON_PRESSED:
      case ONE_DOT_BUTTON_PRESSED:
        if(onedot_twodot_dc == CP_SELECT_ON)
        {
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_CLEAR_PAIRING_INFO, &temp_hspDev, sizeof(btDevInfo));
          headset_database_status_update(DB_EMPTY);
          connection_clear_all_pair_data(HSP);         
          StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
        }
        break;

      }   
      break;


/***************************** BT_CONNECTED_STATE *****************************/
   case BT_CONNECTED_STATE:
    switch(msg_id)
    {
      /* Label 5 in the state diagram */
      case BT_LINKLOST_DISCONNECTED_ACK:
        null_data_nibble_rxd = false;
        SEND_MESSAGE_TO_CONTROLS_MANAGER(AUDIO_DEVICE_DISCONNECTED_STATE_UPDATE, NULL, 0);

        headset_database_status_update(PAIRED_DEV_CONNECTING);
        disconnect_xnl_link(device_1_id);
        // Unload the CSR DSP Application
        if(hw_audio_detect == CP_SELECT_OFF)
        {             
          vm_cmd_send_state = UNLOAD_DSP_CMD;
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(DEVICE_SEND_VM_CMD, NULL, 0);   
        }
        StateTransaction(BT_CONNECTING_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTING_STATE);

        if( WirelessPTTState == BT_CONNECTED_STATE )
        {
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &sppDev, sizeof(btDevInfo));
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &sppDev, sizeof(btDevInfo));

          wirelessPTT_database_status_update(DEV_DISCONNECTING);
        }  
        else
        {
          /* Do a soft reset on a link loss */
          bt_disconnect_reset();
        }
        break;

     /* Label 7a in the state diagram */
     case USER_PTT_PRESS_DETECTED_ACK: 
        /* Switch back to local Mic done in General Case */
        if(gw_legacy_behavior == true)
        {     
          if(hspstatus != DEV_PTT_PRESS_DISCONNECTING)
          {
            SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, NULL, 0);
            headset_database_status_update(DEV_PTT_PRESS_DISCONNECTING);
            SEND_HSP_DISCONNECT_DELAY_TIMER(50, HSP_DISCONNECT_REQUEST_DELAY_TIMEOUT);
          }
        }      
        break;
        
     case BT_DISCONNECTED_ACK: /* vgxt68:  case for BT disconnected(Normal Power Off)*/
        break;
        
     case TWO_DOT_BUTTON_PRESSED:
     case ONE_DOT_BUTTON_PRESSED:
        if(onedot_twodot_dc == CP_SELECT_ON && hspstatus != DEV_PTT_PRESS_DISCONNECTING)
        {
          SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, NULL, 0);
          headset_database_status_update(DEV_PTT_PRESS_DISCONNECTING);
          SEND_HSP_DISCONNECT_DELAY_TIMER(50, HSP_DISCONNECT_REQUEST_DELAY_TIMEOUT);
        }
        break;

      case ONE_DOT_BUTTON_RELEASED:
        break;
        
      case TWO_DOT_BUTTON_RELEASED:
        break;   

      }
      break;

/****************************** IDLE_STATE *****************************/
    case IDLE_STATE: /* When Dongle enter to IDLE_STATE, it will not process any incoming msg. User need to repower cycle the Dongle */
      switch(msg_id)
      {
        default:
          break;
      }
      break;

/****************************** BT_QUALIFICATION_TEST_MODE *****************************/
    case BT_QUAL_TEST_MODE:
      switch(msg_id)
      {
        case BT_PAIRING_SUCCESS_ACK: /* paired success BT device's pairing info */
          SUCC_PAIR_EVENT((btDevInfo*)ptr2message);
          connection_store_pair_data(&hspDev);
          break;
  
        case BT_CONNECT_SUCCESS_ACK:
          SEND_MESSAGE_TO_CONTROLS_MANAGER(BT_TEST_CONNECTED, NULL, 0);
          break;
  
        case BT_DISCONNECTED_ACK:
         SEND_MESSAGE_TO_CONTROLS_MANAGER(BT_TEST_DISCONNECTED, NULL, 0);
         break;
  
        default:
          break;
      }
      break;

/****************************** FATAL_ERROR_STATE *****************************/
    case FATAL_ERROR_STATE:
      switch(msg_id)
      {
        case SYSTEM_WARM_RESET:
          watchdog_enable(500);
          CANCEL_WATCHDOG_RESET_TIMER();
          break;
  
        default:
          break;
      }
      break;
      
/****************************** BT_DISCONNECTED_RESET_STATE *****************************/     
    case BT_DISCONNECTED_RESET_STATE:
      switch(msg_id)
      {
        case BT_DISCONNECT_SYSTEM_RESET:
          watchdog_enable(200);
          CANCEL_WATCHDOG_RESET_TIMER();
          break;
  
        default:
          break;
      }
      break;      
/******************************** Other States ********************************/
    default:
      break;
/******************************** END OF STATE ********************************/
    }
}

/*===========================================================================
 *  Function:       RunStandarMachine
 *  Description:    Runs state machine for Standar Pairing
 *  Returns:        None.
 *===========================================================================*/
void RunStandardStateMachine(void * ptr2message, BT_U8 msg_id)
{
    switch(ConnectionManagerState)
    {
/********************************** INIT_STATE ********************************/
    case INIT_STATE:
      switch(msg_id)
      {
      case CONN_MGR_INIT:
        deviceDiscovery = DISCOVERY_ALL;
        wirelessPTTStateTransaction(INIT_STATE);
        break;

      case BT_INIT_SUCCESS_ACK:
        SEND_MESSAGE_TO_CONTROLS_MANAGER(INIT_STATE_UPDATE, NULL, 0);
        //For Standard Pairing the Software is ready after Mecel Stack and CSR are initialized
        SoftwareReady = true;
        connection_retrieve_pair_data();
        //Atmel and CSR Version has matched
        tc_flags.version_match = TRUE;
        if(isPairingRequest == true) /* User Pairing Request, Dongle to enter Pairing_Standby_State */
        {
          CANCEL_PENDING_PAIRING_REQ_DELAY_TIMER();
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCOVERY_AND_PAIRING_REQUEST, &deviceDiscovery, sizeof(discoveryType));
          StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
        }
        else
        {
          PENDING_PAIRING_REQ_DELAY_TIMER(3000,USER_INVALID_PAIRING_REQUEST_ACK);
        }
        break;
        
      case BT_INIT_CSR_VERSION_MISMATCH:
        /* Let the Gateway controls be functional - MPP and BT not allowed */
        SEND_MESSAGE_TO_CONTROLS_MANAGER(BT_INIT_CSR_FW_MISMATCH_UPDATE, NULL, 0);
        tc_flags.version_match = FALSE;
        break;
        
      case USER_PTT_PRESS_DETECTED_ACK: /* User request to start Device Discovery & Pairing */
        if(SoftwareReady == true)
        {
          CANCEL_PENDING_PAIRING_REQ_DELAY_TIMER();
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCOVERY_AND_PAIRING_REQUEST, &deviceDiscovery, sizeof(discoveryType));
          connection_clear_all_pair_data(HSP);         
          connection_clear_all_pair_data(SPP);         
          StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
        }
        else
        {
          isPairingRequest = true;
        }
        break;

      case USER_INVALID_PAIRING_REQUEST_ACK: /* Invalid user request */
        if(SoftwareReady == true)
        {
          if(hspstatus != DB_EMPTY) /* if hsp data is present in codeplug, then dongle will enter to initate the connect request and transition to <CONNECTING_STATE> */
          {
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_START_CONNECT_REQUEST, &hspDev, sizeof(btDevInfo));
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &hspDev, sizeof(btDevInfo));
            StateTransaction(BT_CONNECTING_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTING_STATE);
          }
          else /* if no hsp data in codeplug, then dongle will start the device discovery & pairing */
          {
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCOVERY_AND_PAIRING_REQUEST, &deviceDiscovery, sizeof(discoveryType));
            StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
          }
        }
        break;
      }
      break;
	  
	  
/*************************** PAIRING_STANDBY_STATE ****************************/
    case PAIRING_STANDBY_STATE:
      switch(msg_id)
      {
      case BT_PAIRING_STARTED_ACK: /* Device Discovery is ended, and pairing process is started */
        StateTransaction(PAIRING_STATE_UPDATE,&ConnectionManagerState,PAIRING_STATE);
        isPairingRequest = false;
        break;

      case BT_DEVICE_INQUIRY_TIMEOUT_ACK:
        isPairingRequest = false;
        StateTransaction(IDLE_STATE_UPDATE,&ConnectionManagerState,IDLE_STATE);
        break;
      }

      break;

/******************************* PAIRING_STATE ********************************/
    case PAIRING_STATE:
      switch(msg_id)
      {
      case BT_PAIRING_SUCCESS_ACK: /* paired success BT device's pairing info */
        SUCC_PAIR_EVENT((btDevInfo*)ptr2message);
        break;

      case BT_HEADSET_PAIRING_FAIL_ACK: /* Headset found but pairing fail */
        isHspPairFail = true;
        break;

      case BT_PAIRING_COMPLETE_ACK: /* pairing complete/ timeout */

        if (isHspPairFail) /* If headset found but pairing fail, then dongle shall transition to pairing standby state to start for headset only */
        {
          if(sppstatus == DB_EMPTY)
          {
            deviceDiscovery = DISCOVERY_ALL;
          }
          else
          {
            deviceDiscovery = DISCOVERY_HSP_ONLY;
          }
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCOVERY_AND_PAIRING_REQUEST, &deviceDiscovery, sizeof(discoveryType));
          isHspPairFail = false;
          StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
        }
        else if(hspstatus == NEW_DEV_CONNECTING) /* headset pairing information is available, then start to connect with headset */
        {
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_START_CONNECT_REQUEST, &temp_hspDev, sizeof(btDevInfo));
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &temp_hspDev, sizeof(btDevInfo));
          StateTransaction(BT_CONNECTING_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTING_STATE);
          if( sppstatus == NEW_DEV_CONNECTING )
          {
             SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &temp_sppDev, sizeof(btDevInfo));
          }
        }
        else if(sppstatus == NEW_DEV_CONNECTING ) /* none of headset pairing info is available, but new SPP pairing info is availabe */
        {
          deviceDiscovery = DISCOVERY_HSP_ONLY;
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCOVERY_AND_PAIRING_REQUEST, &deviceDiscovery, sizeof(discoveryType));
          StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
        }
        else /* neither headset nor SPP pairing info is available */
        {
          deviceDiscovery = DISCOVERY_ALL;
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCOVERY_AND_PAIRING_REQUEST, &deviceDiscovery, sizeof(discoveryType));
          StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
        }
        break;
      }
      break;

/**************************** BT_CONNECTING_STATE *****************************/
    case BT_CONNECTING_STATE:
      switch(msg_id)
      {
      case BT_CONNECT_TIMEOUT_ACK:

        if(CompareBTAddress(hspDev.addr,(((btDevInfo *)ptr2message)->addr)))
        {
          hspConnTimeoutCount ++;
          if(hspstatus == NEW_DEV_CONNECTING)/* For NEW Bluetooth device, Dongle will transition to <PAIRING_STANDBY_STATE> after timeout */
          {
            if (hspConnTimeoutCount < MAX_HSP_CONN_RETRY)
            {
              SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_START_CONNECT_REQUEST, &hspDev, sizeof(btDevInfo));
            }
            else
            {
              deviceDiscovery = DISCOVERY_ALL;
              SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCOVERY_AND_PAIRING_REQUEST, &deviceDiscovery, sizeof(discoveryType))
              StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
            }
          }
          else if (hspstatus == PAIRED_DEV_CONNECTING)  /* For previously paired device, Dongle will stay at connecting state and resend the connect request */
          {
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_START_CONNECT_REQUEST, &hspDev, sizeof(btDevInfo));
          }
        }
        break;

      case BT_CONNECT_SUCCESS_ACK:
        if(((btDevInfo *)ptr2message)->profile == HSP)
        {

          if( (hspstatus == NEW_DEV_CONNECTING)
          && (CompareBTAddress(temp_hspDev.addr,(((btDevInfo *)ptr2message)->addr))))
          {
            hspConnTimeoutCount = 0;
            if(hspstatus == NEW_DEV_CONNECTING)
            {
              connection_store_pair_data(&temp_hspDev);
              clear_pairing_database(&temp_hspDev);
              headset_database_status_update(DEV_CONNECTED);

              /* in this case hsp and spp with the same address is not going to happen */
            }
          }
          else if( (hspstatus == PAIRED_DEV_CONNECTING)
          && (CompareBTAddress(hspDev.addr,(((btDevInfo *)ptr2message)->addr))))
          {
              headset_database_status_update(DEV_CONNECTED);
          }


          StateTransaction(BT_CONNECTED_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTED_STATE);
          SEND_MESSAGE_TO_CONNECTION_MNGR(CONN_MGR_SPP_CHECK, NULL, 0); /* To check if there is a new SPP device to be connecting */

          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &hspDev, sizeof(btDevInfo));
        }
        else if(CompareBTAddress(sppDev.addr,(((btDevInfo *)ptr2message)->addr)))
        {
           SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &sppDev, sizeof(btDevInfo));
        }
        break;

      case BT_HS_RING_TIMEOUT:
        if(hspstatus == NEW_DEV_CONNECTING)
        {
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &hspDev, sizeof(btDevInfo));
          ringTimeout = true;
        }
        else
        {
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_HS_ACCEPT_CKPD_CMND_REQUEST, NULL, 0);
        }
        /* For previously connected headset, no ring timeout, dongle will wait for the AT+CKPD=200 from headset for audio link establishment */
        break;

      case BT_DISCONNECTED_ACK:
        if(CompareBTAddress(hspDev.addr,(((btDevInfo *)ptr2message)->addr)))
        {
          if(hspstatus == NEW_DEV_CONNECTING || ringTimeout)
          {
            deviceDiscovery = DISCOVERY_ALL;
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCOVERY_AND_PAIRING_REQUEST, &deviceDiscovery, sizeof(discoveryType));
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
            StateTransaction(PAIRING_STANDBY_STATE_UPDATE,&ConnectionManagerState,PAIRING_STANDBY_STATE);
            ringTimeout = false;
          }
          else if (hspstatus == PAIRED_DEV_CONNECTING && (ConnectionManagerPrevState != BT_CONNECTED_STATE))
          {
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_START_CONNECT_REQUEST, &hspDev, sizeof(btDevInfo));
          }

        }
        else if(CompareBTAddress(sppDev.addr,(((btDevInfo *)ptr2message)->addr)))
        {
          wirelessPTTStateTransaction(BT_DISCONNECTED_STATE);
        }
        break;

      //wrn637 : this message has been remove from bluetooth manager.
      case BT_CONNECT_REQUEST: /* Exception handling when dongle receive the SPP connect request but hsp is not yet connected */
       if(CompareBTAddress(sppDev.addr,(((btDevInfo *)ptr2message)->addr)))
        {
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_START_CONNECT_REQUEST, &hspDev, sizeof(btDevInfo));
        }
        break;
      }
      break;


/***************************** BT_CONNECTED_STATE *****************************/
    case BT_CONNECTED_STATE:
      switch(msg_id)
      {
      case CONN_MGR_SPP_CHECK:
        if(sppstatus == NEW_DEV_CONNECTING ) 
        {
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &temp_sppDev, sizeof(btDevInfo));
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_START_CONNECT_REQUEST, &temp_sppDev, sizeof(btDevInfo));
          wirelessPTTStateTransaction(BT_CONNECTING_STATE);
        }
        else if(sppstatus == PAIRED_DEV_CONNECTING)
        {
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &sppDev, sizeof(btDevInfo));
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_START_CONNECT_REQUEST, &sppDev, sizeof(btDevInfo));
          wirelessPTTStateTransaction(BT_CONNECTING_STATE);
        }       
        break;

      case BT_CONNECT_TIMEOUT_ACK: /* This connect timeout is used for new SPP device  */
        if( (sppstatus == NEW_DEV_CONNECTING)
          && (CompareBTAddress(sppDev.addr,(((btDevInfo *)ptr2message)->addr))) )
        {
          sppConnTimeoutCount ++;
          if (sppConnTimeoutCount < MAX_SPP_CONN_RETRY)
          {
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_START_CONNECT_REQUEST, &temp_sppDev, sizeof(btDevInfo));
            wirelessPTTStateTransaction(BT_CONNECTING_STATE);
          }
          else
          {
            if(WirelessPTTState != BT_CONNECTED_STATE)/* Dongle shall ignore the connection timeout ack if it has connected with SPP*/
            {
              wirelessPTTStateTransaction(BT_CONNECTED_STATE); /* If timeout, then dongle will stop the attempt to reconnect with SPP, */
            }                                                  /* it will wait for incoming connect reqeust from SPP */
          }                                                    /* it will wait for incoming connect reqeust from SPP */
        }
        break;

      case BT_CONNECT_SUCCESS_ACK:
        if(((btDevInfo *)ptr2message)->profile == SPP)
        {
          if( (sppstatus == NEW_DEV_CONNECTING)
              && (CompareBTAddress(temp_sppDev.addr,(((btDevInfo *)ptr2message)->addr))) )
          {
            connection_store_pair_data(&temp_sppDev);
            clear_pairing_database(&temp_hspDev);
            wirelessPTT_database_status_update(DEV_CONNECTED);

            /* in this case hsp and spp with the same address is not going to happen */
          }
          else if( (sppstatus == NEW_DEV_CONNECTING)
              && (CompareBTAddress(temp_sppDev.addr,(((btDevInfo *)ptr2message)->addr))) )
          {
            wirelessPTT_database_status_update(DEV_CONNECTED);
          }

          wirelessPTTStateTransaction(BT_CONNECTED_STATE);
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &sppDev, sizeof(btDevInfo));
          sppConnTimeoutCount = 0;
        }
        break;

      case BT_DISCONNECTED_ACK:
        if(CompareBTAddress(hspDev.addr,(((btDevInfo *)ptr2message)->addr)))
        {
          SEND_MESSAGE_TO_CONTROLS_MANAGER(BT_DISCONNECTED_STATE_UPDATE, NULL, 0);
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &sppDev, sizeof(btDevInfo));

          if (WirelessPTTState == BT_CONNECTED_STATE)
          {
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &sppDev, sizeof(btDevInfo));
          }
          else if(WirelessPTTState == BT_CONNECTING_STATE)
          {
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_STOP_SPP_CONNECT_REQUEST, &temp_sppDev, sizeof(btDevInfo));
          }
          /* Dongle stays at connectable mode, it will wait for hsp to initiate the reconnection */
          StateTransaction(BT_CONNECTING_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTING_STATE);
        }
        else if(CompareBTAddress(sppDev.addr,(((btDevInfo *)ptr2message)->addr)))
        {
          /* Dongle stays at connectable mode, it will wait for spp to initiate the reconnection */
          wirelessPTTStateTransaction(BT_DISCONNECTED_STATE);
        }
        break;
      }
      break;

/****************************** IDLE_STATE *****************************/
    case IDLE_STATE: /* When Dongle enter to IDLE_STATE, it will not process any incoming msg. User need to repower cycle the Dongle */
      switch(msg_id)
      {
      default:
        break;
      }
      break;

/****************************** BT_QUALIFICATION_TEST_MODE *****************************/
    case BT_QUAL_TEST_MODE:
      switch(msg_id)
      {

      case BT_PAIRING_SUCCESS_ACK: /* paired success BT device's pairing info */
        SUCC_PAIR_EVENT((btDevInfo*)ptr2message);
        connection_store_pair_data(&hspDev);
        break;

      case BT_CONNECT_SUCCESS_ACK:
        SEND_MESSAGE_TO_CONTROLS_MANAGER(BT_TEST_CONNECTED, NULL, 0);
        break;

     case BT_DISCONNECTED_ACK:
       SEND_MESSAGE_TO_CONTROLS_MANAGER(BT_TEST_DISCONNECTED, NULL, 0);
       break;

      default:
        break;

      }
      break;

/****************************** FATAL_ERROR_STATE *****************************/
    case FATAL_ERROR_STATE:
      switch(msg_id)
      {
      case SYSTEM_WARM_RESET:
        watchdog_enable(500);
        CANCEL_WATCHDOG_RESET_TIMER();
        break;

      default:
        break;
      }
      break;

/******************************** Other States ********************************/
    default:
      break;
/******************************** END OF STATE ********************************/
    }
}
/*=============================================================================
 FUNCTION: send_gw_version_info

 DESCRIPTION: Send GW SW Version 

 ARGUMENTS PASSED: device ID

 RETURN VALUE: TRUE/FALSE
==============================================================================*/
UINT8_T send_gw_version_info(UINT16_T xnl_devid)
{
  version_buff[0] = (UINT8_T)CURRENT_VERSION_IDENTIFIER;
  version_buff[1] = (UINT8_T)CURRENT_VERSION_PHASE;
  version_buff[2] = (UINT8_T)CURRENT_VERSION_MAJOR;
  version_buff[3] = (UINT8_T)CURRENT_VERSION_MINOR;
    
  return send_xcmp_version_info_msg_rply(xnl_devid, XCMP_MSG_RESULT_SUCCESS, version_buff, 0x04);
}
/*=============================================================================
 FUNCTION: send_gw_product_id

 DESCRIPTION: Send GW Product ID Name

 ARGUMENTS PASSED: device ID

 RETURN VALUE: TRUE/FALSE
==============================================================================*/
UINT8_T send_gw_product_id(UINT16_T xnl_devid)
{
  version_buff[0] = 0x4D; // M
  version_buff[1] = 0x4D; // M
  version_buff[2] = 0x47; // G
  version_buff[3] = 0x00; 
    
  return send_xcmp_version_info_msg_rply(xnl_devid, XCMP_MSG_RESULT_SUCCESS, version_buff, 0x04);
}
/*=============================================================================
 FUNCTION: check_csr_version

 DESCRIPTION: Check the version data from CSR chip and compare it to Atmel FW version

 ARGUMENTS PASSED: version data

 RETURN VALUE: TRUE/FALSE
==============================================================================*/
bool check_csr_version(UINT16_T * ver_info)
{
  if(((ver_info[0] & 0xFF) == (UINT8_T)CURRENT_VERSION_IDENTIFIER)
    && ((ver_info[1] & 0xFF) == (UINT8_T)CURRENT_VERSION_PHASE)
      && ((ver_info[2] & 0xFF) == (UINT8_T)CURRENT_VERSION_MAJOR)
        && ((ver_info[3] & 0xFF) == (UINT8_T)CURRENT_VERSION_MINOR))
  {
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}
/*=============================================================================
 FUNCTION: connection_store_pair_data

 DESCRIPTION: Write pairing information of paired success BT Device
        into the codeplug

 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void connection_store_pair_data(btDevInfo *pairedData)
{
  cplg_param_t code_plug_value;

  if(!isBtQualTestMode)
  {   
     if (pairedData->profile == HSP)
     {
     
        /* write HSP BT address to codeplug */
        memcpy( &hspDev, pairedData, sizeof(btDevInfo));  
        
        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->addr[0])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_AUDIO_ADDR_1,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->addr[4])
          , 2);
        codeplugSetParameter(PAIRING_BT_AUDIO_ADDR_2,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->linkkey[0])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_AUDIO_LINKKEY_1,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->linkkey[1 * 4])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_AUDIO_LINKKEY_2,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->linkkey[2 * 4])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_AUDIO_LINKKEY_3,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->linkkey[3 * 4])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_AUDIO_LINKKEY_4,
                             code_plug_value,
                             0);

        code_plug_value = (cplg_param_t)pairedData->keyType;
        codeplugSetParameter(PAIRING_BT_AUDIO_KEYTYPE,
                             code_plug_value,
                             0);

        code_plug_value = (cplg_param_t)pairedData->pairingType;
        codeplugSetParameter(PAIRING_BT_AUDIO_PAIRTYPE,
                             code_plug_value,
                             0);

        code_plug_value = BLUETOOTH_STANDARD;
        codeplugSetParameter(PAIRING_BT_AUDIO_TRANSACTION_TYPE,
                             code_plug_value,
                             0);

         code_plug_value = AUDIO_DEVICE;
         codeplugSetParameter(PAIRING_BT_AUDIO_DEVICE_TYPE,
                             code_plug_value,
                             0);

         code_plug_value = (cplg_param_t)pairedData->IsNibblerDefaultON;
         codeplugSetParameter(PAIRING_BT_AUDIO_FAST_PTT,
                             code_plug_value,
                             0);
     }

     else if (pairedData->profile == SPP)
     {
        memcpy( &sppDev, pairedData, sizeof(btDevInfo));  

        /* write SPP BT address to codeplug */
        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->addr[0])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_PTT_ADDR_1,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->addr[4])
          , 2);
        codeplugSetParameter(PAIRING_BT_PTT_ADDR_2,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->linkkey[0])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_PTT_LINKKEY_1,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->linkkey[1 * 4])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_PTT_LINKKEY_2,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->linkkey[2 * 4])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_PTT_LINKKEY_3,
                             code_plug_value,
                             0);

        memcpy((void*)&code_plug_value
          , (void*)&(pairedData->linkkey[3 * 4])
          , sizeof(cplg_param_t));
        codeplugSetParameter(PAIRING_BT_PTT_LINKKEY_4,
                             code_plug_value,
                             0);

        code_plug_value = (cplg_param_t)pairedData->keyType;
        codeplugSetParameter(PAIRING_BT_PTT_KEYTYPE,
                             code_plug_value,
                             0);

        code_plug_value = (cplg_param_t)pairedData->pairingType;
        codeplugSetParameter(PAIRING_BT_PTT_PAIRTYPE,
                             code_plug_value,
                             0);


        code_plug_value = BLUETOOTH_STANDARD;
        codeplugSetParameter(PAIRING_BT_AUDIO_TRANSACTION_TYPE,
                             code_plug_value,
                             0);

        code_plug_value = NON_AUDIO_DEVICE;
        codeplugSetParameter(PAIRING_BT_PTT_DEVICE_TYPE,
                             code_plug_value,
                             0);

        code_plug_value = (cplg_param_t)pairedData->IsNibblerDefaultON;
        codeplugSetParameter(PAIRING_BT_PTT_FAST_PTT,
                             code_plug_value,
                             0);

      }
  }
  else
  {
    memcpy( &hspDev, pairedData, sizeof(btDevInfo));  

    /* write SPP BT address to codeplug */
    memcpy((void*)&code_plug_value
      , (void*)&(pairedData->addr[0])
      , sizeof(cplg_param_t));
    codeplugSetParameter(BQB_BT_AUDIO_ADDR_1,
                         code_plug_value,
                         0);

    memcpy((void*)&code_plug_value
      , (void*)&(pairedData->addr[4])
      , 2);
    codeplugSetParameter(BQB_BT_AUDIO_ADDR_2,
                         code_plug_value,
                         0);

    memcpy((void*)&code_plug_value
      , (void*)&(pairedData->linkkey[0])
      , sizeof(cplg_param_t));
    codeplugSetParameter(PAIRING_BQB_BT_AUDIO_LINKKEY_1,
                         code_plug_value,
                         0);

    memcpy((void*)&code_plug_value
      , (void*)&(pairedData->linkkey[1 * 4])
      , sizeof(cplg_param_t));
    codeplugSetParameter(PAIRING_BQB_BT_AUDIO_LINKKEY_2,
                         code_plug_value,
                         0);

    memcpy((void*)&code_plug_value
      , (void*)&(pairedData->linkkey[2 * 4])
      , sizeof(cplg_param_t));
    codeplugSetParameter(PAIRING_BQB_BT_AUDIO_LINKKEY_3,
                         code_plug_value,
                         0);

    memcpy((void*)&code_plug_value
      , (void*)&(pairedData->linkkey[3 * 4])
      , sizeof(cplg_param_t));
    codeplugSetParameter(PAIRING_BQB_BT_AUDIO_LINKKEY_4,
                         code_plug_value,
                         0);

    code_plug_value = (cplg_param_t)pairedData->keyType;
    codeplugSetParameter(PAIRING_BQB_BT_AUDIO_KEYTYPE,
                         code_plug_value,
                         0);

    code_plug_value = (cplg_param_t)pairedData->pairingType;
    codeplugSetParameter(PAIRING_BQB_BT_AUDIO_PAIRTYPE,
                         code_plug_value,
                         0);

  }

    /* Write to codeplug on every successful connection(MPP or Standard) */
    codeplugWriteback();
  
}
/* End of connection_store_pair_data() */

/*=============================================================================
 FUNCTION: connection_retrieve_pair_data

 DESCRIPTION: Retrieve the pairing information of the previously paired BT
        from the codeplug

 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void connection_retrieve_pair_data()
{
  unsigned char count;
  cplg_param_t cp_parameter_value;

  if( capture_time_semaphore( CONNMNG_SHARE_DB, portMAX_DELAY) == SEM_RET_SUCCESS )
  {
    if(!isBtQualTestMode)
    {
      /* read HSP Pairing Info from codeplug and load to RAM */
      codeplugGetParameter(PAIRING_BT_AUDIO_ADDR_1, &cp_parameter_value, 0);
      *((cplg_param_t * )&(hspDev.addr[0])) = cp_parameter_value;

      codeplugGetParameter(PAIRING_BT_AUDIO_ADDR_2, &cp_parameter_value, 0);
      *((short *)&(hspDev.addr[4])) = (short)(cp_parameter_value >> 16);

      codeplugGetParameter(PAIRING_BT_AUDIO_LINKKEY_1, &cp_parameter_value, 0);
      memcpy( &hspDev.linkkey[0 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BT_AUDIO_LINKKEY_2, &cp_parameter_value, 0);
      memcpy( &hspDev.linkkey[1 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BT_AUDIO_LINKKEY_3, &cp_parameter_value, 0);
      memcpy( &hspDev.linkkey[2 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BT_AUDIO_LINKKEY_4, &cp_parameter_value, 0);
      memcpy( &hspDev.linkkey[3 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BT_AUDIO_KEYTYPE, &cp_parameter_value, 0);
      hspDev.keyType = (SEC_PairingType)cp_parameter_value;

      codeplugGetParameter(PAIRING_BT_AUDIO_PAIRTYPE, &cp_parameter_value, 0);
      hspDev.pairingType = (HCI_KeyType)cp_parameter_value;

      codeplugGetParameter(PAIRING_BT_AUDIO_FAST_PTT, &cp_parameter_value, 0);
      hspDev.IsNibblerDefaultON = (HCI_KeyType)cp_parameter_value;

      for (count = 0; count < BT_ADDR_BYTES; count ++)
      {
        if (hspDev.addr[count] != 0)
        {
          headset_database_status_update(PAIRED_DEV_CONNECTING);
          hspDev.profile = HSP;
          break;
        }
        else
        {
          headset_database_status_update(DB_EMPTY);
        }
      }
      /* read SPP Pairing Info from codeplug and load to RAM */
      codeplugGetParameter(PAIRING_BT_PTT_ADDR_1, &cp_parameter_value, 0);
      *((cplg_param_t * )&(sppDev.addr[0])) = cp_parameter_value;

      codeplugGetParameter(PAIRING_BT_PTT_ADDR_2, &cp_parameter_value, 0);
      *((short *)&(sppDev.addr[4])) = (short)(cp_parameter_value >> 16);

      codeplugGetParameter(PAIRING_BT_PTT_LINKKEY_1, &cp_parameter_value, 0);
      memcpy( &sppDev.linkkey[0 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BT_PTT_LINKKEY_2, &cp_parameter_value, 0);
      memcpy( &sppDev.linkkey[1 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BT_PTT_LINKKEY_3, &cp_parameter_value, 0);
      memcpy( &sppDev.linkkey[2 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BT_PTT_LINKKEY_4, &cp_parameter_value, 0);
      memcpy( &sppDev.linkkey[3 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BT_PTT_KEYTYPE, &cp_parameter_value, 0);
      sppDev.keyType = (SEC_PairingType)cp_parameter_value;

      codeplugGetParameter(PAIRING_BT_PTT_PAIRTYPE, &cp_parameter_value, 0);
      sppDev.pairingType = (HCI_KeyType)cp_parameter_value;

      codeplugGetParameter(PAIRING_BT_PTT_FAST_PTT, &cp_parameter_value, 0);
      sppDev.IsNibblerDefaultON = (HCI_KeyType)cp_parameter_value;

      for (count = 0; count < BT_ADDR_BYTES; count ++)
      {
        if (sppDev.addr[count] != 0)
        {
          wirelessPTT_database_status_update(PAIRED_DEV_CONNECTING);
          sppDev.profile = SPP;
          break;
        }
        else
        {
          wirelessPTT_database_status_update(DB_EMPTY);
        }
      }
    }
    else
    {
       /* read HSP Pairing Info from codeplug and load to RAM */
      codeplugGetParameter(BQB_BT_AUDIO_ADDR_1, &cp_parameter_value, 0);
      *((cplg_param_t * )&(hspDev.addr[0])) = cp_parameter_value;

      codeplugGetParameter(BQB_BT_AUDIO_ADDR_2, &cp_parameter_value, 0);
      *((short *)&(hspDev.addr[4])) = (short)(cp_parameter_value >> 16);

      codeplugGetParameter(PAIRING_BQB_BT_AUDIO_LINKKEY_1, &cp_parameter_value, 0);
      memcpy( &hspDev.linkkey[0 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BQB_BT_AUDIO_LINKKEY_2, &cp_parameter_value, 0);
      memcpy( &hspDev.linkkey[1 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BQB_BT_AUDIO_LINKKEY_3, &cp_parameter_value, 0);
      memcpy( &hspDev.linkkey[2 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BQB_BT_AUDIO_LINKKEY_4, &cp_parameter_value, 0);
      memcpy( &hspDev.linkkey[3 * 4],  &cp_parameter_value, sizeof(UINT32) );

      codeplugGetParameter(PAIRING_BQB_BT_AUDIO_KEYTYPE, &cp_parameter_value, 0);
      hspDev.keyType = (SEC_PairingType)cp_parameter_value;

      codeplugGetParameter(PAIRING_BQB_BT_AUDIO_PAIRTYPE, &cp_parameter_value, 0);
      hspDev.pairingType = (HCI_KeyType)cp_parameter_value;

      for (count = 0; count < BT_ADDR_BYTES; count ++)
      {
        if (hspDev.addr[count] != 0)
        {
          headset_database_status_update(PAIRED_DEV_CONNECTING);
          hspDev.profile = HSP;
          break;
        }
        else
        {
          headset_database_status_update(DB_EMPTY);
        }
      }
    }
    release_semaphore( CONNMNG_SHARE_DB );
  }
}
/* End of connection_retrieve_pair_data() */

/*=============================================================================
 FUNCTION: bt_qualification_test_mode

 DESCRIPTION: Check the codeplug to determine if shall set the Dongle to
                     DISCOVERABLE_MODE
 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
bool bt_qualification_test_mode()
{
  cplg_param_t cp_parameter_value;

  /* To determine if Dongle shall enter to Bluetooth Qualifcation Test Mode */
  codeplugGetParameter(BT_QUALIFICATION_TEST_MODE, &cp_parameter_value, 0);
  if (cp_parameter_value == CP_SELECT_ON) /* need to change to new codeplug index */
  {
    SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_QUAL_TEST_MODE_REQUEST, NULL, 0);
    isBtQualTestMode = true;
    return 1;
  }
  /* Send to Bluetooth Manager if need to set the Dongle to "DISCOVERABLE_MODE */
  /* Discoverable mode is needed for running Bluetooth Qualification Test- PICS */
  codeplugGetParameter(DONGLE_DISCOVERABLE, &cp_parameter_value, 0);
  if (cp_parameter_value == CP_SELECT_ON)
  {
    SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCOVERABLE_MODE_REQUEST, NULL, 0);
  }
  return 0;
}
/* End of host_discovery_mode() */
/*=============================================================================
 FUNCTION: init_xnl_dev_params

 DESCRIPTION: Initialize the parameters required for XNL connection

 ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void init_xnl_dev_params()
{
  //Audio Device
  hsp_xnl_init_params.is_ack_feature_required = TRUE;
  hsp_xnl_init_params.auth_level = 0x00;
  
  //SPP Device
  spp_xnl_init_params.is_ack_feature_required = TRUE;
  spp_xnl_init_params.auth_level = 0x00;
}
/*=============================================================================
 FUNCTION: init_mpp_pkts

 DESCRIPTION:

 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void init_mpp_pkts()
{
  memcpy( &MPP_init_host_pkt.mac_address.bt_mac_byte[0], &localbtaddr, sizeof(BT_BdAddr) );

  MPP_init_host_pkt.link_key_exc.bt_ssp_oob =0;
  MPP_init_host_pkt.link_key_exc.lf_bt_128_key_push=1;
  MPP_init_host_pkt.link_key_exc.bt_link_key_exchange_method_ext_bit=0;
  MPP_init_host_pkt.link_key_exc.extension_byte=0;

  MPP_init_host_pkt.link_enc.e0_enc_type_128_56=0;
  MPP_init_host_pkt.link_enc.bt_link_enc_ext_bit=0;
  MPP_init_host_pkt.link_enc.extension_byte=0;
  MPP_init_host_pkt.continous_pairing = 1;
  MPP_init_host_pkt.mpp_config_type = 0;
  MPP_init_host_pkt.audio_modified.status=0;
  MPP_init_host_pkt.beacon_conf.tiering_code = 0;
  MPP_init_host_pkt.beacon_conf.lf_baseline_operation=0;
  MPP_init_host_pkt.beacon_conf.lf_security_capability.security_extensions_bit_0=0;
  MPP_init_host_pkt.beacon_conf.lf_security_capability.security_extensions_bit_1=0;
  MPP_init_host_pkt.beacon_conf.lf_security_capability.extension_bit=0;
  MPP_init_host_pkt.beacon_conf.extension_byte=0;

}
/* End of init_mpp_pkts()   */

#ifdef MPP_HOST

/*=============================================================================
 FUNCTION: delete_mac_list

 DESCRIPTION: Add a new entry to the linked list.

 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED:

 RETURN VALUE: success or fail
==============================================================================*/
void delete_mac_list()
{
  bt_mac_addr_lookup_node *node;
  bt_mac_addr_lookup_node *current;

  if(BT_HOST_INIT_DB_PTR_TO_MAC_LIST_HEAD != NULL)
  {
    current = BT_HOST_INIT_DB_PTR_TO_MAC_LIST_HEAD;
    node =  current;
      while(current->next!=NULL)
      {
        node =  current;
        current = current->next;
        MPP_OS_free_mem(node);
      }
      MPP_OS_free_mem(current);
      BT_HOST_INIT_DB_PTR_TO_MAC_LIST_HEAD = NULL;
      BT_HOST_DB_PTR_TO_MAC_LIST_HEAD = NULL;
  }

  return;
}
/* End of delete_mac_list()   */


/*=============================================================================
 FUNCTION: init_connection_db

 DESCRIPTION:

 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none

 NOTE: This product will set to MPP pairing always ON, all the RFCOMM will set
       to avaliable even it is connected
       
==============================================================================*/
void init_connection_db()
{ 
  if( MPP_CONNECTION_DB_MUTEX_HANDLE != NULL )
  {
    if(MPP_OS_SemaphoreTake( MPP_CONNECTION_DB_MUTEX, DB_MUTEX_WAIT_TIME) == MPP_SEM_TAKEN)
    {
      MPP_host_connection_db.system_status = 0;

      /* currently we are using one audio device (HSP) and completely blocked PTT device*/

      MPP_host_connection_db.audio_conn_avail = 1; // Allow MPP pairing always with headset
      MPP_host_connection_db.data_conn_avail = 1;  // Allow MPP pairing always with WPTT

      MPP_host_connection_db.num_rfcomm_chan = 2;

      /* For Audio Device BT connection */
      MPP_host_connection_db.rfcomm_in_system[0].rfcomm_status         = RFCOMM_AVAIL;
      MPP_host_connection_db.rfcomm_in_system[0].channel_no            = HSP_RFCOMM_CH;
      MPP_host_connection_db.rfcomm_in_system[0].device_type.device    = AUDIO_DEVICE;
      MPP_host_connection_db.rfcomm_in_system[0].profile_type.nibbler  = TRUE;
      MPP_host_connection_db.rfcomm_in_system[0].profile_type.bt_service_profile_protocol  = TRUE;

      /* For Non Audio Device BT connection */
      MPP_host_connection_db.rfcomm_in_system[1].rfcomm_status         = RFCOMM_AVAIL;
      MPP_host_connection_db.rfcomm_in_system[1].channel_no            = SPP_RFCOMM_CH;
      MPP_host_connection_db.rfcomm_in_system[1].device_type.device    = NON_AUDIO_DEVICE;
      MPP_host_connection_db.rfcomm_in_system[1].profile_type.nibbler  = TRUE;
      MPP_host_connection_db.rfcomm_in_system[1].profile_type.bt_service_profile_protocol  = TRUE;

      calc_random_link_key(&MPP_host_connection_db.link_key);
      MPP_OS_SemaphoreGive( MPP_CONNECTION_DB_MUTEX );
   }
  }
}
/* End of init_connection_db()   */

/*=============================================================================
 FUNCTION: connection_mgr_update_link_key

 DESCRIPTION: Update link key in MPP data base.

 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void connection_mgr_update_link_key()
{

  if( MPP_CONNECTION_DB_MUTEX_HANDLE != NULL )
  {
    if(MPP_OS_SemaphoreTake( MPP_CONNECTION_DB_MUTEX, DB_MUTEX_WAIT_TIME) == MPP_SEM_TAKEN)
    {
      calc_random_link_key(&BT_HOST_DB.link_key);
      MPP_OS_SemaphoreGive( MPP_CONNECTION_DB_MUTEX );
    }
  }
}

/*=============================================================================
 FUNCTION: connection_mgr_update_link_key

 DESCRIPTION:calculate random number to update link key for MPP pairing.

 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void calc_random_link_key(bt_link_key *p_bt_link_key)
{
 int index = 0;

 srand(SYSTEM_TIME);

 if(p_bt_link_key)
 {
  for(index = 0; index < MPP_ENC_LINK_KEY_SIZE; index++)
  {
   p_bt_link_key->link_key_byte[index] = rand();
   }
 }
 return;
}
/*=============================================================================
 FUNCTION: connection_mgr_update_BTAddr

 DESCRIPTION:  Update

 ARGUMENTS PASSED: BT_BdAddr

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void connection_mgr_update_BTAddr(U8 *local_btaddr)
{
  memcpy( &localbtaddr, local_btaddr, sizeof(BT_BdAddr) );
}

/*=============================================================================
 FUNCTION: new_connection_request_timer

 DESCRIPTION:  The timer for Dongle to listen to connect request from
                      remote devices during new connection.

 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void new_headset_connection_request_timer()
{
   /* Read the timer value from codeplug */
    codeplugGetParameter(TIME_LINK_ESTABLISH, &cp_parameter_value, 0);
    NewConnectionTimerValue = (BT_U16)cp_parameter_value;
    if (NewConnectionTimerValue != 0x0)
     {
       CANCEL_NEW_HEADSET_CONNECTION_REQUEST_TIMER();
       NEW_HEADSET_CONNECTION_REQUEST_TIMER(NewConnectionTimerValue, NEW_HEADSET_CONNECTION_REQUEST_TIMEOUT_ACK);  /* start the NEW CONNECTION REQUEST TIMER */
     }

   /* If the value is 0x0, new connection TOT is set to INDEFINITELY, and Dongle shall
   always listen to connection request from BT Device. */
}
/*=============================================================================
 FUNCTION: bt_chip_ping_received

 DESCRIPTION:  Received confirmation back from BT Chip, cancel the Ping timeout
              timer and schedule the next ping

 ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void bt_chip_ping_received()
{
  CANCEL_INTERCHIP_PING_TIMER();
  SEND_INTERCHIP_PING_MSG_TIMER(3000, SEND_INTERCHIP_PING_MSG);
}
/*=============================================================================
 FUNCTION: connection_clear_all_pair_data

 DESCRIPTION:  To clear paired device history

 ARGUMENTS PASSED: none

 REFERENCE ARGUMENTS PASSED: none

 RETURN VALUE: none
==============================================================================*/
void connection_clear_all_pair_data(DEV_PROFILE device)
{
  cplg_param_t code_plug_value = 0;

   if( device == HSP )
   {
     /* write HSP ( AUDIO )BT address and related codeplug fields */
     codeplugSetParameter(PAIRING_BT_AUDIO_ADDR_1, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_AUDIO_ADDR_2, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_AUDIO_LINKKEY_1, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_AUDIO_LINKKEY_2, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_AUDIO_LINKKEY_3, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_AUDIO_LINKKEY_4, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_AUDIO_KEYTYPE, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_AUDIO_PAIRTYPE, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_AUDIO_TRANSACTION_TYPE, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_AUDIO_DEVICE_TYPE, 1, 0);    /* 1 is the default parameter to indicate that it is BT Audio Device  */
     codeplugSetParameter(PAIRING_BT_AUDIO_FAST_PTT, code_plug_value, 0);
     codeplugSetParameter(GW_LEGACY_BEHAVIOR, 1, 0);    /* 1 is the default parameter to indicate GW will follow legacy behavior - gw_legacy_behavior = TRUE*/
     
     /* Reset connection availability in MPP database */
     if( capture_time_semaphore( CONNMNG_SHARE_DB, portMAX_DELAY) == SEM_RET_SUCCESS )
     {
       headset_database_status_update(DB_EMPTY);
       memset( &hspDev, 0x00, sizeof(hspDev) );
       memset( &temp_hspDev, 0x00, sizeof(temp_hspDev) );
       hspDev.profile = HSP;   
       temp_hspDev.profile = HSP;   
       release_semaphore( CONNMNG_SHARE_DB );
     }
     SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_CLEAR_PAIRING_INFO, &temp_hspDev, sizeof(btDevInfo));
     SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &temp_hspDev, sizeof(btDevInfo));
   }
   else if( device == SPP )
   {
     /* write SPP ( NON_AUDIO ) BT address and related codeplug fields */
     codeplugSetParameter(PAIRING_BT_PTT_ADDR_1, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_ADDR_2, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_LINKKEY_1, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_LINKKEY_2, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_LINKKEY_3, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_LINKKEY_4, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_KEYTYPE, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_PAIRTYPE, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_TRANSACTION_TYPE, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_DEVICE_TYPE, code_plug_value, 0);
     codeplugSetParameter(PAIRING_BT_PTT_FAST_PTT, code_plug_value, 0);

     /* Reset connection availability in MPP database */
     if( capture_time_semaphore( CONNMNG_SHARE_DB, portMAX_DELAY) == SEM_RET_SUCCESS )
     {
       wirelessPTT_database_status_update(DB_EMPTY);
       memset( &sppDev, 0x00, sizeof(sppDev) );
       memset( &temp_sppDev, 0x00, sizeof(temp_sppDev) );
       sppDev.profile = SPP;
       temp_sppDev.profile = SPP;
       release_semaphore( CONNMNG_SHARE_DB );
     }
     SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_CLEAR_PAIRING_INFO, &temp_sppDev, sizeof(btDevInfo));
     SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &temp_sppDev, sizeof(btDevInfo));
   }
   else
   {
     fatal_error_reset();
   }

   codeplugWriteback();

}
/*=============================================================================
	FUNCTION: new_wirelessPTT_connection_request_timer

	DESCRIPTION:  The timer for Dongle to listen to connect request from
                      remote devices during new connection.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void new_wirelessPTT_connection_request_timer()
{
   /* Read the timer value from codeplug */
    codeplugGetParameter(TIME_LINK_ESTABLISH, &cp_parameter_value, 0);
    NewConnectionTimerValue = (BT_U16)cp_parameter_value;
    if (NewConnectionTimerValue != 0x0)
     {
       CANCEL_NEW_WIRELESSPTT_CONNECTION_REQUEST_TIMER();
       NEW_WIRELESSPTT_CONNECTION_REQUEST_TIMER(NewConnectionTimerValue, NEW_WIRELESSPTT_CONNECTION_REQUEST_TIMEOUT_ACK);  /* start the NEW CONNECTION REQUEST TIMER */
     }

   /* If the value is 0x0, new connection TOT is set to INDEFINITELY, and Dongle shall
   always listen to connection request from BT Device. */
}
/*=============================================================================
	FUNCTION: mpp_add_node_to_mac_list

	DESCRIPTION: Add a new entry to the linked list.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE: None

	Note: This function will not call for this product, as it will need to enable MPP always
==============================================================================*/
void mpp_add_node_to_mac_list( btDevInfo *devinfo)
{
bt_mac_addr_lookup_node *node;
bt_mac_addr_lookup_node *current;
UNSIGNED_8BIT count;
bt_mac_address bt_add;
 
    for (count =0;count<MPP_BT_MAC_BYTES;count++)
    {
      bt_add.bt_mac_byte[count] = devinfo->addr[(MPP_BT_MAC_BYTES-count)-1];
    }

    if ( node = (bt_mac_addr_lookup_node *)pvPortMalloc(sizeof(bt_mac_addr_lookup_node)) )
    {
        current = BT_HOST_DB_PTR_TO_MAC_LIST_HEAD;
         for (count =0;count<MPP_BT_MAC_BYTES;count++)
         {
         node->device_mac.bt_mac_byte[count] = bt_add.bt_mac_byte[count];
         }
         node->next = NULL;

        if( current == NULL )
        {
        BT_HOST_DB_PTR_TO_MAC_LIST_HEAD = node;
        BT_HOST_INIT_DB_PTR_TO_MAC_LIST_HEAD = node;
        }else
        {
            while (current->next != NULL )
            {
            current = current->next;
            }
        current->next=node;
        }

    }
}
/*=============================================================================
	FUNCTION: mpp_delete_node_to_mac_list

	DESCRIPTION: Delete a entry from the linked list.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE: None
==============================================================================*/
void mpp_delete_node_from_mac_list( btDevInfo *devinfo)
{
 bt_mac_addr_lookup_node *current = BT_HOST_DB_PTR_TO_MAC_LIST_HEAD;
 bt_mac_addr_lookup_node *pre = BT_HOST_DB_PTR_TO_MAC_LIST_HEAD;
 UNSIGNED_8BIT count;
 bool found = 1;
 bt_mac_address bt_add;

 current = BT_HOST_DB_PTR_TO_MAC_LIST_HEAD;

 if ( current != NULL)
 {
     for (count =0;count<MPP_BT_MAC_BYTES;count++)
     {
       bt_add.bt_mac_byte[count] = devinfo->addr[(MPP_BT_MAC_BYTES-count)-1];
     }

     do
     {
       found = 1;
       if(current->device_mac.bt_mac_byte[0] == bt_add.bt_mac_byte[0])
       {
         for (count =1;count<MPP_BT_MAC_BYTES;count++)
         {
           if(current->device_mac.bt_mac_byte[count]!= bt_add.bt_mac_byte[count])
           {
             found = 0;
           }
         }

        
         if( found )
         {
            if( current == BT_HOST_DB_PTR_TO_MAC_LIST_HEAD)/* if first device */
            {
              BT_HOST_DB_PTR_TO_MAC_LIST_HEAD = current->next;
              BT_HOST_INIT_DB_PTR_TO_MAC_LIST_HEAD =  current->next;
              MPP_OS_free_mem(current);
             }else
            {
              pre->next = current->next;
              MPP_OS_free_mem(current);
             }
            return;
         }
        }
       pre = current;
      current = current->next;
     }while(current != NULL);
 }

}

/*=============================================================================
	FUNCTION: reject_pair_handling

	DESCRIPTION:  This function is to check for the cause of reject pair. 

	ARGUMENTS PASSED: mpp_reject_info

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void reject_pair_handling(host_reject_mpp * mpp_reject_info)
{
    
 switch ((mpp_reject_info)->reject.reject_code)
 {
  case MPP_ERR_BAD_MESSAGE:
    break;

  case MPP_ERR_NO_DATA_LINK:      
  case MPP_ERR_NO_AUDIO_LINK:     
  case MPP_ERR_NO_RADIO_SERVICE:
    fatal_error_reset();
    break;

  default:
    break;
  }
}
#endif
/*=============================================================================
  FUNCTION: bt_disconnect_reset

  DESCRIPTION:  On a BT Disconnect do a watch dog reset

  ARGUMENTS PASSED: none

  RETURN VALUE: none
==============================================================================*/
void bt_disconnect_reset(void)
{
  #ifdef DISPLAY_DATABASE_HANDLING_DEBUG
  send_debug_info((char *) __FUNCTION__, "BT DISCONNECT RESET CASE\0", get_time_stamp(), PRIORITY_LOW);
  #endif
        
  StateTransaction(BT_DISCONNECTED_RESET_STATE_UPDATE,&ConnectionManagerState,BT_DISCONNECTED_RESET_STATE);
  SEND_MESSAGE_TO_CONNECTION_MNGR(BT_DISCONNECT_SYSTEM_RESET, NULL, 0);
}
/*=============================================================================
  FUNCTION: fatal_error_reset

  DESCRIPTION:  fatal error handling

  ARGUMENTS PASSED: none

  REFERENCE ARGUMENTS PASSED: none

  RETURN VALUE: none
==============================================================================*/
void fatal_error_reset(void)
{
      #ifdef DISPLAY_DATABASE_HANDLING_DEBUG
      send_debug_info((char *) __FUNCTION__, "EXCEPTION CASE\0", get_time_stamp(), PRIORITY_LOW);
      #endif
          
      StateTransaction(FATAL_ERROR_STATE_UPDATE,&ConnectionManagerState,FATAL_ERROR_STATE); /* Update LED Indication */
      SEND_MESSAGE_TO_CONNECTION_MNGR(SYSTEM_WARM_RESET, NULL, 0);
}

/*=============================================================================
  FUNCTION: headset_database_status_update

  DESCRIPTION:  headset_database_status_update

  ARGUMENTS PASSED: device_status

  REFERENCE ARGUMENTS PASSED: none

  RETURN VALUE: none
==============================================================================*/
void headset_database_status_update(device_status newstatus)
{

  #ifdef DISPLAY_DATABASE_HANDLING_DEBUG
  switch(newstatus)
  {
    case DB_EMPTY:
      send_debug_info((char *) __FUNCTION__, "DB_EMPTY\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case NEW_DEV_CONNECTING:
      send_debug_info((char *) __FUNCTION__, "NEW_DEV_CONNECTING\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case PAIRED_DEV_CONNECTING:
      send_debug_info((char *) __FUNCTION__, "PAIRED_DEV_CONNECTING\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case DEV_CONNECTED:
      send_debug_info((char *) __FUNCTION__, "DEV_CONNECTED\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case SECOND_DEV_PAIRED_TOUCH_ACTION:
      send_debug_info((char *) __FUNCTION__, "SECOND_DEV_PAIRED_TOUCH_ACTION\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case SECOND_DEV_PAIRED_SWAP:
      send_debug_info((char *) __FUNCTION__, "SECOND_DEV_PAIRED_SWAP\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case SPP_SWITCH_TO_HSP:                  
      send_debug_info((char *) __FUNCTION__, "SPP_SWITCH_TO_HSP\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case HSP_SWITCH_TO_SPP:
      send_debug_info((char *) __FUNCTION__, "HSP_SWITCH_TO_SPP\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case DEV_DISCONNECTING:
      send_debug_info((char *) __FUNCTION__, "DEV_DISCONNECTING\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BLUETOOTH_DATABASE_ERROR:
      send_debug_info((char *) __FUNCTION__, "BLUETOOTH_DATABASE_ERROR\0", get_time_stamp(), PRIORITY_LOW);
      break;
    
    case DEV_PTT_PRESS_DISCONNECTING:
      send_debug_info((char *) __FUNCTION__, DEV_PTT_PRESS_DISCONNECTING\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    default:
     send_debug_info((char *) __FUNCTION__, "defautl\0", get_time_stamp(), PRIORITY_LOW);
     break;

  } 
  #endif

  hspstatus = newstatus;
  return;
}

/*=============================================================================
  FUNCTION: wirelessPTT_database_status_update

  DESCRIPTION:  wirelessPTT_database_status_update

  ARGUMENTS PASSED: device_status

  REFERENCE ARGUMENTS PASSED: none

  RETURN VALUE: none
==============================================================================*/
void wirelessPTT_database_status_update(device_status newstatus)
{

  #ifdef DISPLAY_DATABASE_HANDLING_DEBUG
  switch(newstatus)
  {
    case DB_EMPTY:
      send_debug_info((char *) __FUNCTION__, "DB_EMPTY\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case NEW_DEV_CONNECTING:
      send_debug_info((char *) __FUNCTION__, "NEW_DEV_CONNECTING\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case PAIRED_DEV_CONNECTING:
      send_debug_info((char *) __FUNCTION__, "PAIRED_DEV_CONNECTING\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case DEV_CONNECTED:
      send_debug_info((char *) __FUNCTION__, "DEV_CONNECTED\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case SECOND_DEV_PAIRED_TOUCH_ACTION:
      send_debug_info((char *) __FUNCTION__, "SECOND_DEV_PAIRED_TOUCH_ACTION\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case SECOND_DEV_PAIRED_SWAP:
      send_debug_info((char *) __FUNCTION__, "SECOND_DEV_PAIRED_SWAP\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case SPP_SWITCH_TO_HSP:                  
      send_debug_info((char *) __FUNCTION__, "SPP_SWITCH_TO_HSP\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case HSP_SWITCH_TO_SPP:
      send_debug_info((char *) __FUNCTION__, "HSP_SWITCH_TO_SPP\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case DEV_DISCONNECTING:
      send_debug_info((char *) __FUNCTION__, "DEV_DISCONNECTING\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    case BLUETOOTH_DATABASE_ERROR:
      send_debug_info((char *) __FUNCTION__, "BLUETOOTH_DATABASE_ERROR\0", get_time_stamp(), PRIORITY_LOW);
      break;
      
    default:
     send_debug_info((char *) __FUNCTION__, "defautl\0", get_time_stamp(), PRIORITY_LOW);
     break;

  } 
  #endif

  sppstatus = newstatus;
  return;
}


/*=============================================================================
  FUNCTION: clear_pairing_database

  DESCRIPTION:  clear_pairing_database

  ARGUMENTS PASSED: none

  REFERENCE ARGUMENTS PASSED: btDevInfo

  RETURN VALUE: none
==============================================================================*/
void clear_pairing_database(btDevInfo* database)
{
   if( capture_time_semaphore( CONNMNG_SHARE_DB, portMAX_DELAY) == SEM_RET_SUCCESS )
   {
     memset( database, 0x00, sizeof(btDevInfo) );
     if( database == &hspDev || database == &temp_hspDev  )
     {
       database->profile = HSP;   
     }
     else if( database == &sppDev || database == &temp_sppDev  )
     {
       database->profile = SPP;   
     } 
     release_semaphore( CONNMNG_SHARE_DB );
   }
}

/*=============================================================================
  FUNCTION: bt_connect_success_ack_hsp_handling

  DESCRIPTION:  bt_connect_success_ack_hsp_handling

  ARGUMENTS PASSED: none

  REFERENCE ARGUMENTS PASSED: btDevInfo

  RETURN VALUE: none
==============================================================================*/
void bt_connect_success_ack_hsp_handling(btDevInfo * dev)
{
  switch(ConnectionManagerState)
  {
    /**************************** BT_CONNECTING_STATE *****************************/
    case BT_CONNECTING_STATE:
      switch(hspstatus)
      {
        /* Label 2 in the state diagram */
        /* Label 3c in the state diagram */
        case NEW_DEV_CONNECTING:
          if(CompareBTAddress(temp_hspDev.addr,dev->addr))
          {
            StateTransaction(DEVICE_CONNECTED_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTED_STATE);
            CANCEL_NEW_HEADSET_CONNECTION_REQUEST_TIMER();
            connection_store_pair_data(&temp_hspDev);
            clear_pairing_database(&temp_hspDev);
            
            headset_database_status_update(DEV_CONNECTED);
            SEND_MESSAGE_TO_CONTROLS_MANAGER(AUDIO_DEVICE_CONNECTED_STATE_UPDATE, NULL, 0);       

            // XNL-XCMP new device register here
            device_1_id = xnl_xcmp_new_device_register(hsp_xnl_init_params);
          }
          else
          {
            fatal_error_reset();
          }
          break;


       /* Label 6 in the state diagram */
       case  PAIRED_DEV_CONNECTING:            
          if(CompareBTAddress(hspDev.addr,dev->addr))
          {
            StateTransaction(DEVICE_CONNECTED_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTED_STATE);
            headset_database_status_update(DEV_CONNECTED);
            SEND_MESSAGE_TO_CONTROLS_MANAGER(AUDIO_DEVICE_CONNECTED_STATE_UPDATE, NULL, 0);       

            // XNL-XCMP new device register here
            device_1_id = xnl_xcmp_new_device_register(hsp_xnl_init_params);
          }
          else
          {
            fatal_error_reset();
          }
          break;

        default:
          fatal_error_reset();
          break;
      }
      break;

    /**************************** BT_CONNECTED_STATE *****************************/
    case BT_CONNECTED_STATE:
      break;

    /**************************** BT_DISCONNECTED_STATE *****************************/
    case BT_DISCONNECTED_STATE:
      break;

    /**************************** DEFAULT *****************************/
    default:
      fatal_error_reset();
      break;
      
  }
  return;
}

/*=============================================================================
  FUNCTION: bt_connect_success_ack_spp_handling

  DESCRIPTION:  bt_connect_success_ack_spp_handling

  ARGUMENTS PASSED: none

  REFERENCE ARGUMENTS PASSED: btDevInfo

  RETURN VALUE: none
==============================================================================*/
void bt_connect_success_ack_spp_handling(btDevInfo * dev)
{
  switch(WirelessPTTState)
  {
    /**************************** BT_CONNECTING_STATE *****************************/
    case BT_CONNECTING_STATE:
      switch(sppstatus)
      {
        /* Label W2 in the state diagram */
        /* Label W5c in the state diagram */       
        case NEW_DEV_CONNECTING:
          if(CompareBTAddress(temp_sppDev.addr,dev->addr))
          {
            CANCEL_NEW_WIRELESSPTT_CONNECTION_REQUEST_TIMER();
            connection_store_pair_data(&temp_sppDev);
            clear_pairing_database(&temp_sppDev);

            wirelessPTT_database_status_update(DEV_CONNECTED);
            wirelessPTTStateTransaction(BT_CONNECTED_STATE);

            // XNL-XCMP new device register here
            device_2_id = xnl_xcmp_new_device_register(spp_xnl_init_params);
          }
          else
          {
            fatal_error_reset();
          }
          break;

        /* Label W4 in the state diagram */       
        case  PAIRED_DEV_CONNECTING:        
          if(CompareBTAddress(sppDev.addr,dev->addr))
          {
            wirelessPTTStateTransaction(BT_CONNECTED_STATE);
            wirelessPTT_database_status_update(DEV_CONNECTED);

            // XNL-XCMP new device register here
            device_2_id = xnl_xcmp_new_device_register(spp_xnl_init_params);
          }
          else
          {
            fatal_error_reset();
          }
          break;

        default:
          fatal_error_reset();
          break;
      }

    /**************************** BT_CONNECTED_STATE *****************************/
    case BT_CONNECTED_STATE:
      break;

    /**************************** BT_DISCONNECTED_STATE *****************************/
    case BT_DISCONNECTED_STATE:
      switch(sppstatus)
      {
        case PAIRED_DEV_CONNECTING:
          if(CompareBTAddress(sppDev.addr,dev->addr))
          {
            wirelessPTTStateTransaction(BT_CONNECTED_STATE);
            wirelessPTT_database_status_update(DEV_CONNECTED);

            // XNL-XCMP new device register here
            device_2_id = xnl_xcmp_new_device_register(spp_xnl_init_params);
          }
          else
          {
            fatal_error_reset();
          }

          break;
      
        default:
          fatal_error_reset();
          break;
      }
      break;
      
    /**************************** DEFAULT *****************************/
    default:
      fatal_error_reset();
      break;
  }
  return;
}


/*=============================================================================
  FUNCTION: bt_pair_success_hsp_handling

  DESCRIPTION:  bt_pair_success_hsp_handling

  ARGUMENTS PASSED: none

  REFERENCE ARGUMENTS PASSED: host_succ_mpp

  RETURN VALUE: none
==============================================================================*/
void bt_pair_success_hsp_handling(host_succ_mpp * dev)
{
  bt_database_handling(dev);
  switch(hspstatus)
  {
    /* Label 1 in the state diagram */       
    /* Label 9 in the state diagram */       
    /* Label 10 in the state diagram */    
    /* Label 12 in the state diagram */       
    case NEW_DEV_CONNECTING:
      clear_pairing_database(&hspDev);

      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
      mpp_succ_pair_event(dev);  /* MPP Pairing with BT device is success */

      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &temp_hspDev, sizeof(btDevInfo));
      StateTransaction(BT_CONNECTING_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTING_STATE);
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &temp_hspDev, sizeof(btDevInfo));
      new_headset_connection_request_timer();
      break;


    /* Label 11a in the state diagram */
    case SECOND_DEV_PAIRED_TOUCH_ACTION:
      if(gw_legacy_behavior == false)
      {
        //Change the state back to DEV_CONNECTED as for Audio Transfer we do not disconnect WRSM
        headset_database_status_update(DEV_CONNECTED);
        //Alternate Switching
        if(current_mic_spkr_used == REMOTE_MIC_SPKR)
        {
          // Send Mute request to WRSM
          SEND_MESSAGE_TO_CONNECTION_MNGR(CONNECTED_DEV_TOUCH_ACTION_MUTE, NULL, 0);
        }
        else if (current_mic_spkr_used == LOCAL_MIC_SPKR)
        {
          // Send Un-Mute request to WRSM
          SEND_MESSAGE_TO_CONNECTION_MNGR(CONNECTED_DEV_TOUCH_ACTION_UNMUTE, NULL, 0);
        }
      }
      else
      {
        SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, NULL, 0);
        SEND_HSP_DISCONNECT_DELAY_TIMER(50, HSP_DISCONNECT_REQUEST_DELAY_TIMEOUT);
      }
      break;
      
    /* Label 3a in the state diagram */
    case SECOND_DEV_PAIRED_SWAP:
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, NULL, 0);
      mpp_succ_pair_event(dev);  /* MPP Pairing with BT device is success */
      SEND_HSP_DISCONNECT_DELAY_TIMER(50, HSP_DISCONNECT_REQUEST_DELAY_TIMEOUT);
      break;

    case SPP_SWITCH_TO_HSP:
      if( ConnectionManagerState == BT_CONNECTED_STATE )
      {
        SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, NULL, 0);
        connection_clear_all_pair_data(SPP);         
        wirelessPTT_database_status_update(DB_EMPTY);

        mpp_succ_pair_event(dev);  /* MPP Pairing with BT device is success */
        headset_database_status_update(SECOND_DEV_PAIRED_SWAP);

        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &hspDev, sizeof(btDevInfo));
      }
      else
      {
        connection_clear_all_pair_data(HSP);         
        connection_clear_all_pair_data(SPP);         

        headset_database_status_update(NEW_DEV_CONNECTING);
        wirelessPTT_database_status_update(DB_EMPTY);

        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &hspDev, sizeof(btDevInfo));
        mpp_succ_pair_event(dev);  /* MPP Pairing with BT device is success */

        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &temp_hspDev, sizeof(btDevInfo));
        StateTransaction(BT_CONNECTING_STATE_UPDATE,&ConnectionManagerState,BT_CONNECTING_STATE);
        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &temp_hspDev, sizeof(btDevInfo));
        new_headset_connection_request_timer();
      }
      break;

    case PAIRED_DEV_CONNECTING:
    case DEV_PTT_PRESS_DISCONNECTING:
    case DB_EMPTY:                 
    case DEV_CONNECTED:     
    case HSP_SWITCH_TO_SPP:
    case DEV_DISCONNECTING:           
    case BLUETOOTH_DATABASE_ERROR:        
    default:
      fatal_error_reset();
      break;
  }
}

/*=============================================================================
  FUNCTION: bt_pair_success_spp_handling

  DESCRIPTION:  bt_pair_success_spp_handling

  ARGUMENTS PASSED: none

  REFERENCE ARGUMENTS PASSED: host_succ_mpp

  RETURN VALUE: none
==============================================================================*/
void bt_pair_success_spp_handling(host_succ_mpp * dev)
{
  bt_database_handling(dev);
  switch(sppstatus)
  {    
    /* Label W1 in the state diagram */       
    /* Label W6 in the state diagram */       
    /* Label W10 in the state diagram */ 
    /* Label W11 in the state diagram */ 
    case NEW_DEV_CONNECTING:
      clear_pairing_database(&sppDev);

      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &sppDev, sizeof(btDevInfo));
      mpp_succ_pair_event(dev);  /* MPP Pairing with BT device is success */

      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &temp_sppDev, sizeof(btDevInfo));
      wirelessPTTStateTransaction(BT_CONNECTING_STATE);
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &temp_sppDev, sizeof(btDevInfo));
      new_wirelessPTT_connection_request_timer();
      break;
    
    case SECOND_DEV_PAIRED_TOUCH_ACTION:
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, NULL, 0);
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_REQ_TONE_DISCONNECTED, NULL, 0);         
      SEND_SPP_DISCONNECT_DELAY_TIMER(50, SPP_DISCONNECT_REQUEST_DELAY_TIMEOUT);
      break;

    /* Label W5a in the state diagram */       
    case SECOND_DEV_PAIRED_SWAP:
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_BT_CONFIG_REQUEST_ZERO_TIMEOUT, NULL, 0);
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_REQ_TONE_DISCONNECTED, NULL, 0);      
      SEND_SPP_DISCONNECT_DELAY_TIMER(50, SPP_DISCONNECT_REQUEST_DELAY_TIMEOUT);
      mpp_succ_pair_event(dev);  /* MPP Pairing with BT device is success */
      break;

    case HSP_SWITCH_TO_SPP:
      connection_clear_all_pair_data(HSP);         
      connection_clear_all_pair_data(SPP);         

      headset_database_status_update(DB_EMPTY);
      wirelessPTT_database_status_update(NEW_DEV_CONNECTING);

      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &sppDev, sizeof(btDevInfo));
      mpp_succ_pair_event(dev);  /* MPP Pairing with BT device is success */

      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &temp_sppDev, sizeof(btDevInfo));
      wirelessPTTStateTransaction(BT_CONNECTING_STATE);
      SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &temp_sppDev, sizeof(btDevInfo));
      new_wirelessPTT_connection_request_timer();
      break;

    case PAIRED_DEV_CONNECTING:
    case DEV_PTT_PRESS_DISCONNECTING:
    case DB_EMPTY:    
    case DEV_CONNECTED:     
    case SPP_SWITCH_TO_HSP:                  
    case DEV_DISCONNECTING:           
    case BLUETOOTH_DATABASE_ERROR:
    default:
      fatal_error_reset();
      break;
   }
}


/*=============================================================================
  FUNCTION: bt_disconnected_ack_hsp_handling

  DESCRIPTION:  bt_disconnected_ack_hsp_handling

  ARGUMENTS PASSED: none

  REFERENCE ARGUMENTS PASSED: btDevInfo

  RETURN VALUE: none
==============================================================================*/
void bt_disconnected_ack_hsp_handling(btDevInfo * dev)
{
  switch(ConnectionManagerState)
  {
    /**************************** BT_CONNECTING_STATE *****************************/
    case BT_CONNECTING_STATE:
      /* EXCEPTION HANDLING */
      SEND_MESSAGE_TO_CONTROLS_MANAGER(AUDIO_DEVICE_DISCONNECTED_STATE_UPDATE, NULL, 0);
      /* protecting for new hsp device that ACL link is up and SCO link fail to establish */
      if(hspstatus == NEW_DEV_CONNECTING)
      {
        disconnect_xnl_link(device_1_id);
      }
      break;
          
    /**************************** BT_CONNECTED_STATE *****************************/
    case BT_CONNECTED_STATE:
      SEND_MESSAGE_TO_CONTROLS_MANAGER(AUDIO_DEVICE_DISCONNECTED_STATE_UPDATE, NULL, 0);
      switch(hspstatus)
      {
        /* In case of User Action on GW: PTT Press & Touch Action forget the Pairing */
        /* Label 11b in the state diagram */
        case SECOND_DEV_PAIRED_TOUCH_ACTION:
 
        /* Label 7c in the state diagram */
        case DEV_PTT_PRESS_DISCONNECTING:

          if(gw_legacy_behavior == true)    
          {
            /* cleaning up first device - as it could send a connect request back before
              processing the BT Config message */
            headset_database_status_update(DB_EMPTY);
            connection_clear_all_pair_data(HSP);    
          
            disconnect_xnl_link(device_1_id);
            null_data_nibble_rxd = false;
            /* Label W7a in the state diagram */
            /* Label W9a in the state diagram */ 
            if( WirelessPTTState == BT_CONNECTED_STATE )
            {
              SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &sppDev, sizeof(btDevInfo));
              SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &sppDev, sizeof(btDevInfo));
  
              wirelessPTT_database_status_update(DEV_DISCONNECTING);
            }  
            else
            {
              bt_disconnect_reset();
            }
          }
          break;             
        /* For these cases remember the Pairing */  
        /* exception case */
        case NEW_DEV_CONNECTING:
        case PAIRED_DEV_CONNECTING:
      
        /* Label 4 in the state diagram */
        case DEV_CONNECTED:
          disconnect_xnl_link(device_1_id);
          null_data_nibble_rxd = false;
          /* Label W7a in the state diagram */
          /* Label W9a in the state diagram */ 
          if( WirelessPTTState == BT_CONNECTED_STATE )
          {
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &sppDev, sizeof(btDevInfo));
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &sppDev, sizeof(btDevInfo));

            wirelessPTT_database_status_update(DEV_DISCONNECTING);
          }  
          else
          {
            bt_disconnect_reset();
          }
          break;        
         
        /* Label 3b in the state diagram */
        case SECOND_DEV_PAIRED_SWAP:
          /* cleaning up first device */
          headset_database_status_update(DB_EMPTY);
          clear_pairing_database(&hspDev);
          disconnect_xnl_link(device_1_id);
          null_data_nibble_rxd = false;
          /* Label W7a in the state diagram */
          /* Label W9a in the state diagram */ 
          if( WirelessPTTState == BT_CONNECTED_STATE )
          {
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_DISALLOW, &sppDev, sizeof(btDevInfo));
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DISCONNECT_REQUEST, &sppDev, sizeof(btDevInfo));

            wirelessPTT_database_status_update(DEV_DISCONNECTING);
          }  
          else
          {
            bt_disconnect_reset();
          }          
          break;

        default:
          fatal_error_reset();
          break;
      }              
      break;

    /**************************** DEFAULT *****************************/
    default:
      fatal_error_reset();
      break;
          
  }
       
}

/*=============================================================================
  FUNCTION: bt_disconnected_ack_spp_handling

  DESCRIPTION:  bt_disconnected_ack_spp_handling

  ARGUMENTS PASSED: none

  REFERENCE ARGUMENTS PASSED: btDevInfo

  RETURN VALUE: none
==============================================================================*/
void bt_disconnected_ack_spp_handling(btDevInfo * dev)
{
  switch(WirelessPTTState)
  {
    /**************************** BT_CONNECTING_STATE *****************************/
    case BT_CONNECTING_STATE:
      /* EXCEPTION HANDLING */
      /* protecting for new hsp device that ACL link is up and SCO link fail to establish */
      if(sppstatus == NEW_DEV_CONNECTING)
      {
        disconnect_xnl_link(device_2_id);
      }
      break;
      
    /**************************** BT_CONNECTED_STATE *****************************/
    case BT_CONNECTED_STATE:
      switch(sppstatus)
      {
        /* exception case */
        case NEW_DEV_CONNECTING:
        case PAIRED_DEV_CONNECTING:

        case SECOND_DEV_PAIRED_TOUCH_ACTION:
          /* cleaning up first device */
          wirelessPTT_database_status_update(DB_EMPTY);
          connection_clear_all_pair_data(SPP);         
          disconnect_xnl_link(device_2_id);
          wirelessPTTStateTransaction(PAIRING_STANDBY_STATE);
          break;
          

        /* Label W5b in the state diagram */       
        case SECOND_DEV_PAIRED_SWAP:
          /* cleaning up first device */
          wirelessPTT_database_status_update(DB_EMPTY);
          clear_pairing_database(&sppDev);
          disconnect_xnl_link(device_2_id);

          /* preparing to establish connection with second device */
          wirelessPTT_database_status_update(NEW_DEV_CONNECTING);
          
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_PAIRING_INFO, &temp_sppDev, sizeof(btDevInfo));
          wirelessPTTStateTransaction(BT_CONNECTING_STATE);
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER(BT_DEV_CONNECTION_REQUEST_ALLOW, &temp_sppDev, sizeof(btDevInfo));
          new_wirelessPTT_connection_request_timer();
          break;

        /* Label W3 in the state diagram */       
        /* Label W7b in the state diagram */   
        /* Label W9b in the state diagram */ 
        /* Label W12c in the state diagram */ 
        case DEV_CONNECTED:
        case DEV_DISCONNECTING:
          wirelessPTT_database_status_update(PAIRED_DEV_CONNECTING);
          wirelessPTTStateTransaction(BT_DISCONNECTED_STATE);
          //Send the disconnected tone request to audio device
          SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_REQ_TONE_DISCON_CLN_XNL_LINK, NULL, 0);   
          break;

        case DEV_PTT_PRESS_DISCONNECTING:
        default:
          fatal_error_reset();
          break;
      }
      break;

    /**************************** DEFAULT *****************************/
    default:
      fatal_error_reset();
      break;
  }
}

                       
void audio_transfer_state_machine(unsigned char msg_id)
{
  switch(audio_transfer_state)
  {
  case GW_NOT_CONNECTED_LOCAL_AUDIO:
    {
      switch(msg_id)
      {
  /*    case BT_CONNECT_SUCCESS_ACK:
        {
          //Handled in the controls task 
        }
        break;
  */
      case USER_PTT_PRESS_DETECTED_ACK:
        {
          select_mic_spkr(LOCAL_MIC_SPKR);          
          //Stop MPP Pairing
          SEND_MESSAGE_TO_MPP_MNGR(SUSPEND_MPP, NULL, MESSAGE_SIZE_ZERO);
          //Update the variable for GW PTT press if GW in Connecting state
          if(ConnectionManagerState == BT_CONNECTING_STATE)
          {
            wrsm_oor_gw_ptt_pressed = TRUE;
          }
        }
        break;
      case USER_PTT_RELEASE_DETECTED_ACK: 
        {
          //Restart MPP Pairing
          SEND_MESSAGE_TO_MPP_MNGR(RESUME_MPP, NULL, MESSAGE_SIZE_ZERO);
        }
        break;
      default:
          break;
      }
      break;
    }
  case GW_CONNECTED_LOCAL_AUDIO:
    {
      switch(msg_id)
      {
   /* case WRSM_PTT_PRESSED: Action taken at the lower layer of nibbler protocol for immediate transfer
        //Switches to Remote Audio
        break;
   */
   /*  //Handled in the controls task    
      case BT_DISCONNECTED_ACK: 
        //Switches to Local Audio          
        break;
   */     
      case USER_PTT_PRESS_DETECTED_ACK:
        {
          select_mic_spkr(LOCAL_MIC_SPKR);
          //Stop MPP Pairing
          SEND_MESSAGE_TO_MPP_MNGR(SUSPEND_MPP, NULL, MESSAGE_SIZE_ZERO);
          audio_transfer_state = GW_CONNECTED_PTT_PRESSED_LOCAL_AUDIO;
        }
        break;
      case SWITCH_TO_LOCAL_AUDIO:
      case USER_PTT_RELEASE_DETECTED_ACK:   
        {
          //Restart MPP Pairing
          SEND_MESSAGE_TO_MPP_MNGR(RESUME_MPP, NULL, MESSAGE_SIZE_ZERO);          
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, SWITCH_TO_LOCAL_AUDIO, 20, NULL);
          }
        }
        break;      
      case CONNECTED_DEV_TOUCH_ACTION_UNMUTE:
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, UNMUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, CONNECTED_DEV_TOUCH_ACTION_UNMUTE, 20, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_ON_MSG_SEND_FAILED;
          }      
          else
          {
            // Use Remote Mic as user brought same device again near GW
            select_mic_spkr(REMOTE_MIC_SPKR);
            //Change State
            audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;               
          }
        } 
        break;
      case CONNECTED_WPTT_PTT_PRESS_ROUTE_AUDIO:
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, ROUTE_AUDIO) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, CONNECTED_WPTT_PTT_PRESS_ROUTE_AUDIO, 20, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_ON_MSG_SEND_FAILED;
          }      
          else
          {
            // Use Remote Mic
            select_mic_spkr(REMOTE_MIC_SPKR);
            //Change State
            audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;               
          }
        } 
        break;        
      case HS_USER_SPKR_UNMUTED_RQST: 
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_rply(device_1_id, XCMP_MSG_RESULT_SUCCESS, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, HS_USER_SPKR_UNMUTED_RQST, 25, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_ON_MSG_SEND_FAILED;                
          }            
          else
          {
            /* Request to Mute the Local Speaker from Remote Device */
            select_mic_spkr(REMOTE_MIC_SPKR);  
             //Change State
            audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;              
          }
        }
        break;  

      case HSP_SW_VER_REPLY_RCVD_MSG: //Send Speaker OFF message after SW Version Request message is sent out
        {
          //Send Mute to WRSM if GW PTT was pressed
          if(wrsm_oor_gw_ptt_pressed == TRUE)
          {
            select_mic_spkr(LOCAL_MIC_SPKR);
            //Reset the OOR GW PTT Pressed variable as we used it
            wrsm_oor_gw_ptt_pressed = FALSE;
            SEND_MESSAGE_TO_CONNECTION_MNGR(SWITCH_TO_LOCAL_AUDIO, NULL, 0);      
          }
        }
        break;
        
      default:
        break;
      }
      break;
    }
  case GW_CONNECTED_REMOTE_AUDIO:
    {
      switch(msg_id)
      {
      case USER_PTT_PRESS_DETECTED_ACK:
        {
          select_mic_spkr(LOCAL_MIC_SPKR);          
          //Stop MPP Pairing
          SEND_MESSAGE_TO_MPP_MNGR(SUSPEND_MPP, NULL, MESSAGE_SIZE_ZERO);
          if(isHSPttPress == true)
          {
            //Change the variable value for HS PTT Press
            wirelessR1_3ptt_handler(false);
            //Cancel the Nibbler Timeout Timer
            Cancel_HS_NibbleTimer();
            //Send Bad Bonk to WRSM 
            if( device_1_id == 0 ) break;
            if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_START, NEGATIVE_INDICATION_TONE, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
            {
              RETRY_XCMP_MSG_TIMER(HSP_BAD_BONK_TONE_REQ_MSG, HS_USER_PTT_PRESS_PLAY_BAD_BONK, 25, NULL);
            }          
          }
          if(isWPTTPttPress == true)
          {
            //Change the variable value for HS PTT Press
            wirelessR1_4ptt_handler(false);
            //Cancel the Nibbler Timeout Timer
            Cancel_WPTT_NibbleTimer();
            //Send Bad Bonk to WRSM 
            if( device_1_id == 0 ) break;
            if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_START, NEGATIVE_INDICATION_TONE, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
            {
              RETRY_XCMP_MSG_TIMER(HSP_BAD_BONK_TONE_REQ_MSG, HS_USER_PTT_PRESS_PLAY_BAD_BONK, 25, NULL);
            }          
          }          
          
          //Change state
          audio_transfer_state = GW_CONNECTED_PTT_PRESSED_LOCAL_AUDIO;
        }
        break;
      case CONNECTED_DEV_TOUCH_ACTION_MUTE:
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, CONNECTED_DEV_TOUCH_ACTION_MUTE, 20, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_OFF_MSG_SEND_FAILED;  
          }
          else
          {
            select_mic_spkr(LOCAL_MIC_SPKR);            
            //Change State
            audio_transfer_state = GW_CONNECTED_LOCAL_AUDIO;           
          }
        }
        break;  
      case HSP_NIBBLER_PTT_PRESS_TMO:
        {
          //Change to local mic and speaker 
          select_mic_spkr(LOCAL_MIC_SPKR);
          //Change the variable value for HS PTT Press
          wirelessR1_3ptt_handler(false);
          //Send the Audio off to WRSM 
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, HSP_NIBBLER_PTT_PRESS_TMO, 20, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_OFF_MSG_SEND_FAILED;              
          }    
          else
          {
            //Change State
            audio_transfer_state = GW_CONNECTED_LOCAL_AUDIO;  
          }
        }
        break;
   /*  //Handled in the controls task    
      case BT_DISCONNECTED_ACK: 
        //Switches to Local Audio  
        break;
   */            
        default:
          break;
      }
      break;
    }
  case GW_CONNECTED_PTT_PRESSED_LOCAL_AUDIO:
    {
      switch(msg_id)
      {
      case USER_PTT_RELEASE_DETECTED_ACK:   
        {
          //Start MPP Pairing
          SEND_MESSAGE_TO_MPP_MNGR(RESUME_MPP, NULL, MESSAGE_SIZE_ZERO);
          //Send the XCMP Message to WRSM
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, SWITCH_TO_LOCAL_AUDIO, 20, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_OFF_MSG_SEND_FAILED; 
          }
          else
          {
            CANCEL_RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG);
            //Change State
            audio_transfer_state = GW_CONNECTED_LOCAL_AUDIO;
          }
        }
        break;  
      case HS_USER_SPKR_UNMUTED_RQST: 
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_rply(device_1_id, XCMP_MSG_RESULT_FAILURE, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, HS_USER_SPKR_UNMUTED_RQST, 25, NULL);
          }            
        }
        break;
      case HS_USER_PTT_PRESS_PLAY_BAD_BONK:
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_logical_tone_control_msg_rqst(device_1_id, XCMP_MSG_TONE_CTRL_START, NEGATIVE_INDICATION_TONE, XCMP_MSG_TONE_VOL_CTRL_CURRENT_VOL) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_BAD_BONK_TONE_REQ_MSG, HS_USER_PTT_PRESS_PLAY_BAD_BONK, 25, NULL);
          }     
        }
        break;
      case HSP_SW_VER_REPLY_RCVD_MSG: //In case when GW Reconnects while GW PTT is pressed, Spkr OFF will be sent out by PTT Release message
        {
          //Change the variable state
          if(wrsm_oor_gw_ptt_pressed == TRUE)
          {
            //Reset the OOR GW PTT Pressed variable
            wrsm_oor_gw_ptt_pressed = FALSE;
          }
        }     
        break;
   /*  //Handled in the controls task    
      case BT_DISCONNECTED_ACK: 
        //Switches to Local Audio        
        break;
   */            
        default:
          break;
      }
      break;
    }
  case GW_LOCAL_AUDIO_SPKR_ON_MSG_SEND_FAILED:
    {
      switch(msg_id)
      {
      case USER_PTT_PRESS_DETECTED_ACK:
        {
          select_mic_spkr(LOCAL_MIC_SPKR);          
          //Stop MPP Pairing
          SEND_MESSAGE_TO_MPP_MNGR(SUSPEND_MPP, NULL, MESSAGE_SIZE_ZERO);
          //Cancel any Pending messages
          CANCEL_RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG);
          //Change State
          audio_transfer_state = GW_CONNECTED_PTT_PRESSED_LOCAL_AUDIO;
        }
        break;
      case CONNECTED_DEV_TOUCH_ACTION_UNMUTE:
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, UNMUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, CONNECTED_DEV_TOUCH_ACTION_UNMUTE, 20, NULL);
          }      
          else
          {
            // Use Remote Mic as user brought same device again near GW
            select_mic_spkr(REMOTE_MIC_SPKR);
            //Change State
            audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;               
          }
        } 
        break;
      case CONNECTED_WPTT_PTT_PRESS_ROUTE_AUDIO:
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, ROUTE_AUDIO) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, CONNECTED_WPTT_PTT_PRESS_ROUTE_AUDIO, 20, NULL);
          }      
          else
          {
            // Use Remote Mic
            select_mic_spkr(REMOTE_MIC_SPKR);
            //Change State
            audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;               
          }
        } 
        break;        
      case HS_USER_SPKR_UNMUTED_RQST: 
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_rply(device_1_id, XCMP_MSG_RESULT_FAILURE, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, HS_USER_SPKR_UNMUTED_RQST, 25, NULL);
          }
          else
          {
            // Use Remote Mic as user brought same device again near GW
            select_mic_spkr(REMOTE_MIC_SPKR);
            //Change State
            audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;               
          }            
        }
        break;    
   /*  //Handled in the controls task    
      case BT_DISCONNECTED_ACK: 
        //Switches to Local Audio   
        break;
   */            
      default:
        break;
      }
      break;
    }
  case GW_LOCAL_AUDIO_SPKR_OFF_MSG_SEND_FAILED:
    {
      switch(msg_id)
      {
   /* case WRSM_PTT_PRESSED: Action taken at the lower layer of nibbler protocol for immediate transfer
        //Switches to Remote Audio
        break;
   */        
   /*  //Handled in the controls task    
      case BT_DISCONNECTED_ACK: 
        //Switches to Local Audio
        break;
   */            
      case USER_PTT_PRESS_DETECTED_ACK:
        {
          select_mic_spkr(LOCAL_MIC_SPKR);          
          //Stop MPP Pairing
          SEND_MESSAGE_TO_MPP_MNGR(SUSPEND_MPP, NULL, MESSAGE_SIZE_ZERO);
          //Cancel any Pending messages
          CANCEL_RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG);
          //Change State
          audio_transfer_state = GW_CONNECTED_PTT_PRESSED_LOCAL_AUDIO;
        }
        break;        
      case SWITCH_TO_LOCAL_AUDIO:
        {
          //Start MPP Pairing
          SEND_MESSAGE_TO_MPP_MNGR(RESUME_MPP, NULL, MESSAGE_SIZE_ZERO);
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, SWITCH_TO_LOCAL_AUDIO, 20, NULL);
          }
          else
          {
            //Change State
            audio_transfer_state = GW_CONNECTED_LOCAL_AUDIO;            
          }
        }
        break;  
      case CONNECTED_DEV_TOUCH_ACTION_MUTE:
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, CONNECTED_DEV_TOUCH_ACTION_MUTE, 20, NULL);
          }
          else
          {
            select_mic_spkr(LOCAL_MIC_SPKR);
            //Change State
            audio_transfer_state = GW_CONNECTED_LOCAL_AUDIO;           
          }
        }
        break;    
      case HSP_NIBBLER_PTT_PRESS_TMO:
        {
          //Send the Audio off to WRSM 
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, HSP_NIBBLER_PTT_PRESS_TMO, 20, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_OFF_MSG_SEND_FAILED;              
          }    
          else
          {
            //Change State
            audio_transfer_state = GW_CONNECTED_LOCAL_AUDIO;  
          }
        }
        break;        
      case CONNECTED_DEV_TOUCH_ACTION_UNMUTE:
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, UNMUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, CONNECTED_DEV_TOUCH_ACTION_UNMUTE, 20, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_ON_MSG_SEND_FAILED;      
          }      
          else
          {
            // Use Remote Mic as user brought same device again near GW
            select_mic_spkr(REMOTE_MIC_SPKR);
            //Change State
            audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;               
          }
        } 
        break;
      case CONNECTED_WPTT_PTT_PRESS_ROUTE_AUDIO:
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_bdst(device_1_id, ALL_SPEAKERS, ROUTE_AUDIO) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, CONNECTED_WPTT_PTT_PRESS_ROUTE_AUDIO, 20, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_ON_MSG_SEND_FAILED;      
          }      
          else
          {
            // Use Remote Mic
            select_mic_spkr(REMOTE_MIC_SPKR);
            //Change State
            audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;               
          }
        } 
        break;   
      case HS_USER_SPKR_UNMUTED_RQST: 
        {
          if( device_1_id == 0 ) break;
          if(send_xcmp_speaker_control_msg_rply(device_1_id, XCMP_MSG_RESULT_FAILURE, ALL_SPEAKERS, MUTE_SPKR) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_SPKR_CTRL_MSG, HS_USER_SPKR_UNMUTED_RQST, 25, NULL);
            //Change State
            audio_transfer_state = GW_LOCAL_AUDIO_SPKR_ON_MSG_SEND_FAILED;
          }
          else
          {
            // Use Remote Mic as user brought same device again near GW
            select_mic_spkr(REMOTE_MIC_SPKR);
            //Change State
            audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;               
          }            
        }
        break;         
        default:
          break;
      }
    }    
    break;
    
  default:
      break;
  }
}
