/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: ams3930_driver.c
#
# --------------------------- General Description -----------------------------
# This file contains the AMS 3930 NFC chip initialization and interface functions
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/

//  #include "mpp_init.h"    //CCMPD01414087 (GM_30Dec2010) No needed here. 
#include "ams3930_driver.h"
#include <string.h>


/********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR         Date Modified Tracking Number Description
* Jaspreet Bajwa 01/08/2010    CCMPD           Initial Creation
* Jaspreet Bajwa 01/21/2010    CCMPD           Working ASM3930 init code 
* JJ Low         11/15/2010    CCMPD01412613   R2.1 MPP state machine
* WRN637         11/13/2013    CCMPD01835515   NFC oscillator trim 
*
*--------------------------- End of History Template-----------------------------
* *****************************************************************************/

/*==============================================================================
                                      EXTERNAL DEFINITIONS
==============================================================================*/

/*==============================================================================
                                    GLOBAL VARIABLES
==============================================================================*/
/* Check AMS data sheet for register settings or http://compass.mot.com/go/316240755 */
// AMS Register Configuration
UNSIGNED_16BIT ams_reg_settings[AMS_MAX_REG_TO_PRG]=
{
    0x0002,   
   
    #if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE)
    0x0112,   
    #endif
    
    #if defined(MPP_OPTION_BOARD)
    0x0113,   
    #endif
    
    0x0202,  
    0x0336,  
    0x0400,   
    0x05A6,   
    0x06A9,  
    0x071a,   
    0x0800,   
    0x0900   
};


/*==============================================================================
                                      EXTERNAL FUNCTIONS
==============================================================================*/

/*==============================================================================
                                      FUNCTION PROTOTYPES
==============================================================================*/
UNSIGNED_8BIT lf_chip_read( UNSIGNED_8BIT reg_no );
void lf_chip_write(UNSIGNED_16BIT data );
void lf_listen_mode();
void lf_clear_rssi();
void lf_trim_osc();
void lf_clear_false();
void mpp_lfc_chip_init(void);
/*==============================================================================
                                      FUNCTION DEFINITIONS
==============================================================================*/
/*=============================================================================
	FUNCTION: lf_chip_read

	DESCRIPTION: This function reads a byte from the AMS chip.

	ARGUMENTS PASSED: UNSIGNED_8BIT

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: UNSIGNED_8BIT
==============================================================================*/
UNSIGNED_8BIT lf_chip_read( UNSIGNED_8BIT reg_no )
#if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE)
{
  
  UNSIGNED_16BIT data;

  HAL_MPP_SPI_INIT_MODE_16_BIT;
    // Disable Interrupts for critical code section.
  MPP_ENTER_CRITICAL;
   /*Register number and Read command are combined before sending*/
  HAL_MPP_SPI_SELECT_CHIP;
  HAL_SPI_WRITE(AMS_READ_CMD|(((UNSIGNED_16BIT)reg_no&AMS_CMD_REG_MASK)<<SHIFT_BY_EIGHT));
  HAL_SPI_READ(&data);
  HAL_MPP_SPI_UNSELECT_CHIP;
    // Re-enable Interrupts.
  MPP_EXIT_CRITICAL;
  
  return data; 
} 
#elif defined(MPP_OPTION_BOARD)
{
  
  unsigned short data;

   /*Register number and Read command are combined before sending*/
  
  data = AMS_READ_CMD|(((unsigned short)reg_no&AMS_CMD_REG_MASK)<<SHIFT_BY_EIGHT);
  spi_driver_read(0 , 0,  (unsigned char *)&data, sizeof(unsigned short));
  return data; 
} 
#endif
/* End of lf_chip_read()   */

/*=============================================================================
	FUNCTION: lf_chip_write

	DESCRIPTION: This function writes a 16bit data into the AMS chip.

	ARGUMENTS PASSED: UNSIGNED_8BIT

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void lf_chip_write(UNSIGNED_16BIT data )
#if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE)  // CCMPD01412613 (JJ 12/11/2010)
{
  HAL_MPP_SPI_INIT_MODE_16_BIT;
  // Disable Interrupts for critical code section.
  MPP_ENTER_CRITICAL;
  HAL_MPP_SPI_SELECT_CHIP;
  HAL_SPI_WRITE(data);
  HAL_MPP_SPI_UNSELECT_CHIP;
  // Re-enable Interrupts.
  MPP_EXIT_CRITICAL;
  
  return;
}
#elif defined(MPP_OPTION_BOARD)
{
  spi_driver_read(0 , 0, (unsigned char *)&data, sizeof(unsigned short));
}
#endif
/* End of lf_chip_write()   */
/*=============================================================================
	FUNCTION: lf_preset_default

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void lf_preset_default()
#if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE)
{
  
  HAL_MPP_SPI_INIT_MODE_8_BIT;
  // Disable Interrupts for critical code section.
  MPP_ENTER_CRITICAL;
  HAL_MPP_SPI_SELECT_CHIP;
 /*clear wakeup line and put AMS chip into listen mode*/
  HAL_SPI_WRITE(AMS_PRESET_DEFAULT);
  HAL_MPP_SPI_UNSELECT_CHIP;
  // Re-enable Interrupts.
  MPP_EXIT_CRITICAL;
         
}     
#elif defined(MPP_OPTION_BOARD)
{
  unsigned char buffer = AMS_PRESET_DEFAULT;

 /*clear wakeup line and put AMS chip into listen mode*/
  spi_driver_read(0 , 0, &buffer, 1);
}     
#endif
     
/* End of lf_preset_default()   */

/*=============================================================================
	FUNCTION: lf_listen_mode

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void lf_listen_mode()
#if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE)
{
  
  HAL_MPP_SPI_INIT_MODE_8_BIT;
  // Disable Interrupts for critical code section.
  MPP_ENTER_CRITICAL;
  HAL_MPP_SPI_SELECT_CHIP;
 /*clear wakeup line and put AMS chip into listen mode*/
  HAL_SPI_WRITE(AMS_LISTEN_MODE_CMD);
  HAL_MPP_SPI_UNSELECT_CHIP;
  // Re-enable Interrupts.
  MPP_EXIT_CRITICAL;
         
}     
#elif defined(MPP_OPTION_BOARD)
{
  unsigned char buffer = AMS_LISTEN_MODE_CMD;

 /*clear wakeup line and put AMS chip into listen mode*/
  spi_driver_read(0 , 0, &buffer, 1);
}     
#endif
     
/* End of lf_listen_mode()   */

/*=============================================================================
	FUNCTION: lf_clear_rssi

	DESCRIPTION: This function will configure UC3B clock.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void lf_clear_rssi()
#if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE)
{


  HAL_MPP_SPI_INIT_MODE_8_BIT;
  // Disable Interrupts for critical code section.
  
  MPP_ENTER_CRITICAL;
  HAL_MPP_SPI_SELECT_CHIP;
/* clear the RSSI registers of AMS chip*/
  HAL_SPI_WRITE(AMS_CLEAR_RSSI_CMD);
  HAL_MPP_SPI_UNSELECT_CHIP;
  // Re-enable Interrupts.
  MPP_EXIT_CRITICAL;
        
} 
#elif defined(MPP_OPTION_BOARD)
{

  unsigned char buffer = AMS_CLEAR_RSSI_CMD;

 /* clear the RSSI registers of AMS chip*/
  spi_driver_read(0 , 0, &buffer, 1);
 
} 
#endif
/* End of lf_clear_rssi()   */

/*=============================================================================
	FUNCTION: lf_trim_osc

	DESCRIPTION: This function is used to calibrate the AMS chip's internal   
                     RTC circuit. More than 65 clock clycles of 32768 Hz sent.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void lf_trim_osc()
{
#if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE)
  UNSIGNED_8BIT count;
  
  // Disable Interrupts for critical code section.
  MPP_ENTER_CRITICAL;
  
/* because of low baud rate requirement of 32768 and as SPI clock div logic is not suported by ATMEL as of date we need to reduce to clock #define  CLK_DIV_REQ will change its implementation */  

  HAL_PM_CKSEL_DIV;

  

 HAL_MPP_SPI_INIT_MODE_TRIM_OSC;
  
/*send osc trim command to AMS*/
 HAL_MPP_SPI_SELECT_CHIP;
  HAL_SPI_WRITE(AMS_TRIM_OSC_CMD);
  
  for(count=0;count<SEND_MORE_THAN_65_BITS;count++)
  {

    HAL_SPI_WRITE(AMS_LISTEN_MODE_CMD);/* keep on sending data to generate 65 clock cycles for calibration of AMS RTC*/
  }  
  
  HAL_MPP_SPI_SET_LAST_TX; /* mark this as to tell that this was last transfered byte so that chip select can be disabled*/
  HAL_MPP_SPI_UNSELECT_CHIP;
  /*restore the clock if reduced #define  CLK_DIV_REQ will change its implementation*/

  HAL_PM_CKSEL_ORI;    
 
  // Re-enable Interrupts.
  MPP_EXIT_CRITICAL;
#endif
}
/* End of lf_trim_osc()   */

/*=============================================================================
	FUNCTION: lf_clear_false

	DESCRIPTION: This function is to clear false wake-up register of AMS chip.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void lf_clear_false()
#if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE) // CCMPD01412613 (JJ 16/11/2010)
{
 

  HAL_MPP_SPI_INIT_MODE_8_BIT;
  
  // Disable Interrupts for critical code section.
  MPP_ENTER_CRITICAL;
  HAL_MPP_SPI_SELECT_CHIP;
/*clear the false wake up register*/
 HAL_SPI_WRITE(AMS_CLEAR_FALSE_CMD); 
  HAL_MPP_SPI_UNSELECT_CHIP;
  // Re-enable Interrupts.
  MPP_EXIT_CRITICAL;
    
} 
#elif defined(MPP_OPTION_BOARD)
{
  unsigned char buffer = AMS_CLEAR_FALSE_CMD;

 /*clear the false wake up register*/
  spi_driver_read(0 , 0, &buffer, 1);

} 
#endif
/* End of lf_clear_false()   */

/*=============================================================================
	FUNCTION: mpp_lfc_chip_init

	DESCRIPTION: This function is used to initialize the AMS chip for MPP
                     initialization.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/

void mpp_lfc_chip_init(void)
#if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE)  // CCMPD01412613 (JJ 16/11/2010)
{
  UNSIGNED_8BIT reg_no;
  
  lf_preset_default();
  MPP_SYSTEM_DELAY(MILLI_SEC_10);
  lf_preset_default();
  MPP_SYSTEM_DELAY(MILLI_SEC_10);
     
  HAL_MPP_SPI_INIT_MODE_16_BIT;
  
  // Disable Interrupts for critical code section.
  MPP_ENTER_CRITICAL; 
  HAL_MPP_SPI_SELECT_CHIP;

  for(reg_no=0;reg_no<AMS_MAX_REG_TO_PRG;reg_no++)
  {
  
/* pull the register initialization values from the array define above ams_reg_settings*/
    HAL_SPI_WRITE(ams_reg_settings[reg_no]); 
    MPP_SYSTEM_DELAY(MILLI_SEC_10); /*delay is used to make the chip select go disable so that AMS completes its write cycle*/
  }
 
  HAL_MPP_SPI_UNSELECT_CHIP;

  // Re-enable Interrupts.
 MPP_EXIT_CRITICAL; 
 /* clear register and put AMS into listen mode*/ 
  lf_listen_mode();
  lf_clear_rssi();
  lf_clear_false();
  /* if internal RTC used trim osc*/
  if(!(ams_reg_settings[1]&AMS_RTC_USED_MASK))
  {
  lf_trim_osc(); /*RTC calibrated herer*/
  }
}
#elif defined(MPP_OPTION_BOARD)

{
  unsigned char reg_no;
  UNSIGNED_16BIT ams_reg_settings_tmp[AMS_MAX_REG_TO_PRG];
  lf_preset_default();
  MPP_SYSTEM_DELAY(MILLI_SEC_10);
  lf_preset_default();
  MPP_SYSTEM_DELAY(MILLI_SEC_10);
   
  memcpy(ams_reg_settings_tmp, ams_reg_settings, sizeof(ams_reg_settings_tmp));
  for(reg_no=0;reg_no<AMS_MAX_REG_TO_PRG;reg_no++)
  {
  
/* pull the register initialization values from the array define above ams_reg_settings*/
	spi_driver_read(0 , 0, (unsigned char *)&ams_reg_settings_tmp[reg_no], sizeof (unsigned short));
        Delay (10); /*delay is used to make the chip select go disable so that AMS completes its write cycle*/
  }

 /* clear register and put AMS into listen mode*/ 
  lf_listen_mode();
  lf_clear_rssi();
  lf_clear_false();
  if(!(ams_reg_settings[1]&AMS_RTC_USED_MASK))
  {
  lf_trim_osc(); /*RTC calibrated herer*/
  }
 
}
#endif
/* End of mpp_lfc_chip_init()   */
