/*##############################################################################
#
# 	                      FILE SPECIFICATION
# 		COPYRIGHT 2011,2013-2015 MOTOROLA SOLUTIONS,INC. ALL RIGHTS RESERVED.
#			MOTOROLA CONFIDENTIAL RESTRICTED
#
################################################################################
#
# FILE NAME: codeplug_ui.c
#
# --------------------------- General Description -----------------------------
# This file contains the implementation code for the Codeplug. This file 
# contains functions to read and write to the Codeplug's memories.
#
#
********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR            Date Modified  CR Tracking Number  Description
* Jonathan James    05/12/2010     CCMPD01346766       Initial Creation
* YewFoong Lim      09/08/2010     CCMPD01389483       Added default values for 4 bytes of 
*                                                      new added codeplug fields
* PeiSee Toh        09/20/2010     CCMPD01393923       Catchup from 2.0 WIRELESS_HEADSET_SW_I08.00.00 and change the units of timers to miliseconds
* PeiSee Toh        09/21/2010     CCMPD01393922       Added default value for new created codeplug fields 
* PeiSee Toh        10/06/2010     CCMPD01399521       Change the default value for 1.2 codeplug fields
* PeiSee Toh        10/29/2010     CCMPD01408759       Change the default value for timers
* Tan Seng Kai      10/29/2010     CCMPD01408859       Change default codeplug value to 8 for device inquiry
* mahesh guru       11/18/2010     CCMPD01414087       Default paring method
* YewFoong          11/18/2010     CCMPD01411683       Added new parameters "BT_QUALIFICATION_TEST_MODE", "BQB_HSP_ADDR_1",
*                                                      "BQB_HSP_ADDR_2" for BT Qualification Test Mode usage
* Tan Seng Kai      12/08/2010     CCMPD01451448       Linkkey added in codeplug
* Mahes             Jan/06/2011    CCMPD01460340       Software catch-up to dongle R1.2 version R01.00.00
* PeiSee Toh        01/17/2011     CCMPD01463473       Define new codeplug fields for R2.1 Dongle 
* PeiSee Toh        01/25/2011     CCMPD01466788       Added new codeplug field to store the NFC pairing info
* PeiSee Toh        01/31/2011     CCMPD01468334       Change the parameter of codeplug field PAIRING_REC_RFCOMM_NUM_CH, PAIRING_BT_AUDIO_DEVICE_TYPE and PAIRING_BT_PTT_DEVICE_TYPE
* PeiSee Toh        02/10/2011     CCMPD01471881       Added new codeplug field of "TIME_CONNECTION" and "POLL_RATE_SETTING", changed the parameter of "TIME_RECONNECT"
* PeiSee Toh        04/15/2011     CCMPD01498419       Changed the duration of TIME_RECONNECT to infinitely
* wrn637            10/16/2013     CCMPD01826478       Audio detection handling
* WRN637            01/27/2014     CCMPD01856321       OneDot TwoDot disconnection
* WRN637            01/27/2014     CCMPD01858093       Disable Programmable btn DC Link
* Abhishek Trivedi  03/10/2014	   CCMPD01870625       CSR DSP Audio Detect Integration Changes
* Abhishek Trivedi  04/04/2014     CCMPD01880219       Allow EDR for std_pairing
* Abhishek Trivedi  04/04/2014     CCMPD01882874       Increase the Link Establishment Timeout and CSR Audio Detect Gains
* Abhishek Trivedi  04/15/2014     CCMPD01883363       Change the Audio Detection Threshold Value
* Abhishek Trivedi  04/24/2014     CCMPD01886810       Change CSR Audio Detect Gain values
* Abhishek Trivedi  05/08/2014     CCMPD01891233       Change CSR Audio Detect Gain and Threshold values
* Abhishek Trivedi  05/22/2014     CCMPD01896550       Add Nibbler PTT timeout to codeplug
* Abhishek Trivedi  10/22/2014     CCMPD01938895       Audio Switch
* Abhishek Trivedi  01/14/2015     CCMPD01958122       GW supports infinte pairing
*--------------------------- End of History Template----------------------------
*******************************************************************************/
//------------------------------------------------------------------------------
//      Include Files
//------------------------------------------------------------------------------

#include "compiler.h"

#include "string.h"
#include "flashc.h"

#include "codeplug_ui.h"
//------------------------------------------------------------------------------
//      Preprocessor Definitions
//------------------------------------------------------------------------------

/* Hidden Parameter List */
#define CODEPLUG_HEADER_INFO      0
#define CODEPLUG_CHECKSUM         1
#define CODEPLUG_TRANSID_MARKER   CODEPLUG_PARAMETER_MAX_COUNT - 1

/* Transaction ID Default Key */
#define CODEPLUG_TRANSID_DEFAULT  0x86C50000
#define CODEPLUG_TRANSID_START    0x86C50001
#define CODEPLUG_TRANSID_END      0x86C5FFFF

/******************************************************************************* 
NOTE: If CODEPLUG_DEBUG  
      Then: Automatically "fix" Codeplug with implementation defined defaults
*******************************************************************************/
//------------------------------------------------------------------------------
//      Global Variables
//------------------------------------------------------------------------------

/* Flag for implementation to store which partition is active during runtime. */
int cplg_active = 0;

/* 512 bytes (1 physical page) NVRAM used for default data */
const cplg_param_t codeplug_default[CODEPLUG_PARAMETER_MAX_COUNT] @ "CODEPLUG_DEFAULT";

/* 1024 bytes (2 physical pages) NVRAM */
const cplg_param_t codeplug_nvram[2][CODEPLUG_PARAMETER_MAX_COUNT] @ "CODEPLUG"; 

/* 512 bytes (size of one physical page) RAM */
cplg_param_t codeplug_buffer[CODEPLUG_PARAMETER_MAX_COUNT]; 

//------------------------------------------------------------------------------
//      Function Implementation
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//      Exception Trap Inline Function
//------------------------------------------------------------------------------
extern void exceptional_loop()
{
  AVR32_ENTER_CRITICAL_REGION();
  while(1);
  }
}
/*=============================================================================
	FUNCTION: codeplug_restore_from_default

	DESCRIPTION: This function will pull from the backup location where a
                     default codeplug can be stored.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none

        PRECONDITIONS: 
            A) The Codeplug is in an invalid state.           
            B) This function is called only upon initialization.
            C) The Codeplug default area must not be invalid.

        POSTCONDITIONS: 
            The Codeplug is restored to a default state.
==============================================================================*/
void codeplug_restore_from_default()
{
  AVR32_ENTER_CRITICAL_REGION();
  /* Load the default into the RAM buffer. */
  memcpy((void *)codeplug_buffer, 
         (void *)codeplug_default, 
         CODEPLUG_PARTITION_SIZE);
  
  /* Perform a double write to ensure no old data is selected. */
  codeplugWriteback();
  codeplugWriteback();  

  AVR32_LEAVE_CRITICAL_REGION();
}

/*=============================================================================
	FUNCTION: codeplug_preload_default

	DESCRIPTION: This function contains hard-coded default codeplug values
                     that can be used for debugging or to recover from a 
                     catastrophic Codeplug error that leaves the codeplug in an
                     unusable state.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none

        PRECONDITIONS: 
            A) The Codeplug is in an invalid state.           
            B) This function is called only upon initialization.

        POSTCONDITIONS: 
            The Codeplug is in a valid state.
==============================================================================*/
#ifdef CODEPLUG_DEBUG 
static void codeplug_preload_default()
{   
    /* Clear the Codeplug RAM buffer */  
    memset((void*)codeplug_buffer, 0, sizeof(CODEPLUG_PARTITION_SIZE));
    
    /* Copy the defaults into the buffer */
    codeplug_buffer[0]                              = 0x0FA386C5;    /* Codeplug Header Information */
    codeplug_buffer[1]                              = 0xFFFFFFFF;    /* Checksum */
    codeplug_buffer[RE_PAIRING_CYCLE]               = 0x1;           /* RE_PAIRING_CYCLE */
    codeplug_buffer[PAIRING_METHOD]                 = 0x1;           /* PAIRING_METHOD */ 
    codeplug_buffer[TIME_PAIRING_MODE]              = 0x0;           /* TIME_PAIRING_MODE in miliseconds unit */ 
    codeplug_buffer[TIME_RECONNECT]                 = 0x0;           /* TIME_RECONNECT in miliseconds unit */ 
    codeplug_buffer[TIME_LINK_ESTABLISH]            = 0x00002EE0;    /* TIME_LINK_ESTABLISH in miliseconds unit */
    codeplug_buffer[TIME_DEVICE_INQUIRY]            = 0x0008;        /* TIME_DEVICE_INQUIRY in seconds unit*/
    codeplug_buffer[PAIRING_BT_AUDIO_ADDR_1]        = 0x0;           /* PAIRING_BT_AUDIO_ADDR_1 */
    codeplug_buffer[PAIRING_BT_AUDIO_ADDR_2]        = 0x0;           /* PAIRING_BT_AUDIO_ADDR_2 */
    codeplug_buffer[PAIRING_BT_PTT_ADDR_1]          = 0x0;           /* PAIRING_BT_PTT_ADDR_1 */
    codeplug_buffer[PAIRING_BT_PTT_ADDR_2]          = 0x0;           /* PAIRING_BT_PTT_ADDR_2 */
    codeplug_buffer[RF_TEST_CONFIGURATION_1]        = 0x0;           /* RF_TEST_CONFIGURATION_1 */
    codeplug_buffer[RF_TEST_CONFIGURATION_2]        = 0x0;           /* RF_TEST_CONFIGURATION_2 */
    codeplug_buffer[PAIRING_REC_RFCOMM_NUM_CH]      = 0x2;           /* PAIRING_REC_RFCOMM_NUM_CH */
    codeplug_buffer[PAIRING_REC_RFCOMM_BT_AUDIO]    = 0x0;           /* PAIRING_REC_RFCOMM_BT_AUDIO */
    codeplug_buffer[PAIRING_REC_RFCOMM_BT_PTT]      = 0x0;           /* PAIRING_REC_RFCOMM_BT_PTT */
    codeplug_buffer[DONGLE_DISCOVERABLE]            = 0x0;           /* DONGLE_DISCOVERABLE */
    codeplug_buffer[BT_QUALIFICATION_TEST_MODE]     = 0x0;          /* Bluetooth Qualification Test Mode */
    codeplug_buffer[BQB_BT_AUDIO_ADDR_1]            = 0x0;          /* BQB_BT_AUDIO_ADDR_1 */
    codeplug_buffer[BQB_BT_AUDIO_ADDR_2]            = 0x0;          /* BQB_BT_AUDIO_ADDR_2 */  
    codeplug_buffer[PAIRING_BT_AUDIO_LINKKEY_1]     = 0x0;          /* PAIRING_BT_AUDIO_LINKKEY_1 */
    codeplug_buffer[PAIRING_BT_AUDIO_LINKKEY_2 ]    = 0x0;          /* PAIRING_BT_AUDIO_LINKKEY_2 */
    codeplug_buffer[PAIRING_BT_AUDIO_LINKKEY_3]     = 0x0;          /* PAIRING_BT_AUDIO_LINKKEY_3 */
    codeplug_buffer[PAIRING_BT_AUDIO_LINKKEY_4]     = 0x0;          /* PAIRING_BT_AUDIO_LINKKEY_4 */
    codeplug_buffer[PAIRING_BT_AUDIO_KEYTYPE]       = 0x0;          /* PAIRING_BT_AUDIO_KEYTYPE */
    codeplug_buffer[PAIRING_BT_AUDIO_PAIRTYPE]      = 0x0;          /* PAIRING_BT_AUDIO_PAIRTYPE */
    codeplug_buffer[PAIRING_BT_PTT_LINKKEY_1]       = 0x0;          /* PAIRING_BT_PTT_LINKKEY_1 */
    codeplug_buffer[PAIRING_BT_PTT_LINKKEY_2]       = 0x0;          /* PAIRING_BT_PTT_LINKKEY_2 */
    codeplug_buffer[PAIRING_BT_PTT_LINKKEY_3]       = 0x0;          /* PAIRING_BT_PTT_LINKKEY_3 */
    codeplug_buffer[PAIRING_BT_PTT_LINKKEY_4]       = 0x0;          /* PAIRING_BT_PTT_LINKKEY_4 */
    codeplug_buffer[PAIRING_BT_PTT_KEYTYPE]         = 0x0;          /* PAIRING_BT_PTT_KEYTYPE */
    codeplug_buffer[PAIRING_BT_PTT_PAIRTYPE]        = 0x0;          /* PAIRING_BT_PTT_PAIRTYPE */
    codeplug_buffer[PAIRING_BQB_BT_AUDIO_LINKKEY_1] = 0x0;          /* PAIRING_BQB_BT_AUDIO_LINKKEY_1 */
    codeplug_buffer[PAIRING_BQB_BT_AUDIO_LINKKEY_2] = 0x0;          /* PAIRING_BQB_BT_AUDIO_LINKKEY_2 */
    codeplug_buffer[PAIRING_BQB_BT_AUDIO_LINKKEY_3] = 0x0;          /* PAIRING_BQB_BT_AUDIO_LINKKEY_3 */
    codeplug_buffer[PAIRING_BQB_BT_AUDIO_LINKKEY_4] = 0x0;          /* PAIRING_BQB_BT_AUDIO_LINKKEY_4 */
    codeplug_buffer[PAIRING_BQB_BT_AUDIO_KEYTYPE]   = 0x0;          /* PAIRING_BQB_BT_AUDIO_KEYTYPE */
    codeplug_buffer[PAIRING_BQB_BT_AUDIO_PAIRTYPE]  = 0x0;          /* PAIRING_BQB_BT_AUDIO_PAIRTYPE */
    codeplug_buffer[PAIRING_BT_AUDIO_TRANSACTION_TYPE]   = 0x0;       /* PAIRING_BT_AUDIO_TRANSACTION_TYPE */
    codeplug_buffer[PAIRING_BT_AUDIO_DEVICE_TYPE]        = 0x1;       /* PAIRING_BT_AUDIO_DEVICE_TYPE */
    codeplug_buffer[PAIRING_BT_AUDIO_FAST_PTT]           = 0x0;       /* PAIRING_BT_AUDIO_FAST_PTT */
    codeplug_buffer[PAIRING_BT_PTT_TRANSACTION_TYPE]     = 0x0;       /* PAIRING_BT_PTT_TRANSACTION_TYPE */
    codeplug_buffer[PAIRING_BT_PTT_DEVICE_TYPE]          = 0x0;       /* PAIRING_BT_PTT_DEVICE_TYPE */
    codeplug_buffer[PAIRING_BT_PTT_FAST_PTT]             = 0x0;       /* PAIRING_BT_PTT_FAST_PTT */
    codeplug_buffer[TIME_CONNECTION]                     = 0x0;       /* TIME_CONNECTION */
    codeplug_buffer[POLL_RATE_SETTING]                   = 0x00002BF2;/* POLL_RATE_SETTING */
    codeplug_buffer[MUTE_DELAY_TIME]                     = 2000;      /* MUTE_DELAY_TIME */
    codeplug_buffer[ENABLE_ONEDOT_TWODOT_DISCONNECTION]  = 0x0;       /* ENABLE_ONEDOT_TWODOT_DISCONNECTION */
    codeplug_buffer[CSR_DSP_SPEAKER_GAIN]                = 0x09;      /* CSR_DSP_SPEAKER_GAIN */
    codeplug_buffer[CSR_DSP_MIC_GAIN]                    = 0x06;      /* CSR_DSP_MIC_GAIN */
    codeplug_buffer[CSR_DSP_AUDIO_LINE_DELAY]            = 0x0;       /* CSR_DSP_AUDIO_LINE_DELAY */   
    codeplug_buffer[CSR_DSP_AUDIO_DETECT_THRESHOLD]      = 0x00007C41;/* CSR_DSP_AUDIO_DETECT_THRESHOLD */ 
    codeplug_buffer[ENABLE_EDR_STD_PAIRING]              = 0x0;       /* ENABLE_EDR_STD_PAIRING */
    codeplug_buffer[ENABLE_HW_AUDIO_DETECT]              = 0x0;       /* ENABLE_HW_AUDIO_DETECT */
    codeplug_buffer[NIBBLER_PTT_TIMEOUT]                 = 300;       /* NIBBLER_PTT_TIMEOUT */
    codeplug_buffer[GW_LEGACY_BEHAVIOR]                  = 0x1;       /* GW_LEGACY_BEHAVIOR */
    codeplug_buffer[GW_NEVER_REPAIR]                     = 0x0;       /* GW_NEVER_REPAIR */    
    codeplug_buffer[DISABLE_GW_USB_HID]                  = 0x0;       /* DISABLE_GW_USB_HID */    
    /* CODEPLUG_TRANSID_MARKER */
    codeplug_buffer[CODEPLUG_TRANSID_MARKER] = 0x86C50000; 
  
    /* Perform a double-writeback to ensure the active and the inactive 
      partition are mirrored and both valid. */
    codeplugWriteback();
    codeplugWriteback();    
    
}
#endif

/*=============================================================================
	FUNCTION: codeplug_update_checksum

	DESCRIPTION: This function calculates the current checksum of the data
                     held within the RAM buffer currently.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none

        PRECONDITIONS: 
            A) This function call is within a CRITICAL SECTION and INTERRUPTS
            are disabled.

            B) This function is only to be used for updating the RAM buffer
               part of the Codeplug. It has no context with NVRAM except after
               the RAM buffer has been written to NVRAM.

        POSTCONDITIONS: 
            The checksum is updated in the RAM buffer.
==============================================================================*/
static void codeplug_update_checksum()
{
  cplg_param_t checksum = 0;
  
  /* Update checksum */
  for(int i = 2; i < CODEPLUG_PARAMETER_MAX_COUNT - 2; i++)
    checksum += codeplug_buffer[i]; /* Sum all parameters. */
  checksum = ~checksum; /* 1's Complement operation. */
  checksum += 1;        /* 2's Complement operation. */
  codeplug_buffer[1] = checksum; /* Copy calculated checksum to RAM buffer. */
}

/*=============================================================================
	FUNCTION: codeplug_check_checksum

	DESCRIPTION:  This function checks whether the checksum input as a 
                      parameter matches the

	ARGUMENTS PASSED: 
                      checksum - the checksum to evaluate against the current
                                 data in NVRAM.
                      block - corresponds to which Codeplug partition this 
                              check applies to.

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none

        PRECONDITIONS: 
            A) This function call is within a CRITICAL SECTION and INTERRUPTS
            are disabled.

            B) This function is only to be used when evaluating the NVRAM part
            of the Codeplug. It has no context with the RAM buffer. ie. Do not 
            pass the checksum of the RAM buffer to this function as an argument.

            C) 'block' must be 0 or 1.

        POSTCONDITIONS: none
==============================================================================*/
static bool codeplug_check_checksum(cplg_param_t checksum, int block)
{

  cplg_param_t target_checksum = 0;
  
  /* Check partition <block>'s checksum. */
  /* Sum the parameter data. */
  for(int i = 2; i < CODEPLUG_PARAMETER_MAX_COUNT - 2; i++)
    target_checksum += codeplug_nvram[block][i];
  
  /* Check between the sum and the checksum. */
  if(target_checksum + checksum == 0)
    return true; /* The data is valid by the checksum. */
  else 
    return false; /* The data is invalid by the checksum. */
}

/*=============================================================================
	FUNCTION: codeplugCompareParameter

	DESCRIPTION: This function calculates the current checksum of the data
                     held within the RAM buffer with data held within the NVRAM.                     

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: Returns 'true' if the RAM buffer value matches the NVRAM
                      value.  

        PRECONDITIONS: none

        POSTCONDITIONS: none            
==============================================================================*/
bool codeplugCompareParameter(int index)
{
  /* Compare the value stored in RAM and NVRAM and return whether they match */
  return codeplug_buffer[index] == codeplug_nvram[cplg_active][index];
}

/*=============================================================================
	FUNCTION: codeplugGetParameter

	DESCRIPTION: This function is used to extract parameters' values from
                     the Codeplug.

	ARGUMENTS PASSED:
          index - The Codeplug parameter index location in memory/storage.          
          options - Options passed to the function.

        REFERENCE ARGUMENTS PASSED:
          value - A pointer to the address of the variable the caller expects
                  the Codeplug parameter to be stored in upon return.
	
        RETURN VALUE: A status indicator of either CODEPLUG_OK or CODEPLUG_FAIL.

        PRECONDITIONS:
          A) Argument 'index' must be a valid Codeplug parameter.
          B) 'index' must be within range of the partition size of the Codeplug.
          C) Argument 'value' must not be NULL.
          D) codeplugInit must return CODEPLUG_OK prior to this call or else
             unpredictable results may occur.

        POSTCONDITIONS:
          Location pointed to by value will contain the value of the Codeplug
          parameter specified if CODEPLUG_OK is returned.
==============================================================================*/
int codeplugGetParameter(int index, cplg_param_t *value, int options)
{
  int ret = CODEPLUG_FAIL;   
  
  /* Ensure that the index is within range */
  if(index > 1 && index < CODEPLUG_PARAMETER_MAX_COUNT - 2)
  {  
    /* Fetch the value from the RAM buffer */
    *value = codeplug_buffer[index];      
    
    /* If the flag is set to check the RAM and NVRAM values stored, then
        perform the check. */
    if((options & CPLG_CHK_SYNC) 
       == CPLG_CHK_SYNC)
    {      
      if(codeplugCompareParameter(index))
      {
        /* they are in sync. */
        ret |= CODEPLUG_NVRAM_SYNC;
      }
      else
      {
        /* they are not in sync. */
        ret |= CODEPLUG_NVRAM_NSYNC;
      }      
    }  
    
    /* OR the return value to indicate that the operation completed OK */
    ret |= CODEPLUG_OK;
  }  
  
  return ret;
}

/*=============================================================================
	FUNCTION: codeplugSetParameter

	DESCRIPTION:  This function is used to modify the Codeplug parameters'
                      values.

	ARGUMENTS PASSED:
          index - The Codeplug parameter index location in memory/storage.          
          options - Options passed to the function.
          value - The value that is intended to be stored.

        REFERENCE ARGUMENTS PASSED: none
          
        RETURN VALUE: A status indicator of either CODEPLUG_OK or CODEPLUG_FAIL.

        PRECONDITIONS:
          A) Argument 'index' must be a valid Codeplug parameter.
          B) 'index' must be within range of the partition size of the Codeplug.
          C) codeplugInit must return CODEPLUG_OK prior to this call or else
             unpredictable results may occur.
        POSTCONDITION:
          A)  'value' is written to the buffer at the location where the 
              parameter is stored.
          B)  If the 'flags' specify NVRAM writeback, in addition to 
              postcondition 'A', the NVRAM will also store 'value' immediately.
              
==============================================================================*/
int codeplugSetParameter(int index, cplg_param_t value, int options)
{ 
  int ret = CODEPLUG_FAIL;
  
  /* Ensure that the index is within range */
  if(index > 1 && index < CODEPLUG_PARAMETER_MAX_COUNT - 2)
  {
    /* Update the RAM buffer and signal that the operation completed OK */
    codeplug_buffer[index] = value;
    ret |= CODEPLUG_OK;
  }
  else
  {
    /* If the index is not within range, return FAIL and exit */
    return CODEPLUG_FAIL;
  }
  
  /* If the option in the flag was set to writeback to NVRAM */
  if( ( options & CPLG_FLUSH ) == CPLG_FLUSH)
  {
    /* Writeback */
    if( codeplugWriteback() != CODEPLUG_OK )
    {
      /* The writeback did not successfully complete. Report FAIL. */
      return CODEPLUG_FAIL;
    }
    else
    {
      /* The writeback did successfully complete. Report OK. */
      ret |= CODEPLUG_OK;
    }
  }      

  return ret;
}

/*=============================================================================
	FUNCTION: codeplugInit

	DESCRIPTION:  This function initializes the Codeplug and checks the
                      validity of the Codeplug before any Codeplug operations
                      are performed.

	ARGUMENTS PASSED: none

        REFERENCE ARGUMENTS PASSED: none
          
        RETURN VALUE: A status indicator of either CODEPLUG_OK or CODEPLUG_FAIL.

        PRECONDITIONS: none

        POSTCONDITION: Codeplug is determined to be valid or invalid. This is 
                       indicated by the return value.              
==============================================================================*/
int codeplugInit()
{
  int valid_count = 0;


  int marked_active;
  int ret;

  /*  Copy the Codeplug information and checksum into RAM for both partitions.*/
  cplg_param_t header1 = codeplug_nvram[0][0];
  cplg_param_t header2 = codeplug_nvram[1][0];
  cplg_param_t checksum1 = codeplug_nvram[0][1];
  cplg_param_t checksum2 = codeplug_nvram[1][1];
  cplg_param_t transid1 = codeplug_nvram[0][CODEPLUG_TRANSID_MARKER];
  cplg_param_t transid2 = codeplug_nvram[1][CODEPLUG_TRANSID_MARKER];
  
  /* Check to see if the header information is valid for all partitions. */  
  /* Start with the first partition's block identifier */
  /* NOTE: The valid_count is OR'ed to indicate which partition is valid. */
  if(MSH(header1) == CODEPLUG_HEADER_MAGIC_NUMBER)

  { 
    /* The first partition's identifier is intact, check the second partition.*/
    if(MSH(header2) == CODEPLUG_HEADER_MAGIC_NUMBER)
    {
      /* Both partition identifiers are valid. */
      valid_count |= 2;
    }
    else
    {
      /* Only partition 1's identifier is valid. */
      valid_count |= 1; 
    }
  }
  /* Partition 1's identifier was not valid. */
  else
  {
    /* Check partition 2's identifier*/
    if(MSH(header2) == CODEPLUG_HEADER_MAGIC_NUMBER)
    {
      /* Only partition 2's identifier is valid. */
      valid_count |= 2; 
    }
    /* No partition has a valid identifier. */
    else
    {
      /* Neither partition has a valid identifier. */
      valid_count = 0; 
    }
  }
  
  /* If neither partition is marked as a Codeplug partition, then we must 
     indicate a failure immediately. */
  if(valid_count == 0)
  {    
#ifdef CODEPLUG_DEBUG
    codeplug_preload_default();
    return CODEPLUG_OK;
#else
    return CODEPLUG_FAIL;
#endif
  }
  
  /* Validate the data by reading the checksum. */
  /* Start by checking which partition(s) have valid identifiers. */
  /* If partition 1's identifier is valid... */
  if( valid_count & 1 ) 
  {
    /* Check partition 1's checksum. */    
    /* Check between the sum and the checksum. */
    if(codeplug_check_checksum(checksum1, 0) == true)
      valid_count |= 1; /* The checksum is valid for partition 1. */
    else
      valid_count ^= 1; /* The checksum is not valid for partition 1. */
  }

  /* If partition 2's identifier is valid... */


  if( valid_count & 2 )
  {
    /* Check partition 2's checksum. */
    if(codeplug_check_checksum(checksum2, 1) == true)
      valid_count |= 2; /* The checksum is valid for partition 2. */
    else
      valid_count ^= 2; /* The checksum is not valid for partition 2. */
  }
  
  /* Discover which partition is the last active partition. */
  /* If no partition is valid because the checksum failed on all... */
  if(valid_count == 0)
  {
#ifdef CODEPLUG_DEBUG
    codeplug_preload_default();
    return CODEPLUG_OK;
#else
    
    return CODEPLUG_FAIL;
#endif
  }
  /*  If at least one partition is valid by the identifier and the checksum 
      evaluated as valid. */
  else 
  {  
    /*  Check the first partition's header to determine whether it is the active 
        partition. */
    if(LSH(header1) == CODEPLUG_ACTIVE_MARKER) 
    {
      marked_active = 1;  

      /*  Check the second partition's header to determine if it is also marked 
          active. */
      if(LSH(header2) == CODEPLUG_ACTIVE_MARKER)
      { 
        /* Both partitions seem to be marked active. */
        
        /* Check the Transaction ID counter marker for tie breaker */        
        if(transid1 == CODEPLUG_TRANSID_DEFAULT || 
           transid2 == CODEPLUG_TRANSID_DEFAULT)
        {
          if(transid1 == CODEPLUG_TRANSID_DEFAULT &&
             transid2 == CODEPLUG_TRANSID_DEFAULT)
          {
            /*  Must be default case. Arbitrary to pick.
            Always the first partition will be selected.
            If there is difference between the two partitions, 
            then it shall be ignored. */
            marked_active = 1;
          }
          else /* Otherwise, it is one or the other. */
          {
            /* Is it the first? */
            if(transid1 == CODEPLUG_TRANSID_DEFAULT)
            {
              marked_active = 1;
            }
            /* Or is it the second? */
            else
            {
              marked_active = 2;
            }
          }
        }
        /* Boundary Case 1 */
        else if(transid1 == CODEPLUG_TRANSID_START && 
                transid2 == CODEPLUG_TRANSID_END)
        {  
          marked_active = 1;
        }
        /* Boundary Case 2 */
        else if(transid1 == CODEPLUG_TRANSID_END &&
                transid2 == CODEPLUG_TRANSID_START)
        {
          marked_active = 2;
        }
        /* Normal Case 1 */
        else if(transid1 < transid2)
        {
          marked_active = 2;
        }
        /* Normal Case 2 */
        else if(transid1 > transid2)
        {
          marked_active = 1;
        }
        else
        {
          /* Exception Case - Should never reach this point */
          exceptional_loop();          
        }                
      }
    }
    /* Is the second partition active? The first was not marked active. */
    else if(LSH(header2) == CODEPLUG_ACTIVE_MARKER)
    {
      //load 2 into memory
      marked_active = 2;
    }
    else 
    {
#ifdef CODEPLUG_DEBUG
      codeplug_preload_default();
      return CODEPLUG_OK;
#else
      return CODEPLUG_FAIL;
#endif
    }    
  }
    
  cplg_active = marked_active - 1;
  AVR32_ENTER_CRITICAL_REGION();
  ret = memcpy((void *)&codeplug_buffer[0], 
               (void *)&codeplug_nvram[cplg_active][0], 
               CODEPLUG_PARTITION_SIZE) 
    == &codeplug_buffer ? CODEPLUG_OK : CODEPLUG_FAIL;
  AVR32_LEAVE_CRITICAL_REGION();
  return ret;
}

/*=============================================================================
	FUNCTION: codeplugWriteback

	DESCRIPTION:  

	ARGUMENTS PASSED: none

        REFERENCE ARGUMENTS PASSED: none
          
        RETURN VALUE: A status indicator of either CODEPLUG_OK or CODEPLUG_FAIL.

        PRECONDITIONS: none

        POSTCONDITION: Codeplug is determined to be valid or invalid. This is 
                       indicated by the return value.              
==============================================================================*/
int codeplugWriteback()
{   

  cplg_param_t old_header; 

  AVR32_ENTER_CRITICAL_REGION();
  
  /* Create a mask update for the previously active header. */
  old_header = 0x0FA30000;
  
  /* Switch the active partition (implementation). */
  if(cplg_active == 0)
    cplg_active = 1;
  else
    cplg_active = 0;  
  
  /* Update checksum stored in RAM buffer before write. */
  codeplug_update_checksum();
  
  /* Increment Transaction Counter (CODEPLUG_TRANSID_MARKER). */
  if(codeplug_buffer[CODEPLUG_PARAMETER_MAX_COUNT - 1] == 0x86C5FFFF) 
     codeplug_buffer[CODEPLUG_PARAMETER_MAX_COUNT - 1] = 0x86C50001;
  else
    codeplug_buffer[CODEPLUG_PARAMETER_MAX_COUNT - 1]++;
  
  /* Write updated partition to NVRAM. */
  (void)flashc_memcpy((void *)codeplug_nvram[cplg_active], 
                      (void *)codeplug_buffer, 
                      CODEPLUG_PARTITION_SIZE, 
                      true);  
  
  AVR32_LEAVE_CRITICAL_REGION();
  
  /*  Check to determine whether the driver reports an error from 
      last operation. */
  if(flashc_is_programming_error())
  {    
    return CODEPLUG_FAIL;
  }
  /*  Otherwise, if the driver did not indicate a failure, the partition was 
      successfully written. */
  else 
  {
    /*  The write was succcessful, toggle the previously active page to 
        inactive. */
    /*  NOTE: This is done using the technique of toggling 1's to 0 without an 
        page erase. */
    flashc_memcpy((void *)&codeplug_nvram[cplg_active == 0 ? 1 : 0][0],
                  &old_header, 
                  sizeof(cplg_param_t), 
                  false);
    
    /* Check to see if the toggle operation failed. */
    /* If there was an error, indicate FAIL. */
    if(flashc_is_programming_error())
      return CODEPLUG_FAIL;
    /* The operation was successful. */
    else
      return CODEPLUG_OK;
  }
}

