/*################################################################################
#
# 						FILE SPECIFICATION
# 					COPYRIGHT 2011-2015 MOTOROLA SOLUTIONS,INC. ALL RIGHTS RESERVED.
#						MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: controls.c
#
# --------------------------- General Description -----------------------------
# This file contains all the function to initialize and report button presses.
#
#
#
********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR            Date Modified	CR Tracking Number 	Description
* Abhishek Trivedi	4/14/2010	CCMPD01331626		Initial Creation
* Abhishek Trivedi      4/27/2010       CCMPD01341119           Button Debounce Timer logic
* Abhishek Trivedi      4/30/2010       CCMPD01339858           Factory Test Command Addition
* Aneeket Patkar        5/07/2010       CCMPD01344099           Nibbler Data Integration in Headset
* Rhee Zhang            5/07/2010       CCMPD01345347           Cleanup IAR Compiler Warnings
* Aneeket Patkar       05/14/2010       CCMPD01347254           Porting to Secure Wireless
* Abhishek Trivedi     05/14/2010       CCMPD01346767           XNL Initialization
* Abhishek Trivedi     05/18/2010       CCMPD01349944           PTT_RESEND
* Abhishek Trivedi     05/25/2010       CCMPD01353007           Buttons & GPIO Processing
* Abhishek Trivedi     06/03/2010       CCMPD01356341           XNL Cleanups
* Aneeket Patkar       06/04/2010       CCMPD01355418           Add XCMP command support
* Phong Tran           06/18/2010       CCMPD01331569           Moved OS_ENTER_SWITCHING_ISR statement up
* Pei See              06/29/2010       CCMPD01373324           Initial draft for non secure wireless dongle1.2
* PeiSee Toh           08/04/2010       CCMPD01376817           Add in the infinite loop in control manager task
* Sevuhan Ramanathan   08/06/2010       CCMPD01376817           Caught up with I06.00.05 version of R2.0 to correct the PTT button interrupt
* Sevuhan Ramanathan   09/06/2010       CCMPD01386212           LED blinking for different dongle states
* Sevuhan Ramanathan   09/06/2010       CCMPD01388941           Added PTT_function,PTT_long_press_timer, PTT_press_pre_init for detecting,handling different PTT presses
* YewFoong Lim         09/07/2010       CCMPD01389483           Updated the naming convention of the inter-task messages sent between Control Manager & Connection Manager
* Sevuhan Ramanathan   09/08/2010       CCMPD01390094           Modifying LED_blink function to include atmel test mode condition set by test_commands_manager
* Tan Seng Kai	       09/14/2010       CCMPD01392415           Change bluetooth link status to active low. Low = Connected, High = not connected.
* Sevuhan Ramanathan   09/30/2010       CCMPD01397192           Change message retrieval time in control_manager_task to a smaller value to not affect the blinking rate
* Sevuhan Ramanathan   09/30/2010       CCMPD01397222           Changed PTT_function to assert/de-assert Gcai Gpio 3 irrespective of dongle state depending only on PTT status
* Sevuhan Ramanathan   10/7/2010        CCMPD01399974           modified LED blink function,added LED_timer function
* Tan Seng Kai         10/06/2010       CCMPD01399559           RADIO_INTERFACE_MANAGER_TASK,MPP_TASK,NIBBLER_TASK Compile sw
*                                                               added to remove Radio interface and mpp task
* PeiSee Toh           10/06/2010       CCMPD01397210           Modified the ptt_button_debounce function to send the PTT status to test commands manager.
* Sevuhan Ramanathan   10/06/2010       CCMPD01397694           added wireless_ptt_handler,modified ptt_function for supporting wireless PTT
* YewFoong Lim         10/06/2010       CCMPD01398513           Changed the state to PAIRING_STANDBY_STATE to start the blinking of pairing LED indication
* PeiSee Toh           10/11/2010       CCMPD01400975           Modified the ptt_button_debounce function by removing the button test mode checking
* Sevuhan Ramanathan   10/28/2010       CCMPD01408045           Fixed PTT function to send PTT press/release detection message to bluetooth manager
* Sevuhan Ramanathan   10/30/2010       CCMPD01408763           long PTT press detection checking for 3000ms press
* JJ Low               11/08/2010       CCMPD01410054           Compiler warning resolution for R2.1
* YK - JWPM67          Dec/1/2010       CCMPD01413200           (a) Catch-up to NON_SECURE_R1.2A_ITR05_I05.00.09
*                                                               (b) Created the audio_route_switch_sensing function in Control Manager.
*                                                               (c) Moved the MPP ISR to be in gpio_group2_irq1_controls_isr(void)
* YK - JWPM67          Dec/6/2010       CCMPD01450340           Checked condition DONGLE_PRESENT_ON_OPT_SEL1 before asserting RADIO_PTT_GPIO_PIN.
* Sevuhan Ramanathan   11/03/2010       CCMPD01409605           Added audio detection circuit ISR to send audio mute/unmute message to bluetooth manager based 
*                                                               on audio detection circuit output.
* YewFoong             11/18/2010       CCMPD01411683           Added LED indication update for BQB test mode, and send message to BT manager when PTT is pressed
* YewFoong             11/24/2010       CCMPD01419110           Added Long PTT Press detection during BQB mode
* Tan Seng Kai         11/25/2010       CCMPD01419120           Tc.h is needed when removing nibbler task
* Sevuhan Ramanathan   12/02/2010       CCMPD01448401           Changed LED indication for "BT_CONNECTING_STATE_UPDATE" to SOLID,Modified BQB test mode PTT functions
*                                                               and change name of PTT_PTT_long_press_timer to PTT_long_press_handler.
* Sevuhan Ramanathan   12/11/2010       CCMPD01453013           LED indication change for pairng standby state update to 1000ms ON,100 ms OFF, removal of fatal
*                                                               error state inidication.
* YewFoong             12/10/2010       CCMPD01452415           Added OFF LED for IDLE_STATE
* Mahes                Jan/06/2011      CCMPD01460340           (a) Software catup to dongle R1.2 version R01.00.00
*                                                               (b) Fixed the LED bug during Atmel Test mode.     
* Mahesh               11-Jan-2011                               Removed BT_Link Simulation.
* Sevuhan Ramanathan   01/13/2011       CCMPD01462651            Make RAD_RXD line low so that RAD_TXD stays low when connected to radio and does not start USB mode
* Sevuhan Ramanathan   01/26/2011       CCMPD01466905            PTT Press/Release checking done on the audio route switch toggled from radio to 2.1 Adapter
* Sevuhan Ramanathan   02/12/2011       CCMPD01471858            Added messages  DEVICE_CONNECTED_STATE_UPDATE,AUDIO_DEVICE_CONNECTED_STATE_UPDATE,AUDIO_DEVICE_DISCONNECTED_STATE_UPDATE
* Sevuhan Ramanathan   02/12/2011       CCMPD01472726            Modified Audio route switch detection to be based on debouncing with 3ms
* Tan Seng Kai         02/21/2011       CCMPD01475738            Change Intertask message to HEADSET_SPKR_CONTROL, and remove HEADSET_SPKR_MUTE ,HEADSET_SPKR_UNMUTE
* Sevuhan Ramanathan   03/07/2011       CCMPD01481275            BT_LED_DARK_MODE and BT_NORMAL_LED_MODE handling to enter dark mode and back to normal mode.   
* Toh Pei See          04/15/2011       CCMPD01498419            First baseline version for R2.19A by removing the audio routing switch handler, changing the PTT handler and USB initialization
* Sevuhan Ramanathan   04/20/2011       CCMPD01498417            Renamed PA28 and onhook read pin to LED_PIN & BTLINK_PIN  
* Sevuhan Ramanathan   12/05/2011       CCMPD01594785            USB detection mechanism
* Sevuhan Ramanathan   01/04/2012       CCMPD01600891            added hook_status_control(),hook_init() for hook controlling in 2.17
* Sean                 08/28/2013       CCMPD01806989            PTT to drop connection in 2.17
* WRN637               09/17/2013       CCMPD01815073            Add Button handling for R2.17 Wireless Gateway P0B board
* wrn637               09/17/2013       CCMPD01808031            Remove USB HID feature
* Abhishek Trivedi     09/02/2013       CCMPD01827271            XNL-XCMP Host Support
* wrn637               10/16/2013       CCMPD01826478            Audio detection handling
* WRN637               20/12/2013       CCMPD01833277            WPTT state machine
* WRN637               06/01/2014       CCMPD01849533            Resolve compilation warming
* Abhishek Trivedi     03/10/2014       CCMPD01870625            CSR DSP Audio Detect Integration Changes
* Abhishek Trivedi     04/11/2014       CCMPD01882874            Change in Mute delay time when PTT is pressed, Version mismatch
* Abhishek Trivedi     06/25/2014       CCMPD01906926            Audio Route (Mic Select) Fix
* Abhishek Trivedi     10/22/2014       CCMPD01938895            Audio Switch
* Abhishek Trivedi     10/24/2014       CCMPD01941655            Audio Switch Redesign
* Abhishek Trivedi     11/02/2014       CCMPD01943851            Codeplug Read Request for Talk Permit Tone issue
* Abhishek Trivedi     11/29/2014       CCMPD01957210            WPTT Support for Audio Switch
* Aneeket Patkar       12/23/2014       CCMPD01957505            Codeplug Read Request for Talk Permit Tone issue - Bug Fix
*--------------------------- End of History Template-----------------------------
* ******************************************************************************/
/* ***************************************************************************
                                HEADER INCLUDE FILES
   ************************************************************************** */
#include "controls.h"

#include "TaskConfig.h"
#include "FreeRTOS.h"
#include "task.h"
#include "bluetooth_manager.h"
#include "ConnectionManager.h"
#include "initialization.h" 
#include "codeplug_ui.h"
#include "usb_lib.h"

#ifdef RADIO_INTERFACE_MANAGER_TASK
#include "utilities.h"
#include "nibbler.h"
#include "tc.h"
#include "usart.h"
#endif


#ifdef NIBBLER_TASK
#include "nibbler_task.h"
#endif

/* ***************************************************************************
                                GLOBAL VARIABLES
   ************************************************************************** */
volatile ULONG32_T debounce_ms_tick =  PTT_BUTTON_DEBOUNCE;

unsigned char ctrl_state = DONGLE_OFF_STATE;
bool LED_init_once=FALSE;
bool reload_time = TRUE ;
bool LED_on_status = TRUE;
unsigned int init_rate=0;
bool if_long_ptt_detect_msg_sent = false;
unsigned char ctrl_state_for_led = DONGLE_OFF_STATE;  /* a copy of ctrl_state that is storing led state */ 

static unsigned char atmel_test_mode = TEST_OFF; /* Atmel Test mode (default is OFF) */  

static buttons_pin_t buttons[MAX_NUM_BTN];
static bool button_timer_flag = FALSE;    /* TRUE = Timer previously started, FALSE = Timer has not been started */

extern U32 mute_delay; 
extern cplg_param_t hw_audio_detect;
extern bool gw_legacy_behavior;
static UINT32_T ptt_press_mute_delay = 100;

bool current_mic_spkr_used = LOCAL_MIC_SPKR;
bool wrsm_oor_gw_ptt_pressed = FALSE;
bool allow_audio_xfer = TRUE;
bool LED_mode;
volatile bool local_PttPress;
volatile bool isHSPttPress;
volatile bool isWPTTPttPress;
bool  isDetAudmute = true;  /* false = unmute, true = mute */
extern bool gw_usb_hid_disabled_glb;
bool HID_One_Dot_Pressed = FALSE; 
bool HID_Two_Dot_Pressed = FALSE; 
static UINT8_T hid_report_message[] = {0,0,0,0,0,0,0,0};

#define  DARK_MODE false
#define  NORMAL_MODE true
#define  HID_ID_ONE_DOT 0x6B
#define  HID_ID_TWO_DOT 0x6C
#define  HID_ID_RELEASED 0x00
/* ***************************************************************************
                                EXTERNAL VARIABLES
   ************************************************************************** */
extern connectionManagerStateType ConnectionManagerState;
extern ms_task_msg_t controls_radio_interface_button_msg;
extern UINT16_T payload_length_remaining;
extern tc_flags_t tc_flags;
extern  xQueueHandle Controls_Radio_Interface_Q;
extern UINT32_T apply_tpt_issue_delay;

extern int usb_radio_detected_glb;
extern audio_transfer_state_t audio_transfer_state;
/* ***************************************************************************
                                EXTERNAL FUNCTIONS
   ************************************************************************** */
extern void send_button_broadcast(UINT8_T chipset, UINT8_T button_id, UINT8_T state);
/* ***************************************************************************
                                FUNCTION PROTOTYPES
   ************************************************************************** */
inline void ptt_button_debounce(UINT8_T btn_no);
inline void audio_detect_debounce(UINT8_T btn_no);
inline void button_debounce(UINT8_T btn_no);
static void radio_PTT_function(void);

void LED_blink(portTickType T_ON,portTickType T_OFF);

void PTT_function(controls_button_state_t ptt_current_button_state);

void LED_timer(portTickType time);

void setup_button_processing(void);
void one_dot_programable_button_handler(controls_button_state_t button_state);
void two_dot_programable_button_handler(controls_button_state_t button_state);
void control_manager_init(void);
void send_hid_report(void);

controls_button_state_t get_button_state(UINT8_T btn_no);

/* This is a timer for the Reverse Nibbler mute delay handling */
TimerDescriptorType MutedelayTimer;
#define    MUTE_DELAY_TIMER(time, message)      TaskCancelTimer(&MutedelayTimer); \
           MutedelayTimer.TimerMatchValue = SYSTEM_TIME + time; \
           MutedelayTimer.TaskToNotify = CONTROLS_MANAGER_TASK; \
           MutedelayTimer.MessageID = message; \
           MutedelayTimer.Message = NULL; \
           MutedelayTimer.Next = NULL; \
           ScheduleTimer(&MutedelayTimer);
           
#define CANCEL_MUTE_DELAY_TIMER()          TaskCancelTimer(&MutedelayTimer); 

/* This is a timer for the talk permit tone feedback issue delay handling */
TimerDescriptorType TPTdelayTimer;
#define    TPT_DELAY_TIMER(time, message)      TaskCancelTimer(&TPTdelayTimer); \
           TPTdelayTimer.TimerMatchValue = SYSTEM_TIME + time; \
           TPTdelayTimer.TaskToNotify = CONTROLS_MANAGER_TASK; \
           TPTdelayTimer.MessageID = message; \
           TPTdelayTimer.Message = NULL; \
           TPTdelayTimer.Next = NULL; \
           ScheduleTimer(&TPTdelayTimer);
           
#define CANCEL_TPT_DELAY_TIMER()          TaskCancelTimer(&TPTdelayTimer); 

/* This is a timer to send a HID report message */
TimerDescriptorType HidMessageTimer;
#define    HID_REPORT_TIMER(time, message)      TaskCancelTimer(&HidMessageTimer);\
           HidMessageTimer.TimerMatchValue = \
           SYSTEM_TIME + time; \
           HidMessageTimer.TaskToNotify = CONTROLS_MANAGER_TASK; \
           HidMessageTimer.MessageID = message; \
           HidMessageTimer.Message = NULL; \
           HidMessageTimer.Next = NULL; \
           ScheduleTimer(&HidMessageTimer)

#define CANCEL_HID_REPORT_TIMER()          TaskCancelTimer(&HidMessageTimer)

/*=============================================================================
	FUNCTION: controls_manager_task

	DESCRIPTION:The control manager task

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void controls_manager_task(unsigned long delay)
{
 ms_task_msg_t *ctrl_msg;
 unsigned int init_blink_num = 3;
 ConnectionManagerState = BT_QUAL_TEST_MODE; 
 LED_mode = NORMAL_MODE;
 
 while(TRUE)
 {
  ctrl_msg=(ms_task_msg_t *)Get_Msg_w_Time_Out(0x0010);//waits for 16ms to get latest msg from connection manager
  if(ctrl_msg != NULL)
    {
       ctrl_state_for_led = ctrl_state;   
       ctrl_state= ctrl_msg->ros_msg.radio_if_msg.sub_opcode; // get sub opcode 
       reload_time = TRUE;
       OS_free(ctrl_msg);
     }
  
  switch(ctrl_state)
  {
      case PTT_BUTTON_BUTTON_PRESSED:
        ctrl_state = ctrl_state_for_led;
        SEND_MESSAGE_TO_CONNECTION_MNGR(USER_PTT_PRESS_DETECTED_ACK,NULL,0);  
        break;
        
      case PTT_BUTTON_BUTTON_RELEASED:
        ctrl_state = ctrl_state_for_led;  
        SEND_MESSAGE_TO_CONNECTION_MNGR(USER_PTT_RELEASE_DETECTED_ACK,NULL,0);  
        break;

      case AUDIO_DET_ACTIVATED:
        ctrl_state = ctrl_state_for_led;  
        /* Mute message is already send to BT manager in ISR, this message just cancel mute delay timer. */        
        CANCEL_MUTE_DELAY_TIMER();
        break;

      case AUDIO_DET_DEACTIVATED:
        ctrl_state = ctrl_state_for_led; 
        if ((local_PttPress == true)
            ||(isHSPttPress == true)
            ||(isWPTTPttPress == true))   
        {
          MUTE_DELAY_TIMER(ptt_press_mute_delay,MUTE_DELAY_TIME_TIMEOUT);
        }
        else
        {
          MUTE_DELAY_TIMER(mute_delay,MUTE_DELAY_TIME_TIMEOUT);
        }
        break;

      case CSR_DSP_AUDIO_DET_ACTIVATED:
        ctrl_state = ctrl_state_for_led;  
        /* Mute message is already send to BT manager in ISR, this message just cancel mute delay timer. */        
        CANCEL_MUTE_DELAY_TIMER();
          gpio_set_gpio_pin(MIC_SELECT_GPIO_PIN);  /* select local Mic */
        CANCEL_TPT_DELAY_TIMER();
          TPT_DELAY_TIMER(apply_tpt_issue_delay, TPT_DELAY_TIMEOUT);
        break;
        
      case TPT_DELAY_TIMEOUT:
        ctrl_state = ctrl_state_for_led;
        CANCEL_TPT_DELAY_TIMER();       
          // If the Mic Speaker selection did not change while the timer was running
          if(current_mic_spkr_used == REMOTE_MIC_SPKR)
          {
            gpio_clr_gpio_pin(MIC_SELECT_GPIO_PIN);  /* select remote Mic */
          }
        break;

      case CSR_DSP_AUDIO_DET_DEACTIVATED:
        ctrl_state = ctrl_state_for_led; 
        if ((local_PttPress == true)
            ||(isHSPttPress == true)
            ||(isWPTTPttPress == true))   
        {
          MUTE_DELAY_TIMER(ptt_press_mute_delay,MUTE_DELAY_TIME_TIMEOUT);
        }
        else
        {
          MUTE_DELAY_TIMER(mute_delay,MUTE_DELAY_TIME_TIMEOUT);
        }
        break;
        
      case MUTE_DELAY_TIME_TIMEOUT:
        ctrl_state = ctrl_state_for_led; 
        if(hw_audio_detect == CP_SELECT_ON)
        {
          if( get_button_state(AUDIO_DET) == AUDIO_NOTDETECTED)
          {
            isDetAudmute = true;
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(HEADSET_SPKR_CONTROL, NULL, 0);
          }
          else if ( get_button_state(AUDIO_DET) == DEBOUNCED)
          {
            MUTE_DELAY_TIMER(10,MUTE_DELAY_TIME_TIMEOUT);
          }          
        }
        else
        {
          if( get_button_state(CSR_DSP_AUD_DET) == AUDIO_NOTDETECTED)
          {
            isDetAudmute = true;
            SEND_MESSAGE_TO_BLUETOOTH_MANAGER(HEADSET_SPKR_CONTROL, NULL, 0);
          }
          else if ( get_button_state(CSR_DSP_AUD_DET) == DEBOUNCED)
          {
            MUTE_DELAY_TIMER(10,MUTE_DELAY_TIME_TIMEOUT);
          }
        }
        break;
      
      case ONE_DOT_BUTTON_PRESSED:
        ctrl_state = ctrl_state_for_led;  
        one_dot_programable_button_handler(PRESSED);
        SEND_MESSAGE_TO_CONNECTION_MNGR(ONE_DOT_BUTTON_PRESSED,NULL,0);  
        break;

      case ONE_DOT_BUTTON_RELEASED:
        ctrl_state = ctrl_state_for_led;  
        one_dot_programable_button_handler(RELEASED);
        SEND_MESSAGE_TO_CONNECTION_MNGR(ONE_DOT_BUTTON_RELEASED,NULL,0);  
        break;

      case TWO_DOT_BUTTON_PRESSED:
        ctrl_state = ctrl_state_for_led;  
        two_dot_programable_button_handler(PRESSED);
        SEND_MESSAGE_TO_CONNECTION_MNGR(TWO_DOT_BUTTON_PRESSED,NULL,0);  
        break;
        
      case TWO_DOT_BUTTON_RELEASED:
        ctrl_state = ctrl_state_for_led;  
        two_dot_programable_button_handler(RELEASED);
        SEND_MESSAGE_TO_CONNECTION_MNGR(TWO_DOT_BUTTON_RELEASED,NULL,0);  
        break;      
  
      case DONGLE_OFF_STATE:/*dongle init state*/
      break;

      case INIT_STATE_UPDATE: /* software intiliazation success,the LED blink is 3 times */
        if(!LED_init_once)
        {      
          control_manager_init();
          while(init_rate < (init_blink_num))/*Condition to check LED blinks only 3 times*/
          {
            if(init_rate == (init_blink_num -1) )
            {
            LED_blink((portTickType)200,(portTickType)1000 );
            }
            else
            {
             LED_blink((portTickType)200,(portTickType)200 );
            }
           }
          LED_init_once=TRUE;
        }
      break;
      
    case BT_INIT_CSR_FW_MISMATCH_UPDATE:
      control_manager_init();
      LED_blink((portTickType)500,(portTickType)500 );
      break;
    case PAIRING_STANDBY_STATE_UPDATE: 
      LED_blink((portTickType)1000,(portTickType)100);
      break;

    case BT_CONNECTING_STATE_UPDATE:/* attempt to connect or reconnect with headset */
      LED_blink((portTickType)1000,(portTickType)0);
    break;

    case BT_TEST_CONNECTED:/*BQB mode message from connection manager*/
    case BT_CONNECTED_STATE_UPDATE:/* headset connected indication,BTLink status activate*/ 
        // Do not allow audio to transfer if GW PTT was pressed while WRSM was Out of range (Link Lost)
        if(wrsm_oor_gw_ptt_pressed == FALSE)
        {
          select_mic_spkr(REMOTE_MIC_SPKR);
          audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;
        }
        else
        {
          if(local_PttPress == false)
          {
            audio_transfer_state = GW_CONNECTED_LOCAL_AUDIO;
          }
          else  //if WRSM connected while GW PTT pressed
          {
            audio_transfer_state = GW_CONNECTED_PTT_PRESSED_LOCAL_AUDIO;
          }        
        }  
      LED_blink((portTickType)50,(portTickType)4000);
      break;
    
    case AUDIO_DEVICE_CONNECTED_STATE_UPDATE:
        // Do not allow audio to transfer if GW PTT was pressed while WRSM was Out of range (Link Lost)
        if(wrsm_oor_gw_ptt_pressed == FALSE)
        {
          select_mic_spkr(REMOTE_MIC_SPKR);
          audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;
        }
        else
        {
          if(local_PttPress == false)
          {
            audio_transfer_state = GW_CONNECTED_LOCAL_AUDIO;
          }
          else  //if WRSM connected while GW PTT pressed
          {
            audio_transfer_state = GW_CONNECTED_PTT_PRESSED_LOCAL_AUDIO;
          }        
        }  
      ctrl_state = ctrl_state_for_led; 
      break;
    
    case DEVICE_CONNECTED_STATE_UPDATE:
    
      LED_blink((portTickType)50,(portTickType)4000);  
      break;  
    
    case AUDIO_DEVICE_DISCONNECTED_STATE_UPDATE:
    case BT_TEST_DISCONNECTED:/*BQB mode message from connection manager*/
    case BT_DISCONNECTED_STATE_UPDATE:/*headset disconnected,BTLink status deactivate*/
       
      select_mic_spkr(LOCAL_MIC_SPKR);
      audio_transfer_state = GW_NOT_CONNECTED_LOCAL_AUDIO;
      if (ConnectionManagerState == BT_QUAL_TEST_MODE)/*BQB mode only*/
      {
        LED_blink((portTickType)1000,(portTickType)0);  
      }
      ctrl_state = ctrl_state_for_led;
      break;

    case IDLE_STATE_UPDATE:/* Dongle enter to Idle state Wehen the Device Inquiry Timeout = 61.44 */
    
      LED_blink((portTickType)0,(portTickType)1000);  
      break;
    
    case BT_QUAL_TEST_MODE_UPDATE:/*headset disconnected,BTLink status deactivate(BQB mode message from connection manager)*/
    
      LED_blink((portTickType)1000,(portTickType)0);  
      break;

    case TC_ATMEL_TEST_MODE_OFF: //CCMPD01453712
      /* Not in Atmel Test mode */
      atmel_test_mode = TEST_OFF; 
      ctrl_state = ctrl_state_for_led;  /* to restore led state */
      break;
      
    case TC_ATMEL_TEST_MODE_ON: //CCMPD01453712
      /* Atmel Test mode is ON  */
      atmel_test_mode = TEST_ON; 
      ctrl_state = ctrl_state_for_led;  /* to restore led state */
      break;
    
    case BT_LED_DARK_MODE:
      LED_mode = DARK_MODE;
      ctrl_state = ctrl_state_for_led;
      break;
  
    case BT_NORMAL_LED_MODE:
      LED_mode = NORMAL_MODE; 
      ctrl_state = ctrl_state_for_led;
      break;  
   
    case SEND_HID_REPORT_MESSAGE:
      ctrl_state = ctrl_state_for_led;  
      send_hid_report(); 
      break; 

    default:
      break;
   }
 } 
}  
/*=============================================================================
	FUNCTION: gpio_group0_controls_isr

	DESCRIPTION: Initialize the buttons and audio path ISR

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE:
==============================================================================*/
#define GPIO_INTERRUPT_CLEAR_DISABLE(id)  \
                  while (gpio_get_pin_interrupt_flag( id )) \
                    { \
                      gpio_enable_pin_interrupt(id, GPIO_PIN_CHANGE); \
                      gpio_clear_pin_interrupt_flag(id); \
                      gpio_disable_pin_interrupt(id); \
                    };

#if __GNUC__
	__attribute__((__naked__))
#elif __ICCAVR32__
	#pragma shadow_registers = full   // Naked.
#endif

void gpio_group0_controls_isr(void)
{

  OS_ENTER_SWITCHING_ISR;
  bool button_interrupt_det = FALSE;
  
  if (gpio_get_pin_interrupt_flag( PTT_MFB_GPIO_PIN ))
  {
    GPIO_INTERRUPT_CLEAR_DISABLE(PTT_MFB_GPIO_PIN);

    /* save the current detected state for debouncing the button press/release */
    buttons[PTT_BUTTON].Button_Detect_State = (controls_button_state_t)gpio_get_pin_value(PTT_MFB_GPIO_PIN);

    // load with debounce time
    buttons[PTT_BUTTON].Debounce_Counter = PTT_BTN_DEBOUNCE_COUNTER; 

    button_interrupt_det = TRUE;
  }

   
  if (gpio_get_pin_interrupt_flag( AUDIO_DETECT_GPIO_PIN ))
  {
    GPIO_INTERRUPT_CLEAR_DISABLE(AUDIO_DETECT_GPIO_PIN);

    // save the current detected state for debouncing the button press/release 
    buttons[AUDIO_DET].Button_Detect_State = (controls_button_state_t)gpio_get_pin_value(AUDIO_DETECT_GPIO_PIN);
  
    // load with debounce time
    buttons[AUDIO_DET].Debounce_Counter = AUDIO_DET_DEBOUNCE_COUNTER;
  
    button_interrupt_det = TRUE;
  }

  if (gpio_get_pin_interrupt_flag( CSR_DSP_AUDIO_DETECT_GPIO_PIN ))
  {
    GPIO_INTERRUPT_CLEAR_DISABLE(CSR_DSP_AUDIO_DETECT_GPIO_PIN);
      /* save the current detected state for debouncing the button press/release */
    buttons[CSR_DSP_AUD_DET].Button_Detect_State = (controls_button_state_t)gpio_get_pin_value(CSR_DSP_AUDIO_DETECT_GPIO_PIN);

    // load with debounce time
    buttons[CSR_DSP_AUD_DET].Debounce_Counter = AUDIO_DET_DEBOUNCE_COUNTER;

    button_interrupt_det = TRUE;
  }
  
  if (gpio_get_pin_interrupt_flag( ONE_DOT_DETECT_GPIO_PIN ))
  {
    GPIO_INTERRUPT_CLEAR_DISABLE(ONE_DOT_DETECT_GPIO_PIN);

    /* save the current detected state for debouncing the button press/release */
    buttons[ONE_DOT].Button_Detect_State = (controls_button_state_t)gpio_get_pin_value(ONE_DOT_DETECT_GPIO_PIN);

    // load with debounce time
    buttons[ONE_DOT].Debounce_Counter = BTN_DEBOUNCE_COUNTER;

    button_interrupt_det = TRUE;
  }

  if (gpio_get_pin_interrupt_flag( TWO_DOT_DETECT_GPIO_PIN ))
  {
    GPIO_INTERRUPT_CLEAR_DISABLE(TWO_DOT_DETECT_GPIO_PIN);

    /* save the current detected state for debouncing the button press/release */
    buttons[TWO_DOT].Button_Detect_State = (controls_button_state_t)gpio_get_pin_value(TWO_DOT_DETECT_GPIO_PIN);

    // load with debounce time
    buttons[TWO_DOT].Debounce_Counter = BTN_DEBOUNCE_COUNTER;

    button_interrupt_det = TRUE;
  }

  /* debouncing timer is not started then start it*/
  if( button_timer_flag == FALSE && button_interrupt_det == TRUE)
  {
    button_timer_flag = TRUE;

    /* Set RC value for 1ms debounce. */
    tc_write_rc(DEBOUNCE_TC_HANDLE, PTT_DEBOUNCE_TC_CHANNEL, DEBOUNCE_MS_COUNT);    

    /* Start the timer/counter for debounce routine. */
    tc_start(DEBOUNCE_TC_HANDLE, PTT_DEBOUNCE_TC_CHANNEL);
  }
  
  OS_EXIT_SWITCHING_ISR;
}
/* End of gpio_group0_controls_isr() */


/*=============================================================================
	FUNCTION: gpio_group2_irq1_controls_isr

	DESCRIPTION: ISR for AUDIO_ROUTE_SWITCH_GPIO_PIN (PA08)

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE:
==============================================================================*/

#if __GNUC__
	__attribute__((__naked__))
#elif __ICCAVR32__
	#pragma shadow_registers = full   // Naked.
#endif

void gpio_group2_irq1_controls_isr(void)       //CCMPD01413200
{ /* ISR for GPIO int Group 2 IRQ1 */

  OS_ENTER_SWITCHING_ISR;

     if (gpio_get_pin_interrupt_flag(MPP_WAKEUP_INTERRUPT_GPIO_PIN))
     {  /* MPP wake up is having higher priority */
        /* same processing as in mpp_wakeup_isr() */
        gpio_clear_pin_interrupt_flag(MPP_WAKEUP_INTERRUPT_GPIO_PIN);
        MPP_UART_PORT->cr |= AVR32_USART_CR_RXEN_MASK;
     }
  
  OS_EXIT_SWITCHING_ISR;
}
/* End of gpio_group2_irq1_controls_isr() */

/*=============================================================================
	FUNCTION: debounce_tc_isr

	DESCRIPTION:

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
#if __GNUC__
	__attribute__((__naked__))
#elif __ICCAVR32__
	#pragma shadow_registers = full   // Naked.
#endif

static void debounce_tc_isr( void )
{
  /* This ISR can cause a context switch, so the first statement must be a
     call to the OS_ENTER_SWITCHING_ISR macro.  This must be BEFORE any
     variable declarations. */
  OS_ENTER_SWITCHING_ISR;

  /* Clear the interrupt flag. This is a side effect of reading the TC SR.*/
  tc_read_sr(DEBOUNCE_TC_HANDLE, PTT_DEBOUNCE_TC_CHANNEL);

  if( buttons[PTT_BUTTON].Debounce_Counter != 0 ) ptt_button_debounce(PTT_BUTTON);
  if( buttons[AUDIO_DET].Debounce_Counter != 0 ) audio_detect_debounce(AUDIO_DET);
  if( buttons[ONE_DOT].Debounce_Counter != 0 ) button_debounce(ONE_DOT);
  if( buttons[TWO_DOT].Debounce_Counter != 0 ) button_debounce(TWO_DOT);
  if( buttons[CSR_DSP_AUD_DET].Debounce_Counter != 0 ) audio_detect_debounce(CSR_DSP_AUD_DET);

  if( buttons[PTT_BUTTON].Debounce_Counter == 0
    && buttons[AUDIO_DET].Debounce_Counter == 0
    && buttons[ONE_DOT].Debounce_Counter == 0
    && buttons[TWO_DOT].Debounce_Counter == 0
    && buttons[CSR_DSP_AUD_DET].Debounce_Counter == 0)
  {
    button_timer_flag = FALSE;
    tc_stop(DEBOUNCE_TC_HANDLE, PTT_DEBOUNCE_TC_CHANNEL);
  }   
  OS_EXIT_SWITCHING_ISR;
}

/*=============================================================================
	FUNCTION: ptt_debounce_clk_init

	DESCRIPTION: This function is used to initialize the debounce timer clock.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void ptt_debounce_clk_init()
{
  /* Options for waveform genration. Timer config */
  tc_waveform_opt_t debounce_timer_opt =
  {
    .channel  = PTT_DEBOUNCE_TC_CHANNEL,          // Channel selection.

    .bswtrg   = TC_EVT_EFFECT_NOOP,               // Software trigger effect on TIOB.
    .beevt    = TC_EVT_EFFECT_NOOP,               // External event effect on TIOB.
    .bcpc     = TC_EVT_EFFECT_NOOP,               // RC compare effect on TIOB.
    .bcpb     = TC_EVT_EFFECT_NOOP,               // RB compare effect on TIOB.

    .aswtrg   = TC_EVT_EFFECT_NOOP,               // Software trigger effect on TIOA.
    .aeevt    = TC_EVT_EFFECT_NOOP,               // External event effect on TIOA.
    .acpc     = TC_EVT_EFFECT_NOOP,               // RC compare effect on TIOA: toggle.
    .acpa     = TC_EVT_EFFECT_NOOP,               // RA compare effect on TIOA: toggle (other possibilities are none, set and clear).


    .wavsel   = TC_WAVEFORM_SEL_UP_MODE_RC_TRIGGER, // Waveform selection: Up mode without automatic trigger on RC compare.
    .enetrg   = FALSE,                              // External event trigger enable.
    .eevt     = 0,                                  // External event selection.
    .eevtedg  = TC_SEL_NO_EDGE,                     // External event edge selection.
    .cpcdis   = FALSE,                              // Counter disable when RC compare.
    .cpcstop  = FALSE,                              // Counter clock stopped with RC compare.

    .burst    = FALSE,                              // Burst signal selection.
    .clki     = FALSE,                              // Clock inversion.
    .tcclks   = TC_CLOCK_SOURCE_TC3                 // Internal source clock 3, connected to fPBA / 8.
  };

  static const tc_interrupt_t DEBOUNCE_TC_INTERRUPT =
  {
    .etrgs = 0,
    .ldrbs = 0,
    .ldras = 0,
    .cpcs  = 1,
    .cpbs  = 0,
    .cpas  = 0,
    .lovrs = 0,
    .covfs = 0
  };

  ENTER_CRITICAL;

  INTC_register_interrupt((__int_handler)&debounce_tc_isr, DEBOUNCE_TC_IRQ, AVR32_INTC_INT0);

  tc_init_waveform(DEBOUNCE_TC_HANDLE, &debounce_timer_opt);  // Initialize the timer/counter waveform.

  /* Set the compare triggers.
  // Remember TC counter is 16-bits, so counting second is not possible with fPBA = 4 MHz.
  // We configure it to count ms.
  // We want: (1/(fPBA/8)) * RC = 0.001 s, hence RC = (fPBA/8) / 1000 = 500 to get an interrupt every 1 ms.*/
  tc_write_rc(DEBOUNCE_TC_HANDLE, PTT_DEBOUNCE_TC_CHANNEL, DEBOUNCE_MS_COUNT);    // Set RC value.

  tc_configure_interrupts( DEBOUNCE_TC_HANDLE, PTT_DEBOUNCE_TC_CHANNEL, &DEBOUNCE_TC_INTERRUPT );
  EXIT_CRITICAL;
}
/* End of ptt_debounce_clk_init()   */

/*=============================================================================
	FUNCTION: ptt_button_debounce

	DESCRIPTION: Debounce routine for the PTT button press (7 ms)

	ARGUMENTS PASSED: current PTT button state

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
inline void ptt_button_debounce(UINT8_T btn_no)
{
  /* Check for Button State Detected and Current Button State */
  if (buttons[btn_no].Button_Detect_State == (controls_button_state_t)gpio_get_pin_value(buttons[btn_no].button_pin_no))
  {
    
    buttons[btn_no].Debounce_Counter--;
     //Successfully Debounced
    if (buttons[btn_no].Debounce_Counter == 0)
    {
      // If button state has changed, update prev state and send out button state
      if(buttons[btn_no].Prev_Button_State != buttons[btn_no].Button_Detect_State)
      {  
        //keep track of prev successful button state
        buttons[btn_no].Prev_Button_State = buttons[btn_no].Button_Detect_State;

        
        if(buttons[btn_no].Prev_Button_State == PRESSED) 
        {
          PTT_function(PRESSED);
          
          SEND_MESSAGE_TO_CONTROLS_MANAGER_FROM_ISR(buttons[btn_no].Active_Msg
                , NULL
                , 0);
        }
        else
        {
          PTT_function(RELEASED);

          SEND_MESSAGE_TO_CONTROLS_MANAGER_FROM_ISR(buttons[btn_no].Inactive_Msg
                , NULL
                , 0);
        }
      }
      // re-enable interrupt
      gpio_enable_pin_interrupt(buttons[btn_no].button_pin_no, GPIO_PIN_CHANGE);
    }
  }
  else
  {
       // Failed debounce, restart debouncing
       buttons[btn_no].Button_Detect_State = (controls_button_state_t)gpio_get_pin_value(buttons[btn_no].button_pin_no);
       // load with debounce time
       buttons[btn_no].Debounce_Counter = PTT_BTN_DEBOUNCE_COUNTER;
  }
}
/* End of ptt_button_debounce() */

/*=============================================================================
	FUNCTION: audio_detect_debounce

	DESCRIPTION: Debounce routine for the audio detect

	ARGUMENTS PASSED: audio detect state

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
inline void audio_detect_debounce(UINT8_T btn_no)
{
  /* Check for Button State Detected and Current Button State */
  if (buttons[btn_no].Button_Detect_State == (controls_button_state_t)gpio_get_pin_value(buttons[btn_no].button_pin_no))
  {
    
    buttons[btn_no].Debounce_Counter--;
     //Successfully Debounced
    if (buttons[btn_no].Debounce_Counter == 0)
    {
      // If button state has changed, update prev state and send out button state
      if(buttons[btn_no].Prev_Button_State != buttons[btn_no].Button_Detect_State)
      {  
        //keep track of prev successful button state
        buttons[btn_no].Prev_Button_State = buttons[btn_no].Button_Detect_State;

        /* audio detected */
        if(buttons[btn_no].Prev_Button_State == AUDIO_DETECTED) 
        {
          isDetAudmute = false;
          SEND_MESSAGE_TO_BLUETOOTH_MANAGER_FROM_ISR(HEADSET_SPKR_CONTROL, NULL, 0); 

          SEND_MESSAGE_TO_CONTROLS_MANAGER_FROM_ISR(buttons[btn_no].Active_Msg
            , NULL
            , 0);
        }
        /* audio NOT detected */
        else
        {
          SEND_MESSAGE_TO_CONTROLS_MANAGER_FROM_ISR(buttons[btn_no].Inactive_Msg
            , NULL
            , 0);
        }

      }
      // re-enable interrupt
      gpio_enable_pin_interrupt(buttons[btn_no].button_pin_no, GPIO_PIN_CHANGE);
    }
  }
  else
  {
       // Failed debounce, restart debouncing
       buttons[btn_no].Button_Detect_State = (controls_button_state_t)gpio_get_pin_value(buttons[btn_no].button_pin_no);
       // load with debounce time
       buttons[btn_no].Debounce_Counter = AUDIO_DET_DEBOUNCE_COUNTER;
  }
}

/*=============================================================================
	FUNCTION: button_debounce

	DESCRIPTION: Debounce routine for the other button press

	ARGUMENTS PASSED: button state

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
inline void button_debounce(UINT8_T btn_no)
{
  /* Check for Button State Detected and Current Button State */
  if (buttons[btn_no].Button_Detect_State == (controls_button_state_t)gpio_get_pin_value(buttons[btn_no].button_pin_no))
  {
    
    buttons[btn_no].Debounce_Counter--;
     //Successfully Debounced
    if (buttons[btn_no].Debounce_Counter == 0)
    {
      // If button state has changed, update prev state and send out button state
      if(buttons[btn_no].Prev_Button_State != buttons[btn_no].Button_Detect_State)
      {  
        //keep track of prev successful button state
        buttons[btn_no].Prev_Button_State = buttons[btn_no].Button_Detect_State;

        
        if(buttons[btn_no].Prev_Button_State == PRESSED) 
        {
            SEND_MESSAGE_TO_CONTROLS_MANAGER_FROM_ISR(buttons[btn_no].Active_Msg
                , NULL
                , 0);
        }
        else
        {
            SEND_MESSAGE_TO_CONTROLS_MANAGER_FROM_ISR(buttons[btn_no].Inactive_Msg
                , NULL
                , 0);
        }
      }      // re-enable interrupt
      gpio_enable_pin_interrupt(buttons[btn_no].button_pin_no, GPIO_PIN_CHANGE);
    }
  }
  else
  {
       // Failed debounce, restart debouncing
       buttons[btn_no].Button_Detect_State = (controls_button_state_t)gpio_get_pin_value(buttons[btn_no].button_pin_no);
       // load with debounce time
       buttons[btn_no].Debounce_Counter = BTN_DEBOUNCE_COUNTER;
  }
}
/*=============================================================================
	FUNCTION: controls_init

	DESCRIPTION: Initialize the buttons and audio path

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE:
==============================================================================*/
void controls_init(void)
{
  //Enable the interrupts
  ptt_debounce_clk_init();
  
  if (usb_radio_detected_glb == 1)
  {
    /* Start a timer to send HID report message every 100ms */
    HID_REPORT_TIMER(100, SEND_HID_REPORT_MESSAGE);
  }
}
/* end of controls_init()*/
#ifdef RADIO_INTERFACE_MANAGER_TASK
/*=============================================================================
	FUNCTION: BT_usart_putchar

	DESCRIPTION: This function will put the specified byte on BC05_UART_PORT

	ARGUMENTS PASSED: byte to be tx.

	REFERENCE ARGUMENTS PASSED: none.

	RETURN VALUE: TRUE - none
==============================================================================*/
void BT_usart_putchar(unsigned char c)
{
  while (!(BC05_UART_PORT->csr & AVR32_USART_CSR_TXRDY_MASK));
  BC05_UART_PORT->thr = c;
}
#endif

/*=============================================================================
	FUNCTION: LED_blink

	DESCRIPTION: Blinks the LED depending on the ON and OFF duration passed
                     as parameter

	ARGUMENTS PASSED:T_ON - LED ON time,T_OFF - LED OFF time

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE:
==============================================================================*/
void LED_blink(portTickType T_ON,portTickType T_OFF)
{
  static portTickType time_loader_on;
  static portTickType time_loader_off;
  if(atmel_test_mode == TEST_OFF)   
  { /* LED blinking pattern is displayed only when chip is not in test mode. */
    if((reload_time == TRUE) && (LED_on_status == true)&& (T_ON > 0))
    {
      time_loader_on = SYSTEM_TIME + T_ON;
      if(LED_mode == NORMAL_MODE)
      {  
        gpio_set_gpio_pin(LED_PIN);
      }
      else if(LED_mode == DARK_MODE)
      {
        gpio_clr_gpio_pin(LED_PIN);
      }
    }
    else if((reload_time == TRUE) && (LED_on_status == false) && (T_OFF > 0))
    {
      time_loader_off = SYSTEM_TIME + T_OFF;
      gpio_clr_gpio_pin(LED_PIN);
    }
    if(LED_on_status == TRUE)
    {
      LED_timer(time_loader_on);
    }
    else if(LED_on_status == false)
    {
      LED_timer(time_loader_off);
    }
  }
  else if(atmel_test_mode == TEST_ON) 
  {
    //No Led blinking pattern can be actived as chip is in test mode.
  }
}
/*=============================================================================
	FUNCTION: radio_PTT_function

	DESCRIPTION: All PTT inputs handling

	ARGUMENTS PASSED:none

	RETURN VALUE:none
==============================================================================*/
static void radio_PTT_function(void)
{
  if ((local_PttPress == true)
    ||(isHSPttPress == true)
    ||(isWPTTPttPress == true))   
  {
    gpio_clr_gpio_pin(RADIO_MMP_PTT_GPIO_PIN);   /* Asserting PTT to radio */
  }
  else
  {
    gpio_set_gpio_pin(RADIO_MMP_PTT_GPIO_PIN);
  } 
}
/*=============================================================================
	FUNCTION: PTT_function

	DESCRIPTION: Local PTT button handling

	ARGUMENTS PASSED:ptt_current_button_state - button state detected in the
                         timer debounce isr.

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE:
==============================================================================*/
void PTT_function(controls_button_state_t ptt_current_button_state)
{
  /*irrespective of the dongle state the Gcai Gpio3 will be asserted/de-asserted
  depending on PTT button state*/ 

  if( ptt_current_button_state == PRESSED)
  {
    local_PttPress = true;
  }
  else
  {
    local_PttPress = false;
  }
  //Assert-Deassert PTT 
  radio_PTT_function();
}
/*end of PTT_function*/


/*=============================================================================
	FUNCTION:  LED_timer(portTickType time)

	DESCRIPTION:polling function that sends toggle message to the LED_blink function to toggle LED
	
        ARGUMENTS PASSED:portTickType time - time after which the LED must be toggled

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE:
==============================================================================*/
void LED_timer(portTickType time)
{
  portTickType newcurrenttime;
  newcurrenttime = xTaskGetTickCount();
  if(((time - (portTickType)16) < newcurrenttime) || ((time - newcurrenttime) > ONE_MINUTE))
  {
    reload_time = true;
    if(LED_on_status == true)
    {
      LED_on_status = false;
    }
    else if(LED_on_status == false)
    {
       LED_on_status = true;
        if(ctrl_state == INIT_STATE_UPDATE)
      {
        init_rate++;
      }
    }
  }
  else
  {
    reload_time = FALSE;
  }
}
/* end of LED_timer */
/*=============================================================================
	FUNCTION:wirelessR1_3ptt_handler(bool R1_3PttPressed)

	DESCRIPTION:This function is for nibbler decoder task to call for notifying radio
                    on wireless R1.3 ptt press/release

	ARGUMENTS PASSED:bool R1_3PttPressed

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE:
==============================================================================*/
void wirelessR1_3ptt_handler(bool R1_3PttPressed)
{
    isHSPttPress = R1_3PttPressed;
    
    if((gw_legacy_behavior == FALSE) && (isHSPttPress == TRUE) && (current_mic_spkr_used == LOCAL_MIC_SPKR))
    {
      if(local_PttPress == FALSE)
      {
        select_mic_spkr(REMOTE_MIC_SPKR);
        audio_transfer_state = GW_CONNECTED_REMOTE_AUDIO;
      }
      else
      {
        //De-Assert the Headset PTT press as local PTT is pressed
        isHSPttPress = false;
        //Cancel the Nibbler PTT Timeout
        Cancel_HS_NibbleTimer();
        //Send the message once to play the 'bad bonk' tone on WRSM
        SEND_MESSAGE_TO_CONNECTION_MNGR(HS_USER_PTT_PRESS_PLAY_BAD_BONK, NULL, 0);
      }
    }
    
  //Assert-Deassert PTT 
  radio_PTT_function();
}
/*end of wireless_ptt_handler*/
/*=============================================================================
	FUNCTION: wirelessR1_4ptt_handler(bool R1_4PttPressed)

	DESCRIPTION:This function is for nibbler decoder task to call for notifying radio
                    on wireless R1.4 ptt press/release

	ARGUMENTS PASSED:R1_4PttPressed - parameter indicating state of the wireless R1.4PTT

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE:
==============================================================================*/
void wirelessR1_4ptt_handler(bool R1_4PttPressed)
{
    isWPTTPttPress = R1_4PttPressed;

    if( ConnectionManagerState == BT_CONNECTED_STATE )
    {
      if((gw_legacy_behavior == FALSE) && (isWPTTPttPress == TRUE) && (current_mic_spkr_used == LOCAL_MIC_SPKR))
      {
        if(local_PttPress == FALSE)
        {
          //Route audio to WRSM as it is not aware of WPTT PTT press
          SEND_MESSAGE_TO_CONNECTION_MNGR(CONNECTED_WPTT_PTT_PRESS_ROUTE_AUDIO, NULL, 0);
        }
        else
        {
          //De-Assert the Headset PTT press as local PTT is pressed
          isWPTTPttPress = false;
          //Cancel the Nibbler PTT Timeout
          Cancel_WPTT_NibbleTimer();
          //Send the message once to play the 'bad bonk' tone on WRSM
          SEND_MESSAGE_TO_CONNECTION_MNGR(HS_USER_PTT_PRESS_PLAY_BAD_BONK, NULL, 0);
        }
      }
    
      //Assert-Deassert PTT 
      radio_PTT_function();
    }
}
/* end of wireless_ptt_handler*/


/*=============================================================================
	FUNCTION: setup_button_processing
	DESCRIPTION: This function will setup the pins and interrupt for non-PTT buttons.
                     
	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: 
==============================================================================*/
void setup_button_processing(void)
{  
  bool button_interrupt_det = FALSE;
  
  // Assign Atmel physical pin to the array location 
  buttons[PTT_BUTTON].button_pin_no     = PTT_MFB_GPIO_PIN;
  buttons[AUDIO_DET].button_pin_no      = AUDIO_DETECT_GPIO_PIN;
  buttons[ONE_DOT].button_pin_no        = ONE_DOT_DETECT_GPIO_PIN;
  buttons[TWO_DOT].button_pin_no        = TWO_DOT_DETECT_GPIO_PIN;
  buttons[CSR_DSP_AUD_DET].button_pin_no = CSR_DSP_AUDIO_DETECT_GPIO_PIN;
  
  buttons[PTT_BUTTON].button_detection  = INTERRUPT_BASE;
  buttons[AUDIO_DET].button_detection   = INTERRUPT_BASE;
  buttons[ONE_DOT].button_detection     = INTERRUPT_BASE;
  buttons[TWO_DOT].button_detection     = INTERRUPT_BASE;
  buttons[CSR_DSP_AUD_DET].button_detection = INTERRUPT_BASE;
    
  buttons[PTT_BUTTON].Active_Msg        = PTT_BUTTON_BUTTON_PRESSED;
  buttons[AUDIO_DET].Active_Msg         = AUDIO_DET_ACTIVATED;
  buttons[ONE_DOT].Active_Msg           = ONE_DOT_BUTTON_PRESSED;
  buttons[TWO_DOT].Active_Msg           = TWO_DOT_BUTTON_PRESSED;
  buttons[CSR_DSP_AUD_DET].Active_Msg   = CSR_DSP_AUDIO_DET_ACTIVATED;
    
  buttons[PTT_BUTTON].Inactive_Msg      = PTT_BUTTON_BUTTON_RELEASED;
  buttons[AUDIO_DET].Inactive_Msg       = AUDIO_DET_DEACTIVATED;
  buttons[ONE_DOT].Inactive_Msg         = ONE_DOT_BUTTON_RELEASED;
  buttons[TWO_DOT].Inactive_Msg         = TWO_DOT_BUTTON_RELEASED;
  buttons[CSR_DSP_AUD_DET].Inactive_Msg = CSR_DSP_AUDIO_DET_DEACTIVATED;

   /* Register the enable interrupt handler to the interrupt controller and
      enable the interrupt. */
  INTC_register_interrupt((__int_handler)&gpio_group0_controls_isr, PTT_MFB_GPIO_PIN_IRQ, PTT_MFB_GPIO_PIN_IRQ_LEVEL);
  INTC_register_interrupt((__int_handler)&gpio_group0_controls_isr, AUDIO_DETECT_GPIO_PIN_IRQ, AUDIO_DETECT_GPIO_PIN_IRQ_LEVEL);
  INTC_register_interrupt((__int_handler)&gpio_group0_controls_isr, ONE_DOT_DETECT_GPIO_PIN_IRQ, ONE_DOT_DETECT_GPIO_PIN_IRQ_LEVEL);
  INTC_register_interrupt((__int_handler)&gpio_group0_controls_isr, TWO_DOT_DETECT_GPIO_PIN_IRQ, TWO_DOT_DETECT_GPIO_PIN_IRQ_LEVEL);
  INTC_register_interrupt((__int_handler)&gpio_group0_controls_isr, CSR_DSP_AUDIO_DETECT_GPIO_PIN_IRQ, CSR_DSP_AUDIO_DETECT_GPIO_PIN_IRQ_LEVEL);
  
  /* Initialize */
  for(UINT8_T btn_no = 0; btn_no < MAX_NUM_BTN; btn_no++)
  {
    buttons[btn_no].Prev_Button_State   = RELEASED;   
    buttons[btn_no].Debounce_Counter    = 0;
  }
  // For CSR_DSP_AUDIO_DET - Detection is High and no detection is low
 // buttons[CSR_DSP_AUD_DET].Prev_Button_State  = AUDIO_NOTDETECTED;
  
  /* Start button Interrupt */
  for(UINT8_T btn_no = 0; btn_no < MAX_NUM_BTN; btn_no++)
  {
    buttons[btn_no].Button_Detect_State = (controls_button_state_t)gpio_get_pin_value(buttons[btn_no].button_pin_no);
    if(buttons[btn_no].Button_Detect_State != buttons[btn_no].Prev_Button_State) 
    {
      // start button deboucing if button is PRESSED
      if( btn_no == PTT_BUTTON)
      {
        buttons[btn_no].Debounce_Counter = PTT_BTN_DEBOUNCE_COUNTER;
        button_interrupt_det = TRUE;
      }
      else if( btn_no == AUDIO_DET)
      {
        //Do not allow the interrupt to be enabled from the TC ISR
        if(hw_audio_detect == CP_SELECT_ON)
        {
          buttons[btn_no].Debounce_Counter = AUDIO_DET_DEBOUNCE_COUNTER;
          button_interrupt_det = TRUE;
        }
      }
      else if( btn_no == CSR_DSP_AUD_DET)
      {
        //Do not allow the interrupt to be enabled from the TC ISR
        if(hw_audio_detect == CP_SELECT_OFF)
        {
          buttons[btn_no].Debounce_Counter = AUDIO_DET_DEBOUNCE_COUNTER;
          button_interrupt_det = TRUE;
        }
      }        
      else
      {
        buttons[btn_no].Debounce_Counter = BTN_DEBOUNCE_COUNTER;
        button_interrupt_det = TRUE;
      }
    }
    else
    {
      if( btn_no == AUDIO_DET)
      {
        //Do not enable the interrupt if HW Audio Detect not enabled
        if(hw_audio_detect == CP_SELECT_ON)
        {
          gpio_enable_pin_interrupt(buttons[btn_no].button_pin_no, GPIO_PIN_CHANGE); 
        }
      }
      else if( btn_no == CSR_DSP_AUD_DET)
      {
        if(hw_audio_detect == CP_SELECT_OFF)
        {        
          gpio_enable_pin_interrupt(buttons[btn_no].button_pin_no, GPIO_PIN_CHANGE); 
        }
      }
      else
      {
        gpio_enable_pin_interrupt(buttons[btn_no].button_pin_no, GPIO_PIN_CHANGE); 
      } 
    }
  }

  if( button_timer_flag == FALSE && button_interrupt_det == TRUE)
  {
    button_timer_flag = TRUE;

    /* Set RC value for 1ms debounce. */
    tc_write_rc(DEBOUNCE_TC_HANDLE, PTT_DEBOUNCE_TC_CHANNEL, DEBOUNCE_MS_COUNT);    

    /* Start the timer/counter for debounce routine. */
    tc_start(DEBOUNCE_TC_HANDLE, PTT_DEBOUNCE_TC_CHANNEL);
  }
  
}
/* end of void setup_button_processing()*/

/*=============================================================================
	FUNCTION: one_dot_programable_button_handler

	DESCRIPTION:The function handle one_dot_programable_button  event

	ARGUMENTS PASSED:

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE:
==============================================================================*/
void one_dot_programable_button_handler(controls_button_state_t button_state)
{
  if (gw_usb_hid_disabled_glb) /* USB HID is disabled */
  {
    if( button_state == PRESSED)
    {
      gpio_clr_gpio_pin(RADIO_MMP_ONE_DOT_GPIO_PIN);   /* Assert */
    }
    else
    {
      gpio_set_gpio_pin(RADIO_MMP_ONE_DOT_GPIO_PIN);   /* Deassert */
    }
  }
  
  else /* USB HID is enabled */
  {
    if( button_state == PRESSED)
    {
      HID_One_Dot_Pressed = TRUE; 
      if (HID_Two_Dot_Pressed) 
      {
        hid_report_message[3] = HID_ID_ONE_DOT;
      }
      else 
      {
        hid_report_message[2] = HID_ID_ONE_DOT;
      }
    }
    else
    {
      HID_One_Dot_Pressed = FALSE; 
      if (HID_Two_Dot_Pressed) 
      {
        hid_report_message[2] = HID_ID_TWO_DOT;
        hid_report_message[3] = HID_ID_RELEASED;
      }
      else 
      {
        hid_report_message[2] = HID_ID_RELEASED;
      }
    }
    send_hid_report();
  }
}

/*=============================================================================
	FUNCTION: two_dot_programable_button_handler

	DESCRIPTION:The function handle one_dot_programable_button  event

	ARGUMENTS PASSED:

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE:
==============================================================================*/
void two_dot_programable_button_handler(controls_button_state_t button_state)
{
  if (gw_usb_hid_disabled_glb) /* USB HID is disabled */
  {
    if( button_state == PRESSED)
    {
      gpio_clr_gpio_pin(RADIO_MMP_TWO_DOT_GPIO_PIN);   /* Assert */
    }
    else
    {
      gpio_set_gpio_pin(RADIO_MMP_TWO_DOT_GPIO_PIN);   /* Deassert */
    }
  }
  
  else /* USB HID is enabled */
  {
    if( button_state == PRESSED)
    {
      HID_Two_Dot_Pressed = TRUE; 
      if (HID_One_Dot_Pressed) 
      {
        hid_report_message[3] = HID_ID_TWO_DOT;
      }
      else 
      {
        hid_report_message[2] = HID_ID_TWO_DOT;
      }
    }
    else
    {
      HID_Two_Dot_Pressed = FALSE; 
      if (HID_One_Dot_Pressed) 
      {
        hid_report_message[2] = HID_ID_ONE_DOT;
        hid_report_message[3] = HID_ID_RELEASED;
      }
      else 
      {
        hid_report_message[2] = HID_ID_RELEASED;
      }
    }
    send_hid_report();
  }
}

/*=============================================================================
	FUNCTION: control_manager_init

	DESCRIPTION:The function handle all the initilization for control manager

	ARGUMENTS PASSED: NONE

	REFERENCE ARGUMENTS PASSED: NONE

	RETURN VALUE:
==============================================================================*/
void control_manager_init(void)
{
  setup_button_processing();
}

/*=============================================================================
	FUNCTION: get_button_state

	DESCRIPTION:The function will check for the button state

	ARGUMENTS PASSED: button ID

	REFERENCE ARGUMENTS PASSED: Button state

	RETURN VALUE:
==============================================================================*/
controls_button_state_t get_button_state(UINT8_T btn_no)
{
  if(buttons[btn_no].Debounce_Counter != 0)
  {
    return DEBOUNCED;
  }
  else
  {
    return buttons[btn_no].Prev_Button_State;
  }
}

void select_mic_spkr(bool local)
{
  if(local)
  {
    gpio_set_gpio_pin(MIC_SELECT_GPIO_PIN);  /* select local Mic */
    gpio_clr_gpio_pin(RADIO_MMP_VIRTUAL_HOOK_GPIO_PIN);  /* Select Local Speaker L:Radio speaker, H: HS Speaker */
    current_mic_spkr_used = LOCAL_MIC_SPKR;
  }
  else
  {
    gpio_set_gpio_pin(RADIO_MMP_VIRTUAL_HOOK_GPIO_PIN); /* Select HS Speaker */
    gpio_clr_gpio_pin(MIC_SELECT_GPIO_PIN);  /* select remote Mic */    
    current_mic_spkr_used = REMOTE_MIC_SPKR;
  }
}

void send_hid_report(void)
{
  if (usb_radio_detected_glb >= 1) 
  {
    /* send hid report 0000xx0000000000 */
    for (int i=0;i<8;i++) 
      usb_putchar(hid_report_message[i]); 
  
    if (usb_radio_detected_glb == 1)
    {
      /* Start a timer to send HID report message every 100ms */
      HID_REPORT_TIMER(100, SEND_HID_REPORT_MESSAGE);
    }
  }
}