/*################################################################################
#
# 						FILE SPECIFICATION
# 					COPYRIGHT 2011,2014 MOTOROLA SOLUTIONS,INC. ALL RIGHTS RESERVED.
#						MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: <hsp_ag.c>
#
# --------------------------- General Description -----------------------------
#
#
#
#
********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR            	Date Modified	CR Tracking Number  Description
* Niravkumar Rabara	09/23/2010	CCMPD01394958       Headset Profile Audio Gateway implementation. 
* Niravkumar Rabara     10/06/2010      CCMPD014000437      Added in SendAudioConnectionStatusInd to get the BT Address
* Niravkumar Rabara	10/08/2010	CCMPD01400432       Modified RING AT Command to 10 times as per MSC
* Niravkumar Rabara	10/22/2010	CCMPD01405968       Modify nibbler byte variable name
* Niravkumar Rabara     10/27/2010      CCMPD01407554       Added in SendLinkStatusInd to get teh BT Address
* Niravkumar Rabara	10/27/2010	CCMPD01405413       Added eSCO support for audio
* JJ Low                11/08/2010      CCMPD01410054       Compiler warning resolution for R2.1
* PeiSee Toh            12/08/2010      CCMPD01451438       Support the eSCO options:2EV3, 3EV3, 2EV5 and 3EV5. 
* Tan Seng Kai          12/10/2010      CCMPD01452526       Receiving Nibbler data after nibbler establish
* Tan Seng Kai          12/17/2010      CCMPD01455489       EV3 S3 setup
* Tan Segn Kai          12/20/2010      CCMPD01455949       Nibbler sending pointer instead of byte by byte
* Mahes                Jan/06/2011      CCMPD01460340       Software catch-up to dongle R1.2 version R01.00.00
* Tan Seng Kai          01/17/2011      CCMPD01463409       Added in function call for nibbler enable
* Niravkumar Rabara     01/25/2011      CCMPD01466754       Skip the Service Search for the Incomming ACL Connection Request
* Toh Pei See           12/01/2011      CCMPD01587273       Disable the support of EDR packets, only allow Basic Rate packets for eSCO link
* WRN637                27/01/2014      CCMPD01856259       Disable_EDR
*
*--------------------------- End of History Template-----------------------------
* *********************************************************************************/

/*****************************************************************************/
/* Include files															 */
/*****************************************************************************/

#include "hsp_ag.h"
#include "hsp_ag_opt.h"
#include "bt_log.h"
#include "nibbler.h"

/*****************************************************************************/
/* Private macros															 */
/*****************************************************************************/

/*
 * How to use PADAPT
 */
#define PADAPT_INTERFACE_TYPE ((PADAPT_InterfaceType)((HSP_AG_CFG_DISCONNECT_ACL_AFTER_DM_TIME != 0)?\
									(PADAPT_INTERFACE_SIMPLE | PADAPT_GET_DM_SENT_IND):\
									(PADAPT_INTERFACE_SIMPLE)) )


/*
 * Values used for userTimerId in BT_TimeoutReq
 */
#define TIMER_ID_RUN_CONNECTION_STATE_MACHINE		1
#define TIMER_ID_RESTART_RESPONSE					2
#define TIMER_ID_SCO_CONNECT_RESPONSE 			3
#define TIMER_ID_SDAP_SEARCH						4
#define TIMER_ID_PADAPT_CONNECT_REQ 				5
#define TIMER_ID_HS_DISCONNECTED_STATE_IND					0x0a
#define TIMERID_RESEND_LINK_SETTINGS				0x0b
#define TIMER_ID_DM_DISCONNECT_ACL	   				0x0c
#define TIMER_ID_SEND_LINK_STATUS_IND				0x0d

#define TIMER_ID_RESEND_REJECT_CONNECT				0x0e


/*
 * Values for type ConnectionState
 */

#define HS_DISCONNECTED_STATE					0


#define INCOMING_HS_CONNECT_REQUEST_STATE		1	/* Remote Device initiated ACL
													 * connection, sent SDAP
													 * search req
													 */


#define INCOMING_HS_W_4_CONNECT_RSP				2	/* Remote Device initiated ACL
													 * connection, got SDAP
													 * search response. Waiting
													 * for local connect rsp
													 */

#define HS_SERVICE_SEARCH_STATE 				5	/* Connection initiated
													 * locally, sent SDAP search
													 * req
													 */

#define HS_SERVICE_SEARCH_COMPLETE_STATE 		6	/* Connection initiated
													 * locally, Got SDAP search
													 * response
													 */

#define PADAPT_CONNECTING_STATE					7	/* AG has sent a PADAPT
													 * connect request
													 */

#define PADAPT_CONNECTED_STATE					8	/* AG has received
													 * PADAPT connect cfm,
													 * hsDlci != 0 if successful
													 *
													 * or
													 *
													 * AG has sent padapt
													 * connect response
													 * accepting connection
													 */

#define HS_ACL_CONNECTED_STATE						0x0e	/* AT command sequence
														 * is complete
														 */

/*
 * Character constats
 */
#define CR		0x0d	/* Carrige return */
#define LF		0x0a	/* Line feed */


/*
 * Values for type ParseState
 */
#define W_4_A			0
#define W_4_T			1
#define W_4_CMD 		2


/*
 * Wait max two minutes for SDAP, i.e. basically no timeout
 */
#define MAX_SDAP_SEARCH_TIME					120000


 /*
 *	Typecast macros to avoid having to put a typecast at each call
 */
#define SEND_TO_USER(msg, msgType)		SendToUser((BT_Msg*)(msg), (msgType))
#define SEND_REQ(a, b, c, d, e)			SendReq((BT_Msg*)(a), b, c, d, e)


#define INVALID_CONNECTION_HANDLE	0x8000


#define GET_CONNECTION(con) (HSP_AG_validMultiProfile ? ((Connections*) con) : (&connections[0]))

#define USE_MANUAL_AUDIO	(HSP_AG_CFG_BEHAVIOUR & HSP_AG_BEHAVIOUR_MANUAL_AUDIO_ACCEPT)


/*****************************************************************************/
/* Private types															 */
/*****************************************************************************/


/*
 * Union of all messages that HSP Ag can receive
 */
typedef union
{
	BT_Header		header;
	PADAPT_IndConf	padaptIndConf;
	SDAP_Rsp		sdapRsp;
	HCI_Event		hciEvent;
	BT_TimeoutInd	tsrvInd;
	HCI_LH_IndConf	lh;
} HSP_AG_InMsg;

/*
 * Main state information for module, shows
 * state of service level connection
 */
typedef BT_U8	ConnectionState;

/*
 * State of AT parser
 */
typedef BT_U8	ParseState;

/*
 * Struct to store information about a specific connection
 */
typedef struct
{
	/*
	 * The address of the HSP HS
	 */
	BT_BdAddr				hsBdAddr;

	/*
	 * When there is a PADAPT connection , hsDlci identifies
	 * the PADAPT connection to the HS.
	 */
	RFCOMM_Dlci 			hsDlci;

	/*
	 * hsServerChannel is set by the SDAP search procedure in the
	 * beginning of each connection. If there was no matching HS
	 * service it will be set to 0
	 */
	RFCOMM_ServerChannel	hsServerChannel;

	/*
	 * The agreed frame size used for the PADAPT connection
	 */
	RFCOMM_DlcFrameSize 	framesize;

	/*
	 * When the are data and audio connections to the HS
	 * the connection handles will be stored in these varables
	 * The audioConnectionHandle will be used to keep track if 
	 * audio is connected or not.
	 */
	BT_U16					audioConnectionHandle;
	BT_U16					aclConnectionHandle;

	/*
	 * Used to store when resending audio connect response
	 */
	HCI_LinkType			audioLinkType;

	/*
	 * Used to store the accept/reject of the audio link.
	 */
	BT_BOOL					audioAccept;

	/*
	 * Store current status of the PADAPT flow control. BT_TRUE
	 * if it is allowed to send data to AG, BT_FALSE otherwise
	 */
	BT_BOOL 				remoteReadyToReceive;


	/*
	 * State information for HSP AG module,
	 */
	ConnectionState 		connectionState;
	ParseState				parseState;

	/*
	 * These variables are used to prevent profile from using too
	 * many timers. They store the timer id's returned BT_TimeoutReq
	 * for each of the allocated timers. If a variable is 0,
	 * that timer is not in use.
	 *
	 * connectTimer
	 *		Used when sending internal service level connection setup
	 *		commands and when sending commands initiated by user requests.
	 *
	 *	sdapTimer
	 *		Used when resending SDAP commands. Service search is
	 *		performed concurrently with incoming connections so
	 *		a separate timer is needed.
	 *
	 *		It's also used for the context switch when received OK
	 *		and shall send LINK_STATUS_IND(SLC_CONNECTED)
	 *
	 * audioConnectRspTimer
	 *  Used for retransmission of sco accept
	 *  command, incoming audio setup can happen anytime
	 *		so a separate timer is needed.
	 *
	 */

	BT_TimerId					connectTimer;
	BT_TimerId					sdapTimer;
	BT_TimerId					audioConnectRspTimer;

	/*
	 * Used for connect/disconnect to make it
	 * possible to send appropriate confirm when done
	 */
	BT_Primitive				userRequestInProgress;

	/*
	 * Used to assemble at commands complete result one delimited
	 * by AT ...<CR> have been received.
	 */
	BT_U8	*					atCommandBuffer;

	/*
	 * Length of current contents of atCommandBuffer
	 */
	BT_U16						atCommandLength;
} Connections;


/* 
 * Info needed to reject connects from HS while another connections exists
 * or is being setup. Can happen anytime so a separate timer is needed.
 */	
typedef struct
{
	BT_TimerId			timer;
	BT_BdAddr 			bdAddr;
	RFCOMM_Dlci			dlci;
} RejectConnectInfo;


/*****************************************************************************/
/* Public constants & variables 											 */
/*****************************************************************************/

/*
 * Module id assigned to HSP_AG during startup
 */
BT_ModuleId 	BT_RAM	HSP_AG_moduleId = 0;

BT_BOOL 		BT_RAM	HSP_AG_validMultiProfile;

extern nibbler_struct_t headset_nibbler;

/*****************************************************************************/
/* Private constants & variables											 */
/*****************************************************************************/

/*
 * Module id of registered user of HSP HS or 0 if there
 * currently is no registered user
 */
static BT_ModuleId				BT_RAM hspAgUser = 0;


/*
 * Local server channel assigned when registering service. Needs
 * to be saved to make it possible to unregister service
 */
static	RFCOMM_ServerChannel	BT_RAM localServerChannel;


/*
 * Use this instead of HSP_AG_attributeIdList when connecting to
 * avoid allocating a buffer for service name (which might be long)
 */
static const BT_U8 BT_ROM connectAttributeIdList[] =
{
	0x35,	0x09,		/* data element seq 9 bytes
						*/

	0x09, 0x00, 0x01,	/* 0x0001 - UINT16, Service Class Id List
						 */

	0x09, 0x00, 0x04,	/* 0x0004 - UINT16, Protocol descriptor list */
						 
	0x09, 0x01, 0x00 	 /*0x0100 - UINT16, Service name */
														 
};

/*
 * Track the connections.
 */
static Connections * BT_RAM connections;

/*
 * Restarted response timer
 */
static BT_TimerId restartedRspTimer;

/*
 * When receiving PADAPT_DM_SENT_IND, we will
 * optinally tear down the ACL link after a timeout.
 * Store bdAddress to disconnect here while waiting
 * for timeout.
 */
static BT_BdAddr disconnectAclBdaddr;


/*
* Default setup for an audio connect request         
 */
static HCI_LH_AudioSetupOptions	scoSetupSettings[] = 
{

  { 
    BT_FALSE,
    HCI_LINK_TYPE_ESCO_CONNECTION,
    8000,
    8000,
    10, //maxLatency
    0,
    HCI_AT_LEAST_ONE_RETRANSMISSION_OPT_FOR_POWER,
    HCI_SYNCHRONOUS_EV3 
    | HCI_SYNCHRONOUS_DISALLOW_2_EV3
    | HCI_SYNCHRONOUS_DISALLOW_3_EV3
    | HCI_SYNCHRONOUS_DISALLOW_2_EV5
    | HCI_SYNCHRONOUS_DISALLOW_3_EV5                  /* packet type */
  },
  {
    BT_FALSE,
    HCI_LINK_TYPE_ESCO_CONNECTION,
    8000,
    8000,
    10,
    0,
    HCI_AT_LEAST_ONE_RETRANSMISSION_OPT_FOR_POWER,
    HCI_SYNCHRONOUS_EV4 | HCI_SYNCHRONOUS_EV5
    | HCI_SYNCHRONOUS_DISALLOW_2_EV3
    | HCI_SYNCHRONOUS_DISALLOW_3_EV3
    | HCI_SYNCHRONOUS_DISALLOW_2_EV5
    | HCI_SYNCHRONOUS_DISALLOW_3_EV5                  /* packet type */
  },
  { 
    BT_FALSE,
    HCI_LINK_TYPE_SCO_CONNECTION,
    8000,
    8000, 
    7,
    0,
    HCI_NO_RETRANSMISSIONS,
    HCI_SYNCHRONOUS_HV2
    | HCI_SYNCHRONOUS_DISALLOW_2_EV3
    | HCI_SYNCHRONOUS_DISALLOW_3_EV3
    | HCI_SYNCHRONOUS_DISALLOW_2_EV5
    | HCI_SYNCHRONOUS_DISALLOW_3_EV5                  /* packet type */
  },
  { 
    BT_TRUE,
    HCI_LINK_TYPE_SCO_CONNECTION,
    8000,
    8000,
    7,
    0,
    HCI_NO_RETRANSMISSIONS,
    HCI_SYNCHRONOUS_HV1
    | HCI_SYNCHRONOUS_DISALLOW_2_EV3
    | HCI_SYNCHRONOUS_DISALLOW_3_EV3
    | HCI_SYNCHRONOUS_DISALLOW_2_EV5
    | HCI_SYNCHRONOUS_DISALLOW_3_EV5                  /* packet type */
  }
};


static const HCI_LH_AudioSetupOptions *audioSetupOptions = 0;

/*
 * Default setup for an audio connect response     
 */ 
static const HCI_LH_AudioAcceptOptions	scoAcceptSettings =
{
  8000,   /* transmitBandwidth */
  8000,   /* receiveBandwidth */
  10,     /* maxLatency in ms */
  0x00,   /* aircoding: CVSD */
  HCI_AT_LEAST_ONE_RETRANSMISSION_OPT_FOR_POWER,    /* retransmissionEffort */
    HCI_SYNCHRONOUS_HV1
  | HCI_SYNCHRONOUS_HV2
  | HCI_SYNCHRONOUS_HV3
  | HCI_SYNCHRONOUS_EV3
  | HCI_SYNCHRONOUS_EV4
  | HCI_SYNCHRONOUS_EV5
  | HCI_SYNCHRONOUS_DISALLOW_2_EV3
  | HCI_SYNCHRONOUS_DISALLOW_3_EV3
  | HCI_SYNCHRONOUS_DISALLOW_2_EV5
  | HCI_SYNCHRONOUS_DISALLOW_3_EV5                  /* packet type */
};


static const HCI_LH_AudioAcceptOptions	*audioAcceptOptions = &scoAcceptSettings;

/* If data request has returned BT_CONGESTED */
static BT_BOOL					returnedDataCongestion = BT_FALSE;

static RejectConnectInfo rejectConnectInfo;

/*****************************************************************************/
/* Private function prototypes												 */
/*****************************************************************************/
static void Initialise(void) BT_LARGE;

static void ChangeConnectionState(Connections *con, ConnectionState newState) BT_LARGE;

static void HandleHciMsg(const HCI_Event *m) BT_LARGE;
static void HandlePadaptMsg(const PADAPT_IndConf *m) BT_LARGE;
static void HandleSdapMsg(const SDAP_Rsp *m) BT_LARGE;
static void HandleTsrvMsg(const BT_TimeoutInd *m) BT_LARGE;
static void HandleLinkHandlerMsg( const HCI_LH_IndConf *ind) BT_LARGE;

static BT_Result	SendPadaptConnectRsp(Connections *con, BT_BOOL accept) BT_LARGE;
static void SendHciLhScoConnectRsp(Connections *con) BT_LARGE;
static BT_BOOL		SetupHciUsage(void) BT_LARGE;

static BT_Result HandleHspAgConnectReq(const BT_BdAddr bdAddr) BT_LARGE;

static BT_Result HandleHspAgDisconnectReq(Connections *con) BT_LARGE;
static BT_Result HandleAudioConnectReq(Connections *con) BT_LARGE;
static BT_Result HandleAudioConnectRsp(Connections *con, BT_BOOL accept) BT_LARGE;
static BT_Result HandleAudioDisconnectReq(Connections *con) BT_LARGE;

static void RestartAtCommandParser(Connections *con) BT_LARGE;
//static const char* StrIgnoreCaseStr(const char *string, const char *strCharSet); // CCMPD01410054 (JJ 8/11/2010)
//static BT_S32 StrIgnoreCaseNCmp( const char *str1, const char *str2, BT_U32 count ) BT_LARGE;
static BT_Result SendPadaptDataReq(Connections *con, const BT_U8 *dta, BT_U16 length ) BT_LARGE;

static void SendAclDisconnect(BT_BdAddr bdAddr );

static void ConvertPacketTypesFrom11To12(HCI_ScoPacketType packetType11,
										HCI_SynchronousPacketType *packetType12 );



/*****************************************************************************/
/* Public API functions 													 */
/*****************************************************************************/


/* ===========================================================================
 * Function:  HSP_AG_SetScoPacketType
 * Description: 
 * =========================================================================== */
void HSP_AG_SetScoPacketType(HCI_ScoPacketType hciScoPacketType) BT_LARGE
{
	if ( audioSetupOptions == 0 )
	{
		ConvertPacketTypesFrom11To12( hciScoPacketType,
										&scoSetupSettings[0].packetType );
	}
}

/* ===========================================================================
 * Function: 	HSP_AG_ConfigureAudio
 * Description: 
 * =========================================================================== */
void HSP_AG_ConfigureAudio(	const HCI_LH_AudioSetupOptions 	*setupOptions,
							const HCI_LH_AudioAcceptOptions	*acceptOptions )
{
	audioSetupOptions = setupOptions; 
	audioAcceptOptions = acceptOptions; 
}

/* ===========================================================================
 * Function:	HSP_AG_RegisterService
 * Description: 
 * =========================================================================== */
BT_BOOL HSP_AG_RegisterService(BT_ModuleId user,
								const char *securityServiceName,
								RFCOMM_ServerChannel *rfcommServerChannel) BT_LARGE
{
	BT_BOOL BT_RAM res = BT_TRUE;

	if ( hspAgUser )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "User of HSP_AG is already registered"));
		res = BT_FALSE;
	}
	else
	{
		hspAgUser = user;

		/*
		 * Set local server channel
		 */
		if ( *rfcommServerChannel == 0
			&& !PADAPT_NextFreeServerChannel( rfcommServerChannel ) )
		{
				LOG_INFO((HSP_AG_MODULE_ID, "Could not get free server channel from PADAPT"));
		}

		/*
		 * Registration at Security manager.
		 */
		else if ( SEC_RegisterIncomingService( securityServiceName,
									HSP_AG_CFG_SECURITY_LEVEL,
									SEC_ID_RFCOMM,
									*rfcommServerChannel ) != BT_OK)
		{
			LOG_INFO((HSP_AG_MODULE_ID, "Could not register incoming service with SEC"));
		}

		else if ( SEC_RegisterOutgoingService( securityServiceName,
									HSP_AG_CFG_SECURITY_LEVEL,
									SEC_ID_RFCOMM,
									BT_HSP_HS ) != BT_OK)
		{
			LOG_INFO((HSP_AG_MODULE_ID, "Could not register outgoing service with SEC"));
		}

		/*
		 * Register service with PADAPT
		 */
		else if ( !PADAPT_RegisterService(HSP_AG_MODULE_ID,
									*rfcommServerChannel,
									PADAPT_INTERFACE_TYPE,
									HSP_AG_CFG_MIN_FRAME_SIZE,
									HSP_AG_CFG_MAX_FRAME_SIZE,
									HSP_AG_CFG_INITIAL_CREDITS,
									HSP_AG_CFG_CREDITS_TO_SEND,
									HSP_AG_CFG_MIN_CREDITS) )
		{
			LOG_INFO((HSP_AG_MODULE_ID, "Register service with PADAPT failed\n"));
		}

		/*
		 * Register with HCI Driver and subscribe to HCI events
		 */
		else if ( SetupHciUsage() )
		{
			/*
			 * Store the server channel so it will be possible
			 * to unregister.
			 */
			localServerChannel = *rfcommServerChannel;
			res = BT_TRUE;
		}
		else
		{
			/*
			 * Just skip
			 */
		}
	}

	return res;
}


/* ===========================================================================
 * Function:	HSP_AG_UnregisterService
 * Description: 
 * =========================================================================== */
BT_BOOL HSP_AG_UnregisterService(BT_ModuleId user) BT_LARGE
{
	BT_BOOL BT_RAM res = BT_FALSE;

	if ( hspAgUser != 0 && hspAgUser == user )
	{
		BT_TimeoutCancelAll(HSP_AG_MODULE_ID);
		(void)HCI_UnregisterUser(HSP_AG_MODULE_ID);
		(void)PADAPT_UnregisterService(localServerChannel);

		(void)SEC_UnregisterOutgoingService(
						SEC_ID_RFCOMM,
						BT_HSP_HS );

		(void)SEC_UnregisterIncomingService(
									SEC_ID_RFCOMM,
									localServerChannel);
		hspAgUser = 0;
		res = BT_TRUE;
	}
	else
	{
		LOG_INFO((HSP_AG_MODULE_ID, "HSP_AG_UnregisterService failed"));
	}

	return res;
}


/* ===========================================================================
 * Function:	HSP_AG_ChangeSecurityLevel
 * Description: 
 * ==========================================================================*/
BT_Result HSP_AG_ChangeSecurityLevel(SEC_SecurityLevel securityLevel)
{
	BT_Result res = BT_FAILED;

	if ( (res = SEC_ChangeIncomingService(securityLevel,
										SEC_ID_RFCOMM,
										localServerChannel)) != BT_OK ) 
	{
		LOG_INFO((HSP_AG_MODULE_ID, "SEC_ChangeIncomingService failed"));
	}
	else if ( (res = SEC_ChangeOutgoingService(securityLevel,
												SEC_ID_RFCOMM,
												BT_HSP_HS)) != BT_OK )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "SEC_ChangeOutgoingService failed"));
	}

	return res;
}

/* ===========================================================================
 * Function:	HSP_AG_Control
 * Description: 
 * =========================================================================== */
BT_Result HSP_AG_Control(BT_ControlCommand cmd) BT_LARGE
{
	switch ( cmd )
	{
	case BT_INITIALISE:
		Initialise();
		break;

	case BT_REGISTER:
		break;

	case BT_START:
		break;
	}

	return BT_CONTROL_CMD_COMPLETE;
}

/* ===========================================================================
 * Function:	HSP_AG_RequestMsgHandler
 * Description: Dispatcher for all requests received by HSP_AG
 * Returns: 	BT_OK, BT_CONGESTED or BT_CALL_ERROR
 * =========================================================================== */
BT_Result HSP_AG_RequestMsgHandler(BT_DEPRECATED_MSG_CONST BT_Msg *msg) BT_LARGE
{
	BT_Result			   BT_RAM res	= BT_CALL_ERROR;
	const HSP_AG_ReqResp * BT_RAM m 	= (const HSP_AG_ReqResp *)msg;

	switch ( m->header.primitive )
	{
	case HSP_AG_CONNECT_REQ:
		res = HandleHspAgConnectReq(m->connectReq.bdAddr);
		break;

	case HSP_AG_DISCONNECT_REQ:
		res = HandleHspAgDisconnectReq(GET_CONNECTION(m->disconnectReq.handle));
		break;

	case HSP_AG_AUDIO_CONNECT_REQ:
		res = HandleAudioConnectReq(GET_CONNECTION(m->audioConnectReq.handle));
		break;

	case HSP_AG_AUDIO_CONNECT_RSP:
		res = HandleAudioConnectRsp(GET_CONNECTION(m->audioConnectRsp.handle),
									m->audioConnectRsp.accept);
		break;

	case HSP_AG_AUDIO_DISCONNECT_REQ:
		res = HandleAudioDisconnectReq(GET_CONNECTION(m->audioDisconnectReq.handle));
		break;

	case HSP_AG_CONNECT_RSP:
		res = SendPadaptConnectRsp(GET_CONNECTION(m->connectRsp.handle),
								   m->connectRsp.accept);
		break;

	case HSP_AG_DATA_REQ:
		res = SendPadaptDataReq(GET_CONNECTION(m->dataReq.handle),
								m->dataReq.dta, m->dataReq.length);
		returnedDataCongestion = (BT_BOOL) (res == BT_CONGESTED);
		break;

	default:
		break;
	}

	return res;
}


/* ===========================================================================
 * Function:	HSP_AG_IndicationMsgHandler
 * Description: Dispatcher for indications received by HSP_AG
 * Returns: 	BT_OK
 * =========================================================================== */
BT_Result HSP_AG_IndicationMsgHandler(BT_DEPRECATED_MSG_CONST BT_Msg *msg) BT_LARGE
{
	HSP_AG_InMsg *BT_RAM m = (HSP_AG_InMsg*)msg;

	/*
	 * Dispatch to appropriate handler function
	 */
	switch (BT_MSG_CLASS(m->header.primitive ))
	{
	case HCI_MSG_CLASS:
		HandleHciMsg(&m->hciEvent);
		break;

	case HCI_LH_MSG_CLASS:
		HandleLinkHandlerMsg(&m->lh);
		break;

	case SDAP_MSG_CLASS:
		HandleSdapMsg(&m->sdapRsp);
		break;

	case TSRV_MSG_CLASS:
		if ( !BT_TimeoutIsCancelled(m->tsrvInd.timerId) )
		{
			HandleTsrvMsg(&m->tsrvInd);
		}
		break;

	case PADAPT_MSG_CLASS:
		HandlePadaptMsg(&m->padaptIndConf);
		break;
	}

	return BT_OK;
}


/* ===========================================================================
 * Function:	HSP_AG_HandleOrdinalNo
 * Description: Return the ordinal number for specific handle
 * =========================================================================== */
BT_U8 HSP_AG_HandleOrdinalNo(HSP_AG_Handle handle) BT_LARGE
{
	return (BT_U8) (BT_PTRDIFF((Connections*) handle, connections));
}


/*****************************************************************************/
/* Private functions														 */
/*****************************************************************************/

/* ===========================================================================
 * Function:	FreeConnection
 * Description: Frees a specific connection and assign default values.
 * =========================================================================== */
static void FreeConnection(Connections *con)
{
	con->audioConnectionHandle = INVALID_CONNECTION_HANDLE;
	con->aclConnectionHandle	= INVALID_CONNECTION_HANDLE;
	con->remoteReadyToReceive	= BT_FALSE;
	con->connectionState		= HS_DISCONNECTED_STATE;
	con->parseState 			= W_4_A;

	con->connectTimer			= 0;
	con->sdapTimer				= 0;
	con->audioConnectRspTimer   = 0;

	con->userRequestInProgress	= 0;

	con->atCommandLength		= 0;

}


/* ===========================================================================
 * Function:	GetConnectionUserRequest
 * Description: Check if there's a user request running amongst the connections
 * Returns: 	Pointer to the connection using p, otherwise 0
 * =========================================================================== */
static Connections *GetConnectionUserRequest(BT_Primitive p) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS
		   &&	con == 0 )
	{
		/*
		 * If no primitive is requested, return the first
		 * connection which has a user request running
		 */
		if (	p == 0x0000
			&&	connections[n].connectionState != HS_DISCONNECTED_STATE
			&&	connections[n].userRequestInProgress != 0x0000 )
		{
			con = &connections[n];
		}
		/*
		 * If a primitive is requested, return the first
		 * connection which user request is equal to the requested one
		 */
		else if ( p != 0x0000
			&&	connections[n].connectionState != HS_DISCONNECTED_STATE
			&&	connections[n].userRequestInProgress == p )
		{
			con = &connections[n];
		}
		else
		{
			/*
			 * Try the next connection
			 */
		}

		n++;
	}

	return con;
}


/* ===========================================================================
 * Function:	GetConnectionBdAddr
 * Description: Get a connection with a specific bdAddr.
 * Returns: 	A pointer to a connection
 * =========================================================================== */
static Connections *GetConnectionBdAddr(const BT_BdAddr bdAddr) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS
		   &&	con == 0 )
	{
		if (	connections[n].connectionState != HS_DISCONNECTED_STATE
			&&	!memcmp(bdAddr, connections[n].hsBdAddr, 6) )
		{
			con = &connections[n];
		}

		n++;
	}

	return con;
}
/* ===========================================================================
 * Function:	GetScoStatus
 * Description: Get a connection with a specific bdAddr.
 * Returns: 	A pointer to a connection
 * =========================================================================== */
BT_U8 GetScoStatus(BT_BdAddr btDev) BT_LARGE
{
	Connections *con;
	if((con = GetConnectionBdAddr(btDev)) != 0)
	{
		/* Check for the SCO Connection */
		if(con->audioConnectionHandle == INVALID_CONNECTION_HANDLE)
			return SCO_NOT_CONNECTED;
		else
			return SCO_CONNECTED;
	}
	else
		return UNKNOWN_CONNECTION;

}

/* ===========================================================================
 * Function:	GetConnectionState
 * Description: Get a connection with a specific state
 * Returns: 	A pointer to a connection
 * =========================================================================== */
static Connections *GetConnectionState(ConnectionState state) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS
		   &&	con == 0 )
	{
		if ( connections[n].connectionState == state )
		{
			con = &connections[n];
		}

		n++;
	}

	return con;
}


/* ===========================================================================
 * Function:	GetConnectionConHnd
 * Description: Get a connection with a specific HCI connections handle
 * Returns: 	A pointer to a connection
 * =========================================================================== */
static Connections *GetConnectionConHnd(HCI_LinkType linkType, BT_U16 conHnd) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS
		   &&	con == 0 )
	{
		if ( linkType == HCI_LINK_TYPE_SCO_CONNECTION
		 && connections[n].audioConnectionHandle == conHnd )
		{
			con = &connections[n];
		}
		else if (	linkType == HCI_LINK_TYPE_ACL_CONNECTION
				 && connections[n].connectionState != HS_DISCONNECTED_STATE
				 && connections[n].aclConnectionHandle == conHnd )
		{
			con = &connections[n];
		}

		n++;
	}

	return con;
}


/* ===========================================================================
 * Function:	NewConnection
 * Description: Create a connection with a specific bdAddr.
 * Returns: 	A pointer to a connection
 * =========================================================================== */
static Connections *NewConnection(const BT_BdAddr bdAddr) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS
		   &&	con == 0 )
	{
		if ( connections[n].connectionState == HS_DISCONNECTED_STATE 
			&& 	connections[n].audioConnectionHandle == INVALID_CONNECTION_HANDLE )
		{
			con = &connections[n];

			/*
			 * Init the connection and store data about the HS
			 */
			FreeConnection(con);
			memcpy(con->hsBdAddr, bdAddr, 6);
		}

		n++;
	}

	return con;
}

/* ===========================================================================
 * Function:	VerifyConnectionHandle
 * Description: Verify that the supplied handle is valid.
 * Returns: 	TRUE if valid handle
 * =========================================================================== */
static BT_BOOL VerifyConnectionHandle(const Connections *handle) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	BT_BOOL validHandle = BT_FALSE;

	while ( 	n < HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS
		   &&	!validHandle )
	{
		validHandle = (handle == &connections[n]);
		n++;
	}

	return validHandle;
}



/* ===========================================================================
 * Function:	SendPadaptDataReq
 * Description:
 * =========================================================================== */
static BT_Result SendPadaptDataReq(Connections *con, 
								   const BT_U8 *dta, BT_U16 length ) BT_LARGE
{
	PADAPT_DataReq		BT_RAM req;
	BT_Result			BT_RAM res = BT_CALL_ERROR;
	//const char			BT_RAM ok[] = {CR, LF, 'O', 'K', CR, LF, 0x00}; // CCMPD01410054 (JJ 8/11/2010)


	if ( !con )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection handle"));
	}
	
		req.header.primitive	= PADAPT_DATA_REQ;
		req.header.receiver 	= BT_PADAPT;
		req.header.sender		= HSP_AG_MODULE_ID;

		req.credits 			= 0;
		memcpy( req.bdAddr, con->hsBdAddr, 6);
		req.dlci = con->hsDlci;

		req.dataBufferPtr = dta;
		req.bufferLength = length;

		res = HSP_AG_SEND_REQ(&req);

	return res;
}


/* ===========================================================================
 * Function:	ChangeConnectionState
 * Description: Change the connection state. A function is used
 *				since it simplifies test and debug
 * =========================================================================== */
static void ChangeConnectionState(Connections *con,
								  ConnectionState newState) BT_LARGE
{
	con->connectionState = newState;
}


/* ===========================================================================
 * Function:	CancelTimers
 * Description: Cancel running timers
 * =========================================================================== */
static void CancelTimers(Connections *con) BT_LARGE
{
	if ( con->sdapTimer )
	{
		BT_TimeoutCancel(con->sdapTimer);
		con->sdapTimer				= 0;
	}
	
	if ( con->audioConnectRspTimer )
	{
		BT_TimeoutCancel(con->audioConnectRspTimer);
		con->audioConnectRspTimer   = 0;
	}
}


/* ===========================================================================
 * Function:	SendToUser
 * Description: Create header and send message to registered user
 * =========================================================================== */
static void SendToUser(BT_Msg *ind, BT_Primitive primitive) BT_LARGE
{
	ind->header.primitive	= primitive;
	ind->header.receiver	= hspAgUser;
	ind->header.sender		= HSP_AG_MODULE_ID;

	if ( hspAgUser != 0 )
	{
		(void)HSP_AG_SEND_IND(ind);
	}
	else
	{
		LOG_INFO((HSP_AG_MODULE_ID, "No service registered with HSP_AG profile"));
	}
}


/* ===========================================================================
 * Function:	SendReq
 * Description: Fill in sender field of header and send a request message
 *				Optionally a timer is started if sending is congested .
 *
 * Parameters:	msg
 *					Message to send
 *
 *				userTimerId
 *					ID of timer to start if sending is BT_CONGESTED. If this
 *					is 0, no timer is started.
 *
 *				timerId
 *					If a timer was started, its id is store in the variable
 *					pointed to by this variable
 *
 *				timeout
 *					The length of the timeout if a timer is started.
 *
 *				con
 *					Pointer to the connection used
 *
 * Returns: 	BT_FALSE if sending did not succeed (even if timer
 *				was started). BT_TRUE if message was sent.
 * =========================================================================== */
static BT_BOOL SendReq(BT_Msg *msg, BT_U8 userTimerId,
					   BT_TimerId *timerId, BT_U32 timeout,
					   Connections *con) BT_LARGE
{
	BT_BOOL 	BT_RAM	sentRightAway = BT_TRUE;
	BT_Result	BT_RAM	res;
	

	msg->header.sender		= HSP_AG_MODULE_ID;

	res = HSP_AG_SEND_REQ(msg);
	
	if ( res == BT_CONGESTED )
	{
		/*
		 * If a timer id was passed to this function, start it
		 * and indicate to caller that the request was delayed
		 */
		if ( userTimerId != 0 )
		{
			*timerId = BT_TimeoutReq(HSP_AG_MODULE_ID, userTimerId,
									con, timeout);
		}

		sentRightAway = BT_FALSE;
	}

	return sentRightAway;
}

/* ===========================================================================
 * Function:	SetupHciUsage
 * Description: Register with HCI driver and subscribe to events
 * Returns: 	BT_TRUE if success, BT_FALSE otherwise
 * =========================================================================== */
static BT_BOOL SetupHciUsage(void) BT_LARGE
{
	BT_BOOL BT_RAM res = BT_TRUE;

	if ( HCI_RegisterUser(HSP_AG_MODULE_ID) != BT_OK )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Register as user of HCI Driver failed\n"));
		res = BT_FALSE;
	}
	else
	{
		(void)HCI_EventSubscribe( HSP_AG_MODULE_ID, HCI_CONNECTION_REQUEST_EVENT);
		(void)HCI_EventSubscribe( HSP_AG_MODULE_ID, HCI_LH_AUDIO_CONNECT_COMPLETE_IND);
		(void)HCI_EventSubscribe( HSP_AG_MODULE_ID, HCI_DISCONNECTION_COMPLETE_EVENT);

		(void)HCI_EventSubscribe( HSP_AG_MODULE_ID, HCI_CONTROLLER_RESTARTED_IND);
	}

	return res;
}


/* ===========================================================================
 * Function:	SendPadaptConnectReq
 * Description: Create and send a PADAPT_CONNECT_REQ.
 * =========================================================================== */
static void SendPadaptConnectReq(Connections *con) BT_LARGE
{
	PADAPT_ConnectReq	BT_RAM req;

	req.header.receiver 	= BT_PADAPT;
	req.header.primitive	= PADAPT_CONNECT_REQ;

	memcpy( req.bdAddr, con->hsBdAddr, 6 );
	req.serverChannel	= con->hsServerChannel;

	req.serviceId		= BT_HSP_HS;
	req.maxFramesize	= HSP_AG_CFG_MAX_FRAME_SIZE;
	req.minFramesize	= HSP_AG_CFG_MIN_FRAME_SIZE;
	req.framesize		= HSP_AG_CFG_FRAME_SIZE;
	req.creditsToSend	= HSP_AG_CFG_CREDITS_TO_SEND;
	req.minCredits		= HSP_AG_CFG_MIN_CREDITS;
	req.interfaceType	= PADAPT_INTERFACE_SIMPLE;
	req.initialCredits	= HSP_AG_CFG_INITIAL_CREDITS;

	(void)SEND_REQ(&req, TIMER_ID_PADAPT_CONNECT_REQ, &con->connectTimer,
					HSP_AG_CFG_RESEND_TIME, con);
}


/* ===========================================================================
 * Function:	SendAudioConnectionStatusInd
 * Description: Send HSP_AG_AUDIO_CONNECTION_STATUS_IND to registered user
 * =========================================================================== */
static void SendAudioConnectionStatusInd(Connections *con,
										BT_BOOL audioConnected,
										HCI_Error reason) BT_LARGE
{
	HSP_AG_AudioConnectionStatusInd BT_RAM ind;
	
	memcpy(ind.bdAddr,con->hsBdAddr,sizeof(BT_BdAddr));
	ind.audioConnected			= audioConnected;
	ind.handle					= con;
	ind.audioConnectionHandle	= con->audioConnectionHandle;
	ind.linkType				= con->audioLinkType;
	ind.reason					= reason;

	SEND_TO_USER(&ind, HSP_AG_AUDIO_CONNECTION_STATUS_IND );
}

/* ===========================================================================
 * Function:  SendHciLhAudioConnectReq
 * Description: Send HCI_LH_AUDIO_CONNECT_REQ
 * =========================================================================== */
static BT_Result SendHciLhAudioConnectReq(const Connections *con) BT_LARGE
{
	HCI_LH_AudioConnectReq	BT_RAM audioConnectReq;

	/*
	 * Message header
	 */
	audioConnectReq.header.sender 	= HSP_AG_MODULE_ID;
	audioConnectReq.header.primitive= HCI_LH_AUDIO_CONNECT_REQ;
	audioConnectReq.header.receiver	= BT_LookupProfile(BT_HCI_LH);

	/*
	 * Message parameters
	 */
	audioConnectReq.connectionHandle	= con->aclConnectionHandle;
	audioConnectReq.setupOptions		= (audioSetupOptions?audioSetupOptions:scoSetupSettings);

	/*
	 * Send request
	 */
	return HSP_AG_SEND_REQ((BT_Msg*)&audioConnectReq);
}

/* ===========================================================================
 * Function:  SendDisconnectScoConnection
 * Description: Send HCI_DISCONNECT for SCO link
 * =========================================================================== */
static BT_Result SendDisconnectScoConnection(const Connections *con) BT_LARGE
{
	HCI_Disconnect BT_RAM disconnect;

	/*
	 * Message header
	 */
	disconnect.header.sender	= HSP_AG_MODULE_ID;
	disconnect.header.primitive = HCI_DISCONNECT;
	disconnect.header.receiver	= BT_HCI;

	/*
	 * Message parameters
	 */
	disconnect.connectionHandle = con->audioConnectionHandle;
	disconnect.reason = HCI_OTHER_END_TERMINATED_USER_ENDED_CONNECTION;

	/*
	 * Send request
	 */
	return HSP_AG_SEND_REQ((BT_Msg*)&disconnect);
}

/* ===========================================================================
 * Function:  SendAclDisconnect
 * Description: Creates and sends a HCI disconnection message.
 * Returns: 	-
 * ==========================================================================*/
static void SendAclDisconnect(BT_BdAddr bdAddr )
{
	BT_U16			connectionHandle;
	HCI_Disconnect disconnect;

	if ( HCI_GetConnectionHandle(bdAddr, &connectionHandle ) )
	{
		/* 
		 * Message header 
		 */
		disconnect.header.primitive = HCI_DISCONNECT;
		disconnect.header.sender = HSP_AG_MODULE_ID;
		disconnect.header.receiver = BT_HCI;

		/* 
		 * Message parameters 
		 */
		disconnect.connectionHandle = connectionHandle;
		disconnect.reason = HCI_OTHER_END_TERMINATED_USER_ENDED_CONNECTION;

		if ( BT_SEND(&disconnect) == BT_CONGESTED )
		{
			(void)BT_TimeoutReq(HSP_AG_MODULE_ID, 
			   TIMER_ID_DM_DISCONNECT_ACL,
									0, 
									HSP_AG_CFG_RESEND_TIME);
		}
		
	}
}

/* ===========================================================================
 * Function:	HandleAudioConnectReq
 * Description: Handle audio transfer requested by AG side
 * =========================================================================== */
static BT_Result HandleAudioConnectReq(Connections *con) BT_LARGE
{
	BT_Result BT_RAM res = BT_CALL_ERROR;

	ChangeConnectionState(con, HS_ACL_CONNECTED_STATE); // Nirav
	if ( !VerifyConnectionHandle(con) ) 
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection handle"));
	}
	/*
	 * Check if there's an user request running.
	 * If so, wait until its completed.
	 */
	else if ( GetConnectionUserRequest(0) )
	{
		res = BT_CONGESTED;
	}
	else if (	con->connectionState != HS_ACL_CONNECTED_STATE
		  || con->audioConnectionHandle != INVALID_CONNECTION_HANDLE )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection state (ACL %d, SCO %d)",
				(int)con->connectionState,
		  (int)con->audioConnectionHandle));
	}
	else if ( (res = SendHciLhAudioConnectReq(con)) == BT_OK )
	{
		con->userRequestInProgress	= HSP_AG_AUDIO_CONNECT_REQ;
		res 						= BT_OK;
	}
	else
	{
		/*
		 * Just return
		 */
	}

	return res;
}


/* ===========================================================================
 * Function:	HandleAudioConnectRsp
 * Description: Handle audio transfer requested by HS side
 * =========================================================================== */
static BT_Result HandleAudioConnectRsp(Connections *con, BT_BOOL accept) BT_LARGE
{
	BT_Result BT_RAM res = BT_CALL_ERROR;

	if ( !VerifyConnectionHandle(con) ) 
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection handle"));
	}
	/*
	 * Check if there's an user request running.
	 * If so, wait until its completed.
	 */
	else if ( GetConnectionUserRequest(0) )
	{
		res = BT_CONGESTED;
	}
	else if (	con->connectionState != HS_ACL_CONNECTED_STATE
		  || con->audioConnectionHandle != INVALID_CONNECTION_HANDLE )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection state (ACL %d, SCO %d)",
				(int)con->connectionState,
		  (int)con->audioConnectionHandle));
	}
	else
	{
		con->audioAccept = accept;
		SendHciLhScoConnectRsp(con);
		res = BT_OK;
	}

	return res;
}


/* ===========================================================================
 * Function:  HandleAudioDisconnectReq
 * Description: Handle audio transfer requested by HS side
 * =========================================================================== */
static BT_Result HandleAudioDisconnectReq(Connections *con) BT_LARGE
{
	BT_Result BT_RAM res = BT_CALL_ERROR;

	if ( !VerifyConnectionHandle(con) )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection handle"));
	}
	/*
	 * Check if there's an user request running.
	 * If so, wait until its completed.
	 */
	else if ( GetConnectionUserRequest(0) )
	{
		res = BT_CONGESTED;
	}
	else if (	con->connectionState != HS_ACL_CONNECTED_STATE
		  || con->audioConnectionHandle == INVALID_CONNECTION_HANDLE )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection state (ACL %d, SCO %d)",
				(int)con->connectionState,
		  (int)con->audioConnectionHandle));
	}
	else if ( (res = SendDisconnectScoConnection(con)) == BT_OK )
	{
		con->userRequestInProgress = HSP_AG_AUDIO_DISCONNECT_REQ;
	}
	else
	{
		/*
		 * Just return
		 */
	}

	return res;
}


/* ===========================================================================
 * Function:	SendServiceSearchAttrReq
 * Description: Send an SDAP_SERVICE_SEARCH_ATTR_REQ to search for HS AG
 *				service
 * =========================================================================== */
static void SendSdapServiceSearchAttrReq(Connections *con) BT_LARGE
{
	SDAP_ServiceSearchAttrReq	BT_RAM attrReq;

	/* Message header */
	attrReq.header.primitive	= SDAP_SERVICE_SEARCH_ATTR_REQ;
	attrReq.header.receiver 	= BT_SDAP;

	/* Message parameters */
	attrReq.noOfBdAddr	= 1;

	attrReq.bdAddr = &con->hsBdAddr;

	attrReq.remDevRelation	= SDAP_NO_RELATION;
	attrReq.searchPattern	= (SDP_ConstDePtr)HSP_AG_serviceSearchPattern;
	attrReq.attributeId 	= (SDP_ConstDePtr)connectAttributeIdList;
	attrReq.getRemDevName	= BT_FALSE;

	attrReq.maxNoOfBytes	= HSP_AG_SDAP_RECEIVE_BUFFER_SIZE;
	attrReq.maxSearchTime	= MAX_SDAP_SEARCH_TIME;

	(void)SEND_REQ(&attrReq, TIMER_ID_SDAP_SEARCH,
					&con->sdapTimer, HSP_AG_CFG_RESEND_TIME, con);
}


/* ===========================================================================
 * Function:	SendLinkSettingsReq
 * Description:
 * =========================================================================== */
static void SendLinkSettingsReq(Connections *con) BT_LARGE
{
	HCI_LH_LinkSettingsReq	BT_RAM req;

	req.header.primitive	= HCI_LH_LINK_SETTINGS_REQ;
	req.header.sender		= HSP_AG_MODULE_ID;
	req.header.receiver 	= BT_LookupProfile( BT_HCI_LH );

	req.connectionHandle	= con->aclConnectionHandle;
	req.settings			= HSP_AG_CFG_LINK_HANDLER_SETTINGS;

	(void)SEND_REQ(&req, TIMERID_RESEND_LINK_SETTINGS,
					&con->connectTimer, HSP_AG_CFG_RESEND_TIME, con);
}

/* ===========================================================================
 * Function:	SendLinkStatusInd
 * Description: Send a HSP_AG_LINK_STATUS_IND to registered user
 * =========================================================================== */
static void SendLinkStatusInd(Connections *con) BT_LARGE
{
	HSP_AG_LinkStatusInd BT_RAM ind;

	if ( con->connectionState == HS_ACL_CONNECTED_STATE )
	{
		ind.aclStatus = HS_ACL_CONNECTED_STATE;
	}
	else
	{
		ind.aclStatus = HS_DISCONNECTED_STATE;
	}
	
	ind.handle = con;
        memcpy(ind.bdAddr,con->hsBdAddr,6);
	SEND_TO_USER(&ind, HSP_AG_LINK_STATUS_IND );
}

/* ===========================================================================
 * Function:	SendConnectCfm
 * Description: If CONNECT_REQ was sent by user, report result
 * =========================================================================== */
static void SendConnectCfm( Connections *con,
						   HSP_AG_Result res,
						   RFCOMM_DlcFrameSize	fsize ) BT_LARGE
{
	HSP_AG_ConnectCfm BT_RAM cfm;

	/*
	 * Clear the user request in progress
	 */
	con->userRequestInProgress	= 0;

	cfm.result					= res;
	cfm.framesize				= fsize;
	memcpy(cfm.bdAddr, con->hsBdAddr, 6);

	/*
	 * Only return a handle if the result is a success.
	 */
	cfm.handle					= (res == HSP_AG_OK) ? con : 0;

	SEND_TO_USER(&cfm, HSP_AG_CONNECT_CFM );

}

/* ===========================================================================
 * Function:	RunConnectionSetupStateMachine
 * Description: If service level connection is not established, perform the
 *				next step in the connection setup sequence.
 * =========================================================================== */
static void RunConnectionSetupStateMachine(Connections *con) BT_LARGE
{

	switch ( con->connectionState )
	{
	case HS_DISCONNECTED_STATE:
		con->hsServerChannel = 0;
		SendSdapServiceSearchAttrReq(con);
		ChangeConnectionState(con, HS_SERVICE_SEARCH_STATE);
		break;

	case HS_SERVICE_SEARCH_COMPLETE_STATE:
		if ( con->hsServerChannel != 0 )
		{
			con->hsDlci = 0;
			SendPadaptConnectReq(con);
			ChangeConnectionState(con,	PADAPT_CONNECTING_STATE );
		}
		else
		{
			FreeConnection(con);
			/*
			 * Report failure
			 */
			SendConnectCfm(con, HSP_AG_SERVICE_SEARCH_FAILED, 0);
		}
		break;

	case PADAPT_CONNECTED_STATE:
		if ( con->hsDlci != 0 )
		{
			ChangeConnectionState(con, HS_ACL_CONNECTED_STATE);
			con->parseState = W_4_A;
			//SendLinkSettingsReq(con);

		}
		else
		{
			FreeConnection(con);
			/*
			 * Report failure
			 */
			SendConnectCfm(con, HSP_AG_RFCOMM_CONNECT_FAILED, 0);
		}
		break;
	}
}


/* ===========================================================================
 * Function:	SendAudioConnectCfm
 * Description:
 * =========================================================================== */
static void SendAudioConnectCfm(Connections *con, BT_BOOL success) BT_LARGE
{
	HSP_AG_AudioConnectCfm cfm;

	/*
	 * Confirm sent, no request in progress
	 */
	con->userRequestInProgress = 0;

	cfm.success = success;
	cfm.handle	= con;

	SEND_TO_USER(&cfm, HSP_AG_AUDIO_CONNECT_CFM );
}

/* ===========================================================================
 * Function:	SendAudioConnectInd
 * Description:
 * =========================================================================== */
static void SendAudioConnectInd(Connections *con) BT_LARGE
{
	HSP_AG_AudioConnectInd ind;

	ind.handle	= con;

	SEND_TO_USER(&ind, HSP_AG_AUDIO_CONNECT_IND);
}


/* ===========================================================================
 * Function:  SendAudioDisconnectCfm
 * Description:
 * =========================================================================== */
static void SendAudioDisconnectCfm(Connections *con, BT_BOOL success) BT_LARGE
{
	HSP_AG_AudioDisconnectCfm cfm;

	/*
	 * Confirm sent, no request in progress
	 */
	con->userRequestInProgress = 0;

	cfm.success = success;
	cfm.handle	= con;

	SEND_TO_USER(&cfm, HSP_AG_AUDIO_DISCONNECT_CFM );
}


/* ===========================================================================
 * Function:	StartHandlingIncomingConnectionAttempt
 * Description:
 * =========================================================================== */
static void StartHandlingIncomingConnectionAttempt(Connections *con, 
												   const BT_BdAddr bdAddr) BT_LARGE
{
        HSP_AG_ConnectInd BT_RAM ind;
	memcpy(con->hsBdAddr, bdAddr, 6);

	ChangeConnectionState(con, INCOMING_HS_CONNECT_REQUEST_STATE);

	/*
	 * Save connection handle for future use
	 */
	(void)HCI_GetConnectionHandle( con->hsBdAddr, &con->aclConnectionHandle);

	/*
	 * Start the service search to retrieve the AGs features
	 */
        /* Bypass the Service Serach */
	/* SendSdapServiceSearchAttrReq(con); */
        
        ChangeConnectionState(con, INCOMING_HS_W_4_CONNECT_RSP);

	/*
	 * Notify user
	 */
	memcpy( ind.bdAddr, con->hsBdAddr, sizeof( BT_BdAddr ) );
	ind.framesize	= con->framesize;
	ind.handle		= con;

        SEND_TO_USER(&ind, HSP_AG_CONNECT_IND);
}


/* ===========================================================================
 * Function:  SendPadaptDisconnectReq
 * Description:
 * =========================================================================== */
static BT_Result SendPadaptDisconnectReq(Connections *con) BT_LARGE
{
	PADAPT_DisconnectReq BT_RAM req;
	BT_Result				BT_RAM res = BT_CONGESTED;

	req.header.sender		= HSP_AG_MODULE_ID;
	req.header.primitive 	= PADAPT_DISCONNECT_REQ;
	req.header.receiver 	= BT_PADAPT;

	req.dlci				= con->hsDlci;
	memcpy(req.bdAddr, con->hsBdAddr, sizeof(BT_BdAddr));

	if ( 	con->connectTimer == 0
		&& 	(res = HSP_AG_SEND_REQ((BT_Msg*)&req)) == BT_OK )
	{
		/*
		 * Cancel possibly running timers
		 */
		CancelTimers(con);

		/*
		 * Free the connection
		 */
		FreeConnection(con);

		/*
		 * Switch context before sending link status ind
		 */
		con->connectTimer = BT_TimeoutReq(HSP_AG_MODULE_ID,
									TIMER_ID_HS_DISCONNECTED_STATE_IND,
										con, 0);
	}

	return res;
}


/* ===========================================================================
 * Function:  HandleHspAgDisconnectReq
 * Description:
 * =========================================================================== */
static BT_Result HandleHspAgDisconnectReq(Connections *con) BT_LARGE
{
	BT_Result BT_RAM res = BT_CALL_ERROR;

	if ( !VerifyConnectionHandle(con) )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection handle"));
	}
	else if ( con->audioConnectionHandle != INVALID_CONNECTION_HANDLE )
	{
		/*
		 * The audio connection has to be disconnected first
		 */
		LOG_INFO((HSP_AG_MODULE_ID, "Illegal action. Audio not disconnected"));
		res = BT_CALL_ERROR;
	}
	
	else if ( con->connectionState == HS_ACL_CONNECTED_STATE )
	{
		/*
		 * Disconnect using PADAPT (there is no response to this command)
		 */
		res = SendPadaptDisconnectReq(con);
	}
	else
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection state (ACL %d)",
				(int)con->connectionState));
	}

	return res;
}

/* ===========================================================================
 * Function:	HandleHspAgConnectReq
 * Description:
 * =========================================================================== */
static BT_Result HandleHspAgConnectReq(const BT_BdAddr bdAddr) BT_LARGE
{
	BT_Result BT_RAM res = BT_OK;
	Connections *con;

	/*
	 * Check if there's an user request running.
	 * If so, wait until its completed.
	 */
	if ( GetConnectionUserRequest(0) )
	{
		res = BT_CONGESTED;
	}
	/*
	 * Only one connection per bdAddr is allowed
	 */
	else if ( (con = GetConnectionBdAddr(bdAddr)) != 0 )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Only one connection per bdAddr is allowed"));
		res = BT_CALL_ERROR;
	}
	/*
	 * Try to find a spare connection. Return BT_CONGESTED if everyone
	 * is allocated.
	 */
	else if ( (con = NewConnection(bdAddr)) == 0 )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "No resources left, max number of "
			"connections reached."));
		res = BT_CONGESTED;
	}
	else
	{
		/*
		 * Save primitive so that appropriate cfm can be sent after connect
		 * complete/failed
		 */
		con->userRequestInProgress = HSP_AG_CONNECT_REQ;

		/*
		 * Switch context before running connect state machine to avoid sending result
		 * in request handler in case an error occurs
		 */
		con->connectTimer = BT_TimeoutReq(HSP_AG_MODULE_ID,
									TIMER_ID_RUN_CONNECTION_STATE_MACHINE,
										con, 0);
	}

	return res;
}


/* ===========================================================================
 * Function: 	SendHciLhScoConnectRsp
 * Description: Send a HCI_LH_AUDIO_CONNECT_RSP to accept an audio
 *				connection
 * =========================================================================== */
static void SendHciLhScoConnectRsp(Connections *con) BT_LARGE
{
	HCI_LH_AudioConnectRsp	BT_RAM rsp;

	rsp.header.primitive	= HCI_LH_AUDIO_CONNECT_RSP;
	rsp.header.receiver		= BT_LookupProfile(BT_HCI_LH);

	memcpy(rsp.bdAddr, con->hsBdAddr, 6);
	rsp.linkType = con->audioLinkType;
	rsp.status = (HCI_Error)(con->audioAccept?HCI_OK:HCI_HOST_REJECTED_LIMITED_RESOURCES);
	rsp.audioAcceptOptions = audioAcceptOptions;
	/*
	 * Just discard audio connection requests if there are several at
	 * the same time
	 */
	if ( !con->audioConnectRspTimer )
	{
		(void) SEND_REQ(&rsp, TIMER_ID_SCO_CONNECT_RESPONSE,
						&con->audioConnectRspTimer, HSP_AG_CFG_RESEND_TIME, con);
	}
}


/* ===========================================================================
 * Function:	SendRestartResponse
 * Description: Send HCI_CONTROLLER_RESTARTED_RSP
 * =========================================================================== */
static void SendRestartResponse(void) BT_LARGE
{
	BT_Msg BT_RAM msg;

	msg.header.primitive	= HCI_CONTROLLER_RESTARTED_RSP;
	msg.header.receiver 	= BT_HCI;

	(void)SEND_REQ(&msg, TIMER_ID_RESTART_RESPONSE,
				&restartedRspTimer, HSP_AG_CFG_RESEND_TIME, 0);
}


/* ===========================================================================
 * Function:	RestartedCfmUserCommand
 * Description: Sends a negative confirm to any user command
 *				that may be running
 * =========================================================================== */
static void RestartedCfmUserCommand(Connections *con) BT_LARGE
{
	/*
	 * Send appropriate confirm negative,
	 * userRequestInProgressis cleared in each
	 * send cfm function
	 */
	switch ( con->userRequestInProgress )
	{
	case 0:
		/*
		 * No active request
		 */
		break;

	case HSP_AG_AUDIO_CONNECT_REQ:
		SendAudioConnectCfm(con, BT_FALSE);
		break;

	case HSP_AG_AUDIO_DISCONNECT_REQ:
		SendAudioDisconnectCfm(con, BT_TRUE);
		break;

	case HSP_AG_CONNECT_REQ:
		SendConnectCfm(con, HSP_AG_HOST_CONTROLLER_RESTART, 0 );
		break;

	default:
		LOG_INFO(( HSP_AG_MODULE_ID,
			"Restart handling, unknown user request %d, Please report to Mecel",
				(int) con->userRequestInProgress ));
	}
}


/* ===========================================================================
 * Function:	HandleHciMsg
 * Description: Main handler for all received HCI messages
 * =========================================================================== */
static void HandleHciMsg(const HCI_Event *m) BT_LARGE
{
	BT_U8 BT_RAM n;
	Connections * BT_RAM con;

	switch ( m->header.primitive )
	{
	case HCI_CONNECTION_REQUEST_EVENT:
		/*
		 * If the audio connection is requested from the bdAddress that
		 * is set as HS, and if this profile is active, accept audio 
		 * connection 
		 */
		if ( ( m->connectionRequestEvent.linkType == HCI_LINK_TYPE_SCO_CONNECTION
			|| m->connectionRequestEvent.linkType == HCI_LINK_TYPE_ESCO_CONNECTION )
			&&	(con = GetConnectionBdAddr(m->connectionRequestEvent.bdAddr)) != 0
			&&	con->connectionState != HS_DISCONNECTED_STATE )
		{
			con->audioLinkType = m->connectionRequestEvent.linkType;

			/*
			 * Respond automatically or handle the request to the user
			 */
			if ( USE_MANUAL_AUDIO )
			{
				SendAudioConnectInd(con);
			}
			else
			{
				/*
				 * Always accept
				 */
				con->audioAccept = BT_TRUE;
				SendHciLhScoConnectRsp(con);
			}
		}
		break;


	case HCI_DISCONNECT_STATUS_EVENT:
		if ( (con = GetConnectionUserRequest(HSP_AG_AUDIO_DISCONNECT_REQ)) != 0
		 	&&  m->disconnectStatusEvent.status != HCI_OK )
		{
		 	SendAudioDisconnectCfm(con, BT_FALSE);
		}
		break;

	case HCI_DISCONNECTION_COMPLETE_EVENT:
		/*
		 * Notify user if audio link goes down
		 */
		if ( (con = GetConnectionConHnd(HCI_LINK_TYPE_SCO_CONNECTION, 
				m->disconnectionCompleteEvent.connectionHandle)) == 0 )
		{
				/*
				 * Just drop
				 */
		}
		else if ( m->disconnectionCompleteEvent.status == HCI_OK )
		{
			con->audioConnectionHandle	= INVALID_CONNECTION_HANDLE;
			con->audioLinkType			= 0xff;

			if ( con->userRequestInProgress == HSP_AG_AUDIO_DISCONNECT_REQ )
			{
				SendAudioDisconnectCfm(con, BT_TRUE);
			}

			SendAudioConnectionStatusInd(con, BT_FALSE, m->disconnectionCompleteEvent.reason);
		}
		else if ( con->userRequestInProgress == HSP_AG_AUDIO_DISCONNECT_REQ )
		{
			SendAudioDisconnectCfm(con, BT_FALSE);
		}
		else
		{
			/*
			 * Just drop
			 */
		}
		break;


	case HCI_LH_AUDIO_CONNECT_COMPLETE_IND:
		/*
		 * If this is SCO connection complete from the bdAddress that
		 * is set as HS, and if this profile is active, check and report status
		 */
		if ( (con = GetConnectionBdAddr(m->lhAudioConnectCompleteInd.bdAddr)) != 0
			&&	con->connectionState != HS_DISCONNECTED_STATE
			&&	m->lhAudioConnectCompleteInd.status == HCI_OK )
		{
			con->audioConnectionHandle = m->lhAudioConnectCompleteInd.audioConnectionHandle;
			con->audioLinkType = m->lhAudioConnectCompleteInd.linkType;

			SendAudioConnectionStatusInd(con, BT_TRUE, HCI_OK);
		}
		break;

	case HCI_CONTROLLER_RESTARTED_IND :
		BT_TimeoutCancelAll( HSP_AG_MODULE_ID );
		returnedDataCongestion = BT_FALSE;
		/*
		 * Reset module state and respond to HCI
		 */
		for (n = 0; n < HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS; n++)
		{
			/*
			 * Try to send a negative confirm. The userRequestInProgress will
			 * take care if the confirm is sent or not
			 */
			RestartedCfmUserCommand(&connections[n]);

			ChangeConnectionState(&connections[n],	HS_DISCONNECTED_STATE );

			if ( connections[n].audioConnectionHandle != INVALID_CONNECTION_HANDLE )
			{
				connections[n].audioLinkType = 0xff;
				SendAudioConnectionStatusInd(&connections[n], BT_FALSE, HCI_UNSPECIFIED_ERROR);
			}
			connections[n].audioConnectionHandle = INVALID_CONNECTION_HANDLE;
		}

		SendRestartResponse();
		break;

	default:
		break;
	}

}


/* ===========================================================================
 * Function:	HandleSdapSearchCompleteRsp
 * Description: Handle SDAP_SEARCH_COMPLETE_RSP
 * =========================================================================== */
static void HandleSdapSearchCompleteRsp(void) BT_LARGE
{
	Connections * BT_RAM con;
	HSP_AG_ConnectInd BT_RAM ind;

	/*
	 * If service search was performed as part of an outgoing
	 * connection, continue connection setup sequence
	 */
	if ( (con = GetConnectionState(HS_SERVICE_SEARCH_STATE)) != 0 )
	{
		ChangeConnectionState(con, HS_SERVICE_SEARCH_COMPLETE_STATE);
		RunConnectionSetupStateMachine(con);
	}
	else if ( (con = GetConnectionState(INCOMING_HS_CONNECT_REQUEST_STATE)) != 0 )
	{
		ChangeConnectionState(con, INCOMING_HS_W_4_CONNECT_RSP);

		/*
		 * Notify user
		 */
		memcpy( ind.bdAddr, con->hsBdAddr, sizeof( BT_BdAddr ) );
		ind.framesize	= con->framesize;
		ind.handle		= con;

		SEND_TO_USER(&ind, HSP_AG_CONNECT_IND);
	}
}


/* ===========================================================================
 * Function:	HandleSdapServiceSearchAttrRsp
 * Description: Extract attribute values from the search result
 *				returned by SDAP
 * =========================================================================== */
static void HandleSdapServiceSearchAttrRsp(
								const SDAP_ServiceSearchAttrRsp *rsp) BT_LARGE
{
	HSP_AG_Position 		BT_RAM position;
	HSP_AG_HsServiceRecord	BT_RAM hsServiceRecord;
	Connections 		*	BT_RAM con;

	if (	(con = GetConnectionBdAddr(rsp->bdAddr)) != 0
		&&	rsp->result == SDAP_OK )
	{
		position = HSP_AG_StartInterpretSearchResult(rsp->dataBuffer);

		while ( position != 0 )
		{
			if ( HSP_AG_InterpretSearchResult(rsp->dataBuffer,
											  rsp->noOfBytes,
											&hsServiceRecord,
											&position) != BT_OK )
			{
				/*
				 * Drop and take the next one if any
				 */
			}
			else if ( !(hsServiceRecord.validAttributes & HSP_AG_VALID_SERVICE_CLASS_ID_LIST) )
			{
				/*
				 * The service class id is missing even if it's mandatory.
				 * Use the information and hope it's a handfree unit.
				 */
				con->hsServerChannel = hsServiceRecord.serverChannel;

			}
			else if ( hsServiceRecord.serviceClassIdList[0] == 0x1108 )
			{
				/*
				 * The service class id is present and it's a handsfree unit
				 */
				con->hsServerChannel = hsServiceRecord.serverChannel;

			}
			else
			{
				/*
				 * The service class id is present and it's a audio gateway
				 * Skip and try the next one if any.
				 */
			}
		}
	}
}


/* ===========================================================================
 * Function:	HandleSdapMsg
 * Description: Handle all messages received from SDAP module.
 * =========================================================================== */
static void HandleSdapMsg(const SDAP_Rsp *m) BT_LARGE
{
	switch ( m->header.primitive )
	{
	case SDAP_SERVICE_SEARCH_ATTR_RSP:
		HandleSdapServiceSearchAttrRsp(&m->searchAttrRsp);
		break;

	case SDAP_SEARCH_COMPLETE_RSP:
		HandleSdapSearchCompleteRsp();
		break;
	}
}


/* ===========================================================================
 * Function:	HandlePadaptConnectCfm
 * Description: Handle PADAPT_CONNECT_CFM message
 * =========================================================================== */
static void HandlePadaptConnectCfm(const PADAPT_ConnectCfm *connectCfm) BT_LARGE
{
	Connections * BT_RAM con;

	/*
	 * Find the connection associated with the bdAddr
	 */
	if ( (con = GetConnectionBdAddr(connectCfm->bdAddr)) != 0 )
	{
		ChangeConnectionState(con, PADAPT_CONNECTED_STATE);

		if ( connectCfm->accept )
		{
			con->hsDlci = connectCfm->dlci;
			RestartAtCommandParser(con);

			con->remoteReadyToReceive = BT_FALSE;
			SendConnectCfm(con, HSP_AG_OK, connectCfm->framesize );
		}

		/*
		 * Save connection handle for future use
		 */
		(void)HCI_GetConnectionHandle(con->hsBdAddr, &con->aclConnectionHandle);

		RunConnectionSetupStateMachine(con);
	}
	else
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Discarding PADAPT_CONNECT_CFM since no valid connection"));
	}
}


/* ===========================================================================
 * Function:	RestartAtCommandParser
 * Description:
 * =========================================================================== */
static void RestartAtCommandParser(Connections *con) BT_LARGE
{
	con->atCommandLength	= 0;
	con->parseState 		= W_4_A;
}

/* ===========================================================================
 * Function:	AddToAtCommandBuffer
 * Description: Add one character to the at command buffer
 * =========================================================================== */
static void AddToAtCommandBuffer(Connections *con, BT_U8 c)
{
	/*
	 * Add to buffer if there is room, last position is always 0
	 */
	if ( con->atCommandLength < HSP_AG_CFG_AT_COMMAND_MAX_SIZE - 1)
	{
		con->atCommandBuffer[con->atCommandLength++]	= c;
		con->atCommandBuffer[con->atCommandLength]	= '\0';
	}
	else
	{
		LOG_INFO(( HSP_AG_MODULE_ID,
			"AT command too long, discarding character \'%c\'\n", c));
	}
}



/* ===========================================================================
 * Function:	HandleAtCommand
 * Description:
 * =========================================================================== */
static void HandleAtCommand(Connections *con, char *atCommand)
{
	HSP_AG_AtInd		BT_RAM	ind;
	
	ind.details = atCommand;
	ind.handle	= con;

	SEND_TO_USER(&ind, HSP_AG_AT_IND);
}
/* ===========================================================================
 * Function:	HandleAtCommand
 * Description:
 * =========================================================================== */
static void HandleNibblerByte(Connections *con, BT_U8* nibbler, BT_U8 size)
{
	HSP_AG_NibblerInd		BT_RAM	ind;
	
	ind.nibbler = nibbler;
	ind.handle	= con;
	ind.size	  = size;

	SEND_TO_USER(&ind, HSP_AG_NIBBLER_IND);
}

/* ===========================================================================
 * Function:	ParseCharacter
 * Description: Parser for result codes. Looks for pattern
 *				AT .... <CR> When found, a handler for complete
 *				At commands is called.
 * =========================================================================== */
static void ParseCharacter(Connections *con, BT_U8 c) BT_LARGE
{
	switch ( con->parseState )
	{
	case W_4_A:
		if ( c == 'A' || c == 'a' )
		{
			AddToAtCommandBuffer(con, c);
			con->parseState = W_4_T;
		}
		break;

	case W_4_T:
		if ( c == 'T' || c == 't' )
		{
			AddToAtCommandBuffer(con, c);
			con->parseState = W_4_CMD;
		}
		else if ( c == 'A' || c == 'a' )
		{
			RestartAtCommandParser(con);
			AddToAtCommandBuffer(con, c);
			con->parseState = W_4_T;
		}
		else
		{
			RestartAtCommandParser(con);
		}
		break;

	case W_4_CMD:
		if ( c != CR )
		{
			AddToAtCommandBuffer(con, c);
		}
		else
		{
			HandleAtCommand(con, (char*)con->atCommandBuffer);
			RestartAtCommandParser(con);
		}
		break;
	}
}

/* ===========================================================================
 * Function:	SendNibbleByte
 * Description: Send the NibbleByte for Handling
 * =========================================================================== */
static void SendNibbleByte(Connections *con, BT_U8* c, BT_U8 size) BT_LARGE
{
  memcpy(con->atCommandBuffer, c, size);
  con->atCommandLength = size;

  HandleNibblerByte(con, (BT_U8*)con->atCommandBuffer, (BT_U8)con->atCommandLength);
}

/* ===========================================================================
 * Function:	HandlePadaptDataInd
 * Description: Handler for PADAPT_DATA_IND messages
 * =========================================================================== */
static void HandlePadaptDataInd(const PADAPT_DataInd *dataInd) BT_LARGE
{
	Connections * BT_RAM con;
	BT_U16 BT_RAM i;

	/*
	 * Find the connection associated with the bdAddr
	 * Only one connection per bdAddr is allowed so we don't
	 * have to check the dlci
	 */
	if ( (con = GetConnectionBdAddr(dataInd->bdAddr)) != 0 )
	{
		/*
		 * Check if the length of the message > 0.
		 * If the length == 0 => message contains credits =>
		 * do nothing. (PADAPT takes care of the credits)
		 */
	   if ( dataInd->bufferLength > 0 )
    {
      if( nibbler_get_enable_status(&headset_nibbler) == NIBBLER_ENABLE )
      {  
        SendNibbleByte(con, dataInd->dataBufferPtr, dataInd->bufferLength);
      }
      else
      {
        for ( i = 0; i < dataInd->bufferLength; i++ )
        {
          ParseCharacter(con, dataInd->dataBufferPtr[i]);
        }
      }
    }

	}
	else
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Discarding PADAPT_DATA_IND since no valid connection"));
	}
}


/* ===========================================================================
 * Function:	SendPadaptConnectRsp
 * Description: Try to send a PADAPT_CONNECT_RSP, continue with service
 *				level connection setup if successful.
 * Returns: 	BT_OK, BT_CONGESTED or BT_CALL_ERROR
 * =========================================================================== */
static BT_Result SendPadaptConnectRsp(Connections *con, BT_BOOL accept) BT_LARGE
{
	BT_Result			BT_RAM res = BT_CONGESTED;
	PADAPT_ConnectRsp	BT_RAM rsp;


	rsp.header.sender		= HSP_AG_MODULE_ID;
	rsp.header.primitive	= PADAPT_CONNECT_RSP;
	rsp.header.receiver 	= BT_PADAPT;

	rsp.dlci				= con->hsDlci;
	rsp.accept				= accept;
	memcpy(rsp.bdAddr, con->hsBdAddr, sizeof(BT_BdAddr) );

	if ( !VerifyConnectionHandle(con) )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Invalid connection handle"));
		res = BT_CALL_ERROR;
	}
	/*
	 * If the sending the command was not delayed, continue with connect
	 * sequence
	 */
	else
	{
		res = HSP_AG_SEND_REQ((BT_Msg*)&rsp);
		
		/*
		 * If connection is terminated while waiting for the users 
		 * connect response. PADAPT will return BT_CALL_ERROR
		 */
		if ( res == BT_CALL_ERROR 
			&& con->connectionState == INCOMING_HS_W_4_CONNECT_RSP )
		{			
			FreeConnection(con);
		}
		else if ( res == BT_CONGESTED )
		{
			/*
			 * Return congested and try later
			 */
		}
		/*
		 * The message was sent
		 */
		else if ( accept )
		{
			ChangeConnectionState(con, PADAPT_CONNECTED_STATE);
			RunConnectionSetupStateMachine(con);
		}
		else
		{
			FreeConnection(con);
		}
	}

	return res;
}


/* ===========================================================================
 * Function:	SendPadaptConnectRspNegative
 * Description: 
 * Returns: 	
 * =========================================================================== */
static void SendPadaptConnectRspNegative(void)
{
	PADAPT_ConnectRsp	BT_RAM rsp;

	rsp.header.sender		= HSP_AG_MODULE_ID;
	rsp.header.primitive	= PADAPT_CONNECT_RSP;
	rsp.header.receiver 	= BT_PADAPT;

	rsp.accept				= BT_FALSE;
	memcpy(rsp.bdAddr, rejectConnectInfo.bdAddr, sizeof(BT_BdAddr) );
	rsp.dlci				= rejectConnectInfo.dlci;
	
	if ( HSP_AG_SEND_REQ((BT_Msg*)&rsp) == BT_CONGESTED )
	{
		rejectConnectInfo.timer= 
			BT_TimeoutReq(HSP_AG_MODULE_ID, 
						TIMER_ID_RESEND_REJECT_CONNECT,
						0, 
						HSP_AG_CFG_RESEND_TIME );
	}
}


/* ===========================================================================
 * Function:	HandlePadaptConnectInd
 * Description: Notify user of connection request or discard it
 * =========================================================================== */
static void HandlePadaptConnectInd(const PADAPT_ConnectInd *connectInd) BT_LARGE
{
	Connections * BT_RAM con;

	/*
	 * Only one connection per bdAddr is allowed
	 */
	if ( (con = GetConnectionBdAddr(connectInd->bdAddr)) != 0 )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Only one connection per bdAddr is allowed"));

		if ( rejectConnectInfo.timer == 0 )
		{
			LOG_INFO((HSP_AG_MODULE_ID, "Rejecting connection attempt, " ));

			memcpy( rejectConnectInfo.bdAddr, connectInd->bdAddr, 6 );
			rejectConnectInfo.dlci = connectInd->dlci;

			SendPadaptConnectRspNegative();
		}
		else
		{
			LOG_INFO((HSP_AG_MODULE_ID, "Discarding connection attempt, " ));
		}
	}

	/*
	 * Try to find a spare connection. Discard if everyone
	 * is allocated.
	 */
	else if ( (con = NewConnection(connectInd->bdAddr)) == 0 )
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Discarding connection attempt, "
						"no resources left"));
	}
	/*
	 * Start an service search attribute request before notifying the user
	 * and there fore complete the RFCOMM channel setup.
	 */
	else
	{
		con->hsDlci 	= connectInd->dlci;
		con->framesize	= connectInd->framesize;

		StartHandlingIncomingConnectionAttempt(con, connectInd->bdAddr);
	}
}


/* ===========================================================================
 * Function:  HandlePadaptDisconnectInd
 * Description: Update state and notify registered user
 * =========================================================================== */
static void HandlePadaptDisconnectInd(const PADAPT_DisconnectInd *disconnectInd) BT_LARGE
{
	Connections * BT_RAM con;

	/*
	 * Find the connection associated with the bdAddr
	 * Only one connection per bdAddr is allowed so we don't
	 * have to check the dlci
	 */
	if ( (con = GetConnectionBdAddr(disconnectInd->bdAddr)) != 0 )
	{
		ChangeConnectionState(con, HS_DISCONNECTED_STATE);

		/*
		 * The SCO might be in a "middle" state
		 * and will therefore never receive a 
		 * CONNECTION_COMPLETE
		 * Try to send a negative confirm. The userRequestInProgress will
		 * take care if the confirm is sent or not
		 */
		RestartedCfmUserCommand(con);

		switch ( con->userRequestInProgress )
		{
		case HSP_AG_AUDIO_DISCONNECT_REQ:
			con->audioConnectionHandle = INVALID_CONNECTION_HANDLE;
			break;
		}

		/*
		 * Cancel possibly running timers
		 */
		CancelTimers(con);

		/*
		 * Notify the user about the disconnection
		 */
		SendLinkStatusInd(con);
	}
	else
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Discarding PADAPT_DISCONNECT_IND since no valid connection"));
	}
}


/* ===========================================================================
 * Function:	HandlePadaptFlowInd
 * Description: Send the flow indication to the user
 * ==========================================================================*/
static void HandlePadaptFlowInd(const PADAPT_FlowInd *flowInd)
{
	HSP_AG_FlowInd	BT_RAM ind;
	Connections *	BT_RAM con;

	/*
	 * Find the connection associated with the bdAddr
	 * Only one connection per bdAddr is allowed so we don't
	 * have to check the dlci
	 */
	if ( (con = GetConnectionBdAddr(flowInd->bdAddr)) != 0 )
	{
		con->remoteReadyToReceive	= flowInd->remoteReadyToReceive;

		ind.remoteReadyToReceive	= con->remoteReadyToReceive;
		ind.handle					= con;
                memcpy(ind.bdAddr, con->hsBdAddr,6);
		SEND_TO_USER(&ind, HSP_AG_FLOW_IND );
	}
	else
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Discarding PADAPT_FLOW_IND since no valid connection"));
	}
}

/* ===========================================================================
 * Function:	HandlePadaptConfigInd
 * Description:
 * ==========================================================================*/
static void HandlePadaptConfigInd(const PADAPT_ConfigInd *configInd)
{
	HSP_AG_ConfigInd	BT_RAM ind;
	Connections 	*	BT_RAM con;

	/*
	 * Find the connection associated with the bdAddr
	 * Only one connection per bdAddr is allowed so we don't
	 * have to check the dlci
	 */
	if ( (con = GetConnectionBdAddr(configInd->bdAddr)) != 0 )
	{
		ind.framesize	= configInd->framesize;
		ind.handle		= con;

		SEND_TO_USER(&ind, HSP_AG_CONFIG_IND );
	}
	else
	{
		LOG_INFO((HSP_AG_MODULE_ID, "Discarding PADAPT_CONFIG_IND since no valid connection"));
	}
}

/* ===========================================================================
 * Function: 	HandlePadaptNotCongestedInd
 * Description: Handles PADAPT_NOT_CONGESTED_IND
 * Returns:		-
 * =========================================================================== */
static void HandlePadaptNotCongestedInd(void)
{
	HSP_AG_NotCongestedInd 	BT_RAM ind;

	/* Only send notification if there has been a data congestion */
	if ( returnedDataCongestion )
	{
		returnedDataCongestion = BT_FALSE;

		SEND_TO_USER(&ind, HSP_AG_NOT_CONGESTED_IND);
	}
}



/* ===========================================================================
 * Function:	HandlePadaptMsg
 * Description: Handle all messages received from PADAPT.
 * =========================================================================== */
static void HandlePadaptMsg(const PADAPT_IndConf *m) BT_LARGE
{
	switch ( m->header.primitive )
	{
	case PADAPT_CONNECT_CFM:
		HandlePadaptConnectCfm(&m->connectCfm);
		break;

	case PADAPT_CONFIG_IND:
		HandlePadaptConfigInd(&m->configInd );
		break;

	case PADAPT_CONNECT_IND:
		HandlePadaptConnectInd(&m->connectInd );
		break;

	case PADAPT_DISCONNECT_IND:
		HandlePadaptDisconnectInd(&m->disconnectInd);
		break;

	case PADAPT_DATA_IND:
		HandlePadaptDataInd(&m->dataInd);
		break;

	case PADAPT_FLOW_IND:
		HandlePadaptFlowInd(&m->flowInd);
		break;

	case PADAPT_DM_SENT_IND:
		if ( HSP_AG_CFG_DISCONNECT_ACL_AFTER_DM_TIME )
		{
			memcpy( disconnectAclBdaddr, m->dmSentInd.bdAddr, 6 );
			
			(void)BT_TimeoutReq(HSP_AG_MODULE_ID, TIMER_ID_DM_DISCONNECT_ACL,
					0, HSP_AG_CFG_DISCONNECT_ACL_AFTER_DM_TIME );
		}
		break;
		
	case PADAPT_NOT_CONGESTED_IND:
		HandlePadaptNotCongestedInd();
		break;

	default:
		LOG_INFO((HSP_AG_MODULE_ID, "PADAPT primitive 0x%02x does not exist", m->header.primitive));
		break;
	}
}


/* ===========================================================================
 * Function:	HandleTsrvMsg
 * Description: Handle messages from Time Server.
 * =========================================================================== */
static void HandleTsrvMsg(const BT_TimeoutInd *m) BT_LARGE
{
	/*
	 * m->userTimerData might be zero (m->userTimerId equal to 
	 * TIMER_ID_RESTART_RESPONSE) but then is not con used.
	 */
	Connections *	BT_RAM con = (Connections *)m->userTimerData;

	/*
	 * Dispatch to the appropriate handler functionm
	 */
	switch ( m->userTimerId )
	{
	case TIMER_ID_RUN_CONNECTION_STATE_MACHINE:
		con->connectTimer = 0;
		RunConnectionSetupStateMachine(con);
		break;

	case TIMER_ID_HS_DISCONNECTED_STATE_IND:
		con->connectTimer = 0;
		SendLinkStatusInd(con);
		break;

	case TIMER_ID_RESTART_RESPONSE:
		restartedRspTimer = 0;
		SendRestartResponse();
		break;

	case TIMER_ID_SCO_CONNECT_RESPONSE:
		con->audioConnectRspTimer = 0;
		SendHciLhScoConnectRsp(con);
		break;

	case TIMER_ID_PADAPT_CONNECT_REQ:
		con->connectTimer = 0;
		SendPadaptConnectReq(con);
		break;

	case TIMER_ID_SDAP_SEARCH:
		con->sdapTimer = 0;
		SendSdapServiceSearchAttrReq(con);
		break;

	case TIMERID_RESEND_LINK_SETTINGS:
		con->connectTimer = 0;
		SendLinkSettingsReq(con);
		break;

	case TIMER_ID_DM_DISCONNECT_ACL:
		SendAclDisconnect( disconnectAclBdaddr );
		break;
		
	case TIMER_ID_SEND_LINK_STATUS_IND:
		con->sdapTimer = 0;
		SendLinkStatusInd(con);
		break;

	case TIMER_ID_RESEND_REJECT_CONNECT:
		rejectConnectInfo.timer = 0;
		SendPadaptConnectRspNegative();
		break;
	}
}


/* ===========================================================================
 * Function:	HandleLinkHandlerMsg
 * Description:
 * =========================================================================== */
static void HandleLinkHandlerMsg(const HCI_LH_IndConf *ind)
{
	Connections * BT_RAM con;

	switch (ind->header.primitive )
	{
	case HCI_LH_LINK_SETTINGS_CFM:
		if ( (con = GetConnectionConHnd(HCI_LINK_TYPE_ACL_CONNECTION, ind->linkSettingsCfm.connectionHandle)) != 0 )
		{
			if ( !ind->linkSettingsCfm.success )
			{
				LOG_INFO((HSP_AG_MODULE_ID, "Failed to configure link handling"));
			}

			ChangeConnectionState(con, HS_ACL_CONNECTED_STATE );
		}
		break;

	case HCI_LH_AUDIO_CONNECT_CFM:
		/*
		 * Audio Connect
		 */
		if ( (con = GetConnectionUserRequest(HSP_AG_AUDIO_CONNECT_REQ)) != 0 )
		{
			SendAudioConnectCfm(con, (BT_BOOL)(ind->audioConnectCfm.status == HCI_OK));
		}
		break;
	}
}


/* ===========================================================================
 * Function:	Initialise
 * Description: Handle initialise startup phase
 * =========================================================================== */
static void Initialise(void) BT_LARGE
{
	BT_U16 BT_RAM fatalError = 0;
	BT_U8  BT_RAM n = 0;

	if ( (fatalError = HSP_AG_OptInitialise()) != 0 )
	{
		/* Just quit */
	}
	else if ( BT_LookupProfile(BT_HCI_LH) == 0 )
	{
		fatalError = HSP_AG_ERR_HCI_LH_MISSING;
	}
	else if ( (connections = (Connections*)
				BT_MALLOC(
					HSP_AG_MODULE_ID,
					HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS*sizeof(Connections))) == 0	)
	{
		fatalError = HSP_AG_ERR_MAX_NO_OF_HS_CONNECTIONS;
	}
	else
	{
		/*
		 * Se comments at top of file about timer variables for an
		 * explanation on number of timers needed
		 *
		 * 3 per connection + rejectConnectTimer
		 */
		BT_AllocateTimers(HSP_AG_MODULE_ID, 1 + 3*HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS);
		rejectConnectInfo.timer = 0;
		
		/*
		 * Allocate a command buffer and init each connection
		 */
		while ( 	n < HSP_AG_CFG_MAX_NO_OF_HS_CONNECTIONS
			   &&	!fatalError )
		{
			if ( (connections[n].atCommandBuffer = (BT_U8*)
						BT_MALLOC(
							HSP_AG_MODULE_ID,
							HSP_AG_CFG_AT_COMMAND_MAX_SIZE*sizeof(BT_U8))) == 0  )
			{
				fatalError = HSP_AG_ERR_AT_COMMAND_MAX_SIZE;
			}

			FreeConnection(&connections[n]);

			n++;
		}
	}

	/*
	 * If an error occured, report it to controlling app
	 */
	if ( fatalError )
	{
		BT_FatalError(fatalError);
	}
}
/* ===========================================================================
 * Function:    ConvertPacketTypesFrom11To12
 * Description: Converts packet type encoding from BT 1.1 to BT 1.2.
 * =========================================================================== */
static void ConvertPacketTypesFrom11To12( HCI_ScoPacketType packetType11,
										HCI_SynchronousPacketType *packetType12 )
{
	*packetType12 = 0;

	if ( packetType11 & HCI_SCO_HV1 )
	{
		*packetType12 |= HCI_SYNCHRONOUS_HV1;
	}

	if ( packetType11 & HCI_SCO_HV2 )
	{
		*packetType12 |= HCI_SYNCHRONOUS_HV2;
	}

	if ( packetType11 & HCI_SCO_HV3 )
	{
		*packetType12 |= HCI_SYNCHRONOUS_HV3;
	}
}




