/*################################################################################
#
# 						FILE SPECIFICATION
# 					COPYRIGHT 2011,2013-2015 MOTOROLA SOLUTIONS,INC. ALL RIGHTS RESERVED.
#						MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: initialization.c
#
# --------------------------- General Description -----------------------------
# This file contains the function definitions for the Initialization of the 
# Dongle code.
#
#
*--------------------------- End of History Template-----------------------------
# --------------------------- HEADER FILE INCLUDES ----------------------------*/
/* Environment header files. */
#include "pm.h"

/* Scheduler header files. */
#include "FreeRTOS.h"

#include "wdt.h"
#include "initialization.h"
#include "cycle_counter.h"
#include "gpio.h"
#include "controls.h"
#include "adc.h"
#include "TaskConfig.h"
#include "codeplug_ui.h"

#ifdef RADIO_INTERFACE_MANAGER_TASK
#include "radio_interface.h"
#endif

#ifdef MPP_TASK
#include "mpp_init.h"
#endif

/* Device Specific Includes*/
#include "config.h"
/********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* * AUTHOR            Date Modified	  CR Tracking Number 	    Description
* Aneeket Patkar      02/03/2010          CCMPD01310299_Atmel_Init  Initial Creation
* Abhishek Trivedi    04/20/2010          CCMPD01331626             Move the Speaker & Mic path functions to controls.c
* Aneeket Patkar      05/25/2010          CCMPD01347254             Porting to Secure-Headset Code
* YewFoong            06/25/2010          CCMPD                     Initial draft for non-secure wireless dongle R1.2
* Sevuhan Ramanathan  07/05/2010                                    Modified configure_gpio function for reuse in R1.2 wireless adaptor code
* Tan Seng Kai	      09/14/2010          CCMPD01392415             Change init state for bluetooth link status to High
* Sevuhan Ramanathan  09/30/2010          CCMPD01397192             set init value for LED pin PA28 to low 
* Tan Seng Kai        10/06/2010          CCMPD01399559             RADIO_INTERFACE_MANAGER_TASK,MPP_TASK,NIBBLER_TASK Compile sw added 
*                                                                   to remove Radio interface and mpp task
* Toh Pei See         10/15/2010          CCMPD01393923             Added the codeplug initialization under initialize_dongle_features function 
* WRN637	 		        10/20/2010   	      CCMPD01394399		          Added in init for CSR chip
* YK - JWPM67          Nov/15/2010      CCMPD01413200               Set PA08 (Audio Route Switch) as GPIO
* JJ Low              11/16/2010          CCMPD01412613            Removed redundant I/O and moved MPP I/O configuration to mpp_init.c
* YK - JWPM67         Dec/2/2010          CCMPD01413201            Re-wrote the comment for GPIO PB08.
* JJ Low ckpd36       12/08/2010         CCMPD01440674              To enable/disable USB module
* YewFoong            12/20/2010          CCMPD01455955             Disable watchdog when program is started
* Mahes               Jan/06/2011         CCMPD01460340             Software catch-up to dongle R1.2 version R01.00.00
* Sevuhan Ramanathan  04/20/2011          CCMPD01498417             Modified R2.1 code to enable USB mode always in R2.19
* Sevuhan Ramanathan  12/29/2011          CCMPD01600891             R2.17 pin mapping and initial state setting
* Sevuhan Ramanathan  12/21/2011          CCMPD01594785               remove usb init by default, programmable buttons initialization
* WRN637              9/11/2013           CCMPD01808031             REMOVE USB HID Support and initial mapping for P0A board
* WRN637              09/17/2013          CCMPD01815073             Add Button handling for R2.17 Wireless Gateway P0B board
* Abhishek Trivedi    09/02/2013          CCMPD01827271             XNL XCMP Host Interface
* Abhishek Trivedi    03/10/2014	  CCMPD01870625		    CSR DSP Audio Detect Integration Changes
* Aneeket Patkar      03/26/2014          CCMPD01876782             Initialization Sequence update - Create MPP State Machine after BT initialization
* Abhishek Trivedi    04/04/2014          CCMPD01880139             Reset on DSP Command failure
*--------------------------- End of History Template-----------------------------
* *********************************************************************************/


//------------------------------------------------------------------------------
//      Defines
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//      GLOBAL VARIABLES
//------------------------------------------------------------------------------
volatile avr32_pm_t* pm = &AVR32_PM;
volatile avr32_adc_t *adc = &AVR32_ADC;

/* 0: CDC USB 
   1: HID USB with periodic message 
   2: HID USB without periodic message */
int usb_radio_detected_glb = 0; 
bool gw_usb_hid_disabled_glb = 0; /* 0: USB HID; 1: not USB HID */

static const gpio_map_t DONGLE_ATMEL_IO_FUNCTION_MAP =
{
  {PA03, PA03_GPIO_FUNCTION},     /* NC                   (Not using)                     */
  {PA04, PA04_GPIO_FUNCTION},     /* IN,  ONE_DOT_DETECT_GPIO_PIN                         */
  {PA05, PA05_GPIO_FUNCTION},     /* IN   TWO_DOT_DETECT_GPIO_PIN                         */ 
  {PA06, PA06_GPIO_FUNCTION},     /* IN,  PTT_MFB_GPIO_PIN                                */ 
  {PA07, PA07_GPIO_FUNCTION},     /* IN,  AUDIO_DETECT_GPIO_PIN                           */ 
  {PA08, PA08_GPIO_FUNCTION},     /* IN,  PHYSICAL_HOOK_DETECT_GPIO_PIN        (Not using)*/ 
  {PA09, PA09_GPIO_FUNCTION},     /* IN,  I2C_SCL         (Not using)                     */ 
  {PA10, PA10_GPIO_FUNCTION},     /* IN,  I2C_SDA         (Not using)                     */ 
  {PA11, PA11_GPIO_FUNCTION},     /* NC                   (Not using)                     */ 
                                  /* PA12 - NFC_WAKE_INT                                  */
  {PA13, PA13_GPIO_FUNCTION},     /* IN,  SW_TEST_UP      (Not using)                     */
  {PA14, PA14_GPIO_FUNCTION},     /* PA14 - SPI_MOSI                                      */
  {PA15, PA15_GPIO_FUNCTION},     /* OUT, ATMEL_CSR_UPGRADE_SELECT_GPIO_PIN               */ 
  {PA16, PA16_GPIO_FUNCTION},     /* PA16 - SPI_CS0                                       */
  {PA17, PA17_GPIO_FUNCTION},     /* PA17 - SPI_CLK                                       */
                                  /* PA18 - XIN16M                                        */
                                  /* PA19 - XOUT16M                                       */
  {PA20, PA20_GPIO_FUNCTION},     /* OUT, RADIO_MMP_PTT_GPIO_PIN                          */ 
                                  /* PA21 - NFC_TX_CLK                                    */
  {PA22, PA22_GPIO_FUNCTION},     /* OUT, UP_RST_BT_GPIO_PIN                              */ 
  {PA23, PA23_GPIO_FUNCTION},     /* OUT, RADIO_MMP_ONE_DOT_GPIO_PIN                      */ 
  {PA24, PA24_GPIO_FUNCTION},     /* OUT, RADIO_MMP_TWO_DOT_GPIO_PIN                      */ 
  {PA25, PA25_GPIO_FUNCTION},     /* PA25 - SPI_MISO                                      */
                                  /* PA26 - UP_BT_UART2                                   */
                                  /* PA27 - BT_UP_UART2                                   */
  {PA28, PA28_GPIO_FUNCTION},     /* OUT, LED_PIN                                         */ 
  {PA29, PA29_GPIO_FUNCTION},     /* IN,  DEBUG_GPIO_PIN           (Not using)            */ 
  {PA30, PA30_GPIO_FUNCTION},     /* NC                   (Not using)                     */ 
  {PA31, PA31_GPIO_FUNCTION},     /* IN,  UPGRADE_DETECT_GPIO_PIN2     (Not using)        */ 
  {PB00, PB00_GPIO_FUNCTION},     /* PB00 - BT_UART2_RTS                                  */
  {PB01, PB01_GPIO_FUNCTION},     /* PB01 - UP_UART2_RTS                                  */
  {PB02, PB02_GPIO_FUNCTION},     /* IN,  BOARD_FAIL_DETECT_PIN     (Not_using)           */ 
  {PB03, PB03_GPIO_FUNCTION},     /* IN,  BOARD_ID_DETECT_PIN        (Not_using)          */ 
  {PB04, PB04_GPIO_FUNCTION},     /* OUT, MIC_SELECT_GPIO_PIN                             */ 
  {PB05, PB05_GPIO_FUNCTION},     /* IN,  USB_SELECT_GPIO_PIN     (Not using)             */ 
  {PB06, PB06_GPIO_FUNCTION},     /* IN,  CSR_DSP_AUDIO_DETECT        I2C_REQ(schematic)  */ 
  {PB07, PB07_GPIO_FUNCTION},     /* OUT, USB_PIO_VBUS    (Not using)                     */ 
  {PB08, PB08_GPIO_FUNCTION},     /* OUT, RADIO_MMP_VIRTUAL_HOOK_GPIO_PIN                 */ 
  {PB09, PB09_GPIO_FUNCTION},     /* IN,  UPGRADE_DETECT_GPIO_PIN     (Not using)         */ 

};




//------------------------------------------------------------------------------
//      External Variables
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//      External Functions
//------------------------------------------------------------------------------

extern void wdt_clear(void);
extern void pm_switch_to_osc0(volatile avr32_pm_t *pm, unsigned int fosc0, unsigned int startup);
extern void pm_cksel(volatile avr32_pm_t *pm, unsigned int pbadiv, unsigned int pbasel,unsigned int pbbdiv,
                     unsigned int pbbsel,unsigned int hsbdiv,unsigned int hsbsel);
extern void xnl_init_component(void);
//------------------------------------------------------------------------------
//      Internal Functions' Prototype Declaration
//------------------------------------------------------------------------------
void initialize_system(void);
void configure_atmel_clock(void);
void configure_GPIO(void);
void configure_atmel_usb_clock(void);
void watchdog_enable(int watchdog_period);
void watchdog_disable(void);
void watchdog_reset(void);
int get_reset_cause(void);
void usb_cdc_init(void);
void debug_manager_init(void);
void test_commands_manager_init(void);
void bluetooth_manager_init(void);
void unconnected_pin_handler(void);
void usb_hid_detection(void);
//------------------------------------------------------------------------------
//      Internal Functions' Definition Begins Here
//------------------------------------------------------------------------------

/*=============================================================================
	FUNCTION: initialize_system

	DESCRIPTION: This function will be called from "main". The task is to 
                    initialize all components.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void initialize_system(void)
{  
  /* Disable Watchdog */
  wdt_disable();
  
  /* Configure the Atmel clock. */
  configure_atmel_clock();
  
  /* Configure the GPIO */
  configure_GPIO();
  
  /* Configure USB clock */
  configure_atmel_usb_clock();
  
  /* Configure ADC */
  AVR32_ADC.mr |= 0x1 << AVR32_ADC_MR_PRESCAL_OFFSET;
  adc_configure(adc);
 
}
/* End of initialize_system()   */
/*=============================================================================
	FUNCTION: initialize_dongle_features

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void initialize_dongle_features(void)
{   
    /* Initialize the codeplug */
    codeplugInit();
    
    /* Check and configure USB HID feature */
    usb_hid_detection();
    
    /* Initialize debug manager */
    debug_manager_init();

      /* Initialize USB CDC communication */
    usb_cdc_init();
    /* Initialize test commands manager */
    test_commands_manager_init();
  
    /* Initialize the XNL-XCMP functionality */
    xnl_init_component();    
    
    if (usb_radio_detected_glb == 0) // else: controls_init() after USB has been enumerated
      controls_init();
}
/* End of initialize_dongle_features()   */

/*=============================================================================
	FUNCTION: configure_atmel_clock

	DESCRIPTION: This function will configure UC3B clock.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void configure_atmel_clock(void)
{
  // Set System clock source to the 16 MHz crystal connected to Osc0.
  pm_switch_to_osc0(&AVR32_PM, DONGLE_FOSC0_CLK, OSC0_STARTUP); 
  
  // Configure the Main clock, HSB, PBB and PBA to 4MHz by programming the CKSEL register in Power Manager.
  pm_cksel(&AVR32_PM, PBA_DIV, PBA_SEL,PBB_DIV, PBB_SEL, CLK_HSB_DIV, CLK_HSB_SEL);  
  
}
/* End of configure_atmel_clock()   */
/*=============================================================================
	FUNCTION: configure_atmel_usb_clock

	DESCRIPTION: This function will configure UC3B USB PLL clock.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void configure_atmel_usb_clock(void)
{
  // Configure the PLL1 to run at 96 MHz from the 16MHz cyrstal connected at Osc0
  pm_pll_setup(&AVR32_PM,
                DONGLE_USB_PLL_SELECT,       // PLL 1 Selected
                DONGLE_USB_PLL_PLLMUL,       // Multiply by 6
                DONGLE_USB_PLL_PLLDIV,       // Divide by 1
                DONGLE_USB_PLL_PLLOSC,       // Osc0 selected as source for PLL 
                DONGLE_USB_PLL_PLLCOUNT);    // 16 slow clock cycles
  
  // Set the PLL Option fields
  pm_pll_set_option(&AVR32_PM,
                     DONGLE_USB_PLL_SELECT,        // PLL 1 Selected
                     DONGLE_USB_PLL_PLLOPT_BIT0,   // VCO freq : 80MHz<fvco<180MHz
                     DONGLE_USB_PLL_PLLOPT_BIT1,   // fpll = fvco/2 = 48MHz 
                     DONGLE_USB_PLL_PLLOPT_BIT2    // Wide BW mode disabled
                    );
  
  // Start PLL1. wait for lock
  pm_pll_enable(&AVR32_PM,DONGLE_USB_PLL_SELECT);
  
  // Wait for PLL1 locked.
  pm_wait_for_pll1_locked(&AVR32_PM);
  
  // Setup the Generic Clock Control reg, GCCTRL, for the PLL
  pm_gc_setup(&AVR32_PM,
              AVR32_PM_GCLK_USBB,             // 
              DONGLE_USB_PLL_GCCTRL_OSCSEL,  // Select PLL1 as source
              DONGLE_USB_PLL_GCCTRL_PLLSEL,  // Select PLL1 as source
              DONGLE_USB_PLL_GCCTRL_DIVEN,   // Disable Divide
              DONGLE_USB_PLL_GCCTRL_DIV      // Divide =0
              );
  
  // Enable the Generic Clock
  pm_gc_enable(&AVR32_PM, AVR32_PM_GCLK_USBB);  
   
}
/*=============================================================================
	FUNCTION: configure_GPIO

	DESCRIPTION: Configure GPIO pins. Enable pull-ups for necessary pins.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void configure_GPIO(void)
{
  gpio_enable_module( DONGLE_ATMEL_IO_FUNCTION_MAP, sizeof(DONGLE_ATMEL_IO_FUNCTION_MAP)/sizeof( DONGLE_ATMEL_IO_FUNCTION_MAP[0]));/*enabling the gpio pin map*/ 


  /* Unused GPIO - Set to Output Low */
  gpio_clr_gpio_pin(PA03); /* NC */
  gpio_clr_gpio_pin(PA11); /* NC */
  gpio_clr_gpio_pin(PA30); /* NC */
  gpio_clr_gpio_pin(DEBUG_GPIO_PIN); /* NC */

  /* Unused Interchip - set to input */
  gpio_configure_pin(PA13,GPIO_DIR_INPUT); /* SW_TEST_UP*/
  gpio_configure_pin(PA09,GPIO_DIR_INPUT); /* I2C_SCL */
  gpio_configure_pin(PA10,GPIO_DIR_INPUT); /* I2C_SDA */
  gpio_configure_pin(PB06,GPIO_DIR_INPUT); /* CSR_DSP_AUDIO_DET  (I2C_REQ) */
  gpio_configure_pin(PB07,GPIO_DIR_INPUT); /* USB_PIO_VBUS */
  gpio_enable_pin_pull_up(PB06);
  
  /* Reserved GPIO function for future expension - set to input */
  gpio_configure_pin(PHYSICAL_HOOK_DETECT_GPIO_PIN,GPIO_DIR_INPUT); /* HOOK_DET*/
  gpio_configure_pin(BOARD_FAIL_DETECT_PIN,GPIO_DIR_INPUT);         /* DONGLE_FAIL*/
  gpio_configure_pin(BOARD_ID_DETECT_PIN,GPIO_DIR_INPUT);           /* BOARDID0*/
  gpio_configure_pin(USB_SELECT_GPIO_PIN,GPIO_DIR_INPUT);           /* UPGRADE_SEL*/
  gpio_configure_pin(UPGRADE_DETECT_GPIO_PIN,GPIO_DIR_INPUT);       /* UPGRADE_DET*/
  gpio_configure_pin(UPGRADE_DETECT_GPIO_PIN2,GPIO_DIR_INPUT);      /* UPGRADE_DET2*/

  /* Initialize Input Pin */
  gpio_configure_pin(PTT_MFB_GPIO_PIN,GPIO_DIR_INPUT);              /* DONGLE_PTT */
  gpio_configure_pin(AUDIO_DETECT_GPIO_PIN,GPIO_DIR_INPUT);         /* AUDIO_DETECT */
  gpio_configure_pin(ONE_DOT_DETECT_GPIO_PIN,GPIO_DIR_INPUT);       /* ONEDOT DETECT */
  gpio_configure_pin(TWO_DOT_DETECT_GPIO_PIN,GPIO_DIR_INPUT);       /* TWODOT DETECT */
  gpio_enable_pin_pull_up(AUDIO_DETECT_GPIO_PIN);
  
  /* Initialize Output Pin */
  gpio_clr_gpio_pin(ATMEL_CSR_UPGRADE_SELECT_GPIO_PIN);             /* USB_SWITCH_CTL */
  gpio_clr_gpio_pin(RADIO_MMP_VIRTUAL_HOOK_GPIO_PIN);               /* VIRTUAL_HOOK */
  gpio_clr_gpio_pin(LED_PIN);                                       /* LED_OFF*/ 
  gpio_set_gpio_pin(RADIO_MMP_PTT_GPIO_PIN);                        /* RADIO_PTT */                              
  gpio_set_gpio_pin(UP_RST_BT_GPIO_PIN);                            /* UP_RST_BT */
  gpio_set_gpio_pin(MIC_SELECT_GPIO_PIN);                           /* MICSELECT */
  gpio_set_gpio_pin(RADIO_MMP_ONE_DOT_GPIO_PIN);                    /* ONEDOT */
  gpio_set_gpio_pin(RADIO_MMP_TWO_DOT_GPIO_PIN);                    /* TWODOT */

  /* Unused SPI MOSI, CS0, MISO - Set to input Pullup */
  gpio_configure_pin(PA14,GPIO_DIR_INPUT);
  gpio_configure_pin(PA16,GPIO_DIR_INPUT);
  gpio_configure_pin(PA25,GPIO_DIR_INPUT);
  gpio_enable_pin_pull_up(PA14);
  gpio_enable_pin_pull_up(PA16);      
  gpio_enable_pin_pull_up(PA25);

  /* Unused SPI CLK - Set to output Low */
  gpio_clr_gpio_pin(PA17);
  
  /* Unused UART2 - Set to input Pullup */
  gpio_configure_pin(PB00,GPIO_DIR_INPUT);
  gpio_configure_pin(PB01,GPIO_DIR_INPUT);
  gpio_enable_pin_pull_up(PB00);      
  gpio_enable_pin_pull_up(PB01);
  
}
/* End of configure_GPIO*/
/*=============================================================================
	FUNCTION: watchdog_enable

	DESCRIPTION: This function will enable Watchdog Timer.

	ARGUMENTS PASSED: time ( in milliseconds )

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void watchdog_enable(int watchdog_period)
{ 
  // Enable the Watchdog. Convert the timer value to microsecsonds  as per 
  // Atmel function implementation.
  // Save current value in GPLP register 
  pm_write_gplp(pm,0,1000000);
  wdt_enable((unsigned long) (watchdog_period*1000));
  
}
/* End of watchdog_enable()   */

/*=============================================================================
	FUNCTION: watchdog_disable

	DESCRIPTION: This function will disable Watchdog Timer.

	ARGUMENTS PASSED: none.

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void watchdog_disable(void)
{
  // Atmel  implemented function. 
  //This function will disable the Watchdog Timer.
  wdt_disable();
}
/* End of watchdog_disable()   */

/*=============================================================================
	FUNCTION: watchdog_reset

	DESCRIPTION: This function will reset Watchdog Timer. This function needs
                     to be called before the Wathdog Timer period to prevent the
                     Watchdog from resetting the device.

	ARGUMENTS PASSED: none.

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void watchdog_reset(void)
{
  // Atmel  implemented function. 
  //This function will reset/clear/restart the Watchdog Timer.
  wdt_clear();
}
/* End of watchdog_reset()   */

/*=============================================================================
	FUNCTION: get_reset_cause

	DESCRIPTION: This function is a wrapper to provide the reason for reset.
                     For UC3B, the reset status can be directly accessed from 
                     the RCAUSE register.
	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: int. Reason for RESET.
==============================================================================*/
int get_reset_cause(void)
{
  unsigned int reset_status =0x00;
  int reset_cause = 0;
  reset_status = pm->rcause;
  // check is reset caused by Watchdog
   if ( reset_status & AVR32_PM_RCAUSE_WDT_MASK)
   {
     reset_cause = WATCHDOG_RESET; 
   }
   // check if reset is caused by Brown Out Reset
   else if ( reset_status & AVR32_PM_RCAUSE_BOD_MASK)
   {
     reset_cause = BROWNOUT_RESET; 
   }
   // check if reset is caused by External Reset
   else if ( reset_status & AVR32_PM_RCAUSE_EXT_MASK )
   {
     reset_cause = EXTERNAL_RESET; 
   }
   // check if reset is caused by Power On Reset
   else if ( reset_status & AVR32_PM_RCAUSE_POR_MASK)
   {
     reset_cause = POWERON_RESET; 
   }
   else if (reset_status & AVR32_PM_RCAUSE_SLEEP_MASK)
   {
     reset_cause = SLEEP_RESET;
   }
   else if (reset_status & AVR32_PM_RCAUSE_CPUERR_MASK)
   {
     reset_cause = CPU_ERR_RESET;
   }   
   
   return (reset_cause);
}
/* End of get_reset_cause()   */

/*=============================================================================
	FUNCTION: usb_hid_detection

	DESCRIPTION: Check and configure USB HID feature.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void usb_hid_detection(void)
{
  cplg_param_t temp_cp_value;
  
  //Get the Codeplug value for USB HID feature
  codeplugGetParameter(DISABLE_GW_USB_HID, &temp_cp_value, 0);
  gw_usb_hid_disabled_glb = (bool)temp_cp_value;
  
  if (!gw_usb_hid_disabled_glb && !gpio_get_pin_value(PB09)) // USB HID radio connected
  {
    gpio_configure_pin(PB05, GPIO_DIR_OUTPUT | GPIO_INIT_HIGH); // USB/GPIO -> USB
    gpio_set_gpio_pin(PB05);
    usb_radio_detected_glb = 1;
  }
  else if (gw_usb_hid_disabled_glb && !gpio_get_pin_value(PB09)) // non-USB HID radio connected
  {
    gpio_configure_pin(PB05, GPIO_DIR_OUTPUT); // USB/GPIO -> GPIO
    gpio_clr_gpio_pin(PB05);
    usb_radio_detected_glb = 0;
  }
}
/* End of usb_hid_detection() */
