/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: mpp_init.c
#
# --------------------------- General Description -----------------------------
# This file contains the MPP module initialization and interface functions
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/
#include "mpp_init.h"
#include "mpp_statemachine.h"
#include "mpp_msg.h"
#include "ams3930_driver.h"
#include "spi.h"
/********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR            Date Modified Tracking Number       Description
* Jaspreet Bajwa    01/08/2010    CCMPD                 Initial Creation
* Jaspreet Bajwa    01/21/2010    CCMPD                 UART SPI initialization routines and setting 
* JJ Low            11/05/2010    CCMPD01410054         Compiler warning resolution for R2.1
* JJ Low            11/16/2010    CCMPD01412613         R2.1 MPP state machine
* YK - JWPM67       Dec/1/2010    CCMPD01413200         Moved the MPP Wakeup ISR to "controls.c"
* Mahes             Jan/10/2011   CCMPD01460340         To solve linker issue: Undefined external "send_inter_task_msg" by
*                                                       adding compiler switch for duplicate function.[send_inter_task_msg]  
* Mahesh            07-Dec-2011   CCMPD01460638         Added  initial MPP inter task messages
* WRN637            11/13/2013    CCMPD01835515         NFC oscillator trim 
*--------------------------- End of History Template----------------------------
* *****************************************************************************/

/*==============================================================================
                                      EXTERNAL DEFINITIONS
==============================================================================*/
extern UNSIGNED_16BIT ams_reg_settings [];
extern UNSIGNED_8BIT host_cpb_rep_msg_txed;


#if defined(MPP_SECURE_DONGLE) //CCMPD01413200
extern void gpio_group2_irq1_controls_isr(void);
#endif

/*==============================================================================
                                    GLOBAL VARIABLES
==============================================================================*/
/* pairing timer handel*/
SIGNED_8BIT                mpp_time_to_pair;
static spi_mode present_spi_mode = 0xFF;
#ifdef MPP_DEVICE
device_reject_mpp             device_pairing_reject;
init_device_mpp_pkt           device_pkt;
init_device_mpp_pkt           *ptr2init_device_mpp_pkt = NULL;
device_succ_mpp               device_succ_msg;
#endif

#ifdef MPP_HOST
host_succ_mpp                   host_paired_succ;
connection_database             host_connection_db;
init_host_mpp_pkt               host_pkt;
host_mpp_config                 run_time_config;
mpp_beacon                      beacon_brdcast;
host_reject_mpp                 host_pairing_reject;
UNSIGNED_8BIT               rfcomm_taken[NUM_MAX_MPP_RFCOMM_CH];
init_host_mpp_pkt               *ptr2init_host_mpp_pkt = NULL;
connection_database             *ptr2connection_database = NULL;
#endif


UNSIGNED_8BIT mpp_debug_enable =1;
/*==============================================================================
                                     FUNCTION PROTOTYPES
==============================================================================*/
UNSIGNED_8BIT get_inter_task_msg(UNSIGNED_8BIT *msgid, void **ptr2message, UNSIGNED_16BIT *message_size, UNSIGNED_32BIT_LONG delay );  //CCMPD01460340(GM_Jan/10/2011)
void mpp_hw_tx_clk_init();


/************************************* SPI Support Functionality *******************************************************/

// Select this clock for EVK1101 board
#ifdef MPP_EVK1101
UNSIGNED_32BIT_LONG cpu_clk[MPP_SPI_MODES] = {
12000000,
12000000,
6000000
};
#endif

#ifdef MPP_SECURE_DONGLE
UNSIGNED_32BIT_LONG cpu_clk[MPP_SPI_MODES] = {
MPP_PBA_CLOCK_HZ,
MPP_PBA_CLOCK_HZ,
MPP_PBA_CLOCK_HZ
};
#endif

#ifdef MPP_OPTION_BOARD
UNSIGNED_32BIT_LONG cpu_clk[MPP_SPI_MODES] = {
48000000,
48000000,
6000000
};
#endif

//Select this clock for P0
#ifdef MPP_P0_BOARD
UNSIGNED_32BIT_LONG cpu_clk[MPP_SPI_MODES] = {
MPP_PBA_CLOCK_HZ,
MPP_PBA_CLOCK_HZ,
MPP_PBA_CLOCK_HZ
};
#endif

UNSIGNED_8BIT mode[MPP_SPI_MODES] = {
6,
6,
0
};
/*==============================================================================
                                      EXTERNAL FUNCTIONS
==============================================================================*/

/*==============================================================================
                                      FUNCTION PROTOTYPES
==============================================================================*/
static SIGNED_32BIT_LONG debug_port_isr( void );
static SIGNED_32BIT_LONG mpp_serial_port_isr( void );
static void debug_uart_isr( void );
static void mpp_uart_isr( void );
void mpp_debug_comm_create();
void mpp_hw_comm_create();
void mpp_spi_init(avr32_spi_t  *spi, spi_mode spi_mode);
void mpp_sw_init(void);
void mpp_serialport_init();
void mpp_wakeup_setup();
void mpp_hw_peripherals_init(void);
void mpp_gpio_init(void);

/*==============================================================================
                                      FUNCTION DEFINITIONS
==============================================================================*/
#if defined(MPP_OPTION_BOARD)
/*new functions for OB need cleanup*/
void ob_lf_chip_damper_off(void){
	UNSIGNED_16BIT damper_on;
	damper_on =  ams_reg_settings[1]&CLEAR_AMS_ANTENNA_DAMPER_OFF;
	spi_driver_read(0 , 0, (unsigned char *)&damper_on, sizeof (unsigned short));
}

void ob_lf_chip_damper_on(void){
	UNSIGNED_16BIT damper_on;
	damper_on =  ams_reg_settings[1]|SET_AMS_ANTENNA_DAMPER_ON;
	spi_driver_read(0 , 0, (unsigned char *)&damper_on, sizeof (unsigned short));
}
#endif
/*=============================================================================
	FUNCTION: mpp_power_save_mode

	DESCRIPTION:turn off all the peripherals

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED:none

	RETURN VALUE: none
==============================================================================*/
void mpp_power_save_mode(){
 
   HAL_MPP_TX_CLK_STOP;/* stop 125 khz clock*/
   HAL_MPP_UART_RX_DIS; /* disable MPP UART which will be enabled in wakeup interrupt*/
   HAL_MPP_UART_TX_DIS;  /* disbable mpp tc uart*/
   
   #if defined(MPP_OPTION_BOARD)
         /* enable wakeup interrupt*/
	   eic_clear_interrupt_line(&AVR32_EIC, EXT_INT1);
	   eic_enable_interrupt_line(&AVR32_EIC, EXT_INT1);
	   /*put ams into listen mode*/
	   clear_wake_line();
   #endif
   
   #if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE)
   HAL_MPP_DISABLE_WAKEUP_INT; /*diable wakeup interrupt*/
   HAL_MPP_SPI_INIT_MODE_POWER_DWN; /*SPI IN POWER DOWN MODE*/
   lf_chip_write(AMS_POWER_DOWN_CMD); /* AMS CHIP IN POWER DOWN MODE*/
   #endif
      
   if(mpp_time_to_pair>=0)
   {
     vr_free_timer(mpp_time_to_pair);
   }
   
   if(MPP_SERIAL_PORT_EVENT_HANDLE !=NULL)
   {
     MPP_OS_SemaphoreTake( mpp_sem_serial_port_event,0); /* just clear  semaphore */
     MPP_OS_QueueDelete((xQueueHandle)MPP_SERIAL_PORT_EVENT_HANDLE);
     MPP_SERIAL_PORT_EVENT_HANDLE = 0;
     
   }  

#ifdef MPP_HOST
#ifdef DEL_DB_MUTEX_ON_POWER_DOWN    
   if(MPP_CONNECTION_DB_MUTEX_HANDLE !=NULL)
   {
     if(MPP_OS_SemaphoreTake( MPP_CONNECTION_DB_MUTEX, MAX_DELAY) == MPP_SEM_TAKEN)
     { 
     MPP_OS_QueueDelete((xQueueHandle)MPP_CONNECTION_DB_MUTEX_HANDLE);
     MPP_CONNECTION_DB_MUTEX_HANDLE = 0;
     }
   } 
#endif   
#endif
   
return;
}

/*=============================================================================
	FUNCTION: HAL_MPP_READ_BYTE_SERIAL_PORT

	DESCRIPTION: start the pairing time

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED:none*

	RETURN VALUE: none
==============================================================================*/
void mpp_start_pairing_timer(){
/* start timer with maximum timeout value*/
  vr_write_timer(mpp_time_to_pair, MILLI_SEC_2000);
}
/* End of mpp_start_pairing_timer()   */

/*=============================================================================
	FUNCTION: mpp_get_pairing_time

	DESCRIPTION: return the time took in pairing process

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: time in msec

	RETURN VALUE: none
==============================================================================*/
UNSIGNED_16BIT mpp_get_pairing_time(){
 UNSIGNED_16BIT time_in_ms;
/* stop timer and minus the remaining time returned by stop timer form the original time set the substraction result will the the total time taken from timer start to stop */
 time_in_ms = MILLI_SEC_2000 - vr_stop_timer(mpp_time_to_pair);
 
 return time_in_ms;
}
/* End of mpp_get_pairing_time()   */


#ifndef MPP_SECURE_DONGLE /* Duplicate functions */ //CCMPD01460340(GM_Jan/10/2011)
/*=============================================================================
	FUNCTION: send_inter_task_msg

	DESCRIPTION: send pic message.

	ARGUMENTS PASSED: message id , pointer to message and mesage size

	REFERENCE ARGUMENTS PASSED: 

	RETURN VALUE: none
==============================================================================*/

void send_inter_task_msg(int taskId, UNSIGNED_8BIT msgid, void **ptr2message, UNSIGNED_16BIT message_size, UNSIGNED_8BIT op_code)
{
ms_task_msg_t *msg ;
/* fill in the buffer*/
msg = (ms_task_msg_t *) Get_Buf(Get_Res_Pool_Id(POOL_USER_MSG));
                    
                      msg->ros_msg.radio_if_msg.msg_hdr.msg_type  = USER_MSG; /* application message*/
                      msg->ros_msg.radio_if_msg.msg_hdr.op_code   = op_code; /*system level op code*/
                      msg->ros_msg.radio_if_msg.sub_opcode        = msgid; /* task level sub opcode*/
                      msg->ros_msg.radio_if_msg.data              = *ptr2message; /* pointer to message*/
                      msg->ros_msg.radio_if_msg.data_len          = message_size; /* size of message which may or maynot be used as data struct are shared*/

Put_Msg( (BUF_PTR)msg, taskId ); /* send to connection manager task*/

} 
/* End of mpp_send_inter_task_msg()   */
#endif //#ifndef(MPP_SECURE_DONGLE)  //CCMPD01460340(GM_Jan/10/2011)

/*=============================================================================
	FUNCTION: get_inter_task_msg

	DESCRIPTION: send pic message.

	ARGUMENTS PASSED: 

	REFERENCE ARGUMENTS PASSED: message id , pointer to message and mesage size

	RETURN VALUE: arguments passed
==============================================================================*/
UNSIGNED_8BIT get_inter_task_msg(UNSIGNED_8BIT *msgid, void **ptr2message, UNSIGNED_16BIT *message_size, UNSIGNED_32BIT_LONG delay ){

ms_task_msg_t *msg;
    
if(msg = (ms_task_msg_t *)Get_Msg_w_Time_Out(delay))/*get message from connection manager task and any other one*/
{

                    switch (msg->ros_msg.msg_hdr.op_code){
                    case SHUTDOWN:
            
                    MPP_OS_free_mem(msg);
                    vr_destroy_task(NULL); /* delete task if shutdown message recieved*/
                    break;
                    }
                    
*msgid = msg->ros_msg.radio_if_msg.sub_opcode; /* get sub opcode*/
*ptr2message = msg->ros_msg.radio_if_msg.data; /* get pointer to data*/
*message_size = msg->ros_msg.radio_if_msg.data_len;
MPP_OS_free_mem(msg);
return MPP_PASS;
} 
return MPP_FAIL;
}
/* End of get_inter_task_msg()   */

/*=============================================================================
	FUNCTION: HAL_MPP_READ_BYTE_SERIAL_PORT

	DESCRIPTION: Read a byte from Serial port.

	ARGUMENTS PASSED: UNSIGNED_32BIT_LONG

	REFERENCE ARGUMENTS PASSED: UNSIGNED_8BIT *

	RETURN VALUE: none
==============================================================================*/
UNSIGNED_8BIT mpp_read_byte_from_serialport(volatile UNSIGNED_8BIT *data, UNSIGNED_32BIT_LONG delay, UNSIGNED_8BIT whitening){
  MPP_ENTER_CRITICAL;
  MPP_UART_PORT->ier = AVR32_USART_IER_RXRDY_MASK; /*set ready mask to enable interrupt*/
  MPP_EXIT_CRITICAL;
  if(MPP_OS_SemaphoreTake( mpp_sem_serial_port_event, delay ) == MPP_SEM_TAKEN)/* check if there is event from UART rx interrupt*/
  {
    MPP_ENTER_CRITICAL;
   *data = (MPP_UART_PORT->rhr) ^ whitening; /* copy data*/
    MPP_EXIT_CRITICAL;

    return MPP_PASS;
  }
  return MPP_FAIL;
}
/* End of HAL_MPP_READ_BYTE_SERIAL_PORT()   */

/*=============================================================================
	FUNCTION: mpp_write_byte_to_serialport

	DESCRIPTION: Write a byte to serial port.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: UNSIGNED_8BIT *

	RETURN VALUE: none
==============================================================================*/
UNSIGNED_8BIT mpp_write_byte_to_serialport(volatile UNSIGNED_8BIT *data, UNSIGNED_8BIT whitening){

            MPP_ENTER_CRITICAL;
            MPP_UART_PORT->thr = (*data) ^ whitening; /*put data into tx register*/
            MPP_UART_PORT->ier = AVR32_USART_IER_TXRDY_MASK; /*enable tx ready interrupt*/
            MPP_EXIT_CRITICAL;
            if(MPP_OS_SemaphoreTake( mpp_sem_serial_port_event, MILLI_SEC_50 ) == MPP_SEM_TAKEN )/*wait till the tx register gets empty to avoid overwriting this register*/
            {
                 //MPP_UART_PORT->idr = AVR32_USART_IDR_TXRDY_MASK;
                 return MPP_PASS;
            }
    

  return MPP_FAIL;
  
}
/* End of mpp_write_byte_to_serialport()   */

/*=============================================================================
	FUNCTION: mpp_wait_tx_complete

	DESCRIPTION: This non-blocking function will implement wait till the 
                      entire MPP packet has been transmitted. 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_wait_tx_complete()
{
  
   if(MPP_OS_SemaphoreTake( mpp_sem_serial_port_event, MILLI_SEC_50 ) == MPP_SEM_TAKEN)/*wait till the tx register gets empty to avoid overwriting this register and on last byte tx wait to complete the tx so that tx clk can be disabled*/
            {
            //  MPP_UART_PORT->idr = AVR32_USART_IDR_TXRDY_MASK;
                 
            }
    
  
}
/* End of mpp_wait_tx_complete()   */
/*=============================================================================
	FUNCTION: mpp_trigger_tx

	DESCRIPTION: will enable tx init or dma or tx empty int based on implementation

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_trigger_tx()
{
 MPP_UART_PORT->ier = AVR32_USART_IER_TXEMPTY_MASK; /* enable tx empty interrupt here */
}
/* End of mpp_trigger_tx()   */

/*=============================================================================
	FUNCTION: mpp_hw_setup_idle_mode

	DESCRIPTION: Configure MPP hardware for idle state.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_hw_setup_idle_mode()
{
   host_cpb_rep_msg_txed = MPP_FAIL; 
   HAL_MPP_UART_RX_DIS; /* disable MPP UART which will be enabled in wakeup interrupt*/
   HAL_MPP_UART_TX_DIS;  /* disbable mpp tc uart*/
   HAL_MPP_TX_CLK_STOP;/* stop 125 khz clock*/
   
    #if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE) // CCMPD01412613 (YK_17Nov2010)
      #ifdef AMS_RX_PATH_DAMPER
        lf_chip_write(ams_reg_settings[1]|SET_AMS_ANTENNA_DAMPER_ON ); /* Turn Antenna Damper ON */
      #endif  
    #endif
  
    #if defined(MPP_OPTION_BOARD)
      #ifdef AMS_RX_PATH_DAMPER
        /* Turn Antenna Damper ON */
        ob_lf_chip_damper_on();
      #endif
    #endif
}
/* End of mpp_hw_setup_idle_mode */
/*=============================================================================
	FUNCTION: mpp_hw_setup_rx_mode

	DESCRIPTION: Configure MPP hardware for receiving NFC data.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_hw_setup_rx_mode()
{
   volatile UNSIGNED_8BIT data;
   HAL_MPP_UART_RX_RESET;
   HAL_MPP_UART_RX_DIS; /* disable MPP UART which will be enabled in wakeup interrupt*/
   HAL_MPP_TX_CLK_STOP;/* stop 125 khz clock*/
   MPP_UART_PORT->idr = AVR32_USART_IDR_RXRDY_MASK;
   MPP_UART_PORT->idr = AVR32_USART_CSR_TXEMPTY_MASK; 
   MPP_UART_PORT->idr = AVR32_USART_IDR_TXRDY_MASK;
   HAL_MPP_UART_TX_DIS;  /* disbable mpp tc uart*/
   
   data = MPP_UART_PORT->rhr; /* clear data if any in RX buffer*/
  
   #if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE) // CCMPD01412613 (YK_17Nov2010)
  
        #ifdef AMS_RX_PATH_DAMPER
           lf_chip_write(ams_reg_settings[1]&CLEAR_AMS_ANTENNA_DAMPER_OFF ); /* Turn Antenna Damper OFF */
        #endif   
         
         HAL_MPP_ENABLE_WAKEUP_INT; /* enable wakeup interrupt*/
         lf_listen_mode(); /*put ams into listen mode*/
   #endif
   
   #if defined(MPP_OPTION_BOARD)
          #ifdef AMS_RX_PATH_DAMPER
            ob_lf_chip_damper_off();
            /* Turn Antenna Damper OFF */
          #endif
          /* enable wakeup interrupt*/
	   eic_clear_interrupt_line(&AVR32_EIC, EXT_INT1);
	   eic_enable_interrupt_line(&AVR32_EIC, EXT_INT1);
	   /*put ams into listen mode*/
	   clear_wake_line();
   #endif
   
   MPP_OS_SemaphoreTake( mpp_sem_serial_port_event,0); /* just clear  semaphore */
}
/* End of mpp_hw_setup_rx_mode()   */

/*=============================================================================
	FUNCTION: mpp_hw_setup_tx_mode

	DESCRIPTION: Configure MPP hardware for transmitting NFC data.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_hw_setup_tx_mode()
{
  #if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE) // CCMPD01412613 (JJ 18/11/2010)
      #ifdef AMS_RX_PATH_DAMPER
        lf_chip_write(ams_reg_settings[1]|SET_AMS_ANTENNA_DAMPER_ON ); /* Turn Antenna Damper ON */
      #endif  
      HAL_MPP_DISABLE_WAKEUP_INT; /*diable wakeup interrupt*/
  #endif
  
  #if defined(MPP_OPTION_BOARD)
      #ifdef AMS_RX_PATH_DAMPER
        /* Turn Antenna Damper ON */
        ob_lf_chip_damper_on();
      #endif

      /*diable wakeup interrupt*/
        eic_disable_interrupt_line(&AVR32_EIC, EXT_INT1);
  #endif
  
  HAL_MPP_UART_RX_DIS;/*disable rx uart*/
  HAL_MPP_UART_TX_EN;  /*enable tx uart*/
  HAL_MPP_TX_CLK_START; /*start 125khz clock*/
  MPP_UART_PORT->idr = AVR32_USART_IDR_RXRDY_MASK;
  MPP_UART_PORT->idr = AVR32_USART_CSR_TXEMPTY_MASK; 
  MPP_UART_PORT->idr = AVR32_USART_IDR_TXRDY_MASK;  
  MPP_OS_SemaphoreTake( mpp_sem_serial_port_event,0);/* just clear  semaphore */
  
}
/* End of mpp_hw_setup_tx_mode()   */
/*=============================================================================
	FUNCTION: mpp_spi_reset

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/ 
void mpp_spi_reset(volatile avr32_spi_t *spi)
{
spi->cr = AVR32_SPI_CR_SWRST_MASK;
return;
}
/* end of mpp_spi_reset*/
/*=============================================================================
	FUNCTION: spi_initMaster

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/ 

void mpp_spi_initMaster(volatile avr32_spi_t *spi, const spi_options_t *options)
{
  
typedef union
{
  unsigned long                 mr;
  avr32_spi_mr_t                MR;
} u_avr32_spi_mr_t;

  u_avr32_spi_mr_t u_avr32_spi_mr;

  // Master Mode.
  u_avr32_spi_mr.mr = spi->mr;
  u_avr32_spi_mr.MR.mstr = 1;
  u_avr32_spi_mr.MR.modfdis = options->modfdis;
  u_avr32_spi_mr.MR.llb = 0;
  u_avr32_spi_mr.MR.pcs = (1 << AVR32_SPI_MR_PCS_SIZE) - 1;
  spi->mr = u_avr32_spi_mr.mr;

  return;
}
/* end of spi_initMaster*/

#define SPI_16_BIT_MODE	0x0080
#define SPI_8_BIT_MODE	0xFF0F	

/*=============================================================================
	FUNCTION: mpp_spi_8bit_mode

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/ 

void mpp_spi_8bit_mode()
{

  switch(MPP_SPI_CHANNEL) {
    case 0:
      SPI_FOR_AMS_CHIP->csr0 = SPI_FOR_AMS_CHIP->csr0&SPI_8_BIT_MODE;
      break;
    case 1:
      SPI_FOR_AMS_CHIP->csr1 = SPI_FOR_AMS_CHIP->csr1&SPI_8_BIT_MODE;
      break;
    case 2:
      SPI_FOR_AMS_CHIP->csr2 = SPI_FOR_AMS_CHIP->csr2&SPI_8_BIT_MODE;
      break;
    case 3:
      SPI_FOR_AMS_CHIP->csr3 = SPI_FOR_AMS_CHIP->csr3&SPI_8_BIT_MODE;
      break;
    default:
      return;
  }

  return;
}
/* end of mpp_spi_8bit_mode*/
/*=============================================================================
	FUNCTION: mpp_spi_16bit_mode

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/ 

void mpp_spi_16bit_mode()
{

  switch(MPP_SPI_CHANNEL) {
    case 0:
      SPI_FOR_AMS_CHIP->csr0 = SPI_FOR_AMS_CHIP->csr0|SPI_16_BIT_MODE;
      break;
    case 1:
      SPI_FOR_AMS_CHIP->csr1 = SPI_FOR_AMS_CHIP->csr1|SPI_16_BIT_MODE;
      break;
    case 2:
      SPI_FOR_AMS_CHIP->csr2 = SPI_FOR_AMS_CHIP->csr2|SPI_16_BIT_MODE;
      break;
    case 3:
      SPI_FOR_AMS_CHIP->csr3 = SPI_FOR_AMS_CHIP->csr3|SPI_16_BIT_MODE;
      break;
    default:
      return;
  }

  return;
}
/* end of mpp_spi_16bit_mode*/


/*=============================================================================
	FUNCTION: mpp_spi_init

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/ 
void mpp_spi_init(avr32_spi_t  *spi, spi_mode mpp_spi_mode)
{ 
 spi_options_t SPI_OPTIONS_MASTER[MPP_SPI_MODES] =
{
  /*16 bit*/
  {
  //! The SPI channel to set up.
  MPP_SPI_CHANNEL, //UNSIGNED_8BIT reg;

  //! Preferred baudrate for the SPI.
  MPP_SPI_RUN_TIME_BAUD_RATE, //UNSIGNED_32BIT_LONG baudrate;

  //! Number of bits in each character (8 to 16).
  16, //UNSIGNED_8BIT bits;

  //! Delay before first clock pulse after selecting slave (in PBA clock periods).
  0, //UNSIGNED_8BIT spck_delay;

  //! Delay between each transfer/character (in PBA clock periods).
  0, //UNSIGNED_8BIT trans_delay;

  //! Sets this chip to stay active after last transfer to it.
  0, //UNSIGNED_8BIT stay_act;

  //! Which SPI mode to use when transmitting.
  1, //UNSIGNED_8BIT spi_mode;

  //! Disables the mode fault detection.
  //! With this bit cleared, the SPI master mode will disable itself if another
  //! master tries to address it.
  1, //UNSIGNED_8BIT modfdis;
},
  /*8 bit*/
  {//! The SPI channel to set up.
  MPP_SPI_CHANNEL, //UNSIGNED_8BIT reg;

  //! Preferred baudrate for the SPI.
  MPP_SPI_RUN_TIME_BAUD_RATE, //UNSIGNED_32BIT_LONG baudrate;

  //! Number of bits in each character (8 to 16).
  8, //UNSIGNED_8BIT bits;

  //! Delay before first clock pulse after selecting slave (in PBA clock periods).
  0, //UNSIGNED_8BIT spck_delay;

  //! Delay between each transfer/character (in PBA clock periods).
  0, //UNSIGNED_8BIT trans_delay;

  //! Sets this chip to stay active after last transfer to it.
  0, //UNSIGNED_8BIT stay_act;

  //! Which SPI mode to use when transmitting.
  1, //UNSIGNED_8BIT spi_mode;

  //! Disables the mode fault detection.
  //! With this bit cleared, the SPI master mode will disable itself if another
  //! master tries to address it.
  1, //UNSIGNED_8BIT modfdis;
  },
  /*AMS OSC trim*/
  {
  //! The SPI channel to set up.
  MPP_SPI_CHANNEL, //UNSIGNED_8BIT reg;

  //! Preferred baudrate for the SPI.
  MPP_SPI_AMS_OSC_TRIM_BAUD_RATE, //UNSIGNED_32BIT_LONG baudrate;

  //! Number of bits in each character (8 to 16).
  8, //UNSIGNED_8BIT bits;

  //! Delay before first clock pulse after selecting slave (in PBA clock periods).
  0, //UNSIGNED_8BIT spck_delay;

  //! Delay between each transfer/character (in PBA clock periods).
  0, //UNSIGNED_8BIT trans_delay;

  //! Sets this chip to stay active after last transfer to it.
  1, //UNSIGNED_8BIT stay_act;

  //! Which SPI mode to use when transmitting.
  1, //UNSIGNED_8BIT spi_mode;

  //! Disables the mode fault detection.
  //! With this bit cleared, the SPI master mode will disable itself if another
  //! master tries to address it.
  0, //UNSIGNED_8BIT modfdis;
  }  
} ;

  if(present_spi_mode != mpp_spi_mode)
  {
    MPP_ENTER_CRITICAL;
    {
      present_spi_mode = mpp_spi_mode;
      if(mpp_spi_mode == SPI_MODE_POWER_DOWN)
      {
      spi_disable(spi);
      return;
      }
       
      mpp_spi_initMaster(spi, &SPI_OPTIONS_MASTER[mpp_spi_mode]);
      //spi_selectionMode(spi,0,0,mode[mpp_spi_mode]); /*last 0 or 6 fix this*/
      spi_setupChipReg(spi, &SPI_OPTIONS_MASTER[mpp_spi_mode], cpu_clk[mpp_spi_mode]);
      /* Enable receiver and transmitter... */
      spi_enable(spi);
    }
   MPP_EXIT_CRITICAL; 
    }
 return;
}
/* End of mpp_spi_init()   */

/*=============================================================================
	FUNCTION: mpp_uart_isr

	DESCRIPTION: USART0 interrupt service routine.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/  
#if __GNUC__
	__attribute__((__naked__))
#elif __ICCAVR32__
	#pragma shadow_registers = full   // Naked.
#endif

static void mpp_uart_isr( void )
{
  /* This ISR can cause a context switch, so the first statement must be a
  call to the MPP_OS_ENTER_SWITCHING_ISR macro.  This must be BEFORE any
  variable declarations. */
MPP_OS_ENTER_SWITCHING_ISR;

  mpp_serial_port_isr();
  
  /* Exit the ISR.  If a task was woken by either a character being received
  or transmitted then a context switch will occur. */

MPP_OS_EXIT_SWITCHING_ISR;
}
/* End of mpp_uart_isr()   */


#if __GNUC__
	__attribute__((__noinline__))
#elif __ICCAVR32__
	#pragma optimize = no_inline
#endif
static SIGNED_32BIT_LONG mpp_serial_port_isr( void )
{

  SIGNED_32BIT_LONG     xHigherPriorityTaskWoken = MPP_FAIL;
  UNSIGNED_32BIT_LONG     ulStatus;

  MPP_ENTER_CRITICAL;
  /* What caused the interrupt? */
  ulStatus = MPP_UART_PORT->csr & MPP_UART_PORT->imr;
  /* check if tx UART is ready means TX buffer is empty*/
  if (ulStatus & AVR32_USART_CSR_TXRDY_MASK)
  {   
    MPP_UART_PORT->idr = AVR32_USART_IDR_TXRDY_MASK;     
  }
  
  /*check if TX is complete on physical line*/
  if (ulStatus & AVR32_USART_CSR_TXEMPTY_MASK)
  {
    MPP_UART_PORT->idr = AVR32_USART_CSR_TXEMPTY_MASK; 
  }
  /* check if we have received any character*/

  if (ulStatus & AVR32_USART_CSR_RXRDY_MASK)
  {    
    MPP_UART_PORT->idr = AVR32_USART_IDR_RXRDY_MASK;
  }

  MPP_OS_SemaphoreGiveFromISR( MPP_SERIAL_PORT_EVENT_HANDLE, &xHigherPriorityTaskWoken );
  
 MPP_EXIT_CRITICAL;  
   
  return (xHigherPriorityTaskWoken );
}

/*=============================================================================
	FUNCTION: mpp_serialport_init

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/   
void mpp_serialport_init()
{  
  // USART options.
  usart_options_t MPP_USART_OPTIONS =
  {
    .baudrate     = 1200,
    .charlength   = 8,
    .paritytype   = USART_NO_PARITY,
    .stopbits     = USART_1_STOPBIT,
    .channelmode  = USART_NORMAL_CHMODE
  };
  

  /* Configure USART. */  
  MPP_ENTER_CRITICAL;
  {
    
    
    // Initialize USART in RS232 mode.
    usart_init_rs232(MPP_UART_PORT, &MPP_USART_OPTIONS, MPP_PBA_CLOCK_HZ);

    
    /* We're not fully done yet: disable receiver and transmitter. */

    MPP_UART_PORT->cr |= AVR32_USART_CR_RXDIS_MASK | AVR32_USART_CR_TXDIS_MASK;
    
    /* Register the USART interrupt handler to the interrupt controller and
    enable the USART interrupt. */

    INTC_register_interrupt((__int_handler)&mpp_uart_isr, MPP_PORT_USART_IRQ, AVR32_INTC_INT1);
    
  }
 MPP_EXIT_CRITICAL;  
  return;
}
/* End of mpp_serialport_init()   */



/*==============================================================================
                                       FUNCTION DEFINITIONS
==============================================================================*/
/* initialize TC to generate 125000 Hz signal */



/*=============================================================================
	FUNCTION: mpp_hw_tx_clk_init

	DESCRIPTION: This function is used to initialize the 125KHz clock.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/ 
void mpp_hw_tx_clk_init()
{  
  // Options for waveform genration.
  tc_waveform_opt_t waveform_opt =
  {
    .channel  = MPP_TC_CHANNEL,                         // Channel selection.
    
    .bswtrg   = TC_EVT_EFFECT_NOOP,                 // Software trigger effect on TIOB.
    .beevt    = TC_EVT_EFFECT_NOOP,                 // External event effect on TIOB.
    // PC and PB used in EVK101 as B1 used to output clock.
    #ifdef MPP_EVK1101
    .bcpc     = TC_EVT_EFFECT_TOGGLE,               // RC compare effect on TIOB.
    .bcpb     = TC_EVT_EFFECT_TOGGLE,               // RB compare effect on TIOB.
    #endif
    // PC and PA not used in P0.
    #ifdef MPP_P0_BOARD
    .bcpc     = TC_EVT_EFFECT_NOOP,               // RC compare effect on TIOB.
    .bcpb     = TC_EVT_EFFECT_NOOP,               // RB compare effect on TIOB.
    #endif
    
    #ifdef  MPP_OPTION_BOARD 
    .bcpc     = TC_EVT_EFFECT_NOOP,               // RC compare effect on TIOB.
    .bcpb     = TC_EVT_EFFECT_NOOP,               // RB compare effect on TIOB.
    #endif
    
    #ifdef  MPP_SECURE_DONGLE
    .bcpc     = TC_EVT_EFFECT_NOOP,               // RC compare effect on TIOB.
    .bcpb     = TC_EVT_EFFECT_NOOP,               // RB compare effect on TIOB.
    #endif
    
    .aswtrg   = TC_EVT_EFFECT_NOOP,                 // Software trigger effect on TIOA.
    .aeevt    = TC_EVT_EFFECT_NOOP,                 // External event effect on TIOA.
    
    // PC and PA not used in P0.
    #ifdef MPP_EVK1101
    .acpc     = TC_EVT_EFFECT_NOOP,                 // RC compare effect on TIOA: toggle.
    .acpa     = TC_EVT_EFFECT_NOOP,                 // RA compare effect on TIOA: toggle (other possibilities are none, set and clear).
    #endif
    // PPC and PA used in EVK101 as A1 used to output clock.
    #ifdef MPP_P0_BOARD
    .acpc     = TC_EVT_EFFECT_TOGGLE,                 // RC compare effect on TIOA: toggle.
    .acpa     = TC_EVT_EFFECT_TOGGLE,                 // RA compare effect on TIOA: toggle (other possibilities are none, set and clear)
    #endif
    
    #ifdef  MPP_OPTION_BOARD 
    .acpc     = TC_EVT_EFFECT_TOGGLE,                 // RC compare effect on TIOA: toggle.
    .acpa     = TC_EVT_EFFECT_TOGGLE,                 // RA compare effect on TIOA: toggle (other possibilities are none, set and clear)
    #endif
    
    #ifdef  MPP_SECURE_DONGLE
    .acpc     = TC_EVT_EFFECT_TOGGLE,                 // RC compare effect on TIOA: toggle.
    .acpa     = TC_EVT_EFFECT_TOGGLE,                 // RA compare effect on TIOA: toggle (other possibilities are none, set and clear)
    #endif
    
    .wavsel   = TC_WAVEFORM_SEL_UP_MODE_RC_TRIGGER, // Waveform selection: Up mode without automatic trigger on RC compare.
    .enetrg   = FALSE,                              // External event trigger enable.
    .eevt     = TC_EXT_EVENT_SEL_XC0_OUTPUT,        // External event selection.
    .eevtedg  = TC_SEL_NO_EDGE,                     // External event edge selection.
    .cpcdis   = FALSE,                              // Counter disable when RC compare.
    .cpcstop  = FALSE,                              // Counter clock stopped with RC compare.
    
    .burst    = TC_BURST_NOT_GATED,                 // Burst signal selection.
    .clki     = TC_CLOCK_RISING_EDGE,               // Clock inversion.
    .tcclks   = TC_CLOCK_SOURCE_TC2                 // Internal source clock 2, connected to fPBA / 2.
  };
  MPP_ENTER_CRITICAL;  
  
  tc_init_waveform(MPP_TC_HANDLE, &waveform_opt);  // Initialize the timer/counter waveform.
  
  // The TC module uses Waveform mode to generate the 125KHz clock.
  // EVK1101 uses TC and TB. P0 uses TA and TC.
  #ifdef MPP_EVK1101  
  tc_write_rb(MPP_TC_HANDLE, MPP_TC_CHANNEL,MPP_TC_MODULE_RB_VALUE);    // Set RB value. 
  #endif
  
  #ifdef MPP_P0_BOARD  
  tc_write_ra(MPP_TC_HANDLE, MPP_TC_CHANNEL,MPP_TC_MODULE_RA_VALUE);    // Set RA value.   
  #endif
  
  #ifdef MPP_OPTION_BOARD
  tc_write_ra(MPP_TC_HANDLE, MPP_TC_CHANNEL,MPP_TC_MODULE_RA_VALUE);    // Set RA value.   
  #endif
  
  #ifdef MPP_SECURE_DONGLE
  tc_write_ra(MPP_TC_HANDLE, MPP_TC_CHANNEL,MPP_TC_MODULE_RA_VALUE);    // Set RA value.   
  #endif
  
  tc_write_rc(MPP_TC_HANDLE, MPP_TC_CHANNEL,MPP_TC_MODULE_RC_VALUE);    // Set RC value.
  
 MPP_EXIT_CRITICAL;
}
/* End of mpp_hw_tx_clk_init()   */
 /*=============================================================================
	FUNCTION: mpp_wakeup_setup

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/ 
#if defined(MPP_OPTION_BOARD)
void mpp_wakeup_setup()
{  
  
}  
#else

void mpp_wakeup_setup()
{  

   /* Register the Wakeup  interrupt handler to the interrupt controller and
  enable the interrupt. */
  
#if defined(MPP_SECURE_DONGLE) //CCMPD01413200
  /* Different ISR for R2.1 */
  INTC_register_interrupt((__int_handler)&gpio_group2_irq1_controls_isr, MPP_WAKEUP_IRQ, MPP_WAKEUP_IRQ_LEVEL);
#else
  INTC_register_interrupt((__int_handler)&mpp_wakeup_isr, MPP_WAKEUP_IRQ, MPP_WAKEUP_IRQ_LEVEL);
#endif
  
  /* enable GPIO interrupt */
  gpio_enable_pin_interrupt(MPP_WAKEUP_INTERRUPT_GPIO_PIN, GPIO_RISING_EDGE);   
}
#endif

/* End of mpp_wakeup_setup()   */     
/*=============================================================================
	FUNCTION: mpp_wakeup_isr

	DESCRIPTION: This function implements the ISR for interrupt received from 
                     the Wake-Up pin of the Austria Micro Systems chip.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/


#ifdef MPP_EVK1101   
#if __GNUC__
	__attribute__((__naked__))
#elif __ICCAVR32__
	#pragma shadow_registers = full   // Naked.
#endif
void mpp_wakeup_isr( void )
{
  /* This ISR can cause a context switch, so the first statement must be a
  call to the MPP_OS_ENTER_SWITCHING_ISR macro.  This must be BEFORE any
  variable declarations. */
  MPP_OS_ENTER_SWITCHING_ISR;  
 
  MPP_UART_PORT->cr |= AVR32_USART_CR_RXEN_MASK;
  gpio_clear_pin_interrupt_flag(MPP_WAKEUP_INTERRUPT_GPIO_PIN);

  MPP_OS_EXIT_SWITCHING_ISR;
}
#endif

#if defined(MPP_P0_BOARD) 
#if __GNUC__
	__attribute__((__naked__))
#elif __ICCAVR32__
	#pragma shadow_registers = full   // Naked.
#endif
void mpp_wakeup_isr( void )
{ /* For R2.1 (secure dongle), the same processing has been moved to gpio_group2_irq1_controls_isr(void)       //CCMPD01413200 */
  volatile avr32_gpio_port_t *gpio_port = &MPP_GPIO.port[MPP_WAKEUP_INTERRUPT_GPIO_PIN >> 5];
  UNSIGNED_32BIT_LONG int_status;
  
  /* This ISR can cause a context switch, so the first statement must be a
  call to the MPP_OS_ENTER_SWITCHING_ISR macro.  This must be BEFORE any
  variable declarations. */
  MPP_OS_ENTER_SWITCHING_ISR;
  int_status =  gpio_port->ifr &  gpio_port->ier;
  
  switch(int_status){
  
  case MPP_WAKEUP_INTERRUPT_GPIO_PIN_MASK:
    gpio_clear_pin_interrupt_flag(MPP_WAKEUP_INTERRUPT_GPIO_PIN);
    MPP_UART_PORT->cr |= AVR32_USART_CR_RXEN_MASK;
    break;
  
  default:
    
    break;
  }

  /* Exit the ISR.  If a task was woken by either a character being received
  or transmitted then a context switch will occur. */

  MPP_OS_EXIT_SWITCHING_ISR;
}
#elif defined(MPP_OPTION_BOARD)

void mpp_wakeup_isr( void )
{
  //MPP_OS_ENTER_SWITCHING_ISR;
 MPP_ENTER_CRITICAL;
 //DBG_MSG("\r\nNFC_WAKE!!!\r\n");

 eic_clear_interrupt_line(&AVR32_EIC, EXT_INT1);
 eic_disable_interrupt_line(&AVR32_EIC, EXT_INT1);

 MPP_UART_PORT->cr |= AVR32_USART_CR_RXEN_MASK;
 MPP_EXIT_CRITICAL;
//MPP_OS_EXIT_SWITCHING_ISR;
}
#endif

/* End of mpp_wakeup_isr()   */


/*=============================================================================
	FUNCTION: mpp_gpio_init

	DESCRIPTION: This function will intialize all the GPIO necessary for the MPP module.
                     GPIO are SPI (4), USART(2), DEBUG_USART(2) and 125KHz Clk(1).
                     Initialization is based on the "gpio_enable_module()" function 
                     implementation of AVR framework 1.5.0

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/   
void mpp_gpio_init(void)
{
  /*   MPP_GPIO defines. this structure is specific to AVR. 
    Specific Pin mapping is dependant on whether the board is EVK1101 or P0. */
  // Instantiate the GPIO structure
  gpio_map_t MPP_GPIO_MAP =
  {
    // MPP SPI pin mapping
    {MPP_SPI_NPCS_GPIO_PIN, MPP_SPI_NPCS_GPIO_PIN_FUNCTION },//{ AVR32_SPI_NPCS_0_0_PIN, AVR32_SPI_NPCS_0_0_FUNCTION },
    {MPP_SPI_MISO_GPIO_PIN, MPP_SPI_MISO_GPIO_PIN_FUNCTION},//{ AVR32_SPI_MISO_0_0_PIN, AVR32_SPI_MISO_0_0_FUNCTION },  
    {MPP_SPI_SCK_GPIO_PIN, MPP_SPI_SCK_GPIO_PIN_FUNCTION },//{ AVR32_SPI_SCK_0_0_PIN, AVR32_SPI_SCK_0_0_FUNCTION },
    {MPP_SPI_MOSI_GPIO_PIN, MPP_SPI_MOSI_GPIO_PIN_FUNCTION },//{ AVR32_SPI_MOSI_0_0_PIN, AVR32_SPI_MOSI_0_0_FUNCTION }
     #ifdef MPP_EVK1101
    { AVR32_SPI_NPCS_0_0_PIN, AVR32_SPI_NPCS_0_0_FUNCTION },
     #endif
    // MPP USART pin mapping
    {MPP_USART_RXD_GPIO_PIN, MPP_USART_RXD_GPIO_PIN_FUNCTION},//{ AVR32_USART0_RXD_0_1_PIN, AVR32_USART0_RXD_0_1_FUNCTION },
    {MPP_USART_TXD_GPIO_PIN, MPP_USART_TXD_GPIO_PIN_FUNCTION},//{ AVR32_USART0_TXD_0_1_PIN, AVR32_USART0_TXD_0_1_FUNCTION },

    #ifdef MPP_DEBUG
    // MPP DEBUG USART pin mapping
    {MPP_DEBUG_USART_RXD_GPIO_PIN, MPP_DEBUG_USART_RXD_GPIO_PIN_FUNCTION},//{ serialPORT_USART_RX_PIN, serialPORT_USART_RX_FUNCTION },
    {MPP_DEBUG_USART_TXD_GPIO_PIN, MPP_DEBUG_USART_TXD_GPIO_PIN_FUNCTION},// { serialPORT_USART_TX_PIN, serialPORT_USART_TX_FUNCTION }
    #endif

    // MPP NFC 125 KHz CLOCK OUT pin mapping
    {MPP_125KHz_CLOCK_OUT_GPIO_PIN, MPP_125KHz_CLOCK_OUT_GPIO_PIN_FUNCTION},//{ AVR32_TC_B1_0_1_PIN, AVR32_TC_B1_0_1_FUNCTION } 
    // MPP WAKE pin mapping
    {MPP_WAKEUP_INTERRUPT_GPIO_PIN, MPP_WAKEUP_INTERRUPT_GPIO_PIN_FUNCTION},
  };
  
  // Configure the pin functionality.
  gpio_enable_module( MPP_GPIO_MAP, sizeof( MPP_GPIO_MAP ) / sizeof( MPP_GPIO_MAP[0] ) );
  gpio_enable_pin_pull_up(MPP_USART_RXD_GPIO_PIN);
  gpio_enable_pin_pull_up(MPP_USART_TXD_GPIO_PIN);
  gpio_enable_pin_pull_up(MPP_WAKEUP_INTERRUPT_GPIO_PIN);
  gpio_enable_pin_pull_up(MPP_125KHz_CLOCK_OUT_GPIO_PIN);
}
/* End of mpp_gpio_init()   */

/*=============================================================================
	FUNCTION: mpp_sw_init

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/   


  
/****************MOTO CODE********************/
void mpp_sw_init(void)
{
    host_cpb_rep_msg_txed = MPP_FAIL; 
   #ifdef MPP_HOST
  /* this has to be in connection manager*/
    MPP_OS_SemaphoreCreateMutex(MPP_CONNECTION_DB_MUTEX);
    if(MPP_CONNECTION_DB_MUTEX_HANDLE == NULL)
    {
      mpp_send_msg_to_app(MPP_ERROR);
      return;
    }
   #endif  
    
    MPP_OS_SemaphoreCreateBinary(mpp_sem_serial_port_event);
    if( MPP_SERIAL_PORT_EVENT_HANDLE == NULL )
    {
      mpp_send_msg_to_app(MPP_ERROR);
      return;
    }

    if((mpp_time_to_pair = vr_get_timer())<0){
      mpp_send_msg_to_app(MPP_ERROR);
      return;
    }
    
  #ifdef MPP_DEVICE
    mpp_send_msg_to_app(INIT_DEVICE_MPP_PKT_REQ);
  #endif
    
  #ifdef MPP_HOST
    mpp_send_msg_to_app(INIT_HOST_MPP_PKT_REQ);
    mpp_send_msg_to_app(HOST_CONN_ALLOC_RES_DB_REQ);
  #endif

}

/*=============================================================================
	FUNCTION: mpp_hw_peripherals_init

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/   
void mpp_hw_peripherals_init(void)
{
#if defined(MPP_EVK1101) || defined(MPP_P0_BOARD) || defined(MPP_SECURE_DONGLE) // CCMPD01412613 (JJ 16/11/2010)
  MPP_ENTER_CRITICAL;  
  /* Enable GPIO */ 
  mpp_gpio_init();  
  mpp_spi_reset(SPI_FOR_AMS_CHIP);
  mpp_spi_init(SPI_FOR_AMS_CHIP, SPI_MODE_16_BIT);
  mpp_serialport_init();  
  mpp_wakeup_setup();
  mpp_hw_tx_clk_init();
  mpp_lfc_chip_init(); 
  mpp_spi_init(SPI_FOR_AMS_CHIP, SPI_MODE_16_BIT);
  MPP_EXIT_CRITICAL;
#endif
  
#if defined(MPP_OPTION_BOARD)
  MPP_ENTER_CRITICAL;
  /* Enable GPIO */
  //mpp_gpio_init();
  mpp_serialport_init();
  //mpp_wakeup_setup();
  mpp_hw_tx_clk_init();
  mpp_lfc_chip_init();
  MPP_EXIT_CRITICAL;
#endif         
}
/* End of mpp_hw_peripherals_init()   */
