/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: mpp_msg.c
#
# --------------------------- General Description -----------------------------
# This file contains the definations of MPP messages and its initialization
#
# --------------------------- HEADER FILE INCLUDES ---------------------------*/

#include "mpp_init.h"
#include "mpp_msg.h"
#include "mpp_statemachine.h"
#include <systemLogger.h>

/*******************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR            Date Modified Tracking Number Description
* Jaspreet Bajwa 01/08/2010    CCMPD              Initial Creation
* Rhee Zhang     05/07/2010    CCMPD01345347      Cleanup IAR Compiler Warnings
* Mahesh.G       01-Feb-2011   CCMPD01468897      Reverse checking of BT  Address in "mpp_search_mac_from_list"
* WRN637         07/11/2013    CCMPD01825398      MPP Class 1 RFPA handling
*
*--------------------------- End of History Template----------------------------
* *****************************************************************************/

/*==============================================================================
                                      EXTERNAL DEFINITIONS
==============================================================================*/


/*==============================================================================
                                    GLOBAL VARIABLES
==============================================================================*/
UNSIGNED_8BIT host_cpb_rep_msg_txed = MPP_FAIL;
extern UNSIGNED_16BIT mpp_get_pairing_time();
extern void mpp_start_pairing_timer();
static UNSIGNED_16BIT  valid_beacon_proc_total = 0;
static UNSIGNED_8BIT   valid_data_msg_proc_total = 0;
static UNSIGNED_8BIT run_once = 0;
portTickType start_tick_ms, end_tick_ms, mpp_fail_tick_ms;
/*==============================================================================
                                      EXTERNAL FUNCTIONS
==============================================================================*/

/*==============================================================================
                                      FUNCTION PROTOYPES
==============================================================================*/

UNSIGNED_8BIT mpp_read_byte_from_serialport(volatile UNSIGNED_8BIT *data, UNSIGNED_32BIT_LONG delay, UNSIGNED_8BIT whitening);
UNSIGNED_8BIT mpp_write_byte_to_serialport(volatile UNSIGNED_8BIT *data, UNSIGNED_8BIT whitening);
void mpp_send_host_cpb_rep();
void mpp_process_host_cpb_rep();
void mpp_send_dev_cpb_brdcst();
void mpp_send_host_reject_msg();
void mpp_process_dev_cpb_brdcst();
void mpp_send_beacon_brdcst();
void mpp_send_beacon_conf();
void mpp_process_beacon_brdcst();
void mpp_process_beacon_conf();
void mpp_send_wakeup_sequence();


/*==============================================================================
                                      FUNCTION DEFINITIONS
==============================================================================*/
/*=============================================================================
	FUNCTION: crc_ccitt_update

	DESCRIPTION:

	ARGUMENTS PASSED: UNSIGNED_16BIT, UNSIGNED_8BIT

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: UNSIGNED_16BIT
==============================================================================*/
UNSIGNED_16BIT crc_ccitt_update( UNSIGNED_16BIT crc, UNSIGNED_8BIT data )
{
  UNSIGNED_16BIT tmp;

  tmp = data ^ (crc >> 8);
  tmp ^= tmp >> 4;
  crc = (crc << 8) ^ tmp ^ (tmp << 5) ^ (tmp << 12);
  return crc;
}
/* End of crc_ccitt_update()   */


 /*=============================================================================
	FUNCTION: mpp_send_crc

	DESCRIPTION: Check the CRC on message
	ARGUMENTS PASSED: crc calculated so far

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_send_crc(UNSIGNED_16BIT crcCalValue){
 UNSIGNED_8BIT mppDataOut;
 mppDataOut = (crcCalValue>>8)&0x00ff;
/*send msb of crc */
    HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, NO_WHITENING);

    #ifdef MPP_DEBUG
    MPP_PUT_DEBUG_BYTE(mppDataOut);
    #endif

    mppDataOut = crcCalValue&0x00ff;
/* send lsb of crc*/
    HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, NO_WHITENING);

    #ifdef MPP_DEBUG
    MPP_PUT_DEBUG_BYTE(mppDataOut);
    #endif
}
/* End of mpp_send_crc()   */
/*=============================================================================
	FUNCTION: mpp_check_crc

	DESCRIPTION: Check the CRC on message
	ARGUMENTS PASSED: crc calculated so far

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: PASS, FAIl, TIMEOUT
==============================================================================*/
UNSIGNED_8BIT mpp_check_crc(UNSIGNED_16BIT crcCalValue){

UNSIGNED_8BIT rxed_byte;
rxed_byte=0;
/*rx crc byte and compare against the msb of crc calculated so far*/
        if(HAL_MPP_READ_BYTE_SERIAL_PORT(&rxed_byte, MPP_UART_INTER_BYTE_DELAY, NO_WHITENING))/*try to read first byte of CRC*/
        {

          #ifdef MPP_DEBUG
          MPP_PUT_DEBUG_BYTE(rxed_byte);
          #endif

          if( !(((crcCalValue&0xff00)>>8) == rxed_byte) )
          {
             return MPP_FAIL;
          }

        }
        else
        {
            return MPP_TIME_OUT;
         }

 /*rx crc byte and compare against the lsb of crc calculated so far*/
        if(HAL_MPP_READ_BYTE_SERIAL_PORT(&rxed_byte, MPP_UART_INTER_BYTE_DELAY, NO_WHITENING)) /*try to read second byte of CRC*/
        {
            #ifdef MPP_DEBUG
            MPP_PUT_DEBUG_BYTE(rxed_byte);
            #endif
            if( !((crcCalValue&0x00ff) == rxed_byte) )
            {
               return MPP_FAIL;
            }
            else
            {
                return MPP_PASS;
            /* success */
          }

        }
        else
        {
            return MPP_TIME_OUT;

        }
}
/* End of mpp_check_crc()   */

/*                  COMMON FOR DEVICE AND HOST                               */
/*=============================================================================
	FUNCTION: mpp_send_wakeup_sequence

	DESCRIPTION: Send the wakeup pattern

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_send_wakeup_sequence()
{

  UNSIGNED_8BIT mppDataOut;
  /* wakeup pattern byte 1 taking into consideration that
  1)UART is tx least significant bit first and AMS chip is interpretting first incomming bit as most significant one
  2)AMS always detect wake up sequence which is manchester coded
  3)AMS follows OOK which is 1 as carrier and 0 as no carrier and our hardware design is sending 1 as no carrier and 0 as carrier
  4) Wake up sequence consists of 20 bits we are sending 16 bit as data and taking 4 bits as start / stop bits added bu UART.
  5) Wake up sequence consists of carrier burst+ preamble +patter as define in AMS 3930 data sheet
  */

  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_STRING( "\r\n SENDING PATTERN: \0" );
  #endif
  mppDataOut = MPP_PATTER_BYTE_1;
  HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, NO_WHITENING);

  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_BYTE(mppDataOut);
  #endif
  /* wakeup pattern byte 2 taking into consideration above mentioned facts */
  mppDataOut = MPP_PATTER_BYTE_2; /* wakeup pattern byte 1*/

  HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, NO_WHITENING);

  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_BYTE(mppDataOut);
  #endif

  //mpp_trigger_tx(); /* trigger the tx kept here for different implementaions like for DMA based transmission and queue based tx handling, in this implementaion its serving the purpose to insure that last byte is completely tx on the physical line so that we can disbale tx clock*/

  return;
}
/* End of mpp_send_wakeup_sequence()   */




/**                                                                          **/

/*                     DEVICE MESSAGES START FROM HERE                          */

/**                                                                          **/
#ifdef MPP_DEVICE
/*=============================================================================
	FUNCTION: mpp_send_ack_to_host

	DESCRIPTION: Send the final ACK indicating that Host capability message is received.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_send_ack_to_host()
{
  UNSIGNED_8BIT mppDataOut;
  mppDataOut = MPP_ACK_TO_HOST_BYTE;

    #ifdef MPP_DEBUG
    MPP_PUT_DEBUG_STRING( "\r\n SEND ACK TO HOST: \0" );
    #endif

    #ifdef MPP_DEBUG
    MPP_PUT_DEBUG_BYTE(mppDataOut);
    #endif

    HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, WHITENING_BYTE);
    mpp_trigger_tx();

  return;
}
/* End of mpp_send_ack_to_host()   */
/*=============================================================================
	FUNCTION: mpp_send_beacon_brdcst

	DESCRIPTION: Send the MPP beacon Broadcast message (Host side).

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/

void mpp_send_beacon_brdcst()
{
 UNSIGNED_8BIT mppDataOut;
 UNSIGNED_16BIT crcCalValue;
 mpp_pkt_byte byte_to_add;

  byte_to_add = BB_BYTE1;
  crcCalValue = INIT_CRC_CCITT;  /* init crc variable*/
  mppDataOut = 0;

  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_STRING( "\r\n SEND BEACON BRDCAST: \0" );
  #endif
  for(;;){

    switch(byte_to_add){

    case BB_BYTE1:
      /*compose first beacon broadcast byte as per MPP ICD , but this will not be used in release 1*/
      mppDataOut = BT_DEVICE_TIERING_CODE&0x0F;
      mppDataOut = mppDataOut | (BT_DEVICE_LF_BASLINE_OPER<<4);
      mppDataOut = mppDataOut | (BT_DEVICE_LF_SEC_CAP_EXT_BIT_0<<5);
      mppDataOut = mppDataOut | (BT_DEVICE_LF_SEC_CAP_EXT_BIT_1<<6);
      mppDataOut = mppDataOut | (BT_DEVICE_LF_SEC_CAP_EXT_BIT<<7);

        if(BT_DEVICE_LF_SEC_CAP_EXT_BIT) /* if extension bit set go and add extension byte else add crc not used in release 1*/
        {
          byte_to_add = BB_EXT_BYTE;
        }
        else
        {
          byte_to_add = BYTE_CRC;
        }
      break;

    case BB_EXT_BYTE:
      mppDataOut = BT_DEVICE_LF_SEC_CAP_EXT_BYTE; /* add extension byte*/
     break;

    case BYTE_CRC:
     mpp_send_crc(crcCalValue); /* calulate and send crc*/
     mpp_trigger_tx();      /* trigger the tx kept here for different implementaions like for DMA based transmission and queue based tx handling, in this implementaion its serving the purpose to insure that last byte is completely tx on the physical line so that we can disbale tx clock*/
     mpp_start_pairing_timer();
     return;

    default:
      #ifdef MPP_DEBUG
      MPP_PUT_DEBUG_STRING( "\r\n DEFAULT CASE ERROR: \0" );
      #endif
      mpp_send_crc(crcCalValue);
      mpp_trigger_tx();
      return;
    }
    crcCalValue = crc_ccitt_update(crcCalValue, mppDataOut); /* update crc on each loop*/
    HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, WHITENING_BYTE);

    #ifdef MPP_DEBUG
    MPP_PUT_DEBUG_BYTE(mppDataOut);
    #endif
  }


}
/* End of mpp_send_beacon_brdcst()   */



/*=============================================================================
	FUNCTION: mpp_send_dev_cpb_brdcst

	DESCRIPTION: Send the MPP Capability Broadcast message (Device side).

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_send_dev_cpb_brdcst()
{
 UNSIGNED_16BIT crcCalValue;
 UNSIGNED_8BIT mppDataOut;
 UNSIGNED_8BIT total_bytes_to_send, mac_byte_no, service_entry_no;
 mpp_pkt_byte byte_to_add, tt_type_bt_add_byte;
  tt_type_bt_add_byte = BT_ENC_TYPE;
  byte_to_add = BYTE_1_PL_TT;
  crcCalValue = INIT_CRC_CCITT;
  mppDataOut = 0;

  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_STRING( "\r\n DEV CPB BRDCAST: \0" );
  #endif

  for(;;)
  {
    switch(byte_to_add)
    {

      case BYTE_1_PL_TT:
      /* compose first byte of device capability*/
        mppDataOut = BT_DEVICE_TRANSACT_TYPE;
        total_bytes_to_send = MIN_BYTES_IN_DEV_CPB_BRDCAST + BT_DEVICE_LINK_KEY_EXT_BIT + BT_DEVICE_LINK_ENC_EXT_BIT + BT_DEVICE_SERVICES_TO_REQUEST; /*estinate the total bytes to send based on extension bits and number of services going to be requested*/
        /* set the payload bits as per MPP ICD*/
        if(total_bytes_to_send<=15)
        {
          mppDataOut = mppDataOut|(PAYLOAD_15B_OR_LESS_VALUE_2<<5);
        }
        else if(total_bytes_to_send<=22)
        {
          mppDataOut = mppDataOut|(PAYLOAD_22B_OR_LESS_VALUE_3<<5);
        }
        else if(total_bytes_to_send<=31)
        {
          mppDataOut = mppDataOut|(PAYLOAD_31B_OR_LESS_VALUE_4<<5);
        }
        else if(total_bytes_to_send<=41)
        {
          mppDataOut = mppDataOut|(PAYLOAD_41B_OR_LESS_VALUE_5<<5);
        }

        byte_to_add = (mpp_pkt_byte)BT_DEVICE_TRANSACT_TYPE; /* only one transaction is handled in release 1 , pulled from dev init struct filled by connection manager*/
        if(BT_DEVICE_TRANSACT_TYPE == BYTE_2_FOR_TT_TYPE_BLUETOOTH){ tt_type_bt_add_byte = BT_ENC_TYPE;} /* id BT type then add encryption tpe*/
        break;

      case BYTE_2_FOR_TT_TYPE_BLUETOOTH:
      /* sub types for BT transaction*/
        switch (tt_type_bt_add_byte)
        {

          case BT_ENC_TYPE:

            /* all datastructs need to be initialized with values at their offsets */
            /* add link key capabilty as per MPP ICD*/
            mppDataOut = (BT_DEVICE_SSP_OOB)|(BT_DEVICE_128_KEY_PUSH<<1)|(BT_DEVICE_LINK_KEY_EXT_BIT<<3);
            if(BT_DEVICE_LINK_KEY_EXT_BIT ) /*if ext bit is set add the msb of extension byte*/
            {
              mppDataOut = mppDataOut |(BT_DEVICE_LINK_KEY_EXT_BYTE&0xF0);
              tt_type_bt_add_byte = BT_ENC_TYPE_BYTE2;      /* go to add byte 2 of link enc type*/
            }
            else
            {
              mppDataOut = mppDataOut | ((BT_DEVICE_LINK_ENC_TYPE_128_56<<4)|(BT_DEVICE_LINK_ENC_EXT_BIT<<7));      /* add link encryption type here if link key ext bit is not set*/
              if(BT_DEVICE_LINK_ENC_EXT_BIT) /* if link enc ext bit is set go to add byte two and add that extension byte there*/
              {
                tt_type_bt_add_byte = BT_ENC_TYPE_BYTE2;
              }
              else
              {
                tt_type_bt_add_byte = BT_DEV_TYPE_AND_SERVICE_TYPE;        /* else just move to add services tobe requested*/
              }
            }
            break;

        case BT_ENC_TYPE_BYTE2:

          if(BT_DEVICE_LINK_KEY_EXT_BIT) /* if link key ext bit is set add link key extension lsb byte*/
          {
            mppDataOut = BT_DEVICE_LINK_KEY_EXT_BYTE&0x0F;
            mppDataOut = mppDataOut |((BT_DEVICE_LINK_ENC_TYPE_128_56<<4)|(BT_DEVICE_LINK_ENC_EXT_BIT<<7));     /* in msb add link enc capability*/
            if(BT_DEVICE_LINK_ENC_EXT_BIT)/* if link enc ext bit is set go to byte 3 and add the extension byte there */
            {
              tt_type_bt_add_byte = BT_ENC_TYPE_BYTE3;
            }
            else
            {
              tt_type_bt_add_byte = BT_DEV_TYPE_AND_SERVICE_TYPE; /* else add services*/
            }
          }
          else if(BT_DEVICE_LINK_ENC_EXT_BIT)/* if link enc extension is to be added add here and move to add services*/
          {
            mppDataOut = BT_DEVICE_LINK_ENC_EXT_BYTE;
            tt_type_bt_add_byte = BT_DEV_TYPE_AND_SERVICE_TYPE;
          }
          break;

        case BT_ENC_TYPE_BYTE3:

          mppDataOut = BT_DEVICE_LINK_ENC_EXT_BYTE; /* add link enc extension byet*/
          tt_type_bt_add_byte = BT_DEV_TYPE_AND_SERVICE_TYPE;

          break;


        case BT_DEV_TYPE_AND_SERVICE_TYPE:

          mppDataOut = BT_DEVICE_DEVICE_TYPE&0x03; /* add device type as per MPP ICD*/
          mppDataOut = mppDataOut | (BT_DEVICE_SERVICES_TO_REQUEST<<5);        /* add number of services to be requested from host*/
          if(!BT_DEVICE_SERVICES_TO_REQUEST)
          {
            mac_byte_no=0; tt_type_bt_add_byte = BT_MAC_ADDRESS; /*if no service to request go to add mac address*/
          }
          else /* else service fields*/
          {
            service_entry_no=0; tt_type_bt_add_byte = BT_SERVICE_TYPE_STRUCT_BYTES;
          }
          break;

        case BT_SERVICE_TYPE_STRUCT_BYTES:
        /* add nibbler capability and profile type*/
          mppDataOut = BT_DEVICE_SERVICE_TYPE_NIBBLER(service_entry_no)|(BT_DEVICE_SERVICE_TYPE_PROFILE(service_entry_no)<<1);
          ++service_entry_no;

          if (service_entry_no>=BT_DEVICE_SERVICES_TO_REQUEST) /* when all added go to add mac address*/
          {
            mac_byte_no=0;
            tt_type_bt_add_byte = BT_MAC_ADDRESS;
          }
          break;

        case BT_MAC_ADDRESS:
        /*considering MSB at array index 0*/
        /* add mac bytes here*/
        mppDataOut = BT_DEVICE_MAC_ADDRESS(mac_byte_no++);
        if(mac_byte_no>=6)
        {
          byte_to_add = BYTE_CRC; /* when done move to add crc*/
        }
        break;

        default:
          #ifdef MPP_DEBUG
          MPP_PUT_DEBUG_STRING( "\r\n DEFAULT CASE ERROR: \0" );
          #endif
          mpp_send_crc(crcCalValue);
          mpp_trigger_tx();
          return;


        }
        break;

      case BYTE_CRC:

      mpp_send_crc(crcCalValue);/* add crc and send*/

      mpp_trigger_tx();/* trigger the tx kept here for different implementaions like for DMA based transmission and queue based tx handling, in this implementaion its serving the purpose to insure that last byte is completely tx on the physical line so that we can disbale tx clock*/

      return;



      default:
        #ifdef MPP_DEBUG
        MPP_PUT_DEBUG_STRING( "\r\n DEFAULT CASE ERROR: \0" );
        #endif
        mpp_send_crc(crcCalValue);
        mpp_trigger_tx();
        mpp_send_msg_to_app(MPP_ERROR);
        return;

    }


    crcCalValue = crc_ccitt_update(crcCalValue, mppDataOut); /* keep on updating the CRC*/
    HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, WHITENING_BYTE);
    #ifdef MPP_DEBUG
    MPP_PUT_DEBUG_BYTE(mppDataOut);
    #endif
  }

}
/* End of mpp_send_dev_cpb_brdcst()   */

/*=============================================================================
	FUNCTION: mpp_process_beacon_conf

	DESCRIPTION: Process the MPP beacon confirmation message (Device side).

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_process_beacon_conf()
{
  UNSIGNED_8BIT rxed_byte, byte_to_decode;
  UNSIGNED_16BIT crcCalValue;
  rxed_byte=0;
  UNSIGNED_32BIT_LONG time_to_wait = MPP_WAIT_FOR_FIRST_BYTE;
  byte_to_decode = BC_BYTE1;
  crcCalValue = INIT_CRC_CCITT;
  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_STRING( "\r\n BEACON CONF DECODE: \0" );
  #endif

for(;;)
{

      if(HAL_MPP_READ_BYTE_SERIAL_PORT(&rxed_byte, time_to_wait, WHITENING_BYTE)) /* read byte from serial port within given time*/
      {

          #ifdef MPP_DEBUG
          MPP_PUT_DEBUG_BYTE(rxed_byte);
          #endif
          crcCalValue = crc_ccitt_update(crcCalValue, rxed_byte); /*up date crc*/

          switch(byte_to_decode)
          {

          case BC_BYTE1:
            /* copy to dev successful message it will be discarde if crc is not correct or timeout happens*/
              BT_DEV_SUCC_TIERING_CODE         = rxed_byte&0x0F;
              BT_DEV_SUCC_LF_BASLINE_OPER      = rxed_byte>>4;
              BT_DEV_SUCC_LF_SEC_CAP_EXT_BIT_0 = rxed_byte>>5;
              BT_DEV_SUCC_LF_SEC_CAP_EXT_BIT_1 = rxed_byte>>6;
              BT_DEV_SUCC_LF_SEC_CAP_EXT_BIT   = rxed_byte>>7;



            if(BT_DEV_SUCC_LF_SEC_CAP_EXT_BIT)/* if extension bit is set even tho this maybe set due to bit corruption it will timeout and fail*/
            {
              byte_to_decode = BC_EXT_BYTE;
            }
            else
            {

              switch(mpp_check_crc(crcCalValue)) /* else check crc and change the state accordingly*/
              {
              case MPP_PASS:
                mppState = DEV_CPB_BRDCST_TX; /* if okay go forward*/
                break;
              case MPP_FAIL:
               mppState = BEACON_BRDCST_TX; /* not okay then go back and restart the pairing*/
               #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                  BT_DEVICE_PAIRING_REJECT_CODE = MPP_ERR_CRC_FAIL;/*bad crc error*/
                  BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                  BT_DEVICE_PAIRING_TIME_OUT = 0x02;
                  mpp_send_msg_to_app(REJECT_PAIR);
               #endif
                break;
              case MPP_TIME_OUT:
                mppState = BEACON_BRDCST_TX;/* not okay then go back and restart the pairing*/
               #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
/*                  BT_DEVICE_PAIRING_REJECT_CODE = 0xff; //Time Out error
                  BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                  BT_DEVICE_PAIRING_TIME_OUT = 0x02;
                  mpp_send_msg_to_app(REJECT_PAIR); */
               #endif
                break;

              default:
                mppState = BEACON_BRDCST_TX;
                return;

              }
            return;
            }
            time_to_wait = MPP_UART_INTER_BYTE_DELAY;
            break;

          case BC_EXT_BYTE:

               BT_DEV_SUCC_LF_SEC_CAP_EXT_BYTE = rxed_byte ; /* if ext byte received*/

              switch(mpp_check_crc(crcCalValue))
              {
              case MPP_PASS:
                mppState = DEV_CPB_BRDCST_TX;/* if okay go forward*/
              break;
              case MPP_FAIL:
               mppState = BEACON_BRDCST_TX;/* not okay then go back and restart the pairing*/
               #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                  BT_DEVICE_PAIRING_REJECT_CODE = MPP_ERR_CRC_FAIL;/*bad crc error*/
                  BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                  BT_DEVICE_PAIRING_TIME_OUT = 0x02;
                  mpp_send_msg_to_app(REJECT_PAIR);
               #endif
              break;
              case MPP_TIME_OUT:
               mppState = BEACON_BRDCST_TX;/* not okay then go back and restart the pairing*/
               #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
/*                  BT_DEVICE_PAIRING_REJECT_CODE = 0xff; //Time Out error
                  BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                  BT_DEVICE_PAIRING_TIME_OUT = 0x02;
                  mpp_send_msg_to_app(REJECT_PAIR); */
               #endif
              break;

              default:
                mppState = BEACON_BRDCST_TX;
                return;

              }
            return;


          default:
            #ifdef MPP_DEBUG
            MPP_PUT_DEBUG_STRING( "\r\n DEFAULT CASE ERROR: \0" );
            #endif
            mppState = BEACON_BRDCST_TX;
            return;

        }
      }else
      {
           mppState = BEACON_BRDCST_TX; /* restart if timeout*/
           #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
/*              BT_DEVICE_PAIRING_REJECT_CODE = 0xff; //Time Out error
              BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
              BT_DEVICE_PAIRING_TIME_OUT = 0x02;
              mpp_send_msg_to_app(REJECT_PAIR); */
           #endif
           return;
      }
}

}
/* End of mpp_process_beacon_conf()   */

/*=============================================================================
	FUNCTION: mpp_process_host_cpb_rep

	DESCRIPTION: Process the MPP Capability Reply message recevived from Host.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_process_host_cpb_rep(){

UNSIGNED_16BIT crcCalValue;
UNSIGNED_8BIT rxed_byte;
UNSIGNED_32BIT_LONG time_to_wait = MPP_WAIT_FOR_FIRST_BYTE;/* wait for first byte*/
UNSIGNED_8BIT payload_length;
UNSIGNED_8BIT byte_to_decode = BYTE_1_PL_TT;
crcCalValue = INIT_CRC_CCITT;
UNSIGNED_8BIT rfcomm_entry_no, mac_byte_no, bt_link_key_byte_no;

rxed_byte = 0;

#ifdef MPP_DEBUG
MPP_PUT_DEBUG_STRING( "\r\nHOST CPB REPLY MSG DECODE: \0" );
#endif
for(;;)
{

    if(HAL_MPP_READ_BYTE_SERIAL_PORT(&rxed_byte, time_to_wait, WHITENING_BYTE))
    {

    crcCalValue = crc_ccitt_update(crcCalValue, rxed_byte);

      switch(byte_to_decode){

        case BYTE_1_PL_TT:

          payload_length = rxed_byte&0x07;
          /*check the payload length*/
          if(payload_length==0)
          {/* if zero raise error  */
              BT_DEVICE_PAIRING_REJECT_CODE = MPP_ERR_BAD_MESSAGE;
              BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();/*get time to pair*/
              mppState = MPP_FAILURE;
              return;
          }

          if(payload_length==1)
          {
            /*if 1 that means its reject code process it here*/
            BT_DEVICE_PAIRING_REJECT_CODE = (rxed_byte>>3)&0x1F;/* copy reject code */
            BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();/*get pairing time*/
            #ifdef MPP_DEBUG
            MPP_PUT_DEBUG_BYTE(rxed_byte);
            #endif

            switch(mpp_check_crc(crcCalValue))
            {
            case MPP_PASS:
              mppState = MPP_FAILURE;/* if crc pass go to mpp failure state with reject code set above*/
              break;
            case MPP_FAIL:
              BT_DEVICE_PAIRING_REJECT_CODE = MPP_ERR_BAD_MESSAGE; /* set as bad message and go to mpp failure state*/
              mppState = MPP_FAILURE;
               #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                  BT_DEVICE_PAIRING_REJECT_CODE = MPP_ERR_CRC_FAIL;/*bad crc error*/
                  BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                  BT_DEVICE_PAIRING_TIME_OUT = 0x04;
                  mpp_send_msg_to_app(REJECT_PAIR);
               #endif
              break;
            case MPP_TIME_OUT:
               mppState = BEACON_BRDCST_TX; /*  time out go back to tx broadcast state*/
               #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                  BT_DEVICE_PAIRING_REJECT_CODE = 0xff; /*Time Out error*/
                  BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                  BT_DEVICE_PAIRING_TIME_OUT = 0x04;
                  mpp_send_msg_to_app(REJECT_PAIR);
               #endif
              break;

            default:
              mppState = BEACON_BRDCST_TX;
              return;

            }
           return;


          }
          else
          {
                /* else start copy the link key cap to be sent to connection manager*/
          BT_DEV_SUCC_SSP_OOB = rxed_byte>>4;
          BT_DEV_SUCC_128_KEY_PUSH  = rxed_byte>>5;
          BT_DEV_SUCC_LINK_KEY_EXT_BIT = rxed_byte>>7;

          if(BT_DEV_SUCC_LINK_KEY_EXT_BIT)/* if ext bit set get that byte*/
          {

          byte_to_decode =BT_ENC_TYPE_BYTE2;

          }
          else
          {

          byte_to_decode =BT_ENC_TYPE;/* else go to link enc byte decode*/

          }

          }
          time_to_wait = MPP_UART_INTER_BYTE_DELAY;/* wait based on interbyte delay*/
        break;

        case BT_ENC_TYPE_BYTE2:

          BT_DEV_SUCC_LINK_KEY_EXT_BYTE = rxed_byte;
          byte_to_decode =BT_ENC_TYPE;

        break;

        case BT_ENC_TYPE:
                                              /* all datastructs need to be initialized with values at their offsets */
          BT_DEV_SUCC_LINK_ENC_TYPE_128_56  = rxed_byte; /* copy link key stuff here*/
          BT_DEV_SUCC_LINK_ENC_EXT_BIT  = rxed_byte>>3;


          if(BT_DEV_SUCC_LINK_ENC_EXT_BIT)/* if ext bit set then copy upper nibble of ext byte*/
          {
          BT_DEV_SUCC_LINK_ENC_EXT_BYTE = rxed_byte&0xF0;
          byte_to_decode = BT_ENC_TYPE_BYTE3;
          }
          else
          {/*else copy number of rfcomm entries received and audio modified indication*/
          BT_DEV_SUCC_RFCOMM_ENTRIES = rxed_byte>>5;
          BT_DEV_SUCC_AUDIO_MODIFIED = rxed_byte>>4;
            if(BT_DEV_SUCC_RFCOMM_ENTRIES)
            {/* if rfcomm entries are there then get those else get macbytes */
              rfcomm_entry_no = 0; byte_to_decode = BT_SERVICE_TYPE_STRUCT_BYTES;
            }
            else
            { mac_byte_no=0; byte_to_decode = BT_MAC_ADDRESS;
            }
          }
          break;


        case BT_ENC_TYPE_BYTE3:
          /*if link enc ext bit was set */
          BT_DEV_SUCC_LINK_ENC_EXT_BYTE = BT_DEV_SUCC_LINK_ENC_EXT_BYTE | rxed_byte&0x0F;
          BT_DEV_SUCC_RFCOMM_ENTRIES = rxed_byte>>5;
          BT_DEV_SUCC_AUDIO_MODIFIED = rxed_byte>>4; /*fixit*/

          if(BT_DEV_SUCC_RFCOMM_ENTRIES)
          {
            rfcomm_entry_no=0; byte_to_decode = BT_SERVICE_TYPE_STRUCT_BYTES;
          }
          else
          {
            mac_byte_no=0; byte_to_decode = BT_MAC_ADDRESS;
          }
          break;

        case BT_SERVICE_TYPE_STRUCT_BYTES:
        /* copy the rfcomm channels from nibbles*/
          BT_DEV_SUCC_RFCOMM(rfcomm_entry_no++) = rxed_byte&0x0F;
          BT_DEV_SUCC_RFCOMM(rfcomm_entry_no++) = rxed_byte>>4;

          if(rfcomm_entry_no>=BT_DEV_SUCC_RFCOMM_ENTRIES)
          {/*when all get copied get mac address*/
          mac_byte_no=0;
          byte_to_decode = BT_MAC_ADDRESS;
          }
          break;

        case BT_MAC_ADDRESS:
        /*considering MSB at array index 0*/
          BT_DEV_SUCC_HOST_MAC_ADDRESS(mac_byte_no++) = rxed_byte;
          if(mac_byte_no>=6)
          {
          bt_link_key_byte_no=0;
          byte_to_decode = BT_LINK_KEY;
          }
        break;


        case BT_LINK_KEY:
        /*considering MSB at array index 0*/
          BT_DEV_SUCC_LINK_KEY(bt_link_key_byte_no++) = rxed_byte;


          if(bt_link_key_byte_no>=16)
          {/* when link key is copied*/
            #ifdef MPP_DEBUG
            MPP_PUT_DEBUG_BYTE(rxed_byte);
            #endif

            switch(mpp_check_crc(crcCalValue))
            {
            case MPP_PASS:
              /* if success ful send ack to host*/

              mppState = ACK_TO_HOST;
              break;
            case MPP_FAIL:
              BT_DEVICE_PAIRING_REJECT_CODE = MPP_ERR_BAD_MESSAGE; /*bad crc error*/
              BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
              mppState = MPP_FAILURE;

               #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                  BT_DEVICE_PAIRING_REJECT_CODE = MPP_ERR_CRC_FAIL;/*bad crc error*/
                  BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                  BT_DEVICE_PAIRING_TIME_OUT = 0x04;
                  mpp_send_msg_to_app(REJECT_PAIR);
               #endif
              break;
            case MPP_TIME_OUT:
               mppState = BEACON_BRDCST_TX;
               #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                  BT_DEVICE_PAIRING_REJECT_CODE = 0xff; /*Time Out error*/
                  BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                  BT_DEVICE_PAIRING_TIME_OUT = 0x04;
                  mpp_send_msg_to_app(REJECT_PAIR);
               #endif
              break;

            default:
              mppState = BEACON_BRDCST_TX;
              return;

            }
           return;
          }
        break;


        default:
            #ifdef MPP_DEBUG
            MPP_PUT_DEBUG_STRING( "\r\n DEFAULT CASE ERROR: \0" );
            #endif
            mppState = BEACON_BRDCST_TX;
            return;

        }
        #ifdef MPP_DEBUG
        MPP_PUT_DEBUG_BYTE(rxed_byte);
        #endif

    }
    else
    {
    mppState = BEACON_BRDCST_TX;
     #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
        BT_DEVICE_PAIRING_REJECT_CODE = 0xff; /*Time Out error*/
        BT_DEVICE_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
        BT_DEVICE_PAIRING_TIME_OUT = 0x04;
        mpp_send_msg_to_app(REJECT_PAIR);
     #endif
    return; /*timeout error*/
    }
}

}
/* End of mpp_process_host_cpb_rep()   */

#endif

#ifdef MPP_HOST
/**                                                                          **/

/*                     HOST MESSAGES START FROM HERE                          */

/**                                                                          **/
/*=============================================================================
	FUNCTION: mpp_search_mac_from_list

	DESCRIPTION:

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE: success or fail
==============================================================================*/
UNSIGNED_8BIT mpp_search_mac_from_list(bt_mac_addr_lookup_node *head, bt_mac_address *ptr2MACstruct)
{

  bt_mac_addr_lookup_node *current;
  UNSIGNED_8BIT count, found=0;
  if(head==NULL)
  {
    return found;
  }
  current = head;

  while(current->next!=NULL)
  {
    for(count =0;count<MPP_BT_MAC_BYTES;count++)
    {
      if(current->device_mac.bt_mac_byte[count] == ptr2MACstruct->bt_mac_byte[count])
      {
        found =1;
      }
      else
       {
        found =0;
        break;
      }
    }
    current = current->next;
  }


  for(count =0;count<MPP_BT_MAC_BYTES;count++)
  {
   if(current->device_mac.bt_mac_byte[count] == ptr2MACstruct->bt_mac_byte[count])
    {
      found =1;
    }
    else
    {
      found =0;
      break;
    }
  }

  return found;
}
/* End of mpp_search_mac_from_list()   */
/*=============================================================================
	FUNCTION: mpp_send_beacon_conf

	DESCRIPTION: Send the MPP beacon confirmation message (Device side).

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_send_beacon_conf()
{

   UNSIGNED_8BIT mppDataOut;
   UNSIGNED_16BIT crcCalValue;
   mpp_pkt_byte byte_to_add;

  byte_to_add = BC_BYTE1;
  crcCalValue = INIT_CRC_CCITT;
  mppDataOut = 0;

  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_STRING( "\r\n SEND BEACON CONF: \0" );
  #endif
#ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
  valid_beacon_proc_total++;
#endif
  for(;;){

    switch(byte_to_add){

    case BC_BYTE1:
      /* form beacon confirmation byte*/
      mppDataOut = BT_HOST_TIERING_CODE & 0x0F;
      mppDataOut = mppDataOut | (BT_HOST_LF_BASLINE_OPER<<4);
      mppDataOut = mppDataOut | (BT_HOST_LF_SEC_CAP_EXT_BIT_0<<5);
      mppDataOut = mppDataOut | (BT_HOST_LF_SEC_CAP_EXT_BIT_1<<6);
      mppDataOut = mppDataOut | (BT_HOST_LF_SEC_CAP_EXT_BIT<<7);

      if(BT_HOST_LF_SEC_CAP_EXT_BIT)
      {
        byte_to_add = BC_EXT_BYTE;
      }
      else
      {
        byte_to_add = BYTE_CRC;
      }
      break;

    case BC_EXT_BYTE:
      /* copy extension byte*/
      mppDataOut = BT_HOST_LF_SEC_CAP_EXT_BYTE;
      byte_to_add = BYTE_CRC;
     break;

    case BYTE_CRC:
      /*calulate crc and send*/
      mpp_send_crc(crcCalValue);
      mpp_trigger_tx();   /* trigger the tx kept here for different implementaions like for DMA based transmission and queue based tx handling, in this implementaion its serving the purpose to insure that last byte is completely tx on the physical line so that we can disbale tx clock*/
      return;


    default:
        #ifdef MPP_DEBUG
        MPP_PUT_DEBUG_STRING( "\r\n DEFAULT CASE ERROR: \0" );
        #endif
        mpp_send_crc(crcCalValue);
        mpp_trigger_tx();
        return;
    }
    crcCalValue = crc_ccitt_update(crcCalValue, mppDataOut);/* update crc*/
    HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, WHITENING_BYTE);
    #ifdef MPP_DEBUG
    MPP_PUT_DEBUG_BYTE(mppDataOut);
    #endif
  }

}
/* End of mpp_send_beacon_conf()   */

/*=============================================================================
	FUNCTION: mpp_send_host_reject_msg

	DESCRIPTION:

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_send_host_reject_msg()
{

  UNSIGNED_16BIT crcCalValue;
  UNSIGNED_8BIT mppDataOut;
  mpp_pkt_byte byte_to_add;

  byte_to_add = BYTE_1_PL_TT;
  crcCalValue = INIT_CRC_CCITT;
  mppDataOut = 0;

  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_STRING( "\r\nHOST REJECT REPLY: \0" );
  #endif
#ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
  valid_data_msg_proc_total++;
#endif
  for(;;)
  {
    switch(byte_to_add)
    {

      case BYTE_1_PL_TT:
        mppDataOut = (BT_HOST_PAIRING_REJECT_CODE<<3)|0x01; /*set 1 for payload and copy the reject code*/
        byte_to_add = BYTE_CRC;
      break;

      case BYTE_CRC:
        mpp_send_crc(crcCalValue);/* calculate the crc*/
        mpp_trigger_tx();
        return;

      default:
            #ifdef MPP_DEBUG
            MPP_PUT_DEBUG_STRING( "\r\n DEFAULT CASE ERROR: \0" );
            #endif
            mpp_send_crc(crcCalValue);
            mpp_trigger_tx();
            return;
    }

    crcCalValue = crc_ccitt_update(crcCalValue, mppDataOut);/*update the crc*/
    HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, WHITENING_BYTE);
    #ifdef MPP_DEBUG
    MPP_PUT_DEBUG_BYTE(mppDataOut);
    #endif
    }

}
/* End of mpp_send_host_reject_msg()   */


/*=============================================================================
	FUNCTION: mpp_send_host_cpb_rep

	DESCRIPTION: Send reply to the MPP Capability Broadcast message received from Host.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/

void mpp_send_host_cpb_rep()
{
UNSIGNED_16BIT crcCalValue;
UNSIGNED_8BIT mppDataOut;
UNSIGNED_8BIT total_bytes_to_send, rfcomm_entry_no;
UNSIGNED_8BIT mac_byte_no, bt_link_key_byte_no;
mpp_pkt_byte byte_to_add;
byte_to_add = BYTE_1_PL_TT;
crcCalValue = INIT_CRC_CCITT;
mppDataOut = 0;

#ifdef MPP_DEBUG
MPP_PUT_DEBUG_STRING( "\r\nHOST CPB REPLY: \0" );
#endif
#ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
  valid_data_msg_proc_total++;
#endif
for(;;)
{

  switch(byte_to_add)
  {

    case BYTE_1_PL_TT:
  /*calulate the number of bytes to send based on ext bits and rfcomm entries*/
      total_bytes_to_send= MIN_BYTES_IN_DEV_CPB_REP + BT_HOST_LINK_KEY_EXT_BIT + BT_HOST_LINK_ENC_EXT_BIT + (UNSIGNED_8BIT)((BT_HOST_SUCC_RFCOMM_INDEX_ENTRIES+1)/2); /*leftover conversion between struct size and bytes to send*/
    /*fill payload field according to MPP ICD*/
      if(total_bytes_to_send<=15)
      {/* fix this */
      mppDataOut =PAYLOAD_15B_OR_LESS_VALUE_2;
      }
      else if(total_bytes_to_send<=22)
      {
      mppDataOut = PAYLOAD_22B_OR_LESS_VALUE_3;
      }
      else if(total_bytes_to_send<=31)
      {
      mppDataOut = PAYLOAD_31B_OR_LESS_VALUE_4;
      }
      else if(total_bytes_to_send<=41)
      {
      mppDataOut = PAYLOAD_41B_OR_LESS_VALUE_5;
      }
      /*put link key cap */
      mppDataOut = mppDataOut|(((BT_HOST_SSP_OOB<<4)|(BT_HOST_128_KEY_PUSH<<5)|(BT_HOST_LINK_KEY_EXT_BIT<<7))&0xF0);


      if(BT_HOST_LINK_KEY_EXT_BIT)
      {
      byte_to_add =BT_ENC_TYPE_BYTE2;/*if ext bit set*/
      }
      else
      {
      byte_to_add = BT_ENC_TYPE;
      }


    break;

    case BT_ENC_TYPE_BYTE2:/* add ext byte*/
      mppDataOut = BT_HOST_LINK_KEY_EXT_BYTE;
      byte_to_add =BT_ENC_TYPE;
    break;



    case BT_ENC_TYPE:
                              /* all datastructs need to be initialized with values at their offsets */
      /*copy link enc capability*/
      mppDataOut = ((BT_HOST_LINK_ENC_TYPE_128_56)|(BT_HOST_LINK_ENC_EXT_BIT<<3));

      if(BT_HOST_LINK_ENC_EXT_BIT)/*if ext bit set copy the ext byte*/
      {
      mppDataOut = mppDataOut |(BT_HOST_LINK_ENC_EXT_BYTE&0xF0);
      byte_to_add = BT_ENC_TYPE_BYTE3;
      }
      else
      {/* copy rfcomm entries and bt host audio modified indication*/
        mppDataOut = mppDataOut | (BT_HOST_SUCC_RFCOMM_INDEX_ENTRIES <<5)|(BT_HOST_AUDIO_MODIFIED<<4);
        if(BT_HOST_SUCC_RFCOMM_INDEX_ENTRIES)
        {
          rfcomm_entry_no=0;byte_to_add = BT_SERVICE_TYPE_STRUCT_BYTES;
        }
        else
        {  mac_byte_no=0;byte_to_add = BT_MAC_ADDRESS; /* if no entries put mac*/
        }
      }
    break;

    case BT_ENC_TYPE_BYTE3:
    /* if ext bit set add lower nibble and rfcomm entries*/
      mppDataOut = mppDataOut |(BT_HOST_LINK_ENC_EXT_BYTE&0x0F);
      mppDataOut = mppDataOut |(BT_HOST_SUCC_RFCOMM_INDEX_ENTRIES<<5)|(BT_HOST_AUDIO_MODIFIED<<4);
      if(BT_HOST_SUCC_RFCOMM_INDEX_ENTRIES)
      {
        rfcomm_entry_no=0; byte_to_add = BT_SERVICE_TYPE_STRUCT_BYTES;
      }
      else
      {
        mac_byte_no=0; byte_to_add = BT_MAC_ADDRESS;
      }

    break;


    case BT_SERVICE_TYPE_STRUCT_BYTES:
    /*copy the rfcomm channel numbers from the DB index*/
      mppDataOut = (BT_HOST_DB_RFCOMM_LOOKUP_CH_NUM(BT_HOST_SUCC_RFCOMM_INDEX_TO_CONN_DB(rfcomm_entry_no+1))<<4)|BT_HOST_DB_RFCOMM_LOOKUP_CH_NUM(BT_HOST_SUCC_RFCOMM_INDEX_TO_CONN_DB(rfcomm_entry_no));
      rfcomm_entry_no =  rfcomm_entry_no+2;

      if(rfcomm_entry_no>=BT_HOST_SUCC_RFCOMM_INDEX_ENTRIES)
      {
      mac_byte_no=0;
      byte_to_add = BT_MAC_ADDRESS;
      }
    break;


    case BT_MAC_ADDRESS:
      /*considering MSB at array index 0*/
      mppDataOut = BT_HOST_MAC_ADDRESS(mac_byte_no++);
      if(mac_byte_no>=6)
      {
      bt_link_key_byte_no=0;
      byte_to_add = BT_LINK_KEY;
      }
    break;


    case BT_LINK_KEY:
    /*considering MSB at array index 0*/

      mppDataOut =  BT_HOST_SUCC_LINK_KEY(bt_link_key_byte_no++);
      if(bt_link_key_byte_no>=16)
      {
      byte_to_add = BYTE_CRC;
      }
    break;

    case BYTE_CRC:
      mpp_send_crc(crcCalValue);
      mpp_trigger_tx();
      return;



    default:
          #ifdef MPP_DEBUG
          MPP_PUT_DEBUG_STRING( "\r\n DEFAULT CASE ERROR: \0" );
          #endif
          mpp_send_crc(crcCalValue);
          mpp_trigger_tx();
          return;

  }

  crcCalValue = crc_ccitt_update(crcCalValue, mppDataOut);
  HAL_MPP_WRITE_BYTE_SERIAL_PORT(&mppDataOut, WHITENING_BYTE);
  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_BYTE(mppDataOut);
  #endif
}

}
/* End of mpp_send_host_cpb_rep()   */


/*=============================================================================
	FUNCTION: mpp_process_beacon_brdcst

	DESCRIPTION: Process the MPP beacon Broadcast message (Host side).

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_process_beacon_brdcst()
{
  UNSIGNED_8BIT byte_to_decode;
  UNSIGNED_8BIT rxed_byte;
  UNSIGNED_16BIT crcCalValue;
  UNSIGNED_32BIT_LONG time_to_wait = MPP_WAIT_FOR_BEACON_BYTE;
  crcCalValue = INIT_CRC_CCITT;
  rxed_byte=0;
  byte_to_decode = BB_BYTE1;
  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_STRING( "\r\n BEACON BRDCAST DECODE: \0" );
  #endif

for(;;)
{

      if(HAL_MPP_READ_BYTE_SERIAL_PORT(&rxed_byte, time_to_wait, WHITENING_BYTE))
      {

          #ifdef MPP_DEBUG
          MPP_PUT_DEBUG_BYTE(rxed_byte);
          #endif
          crcCalValue = crc_ccitt_update(crcCalValue, rxed_byte);

          switch(byte_to_decode)
          {

              case BB_BYTE1:
                /* copy beacon broadcast*/
                beacon_brdcast.tiering_code = rxed_byte&0x0F;
                beacon_brdcast.lf_baseline_operation = rxed_byte>>4;
                beacon_brdcast.lf_security_capability.security_extensions_bit_0 = rxed_byte>>5;
                beacon_brdcast.lf_security_capability.security_extensions_bit_1 = rxed_byte>>6;
                beacon_brdcast.lf_security_capability.extension_bit = rxed_byte>>7;

                if(beacon_brdcast.lf_security_capability.extension_bit)/* if ext bit is set get another byte*/
                {
                  byte_to_decode = BB_EXT_BYTE;
                }
                else
                {

                  switch(mpp_check_crc(crcCalValue))
                  {
                  case MPP_PASS:
                    #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                    if (run_once == 0)
                    {
                      start_tick_ms = xTaskGetTickCount();
                      run_once = 1;
                    }
                    if(host_cpb_rep_msg_txed == MPP_PASS)
                    {
                      BT_HOST_PAIRING_REJECT_CODE = 0xFF; //Time Out error
                      BT_HOST_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                      BT_HOST_PAIRING_TIME_OUT = 0x05;
                      mpp_fail_tick_ms = xTaskGetTickCount();
                      if(valid_beacon_proc_total > 0)
                        BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = (mpp_fail_tick_ms - start_tick_ms);
                      else
                        BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = 0;
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
                      mpp_send_msg_to_app(REJECT_PAIR);
                    }
                    #endif

                    host_cpb_rep_msg_txed = MPP_FAIL;
                    mpp_start_pairing_timer();

                    mppState = BEACON_CONF_TX;

                    break;

                  case MPP_FAIL:
                    #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                      BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_CRC_FAIL; /*Time Out error*/
                      BT_HOST_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                      BT_HOST_PAIRING_TIME_OUT = 0x01;
                      mpp_fail_tick_ms = xTaskGetTickCount();
                      if(valid_beacon_proc_total > 0)
                        BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = (mpp_fail_tick_ms - start_tick_ms);
                      else
                        BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = 0;
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
                      mpp_send_msg_to_app(REJECT_PAIR);
                    #endif
                    break;

                  case MPP_TIME_OUT:
                    if((host_cpb_rep_msg_txed == MPP_PASS) && (rxed_byte == MPP_ACK_TO_HOST_BYTE))
                    {
                        host_cpb_rep_msg_txed = MPP_FAIL;
                        BT_HOST_SUCC_PAIRING_TIME = mpp_get_pairing_time();
                        #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                          end_tick_ms = xTaskGetTickCount();
                          BT_HOST_SUCC_CUMULATIVE_PAIRING_TIME = (end_tick_ms - start_tick_ms);
                          BT_HOST_SUCC_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
                          BT_HOST_SUCC_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
                          valid_beacon_proc_total = 0;
                          valid_data_msg_proc_total = 0;
                          run_once = 0;
                        #endif
                        mpp_send_msg_to_app(SUCC_PAIR); /* send success message to app*/

                        if(BT_HOST_CONTINOUS_PAIRING)
                        {
                        mppState = BEACON_BRDCST_RX;
                        }
                        else
                        {
                        mppState = MPP_STANDBY_COMPL;
                        }
                    }

                    break;

                  default:
                    return;


                  }
                return;
                }
                time_to_wait = MPP_UART_INTER_BYTE_DELAY;/* wait on inter byte delay*/
                break;

              case BB_EXT_BYTE:
              /* copy extension byte*/
                  beacon_brdcast.extension_byte = rxed_byte;

                  switch(mpp_check_crc(crcCalValue))
                  {
                  case MPP_PASS:

                    mppState = BEACON_CONF_TX;
                    #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                    if (run_once == 0)
                    {
                      start_tick_ms = xTaskGetTickCount();
                      run_once = 1;
                    }
                    if(host_cpb_rep_msg_txed == MPP_PASS)
                    {
                      BT_HOST_PAIRING_REJECT_CODE = 0xFF; //Time Out error
                      BT_HOST_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                      BT_HOST_PAIRING_TIME_OUT = 0x05;
                      mpp_fail_tick_ms = xTaskGetTickCount();
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = (mpp_fail_tick_ms - start_tick_ms);
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
                      mpp_send_msg_to_app(REJECT_PAIR);
                    }
                    #endif
                  break;

                  case MPP_FAIL:
                    host_cpb_rep_msg_txed = MPP_FAIL;
                    #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                      BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_CRC_FAIL; /*Time Out error*/
                      BT_HOST_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                      BT_HOST_PAIRING_TIME_OUT = 0x01;
                      mpp_fail_tick_ms = xTaskGetTickCount();
                      if(valid_beacon_proc_total > 0)
                        BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = (mpp_fail_tick_ms - start_tick_ms);
                      else
                        BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = 0;
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
                      mpp_send_msg_to_app(REJECT_PAIR);
                    #endif
                  break;

                  case MPP_TIME_OUT:
                    host_cpb_rep_msg_txed = MPP_FAIL;
                    #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                      BT_HOST_PAIRING_REJECT_CODE = 0xFF; //Time Out error
                      BT_HOST_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                      BT_HOST_PAIRING_TIME_OUT = 0x01;
                      mpp_fail_tick_ms = xTaskGetTickCount();
                      if(valid_beacon_proc_total > 0)
                        BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = (mpp_fail_tick_ms - start_tick_ms);
                      else
                        BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = 0;
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
                      BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
/*                      mpp_send_msg_to_app(REJECT_PAIR); */
                    #endif
                   break;

                  default:
                    return;


                  }
                return;



              default:
                #ifdef MPP_DEBUG
                MPP_PUT_DEBUG_STRING( "\r\n DEFAULT CASE ERROR: \0" );
                #endif

              break;

           }
      }else
      {
          if((host_cpb_rep_msg_txed == MPP_PASS))
          {
              host_cpb_rep_msg_txed = MPP_FAIL;
              BT_HOST_SUCC_PAIRING_TIME = mpp_get_pairing_time();
              #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                end_tick_ms = xTaskGetTickCount();
                BT_HOST_SUCC_CUMULATIVE_PAIRING_TIME = (end_tick_ms - start_tick_ms);
                BT_HOST_SUCC_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
                BT_HOST_SUCC_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
                valid_beacon_proc_total = 0;
                valid_data_msg_proc_total = 0;
                run_once = 0;
              #endif
              mpp_send_msg_to_app(SUCC_PAIR); /* send success message to app*/

              if(BT_HOST_CONTINOUS_PAIRING)
              {
              mppState = BEACON_BRDCST_RX;
              }
              else
              {
              mppState = MPP_STANDBY_COMPL;
              }
          }
          #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
          else
          {
            BT_HOST_PAIRING_REJECT_CODE = 0xFF; //Time Out error
            BT_HOST_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
            BT_HOST_PAIRING_TIME_OUT = 0x01;
            mpp_fail_tick_ms = xTaskGetTickCount();
            if(valid_beacon_proc_total > 0)
              BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = (mpp_fail_tick_ms - start_tick_ms);
            else
              BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = 0;
            BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
            BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
/*            mpp_send_msg_to_app(REJECT_PAIR); */
          }
          #endif
            return;
      }
}


}
/* End of mpp_process_beacon_brdcst()   */

/*=============================================================================
	FUNCTION: mpp_process_dev_cpb_brdcst

	DESCRIPTION: Process the MPP Capability Broadcast message (Host side).

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/

void mpp_process_dev_cpb_brdcst()
{
  UNSIGNED_8BIT rxed_byte;
  UNSIGNED_32BIT_LONG time_to_wait = MPP_WAIT_FOR_FIRST_BYTE;
  UNSIGNED_8BIT payload_length,byte_to_decode,tt_type_bt_decode_byte,mac_byte_no,service_entry_no,count,ser_type,rf_no;
  UNSIGNED_16BIT crcCalValue;

  bt_link_key_exchange_method bt_link_key_exchange_method;
  bt_link_enc                 bt_link_enc;

  crcCalValue = INIT_CRC_CCITT;
  rxed_byte = 0;
  byte_to_decode=BYTE_1_PL_TT;

  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_STRING( "\r\n DEV CPB MSG DECODE: \0" );
  #endif

for(;;)
{

  if(HAL_MPP_READ_BYTE_SERIAL_PORT(&rxed_byte, time_to_wait, WHITENING_BYTE))
  {
  #ifdef MPP_DEBUG
  MPP_PUT_DEBUG_BYTE(rxed_byte);
  #endif
  crcCalValue = crc_ccitt_update(crcCalValue, rxed_byte);

  switch(byte_to_decode)
  {
    case BYTE_1_PL_TT:    /*get transaction type*/
      BT_HOST_SUCC_TRANS_TYPE = rxed_byte&0x1F;
      payload_length = rxed_byte>>5; /* copy payload which is not used*/
      payload_length = payload_length;  /* To Resolve compiler warning */

      byte_to_decode=BT_HOST_SUCC_TRANS_TYPE;
      if (BT_HOST_SUCC_TRANS_TYPE == BYTE_2_FOR_TT_TYPE_BLUETOOTH)/* if trans is BT type which is only one supported in release 1*/
      {
        tt_type_bt_decode_byte=BT_ENC_TYPE;
      }

      time_to_wait = MPP_UART_INTER_BYTE_DELAY;/* set interbyte delay*/
      break;

    case BYTE_2_FOR_TT_TYPE_BLUETOOTH:
  /* start decoding BT trans type payload*/
          switch(tt_type_bt_decode_byte)
          {
            case BT_ENC_TYPE:
        /* copy the link key exc cap*/
              bt_link_key_exchange_method.bt_ssp_oob = rxed_byte;
              bt_link_key_exchange_method.lf_bt_128_key_push = rxed_byte>>1;
              bt_link_key_exchange_method.bt_link_key_exchange_method_ext_bit = rxed_byte>>3;

              if(bt_link_key_exchange_method.bt_link_key_exchange_method_ext_bit)
              {  /*if ext bit is set copy nibble*/
                bt_link_key_exchange_method.extension_byte = rxed_byte&0xF0;
                tt_type_bt_decode_byte = BT_ENC_TYPE_BYTE2;
              }
              else
              {  /* else copy enc type*/
                bt_link_enc.e0_enc_type_128_56 =  rxed_byte>>4;
                bt_link_enc.bt_link_enc_ext_bit = rxed_byte>>7;
                if(bt_link_enc.bt_link_enc_ext_bit)
                {
                  tt_type_bt_decode_byte = BT_ENC_TYPE_BYTE2;
                }
                else
                {
                  tt_type_bt_decode_byte = BT_DEV_TYPE_AND_SERVICE_TYPE;
                }
              }
              break;

            case BT_ENC_TYPE_BYTE2:

              if(bt_link_key_exchange_method.bt_link_key_exchange_method_ext_bit)
              {  /* if ext bit was set copy rest of the byte and then copy rest of enc type*/
                bt_link_key_exchange_method.extension_byte = bt_link_key_exchange_method.extension_byte | rxed_byte&0xF;
                bt_link_enc.e0_enc_type_128_56  = rxed_byte>>4;
                bt_link_enc.bt_link_enc_ext_bit = rxed_byte>>7;

                if(bt_link_enc.bt_link_enc_ext_bit)
                {
                  tt_type_bt_decode_byte = BT_ENC_TYPE_BYTE3;
                }
                else
                {
                  tt_type_bt_decode_byte = BT_DEV_TYPE_AND_SERVICE_TYPE;
                }

              }
              else if(bt_link_enc.bt_link_enc_ext_bit)
              {/* if enc ex bit was set copy ext byte*/
                bt_link_enc.extension_byte = rxed_byte;
                tt_type_bt_decode_byte = BT_DEV_TYPE_AND_SERVICE_TYPE;
              }
              break;

            case BT_ENC_TYPE_BYTE3:

              bt_link_enc.extension_byte = rxed_byte;
              tt_type_bt_decode_byte = BT_DEV_TYPE_AND_SERVICE_TYPE;
              break;

            case BT_DEV_TYPE_AND_SERVICE_TYPE:
        /* copy device type*/
              BT_HOST_SUCC_DEVICE_TYPE = rxed_byte&0x03;
              BT_HOST_SUCC_DEVICE_RFPA = (rxed_byte>>2)&0x01;
              /* see how many services are requested */
              BT_HOST_PAIRING_SERVICES_REQUESTED = rxed_byte>>5;
              if(!BT_HOST_PAIRING_SERVICES_REQUESTED)
              {
                mac_byte_no=0; tt_type_bt_decode_byte = BT_MAC_ADDRESS;
              }
              else
              {
                service_entry_no=0; tt_type_bt_decode_byte = BT_SERVICE_TYPE_STRUCT_BYTES;
              }
              break;

            case BT_SERVICE_TYPE_STRUCT_BYTES:
        /*copy services to host reject message strut that is beig reused over here*/

               BT_HOST_PAIRING_SERVICE_TYPE_NIBBLER(service_entry_no)= rxed_byte;
               BT_HOST_PAIRING_SERVICE_TYPE_PROFILE(service_entry_no++)= rxed_byte>>1;


             if(service_entry_no>=BT_HOST_PAIRING_SERVICES_REQUESTED)/*if all copied then het mac address*/
              {
                mac_byte_no=0;
                tt_type_bt_decode_byte = BT_MAC_ADDRESS;
              }
              break;

            case BT_MAC_ADDRESS:
                  /*considering MSB at array index 0*/

                  BT_HOST_SUCC_DEVICE_MAC(mac_byte_no++) = rxed_byte;

                  if(mac_byte_no>=6)
                  {  /* when mac is copied check crc*/

                      switch(mpp_check_crc(crcCalValue))
                      {
                      case MPP_PASS:

                                    /* process start */
                        /* if host supports SSP OOB see if device has requested that if yes then reject the request*/
                        if(!BT_HOST_SSP_OOB)
                        {
                          if(bt_link_key_exchange_method.bt_ssp_oob)
                          {
                             BT_HOST_PAIRING_REJECT_CODE= BT_SSP_OOB_NOT_SUPPORT;
                            mppState = MPP_FAILURE;
                            return;
                          }
                        }
                        /* if host doest suport 128 it link enc and device request that reject the request*/
                        if(!BT_HOST_LINK_ENC_TYPE_128_56)
                        {
                          if(bt_link_enc.e0_enc_type_128_56)
                          {
                             BT_HOST_PAIRING_REJECT_CODE= BT_LINK_SEC_FAIL;
                            mppState = MPP_FAILURE;
                            return;
                          }
                        }
                        /* if host doesnt support link key push and device request that send reject message back*/
                        if((BT_HOST_128_KEY_PUSH == 0)&&(bt_link_key_exchange_method.lf_bt_128_key_push ==1))
                        {
                           BT_HOST_PAIRING_REJECT_CODE = BT_LINK_SEC_FAIL;
                          mppState = MPP_FAILURE;
                          return;
                        }
            /* if link key extension are not supported send reject message*/
                        if(BT_HOST_LINK_KEY_EXT_BIT != bt_link_key_exchange_method.bt_link_key_exchange_method_ext_bit)
                        {
                           BT_HOST_PAIRING_REJECT_CODE = BT_LINK_SEC_FAIL;
                          mppState = MPP_FAILURE;
                          return;
                        }
             /* if link enc extension are not supported send reject message*/
                        if(BT_HOST_LINK_ENC_EXT_BIT != bt_link_enc.bt_link_enc_ext_bit)
                        {
                           BT_HOST_PAIRING_REJECT_CODE = BT_LINK_SEC_FAIL;
                           BT_HOST_PAIRING_REJECT_TIME_MS =mpp_get_pairing_time();
                          mppState = MPP_FAILURE;
                          return;
                        }

                        if( MPP_CONNECTION_DB_MUTEX_HANDLE != NULL )/* check if mutex is craeted*/
                        {
                           if(MPP_OS_SemaphoreTake( MPP_CONNECTION_DB_MUTEX, DB_MUTEX_WAIT_TIME) == MPP_SEM_TAKEN)/* take the mutex in 50 msec so that host cant access the database at the same time*/
                           {
                                mppState = BEACON_BRDCST_RX;/* just reused to avoid calling semaphore give in multiple if statements*/

                                   /* check linked list here for the BT MAC address blocking if mac is in list reject the request*/
                                if(mpp_search_mac_from_list(BT_HOST_DB_PTR_TO_MAC_LIST_HEAD,&BT_HOST_SUCC_DEVICE_MAC_STRUCT))
                                {

                                	BT_HOST_PAIRING_REJECT_CODE = BT_DEVICE_RECORD_EXISTS;
                                    mppState = MPP_FAILURE;


                                }else if(BT_HOST_DB_SYSTEM_STATUS)/*if system status is bad reject request*/
                                {
                                  BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_NO_RADIO_SERVICE;
                                  mppState = MPP_FAILURE;

                                }else if((BT_HOST_SUCC_DEVICE_TYPE == AUDIO_DEVICE)&&(!BT_HOST_DB_AUDIO_CONN_AVAIL))/*if no audio connection available*/
                                {
                                  BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_NO_AUDIO_LINK;
                                  mppState = MPP_FAILURE;
                                }else if((BT_HOST_SUCC_DEVICE_TYPE == NON_AUDIO_DEVICE)&&(!BT_HOST_DB_DATA_CONN_AVAIL))/*id no data connection is available*/
                                {
                                   BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_NO_DATA_LINK;
                                   mppState = MPP_FAILURE;

                                }

                                if(mppState == MPP_FAILURE)
                                {/*if failed release the mutex and get pairing time*/
                                   BT_HOST_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                                  MPP_OS_SemaphoreGive( MPP_CONNECTION_DB_MUTEX );
                                  return;
                                }

                                 BT_HOST_SUCC_RFCOMM_INDEX_ENTRIES = 0; /*init index to zero*/

                                 for(rf_no = 0;rf_no <NUM_MAX_MPP_RFCOMM_CH;rf_no++)/*lookup all rfcomm entries*/
                                 {
                                   rfcomm_taken[rf_no]=RFCOMM_AVAIL; /*setting this track array to zero in first run*/
                                 }

                                 for(ser_type = 0; ser_type <BT_HOST_PAIRING_SERVICES_REQUESTED;ser_type++ )/* while all service requests are satisfied*/
                                 {

                                    for(rf_no = 0;rf_no <BT_HOST_DB_RFCOMM_ALLOC;rf_no++)/*lookup all rfcomm entries*/
                                    {

                                      if(rfcomm_taken[rf_no]==RFCOMM_TAKEN)/*used a local array to keep track which services are taken so far in this loop as we cant write into DB*/
                                      {
                                      continue;
                                      }
                                        if(BT_HOST_DB_RFCOMM_LOOKUP_RFCOMM_STATUS(rf_no)  == RFCOMM_AVAIL)/* if RFCOMM is available*/
                                        {
                                          if(BT_HOST_SUCC_DEVICE_TYPE == BT_HOST_DB_RFCOMM_LOOKUP_DEVICE_TYPE(rf_no))/*if device type matches*/
                                          {/* if nibbler cap and profile match is found in DB*/
                                            if(( BT_HOST_PAIRING_SERVICE_TYPE_NIBBLER(ser_type)==BT_HOST_DB_RFCOMM_LOOKUP_BT_SERVICE_NIBBLER(rf_no))&&(BT_HOST_PAIRING_SERVICE_TYPE_PROFILE(ser_type)==BT_HOST_DB_RFCOMM_LOOKUP_BT_SERVICE_PROFILE(rf_no)))
                                            {
                                              rfcomm_taken[rf_no]=RFCOMM_TAKEN;/* mark as rfcomm taken*/
                                              BT_HOST_SUCC_RFCOMM_INDEX_TO_CONN_DB(BT_HOST_SUCC_RFCOMM_INDEX_ENTRIES++)=rf_no;/* copy the index for later retrival of rfcomm number*/
                                              break;
                                            }
                                          }
                                        }
                                     }

                                  }
                                MPP_OS_SemaphoreGive( MPP_CONNECTION_DB_MUTEX );/*release semaphore*/
                           }else
                           {
                                  BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_FAILURE_GEN;/* unable to obtain semaphore send reject message*/
                                  BT_HOST_PAIRING_REJECT_TIME_MS =mpp_get_pairing_time();
                                  mppState = MPP_FAILURE;
                                  return;
                           }
                        }else
                        {/*if semaphore not created send fail*/
                                  BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_FAILURE_GEN;
                                  BT_HOST_PAIRING_REJECT_TIME_MS =mpp_get_pairing_time();
                                  mppState = MPP_FAILURE;
                                  return;
                        }

                        if(BT_HOST_SUCC_RFCOMM_INDEX_ENTRIES==BT_HOST_PAIRING_SERVICES_REQUESTED)
                        {
                          /* if number of requests are satisfied from DB send back cap reply*/
                          for(count=0;count<MPP_ENC_LINK_KEY_SIZE;count++)
                          {/*copy link key from the DB*/
                            BT_HOST_SUCC_LINK_KEY(count) = BT_HOST_DB_LINK_KEY_COPY_FROM(count);
                          }
                          mppState = HOST_CPB_REP_TX;/*transmit cap reply to device*/

                        }
                        else
                        {
                           BT_HOST_PAIRING_REJECT_CODE = BT_SERVICE_FAIL;/* if all services not satisfied reject erquest*/
                            BT_HOST_PAIRING_REJECT_TIME_MS =mpp_get_pairing_time();
                            mppState = MPP_FAILURE;


                         }
                        break;
                      case MPP_FAIL:/* crc fail*/
                          BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_BAD_MESSAGE;
                          BT_HOST_PAIRING_REJECT_TIME_MS =mpp_get_pairing_time();
                          mppState = MPP_FAILURE;
                          #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                            BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_CRC_FAIL; /*Time Out error*/
                            BT_HOST_PAIRING_TIME_OUT = 0x03;
                            mpp_fail_tick_ms = xTaskGetTickCount();
                            if(valid_beacon_proc_total > 0)
                              BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = (mpp_fail_tick_ms - start_tick_ms);
                            else
                              BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = 0;
                            BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
                            BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
                         //   mpp_send_msg_to_app(REJECT_PAIR);
                          #endif
                        break;
                      case MPP_TIME_OUT:/*timeout*/
                        mppState = BEACON_BRDCST_RX;
                        #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
                          BT_HOST_PAIRING_REJECT_CODE = 0xFF; /*Time Out error*/
                          BT_HOST_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
                          BT_HOST_PAIRING_TIME_OUT = 0x03;
                          mpp_fail_tick_ms = xTaskGetTickCount();
                          if(valid_beacon_proc_total > 0)
                            BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = (mpp_fail_tick_ms - start_tick_ms);
                          else
                            BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = 0;
                          BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
                          BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
                          mpp_send_msg_to_app(REJECT_PAIR);
                        #endif
                        break;

                      default:
                         mppState = BEACON_BRDCST_RX;
                        return;


                      }
                     return;
                   }
              break;

            default:
               mppState = BEACON_BRDCST_RX;
               return;

            }
      break;
  /* default for main switch*/
      default:/* if trans tyep not supported then reject request*/
                      BT_HOST_PAIRING_REJECT_CODE = MPP_ERR_NO_OTA_COMM_TECH_SUPPORT;
                      BT_HOST_PAIRING_REJECT_TIME_MS =mpp_get_pairing_time();
                      mppState = MPP_FAILURE;
                      return;
      /*error unimplemented message send back error*/

      }/* main switch */

    }/* if*/
    else
    {  /* if time out restart*/
      mppState = BEACON_BRDCST_RX;
      #ifdef MPP_DEBUG_CRC_TIMEOUT_MSG
        BT_HOST_PAIRING_REJECT_CODE = 0xFF; /*Time Out error*/
        BT_HOST_PAIRING_REJECT_TIME_MS = mpp_get_pairing_time();
        BT_HOST_PAIRING_TIME_OUT = 0x03;
        mpp_fail_tick_ms = xTaskGetTickCount();
        if(valid_beacon_proc_total > 0)
          BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = (mpp_fail_tick_ms - start_tick_ms);
        else
          BT_HOST_PAIRING_REJECT_CUMULATIVE_PAIRING_TIME = 0;
        BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_BEACON_PROC = valid_beacon_proc_total;
        BT_HOST_PAIRING_REJECT_CUMULATIVE_VALID_DATA_MSG_PROC = valid_data_msg_proc_total;
        mpp_send_msg_to_app(REJECT_PAIR);
      #endif
      return; /* time out error*/
    }
}  /* for*/

}
/* End of mpp_process_dev_cpb_brdcst()   */
#endif
