/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2014 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: mpp_statemachine.c
#
# --------------------------- General Description -----------------------------
# This file contains the implementation of MPP protocol state machine
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/

#include "ConnectionManager.h"      //CCMPD01414087 (GM_18Nov2010)

#include "mpp_init.h"
#include "mpp_statemachine.h"
#include "mpp_msg.h"

#ifdef MPP_SECURE_DONGLE  // CCMPD01412613 (JJ 12/11/2010)
#include "TaskConfig.h"
#endif


/********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR            Date Modified Tracking Number Description
* Jaspreet Bajwa 01/08/2010    CCMPD       Initial Creation
* Jaspreet Bajwa 01/21/2010    CCMPD       MPP state machine implemented BEACON exchange sequence tested okay, DEVICE capability exchnage message still undertesting
* Jaspreet Bajwa 01/22/2010                MPP Device Capability Message exchanged with dummy bytes
* Jaspreet Bajwa 02/01/2010                MPP device and Host side message encoding / decoding implemented and tested okay
* Rhee Zhang     05/07/2010                Cleanup IAR Compiler Warnings
* JJ Low         11/12/2010     CCMPD01412613    R2.1 MPP state machine
* Mahes          Jan/10/2011     CCMPD01460340   to solve linker issue: Undefined external "send_inter_task_msg"
* Mahesh         30-Dec-2010    CCMPD01414087    use "SEND_MESSAGE_TO_CONNECTION_MNGR()" to replase "MPP_OS_send_inter_task_msg()"
* Mahesh         07-Dec-2011   CCMPD01460638     Added  initial MPP inter task messages
* Sevuhan.R      03-May-2011   CCMPD01503019     Generates a random link key whenever the first device come in pairing range
* Abhishek T.	 04/03/2014			 Do not allow new Audio device to pair
* Abhishek T     10/24/2014    CCMPD01941655     Audio Switch Redesign
*--------------------------- End of History Template----------------------------
* *****************************************************************************/

/*==============================================================================
                                      EXTERNAL DEFINITIONS
==============================================================================*/
//extern void send_inter_task_msg(int taskId, UNSIGNED_8BIT msgid, void **ptr2message, UNSIGNED_16BIT message_size, UNSIGNED_8BIT op_code);  //CCMPD01414087 (GM_30Dec2010)
//extern UNSIGNED_8BIT get_inter_task_msg(UNSIGNED_8BIT *msgid, void **ptr2message, UNSIGNED_16BIT *message_size, UNSIGNED_32BIT_LONG delay ); //CCMPD01414087 (GM_30Dec2010)
extern UNSIGNED_8BIT host_cpb_rep_msg_txed;
extern void mpp_power_save_mode();


/*==============================================================================
                                    GLOBAL VARIABLES
==============================================================================*/
mpp_state_name mppState = MPP_INIT;
UNSIGNED_8BIT mpp_powered_up = 1;
unsigned int isLinkKeyGenerated = 0;
extern connection_database MPP_host_connection_db;
/*==============================================================================
                                      EXTERNAL FUNCTIONS
==============================================================================*/
/*==============================================================================
                                      FUNCTION PROTOTYPES
==============================================================================*/
void mpp_send_msg_to_app(UNSIGNED_8BIT msgID);
void process_msg_from_app(UNSIGNED_32BIT_LONG delay);
extern void calc_random_link_key(bt_link_key *p_bt_link_key);
/*==============================================================================
                                      FUNCTION DEFINITIONS
==============================================================================*/


/*=============================================================================
	FUNCTION: mpp_send_msg_to_app

	DESCRIPTION: Respond to App messages and send approprite request and reply
                      messages as per the MPP protocol.

	ARGUMENTS PASSED: UNSIGNED_8BIT

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_send_msg_to_app(UNSIGNED_8BIT msgid)
{

  void *ptr2message;
  UNSIGNED_16BIT message_size;
  UNSIGNED_8BIT count;

#ifndef MPP_SECURE_DONGLE  //CCMPD01460340(GM_Jan/10/2011)
  SIGNED_32BIT taskId;
#endif

  switch(msgid)
  {

    case REJECT_PAIR:
     /*send reject message to connection manager*/
      #ifdef MPP_DEVICE
      ptr2message = &BT_DEVICE_PAIRING_REJECT;
      message_size = sizeof(BT_DEVICE_PAIRING_REJECT);
      #endif

      #ifdef MPP_HOST
      /*copy tran type and device type*/
      BT_HOST_PAIRING_REJECT_TRANS_TYPE = BT_HOST_SUCC_TRANS_TYPE;
      BT_HOST_PAIRING_REJECT_DEVICE_TYPE = BT_HOST_SUCC_DEVICE_TYPE;
      /*copy mac address of device being rejected*/
      for(count =0;count<MPP_BT_MAC_BYTES;count++)
      {
      BT_HOST_PAIRING_REJECT_DEVICE_MAC(count) = BT_HOST_SUCC_DEVICE_MAC(count) ;
      }

      ptr2message = &BT_HOST_PAIRING_REJECT;
      message_size = sizeof(BT_HOST_PAIRING_REJECT);
      #endif

      break;

    case READY_PAIR:
      /*send ready to pair message to connection manager*/
      ptr2message = NULL;
      message_size = 0;
      msgid = MPP_READY_PAIR;
      break;

    case SUCC_PAIR:
      #ifdef MPP_DEVICE
      /*send device success message to connection manager*/
      ptr2message = &BT_DEV_SUCC_MSG;
      message_size = sizeof(BT_DEV_SUCC_MSG);
      #endif

      #ifdef MPP_HOST
     /*send host success message to connection manager*/
      ptr2message = &BT_HOST_PAIRED_SUCC;
      message_size = sizeof(BT_HOST_PAIRED_SUCC);
      msgid = MPP_SUCC_PAIR;    //CCMPD01414087 (GM_18Nov2010)
      #endif
      break;
    #ifdef MPP_DEVICE
    case INIT_DEVICE_MPP_PKT_REQ:
     /*send init device request message to connection manager*/
      ptr2message = NULL;
      message_size = 0;
      break;

    case IN_RANGE:
      ptr2message = NULL;
      message_size = 0;
      break;
    #endif

    #ifdef MPP_HOST
    case INIT_HOST_MPP_PKT_REQ:
      /*send init host request message to connection manager*/
      ptr2message = NULL;
      message_size = 0;
      msgid = MPP_INIT_HOST_MPP_PKT_REQ;    //CCMPD01414087 (GM_18Nov2010)
      break;

    case HOST_CONN_ALLOC_RES_DB_REQ:
      /*send host connection DB request message to connection manager*/
      ptr2message= NULL;
      message_size = 0;
      msgid = MPP_HOST_CONN_ALLOC_RES_DB_REQ;    //CCMPD01414087 (GM_18Nov2010)
      break;
    #endif
    default:
      /* MPP_ERROR sent from here*/
      ptr2message= NULL;
      message_size = 0;
      break;
  }


#if defined(MPP_SECURE_DONGLE) //CCMPD01460340(GM_Jan/10/2011)

  if (ptr2message == NULL)
  {
    SEND_MESSAGE_TO_CONNECTION_MNGR(msgid, NULL, message_size);
  }
  else
  {
    SEND_MESSAGE_TO_CONNECTION_MNGR(msgid, ptr2message, message_size);
  }
#else
  taskId = MPP_OS_Get_Task_Id(MPP_CONNECTION_MGR);
  if (ptr2message == NULL)
  {
    MPP_OS_send_inter_task_msg(taskId, msgid, NULL, message_size, MPP_OSAL_MSG);
  }
  else
  {
    MPP_OS_send_inter_task_msg(taskId, msgid, &ptr2message, message_size, MPP_OSAL_MSG);
  }
#endif //defined(MPP_SECURE_DONGLE) //CCMPD01460340(GM_Jan/10/2011)

  return;
}
/* End of mpp_send_msg_to_app()   */

/*=============================================================================
	FUNCTION: process_msg_from_app

	DESCRIPTION: Process and decode messages received from App ( Connection manager)

	ARGUMENTS PASSED: UNSIGNED_32BIT_LONG

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void process_msg_from_app(UNSIGNED_32BIT_LONG delay)
{


  void *ptr2message;
  UNSIGNED_8BIT msgid;
  UNSIGNED_16BIT message_size;

 if(MPP_OS_get_inter_task_msg(&msgid, &ptr2message, &message_size, delay))
 {
    switch(msgid)
    {

      case START_PAIR:
        /* handle start pairing message here based on the power state*/
        if(mpp_powered_up){

          #ifdef MPP_DEVICE
          if(mppState == MPP_STANDBY_COMPL)
          {
            mppState = BEACON_BRDCST_TX;
          }
          #endif

        #ifdef MPP_HOST
          if(mppState == MPP_STANDBY_COMPL)
          {
            mppState = BEACON_BRDCST_RX;
          }
        #endif
        }
        else
        {
           mppState = MPP_INIT;
           mpp_powered_up = 1;
        }
        break;

      case STOP_PAIR:
        /* handle stop pairing message here based on the power state*/
        if(mpp_powered_up){
        mpp_power_save_mode();
        mpp_powered_up = 0;
        }
        mppState = MPP_STANDBY_COMPL;
        break;
        
      case SUSPEND_MPP:
        if(mpp_powered_up)
        {
          mppState = MPP_STANDBY_COMPL;
        }
        break;

      case RESUME_MPP:
        if(mpp_powered_up)
        {
          if(mppState == MPP_STANDBY_COMPL)
          {
            mppState = BEACON_BRDCST_RX;
          }
        }
        break;
                
      case MPP_RE_INIT:
        mppState = MPP_INIT;
      break;

      #ifdef MPP_DEVICE
      case INIT_DEVICE_MPP_PKT_RPLY:
        /* copy pointer of device init struct*/
        ptr2init_device_mpp_pkt = (init_device_mpp_pkt *)ptr2message;
        if((ptr2init_device_mpp_pkt!=NULL))
        {
           mpp_send_msg_to_app (READY_PAIR);
           #ifdef MPP_DEBUG
            MPP_PUT_DEBUG_STRING("\r\n PRESS G TO START PAIRING: \0" );
           #endif
        }
        break;
      #endif

      #ifdef MPP_HOST
      case INIT_HOST_MPP_PKT_RPLY:
        /* copy pointer of host init struct if both host init and DB init pointers are received send ready to pair message*/
        ptr2init_host_mpp_pkt = (init_host_mpp_pkt *)ptr2message;
        if((ptr2connection_database!=NULL)&&(ptr2init_host_mpp_pkt!=NULL))
        {
           mpp_send_msg_to_app (READY_PAIR);
          #ifdef MPP_DEBUG
            MPP_PUT_DEBUG_STRING("\r\n PRESS G TO START PAIRING: \0" );
          #endif

        }
        break;

      case HOST_CONN_ALLOC_RES_DB_RPLY:
      /* copy pointer to the connection database if both host init and DB init pointers are received send ready to pair message*/
        ptr2connection_database = (connection_database *)ptr2message;
        if((ptr2connection_database!=NULL)&&(ptr2init_host_mpp_pkt!=NULL))
        {
          mpp_send_msg_to_app (READY_PAIR);
          #ifdef MPP_DEBUG
            MPP_PUT_DEBUG_STRING("\r\n PRESS G TO START PAIRING: \0" );
          #endif

        }
        break;

      case HOST_MPP_CONFIG:
        /* copy continous pairing value from the message if run time config of host is allowed*/
        if(BT_HOST_MPP_CONFIG_TYPE)
        {
          BT_HOST_CONTINOUS_PAIRING = ((host_mpp_config *)ptr2message)->continous_pairing;
        }
        break;
      #endif

      default:
        /* send error message against the unhandled message*/
        mpp_send_msg_to_app(MPP_ERROR);
        break;
    }

}

}
/* End of process_msg_from_app()   */

/*=============================================================================
	FUNCTION: mpp_state_machine

	DESCRIPTION: MPP State Machine Implementation.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void mpp_state_machine( void *pvParameters )
{
#ifdef RTOS_TASK_LOOP
  for(;;)
  {
#endif
    switch(mppState)
    {
#ifdef MPP_DEVICE
        case MPP_INIT:
          mpp_sw_init(); /*init software components like semaphores are send commands to connection manager for required data*/
          mpp_hw_peripherals_init(); /*init all the hardware peripherals*/
          mppState = MPP_STANDBY_COMPL;
          break;

        case MPP_STANDBY_COMPL:
          mpp_hw_setup_idle_mode();
          process_msg_from_app(MAX_DELAY);   /* keep on receiving replies from connection manager and when all requested messages are received the state is changed to BEACON_BRDCST_TX from inside of this fucntion*/
          break;

        case BEACON_BRDCST_TX:
          mpp_hw_setup_tx_mode(); /* put harware into transmission mode*/
          mpp_send_wakeup_sequence(); /* send wake up sequence*/
          //mpp_wait_tx_complete(); /* wait for transmission to complete so that we can send more data */
          //MPP_SYSTEM_DELAY(DELAY_AFTER_PATTERN); /* dealy to introduce some delay between wakeup sequence and actual data so that UART can have sufficient idle time*/
          mpp_send_beacon_brdcst(); /* send beacon broadcast packet*/
          mpp_wait_tx_complete(); /* wait for tx to complete so that we can disable to tx clock*/
          mppState = BEACON_CONF_RX;
          process_msg_from_app(ZERO_DELAY); /*prosess any message comming from connection manager like STOP_PAIR*/
          break;

        case BEACON_CONF_RX:
          mpp_hw_setup_rx_mode();
          mpp_process_beacon_conf(); /* wait for beacon confirmation message and process it state will be changed inside this function*/
          break;

        case DEV_CPB_BRDCST_TX:
          mpp_send_msg_to_app(IN_RANGE);/* send in range message to the app if beacon conf is received successfully*/
          mpp_hw_setup_tx_mode();      /* put harware into transmission mode*/
          mpp_send_wakeup_sequence(); /* send wake up sequence*/
          //mpp_wait_tx_complete(); /* wait for transmission to complete so that we can send more data */
          //MPP_SYSTEM_DELAY(DELAY_AFTER_PATTERN); /* dealy to introduce some delay between wakeup sequence and actual data so that UART can have sufficient idle time*/
          mpp_send_dev_cpb_brdcst(); /* send device capability broadcast message*/
          mpp_wait_tx_complete();/* wait for tx to complete so that we can disable to tx clock*/
          mppState = HOST_CPB_REP_RX;
          break;

        case HOST_CPB_REP_RX:
          mpp_hw_setup_rx_mode();
          mpp_process_host_cpb_rep();/*wait for host capability message and process it state will be changed inside this function*/
          break;

        case MPP_FAILURE:
          mpp_send_msg_to_app(REJECT_PAIR); /* send reject message to connection manager if reject is received from the host*/
          mppState = BEACON_BRDCST_TX;
          break;

        case ACK_TO_HOST:
          mpp_hw_setup_tx_mode();      /* put harware into transmission mode*/
          mpp_send_wakeup_sequence(); /* send wake up sequence*/
          //mpp_wait_tx_complete(); /* wait for transmission to complete so that we can send more data */
          //MPP_SYSTEM_DELAY(DELAY_AFTER_PATTERN); /* dealy to introduce some delay between wakeup sequence and actual data so that UART can have sufficient idle time*/
          mpp_send_ack_to_host(); /* send device capability broadcast message*/
          mpp_wait_tx_complete();/* wait for tx to complete so that we can disable to tx clock*/
          BT_DEV_SUCC_PAIRING_TIME = mpp_get_pairing_time();
          mpp_send_msg_to_app(SUCC_PAIR);
          mppState = MPP_STANDBY_COMPL;
          break;

        default:
          mppState = BEACON_BRDCST_TX;
          process_msg_from_app(ZERO_DELAY);
          break;
#endif

#ifdef MPP_HOST
      /*****************HOST*******************/
        case MPP_INIT:
          mpp_sw_init();/*init software components like semaphores are send commands to connection manager for required data*/
          mpp_hw_peripherals_init();/*init all the hardware peripherals*/
          mppState = MPP_STANDBY_COMPL;
          break;

        case MPP_STANDBY_COMPL:
          mpp_hw_setup_idle_mode();
          process_msg_from_app(MAX_DELAY); /* keep on receiving replies from connection manager and when all requested messages are received send ready to pair and wait for start and goto next state*/
          break;

        case BEACON_BRDCST_RX:
          mpp_hw_setup_rx_mode();
          mpp_process_beacon_brdcst(); /* wait and process the beacon broadcast message*/
          process_msg_from_app(ZERO_DELAY);     /* keep on receiving replies from connection manager like STOP_PAIR */
          break;

        case BEACON_CONF_TX:
           if(!isLinkKeyGenerated) /*generates random link key when first device comes in range after dongle powerup*/
          {  
            calc_random_link_key(&MPP_host_connection_db.link_key);
            isLinkKeyGenerated = 1;
          }
          mpp_hw_setup_tx_mode();  /* put harware into transmission mode*/
          mpp_send_wakeup_sequence();/* send wake up sequence*/
          //mpp_wait_tx_complete(); /* wait for transmission to complete so that we can send more data */
          //MPP_SYSTEM_DELAY(DELAY_AFTER_PATTERN);/* dealy to introduce some delay between wakeup sequence and actual data so that UART can have sufficient idle time*/
          mpp_send_beacon_conf(); /* send beacon confirmation message*/
          mpp_wait_tx_complete();/* wait for tx to complete so that we can disable to tx clock*/
          mppState = DEV_CPB_BRDCST_RX;
          break;


        case DEV_CPB_BRDCST_RX:
          mpp_hw_setup_rx_mode();
          mpp_process_dev_cpb_brdcst(); /* wait and process the device  capability message*/
          break;

        case HOST_CPB_REP_TX:
		  if(check_new_mpp_audio_device(&BT_HOST_PAIRED_SUCC) != FALSE)	
		  {
			mpp_hw_setup_tx_mode();      /* put harware into transmission mode*/
			mpp_send_wakeup_sequence();/* send wake up sequence*/
			//mpp_wait_tx_complete();/* wait for transmission to complete so that we can send more data */
			//MPP_SYSTEM_DELAY(DELAY_AFTER_PATTERN);/* dealy to introduce some delay between wakeup sequence and actual data so that UART can have sufficient idle time*/
			mpp_send_host_cpb_rep();/* send host capability reply message*/
			mpp_wait_tx_complete();/* wait for tx to complete so that we can disable to tx clock*/

			if(mppState == MPP_FAILURE)
			{
			  break;
			}
			host_cpb_rep_msg_txed = MPP_PASS;
			mppState = BEACON_BRDCST_RX;
		  }
		  else
		  {
			mppState = BEACON_BRDCST_RX;
		  }
          break;

        case MPP_FAILURE:
          mpp_hw_setup_tx_mode();   /* put harware into transmission mode*/
          mpp_send_wakeup_sequence(); /* send wake up sequence*/
          //mpp_wait_tx_complete();/* wait for transmission to complete so that we can send more data */
          //MPP_SYSTEM_DELAY(DELAY_AFTER_PATTERN);/* dealy to introduce some delay between wakeup sequence and actual data so that UART can have sufficient idle time*/
          mpp_send_host_reject_msg();
          mpp_wait_tx_complete();/* wait for tx to complete so that we can disable to tx clock*/
          mpp_send_msg_to_app(REJECT_PAIR); /* send the reject message to the connection manager*/
          mppState = BEACON_BRDCST_RX;
          break;

        default:
          mppState = BEACON_BRDCST_RX;
          process_msg_from_app(ZERO_DELAY);
          break;
#endif
    }
#ifdef RTOS_TASK_LOOP
  }
#endif
}
/* End of mpp_state_machine()   */

