/*################################################################################
#
# 						FILE SPECIFICATION
# 					COPYRIGHT 2011,2014 MOTOROLA SOLUTIONS,INC. ALL RIGHTS RESERVED.
#						MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: <spp.c>
#
# --------------------------- General Description -----------------------------
#
#
#
#
********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR                Date Modified	CR Tracking Number 	Description
* Niravkumar Rabara	10/18/2010	CCMPD01402008           Intial Creation       
* Niravkumar Rabara	10/19/2010	CCMPD01404218           Added AT Command for R1.4
* Niravkumar Rabara	10/22/2010	CCMPD01405968           Modify AT Command handling for R1.4 & added Nibbler byte handling
* JJ Low                11/08/2010      CCMPD01410054           Compiler warning resolution for R2.1
* Niravkumar Rabara     11/14/2010      CCMPD01413929           Moved AT Command handling to Bluetooth Manager
                                                                Modified SPP reconnection handling
* Tan Seng Kai      12/08/2010  CCMPD01451448           Remove Unuse function prototype
* Tan Seng Kai      12/10/2010  CCMPD01452526           Receiving Nibbler data after nibbler establish
* Tan Seng Kai        12/20/2010    CCMPD01455949       Nibber sending pointer and size instead of byte by byte sending

* Mahes                Jan/06/2011     CCMPD01460340           Software catch-up to dongle R1.2 version R01.00.00
* Tan Seng Kai        01/17/2011    CCMPD01463409       Added in function call for nibbler enable
* Abhishek Trivedi    04/04/2014    CCMPD01882874       Remove the warnings
*--------------------------- End of History Template-----------------------------
* *********************************************************************************/

/*****************************************************************************/
/* Include files								*/
/*****************************************************************************/


#include "spp_opt.h"
#include "spp.h"
#include "taskconfig.h"
#include "bt_log.h"
#include "bluetooth_manager.h"

#ifdef NIBBLER_TASK
#include "nibbler_task.h"
#endif
#include "nibbler.h"


/*****************************************************************************/
/* Private macros								*/
/*****************************************************************************/

/*
 * How to use PADAPT
 */
#define PADAPT_INTERFACE_TYPE ((PADAPT_InterfaceType)((SPP_DEV_CFG_DISCONNECT_ACL_AFTER_DM_TIME != 0)?\
									(PADAPT_INTERFACE_SIMPLE | PADAPT_GET_DM_SENT_IND):\
									(PADAPT_INTERFACE_SIMPLE)) )


/*
 * Values used for userTimerId in BT_TimeoutReq
 */
#define TIMER_ID_RUN_CONNECTION_STATE_MACHINE		        1
#define TIMER_ID_RESTART_RESPONSE				2

#define TIMER_ID_SDAP_SEARCH					4
#define TIMER_ID_PADAPT_CONNECT_REQ 				5

#define TIMER_ID_WLPTT_DISCONNECTED_STATE_IND			0x0a
#define TIMERID_RESEND_LINK_SETTINGS				0x0b
#define TIMER_ID_DM_DISCONNECT_ACL	   			0x0c
#define TIMER_ID_SEND_LINK_STATUS_IND				0x0d

#define TIMER_ID_RESEND_REJECT_CONNECT				0x0e


/*
 * Values for type ConnectionState
 */

#define WLPTT_DISCONNECTED_STATE					0


#define INCOMING_WLPTT_CONNECT_REQUEST_STATE		1	/* Remote Device initiated ACL
													 * connection, sent SDAP
													 * search req
													 */

#define INCOMING_WLPTT_W_4_CONNECT_RSP				2	/* Remote Device initiated ACL
													 * connection, got SDAP
													 * search response. Waiting
													 * for local connect rsp
													 */

#define WLPTT_SERVICE_SEARCH_STATE 				5	/* Connection initiated
													 * locally, sent SDAP search
													 * req
													 */

#define WLPTT_SERVICE_SEARCH_COMPLETE_STATE 		6	/* Connection initiated
													 * locally, Got SDAP search
													 * response
													 */

#define PADAPT_CONNECTING_STATE					7	/* AG has sent a PADAPT
													 * connect request
													 */

#define PADAPT_CONNECTED_STATE					8	/* AG has received
													 * PADAPT connect cfm,
													 * wlPTTDlci != 0 if successful
													 *
													 * or
													 *
													 * AG has sent padapt
													 * connect response
													 * accepting connection
													 */

#define WLPTT_ACL_CONNECTED_STATE				0x0e	/* AT command sequence
														 * is complete
														 */

/*
 * Character constats
 */
#define CR		0x0d	/* Carrige return */
#define LF		0x0a	/* Line feed */


/*
 * Values for type ParseState
 */
#define W_4_A			0
#define W_4_T			1
#define W_4_CMD 		2


/*
 * Wait max two minutes for SDAP, i.e. basically no timeout
 */
#define MAX_SDAP_SEARCH_TIME					120000


 /*
 *	Typecast macros to avoid having to put a typecast at each call
 */
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
#define SEND_TO_USER(msg, msgType)		SendToUser((BT_Msg*)(msg), (msgType))
#endif

#define SEND_REQ(a, b, c, d, e)			SendReq((BT_Msg*)(a), b, c, d, e)


#define INVALID_CONNECTION_HANDLE	0x8000


#define GET_CONNECTION(con) (SPP_DEV_validMultiProfile ? ((Connections*) con) : (&connections[0]))

#define SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS 1
extern BT_U16 SPP_DEV_OptInitialise(void);    // CCMPD01410054 (JJ 8/11/2010)
/*****************************************************************************/
/* Private types															 */
/*****************************************************************************/


/*
 * Union of all messages that SPP DEV can receive
 */
typedef union
{
	BT_Header		header;
	PADAPT_IndConf	padaptIndConf;
	SDAP_Rsp		sdapRsp;
	HCI_Event		hciEvent;
	BT_TimeoutInd	tsrvInd;
	HCI_LH_IndConf	lh;
} SPP_DEV_InMsg;

/*
 * Main state information for module, shows
 * state of service level connection
 */
typedef BT_U8	ConnectionState;



/*
 * State of AT parser
 */
typedef BT_U8	ParseState;


/*
 * Struct to store information about a specific connection
 */
typedef struct
{
	/*
	 * The address of the SPP DEV
	 */
	BT_BdAddr				wlPTTBdAddr;

	/*
	 * When there is a PADAPT connection , wlPTTDlci identifies
	 * the PADAPT connection to the SPP DEV.
	 */
	RFCOMM_Dlci 			wlPTTDlci;

	/*
	 * wlPTTServerChannel is set by the SDAP search procedure in the
	 * beginning of each connection. If there was no matching SPP DEV
	 * service it will be set to 0
	 */
	RFCOMM_ServerChannel	wlPTTServerChannel;

	/*
	 * The agreed frame size used for the PADAPT connection
	 */
	RFCOMM_DlcFrameSize 	framesize;

	/*
	 * When the are data and audio connections to the SPP
	 * the connection handles will be stored in these varables
	 * The audioConnectionHandle will be used to keep track if 
	 * audio is connected or not.
	 */

	BT_U16					aclConnectionHandle;


	/*
	 * Store current status of the PADAPT flow control. BT_TRUE
	 * if it is allowed to send data to AG, BT_FALSE otherwise
	 */
	BT_BOOL 				remoteReadyToReceive;


	/*
	 * State information for SP DEV module,
	 */
	ConnectionState 		connectionState;
	ParseState				parseState;
	BT_TimerId					connectTimer;
	BT_TimerId					sdapTimer;


	/*
	 * Used for connect/disconnect to make it
	 * possible to send appropriate confirm when done
	 */
	BT_Primitive				userRequestInProgress;

	/*
	 * Used to assemble at commands complete result one delimited
	 * by AT ...<CR> have been received.
	 */
	BT_U8	*					atCommandBuffer;

	/*
	 * Length of current contents of atCommandBuffer
	 */
	BT_U16						atCommandLength;
} Connections;


/* 
 * Info needed to reject connects from SP DEV while another connections exists
 * or is being setup. Can happen anytime so a separate timer is needed.
 */	
typedef struct
{
	BT_TimerId			timer;
	BT_BdAddr 			bdAddr;
	RFCOMM_Dlci			dlci;
} RejectConnectInfo;


/*****************************************************************************/
/* Public constants & variables 											 */
/*****************************************************************************/

/*
 * Module id assigned to SPP_DEV during startup
 */
BT_ModuleId 	BT_RAM	SPP_DEV_moduleId = 0;

BT_BOOL 	BT_RAM	SPP_DEV_validMultiProfile;

extern nibbler_struct_t wirelessPTT_nibbler;

/*****************************************************************************/
/* Private constants & variables											 */
/*****************************************************************************/

/*
 * Module id of registered user of SPP DEV or 0 if there
 * currently is no registered user
 */
static BT_ModuleId				BT_RAM sppDevUser = 0;


/*
 * Local server channel assigned when registering service. Needs
 * to be saved to make it possible to unregister service
 */
static	RFCOMM_ServerChannel	BT_RAM localServerChannel;


/*
 * Use this instead of SPP_DEV_attributeIdList when connecting to
 * avoid allocating a buffer for service name (which might be long)
 */
static const BT_U8 BT_ROM connectAttributeIdList[] =
{
	0x35,	0x09,		/* data element seq 9 bytes */

	0x09, 0x00, 0x01,	/* 0x0001 - UINT16, Service Class Id List */

	0x09, 0x00, 0x04,	/* 0x0004 - UINT16, Protocol descriptor list */
						 
	0x09, 0x01, 0x00 	 /*0x0100 - UINT16, Service name */
														 
};


/*
 * Track the connections.
 */
static Connections * BT_RAM connections;

/*
 * Restarted response timer
 */
static BT_TimerId restartedRspTimer;

/*
 * When receiving PADAPT_DM_SENT_IND, we will
 * optinally tear down the ACL link after a timeout.
 * Store bdAddress to disconnect here while waiting
 * for timeout.
 */
static BT_BdAddr disconnectAclBdaddr;

/* If data request has returned BT_CONGESTED */
static BT_BOOL					returnedDataCongestion = BT_FALSE;

static RejectConnectInfo rejectConnectInfo;
//static BT_BdAddr currentR1_4Addr;
//static BT_U8 sppAtCmdInd;

/*****************************************************************************/
/* Private function prototypes												 */
/*****************************************************************************/
static void Initialise(void) BT_LARGE;

static void ChangeConnectionState(Connections *con, ConnectionState newState) BT_LARGE;

static void HandleHciMsg(const HCI_Event *m) BT_LARGE;
static void HandlePadaptMsg(const PADAPT_IndConf *m) BT_LARGE;
static void HandleSdapMsg(const SDAP_Rsp *m) BT_LARGE;
static void HandleTsrvMsg(const BT_TimeoutInd *m) BT_LARGE;
static void HandleLinkHandlerMsg( const HCI_LH_IndConf *ind) BT_LARGE;

static BT_Result  SendPadaptConnectRsp(Connections *con, BT_BOOL accept) BT_LARGE;

static BT_BOOL	SetupHciUsage(void) BT_LARGE;

static BT_Result HandleSppDevConnectReq(const BT_BdAddr bdAddr) BT_LARGE;

static BT_Result HandleSppDevDisconnectReq(Connections *con) BT_LARGE;

static void RestartAtCommandParser(Connections *con) BT_LARGE;
//static const char* StrIgnoreCaseStr(const char *string, const char *strCharSet);  // CCMPD01410054 (JJ 8/11/2010)
//static BT_S32 StrIgnoreCaseNCmp( const char *str1, const char *str2, BT_U32 count ) BT_LARGE; // CCMPD01410054 (JJ 8/11/2010)

static BT_Result SendPadaptDataReq(Connections *con,const BT_U8 *dta, BT_U16 length ) BT_LARGE;

static void SendAclDisconnect(BT_BdAddr bdAddr );
static void SendSppNibbleByte(Connections *con, BT_U8 *c, BT_U8 size);
/*****************************************************************************/
/* Public API functions 													 */
/*****************************************************************************/

/* ===========================================================================
 * Function:	SPP_DEV_ChangeSecurityLevel
 * Description: 
 * ==========================================================================*/
BT_Result SPP_DEV_ChangeSecurityLevel(SEC_SecurityLevel securityLevel)
{
	BT_Result res = BT_FAILED;

	if ( (res = SEC_ChangeIncomingService(securityLevel,SEC_ID_RFCOMM,
								localServerChannel)) != BT_OK ) 
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "SEC_ChangeIncomingService failed"));
	}
	else if ( (res = SEC_ChangeOutgoingService(securityLevel,SEC_ID_RFCOMM,
								BT_SPP_DEV)) != BT_OK )
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "SEC_ChangeOutgoingService failed"));
	}

	return res;
}


/* ===========================================================================
 * Function:	SPP_DEV_Control
 * Description: 
 * =========================================================================== */
BT_Result SPP_DEV_Control(BT_ControlCommand cmd) BT_LARGE
{
	switch ( cmd )
	{
	case BT_INITIALISE:
		Initialise();
		break;

	case BT_REGISTER:
		break;

	case BT_START:
		break;
	}
	return BT_CONTROL_CMD_COMPLETE;
}


/* ===========================================================================
 * Function:	SPP_DEV_RequestMsgHandler
 * Description: Dispatcher for all requests received by SPP_DEV
 * Returns: 	BT_OK, BT_CONGESTED or BT_CALL_ERROR
 * =========================================================================== */
BT_Result SPP_DEV_RequestMsgHandler(BT_DEPRECATED_MSG_CONST BT_Msg *msg) BT_LARGE
{
	BT_Result			   BT_RAM res	= BT_CALL_ERROR;
	const SPP_DEV_ReqResp * BT_RAM m 	= (const SPP_DEV_ReqResp *)msg;

	switch ( m->header.primitive )
	{
	case SPP_DEV_CONNECT_REQ:
		res = HandleSppDevConnectReq(m->connectReq.bdAddr);
		break;

	case SPP_DEV_DISCONNECT_REQ:
		res = HandleSppDevDisconnectReq(GET_CONNECTION(m->disconnectReq.handle));
		break;

	case SPP_DEV_CONNECT_RSP:
		res = SendPadaptConnectRsp(GET_CONNECTION(m->connectRsp.handle),
								   m->connectRsp.accept);
		break;

	case SPP_DEV_DATA_REQ:
		res = SendPadaptDataReq(GET_CONNECTION(m->dataReq.handle),
								m->dataReq.dta, m->dataReq.length);
		returnedDataCongestion = (BT_BOOL) (res == BT_CONGESTED);
		break;

	default:
		break;
	}

	return res;
}

/* ===========================================================================
 * Function:	SPP_DEV_IndicationMsgHandler
 * Description: Dispatcher for indications received by SPP_DEV
 * Returns: 	BT_OK
 * =========================================================================== */
BT_Result SPP_DEV_IndicationMsgHandler(BT_DEPRECATED_MSG_CONST BT_Msg *msg) BT_LARGE
{
	SPP_DEV_InMsg *BT_RAM m = (SPP_DEV_InMsg*)msg;

	/*
	 * Dispatch to appropriate handler function
	 */
	switch (BT_MSG_CLASS(m->header.primitive ))
	{
	case HCI_MSG_CLASS:
		HandleHciMsg(&m->hciEvent);
		break;

	case HCI_LH_MSG_CLASS:
		HandleLinkHandlerMsg(&m->lh);
		break;

	case SDAP_MSG_CLASS:
		HandleSdapMsg(&m->sdapRsp);
		break;

	case TSRV_MSG_CLASS:
		if ( !BT_TimeoutIsCancelled(m->tsrvInd.timerId) )
		{
			HandleTsrvMsg(&m->tsrvInd);
		}
		break;

	case PADAPT_MSG_CLASS:
		HandlePadaptMsg(&m->padaptIndConf);
		break;
	}

	return BT_OK;
}


/* ===========================================================================
 * Function:	SPP_DEV_HandleOrdinalNo
 * Description: Return the ordinal number for specific handle
 * =========================================================================== */
BT_U8 SPP_DEV_HandleOrdinalNo(SPP_DEV_Handle handle) BT_LARGE
{
	return (BT_U8) (BT_PTRDIFF((Connections*) handle, connections));
}

/* ===========================================================================
 * Function:	FreeConnection
 * Description: Frees a specific connection and assign default values.
 * =========================================================================== */
static void FreeConnection(Connections *con)
{

	con->aclConnectionHandle	= INVALID_CONNECTION_HANDLE;
	con->remoteReadyToReceive	= BT_FALSE;
	con->connectionState		= WLPTT_DISCONNECTED_STATE;
	con->parseState 			= W_4_A;

	con->connectTimer			= 0;
	con->sdapTimer				= 0;

	con->userRequestInProgress	= 0;

	con->atCommandLength		= 0;
}

/* ===========================================================================
 * Function:	GetConnectionUserRequest
 * Description: Check if there's a user request running amongst the connections
 * Returns: 	Pointer to the connection using p, otherwise 0
 * =========================================================================== */
static Connections *GetConnectionUserRequest(BT_Primitive p) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS
		   &&	con == 0 )
	{
		/*
		 * If no primitive is requested, return the first
		 * connection which has a user request running
		 */
		if (	p == 0x0000
			&&	connections[n].connectionState != WLPTT_DISCONNECTED_STATE
			&&	connections[n].userRequestInProgress != 0x0000 )
		{
			con = &connections[n];
		}
		/*
		 * If a primitive is requested, return the first
		 * connection which user request is equal to the requested one
		 */
		else if ( p != 0x0000
			&&	connections[n].connectionState != WLPTT_DISCONNECTED_STATE
			&&	connections[n].userRequestInProgress == p )
		{
			con = &connections[n];
		}
		else
		{
			/*
			 * Try the next connection
			 */
		}

		n++;
	}

	return con;
}


/* ===========================================================================
 * Function:	GetConnectionBdAddr
 * Description: Get a connection with a specific bdAddr.
 * Returns: 	A pointer to a connection
 * =========================================================================== */
static Connections *GetConnectionBdAddr(const BT_BdAddr bdAddr) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS
		   &&	con == 0 )
	{
		if (	connections[n].connectionState != WLPTT_DISCONNECTED_STATE
			&&	!memcmp(bdAddr, connections[n].wlPTTBdAddr, 6) )
		{
			con = &connections[n];
		}

		n++;
	}

	return con;
}


/* ===========================================================================
 * Function:	GetConnectionState
 * Description: Get a connection with a specific state
 * Returns: 	A pointer to a connection
 * =========================================================================== */
static Connections *GetConnectionState(ConnectionState state) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS
		   &&	con == 0 )
	{
		if ( connections[n].connectionState == state )
		{
			con = &connections[n];
		}

		n++;
	}

	return con;
}


/* ===========================================================================
 * Function:	GetConnectionConHnd
 * Description: Get a connection with a specific HCI connections handle
 * Returns: 	A pointer to a connection
 * =========================================================================== */
static Connections *GetConnectionConHnd(HCI_LinkType linkType, BT_U16 conHnd) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS
		   &&	con == 0 )
	{
	       if (	linkType == HCI_LINK_TYPE_ACL_CONNECTION
				 && connections[n].connectionState != WLPTT_DISCONNECTED_STATE
				 && connections[n].aclConnectionHandle == conHnd )
		{
			con = &connections[n];
		}

		n++;
	}

	return con;
}


/* ===========================================================================
 * Function:	NewConnection
 * Description: Create a connection with a specific bdAddr.
 * Returns: 	A pointer to a connection
 * =========================================================================== */
static Connections *NewConnection(const BT_BdAddr bdAddr) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	Connections * BT_RAM con = 0;

	while ( 	n < SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS
		   &&	con == 0 )
	{
		if ( connections[n].connectionState == WLPTT_DISCONNECTED_STATE )
		{
			con = &connections[n];

			/*
			 * Init the connection and store data about the SPP DEV
			 */
			FreeConnection(con);
			memcpy(con->wlPTTBdAddr, bdAddr, 6);
		}

		n++;
	}

	return con;
}

/* ===========================================================================
 * Function:	VerifyConnectionHandle
 * Description: Verify that the supplied handle is valid.
 * Returns: 	TRUE if valid handle
 * =========================================================================== */
static BT_BOOL VerifyConnectionHandle(const Connections *handle) BT_LARGE
{
	BT_U8 BT_RAM n = 0;
	BT_BOOL validHandle = BT_FALSE;

	while ( 	n < SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS
		   &&	!validHandle )
	{
		validHandle = (handle == &connections[n]);
		n++;
	}

	return validHandle;
}



/* ===========================================================================
 * Function:	SendPadaptDataReq
 * Description:
 * =========================================================================== */
static BT_Result SendPadaptDataReq(Connections *con, 
								   const BT_U8 *dta, BT_U16 length ) BT_LARGE
{
    PADAPT_DataReq	BT_RAM req;
    BT_Result		BT_RAM res = BT_CALL_ERROR;
    //const char	BT_RAM ok[] = {CR, LF, 'O', 'K', CR, LF, 0x00}; // CCMPD01410054 (JJ 8/11/2010)

    if ( !con )
    {
      LOG_INFO((SPP_DEV_MODULE_ID, "Invalid connection handle"));
    }
    req.header.primitive	= PADAPT_DATA_REQ;
    req.header.receiver 	= BT_PADAPT;
    req.header.sender		= SPP_DEV_MODULE_ID;

    req.credits 			= 0;
    memcpy( req.bdAddr, con->wlPTTBdAddr, 6);
    req.dlci = con->wlPTTDlci;

    req.dataBufferPtr = dta;
    req.bufferLength = length;

    res = SPP_DEV_SEND_REQ(&req);


    return res;
}


/* ===========================================================================
 * Function:	ChangeConnectionState
 * Description: Change the connection state. A function is used
 *				since it simplifies test and debug
 * =========================================================================== */
static void ChangeConnectionState(Connections *con,
								  ConnectionState newState) BT_LARGE
{
	con->connectionState = newState;
}


/* ===========================================================================
 * Function:	CancelTimers
 * Description: Cancel running timers
 * =========================================================================== */
static void CancelTimers(Connections *con) BT_LARGE
{
	if ( con->sdapTimer )
	{
		BT_TimeoutCancel(con->sdapTimer);
		con->sdapTimer				= 0;
	}
}

#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
/* ===========================================================================
 * Function:	SendToUser
 * Description: Create header and send message to registered user
 * =========================================================================== */
static void SendToUser(BT_Msg *ind, BT_Primitive primitive) BT_LARGE
{
	ind->header.primitive	= primitive;
	ind->header.receiver	= sppDevUser;
	ind->header.sender		= SPP_DEV_MODULE_ID;

	if ( sppDevUser != 0 )
	{
		(void)SPP_DEV_SEND_IND(ind);
	}
	else
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "No service registered with SPP_DEV profile"));
	}
}
#endif

/* ===========================================================================
 * Function:	SendReq
 * Description: Fill in sender field of header and send a request message
 *				Optionally a timer is started if sending is congested .
 *
 * Parameters:	msg
 *					Message to send
 *
 *				userTimerId
 *					ID of timer to start if sending is BT_CONGESTED. If this
 *					is 0, no timer is started.
 *
 *				timerId
 *					If a timer was started, its id is store in the variable
 *					pointed to by this variable
 *
 *				timeout
 *					The length of the timeout if a timer is started.
 *
 *				con
 *					Pointer to the connection used
 *
 * Returns: 	BT_FALSE if sending did not succeed (even if timer
 *				was started). BT_TRUE if message was sent.
 * =========================================================================== */
static BT_BOOL SendReq(BT_Msg *msg, BT_U8 userTimerId,
					   BT_TimerId *timerId, BT_U32 timeout,
					   Connections *con) BT_LARGE
{
	BT_BOOL 	BT_RAM	sentRightAway = BT_TRUE;
	BT_Result	BT_RAM	res;
	

	msg->header.sender		= SPP_DEV_MODULE_ID;

	res = SPP_DEV_SEND_REQ(msg);
	
	if ( res == BT_CONGESTED )
	{
		/*
		 * If a timer id was passed to this function, start it
		 * and indicate to caller that the request was delayed
		 */
		if ( userTimerId != 0 )
		{
			*timerId = BT_TimeoutReq(SPP_DEV_MODULE_ID, userTimerId,
									con, timeout);
		}

		sentRightAway = BT_FALSE;
	}

	return sentRightAway;
}

/* ===========================================================================
 * Function:	SetupHciUsage
 * Description: Register with HCI driver and subscribe to events
 * Returns: 	BT_TRUE if success, BT_FALSE otherwise
 * =========================================================================== */
static BT_BOOL SetupHciUsage(void) BT_LARGE
{
	BT_BOOL BT_RAM res = BT_TRUE;

	if ( HCI_RegisterUser(SPP_DEV_MODULE_ID) != BT_OK )
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Register as user of HCI Driver failed\n"));
		res = BT_FALSE;
	}
	else
	{
		(void)HCI_EventSubscribe( SPP_DEV_MODULE_ID, HCI_CONNECTION_COMPLETE_EVENT);
		(void)HCI_EventSubscribe( SPP_DEV_MODULE_ID, HCI_CONNECTION_REQUEST_EVENT);
		(void)HCI_EventSubscribe( SPP_DEV_MODULE_ID, HCI_LH_AUDIO_CONNECT_COMPLETE_IND);
		(void)HCI_EventSubscribe( SPP_DEV_MODULE_ID, HCI_DISCONNECTION_COMPLETE_EVENT);

		(void)HCI_EventSubscribe( SPP_DEV_MODULE_ID, HCI_CONTROLLER_RESTARTED_IND);
	}

	return res;
}


/* ===========================================================================
 * Function:	SendPadaptConnectReq
 * Description: Create and send a PADAPT_CONNECT_REQ.
 * =========================================================================== */
static void SendPadaptConnectReq(Connections *con) BT_LARGE
{
	PADAPT_ConnectReq	BT_RAM req;

	req.header.receiver 	= BT_PADAPT;
	req.header.primitive	= PADAPT_CONNECT_REQ;

	memcpy( req.bdAddr, con->wlPTTBdAddr, 6 );
	req.serverChannel	= con->wlPTTServerChannel;

	req.serviceId		= BT_SPP_DEV;
	req.maxFramesize	= SPP_DEV_CFG_MAX_FRAME_SIZE;
	req.minFramesize	= SPP_DEV_CFG_MIN_FRAME_SIZE;
	req.framesize		= SPP_DEV_CFG_FRAME_SIZE;
	req.creditsToSend	= SPP_DEV_CFG_CREDITS_TO_SEND;
	req.minCredits		= SPP_DEV_CFG_MIN_CREDITS;
	req.interfaceType	= PADAPT_INTERFACE_SIMPLE;
	req.initialCredits	= SPP_DEV_CFG_INITIAL_CREDITS;

	(void)SEND_REQ(&req, TIMER_ID_PADAPT_CONNECT_REQ, &con->connectTimer,
					SPP_DEV_CFG_RESEND_TIME, con);
}




/* ===========================================================================
 * Function:  SendAclDisconnect
 * Description: Creates and sends a HCI disconnection message.
 * Returns: 	-
 * ==========================================================================*/
static void SendAclDisconnect(BT_BdAddr bdAddr )
{
	BT_U16			connectionHandle;
	HCI_Disconnect disconnect;

	if ( HCI_GetConnectionHandle(bdAddr, &connectionHandle ) )
	{
		/* 
		 * Message header 
		 */
		disconnect.header.primitive = HCI_DISCONNECT;
		disconnect.header.sender = SPP_DEV_MODULE_ID;
		disconnect.header.receiver = BT_HCI;

		/* 
		 * Message parameters 
		 */
		disconnect.connectionHandle = connectionHandle;
		disconnect.reason = HCI_OTHER_END_TERMINATED_USER_ENDED_CONNECTION;

		if ( BT_SEND(&disconnect) == BT_CONGESTED )
		{
			(void)BT_TimeoutReq(SPP_DEV_MODULE_ID, 
			   TIMER_ID_DM_DISCONNECT_ACL,
									0, 
									SPP_DEV_CFG_RESEND_TIME);
		}
		
	}
}



/* ===========================================================================
 * Function:	SendServiceSearchAttrReq
 * Description: Send an SDAP_SERVICE_SEARCH_ATTR_REQ to search for SPP DEV
 *				service
 * =========================================================================== */
static void SendSdapServiceSearchAttrReq(Connections *con) BT_LARGE
{
	SDAP_ServiceSearchAttrReq	BT_RAM attrReq;

	/* Message header */
	attrReq.header.primitive	= SDAP_SERVICE_SEARCH_ATTR_REQ;
	attrReq.header.receiver 	= BT_SDAP;

	/* Message parameters */
	attrReq.noOfBdAddr	= 1;

	attrReq.bdAddr = &con->wlPTTBdAddr;

	attrReq.remDevRelation	= SDAP_NO_RELATION;
	attrReq.searchPattern	= (SDP_ConstDePtr)SPP_DEV_serviceSearchPattern;
	attrReq.attributeId 	= (SDP_ConstDePtr)connectAttributeIdList;
	attrReq.getRemDevName	= BT_FALSE;

	attrReq.maxNoOfBytes	= SPP_DEV_SDAP_RECEIVE_BUFFER_SIZE;
	attrReq.maxSearchTime	= MAX_SDAP_SEARCH_TIME;

	(void)SEND_REQ(&attrReq, TIMER_ID_SDAP_SEARCH,
					&con->sdapTimer, SPP_DEV_CFG_RESEND_TIME, con);
}


/* ===========================================================================
 * Function:	SendLinkSettingsReq
 * Description:
 * =========================================================================== */
static void SendLinkSettingsReq(Connections *con) BT_LARGE
{
	HCI_LH_LinkSettingsReq	BT_RAM req;

	req.header.primitive	= HCI_LH_LINK_SETTINGS_REQ;
	req.header.sender		= SPP_DEV_MODULE_ID;
	req.header.receiver 	= BT_LookupProfile( BT_HCI_LH );

	req.connectionHandle	= con->aclConnectionHandle;
	req.settings			= SPP_DEV_CFG_LINK_HANDLER_SETTINGS;

	(void)SEND_REQ(&req, TIMERID_RESEND_LINK_SETTINGS,
					&con->connectTimer, SPP_DEV_CFG_RESEND_TIME, con);
}

/* ===========================================================================
 * Function:	SendLinkStatusInd
 * Description: Send a SPP_DEV_LINK_STATUS_IND to registered user
 * =========================================================================== */
static void SendLinkStatusInd(Connections *con) BT_LARGE
{
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
	SPP_DEV_LinkStatusInd BT_RAM ind;

	if ( con->connectionState == WLPTT_ACL_CONNECTED_STATE )
	{
		ind.aclStatus = SPP_DEV_CONNECTED;
	}
	else
	{
		ind.aclStatus = SPP_DEV_NOT_CONNECTED;
	}
	
	ind.handle = con;
	memcpy(ind.bdAddr,con->wlPTTBdAddr,6); 
        
        
	SEND_TO_USER(&ind, SPP_DEV_LINK_STATUS_IND );
#endif
}

/* ===========================================================================
 * Function:	SendConnectCfm
 * Description: If CONNECT_REQ was sent by user, report result
 * =========================================================================== */
static void SendConnectCfm( Connections *con,
						   SPP_DEV_Result res,
						   RFCOMM_DlcFrameSize	fsize ) BT_LARGE
{
	SPP_DEV_ConnectCfm BT_RAM cfm;
        //btDevInfo tempDevInfo; // CCMPD01410054 (JJ 8/11/2010)

	/*
	 * Clear the user request in progress
	 */
	con->userRequestInProgress	= 0;

	cfm.result					= res;
	cfm.framesize				= fsize;
	memcpy(cfm.bdAddr, con->wlPTTBdAddr, 6);

	/*
	 * Only return a handle if the result is a success.
	 */
	cfm.handle					= (res == SPP_DEV_OK) ? con : 0;

#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
	SEND_TO_USER(&cfm, SPP_DEV_CONNECT_CFM );
#endif
       //SPPConnectCfm(&cfm);
    
}

/* ===========================================================================
 * Function:	RunConnectionSetupStateMachine
 * Description: connection setup sequence.
 * =========================================================================== */
static void RunConnectionSetupStateMachine(Connections *con) BT_LARGE
{

	switch ( con->connectionState )
	{
	case WLPTT_DISCONNECTED_STATE:
		con->wlPTTServerChannel = 0;
		SendSdapServiceSearchAttrReq(con);
		ChangeConnectionState(con, WLPTT_SERVICE_SEARCH_STATE);
		break;

	case WLPTT_SERVICE_SEARCH_COMPLETE_STATE:
		if ( con->wlPTTServerChannel != 0 )
		{
			con->wlPTTDlci = 0;
			SendPadaptConnectReq(con);
			ChangeConnectionState(con,	PADAPT_CONNECTING_STATE );
		}
		else
		{
			FreeConnection(con);
			/*
			 * Report failure
			 */
			SendConnectCfm(con, SPP_DEV_SERVICE_SEARCH_FAILED, 0);
		}
		break;

	case PADAPT_CONNECTED_STATE:
		if ( con->wlPTTDlci != 0 )
		{
			//ChangeConnectionState(con, W_4_LINK_SETTINGS);
			//con->parseState = W_4_A;
			//SendLinkSettingsReq(con);
			ChangeConnectionState(con, WLPTT_ACL_CONNECTED_STATE);
			con->parseState = W_4_A;
		

		}
		else
		{
			FreeConnection(con);
			/*
			 * Report failure
			 */
			SendConnectCfm(con, SPP_DEV_RFCOMM_CONNECT_FAILED, 0);
		}
		break;
	}
}


/* ===========================================================================
 * Function:	StartHandlingIncomingConnectionAttempt
 * Description:
 * =========================================================================== */
static void StartHandlingIncomingConnectionAttempt(Connections *con, 
												   const BT_BdAddr bdAddr) BT_LARGE
{
        SPP_DEV_ConnectInd BT_RAM ind;
	memcpy(con->wlPTTBdAddr, bdAddr, 6);

	ChangeConnectionState(con, INCOMING_WLPTT_CONNECT_REQUEST_STATE);

	/*
	 * Save connection handle for future use
	 */
	(void)HCI_GetConnectionHandle( con->wlPTTBdAddr, &con->aclConnectionHandle);

	/*
	 * Start the service search to retrieve the AGs features
	 */
	//SendSdapServiceSearchAttrReq(con);
        ChangeConnectionState(con, INCOMING_WLPTT_W_4_CONNECT_RSP);

	/*
	 * Notify user
	 */
	memcpy( ind.bdAddr, con->wlPTTBdAddr, sizeof( BT_BdAddr ) );
	ind.framesize	= con->framesize;
	ind.handle		= con;

        SEND_TO_USER(&ind, SPP_DEV_CONNECT_IND);
}


/* ===========================================================================
 * Function:  SendPadaptDisconnectReq
 * Description:
 * =========================================================================== */
static BT_Result SendPadaptDisconnectReq(Connections *con) BT_LARGE
{
	PADAPT_DisconnectReq BT_RAM req;
	BT_Result				BT_RAM res = BT_CONGESTED;

	req.header.sender		= SPP_DEV_MODULE_ID;
	req.header.primitive 	= PADAPT_DISCONNECT_REQ;
	req.header.receiver 	= BT_PADAPT;

	req.dlci				= con->wlPTTDlci;
	memcpy(req.bdAddr, con->wlPTTBdAddr, sizeof(BT_BdAddr));

	if ( 	con->connectTimer == 0
		&& 	(res = SPP_DEV_SEND_REQ((BT_Msg*)&req)) == BT_OK )
	{
		/*
		 * Cancel possibly running timers
		 */
		CancelTimers(con);

		/*
		 * Free the connection
		 */
		FreeConnection(con);

		/*
		 * Switch context before sending link status ind
		 */
		con->connectTimer = BT_TimeoutReq(SPP_DEV_MODULE_ID,
									TIMER_ID_WLPTT_DISCONNECTED_STATE_IND,
										con, 0);
	}

	return res;
}


/* ===========================================================================
 * Function:  HandleSppDevDisconnectReq
 * Description:
 * =========================================================================== */
static BT_Result HandleSppDevDisconnectReq(Connections *con) BT_LARGE
{
	BT_Result BT_RAM res = BT_CALL_ERROR;

	if ( !VerifyConnectionHandle(con) )
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Invalid connection handle"));
	}
	/*
	else if ( con->connectionState == W_4_LINK_SETTINGS )
	{
		
		 Wait at least until the links settings are complete
		 
		res = BT_CONGESTED;
	}
	*/
	else if ( con->connectionState == WLPTT_ACL_CONNECTED_STATE )
	{
		/*
		 * Disconnect using PADAPT (there is no response to this command)
		 */
		res = SendPadaptDisconnectReq(con);
	}
	else
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Invalid connection state (ACL %d)",
				(int)con->connectionState));
	}

	return res;
}

/* ===========================================================================
 * Function:	HandleSppDevConnectReq
 * Description:
 * =========================================================================== */
static BT_Result HandleSppDevConnectReq(const BT_BdAddr bdAddr) BT_LARGE
{
	BT_Result BT_RAM res = BT_OK;
	Connections *con;

	/*
	 * Check if there's an user request running.
	 * If so, wait until its completed.
	 */
	if ( GetConnectionUserRequest(0) )
	{
		res = BT_CONGESTED;
	}
	/*
	 * Only one connection per bdAddr is allowed
	 */
	else if ( (con = GetConnectionBdAddr(bdAddr)) != 0 )
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Only one connection per bdAddr is allowed"));
		res = BT_CALL_ERROR;
	}
	/*
	 * Try to find a spare connection. Return BT_CONGESTED if everyone
	 * is allocated.
	 */
	else if ( (con = NewConnection(bdAddr)) == 0 )
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "No resources left, max number of "
			"connections reached."));
		res = BT_CONGESTED;
	}
	else
	{
		/*
		 * Save primitive so that appropriate cfm can be sent after connect
		 * complete/failed
		 */
		con->userRequestInProgress = SPP_DEV_CONNECT_REQ;

		/*
		 * Switch context before running connect state machine to avoid sending result
		 * in request handler in case an error occurs
		 */
		con->connectTimer = BT_TimeoutReq(SPP_DEV_MODULE_ID,
									TIMER_ID_RUN_CONNECTION_STATE_MACHINE,
										con, 0);
	}

	return res;
}


BT_Result SendSppConncedReq(SPP_DEV_ConnectReq *req)
{
  BT_Result BT_RAM res = BT_OK;
   res = HandleSppDevConnectReq(req->bdAddr);
   return res;
}
/* ===========================================================================
 * Function:	SendRestartResponse
 * Description: Send HCI_CONTROLLER_RESTARTED_RSP
 * =========================================================================== */
static void SendRestartResponse(void) BT_LARGE
{
	BT_Msg BT_RAM msg;

	msg.header.primitive	= HCI_CONTROLLER_RESTARTED_RSP;
	msg.header.receiver 	= BT_HCI;

	(void)SEND_REQ(&msg, TIMER_ID_RESTART_RESPONSE,
				&restartedRspTimer, SPP_DEV_CFG_RESEND_TIME, 0);
}


/* ===========================================================================
 * Function:	RestartedCfmUserCommand
 * Description: Sends a negative confirm to any user command
 *				that may be running
 * =========================================================================== */
static void RestartedCfmUserCommand(Connections *con) BT_LARGE
{
	/*
	 * Send appropriate confirm negative,
	 * userRequestInProgressis cleared in each
	 * send cfm function
	 */
	switch ( con->userRequestInProgress )
	{
	case 0:
		/*
		 * No active request
		 */
		break;

	case SPP_DEV_CONNECT_REQ:
		SendConnectCfm(con, SPP_DEV_HOST_CONTROLLER_RESTART, 0 );
		break;

	default:
		LOG_INFO(( SPP_DEV_MODULE_ID,
			"Restart handling, unknown user request %d, Please report to Mecel",
				(int) con->userRequestInProgress ));
	}
}
/* ===========================================================================
 * Function:  SendSppDisconnectCfm
 * Description:
 * =========================================================================== */
static void SendSppDisconnectCfm(Connections *con, BT_BOOL success) BT_LARGE
{
        SPP_DEV_DisconnectCfm cfm;
	/*
	 * Confirm sent, no request in progress
	 */
	con->userRequestInProgress = 0;
	cfm.success = success;
        memcpy(&cfm.bdAddr,con->wlPTTBdAddr,6);
        SPPDisconnectCfm(&cfm);

#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
	//SEND_TO_USER(&cfm, HFP_AG_AUDIO_DISCONNECT_CFM );
#endif
}

/* ===========================================================================
 * Function:	HandleHciMsg
 * Description: Main handler for all received HCI messages
 * =========================================================================== */
static void HandleHciMsg(const HCI_Event *m) BT_LARGE
{
	BT_U8 BT_RAM n;
	Connections * BT_RAM con;
	BT_Result		BT_RAM res = BT_CONGESTED;
//	PADAPT_ConnectRsp	BT_RAM rsp;


	switch ( m->header.primitive )
	{
	case HCI_CONNECTION_REQUEST_EVENT:
		break;
	
	case HCI_CONNECTION_COMPLETE_EVENT:
		 break;
	case HCI_DISCONNECT_STATUS_EVENT:
		break;

	case HCI_DISCONNECTION_COMPLETE_EVENT:
		/*
		 * Notify user if disconnection */
		
		if ( (con = GetConnectionConHnd(HCI_LINK_TYPE_SCO_CONNECTION, 
				m->disconnectionCompleteEvent.connectionHandle)) == 0 )
		{
				
				// Just drop
				 
		}
		else if ( (m->disconnectionCompleteEvent.status == HCI_OK)
                          && con->connectionState != WLPTT_DISCONNECTED_STATE )
		{
                  	con->aclConnectionHandle	= INVALID_CONNECTION_HANDLE;
	                con->remoteReadyToReceive	= BT_FALSE;
	                con->connectionState		= WLPTT_DISCONNECTED_STATE;

			//SendSppDisconnectCfm(con, BT_TRUE);
		}
               
		else
		{
			
			 // Just drop
			 
		}
		 
		break;


	case HCI_CONTROLLER_RESTARTED_IND :
		BT_TimeoutCancelAll( SPP_DEV_MODULE_ID );
		returnedDataCongestion = BT_FALSE;
		/*
		 * Reset module state and respond to HCI
		 */
		for (n = 0; n < SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS; n++)
		{
			/*
			 * Try to send a negative confirm. The userRequestInProgress will
			 * take care if the confirm is sent or not
			 */
			RestartedCfmUserCommand(&connections[n]);

			ChangeConnectionState(&connections[n],	WLPTT_DISCONNECTED_STATE );

		}

		SendRestartResponse();
		break;

	default:
		break;
	}
        res = res; // CCMPD01410054 (JJ 8/11/2010)
}

/* ===========================================================================
 * Function:	HandleSdapSearchCompleteRsp
 * Description: Handle SDAP_SEARCH_COMPLETE_RSP
 * =========================================================================== */
static void HandleSdapSearchCompleteRsp(void) BT_LARGE
{
	Connections * BT_RAM con;
	SPP_DEV_ConnectInd BT_RAM ind;

	/*
	 * If service search was performed as part of an outgoing
	 * connection, continue connection setup sequence
	 */
	if ( (con = GetConnectionState(WLPTT_SERVICE_SEARCH_STATE)) != 0 )
	{
		ChangeConnectionState(con, WLPTT_SERVICE_SEARCH_COMPLETE_STATE);
		RunConnectionSetupStateMachine(con);
	}
	else if ( (con = GetConnectionState(INCOMING_WLPTT_CONNECT_REQUEST_STATE)) != 0 )
	{
		ChangeConnectionState(con, INCOMING_WLPTT_W_4_CONNECT_RSP);

		/*
		 * Notify user
		 */
		memcpy( ind.bdAddr, con->wlPTTBdAddr, sizeof( BT_BdAddr ) );
		ind.framesize	= con->framesize;
		ind.handle		= con;
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
		SEND_TO_USER(&ind, SPP_DEV_CONNECT_IND);
#endif
	}
}


/* ===========================================================================
 * Function:	HandleSdapServiceSearchAttrRsp
 * Description: Extract attribute values from the search result
 *				returned by SDAP
 * =========================================================================== */
static void HandleSdapServiceSearchAttrRsp(
								const SDAP_ServiceSearchAttrRsp *rsp) BT_LARGE
{
	SPP_DEV_Position 		BT_RAM position;
	SPP_DEV_ServiceRecord	BT_RAM sppDevServiceRecord;
	Connections 		*	BT_RAM con;

	if (	(con = GetConnectionBdAddr(rsp->bdAddr)) != 0
		&&	rsp->result == SDAP_OK )
	{
		position = SPP_DEV_StartInterpretSearchResult(rsp->dataBuffer);

		while ( position != 0 )
		{
			if ( SPP_DEV_InterpretSearchResult(rsp->dataBuffer,
											  rsp->noOfBytes,
											&sppDevServiceRecord,
											&position) != BT_OK )
			{
				/*
				 * Drop and take the next one if any
				 */
			}
			else if ( !(sppDevServiceRecord.validAttributes & SPP_DEV_VALID_SERVICE_CLASS_ID_LIST) )
			{
				/*
				 * The service class id is missing even if it's mandatory.
				 * Use the information and hope it's a serial port.
				 */
				con->wlPTTServerChannel = sppDevServiceRecord.serverChannel;

			}
			else if ( sppDevServiceRecord.serviceClassIdList[0] == 0x1101 )
			{
				/*
				 * The service class id is present and it's a serial port unit
				 */
				con->wlPTTServerChannel = sppDevServiceRecord.serverChannel;
			}
			else
			{
				/*
				 * The service class id is present and it's a audio gateway
				 * Skip and try the next one if any.
				 */
			}
		}
	}
}


/* ===========================================================================
 * Function:	HandleSdapMsg
 * Description: Handle all messages received from SDAP module.
 * =========================================================================== */
static void HandleSdapMsg(const SDAP_Rsp *m) BT_LARGE
{
	switch ( m->header.primitive )
	{
	case SDAP_SERVICE_SEARCH_ATTR_RSP:
		HandleSdapServiceSearchAttrRsp(&m->searchAttrRsp);
		break;

	case SDAP_SEARCH_COMPLETE_RSP:
		HandleSdapSearchCompleteRsp();
		break;
	}
}


/* ===========================================================================
 * Function:	HandlePadaptConnectCfm
 * Description: Handle PADAPT_CONNECT_CFM message
 * =========================================================================== */
static void HandlePadaptConnectCfm(const PADAPT_ConnectCfm *connectCfm) BT_LARGE
{
	Connections * BT_RAM con;

	/*
	 * Find the connection associated with the bdAddr
	 */
	if ( (con = GetConnectionBdAddr(connectCfm->bdAddr)) != 0 )
	{
		ChangeConnectionState(con, PADAPT_CONNECTED_STATE);

		if ( connectCfm->accept )
		{
			con->wlPTTDlci = connectCfm->dlci;
			RestartAtCommandParser(con);

			con->remoteReadyToReceive = BT_FALSE;
			SendConnectCfm(con, SPP_DEV_OK, connectCfm->framesize );
		}

		/*
		 * Save connection handle for future use
		 */
		(void)HCI_GetConnectionHandle(con->wlPTTBdAddr, &con->aclConnectionHandle);

		RunConnectionSetupStateMachine(con);
	}
	else
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Discarding PADAPT_CONNECT_CFM since no valid connection"));
	}
}


/* ===========================================================================
 * Function:	RestartAtCommandParser
 * Description:
 * =========================================================================== */
static void RestartAtCommandParser(Connections *con) BT_LARGE
{
	con->atCommandLength	= 0;
	con->parseState 		= W_4_A;
}

/* ===========================================================================
 * Function:	AddToAtCommandBuffer
 * Description: Add one character to the at command buffer
 * =========================================================================== */
static void AddToAtCommandBuffer(Connections *con, BT_U8 c)
{
	/*
	 * Add to buffer if there is room, last position is always 0
	 */
	if ( con->atCommandLength < SPP_DEV_CFG_AT_COMMAND_MAX_SIZE - 1)
	{
		con->atCommandBuffer[con->atCommandLength++]	= c;
		con->atCommandBuffer[con->atCommandLength]	= '\0';
	}
	else
	{
		LOG_INFO(( SPP_DEV_MODULE_ID,
			"AT command too long, discarding character \'%c\'\n", c));
	}
}
/* ===========================================================================
 * Function:	HandleNibblerByte
 * Description:
 * =========================================================================== */
static void HandleNibblerByte(Connections *con, BT_U8* nibbler, BT_U8 size)
{
	SPP_DEV_NibblerInd		BT_RAM	ind;

	ind.nibbler = nibbler;
	ind.handle	= con;
  ind.size	  = size;

	SEND_TO_USER(&ind, SPP_DEV_NIBBLER_IND);
}
/* ===========================================================================
 * Function:	HandleAtCommand
 * Description:
 * =========================================================================== */
static void HandleAtCommand(Connections *con, char *atCommand)
{
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
	SPP_DEV_AtInd		BT_RAM	ind;
	//char * 		BT_RAM pos; // CCMPD01410054 (JJ 8/11/2010)
        
        ind.details = atCommand;
	ind.handle	= con;

	SEND_TO_USER(&ind, SPP_DEV_AT_IND);
#endif
}


/* ===========================================================================
 * Function:	ParseCharacter
 * Description: Parser for result codes. Looks for pattern
 *				AT .... <CR> When found, a handler for complete
 *				At commands is called.
 * =========================================================================== */
static void ParseCharacter(Connections *con, BT_U8 c) BT_LARGE
{
	switch ( con->parseState )
	{
	case W_4_A:
		if ( c == 'A' || c == 'a' )
		{
			AddToAtCommandBuffer(con, c);
			con->parseState = W_4_T;
		}
		break;

	case W_4_T:
		if ( c == 'T' || c == 't' )
		{
			AddToAtCommandBuffer(con, c);
			con->parseState = W_4_CMD;
		}
		else if ( c == 'A' || c == 'a' )
		{
			RestartAtCommandParser(con);
			AddToAtCommandBuffer(con, c);
			con->parseState = W_4_T;
		}
		else
		{
			RestartAtCommandParser(con);
		}
		break;

	case W_4_CMD:
		if ( c != CR )
		{
			AddToAtCommandBuffer(con, c);
		}
		else
		{
			HandleAtCommand(con, (char*)con->atCommandBuffer);
			RestartAtCommandParser(con);
		}
		break;
	}
}

static void SendSppNibbleByte(Connections *con, BT_U8 *c, BT_U8 size)
{
  memcpy(con->atCommandBuffer, c, size);
  con->atCommandLength = size;

  HandleNibblerByte(con, (BT_U8*)con->atCommandBuffer, (BT_U8)con->atCommandLength);
}
/* ===========================================================================
 * Function:	HandlePadaptDataInd
 * Description: Handler for PADAPT_DATA_IND messages
 * =========================================================================== */
static void HandlePadaptDataInd(const PADAPT_DataInd *dataInd) BT_LARGE
{
	Connections * BT_RAM con;
	BT_U16 BT_RAM i;

	/*
	 * Find the connection associated with the bdAddr
	 * Only one connection per bdAddr is allowed so we don't
	 * have to check the dlci
	 */
        
	if ( (con = GetConnectionBdAddr(dataInd->bdAddr)) != 0 )
	{
		/*
		 * Check if the length of the message > 0.
		 * If the length == 0 => message contains credits =>
		 * do nothing. (PADAPT takes care of the credits)
		 */
    if ( dataInd->bufferLength > 0 )
    {
      if( nibbler_get_enable_status(&wirelessPTT_nibbler) == NIBBLER_ENABLE )
      {  
          SendSppNibbleByte(con, dataInd->dataBufferPtr, dataInd->bufferLength);
      }
      else
      {
        for ( i = 0; i < dataInd->bufferLength; i++ )
        {
          ParseCharacter(con, dataInd->dataBufferPtr[i]);
        }
      }
                }
	}
        else
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Discarding PADAPT_DATA_IND since no valid connection"));
	}
}


/* ===========================================================================
 * Function:	SendPadaptConnectRsp
 * Description: Try to send a PADAPT_CONNECT_RSP, continue with service
 *				level connection setup if successful.
 * Returns: 	BT_OK, BT_CONGESTED or BT_CALL_ERROR
 * =========================================================================== */
static BT_Result SendPadaptConnectRsp(Connections *con, BT_BOOL accept) BT_LARGE
{
	BT_Result			BT_RAM res = BT_CONGESTED;
	PADAPT_ConnectRsp	BT_RAM rsp;


	rsp.header.sender		= SPP_DEV_MODULE_ID;
	rsp.header.primitive	= PADAPT_CONNECT_RSP;
	rsp.header.receiver 	= BT_PADAPT;

	rsp.dlci				= con->wlPTTDlci;
	rsp.accept				= accept;
	memcpy(rsp.bdAddr, con->wlPTTBdAddr, sizeof(BT_BdAddr) );

	if ( !VerifyConnectionHandle(con) )
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Invalid connection handle"));
		res = BT_CALL_ERROR;
	}
	/*
	 * If the sending the command was not delayed, continue with connect
	 * sequence
	 */
	else
	{
		res = SPP_DEV_SEND_REQ((BT_Msg*)&rsp);
		
		/*
		 * If connection is terminated while waiting for the users 
		 * connect response. PADAPT will return BT_CALL_ERROR
		 */
		if ( res == BT_CALL_ERROR 
			&& con->connectionState == INCOMING_WLPTT_W_4_CONNECT_RSP )
		{			
			FreeConnection(con);
		}
		else if ( res == BT_CONGESTED )
		{
			/*
			 * Return congested and try later
			 */
		}
		/*
		 * The message was sent
		 */
		else if ( accept )
		{
			ChangeConnectionState(con, PADAPT_CONNECTED_STATE);
			RunConnectionSetupStateMachine(con);
		}
		else
		{
			FreeConnection(con);
		}
	}

	return res;
}


/* ===========================================================================
 * Function:	SendPadaptConnectRspNegative
 * Description: 
 * Returns: 	
 * =========================================================================== */
static void SendPadaptConnectRspNegative(void)
{
	PADAPT_ConnectRsp	BT_RAM rsp;

	rsp.header.sender		= SPP_DEV_MODULE_ID;
	rsp.header.primitive	= PADAPT_CONNECT_RSP;
	rsp.header.receiver 	= BT_PADAPT;

	rsp.accept				= BT_FALSE;
	memcpy(rsp.bdAddr, rejectConnectInfo.bdAddr, sizeof(BT_BdAddr) );
	rsp.dlci				= rejectConnectInfo.dlci;
	
	if ( SPP_DEV_SEND_REQ((BT_Msg*)&rsp) == BT_CONGESTED )
	{
		rejectConnectInfo.timer= 
			BT_TimeoutReq(SPP_DEV_MODULE_ID, 
						TIMER_ID_RESEND_REJECT_CONNECT,
						0, 
						SPP_DEV_CFG_RESEND_TIME );
	}
}


/* ===========================================================================
 * Function:	HandlePadaptConnectInd
 * Description: Notify user of connection request or discard it
 * =========================================================================== */
static void HandlePadaptConnectInd(const PADAPT_ConnectInd *connectInd) BT_LARGE
{
	Connections * BT_RAM con;

	/*
	 * Only one connection per bdAddr is allowed
	 */
	if ( (con = GetConnectionBdAddr(connectInd->bdAddr)) != 0 )
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Only one connection per bdAddr is allowed"));

		if ( rejectConnectInfo.timer == 0 )
		{
			LOG_INFO((SPP_DEV_MODULE_ID, "Rejecting connection attempt, " ));

			memcpy( rejectConnectInfo.bdAddr, connectInd->bdAddr, 6 );
			rejectConnectInfo.dlci = connectInd->dlci;

			SendPadaptConnectRspNegative();
		}
		else
		{
			LOG_INFO((SPP_DEV_MODULE_ID, "Discarding connection attempt, " ));
		}
	}

	/*
	 * Try to find a spare connection. Discard if everyone
	 * is allocated.
	 */
	else if ( (con = NewConnection(connectInd->bdAddr)) == 0 )
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Discarding connection attempt, "
						"no resources left"));
	}
	/*
	 * Start an service search attribute request before notifying the user
	 * and there fore complete the RFCOMM channel setup.
	 */
	else
	{
		con->wlPTTDlci 	= connectInd->dlci;
		con->framesize	= connectInd->framesize;

		StartHandlingIncomingConnectionAttempt(con, connectInd->bdAddr);
	}
}


/* ===========================================================================
 * Function:  HandlePadaptDisconnectInd
 * Description: Update state and notify registered user
 * =========================================================================== */
static void HandlePadaptDisconnectInd(const PADAPT_DisconnectInd *disconnectInd) BT_LARGE
{
	Connections * BT_RAM con;

	/*
	 * Find the connection associated with the bdAddr
	 * Only one connection per bdAddr is allowed so we don't
	 * have to check the dlci
	 */
	if ( (con = GetConnectionBdAddr(disconnectInd->bdAddr)) != 0 )
	{
		ChangeConnectionState(con, WLPTT_DISCONNECTED_STATE);

		RestartedCfmUserCommand(con);

		/*
		 * Cancel possibly running timers
		 */
		CancelTimers(con);

		/*
		 * Notify the user about the disconnection
		 */
		SendLinkStatusInd(con);
                //SendSppDisconnectCfm(con, BT_TRUE);
	}
	else
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Discarding PADAPT_DISCONNECT_IND since no valid connection"));
	}
}


/* ===========================================================================
 * Function:	HandlePadaptFlowInd
 * Description: Send the flow indication to the user
 * ==========================================================================*/
static void HandlePadaptFlowInd(const PADAPT_FlowInd *flowInd)
{
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
	SPP_DEV_FlowInd	BT_RAM ind;
#endif
	Connections *	BT_RAM con;

	/*
	 * Find the connection associated with the bdAddr
	 * Only one connection per bdAddr is allowed so we don't
	 * have to check the dlci
	 */
	if ( (con = GetConnectionBdAddr(flowInd->bdAddr)) != 0 )
	{
		con->remoteReadyToReceive	= flowInd->remoteReadyToReceive;
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
		ind.remoteReadyToReceive	= con->remoteReadyToReceive;
		ind.handle			= con;
                memcpy(ind.bdAddr, con->wlPTTBdAddr, sizeof(BT_BdAddr) );
		SEND_TO_USER(&ind, SPP_DEV_FLOW_IND );
#endif
	}
	else
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Discarding PADAPT_FLOW_IND since no valid connection"));
	}
}

/* ===========================================================================
 * Function:	HandlePadaptConfigInd
 * Description:
 * ==========================================================================*/
static void HandlePadaptConfigInd(const PADAPT_ConfigInd *configInd)
{
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
	SPP_DEV_ConfigInd	BT_RAM ind;
	Connections 	*	BT_RAM con;
#endif
	/*
	 * Find the connection associated with the bdAddr
	 * Only one connection per bdAddr is allowed so we don't
	 * have to check the dlci
	 */
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
	if ( (con = GetConnectionBdAddr(configInd->bdAddr)) != 0 )
#else
        if ( (GetConnectionBdAddr(configInd->bdAddr)) != 0 )
#endif
	{
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
		ind.framesize	= configInd->framesize;
		ind.handle		= con;

		//SEND_TO_USER(&ind, SPP_DEV_CONFIG_IND );
                ind=ind;  // CCMPD01410054 (JJ 8/11/2010), solving compiler warning
#endif
	}
	else
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "Discarding PADAPT_CONFIG_IND since no valid connection"));
	}
}

/* ===========================================================================
 * Function: 	HandlePadaptNotCongestedInd
 * Description: Handles PADAPT_NOT_CONGESTED_IND
 * Returns:		-
 * =========================================================================== */
static void HandlePadaptNotCongestedInd(void)
{
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
//	SPP_DEV_NotCongestedInd 	BT_RAM ind;
#endif

	/* Only send notification if there has been a data congestion */
	if ( returnedDataCongestion )
	{
		returnedDataCongestion = BT_FALSE;
#ifndef DISABLE_SEND_TO_USER  // CCMPD01410054 (JJ 8/11/2010)
		//SEND_TO_USER(&ind, SPP_DEV_NOT_CONGESTED_IND);
#endif
	}
}



/* ===========================================================================
 * Function:	HandlePadaptMsg
 * Description: Handle all messages received from PADAPT.
 * =========================================================================== */
static void HandlePadaptMsg(const PADAPT_IndConf *m) BT_LARGE
{
	switch ( m->header.primitive )
	{
	case PADAPT_CONNECT_CFM:
		HandlePadaptConnectCfm(&m->connectCfm);
		break;

	case PADAPT_CONFIG_IND:
		HandlePadaptConfigInd(&m->configInd );
		break;

	case PADAPT_CONNECT_IND:
		HandlePadaptConnectInd(&m->connectInd );
		break;

	case PADAPT_DISCONNECT_IND:
		HandlePadaptDisconnectInd(&m->disconnectInd);
		break;

	case PADAPT_DATA_IND:
		HandlePadaptDataInd(&m->dataInd);
		break;

	case PADAPT_FLOW_IND:
		HandlePadaptFlowInd(&m->flowInd);
		break;

	case PADAPT_DM_SENT_IND:
		if ( SPP_DEV_CFG_DISCONNECT_ACL_AFTER_DM_TIME )
		{
			memcpy( disconnectAclBdaddr, m->dmSentInd.bdAddr, 6 );
			
			(void)BT_TimeoutReq(SPP_DEV_MODULE_ID, TIMER_ID_DM_DISCONNECT_ACL,
					0, SPP_DEV_CFG_DISCONNECT_ACL_AFTER_DM_TIME );
		}
		break;
		
	case PADAPT_NOT_CONGESTED_IND:
		HandlePadaptNotCongestedInd();
		break;

	default:
		LOG_INFO((SPP_DEV_MODULE_ID, "PADAPT primitive 0x%02x does not exist", m->header.primitive));
		break;
	}
}


/* ===========================================================================
 * Function:	HandleTsrvMsg
 * Description: Handle messages from Time Server.
 * =========================================================================== */
static void HandleTsrvMsg(const BT_TimeoutInd *m) BT_LARGE
{
	/*
	 * m->userTimerData might be zero (m->userTimerId equal to 
	 * TIMER_ID_RESTART_RESPONSE) but then is not con used.
	 */
	Connections *	BT_RAM con = (Connections *)m->userTimerData;

	/*
	 * Dispatch to the appropriate handler functionm
	 */
	switch ( m->userTimerId )
	{
	case TIMER_ID_RUN_CONNECTION_STATE_MACHINE:
		con->connectTimer = 0;
		RunConnectionSetupStateMachine(con);
		break;

	case TIMER_ID_WLPTT_DISCONNECTED_STATE_IND:
		con->connectTimer = 0;
		SendLinkStatusInd(con);
		break;

	case TIMER_ID_RESTART_RESPONSE:
		restartedRspTimer = 0;
		SendRestartResponse();
		break;


	case TIMER_ID_PADAPT_CONNECT_REQ:
		con->connectTimer = 0;
		SendPadaptConnectReq(con);
		break;

	case TIMER_ID_SDAP_SEARCH:
		con->sdapTimer = 0;
		SendSdapServiceSearchAttrReq(con);
		break;

	case TIMERID_RESEND_LINK_SETTINGS:
		con->connectTimer = 0;
		SendLinkSettingsReq(con);
		break;

	case TIMER_ID_DM_DISCONNECT_ACL:
		SendAclDisconnect( disconnectAclBdaddr );
		break;
		
	case TIMER_ID_SEND_LINK_STATUS_IND:
		con->sdapTimer = 0;
		SendLinkStatusInd(con);
		break;

	case TIMER_ID_RESEND_REJECT_CONNECT:
		rejectConnectInfo.timer = 0;
		SendPadaptConnectRspNegative();
		break;
   			
   			
   default: break;
                
	}
}


/* ===========================================================================
 * Function:	HandleLinkHandlerMsg
 * Description:
 * =========================================================================== */
static void HandleLinkHandlerMsg(const HCI_LH_IndConf *ind)
{
	Connections * BT_RAM con;

	switch (ind->header.primitive )
	{
	case HCI_LH_LINK_SETTINGS_CFM:
		if ( (con = GetConnectionConHnd(HCI_LINK_TYPE_ACL_CONNECTION, ind->linkSettingsCfm.connectionHandle)) != 0 )
		{
			if ( !ind->linkSettingsCfm.success )
			{
				LOG_INFO((SPP_DEV_MODULE_ID, "Failed to configure link handling"));
			}

			//ChangeConnectionState(con, W_4_CONNECTION_IS_UP );
			ChangeConnectionState(con, WLPTT_ACL_CONNECTED_STATE );
		}
		break;

	}
}


/* ===========================================================================
 * Function:	Initialise
 * Description: Handle initialise startup phase
 * =========================================================================== */
static void Initialise(void) BT_LARGE
{
	BT_U16 BT_RAM fatalError = 0;
	BT_U8  BT_RAM n = 0;

	if ( (fatalError = SPP_DEV_OptInitialise()) != 0 )
	{
		/* Just quit */
	}
	else if ( BT_LookupProfile(BT_HCI_LH) == 0 )
	{
		fatalError = SPP_DEV_ERR_HCI_LH_MISSING;
	}
	else if ( (connections = (Connections*)
				BT_MALLOC(
					SPP_DEV_MODULE_ID,
					SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS*sizeof(Connections))) == 0	)
	{
		fatalError = SPP_DEV_ERR_MAX_NO_OF_WLPTT_CONNECTIONS;
	}
	else
	{
		/*
		 * Se comments at top of file about timer variables for an
		 * explanation on number of timers needed
		 *
		 * 3 per connection + rejectConnectTimer
		 */
		BT_AllocateTimers(SPP_DEV_MODULE_ID, 1 + 3*SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS);
		rejectConnectInfo.timer = 0;
		
		/*
		 * Allocate a command buffer and init each connection
		 */
		while ( 	n < SPP_DEV_CFG_MAX_NO_OF_WLPTT_CONNECTIONS
			   &&	!fatalError )
		{
			if ( (connections[n].atCommandBuffer = (BT_U8*)
						BT_MALLOC(
							SPP_DEV_MODULE_ID,
							SPP_DEV_CFG_AT_COMMAND_MAX_SIZE*sizeof(BT_U8))) == 0  )
			{
				fatalError = SPP_DEV_ERR_AT_COMMAND_MAX_SIZE;
			}

			FreeConnection(&connections[n]);

			n++;
		}
	}
	/*
	 * If an error occured, report it to controlling app
	 */
	if ( fatalError )
	{
		BT_FatalError(fatalError);
	}
}


/* ===========================================================================
 * Function:	SPP_DEV_RegisterService
 * Description: 
 * =========================================================================== */
BT_BOOL SPP_DEV_RegisterService(BT_ModuleId user,
								const char *securityServiceName,
				RFCOMM_ServerChannel *rfcommServerChannel) BT_LARGE
{
	BT_BOOL BT_RAM res = BT_TRUE;

	if ( sppDevUser )
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "User of SPP is already registered"));
		res = BT_FALSE;
	}
	else
	{
		sppDevUser = user;

		/*
		 * Set local server channel
		 */
		if ( *rfcommServerChannel == 0
			&& !PADAPT_NextFreeServerChannel( rfcommServerChannel ) )
		{
				LOG_INFO((SPP_DEV_MODULE_ID, "Could not get free server channel from PADAPT"));
		}

		/*
		 * Registration at Security manager.
		 */
		else if ( SEC_RegisterIncomingService( securityServiceName,
									SPP_DEV_CFG_SECURITY_LEVEL,
									SEC_ID_RFCOMM,
									*rfcommServerChannel ) != BT_OK)
		{
			LOG_INFO((SPP_DEV_MODULE_ID, "Could not register incoming service with SEC"));
		}

		else if ( SEC_RegisterOutgoingService( securityServiceName,
									SPP_DEV_CFG_SECURITY_LEVEL,
									SEC_ID_RFCOMM,
									BT_SPP_DEV ) != BT_OK)
		{
			LOG_INFO((SPP_DEV_MODULE_ID, "Could not register outgoing service with SEC"));
		}

		/*
		 * Register service with PADAPT
		 */
		else if ( !PADAPT_RegisterService(SPP_DEV_MODULE_ID,
									*rfcommServerChannel,
									PADAPT_INTERFACE_TYPE,
									SPP_DEV_CFG_MIN_FRAME_SIZE,
									SPP_DEV_CFG_MAX_FRAME_SIZE,
									SPP_DEV_CFG_INITIAL_CREDITS,
									SPP_DEV_CFG_CREDITS_TO_SEND,
									SPP_DEV_CFG_MIN_CREDITS) )
		{
			LOG_INFO((SPP_DEV_MODULE_ID, "Register service with PADAPT failed\n"));
		}

		/*
		 * Register with HCI Driver and subscribe to HCI events
		 */
		else if ( SetupHciUsage() )
		{
			/*
			 * Store the server channel so it will be possible
			 * to unregister.
			 */
			localServerChannel = *rfcommServerChannel;
			res = BT_TRUE;
		}
		else
		{
			/*
			 * Just skip
			 */
		}
	}
        res = res;  // CCMPD01410054 (JJ 8/11/2010)
	return 1;
}

/* ===========================================================================
 * Function:	SPP_DEV_UnregisterService
 * Description: 
 * =========================================================================== */
BT_BOOL SPP_UnregisterService(BT_ModuleId user) BT_LARGE
{
	BT_BOOL BT_RAM res = BT_FALSE;

	if ( sppDevUser != 0 && sppDevUser == user )
	{
		BT_TimeoutCancelAll(SPP_DEV_MODULE_ID);
		(void)HCI_UnregisterUser(SPP_DEV_MODULE_ID);
		(void)PADAPT_UnregisterService(localServerChannel);

		(void)SEC_UnregisterOutgoingService(
						SEC_ID_RFCOMM,
						BT_SPP_DEV );

		(void)SEC_UnregisterIncomingService(
									SEC_ID_RFCOMM,
									localServerChannel);
		sppDevUser = 0;
		res = BT_TRUE;
	}
	else
	{
		LOG_INFO((SPP_DEV_MODULE_ID, "SPP_DEV_UnregisterService failed"));
	}

	return res;
}


