/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013-2014 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: test_commands_manager.c
#
# --------------------------- General Description -----------------------------
# This file contains the definitions of the Test Commands Manager.
#
# --------------------------- HEADER FILE INCLUDES ---------------------------*/

#include <stdio.h>
#include <string.h>
#include "FreeRTOS.h"
#include "usb_standard_request.h"
#include "debug_manager.h"
#include "test_commands_manager.h"
#include "usb_lib.h"
#include "osal.h"
#include "TaskConfig.h"
#include "gpio.h"
#include "flashc.h"
#include "ConnectionManager.h"
#include "Interchip.h"
#include "initialization.h"
#include "controls.h"
#include "version_string.h"
#include "config.h"
#include "adc.h"
#include "codeplug_ui.h"
#include "bluetooth_manager.h"
#include "ams3930_driver.h"
#include "xnl.h"

#ifdef MPP_TASK
#include "mpp_init.h"
#include "mpp_msg.h"
#endif

typedef struct test_command_t 
{
    UINT16_T opcode;
    UINT8_T chipset;
    UINT16_T trans_id;
    UINT16_T message_size;
    UINT8_T message[TC_MAX_SIZE];
    UINT16_T crc;
} test_command_t;


/*******************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
* AUTHOR                  Date Modified Tracking Number     Description
* Phong Tran              03/18/2010    CCMPD01316656       Initial Creation
* Phong Tran              04/30/2010    CCMPD01339858       CSR DFU usb disconnect, Microphone, Speaker, Access Levels  Button Test Mode
* Aneeket Patkar          06/04/2010    CCMPD01356813       Typecast and Extern edit for debug functions.
* Phong Tran              06/04/2010    CCMPD01355429       BT Addr, LED
* Phong Tran              06/18/2010    CCMPD01331569       Implemented codeplug read/write.
* YewFoong Lim            06/28/2010    CCMPD01373300       Initial draft for non-secure wireless dongle R1.2
* PeiSee Toh              08/04/2010    CCMPD01376817       Cleanup of Parse Test Commands Function  
* Sevuhan Ramanathan      08/04/2010    CCMPD01376817       Added LED control Through Atmel GPIO under MSG_LED 
* YewFoong Lim            09/06/2010    CCMPD01384903       Updated the variable for the Connection Manager StateMachine
* Tan Seng Kai		  09/06/2010    CCMPD01388039       Change Task message sending from interchip to bluetooth manager
* Tan Seng Kai		  09/07/2010    CCMPD01388047       Change the delay time for USB switching back to previous value
* PeiSee Toh              09/08/2010    CCMPD01390094       Added ACMP Test Command to support LED and Atmel Version Display Factory Test
* PeiSee Toh              09/29/2010    CCMPD01397210       Catchup from latest 2.0 firmware, I08.00.00, and do the proper cleanup for Parse Test Commands Function
* PeiSee Toh              10/07/2010    CCMPD01399521       Change the codeplug field in MSG_CP_READ Acmp msg
* Tan Seng Kai            10/06/2010    CCMPD01399559       RADIO_INTERFACE_MANAGER_TASK,MPP_TASK,NIBBLER_TASK Compile sw 
*                                                               added to remove Radio interface and mpp task
* JJ Low                  11/30/2010    CCMPD01446399       Add Test Commander Pairing Mode command
* YK - JWPM67             Dec/14/2010   CCMPD01453712       Send message to controls_manager_task to update atmel_test_mode.
* PeiSee Toh              11/24/2010    CCMPD01419110       Change the maximum index of MSG_CP_READ from "DONGLE_DISCOVERABLE" to "MAX_CODEPLUG_ITEMS"
* Mahes                   Jan/06/2011   CCMPD01460340       Software catch-up to dongle R1.2 version R01.00.00
* Phong Tran              12/06/2013    CCMPD01837173       Added support for XNL/XCMP simulator.
* Abhishek Trivedi        04/04/2014    CCMPD01882874       CSR Version check
* Phong Tran              05/12/2014    CCMPD01885130       Modified to support XXT device
*--------------------------- End of History Template----------------------------
* *****************************************************************************/

/*==============================================================================
                                      EXTERNAL DEFINITIONS
==============================================================================*/
extern xQueueHandle hsp_xnl_xcmp_data_rx_Q;
extern UINT16_T device_1_id;
extern xnl_init_parameters_t hsp_xnl_init_params;

/*==============================================================================
                                    GLOBAL VARIABLES
==============================================================================*/
xQueueHandle data_queue;
xSemaphoreHandle tc_mgr_event;
tc_flags_t tc_flags;
versionType BuildVersion;
tc_button_states_t tc_button_states;

UINT8_T *factory_password = "Factory";
UINT8_T *developer_password = "Developer";

/*==============================================================================
                                      EXTERNAL FUNCTIONS
==============================================================================*/
extern void usb_putstr(UINT8_T* pStr);
extern void test_commands_queue_task(void *pvParameters);
extern cpu_delay_ms(ULONG32_T delay, ULONG32_T fcpu_hz);
extern void usb_disconnect();
extern success_failure_t send_hsp_xnl_msg(UINT8_T *data_tx_ptr, UINT16_T data_tx_length);
extern UINT16_T xnl_xcmp_new_device_register (xnl_init_parameters_t device_param);

/*==============================================================================
                                      FUNCTION PROTOTYPES
==============================================================================*/
void test_commands_manager_init(void);
void test_commands_manager_task(void *pvParameters);
void parse_test_command(UINT8_T *buffer, UINT16_T *count);
UINT16_T update_crc(UINT16_T crc, UINT8_T data);
void send_acmp_reply(test_command_t *test_command, test_command_result_t result);
void send_button_broadcast(UINT8_T chipset, UINT8_T button_id, UINT8_T state);
UINT16_T calculate_request_crc(test_command_t *test_command, UINT16_T crc);   
UINT16_T calculate_reply_crc(test_command_t *test_command, test_command_result_t result, UINT16_T crc);   
test_command_result_t buffer_read_u8(UINT8_T *buffer, UINT16_T *index, UINT16_T *count, UINT8_T *data);
test_command_result_t buffer_read_u16(UINT8_T *buffer, UINT16_T *index, UINT16_T *count, UINT16_T *data);
test_command_result_t buffer_read_u32(UINT8_T *buffer, UINT16_T *index, UINT16_T *count, ULONG32_T *data);
test_command_result_t buffer_read_u64(UINT8_T *buffer, UINT16_T *index, UINT16_T *count, ULONG64_T *data);
UINT8_T array_read_u8(UINT8_T *arr, UINT8_T data);
void array_write_u8(UINT8_T *arr, UINT8_T data);
UINT16_T array_read_u16(UINT8_T *arr, UINT16_T data);
void array_write_u16(UINT8_T *arr, UINT16_T data);
ULONG32_T array_read_u32(UINT8_T *arr, ULONG32_T data);
void array_write_u32(UINT8_T *arr, ULONG32_T data);
void usb_write_u8(UINT8_T data);
void usb_write_u16(UINT16_T data);
void usb_write_u32(ULONG32_T data);


/*==============================================================================
                                      FUNCTION DEFINITIONS
==============================================================================*/

/*=============================================================================
	FUNCTION: test_commands_manager_init

	DESCRIPTION: Inititize the task for the Test Commands Manager
	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/

void test_commands_manager_init(void)
{
    vr_create_task(USB_TEST_CMD_MGR); 
    vr_create_task(USB_TEST_CMD_QUEUE); 
  
    /* Create counting semaphore */
    tc_mgr_event = xSemaphoreCreateCounting(TC_SEMPHR_MAXCOUNT, TC_SEMPHR_INITCOUNT);

    /* Create the generic debug data queue */
    data_queue = xQueueCreate(TC_DATA_QUEUE_LENGTH, (ULONG32_T) sizeof(UINT8_T));
    
    /* Initialize flags */
    tc_flags.debug_enable = FALSE;
    tc_flags.version_match = FALSE;
    tc_flags.button_test_mode = TC_OFF;
    tc_flags.access_level = TC_NO_ACCESS;
    tc_flags.atmel_test_mode = TEST_OFF;
    tc_flags.simulator_enable = FALSE;
    tc_flags.simulator_type = TC_SIM_HOST;
    
    /* Initialize button states */
    tc_button_states.ptt.atmel = TC_RELEASED;
    

}

/*=============================================================================
	FUNCTION: test_commands_manager_task

	DESCRIPTION: The task of the Test Commands Manager
	ARGUMENTS PASSED: pvParameters

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/

void test_commands_manager_task(void *pvParameters)
{
    UINT8_T rx_buffer[TC_RX_BUFFER_LENGTH];
    UINT16_T rx_buf_count;
    
    portTickType xLastWakeTime;
    xLastWakeTime = xTaskGetTickCount();
     
    while (true)
    {
        vTaskDelayUntil(&xLastWakeTime, TC_TASK_DELAY);
    
        /* First, check the device enumeration state */
        if (!Is_device_enumerated()) continue;
           
        /* Receive incoming messages */
        rx_buf_count = 0;
        while (usb_test_hit())  // CCMPD01400441_ckpd36_Framework_V2  (4/10/2010)
        {  
            rx_buffer[rx_buf_count++] = usb_getchar();
        }

        /* Process incoming messages */
        if (rx_buf_count > 0)
        {
            if (tc_flags.simulator_enable)
            {
                if (tc_flags.simulator_type == TC_SIM_HOST)
                {
                    send_hsp_xnl_msg(rx_buffer, rx_buf_count);
                }
                else
                {
                    for (int i = 0; i < rx_buf_count; i++)
                    {
                        xQueueSend(hsp_xnl_xcmp_data_rx_Q, &rx_buffer[i], 0); 
                    }
                }
            }
            else
            {
                parse_test_command(rx_buffer, &rx_buf_count);
            }
        } 
    }
}

/*=============================================================================
	FUNCTION: parse_test_command

	DESCRIPTION: The function parses the test command from the pc.
	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/

void parse_test_command(UINT8_T *buffer, UINT16_T *count)
{
    UINT16_T crc;
    test_command_t test_command = {0};
    void* message; //wrn637: CCMPD01388039
    
    UINT16_T index = 0;
    
    while (*count > 0)
    {      
        /* Reset test command message size */
        test_command.message_size = 0;
      
        /* Retrieve the opcode (2 bytes) */
        if (buffer_read_u16(buffer, &index, count, &test_command.opcode) == TC_FAILURE)
        {
            /* Send ACMP Reply */
            send_acmp_reply(&test_command, TC_FAILURE);
            
            return;
        }
                
        if (test_command.opcode == ACMP_REQUEST)
        {      
            /* Retrieve Chipset */
            if (buffer_read_u8(buffer, &index, count, &test_command.chipset) == TC_FAILURE)
            {
                /* Send ACMP Reply */
                send_acmp_reply(&test_command, TC_INVALID_PARAMS);  
                
                return;
            }

            /* Retrieve the Transaction ID (2 bytes) */
            if (buffer_read_u16(buffer, &index, count, &test_command.trans_id) == TC_FAILURE)
            {
                /* Send ACMP Reply */
                send_acmp_reply(&test_command, TC_INVALID_PARAMS);  
                
                return;
            }
    
            /* Retrieve the Message Size (2 bytes) */
            if (buffer_read_u16(buffer, &index, count, &test_command.message_size) == TC_FAILURE)
            {
                /* Send ACMP Reply */
                send_acmp_reply(&test_command, TC_INVALID_PARAMS);  
                
                return;
            }
          
            /* Check test command size */
            if (test_command.message_size > TC_MAX_SIZE)
            {
                /* Send ACMP Reply */
                send_acmp_reply(&test_command, TC_MAX_SIZE_EXCEEDED);  
                
                return;
            }
            else
            {
                for (UINT16_T i = 0; i < test_command.message_size; i++)
                {
                    if (buffer_read_u8(buffer, &index, count, &test_command.message[i]) == TC_FAILURE)
                    {
                        /* Send ACMP Reply */
                        send_acmp_reply(&test_command, TC_INVALID_PARAMS);  
                        
                        return;
                    }
                }
            }
                                        
            /* Retrieve the crc (2 bytes) */
            if (buffer_read_u16(buffer, &index, count, &test_command.crc) == TC_FAILURE)
            {
                /* Send ACMP Reply */
                send_acmp_reply(&test_command, TC_INVALID_PARAMS);  
                
                return;
            }
    
            /* Calculate crc */
            crc = calculate_request_crc(&test_command, crc);
            
            /* Compare the crc */
            if (crc == test_command.crc)
            {
                if (test_command.message_size > 0)
                {
                    /* Check the message id */
                    switch (test_command.message[0])
                    {
                   
                      case MSG_ENABLE_ACCESS:
                        {
                            UINT8_T mode = 0, pw_size = 0, *pw;
                            
                            /* Retrieve mode */
                            mode = array_read_u8(test_command.message + 1, mode);
                            
                            /* Retrieve password size */
                            pw_size = array_read_u8(test_command.message + 2, pw_size);
                                           
                            /* Update total message size */
                            test_command.message_size = 2;
                            
                            if (mode == TC_FACTORY_ACCESS)
                            {
                                pw = factory_password;
                            }
                            else if (mode == TC_DEVELOPER_ACCESS)
                            {
                                pw = developer_password;
                            }
                            else 
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_MODE);
                                
                                return;
                            }                             
                            
                            /* Compare size */
                            if (pw_size != strlen((char const *)pw))
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_PASSWORD);
                                
                                return;
                            }
                            
                            /* Compare password */
                            if ((pw_size == strlen((char const *)test_command.message + 3)) && strcmp((char const *)pw, (char const *)test_command.message + 3) == 0)
                            {
                                /* Change access level */
                                if (mode == TC_FACTORY_ACCESS)
                                {
                                    tc_flags.access_level = TC_FACTORY_ACCESS;
                                }
                                else
                                {
                                    tc_flags.access_level = TC_DEVELOPER_ACCESS;
                                }
                              
                                /* Send successful ACMP Reply */
                                send_acmp_reply(&test_command, TC_SUCCESS);                                            
                            }                                
                            else
                            {
                                 /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_PASSWORD);
                            }                                                               
   
                            break;
                        }
                    
                        case MSG_CP_READ:
                        {
                            /* Check access level */
                            if (tc_flags.access_level == TC_NO_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }
                            
                            UINT16_T j = 1;
                            cplg_param_t cp_value;
                            
                            for (UINT16_T index = PAIRING_METHOD; index < MAX_CODEPLUG_ITEMS; index++)
                            {
                                if (codeplugGetParameter(index, &cp_value, 0) == CODEPLUG_OK)
                                {
                                    test_command.message[j] = index;
                                    array_write_u32(test_command.message + j + 1, cp_value);
                                    j += 5;
                                }
                                else
                                {
                                    /* Send ACMP Reply */
                                    send_acmp_reply(&test_command, TC_CODEPLUG_ERROR);
                                    
                                    return;                            
                                }
                            }
                                                                          
                            /* Update total message size */
                            test_command.message_size += (j - 1);
                            
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                            
    
                            break;
                        }
                        
                        case MSG_CP_WRITE:
                        {
                            /* Check access level */
                            if (tc_flags.access_level == TC_NO_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }
                            
                            UINT16_T index;
                            cplg_param_t cp_value;

                            for (UINT16_T i = 1; i < test_command.message_size; i += 5)
                            {
                                index = test_command.message[i];
                                cp_value = array_read_u32(test_command.message + i + 1, cp_value);
                                
                                if (codeplugSetParameter(index, cp_value, CPLG_FLUSH) != CODEPLUG_OK)
                                {
                                    /* Send ACMP Reply */
                                    send_acmp_reply(&test_command, TC_CODEPLUG_ERROR);
                                    
                                    return;                            
                                }
                            }

                            /* Update total message size */
                            test_command.message_size = 1;
                            
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                            
                            break;
                        }

                        case MSG_ENTER_BOOTMODE:
                        {                        
                            if (test_command.chipset == TC_ATMEL)
                            {
                                /* Send successful ACMP Reply */
                                send_acmp_reply(&test_command, TC_SUCCESS);   
                                
                                /* Wait for PC to disconnect */
                                Delay(1000);
                                            
                                /* Set Fuse Bit 31 */
                                flashc_set_gp_fuse_bit(FUSE_BIT_GP31, TRUE);
        
                                /* Reset Atmel microcontroller */
                                watchdog_enable(100);
                            }
                            else if (test_command.chipset == TC_CSR)
                            {
                                
			        SEND_MESSAGE_TO_BLUETOOTH_MANAGER(SWITCH_TRANSPORT_LAYER_REQUEST, 
								  message, 
								  MESSAGE_SIZE_ZERO);
  
                                UINT8_T message_id;
                                ms_task_msg_t *msg;

                                msg = (ms_task_msg_t *)Get_Msg_w_Time_Out(CSR_SEND_MSG_TIMEOUT);
                                if (msg != NULL)
                                {  
                                   message_id = msg->ros_msg.radio_if_msg.sub_opcode; /* get sub opcode*/
                                   Free_Buf(msg);
                                   
                                   if (message_id == DFU_MODE_RSP_RECEIVED_FROM_CSR)
                                   {
                                      /* Send successful ACMP Reply */
                                      send_acmp_reply(&test_command, TC_SUCCESS);   
                                      
                                      /* Wait for PC to disconnect */
																			/* Wait for Initialize success */
                                      Delay(2000);

                                      /* Tear down USB connection */
                                      usb_disconnect();
                                      
                                      /* Wait for USB to disconnect */
                                      Delay(1500);

                                      /* Set USB Switch */
                                      gpio_set_gpio_pin(PA15);
                                      
                                      /* Wait for USB switch to settle */
                                      Delay(1000);
                                   }
                                   else
                                   {
                                      /* Send ACMP Reply */
                                      send_acmp_reply(&test_command, TC_CSR_RX_ERROR);
                                   }
                                }
                                else
                                {
                                    /* Send ACMP Reply */
                                    send_acmp_reply(&test_command, TC_CSR_RX_ERROR);
                                }                               
                            }
                            else
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_CHIPSET);
                            }
                            
                            break;
                        }
   
                       case MSG_VERSION_NUMBER:
                        {                        
                            /* Check access level */
                            if (tc_flags.access_level == TC_NO_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }
                            if (tc_flags.version_match == TRUE)
                            {
                              char version[9];
                                  
                              /* Store version length */
                              test_command.message[1] = 9;   
                               
                              versionType BuildVersion = 
                              { .VersionType   = CURRENT_VERSION_IDENTIFIER, 
                                 .Iteration     = CURRENT_VERSION_PHASE, 
                                 .MajorVersion  = CURRENT_VERSION_MAJOR, 
                                 .MinorVersion  =  CURRENT_VERSION_MINOR 
                              };
  
                              sprintf(version, "%c%02d.%02d.%02d", BuildVersion.VersionType, 
                                       BuildVersion.Iteration, BuildVersion.MajorVersion, BuildVersion.MinorVersion);
  
                               /* Store version string */
                              for (UINT16_T i = 0; i < test_command.message[1]; i++)
                                  {
                                      test_command.message[i+2] = version[i];    
                                  }
                                  
                               /* Update message size */
                              test_command.message_size += test_command.message[1] + 1;
  
                               /* Send successful ACMP Reply */
                              send_acmp_reply(&test_command, TC_SUCCESS);                                     
                            }
                            else
                            {
                              /* Send ACMP Reply */
                              send_acmp_reply(&test_command, TC_VERSION_MISMATCH);                              
                            }
                           break;
                        }    
                        
                        case MSG_GPIO:
                        {                        
                            /* Check access level */
                            if (tc_flags.access_level == TC_NO_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }

                            if (test_command.message[1] == TC_GET)
                            {
                                /* Get GPIO value */
                                test_command.message[3] = (UINT8_T) gpio_get_pin_value(test_command.message[2]);    
                                
                                /* Update message size */
                                test_command.message_size += 1;
                            }
                            else if (test_command.message[1] == TC_SET)
                            {
                                if (test_command.message[3] == TC_ON)
                                {
                                    /* Set GPIO pin */
                                    gpio_set_gpio_pin(test_command.message[2]);
                                }
                                else if (test_command.message[3] == TC_OFF)
                                {
                                    /* Clear GPIO pin */
                                    gpio_clr_gpio_pin(test_command.message[2]);
                                }
                                else
                                {
                                    /* Send ACMP Reply */
                                    send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                    
                                    return;
                                }                                  
                            }
                            else
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_OPERATION);     
                                
                                return;
                            }
    
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                            
                                        
                            break;
                        }

                        case MSG_ATMEL_TEST_MODE: 
                        {                        
                            /* Check access level */
                            if (tc_flags.access_level == TC_NO_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }          
                            
                             if (test_command.message[1] == TC_ON)
                            {
                                tc_flags.atmel_test_mode = TEST_ON;
                                SEND_MESSAGE_TO_CONTROLS_MANAGER(TC_ATMEL_TEST_MODE_ON,NULL,MESSAGE_SIZE_ZERO);  //CCMPD01453712
                            }
                            else if (test_command.message[1] == TC_OFF)
                            {
                                tc_flags.atmel_test_mode = TEST_OFF;
                                SEND_MESSAGE_TO_CONTROLS_MANAGER(TC_ATMEL_TEST_MODE_OFF,NULL,MESSAGE_SIZE_ZERO);  //CCMPD01453712
                            }
                            else
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                
                                return;
                            }                                  

                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                     
                                               
                            
                            break;
                        }
                        
                        case MSG_LED: 
                        {                        
                            /* Check access level */
                            if (tc_flags.access_level == TC_NO_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }

                            if (!(test_command.message[1] == BLUE_LED))

                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                
                                return;
                            }
                            /* LED controlled through ATMEL GPIO PIN PA28 */                               
                            if (test_command.message[2] == LED_ON && tc_flags.atmel_test_mode == TEST_ON)
                            {
                              gpio_set_gpio_pin(PA28);
                              /* Update total message size */
                              test_command.message_size = 3;
        
                              /* Send successful ACMP Reply */
                              send_acmp_reply(&test_command, TC_SUCCESS);
                            }
                            else if (test_command.message[2] == LED_OFF && tc_flags.atmel_test_mode == TEST_ON)
                            {
                              gpio_clr_gpio_pin(PA28);
                              /* Update total message size */
                              test_command.message_size = 3;
        
                              /* Send successful ACMP Reply */
                               send_acmp_reply(&test_command, TC_SUCCESS);
                            }
                            else
                            {
                              /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                return;
                            }                   
                                            
                            break;
                        }
            
                        case MSG_BUTTON_TEST_MODE:
                        {                        
                            /* Check access level */
                            if (tc_flags.access_level == TC_NO_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }
                            
                               if (test_command.message[1] == TC_BUTTON_TEST_ATMEL ||
                                   test_command.message[1] == TC_BUTTON_TEST_OFF ||
                                   test_command.message[1] == TC_BUTTON_TEST_CSR ||
                                   test_command.message[1] == TC_BUTTON_TEST_ALL)
                            {
                                tc_flags.button_test_mode = test_command.message[1];
                            }
                           
                              else
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                
                                return;
                            }                                
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                     
                                               
                            break;
                        }                                 
                        #ifdef MPP_TASK
                        case MSG_PAIRING: // CCMPD01446399 (JJ 30/11/2010)
                          
                          /* Check access level */
                          if (tc_flags.access_level == TC_NO_ACCESS)
                          {
                              /* Send ACMP Reply */
                              send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                              return;                            
                          }

                          /* Set Paring Type */
                          if (test_command.message[1] == TC_PAIRING_STANDARD)
                          {
                            SEND_MESSAGE_TO_CONNECTION_MNGR(TC_STANDARD_PAIRING_MODE, NULL, 0);
                          }
                          else if (test_command.message[1] == TC_PAIRING_MPP)
                          {
                            SEND_MESSAGE_TO_CONNECTION_MNGR(TC_MPP_PAIRING_MODE, NULL, 0);
                          }
                          else
                          {
                              /* Send ACMP Reply */
                              send_acmp_reply(&test_command, TC_INVALID_VALUE);
                              return;
                          }           
                          
                          /* Send successful ACMP Reply */
                          send_acmp_reply(&test_command, TC_SUCCESS); 
                          break;
                          
                        case MSG_MPP_DEBUG: // CCMPD01446399 (JJ 30/11/2010)
                        {                        
                            /* Check access level */
                            if (tc_flags.access_level != TC_DEVELOPER_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }

                            /* Check command */
                            switch (test_command.message[1])
                            {
                                case TC_MPP_START_PAIRING:
                                {
                                    SEND_MESSAGE_TO_MPP_MNGR(START_PAIR, NULL, MESSAGE_SIZE_ZERO);
                                    
                                    /* Send successful ACMP Reply */
                                    send_acmp_reply(&test_command, TC_SUCCESS);                                            

                                    break;
                                }
                                case TC_MPP_STOP_PAIRING:
                                {
                                    SEND_MESSAGE_TO_MPP_MNGR(STOP_PAIR, NULL, MESSAGE_SIZE_ZERO);
                                    
                                    /* Send successful ACMP Reply */
                                    send_acmp_reply(&test_command, TC_SUCCESS);                                            

                                    break;
                                }
                                case TC_MPP_SEND_AMS_DIR_CMDS:
                                {
                                    if (test_command.message[2] == 0)
                                    {
                                        lf_listen_mode();
                                    }
                                    else if (test_command.message[2] == 1)
                                    {
                                        lf_clear_rssi();
                                    }
                                    else if (test_command.message[2] == 2)
                                    {
                                        lf_clear_false();  
                                    }
                                    else
                                    {
                                        /* Send ACMP Reply */
                                        send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                        
                                        return;
                                    }          
                                            
                                    /* Send successful ACMP Reply */
                                    send_acmp_reply(&test_command, TC_SUCCESS);                                            

                                    break;
                                }
                                case TC_MPP_PROG_AMS_REG:
                                {
                                    lf_chip_write(((UINT16_T)(test_command.message[2] & 0x3F) << 8) | test_command.message[3]);

                                    /* Send successful ACMP Reply */
                                    send_acmp_reply(&test_command, TC_SUCCESS);                                            

                                    break;
                                }
                                case TC_MPP_AMS_CHIP_REINIT:
                                {
                                    mpp_spi_init(SPI_FOR_AMS_CHIP, SPI_MODE_16_BIT);
                                    mpp_lfc_chip_init();
                                    mpp_spi_init(SPI_FOR_AMS_CHIP, SPI_MODE_16_BIT);

                                    /* Send successful ACMP Reply */
                                    send_acmp_reply(&test_command, TC_SUCCESS);                                            

                                    break;
                                }
                                case TC_MPP_AMS_CHIP_READ_REG:
                                {                                    
                                    test_command.message[3] = lf_chip_read(test_command.message[2]);
                                    
                                    /* Update message size */
                                    test_command.message_size += 1;
                                    
                                    /* Send successful ACMP Reply */
                                    send_acmp_reply(&test_command, TC_SUCCESS);                                            

                                    break;
                                }
                                case TC_MPP_AMS_TRIM_OSC:
                                {
                                    mpp_debug_enable = 1;
                                    mpp_spi_init(SPI_FOR_AMS_CHIP, SPI_MODE_16_BIT);
                                    lf_trim_osc();        
                                    mpp_spi_init(SPI_FOR_AMS_CHIP, SPI_MODE_16_BIT);
                                    
                                    /* Send successful ACMP Reply */
                                    send_acmp_reply(&test_command, TC_SUCCESS);                                            

                                    break;
                                }
                                case TC_MPP_AMS_PWR_DWN:
                                {
                                    lf_chip_write(0x0001); 
                                    
                                    /* Send successful ACMP Reply */
                                    send_acmp_reply(&test_command, TC_SUCCESS);                                            

                                    break;
                                }
                                case TC_MPP_INIT_DEV_MPP_PKT_TC:
                                {
                                    if (test_command.message[2] == 2)
                                    {
                                        BT_HOST_SSP_OOB = test_command.message[3];
                                        BT_HOST_128_KEY_PUSH = test_command.message[4];
                                        BT_HOST_LINK_KEY_EXT_BIT = test_command.message[5];
                                    }
                                    else if (test_command.message[2] == 3)
                                    {
                                        BT_HOST_LINK_ENC_TYPE_128_56 = test_command.message[3];
                                        BT_HOST_LINK_ENC_EXT_BIT = test_command.message[4];
                                    }
                                    else if (test_command.message[2] == 6)
                                    {
                                        BT_HOST_MAC_ADDRESS(0) = test_command.message[3];
                                        BT_HOST_MAC_ADDRESS(1) = test_command.message[4];
                                        BT_HOST_MAC_ADDRESS(2) = test_command.message[5];
                                        BT_HOST_MAC_ADDRESS(3) = test_command.message[6];
                                        BT_HOST_MAC_ADDRESS(4) = test_command.message[7];
                                        BT_HOST_MAC_ADDRESS(5) = test_command.message[8];         
                                    }
                                    else
                                    {
                                        /* Send ACMP Reply */
                                        send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                        
                                        return;
                                    }          

                                    /* Send successful ACMP Reply */
                                    send_acmp_reply(&test_command, TC_SUCCESS);                                            

                                    break;
                                }
                                default:
                                {
                                    /* Send ACMP Reply */
                                    send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                }           
                            }
                                                                    
                            break;
                        }
                        #endif
                         case MSG_BUTTON_STATE:
                        {                        
                            /* Check access level */
                            if (tc_flags.access_level == TC_NO_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }
                            
                            if (test_command.message[1] == TC_PTT_BUTTON) 
                            {
                                test_command.message[2] = tc_button_states.ptt.atmel;
                            }
                          
                            else
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                
                                return;
                            }                                  

                            /* Update message size */
                            test_command.message_size += 2;

                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                     
                                               
                            break;
                        }                                 
                       
                        
                         case MSG_DEBUG_GENERIC:
                        {
                            /* Check access level */
                            if (tc_flags.access_level != TC_DEVELOPER_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }
                          
                            /* Retrieve debug data */
                            for (UINT16_T i = 0; i < test_command.message[2]; i++)
                            {                                                                
                                /* Add debug data to queue */                     
                                if(!xQueueSend(data_queue, &test_command.message[i+3], TC_QUEUE_DELAY))
                                {
                                    send_debug_info((char *) __FUNCTION__, "Unable to add to data queue!", get_time_stamp(), PRIORITY_LOW);
                                }      
                            }                          
                
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                
    
                            if (test_command.message[2] > 0)
                            {
                                xSemaphoreGive(tc_mgr_event);
                            }                       
    
                            break;
                        }
                       
                        case MSG_DEBUG_ON:
                        {
                            /* Check access level */
                            if (tc_flags.access_level != TC_DEVELOPER_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }

                            /* Enable debug flag */
                            tc_flags.debug_enable = TRUE;
                            
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                            
                                  
                            break;
                        }
                        
                        case MSG_DEBUG_OFF:
                        {                
                            /* Check access level */
                            if (tc_flags.access_level != TC_DEVELOPER_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }

                            /* Disable debug flag */
                            tc_flags.debug_enable = FALSE;
                            
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                            
    
                            break;
                        }
                        
                        case MSG_FLASH_READ:
                        case MSG_RAM_READ:
                        {
                            /* Check access level */
                            if (tc_flags.access_level != TC_DEVELOPER_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }

                            ULONG32_T start_addr = 0;
                            UINT16_T size = 0;
                            
                            /* Retrieve start address */
                            start_addr = array_read_u32(test_command.message + 1, start_addr);
                            
                            /* Retrieve size */
                            size = array_read_u16(test_command.message + 5, size);
                
                            /* Copy data from memory */
                            memcpy(test_command.message + 7, (ULONG32_T *)start_addr, size);
                              
                            /* Update total message size */
                            test_command.message_size += size;
                            
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                            
    
                            break;
                        }
                        
                        case MSG_FLASH_WRITE:
                        case MSG_RAM_WRITE:
                        {
                            /* Check access level */
                            if (tc_flags.access_level != TC_DEVELOPER_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }

                            ULONG32_T start_addr = 0;
                            UINT16_T size = 0;
                            
                            /* Retrieve start address */
                            start_addr = array_read_u32(test_command.message + 1, start_addr);
    
                            /* Retrieve size */
                            size = array_read_u16(test_command.message + 5, size);
                                           
                            /* Write codeplug data */
                            flashc_memcpy((ULONG32_T *)start_addr, test_command.message + 7, size, TRUE);
                            
                            /* Update total message size */
                            test_command.message_size = 1;
                            
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                            
    
                            break;
                        }
                        
                        case MSG_MEMORY_INFO:
                        {
                            /* Check access level */
                            if (tc_flags.access_level != TC_DEVELOPER_ACCESS)
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_ACCESS);
                                
                                return;                            
                            }

                            ULONG32_T mem_size = 0;
                            UINT16_T mem_usage = 0;                        
                                                 
                            /* Get memory size */
                            mem_size = flashc_get_flash_size();
                            
                            /* Store memory size */
                            array_write_u32(test_command.message + 1, mem_size);
    
                            /* Get memory usage */
                            // Not Yet Implemented 
                                       
                            /* Store memory size */
                            array_write_u16(test_command.message + 5, mem_usage);
                                          
                            /* Update total message size */
                            test_command.message_size += 6;
                            
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                            
    
                            break;
                        }
                        case MSG_SIMULATOR_CTRL:
                        {                                                    
                            if (test_command.message[1] == TC_ON) 
                            {
                                tc_flags.simulator_enable = TRUE; 
                                
                                if (test_command.message[2] == TC_SIM_HOST) 
                                {
                                    tc_flags.simulator_type = TC_SIM_HOST; 
                                }
                                else if (test_command.message[2] == TC_SIM_DEVICE) 
                                {
                                    tc_flags.simulator_type = TC_SIM_DEVICE; 
                                    device_1_id = xnl_xcmp_new_device_register(hsp_xnl_init_params);

                                }
                                else
                                {
                                    /* Send ACMP Reply */
                                    send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                    
                                    return;
                                }
                            }
                            else if (test_command.message[1] == TC_OFF) 
                            {
                                tc_flags.simulator_enable = FALSE;                           
                            }
                            else
                            {
                                /* Send ACMP Reply */
                                send_acmp_reply(&test_command, TC_INVALID_VALUE);
                                
                                return;
                            }                                  
                            
                            /* Send successful ACMP Reply */
                            send_acmp_reply(&test_command, TC_SUCCESS);                                     
                                                                    
                            break;
                        }
                        default:
                        {
                            /* Send ACMP Reply */
                            send_acmp_reply(&test_command, TC_INVALID_MSG_ID);                
                        }
                    }
                }
                else
                {
                    /* Send ACMP Reply */
                    send_acmp_reply(&test_command, TC_ZERO_SIZE);                
                }
            }
            else
            {
                /* Send ACMP Reply */
                send_acmp_reply(&test_command, TC_INVALID_CRC);  
            }                                  
        }
        else
        {
            /* Send ACMP Reply */
            send_acmp_reply(&test_command, TC_INVALID_OPCODE);                
        }
    }
}

/*=============================================================================
       FUNCTION: send_acmp_reply
       
       DESCRIPTION: Send the reply back to the PC.
       
       ARGUMENTS PASSED: test_command - the test command
                         result - the result
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: none 
==============================================================================*/  

void send_acmp_reply(test_command_t *test_command, test_command_result_t result)
{
    UINT16_T crc = 0;
  
    portENTER_CRITICAL();   
    
    usb_write_u16(ACMP_REPLY);
    usb_write_u8(result);
    usb_write_u8(test_command->chipset);
    usb_write_u16(test_command->trans_id);
    usb_write_u16(test_command->message_size);
    
    for (UINT16_T i = 0; i < test_command->message_size; i++)
    {
        usb_write_u8(test_command->message[i]);
    }  

    /* Calculate the crc */
    crc = calculate_reply_crc(test_command, result, crc);   

    usb_write_u16(crc);
    
    /* Flush usb buffer */
    usb_flush();
    
    portEXIT_CRITICAL();        
}


/*=============================================================================
       FUNCTION: calculate_request_crc
       
       DESCRIPTION: Calculate the crc from the request.
       
       ARGUMENTS PASSED: test_command - the test command
                         crc - the crc
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: crc 
==============================================================================*/  

UINT16_T calculate_request_crc(test_command_t *test_command, UINT16_T crc) 
{
    crc = INIT_CRC_CCITT;
    
    crc = update_crc(crc, test_command->opcode >> 8);
    crc = update_crc(crc, test_command->opcode & 0x00FF);
    crc = update_crc(crc, test_command->chipset);
    crc = update_crc(crc, test_command->trans_id >> 8);
    crc = update_crc(crc, test_command->trans_id & 0x00FF);
    crc = update_crc(crc, test_command->message_size >> 8);
    crc = update_crc(crc, test_command->message_size & 0x00FF);
    
    for (UINT16_T i = 0; i < test_command->message_size; i++)
    {
        crc = update_crc(crc, test_command->message[i]);
    }  
    
    return crc;
}

/*=============================================================================
       FUNCTION: calculate_reply_crc
       
       DESCRIPTION: Calculate the crc from the request.
       
       ARGUMENTS PASSED: test_command - the test command
                         result - the result
                         crc - the crc
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: crc 
==============================================================================*/  

UINT16_T calculate_reply_crc(test_command_t *test_command, test_command_result_t result, UINT16_T crc) 
{
    crc = INIT_CRC_CCITT;
    
    crc = update_crc(crc, ACMP_REPLY >> 8);
    crc = update_crc(crc, ACMP_REPLY & 0x00FF);
    crc = update_crc(crc, result);
    crc = update_crc(crc, test_command->chipset);
    crc = update_crc(crc, test_command->trans_id >> 8);
    crc = update_crc(crc, test_command->trans_id & 0x00FF);
    crc = update_crc(crc, test_command->message_size >> 8);
    crc = update_crc(crc, test_command->message_size & 0x00FF);
    
    for (UINT16_T i = 0; i < test_command->message_size; i++)
    {
        crc = update_crc(crc, test_command->message[i]);
    }  
    
    return crc;
}

/*=============================================================================
       FUNCTION: update_crc
       
       DESCRIPTION: Calculate the new crc with the given byte.
       
       ARGUMENTS PASSED: crc - the calculated crc
                         data - the data to be calculated
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: the calculated crc 
==============================================================================*/  

UINT16_T update_crc(UINT16_T crc, UINT8_T data) 
{ 
  UINT16_T tmp; 
  
  tmp = data ^ (crc >> 8); 
  tmp ^= tmp >> 4; 
  crc = (crc << 8) ^ tmp ^ (tmp << 5) ^ (tmp << 12); 
  return crc; 
} 

/*=============================================================================
       FUNCTION: buffer_read_u8
       
       DESCRIPTION: Read a byte from the rx buffer.
       
       ARGUMENTS PASSED: rx_buf_head - the current position to read
                         rx_buf_count - the number of byte remaining in the buffer
                         u8 - the return byte
       
       REFERENCE ARGUMENTS PASSED: rx_buf_head, rx_buf_count, u8
       
       RETURN VALUE: TC_SUCCESS or TC_FAILURE 
==============================================================================*/  

test_command_result_t buffer_read_u8(UINT8_T *buffer, UINT16_T *index, UINT16_T *count, UINT8_T *data)
{
    if (*count)
    {
        *data = buffer[*index];
        (*count)--;
        (*index)++; 

        return TC_SUCCESS;
    }

    return TC_FAILURE;
}

/*=============================================================================
       FUNCTION: buffer_read_u16
       
       DESCRIPTION: Read a 16-bit from the rx buffer.
       
       ARGUMENTS PASSED: rx_buf_head - the current position to read
                         rx_buf_count - the number of byte remaining in the buffer
                         u16 - the return 16-bit
       
       REFERENCE ARGUMENTS PASSED: rx_buf_head, rx_buf_count, u16
       
       RETURN VALUE: TC_SUCCESS or TC_FAILURE 
==============================================================================*/  

test_command_result_t buffer_read_u16(UINT8_T *buffer, UINT16_T *index, UINT16_T *count, UINT16_T *data)
{
    UINT8_T c;
    
    for (INT16_T i = 1; i >= 0; i--)
    {                                
        if (buffer_read_u8(buffer, index, count, &c) == TC_FAILURE)
        {
            return TC_FAILURE;
        }

        *data |= ((UINT16_T) c) << (i * 8);
    }

    return TC_SUCCESS;
}

/*=============================================================================
       FUNCTION: buffer_read_u32
       
       DESCRIPTION: Read a 32-bit from the rx buffer.
       
       ARGUMENTS PASSED: rx_buf_head - the current position to read
                         rx_buf_count - the number of byte remaining in the buffer
                         u32 - the return 32-bit
       
       REFERENCE ARGUMENTS PASSED: rx_buf_head, rx_buf_count, u32
       
       RETURN VALUE: TC_SUCCESS or TC_FAILURE 
==============================================================================*/  

test_command_result_t buffer_read_u32(UINT8_T *buffer, UINT16_T *index, UINT16_T *count, ULONG32_T *data)
{
    UINT8_T c;
    
    for (INT16_T i = 3; i >= 0; i--)
    {                                
        if (buffer_read_u8(buffer, index, count, &c) == TC_FAILURE)
        {
            return TC_FAILURE;
        }

        *data |= ((ULONG32_T) c) << (i * 8);
    }

    return TC_SUCCESS;
}

/*=============================================================================
       FUNCTION: buffer_read_u64
       
       DESCRIPTION: Read a 32-bit from the rx buffer.
       
       ARGUMENTS PASSED: rx_buf_head - the current position to read
                         rx_buf_count - the number of byte remaining in the buffer
                         u64 - the return 64-bit
       
       REFERENCE ARGUMENTS PASSED: rx_buf_head, rx_buf_count, u64
       
       RETURN VALUE: TC_SUCCESS or TC_FAILURE 
==============================================================================*/  

test_command_result_t buffer_read_u64(UINT8_T *buffer, UINT16_T *index, UINT16_T *count, ULONG64_T *data)
{
    UINT8_T c;
    
    for (INT16_T i = 7; i >= 0; i--)
    {                                
        if (buffer_read_u8(buffer, index, count, &c) == TC_FAILURE)
        {
            return TC_FAILURE;
        }

        *data |= ((ULONG64_T) c) << (i * 8);
    }

    return TC_SUCCESS;
}

/*=============================================================================
       FUNCTION: usb_write_u8
       
       DESCRIPTION: Write an 8-bit to the PC.
       
       ARGUMENTS PASSED: u8 - 8-bit to write
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: none 
==============================================================================*/  

void usb_write_u8(UINT8_T u8)
{
    usb_putchar(u8);
}

/*=============================================================================
       FUNCTION: usb_write_u16
       
       DESCRIPTION: Write a 16-bit to the PC.
       
       ARGUMENTS PASSED: u16 - 16-bit to write
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: none 
==============================================================================*/  

void usb_write_u16(UINT16_T u16)
{
    usb_putchar((UINT16_T)u16 >> 8);
    usb_putchar((UINT16_T)u16 & 0x00FF);
}

/*=============================================================================
       FUNCTION: usb_write_u32
       
       DESCRIPTION: Write a 32-bit to the PC.
       
       ARGUMENTS PASSED: u32 - 32-bit to write
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: none 
==============================================================================*/  

void usb_write_u32(ULONG32_T u32)
{
    usb_putchar((UINT16_T)u32 >> 24);
    usb_putchar((UINT16_T)(u32 >> 16) & 0x000000FF);
    usb_putchar((UINT16_T)(u32 >> 8) & 0x000000FF);
    usb_putchar((UINT16_T)u32 & 0x000000FF);
}

/*=============================================================================
       FUNCTION: array_read_u8
       
       DESCRIPTION: Read an 8-bit from an array.
       
       ARGUMENTS PASSED:  arr - the array to read from
                          u8 - 8-bit to read
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: 8-bit value 
==============================================================================*/  

UINT8_T array_read_u8(UINT8_T *arr, UINT8_T u8)
{
    u8 = *arr;
    return u8;
}

/*=============================================================================
       FUNCTION: array_write_u8
       
       DESCRIPTION: Write an 8-bit to the array.
       
       ARGUMENTS PASSED:  arr - the array to write to
                          u8 - 8-bit to write
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: none 
==============================================================================*/  

void array_write_u8(UINT8_T *arr, UINT8_T u8)
{
    *arr = u8;
}

/*=============================================================================
       FUNCTION: array_read_u16
       
       DESCRIPTION: Read a 16-bit from an array.
       
       ARGUMENTS PASSED:  arr - the array to read from
                          u16 - 16-bit to read
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: 16-bit value 
==============================================================================*/  

UINT16_T array_read_u16(UINT8_T *arr, UINT16_T u16)
{
    u16 = *arr << 8 | *(arr + 1);
    return u16;
}

/*=============================================================================
       FUNCTION: array_write_u16
       
       DESCRIPTION: Write a 16-bit to the array.
       
       ARGUMENTS PASSED:  arr - the array to write to
                          u16 - 16-bit to write
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: none 
==============================================================================*/  

void array_write_u16(UINT8_T *arr, UINT16_T u16)
{
    *arr = (UINT8_T)(u16 >> 8);
    *(arr + 1) = (UINT8_T) u16 & 0x00FF;
}

/*=============================================================================
       FUNCTION: array_read_u32
       
       DESCRIPTION: Read a 32-bit from an array.
       
       ARGUMENTS PASSED:  arr - the array to read from
                          u32 - 32-bit to read
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: 32-bit value 
==============================================================================*/  

ULONG32_T array_read_u32(UINT8_T *arr, ULONG32_T u32)
{ 
    u32 = *arr << 24 | *(arr + 1) << 16 | *(arr + 2) << 8 | *(arr + 3);
    return u32;
}

/*=============================================================================
       FUNCTION: array_write_u32
       
       DESCRIPTION: Write a 32-bit to the array.
       
       ARGUMENTS PASSED:  arr - the array to write to
                          u32 - 32-bit to write
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: none 
==============================================================================*/  

void array_write_u32(UINT8_T *arr, ULONG32_T u32)
{
    *arr = (UINT8_T) (u32 >> 24);
    *(arr + 1) = (UINT8_T) ((u32 >> 16) & 0x000000FF);
    *(arr + 2) = (UINT8_T) ((u32 >> 8) & 0x000000FF);
    *(arr + 3) = (UINT8_T) (u32 & 0x000000FF);
}

/*=============================================================================
       FUNCTION: send_data_to_pc
       
       DESCRIPTION: Send raw data to the PC.
       
       ARGUMENTS PASSED: data - the data to be sent
                         len - the length of the data
       
       REFERENCE ARGUMENTS PASSED: none
       
       RETURN VALUE: SUCC 
==============================================================================*/  

success_failure_t send_data_to_pc(UINT8_T *data, UINT16_T len)
{
    portENTER_CRITICAL();   
             
    /* XCMP Message */
    for (UINT16_T i = 0; i < len; i++)
    {
        usb_write_u8(*(data + i));
    }  
        
    /* Flush usb buffer */
    usb_flush();
        
    portEXIT_CRITICAL(); 
    
    return SUCC;
}