/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013-2014 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: xcmp.c
#
# --------------------------- General Description -----------------------------
#  This file handles the call to the XCMP message that needs to be decoded
#
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/

#include "xcmp.h"
#include "xcmp_config.h"
#include "xnl.h"
/********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
*   AUTHOR              Date Modified         Tracking Number         Description
* Abhishek Trivedi      09/02/2013            CCMPD01827271       XCMP Interfaces
* Abhishek Trivedi      11/8/2013             CCMPD01834482       Updates for XNL-XCMP 
* Abhishek Trivedi      03/18/2014	      CCMPD01873731	  Resolve 2 device connection issue
* Abhishek Trivedi      04/18/2014            CCMPD01885148       System Ping Timeout
* Abhishek Trivedi      10/22/2014            CCMPD01938895       Audio Switch
* Abhishek Trivedi      06/05/2014            CCMPD01900617       Version Request Support
* Abhishek Trivedi      11/02/2014            CCMPD01943851       Codeplug Read Request for Talk Permit Tone issue
**--------------------------- End of History Template-----------------------------
* *********************************************************************************/
/* ***************************************************************************
DEFINES
************************************************************************** */

/* ***************************************************************************
ENUMS & STRUCTURES
************************************************************************** */
typedef enum
{
    XCMP_POWERUP                    = 0, 
    SENDING_XCMP_DEVINIT            = 1,
    WAITING_FOR_XCMP_DEVINIT        = 2,
    SENDING_XCMP_DEVINIT_DONE       = 3,
    XCMP_IDLE                       = 4
} xcmp_states_t;

/* ***************************************************************************
GLOBAL VARIABLES
************************************************************************** */
// A global pointer to receive data from XNL
data_buffer_struct_t *ptr_to_xcmp_payload_glb = NULL;

// A global index to access data from XNL RX Buffer
UINT16_T ptr_new_message_rx_index = 0;

extern UINT16_T number_of_devices_glb;
/* ***************************************************************************
FUNCTION PROTOTYPES
************************************************************************** */

/*=============================================================================
	FUNCTION: xcmp_process_init()

	DESCRIPTION: This is the XCMP initialization function.
==============================================================================*/
xcmp_return_value_t xcmp_process_init(UINT8_T index, UINT16_T xnl_devid, data_buffer_struct_t *payload_data_ptr, UINT8_T payload_length)
{
  xcmp_return_value_t return_value = XCMP_FAILURE;
  ptr_to_xcmp_payload_glb = payload_data_ptr;
  ptr_new_message_rx_index = 0;
  UINT16_T xcmp_opcode = 0;
//  UINT8_T i = 0;
  
  // Check payload size
  if(payload_length >= 2)
  {
    // Extract the XCMP message opcode
    xcmp_opcode = xcmp_get_2bytes();
  }
  else
  {
    ptr_to_xcmp_payload_glb = NULL;
    return XCMP_FAILURE;
  }				
			
  // Check if Dev Initialization Complete message has been sent.
  if(XCMP_MSG_OPCODE_BROADCAST_DEVICE_INITIALIZATION_STATUS == xcmp_opcode)
  {
    return_value = xcmp_msg_rx_wait_device_init(xnl_devid, payload_length);
  
    if(return_value == XCMP_SUCCESS)
    {
      xnl_cancel_enum_timer(xnl_devid);
      // Send the xcmp dev init update
      if(number_of_devices_glb <= 2)
      {
        xcmp_msg_tx_device_init_done(xnl_devid);
      }
/*      else  //TODO: Send the XCMP Dev Init Update to all connected devices
      {
        for(i=0; i < number_of_devices_glb; i++)
        {
        // Send XCMP Dev Init Update followed by Dev init Done
          if(xcmp_msg_tx_device_init_update(xnl_devid))
          {
            continue;
          }
        }
        xcmp_msg_tx_device_init_done(xnl_devid);
      }*/
      // Now the XCMP initialization is done.
      xnl_xcmp_authentication_complete(xnl_devid);
    }
    else
    {
      
    }
  }
  else
  {
    return_value = XCMP_FAILURE;
  }      
      
  ptr_to_xcmp_payload_glb = NULL;
  return return_value;		
}

/*=============================================================================
	FUNCTION: xcmp_process_message()

	DESCRIPTION: This is the "main" XCMP function.  This function process
			the entire state machine for XCMP.  Each state decodes
			incoming XCMP messages.
==============================================================================*/
xcmp_return_value_t xcmp_process_message(UINT16_T xnl_devid, data_buffer_struct_t *payload_buffer_ptr, UINT16_T payload_length)
{
  xcmp_return_value_t return_value = XCMP_FAILURE;
  ptr_to_xcmp_payload_glb = payload_buffer_ptr;
  ptr_new_message_rx_index = 0;
  
  UINT16_T xcmp_opcode = 0; 

  // Check payload size
  if(payload_length >= 2)
  {
    xcmp_opcode = xcmp_get_2bytes();
  }
  else
  {
    // Extract the XCMP message opcode
    ptr_to_xcmp_payload_glb = NULL;
    return XCMP_FAILURE;
  }
	
  switch (xcmp_opcode)
  {

    // Physical User Input broadcast 
    case XCMP_MSG_OPCODE_BROADCAST_PU_INPUT:
    {
      return_value = xcmp_msg_rx_pui_broadcast(xnl_devid, payload_length);
      break;
    }

    // Tone control request - not supported as no Tones played out by Gateway
    /*
    case XCMP_MSG_OPCODE_REQUEST_TONE_CNTRL:
    {
            return_value = xcmp_msg_rx_tone_cntlr_request(xnl_devid, payload_length);
            break;
    }
    */
    // Version reply
    case XCMP_MSG_OPCODE_REPLY_VERSION_INFO:
    {
      return_value = xcmp_msg_rx_ver_info_reply(xnl_devid, payload_length);
      break;						
    }

    // Volume request
    case XCMP_MSG_OPCODE_REQUEST_VOL_CNTRL:
    {
      return_value = xcmp_msg_rx_volume_control_request(xnl_devid, payload_length);
      if(return_value == XCMP_INSUFFICIENT_DATA)
      {
        receive_xcmp_invalid_msg_rqst(xnl_devid, XCMP_MSG_OPCODE_REQUEST_VOL_CNTRL);
      }
      break;
    }
    //Version Request
    case XCMP_MSG_OPCODE_REQUEST_VERSION_INFO:
    {
      return_value = xcmp_msg_rx_ver_info_rqst(xnl_devid, payload_length);
      break; 
    }    
    // Volume reply
    case XCMP_MSG_OPCODE_REPLY_VOL_CNTRL:
    {
      return_value = xcmp_msg_rx_volume_control_reply(xnl_devid, payload_length);
      break;
    }
    //Speaker Control Request
    case XCMP_MSG_OPCODE_REQUEST_SPKR_CNTRL:
    {
      return_value = xcmp_msg_rx_speaker_control_request(xnl_devid, payload_length);
      if(return_value == XCMP_INSUFFICIENT_DATA)
      {
        receive_xcmp_invalid_msg_rqst(xnl_devid, XCMP_MSG_OPCODE_REQUEST_SPKR_CNTRL);
      }
      break;
    }    
    //Speaker Control Broadcast
    case XCMP_MSG_OPCODE_BROADCAST_SPKR_CNTRL:
    {
      return_value = xcmp_msg_rx_speaker_control_broadcast(xnl_devid, payload_length); 
      break;
    }
    // BlueTooth Config
    case XCMP_MSG_OPCODE_REQUEST_BT_CONFIG:
    {
      return_value = xcmp_msg_rx_bt_config_request(xnl_devid, payload_length);
      if(return_value == XCMP_INSUFFICIENT_DATA)
      {
        receive_xcmp_invalid_msg_rqst(xnl_devid, XCMP_MSG_OPCODE_REQUEST_BT_CONFIG);
      }            
      break;
    }			

    // Radio Status
    case XCMP_MSG_OPCODE_REQUEST_RADIO_STATUS:
    {           
      return_value = xcmp_msg_rx_radio_status_request(xnl_devid, payload_length);
      if(return_value == XCMP_INSUFFICIENT_DATA)
      {
        receive_xcmp_invalid_msg_rqst(xnl_devid, XCMP_MSG_OPCODE_REQUEST_RADIO_STATUS);
      }         
      break;
    }    
    // Ping
    case XCMP_MSG_OPCODE_REPLY_PING:
    {
      return_value = xcmp_msg_rx_ping_reply(xnl_devid, payload_length);
      break;
    }
    case XCMP_MSG_OPCODE_REQUEST_PING:
    {
      return_value = xcmp_msg_rx_ping_request(xnl_devid, payload_length);
      break;
    }
    //Codeplug Read
    case XCMP_MSG_OPCODE_REPLY_CPLG_READ:
    {           
      return_value = xcmp_msg_rx_cplg_read_reply(xnl_devid, payload_length);
      break;
    }       
    // Battery Level
    case XCMP_MSG_OPCODE_BROADCAST_BATT_LVL:
    {
      return_value = xcmp_msg_rx_batt_lvl_broadcast(xnl_devid, payload_length);
      break;
    }    
    //Discard Unsupported Messages
    default :
    {
      // Check if request message.
      if(!(XCMP_MSG_REQUEST_OPCODE_MASK & xcmp_opcode))
      {
        // Notify user application
        xcmp_msg_rx_unsupported_xcmp_msg_request(xnl_devid, xcmp_opcode);
        return_value = XCMP_UNSUPPORTED_REQ;
      }
      else
      {
        return_value = XCMP_FAILURE;
      }
      break;
    }
  }
  
  //Reset the pointer
  ptr_to_xcmp_payload_glb = NULL;
  
  return return_value;
} 


