/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013-2015 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: xcmp_config.c
#
# --------------------------- General Description -----------------------------
# This is the XCMP configuration file. Project specific defines for using the
# general XCMP engine in xcmp.c goes here. Specify which XXMP commands your project
# will support in the enum below.
#
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/
#include "xcmp_config.h"
#include "xcmp.h"
#include "xnl.h"
#include "xnl_config.h"
#include "controls.h"
#include <string.h>
/************************************************************************************
*
*--------------------------------- Revision History ----------------------------------
*
*   AUTHOR          Date Modified        Tracking Number        Description
* Abhishek Trivedi  08/25/2013           CCMPD01827271     XNL-XCMP Host
* Abhishek Trivedi  11/8/2013            CCMPD01834482     Updates for XNL-XCMP 
* Abhishek Trivedi  11/11/2013           CCMPD01834144     Change the BT Config XCMP message to send 
                                                           Infinite pairing instead of Immediate pairing
* Abhishek Trivedi  11/19/2013           CCMPD01836292     Add the BT Config Broadcast message to send 
* WRN637            12/20/2013           CCMPD01833277     WPTT state machine
* WRN637            01/07/2014           CCMPD01849957     Volume tone request support
* WRN637            01/22/2014           CCMPD01854392     Move XCMP related handling to HSP_XNL_XCMP_MANAGER_TASK to prevent multiple access to XCMP decoder at the same time  
* Abhishek Trivedi  03/18/2014	         CCMPD01873731	   Resolve 2 device connection issue
* Aneeket Patkar    03/28/2014           CCMPD01877631     Remove Volume ramp feature as per the updated design
* Abhishek Trivedi  04/18/2014           CCMPD01885148     System Ping Timeout         
* Abhishek Trivedi  05/22/2014           CCMPD01897875     Fix Klocwork Issues
* Abhishek Trivedi  10/22/2014           CCMPD01938895     Audio Switch
* Abhishek Trivedi  06/05/2014           CCMPD01900617     Version Request Support
* Abhishek Trivedi  10/24/2014           CCMPD01941655     Audio Switch Redesign
* Abhishek Trivedi  11/02/2014           CCMPD01943851     Codeplug Read Request for Talk Permit Tone issue
* Abhishek Trivedi  01/07/2015           CCMPD01958099     1-dot support for WRSM
* Abhishek Trivedi  01/14/2015           CCMPD01958122     GW supports infinte pairing
**--------------------------- End of History Template-----------------------------
* *********************************************************************************/
/* ***************************************************************************
                                DEFINES
   ************************************************************************** */
#define PHASE_VERSION_CHAR_TWO  2
#define MAJOR_VERSION_CHAR_ONE  4
#define MAJOR_VERSION_CHAR_TWO  5
/* ***************************************************************************
                                GLOBAL VARIABLES
   ************************************************************************** */
volatile UINT8_T accy_step_size = DEFAULT_STEP_SIZE;
volatile UINT16_T accy_vol;
UINT8_T accy_sw_version[MAX_VERSION_SIZE];
char accy_hw_name[MAX_VERSION_SIZE];
UINT16_T hsp_unsupported_xcmp_opcode = 0x0000;
UINT16_T spp_unsupported_xcmp_opcode = 0x0000;
UINT16_T hsp_invalid_msg_xcmp_opcode = 0x0000;
UINT16_T spp_invalid_msg_xcmp_opcode = 0x0000;

ver_msg_state_t ver_msg_state = ACCY_VER_REQ_NOT_SENT;

UINT32_T apply_tpt_issue_delay = 0;
/* ***************************************************************************
                                EXTERNAL VARIABLES
   ************************************************************************** */
extern TimerDescriptorType xnl_msg_retry_timer[];
extern bool hsp_ping_rcvd;
extern bool spp_ping_rcvd;
extern bool gw_legacy_behavior;
extern BT_U16 nibbler_ptt_press_timeout;
extern bool gw_never_repair;
/* ***************************************************************************
                                EXTERNAL FUNCTIONS
   ************************************************************************** */
extern tx_result_t xnl_device_tx_status (const UINT16_T xnl_devid);
extern void one_dot_programable_button_handler(controls_button_state_t button_state);
extern void two_dot_programable_button_handler(controls_button_state_t button_state);
extern void xnl_device_report_failure (UINT16_T xnl_devid, xnl_error_type_t error);
/* ***************************************************************************
                                DEFINES/CONSTANTS
   ************************************************************************** */

// Device Init Status info. - User should config this before using the Host 
const xcmp_msg_device_init_status_broadcast_t dev_init_status_info = 
{
  (UINT16_T)XCMP_MSG_OPCODE_BROADCAST_DEVICE_INITIALIZATION_STATUS,					// XCMP opcode
  (UINT32_T)XCMP_MSG_DEVICE_INIT_STATUS_VERSION_NUMBER,								// XCMP version
  (UINT8_T )XCMP_MSG_DEVICE_INIT_STATUS_TYPE_STATUS,									// Device Init Type
  (UINT8_T )XCMP_MSG_DEVICE_INIT_STATUS_DEVICE_TYPE_RF_TRANSCEIVER,					// Device Type
  (UINT16_T)XCMP_MSG_DEVICE_INIT_STATUS_DEVICE_STATUS_NO_ERRORS,						// Device Status
  (UINT8_T )DEVICE_DESCRIPTOR_SIZE,													// Device Descriptor Size
  {
    {																				// Descriptor 1
      (UINT8_T)XCMP_DEVICE_MIC_TYPE,												// Attribute ID
      (UINT8_T)XCMP_MSG_DEVICE_INIT_STATUS_DESCRIPTOR_DEVICE_MIC_TYPE					// Attribute value
    },
    {																				// Descriptor 2
      (UINT8_T)XCMP_DEVICE_BLUETOOTH_TYPE,												// Attribute ID
      (UINT8_T)XCMP_MSG_DEVICE_INIT_STATUS_DESCRIPTOR_DEVICE_BLUETOOTH_TYPE			// Attribute value
    },
    {																				// Descriptor 3
      (UINT8_T)XCMP_DEVICE_DISPLAY,												// Attribute ID
      (UINT8_T)XCMP_MSG_DEVICE_INIT_STATUS_DESCRIPTOR_DEVICE_VALUE_NO_DISPLAY		// Attribute value
    }
    //...
  }	
};

extern UINT16_T device_1_id;
extern UINT16_T device_2_id;

// Test sync flags
BOOL_T one_dot_test_flag = FALSE;
BOOL_T host_in_emergency_state = FALSE;

TimerDescriptorType bt_system_ping_timer[NUMBER_OF_PING_TIMERS];

/* ***************************************************************************
                                FUNCTION DEFINITIONS
   ************************************************************************** */
/*=============================================================================
	FUNCTION: xcmp_get_2bytes()

	DESCRIPTION: 

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: UINT16_T data
=============================================================================*/
UINT16_T xcmp_get_2bytes()
{
  UINT16_T data = 0;
  data = xcmp_get_1byte();        
  data <<= 8;
  data |= xcmp_get_1byte();
  return data;
} 
/* End of xcmp_get_2bytes() */


//------------------------------------------------------------------------------------------------------------
// Device initialization
//------------------------------------------------------------------------------------------------------------

void receive_xcmp_device_init_msg_bdst(UINT16_T xnl_devid, xcmp_msg_device_init_status_broadcast_t dev_init_message)
{
	//Callback function to be implemented
}


//------------------------------------------------------------------------------------------------------------
// Volume control
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends Volume Control request message. The default Attenuator_Number is set to be
 *				ALL SPEAKERS.
 * Parameters:
 *		function: Specifies the function being requested. See volume_ctrl_func_t.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_volume_control_msg_rqst(UINT16_T xnl_devid, volume_ctrl_func_t function)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_volume_control_request(xnl_devid, function, 0x00, NO_OPTIONAL_PARAM) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}

/* Description: This function sends Volume Control reply message. The default Attenuator_Number is set to be
 *				ALL SPEAKERS.
 * Parameters:
 *		result: XCMP reply result code. Success: 0x00, Failure: 0x01
 *		function: Specifies the function being requested. See volume_ctrl_func_t.
 *		volume_data: Contains the current setting of the specified volume attenuator.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_volume_control_msg_rply(UINT16_T xnl_devid, xcmp_reply_result_t result, volume_ctrl_func_t function, UINT8_T volume_data, UINT8_T send_opt_param)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_volume_control_reply(xnl_devid, result, function, volume_data, send_opt_param) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}

/* Description: This function sends Volume Control broadcast message. The default Attenuator_Number is set to be
 *				ALL SPEAKERS.
 * Parameters:
 *		volume_data: Contains the current setting of the specified volume attenuator.
 *		audio_parameter: Contains Radio audio parameter setting. 
 *						 Radio Speaker Automatic Volume Control Enable/Disable: 0x01/0x00
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_volume_control_msg_bdst(UINT16_T xnl_devid, UINT8_T volume_data, UINT8_T audio_parameter)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_volume_control_broadcast(xnl_devid, volume_data, audio_parameter) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}

/* Description: This function receives Volume Control request message. It assumes that the Attenuator_Number is
 *				ALL_SPEAKER. The volume_data field in the message is ignored.
 * Parameters:
 *		function: Specifies the function being requested. See volume_ctrl_func_t.
*/
void received_xcmp_volume_control_msg_rqst(UINT16_T xnl_devid, xcmp_msg_volume_ctrl_request_msg_t vol_ctrl_req, UINT8_T payload_length)
{
  void * ptr = NULL;

  switch(vol_ctrl_req.function)
  {
    case SET_INCR_SIZE:
    case REPLY_STEPS_PER_INCR:
      {
        if(payload_length > MIN_VOL_CNTRL_RQST_MSG_SIZE)
        {
          if(xnl_devid == device_1_id)
            accy_step_size = vol_ctrl_req.volume_data;
        }
        if(send_xcmp_volume_control_msg_rply(xnl_devid, XCMP_MSG_RESULT_SUCCESS, vol_ctrl_req.function, accy_step_size, SEND_OPTIONAL_PARAM) != TX_SUCCESS)
        {
          ptr = xnl_malloc(sizeof(xcmp_msg_volume_ctrl_request_msg_t));

          if(ptr == NULL)
          {
            // Error, no memory space
            xnl_device_report_failure(xnl_devid, XNL_MEMORY_ALLOC_ERROR);
          }

          memcpy( ptr, &vol_ctrl_req, sizeof(xcmp_msg_volume_ctrl_request_msg_t));
          
          if(xnl_devid == device_1_id)
          {
            RETRY_XCMP_MSG_TIMER(HSP_VOL_LVL_REP_MSG, HSP_SEND_XCMP_VOL_RPLY, 25, ptr);
          }
          else if(xnl_devid == device_2_id)
          {
            RETRY_XCMP_MSG_TIMER(SPP_VOL_LVL_REP_MSG, SPP_SEND_XCMP_VOL_RPLY, 25, ptr);
          }
          else
          {
            xnl_free(ptr);
          }
        }
      }
      break;
    case SPECIFIED_VALUE:
      {
        if(payload_length > MIN_VOL_CNTRL_RQST_MSG_SIZE)
        {
          if(xnl_devid == device_1_id)
            accy_vol = vol_ctrl_req.volume_data;
        }
        if(send_xcmp_volume_control_msg_rply(xnl_devid, XCMP_MSG_RESULT_SUCCESS, vol_ctrl_req.function, 0x00, NO_OPTIONAL_PARAM) != TX_SUCCESS)
        {
          ptr = xnl_malloc(sizeof(xcmp_msg_volume_ctrl_request_msg_t));

          if(ptr == NULL)
          {
            // Error, no memory space
            xnl_device_report_failure(xnl_devid, XNL_MEMORY_ALLOC_ERROR);
          }
          
          memcpy( ptr, &vol_ctrl_req, sizeof(xcmp_msg_volume_ctrl_request_msg_t));
       
          if(xnl_devid == device_1_id)
          {
            RETRY_XCMP_MSG_TIMER(HSP_VOL_LVL_REP_MSG, HSP_SEND_XCMP_VOL_RPLY, 25, ptr);
          }
          else if(xnl_devid == device_2_id)
          {
            RETRY_XCMP_MSG_TIMER(SPP_VOL_LVL_REP_MSG, SPP_SEND_XCMP_VOL_RPLY, 25, ptr);
          }
          else
          {
            xnl_free(ptr);
          }
        }
        else
        {
          if(send_xcmp_volume_control_msg_bdst(device_1_id, accy_vol, 0x00) != TX_SUCCESS)
          {
            RETRY_XCMP_MSG_TIMER(HSP_VOL_BCAST_MSG, SEND_XCMP_VOL_BCAST_NO_TONE, 25, NULL);
          }
          else
          {
          /*  if( device_2_id != 0 ) SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_SEND_XCMP_VOL_BCAST, NULL, 0);        */
          }          
        }
      }
      break;
    default:
      /* Send Unsupported Parameter Reply */ 
      ptr = xnl_malloc(sizeof(xcmp_msg_volume_ctrl_request_msg_t));

      if(ptr == NULL)
      {
        // Error, no memory space
        xnl_device_report_failure(xnl_devid, XNL_MEMORY_ALLOC_ERROR);
      }

      memcpy( ptr, &vol_ctrl_req, sizeof(xcmp_msg_volume_ctrl_request_msg_t));
      
      if(xnl_devid == device_1_id)
      {
        RETRY_XCMP_MSG_TIMER(HSP_VOL_LVL_REP_MSG, HSP_SEND_XCMP_VOL_NEG_RPLY, 25, ptr);
      }
      else if(xnl_devid == device_2_id)
      {
        RETRY_XCMP_MSG_TIMER(SPP_VOL_LVL_REP_MSG, SPP_SEND_XCMP_VOL_NEG_RPLY, 25, ptr);
      }
      else
      {
        xnl_free(ptr);
      }               
      break;
  }  
}

/* Description: This function receives Volume Control reply message. It assumes that the Attenuator_Number is
 *				ALL_SPEAKER. 
 * Parameters:
 *		result: XCMP reply result code. See xcmp_reply_result_t.
 *				If result code is not XCMP_MSG_RESULT_SUCCESS, all other
 *				params should be 0 or NULL.
 *		function: Specifies the function being requested. See volume_ctrl_func_t.
 *		volume_data: Contains the current setting of the specified volume attenuator.
*/
void received_xcmp_volume_control_msg_rply(UINT16_T xnl_devid, xcmp_msg_volume_ctrl_reply_msg_t vol_ctrl_rply, UINT8_T payload_length)
{
  if(vol_ctrl_rply.result == XCMP_MSG_RESULT_SUCCESS)
  {
    switch(vol_ctrl_rply.function)
    {
      case SET_INCR_SIZE:
      case REPLY_STEPS_PER_INCR:
        {
          if(xnl_devid == device_1_id)
          {
            if(payload_length > MIN_VOL_CNTRL_RPLY_MSG_SIZE)
              accy_step_size = vol_ctrl_rply.volume_data;
          }
        }
        break;
      case SPECIFIED_VALUE:
        break;
      default:
        break;
    }
  }
  else
  {
    
  }
}

//------------------------------------------------------------------------------------------------------------
// Speaker control
//------------------------------------------------------------------------------------------------------------
tx_result_t send_xcmp_speaker_control_msg_rqst(UINT16_T xnl_devid, UINT16_T speaker_number, UINT16_T function)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_speaker_control_request(xnl_devid, speaker_number, function) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}

tx_result_t send_xcmp_speaker_control_msg_bdst(UINT16_T xnl_devid, UINT16_T speaker_number, UINT16_T function)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_speaker_control_broadcast(xnl_devid, speaker_number, function) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}

tx_result_t send_xcmp_speaker_control_msg_rply(UINT16_T xnl_devid, xcmp_reply_result_t result, UINT16_T speaker_number, UINT16_T function)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_spkr_control_reply(xnl_devid, result, speaker_number, function) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }  
}
void received_xcmp_spkr_control_msg_rqst(UINT16_T xnl_devid, xcmp_msg_spkr_ctrl_request_msg_t spkr_ctrl_rqst)
{
  if((spkr_ctrl_rqst.spkr_no == ALL_SPEAKERS) || (spkr_ctrl_rqst.spkr_no == BT_SPEAKER))
  {
    if(spkr_ctrl_rqst.function == MUTE_SPKR)  //WRSM Requesting GW to Mute
    {
      SEND_MESSAGE_TO_CONNECTION_MNGR(HS_USER_SPKR_UNMUTED_RQST, NULL, 0);
    }
    else if(spkr_ctrl_rqst.function == UNMUTE_SPKR) //WRSM Requesting GW to Un-Mute
    {
      select_mic_spkr(LOCAL_MIC_SPKR);
    }
  }
}
void received_xcmp_speaker_control_msg_rply(UINT16_T xnl_devid, xcmp_msg_spkr_ctrl_reply_msg_t spkr_ctrl_rply)
{
  if(spkr_ctrl_rply.result != XCMP_MSG_RESULT_SUCCESS)
  {
    
  }
}
void received_xcmp_spkr_control_msg_bcast(UINT16_T xnl_devid, xcmp_msg_spkr_ctrl_broadcast_msg_t spkr_ctrl_bcast)
{
  if(spkr_ctrl_bcast.spkr_no != 0x0000)
  {
    if(spkr_ctrl_bcast.function == MUTE_SPKR)
    {
      SEND_MESSAGE_TO_CONNECTION_MNGR(HS_USER_SPKR_UNMUTED_BCAST, NULL, 0);
    }
    else if(spkr_ctrl_bcast.function == UNMUTE_SPKR)
    {
      select_mic_spkr(LOCAL_MIC_SPKR);
    }
  }  
}
//------------------------------------------------------------------------------------------------------------
// Tone control
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends Tone Control Request message.
 * Parameters:
 *		control: Type of tone action to perform. Stop: 0x00, Start: 0x01
 *		volume: Specifies the volume. Current Volume: 0x00
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_logical_tone_control_msg_rqst(UINT16_T xnl_devid, UINT8_T control, UINT16_T identifier, UINT8_T volume)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_tone_cntlr_request(xnl_devid, control, identifier, volume, NULL, NO_OPTIONAL_PARAM) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}
/* Description: This function sends Tone Control broadcast message.
 * Parameters:
 *		control: Type of tone action to perform. Stop: 0x00, Start: 0x01
 *		volume: Specifies the volume. Current Volume: 0x00
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_tone_control_msg_bdst(UINT16_T xnl_devid, UINT8_T control, UINT8_T volume)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_tone_cntlr_broadcast(xnl_devid, control, XCMP_MSG_TONE_CTRL_ID_ALL_TONES, volume, NULL, NO_OPTIONAL_PARAM) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}

/* Description: This function sends Tone Control reply message.
 * Parameters:
 *		result: XCMP reply result code. Success: 0x00, Failure: 0x01
 *		duration: Value determined by repetition, timebase, and pattern (in ms); 0 means continuous
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_tone_control_msg_rply(UINT16_T xnl_devid, xcmp_reply_result_t result, UINT16_T tone_id, UINT16_T duration)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_tone_cntlr_reply(xnl_devid, result, tone_id, duration) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}

/* Description: This function receives Tone Control request message.
 * Parameters:
 *		tone_control: Type of tone action to perform. Stop: 0x00, Start: 0x01
*/
void received_xcmp_tone_control_msg_rqst(UINT16_T xnl_devid, xcmp_msg_tone_control_request_t tone_ctrl_rqst)
{
//  send_xcmp_tone_control_msg_rply(xnl_devid, XCMP_MSG_RESULT_SUCCESS, tone_ctrl_rqst.ToneIdentifier, 0x0000);
}
/* Description: This function receives Tone Control reply message.
 * Parameters:
 *		
*/
void received_xcmp_tone_control_msg_rply(UINT16_T xnl_devid, xcmp_msg_tone_control_reply_t tone_ctrl_rply)
{
  if(tone_ctrl_rply.result)
  {
   //Start the tone stop timer here?? 
  }
}

//------------------------------------------------------------------------------------------------------------
// Version info
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends a version info request message to accessory.
 * Parameters:
 *		type: The version type the Host supports. See ver_info_request_type_t.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_version_info_msg_rqst(UINT16_T xnl_devid, ver_info_request_type_t type)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_ver_info_request(xnl_devid, type) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
          return TX_BUSY;
  }
}
tx_result_t send_xcmp_version_info_msg_rply(UINT16_T xnl_devid, UINT8_T result, UINT8_T* ver_ptr, UINT8_T size)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
          if (xcmp_msg_tx_ver_info_reply(xnl_devid, result, ver_ptr, size) == SUCC)
                  return TX_SUCCESS;
          else
                  return TX_FAILURE;
  }
  else
  {
          return TX_BUSY;
  }
}

/* Description: The function receives version info sent from accessory.
 * Parameters:
 *		result: XCMP reply result code. See xcmp_reply_result_t.
 *				If result code is not XCMP_MSG_RESULT_SUCCESS, all other
 *				params should be 0 or NULL.
 *		version_buffer: a pointer to the version data buffer.
 *		version_size: the size of data in version_buffer.
 *			Default maximum buffer size is 25, defined in xcmp_msg_ver_info.h
*/
void received_xcmp_ver_info_msg_rply(UINT16_T xnl_devid, xcmp_msg_ver_info_reply_t ver_info_rply)
{
  UINT8_T i = 0;
  if(ver_info_rply.result == XCMP_MSG_RESULT_SUCCESS)
  {
    if(ver_msg_state == ACCY_SW_VER_MSG_SENT)
    {
      for(i=0;(i < ver_info_rply.version_size) && (i < MAX_VERSION_SIZE); i++)
      {
        accy_sw_version[i] = ver_info_rply.version_buffer[i];
      }
       //Add Null to compare the string
      if(i == MAX_VERSION_SIZE)
      {
        accy_sw_version[--i] = '\0';
      }
      else
      {
        accy_sw_version[i] = '\0';
      }     
      ver_msg_state = ACCY_SW_VER_MSG_RCVD;
      //Compare HW name with "MCWRSM"
      if(strcmp(accy_hw_name, "MCWRSM") == 0)
      {
        //Compare with 4th and 5th Major Version Character
        //For WRSM the value should be greater than '01'(Release: R01.01.00) which supports Audio Transfer etc.
        if((accy_sw_version[MAJOR_VERSION_CHAR_TWO] >= '2') ||
           (accy_sw_version[MAJOR_VERSION_CHAR_ONE] >= '1') ||
           (accy_sw_version[PHASE_VERSION_CHAR_TWO] >= '2')  )
        {
          gw_legacy_behavior = FALSE;
        }
        else
        {
          gw_legacy_behavior = TRUE;
        }
      }
      else
      {
        gw_legacy_behavior = TRUE;
      }
      // Send Update to Connection manager
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_SW_VER_REPLY_RCVD_MSG, NULL, 0); 
    }
    else if(ver_msg_state == ACCY_HW_VER_MSG_SENT)
    {
      for(i=0;(i < ver_info_rply.version_size) && (i < MAX_VERSION_SIZE); i++)
      {
        accy_hw_name[i] = ver_info_rply.version_buffer[i];
      }   
      //Add Null to compare the string
      if(i == MAX_VERSION_SIZE)
      {
        accy_hw_name[--i] = '\0';
      }
      else
      {
        accy_hw_name[i] = '\0';
      }
      ver_msg_state = ACCY_HW_VER_MSG_RCVD;
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_SEND_SW_VER_REQ_MSG, NULL, 0); 
    }
  }
  else  //Reply result is Unsupported, Invalid or Failure
  {
    if(ver_msg_state == ACCY_HW_VER_MSG_SENT)
    {
      gw_legacy_behavior = TRUE;
      ver_msg_state = ACCY_HW_VER_MSG_RCVD;
    }
    SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_SEND_SW_VER_REQ_MSG, NULL, 0); 
  }
}
void received_xcmp_ver_info_msg_rqst(UINT16_T xnl_devid, xcmp_msg_ver_info_request_t ver_info_rqst)
{
  if(ver_info_rqst.ver_type == HOST_SW_VER)
  {
    if(xnl_devid == device_1_id)
    {
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_VER_REQ_RCVD, NULL, 0);      
    }
    else if(xnl_devid == device_2_id)
    {
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_VER_REQ_RCVD, NULL, 0);      
    }
  }
  else if(ver_info_rqst.ver_type == XCMP_PRODUCT_ID)
  {
    if(xnl_devid == device_1_id)
    {
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_VER_PRD_ID_REQ_RCVD, NULL, 0);      
    }
    else if(xnl_devid == device_2_id)
    {
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_VER_PRD_ID_REQ_RCVD, NULL, 0);      
    }    
  }
  else
  {
    /* Send Unsupported Parameter Reply */ 
    if(xnl_devid == device_1_id)
    {
      hsp_invalid_msg_xcmp_opcode = XCMP_MSG_OPCODE_REQUEST_VERSION_INFO;
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_RCVD_XCMP_INVALID_MSG, NULL, 0);  
    }
    else if(xnl_devid == device_2_id)
    {
      spp_invalid_msg_xcmp_opcode = XCMP_MSG_OPCODE_REQUEST_VERSION_INFO;
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_RCVD_XCMP_INVALID_MSG, NULL, 0);      
    }
  }
}

//------------------------------------------------------------------------------------------------------------
// BlueTooth Config
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends a BlueTooth Config reply message. Assuming the 'function' field of the message 
 *				is XCMP_MSG_BT_POWER_UP_CONFIG.
 * Parameters:
 *		result: XCMP reply result code. Success: 0x00, Failure: 0x01
 *		function: Functions to perform. Power Up Config: 0x04
 *		accy_type: Types of accessory. OPTION BOARD: 0x00, AUDIO ACCESSORY: 0x01,
 *					DATA ACCESSORY: 0x02, GENERIC / NON-AUDIO: 0x03
 *		pairing_timeout: Value in minutes. After which a disconnected accessory will be de-paired.
 *		out_of_range_timeout: Value in minutes. After which the link will be dropped for an accessory that is out of range.
 *		bt_state: N/A for Audio Accessory.
 *		pairing_type: N/A for Audio Accessory.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_bt_config_msg_rply(UINT16_T xnl_devid, xcmp_reply_result_t result, UINT8_T function, 
					  UINT8_T accy_type, UINT8_T pairing_timeout, UINT8_T out_of_range_timeout, 
					   UINT8_T bt_state, UINT8_T pairing_type)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_bt_config_reply(xnl_devid, result, function, accy_type, pairing_timeout, out_of_range_timeout, bt_state, pairing_type) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}
/* Description: This function sends a BlueTooth Config Broadcast message. Assuming the 'function' field of the message 
 *				is XCMP_MSG_BT_POWER_UP_CONFIG.
 * Parameters:
 *		function: Functions to perform. Power Up Config: 0x04
 *		accy_type: Types of accessory. OPTION BOARD: 0x00, AUDIO ACCESSORY: 0x01,
 *					DATA ACCESSORY: 0x02, GENERIC / NON-AUDIO: 0x03
 *		pairing_timeout: Value in minutes. After which a disconnected accessory will be de-paired.
 *		out_of_range_timeout: Value in minutes. After which the link will be dropped for an accessory that is out of range.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_bt_config_msg_bcast(UINT16_T xnl_devid, UINT8_T function, 
					  UINT8_T accy_type, UINT8_T pairing_timeout, UINT8_T out_of_range_timeout, 
					   UINT8_T bt_state, UINT8_T pairing_type)
{
  if(xnl_device_tx_status(xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_bt_config_bcast(xnl_devid, function, accy_type, pairing_timeout, out_of_range_timeout, bt_state, pairing_type) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}
/* Description: This function receives a BlueTooth Config request message.
 * Parameters:
 *		function: Functions to perform. Power Up Config: 0x04
 *		accy_type: Types of accessory. OPTION BOARD: 0x00, AUDIO ACCESSORY: 0x01,
 *					DATA ACCESSORY: 0x02, GENERIC / NON-AUDIO: 0x03			
*/
void received_xcmp_bt_config_msg_rqst(UINT16_T xnl_devid, xcmp_msg_bt_cfg_request_t bt_cfg_rqst, UINT8_T optional_data_length)
{
  UINT8_T * ptr = NULL;
  
  if(bt_cfg_rqst.function == XCMP_MSG_BT_POWER_UP_CONFIG)
  {
    if(bt_cfg_rqst.data[0] == XCMP_MSG_BT_AUDIO_ACCESSORY_TYPE)
    {
      if(send_xcmp_bt_config_msg_rply(xnl_devid, XCMP_MSG_RESULT_SUCCESS, XCMP_MSG_BT_POWER_UP_CONFIG, XCMP_MSG_BT_AUDIO_ACCESSORY_TYPE,
                                                                                                       XCMP_MSG_BT_IMMEDIATE_PAIRING, 
                                                                                                       XCMP_MSG_BT_8_HR_TIMEOUT, XCMP_MSG_BT_STATE_ON, 
                                                                                                       XCMP_MSG_BT_MPP_PAIRING_TYPE) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_BT_CFG_RPLY_MSG, HSP_XCMP_BT_CONFIG_REQ_RCVD_FAIL, 50, NULL);
      }
      else
      {
        SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_BT_CONFIG_REQ_RCVD_SUCC, NULL, 0);
      }
    }
    else if(bt_cfg_rqst.data[0] == XCMP_MSG_BT_PTT_ACCESSORY_TYPE)
    {
      if(send_xcmp_bt_config_msg_rply(xnl_devid, XCMP_MSG_RESULT_SUCCESS, XCMP_MSG_BT_POWER_UP_CONFIG, XCMP_MSG_BT_PTT_ACCESSORY_TYPE, 
                                                                                                       XCMP_MSG_BT_IMMEDIATE_PAIRING, 
                                                                                                       XCMP_MSG_BT_8_HR_TIMEOUT, XCMP_MSG_BT_STATE_ON,
                                                                                                       XCMP_MSG_BT_MPP_PAIRING_TYPE) != TX_SUCCESS)
      {
        RETRY_XCMP_MSG_TIMER(HSP_BT_CFG_RPLY_MSG, SPP_XCMP_BT_CONFIG_REQ_RCVD_FAIL, 50, NULL);
      }   
      else
      {
        SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_BT_CONFIG_REQ_RCVD_SUCC, NULL, 0);
      }
    }
  }
  else
  {
    ptr = xnl_malloc(sizeof(UINT8_T));

    if(ptr == NULL)
    {
      // Error, no memory space
      xnl_device_report_failure(xnl_devid, XNL_MEMORY_ALLOC_ERROR);
    }
    // Copy the Function to be echoed
    *ptr = (UINT8_T)bt_cfg_rqst.function;
    /* Send Unsupported Parameter Reply */ 
    if(xnl_devid == device_1_id)
    {
      hsp_invalid_msg_xcmp_opcode = XCMP_MSG_OPCODE_REQUEST_BT_CONFIG;
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_RCVD_XCMP_INVALID_MSG, ptr, 1);  
    }
    else if(xnl_devid == device_2_id)
    {
      spp_invalid_msg_xcmp_opcode = XCMP_MSG_OPCODE_REQUEST_BT_CONFIG;
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_RCVD_XCMP_INVALID_MSG, ptr, 1);      
    }  
    else
    {
      xnl_free(ptr);
    }
  }  
}

//------------------------------------------------------------------------------------------------------------
// Shutdown
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends a Shutdown broadcast message.
 * Parameters:
 *		op: Describes the operation that needs to be performed on the device. See shutdown_operation_t.
 *		reason: Describes the reason for a request to shut down the device. Valid when Operation value is Shut 
 *				Down Device. See shutdown_reason_t.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_shutdown_msg_bdst(UINT16_T xnl_devid, shutdown_operation_t op, shutdown_reason_t reason)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_shutdown_broadcast(xnl_devid, op, reason, NO_OPTIONAL_PARAM) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}

//------------------------------------------------------------------------------------------------------------
// Ping
//------------------------------------------------------------------------------------------------------------
/* Description: The function sends a Ping request message.
 * Parameters: 
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_ping_msg_rqst(UINT16_T xnl_devid)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if(xcmp_msg_tx_ping_request(xnl_devid) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}
/* Description: The function sends a Ping request message.
 * Parameters: 
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_ping_msg_rply(UINT16_T xnl_devid, UINT8_T result)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if(xcmp_msg_tx_ping_reply(xnl_devid, result) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}
/* Description: This function receives a Ping request message.
 * Parameters:
 *		device id and ping request message
*/
void received_xcmp_ping_msg_rqst(UINT16_T xnl_devid, xcmp_msg_ping_request_t ping_rqst)
{
  if(xnl_devid == device_1_id)
  {
    SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_PING_MSG, NULL, 0);
  }
  else if(xnl_devid == device_2_id)
  {
    SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_PING_MSG, NULL, 0);
  }
}
/* Description: This function receives a Ping reply message.
 * Parameters:
 *		result: XCMP reply result code. Success: 0x00, Failure: 0x01
*/
void received_xcmp_ping_msg_rply(UINT16_T xnl_devid, xcmp_msg_ping_reply_t ping_rply)
{
  if(ping_rply.result == XCMP_MSG_RESULT_SUCCESS)
  {
    //count the ticks for determining the quality
  }
}
//------------------------------------------------------------------------------------------------------------
// PUI
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends Physical User Input broadcast message.
 * Parameters:
 *		type: Identifies the type of physical input being reported.
 *		pu_id: Identifies which physical input is being reported. 
 *			E.g. Vol_up: 0x0050, Vol_down: 0x0051.
 *		pu_state: Identifies the state of the physical input being reported.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_puinput_msg_bdst(UINT16_T xnl_devid, UINT8_T source, UINT8_T type, UINT16_T pu_id, UINT8_T pu_state)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
      if (xcmp_msg_tx_pui_broadcast(xnl_devid, source, type, pu_id, pu_state, XCMP_MSG_PUI_BUTTON_STATE_MAX) == SUCC)
              return TX_SUCCESS;
      else
              return TX_FAILURE;
  }
  else
  {
      return TX_BUSY;
  }
}

/* Description: This function receives Physical User Input broadcast message.
 * Parameters:
 *		source: Designates the origin of the physical input.
 *		type: Identifies the type of physical input being reported.
 *		id: Identifies which physical input is being reported.
 *			E.g. ACCY 1-dot: 0x0091, ACCY 2-dot: 0x0092, ACCY orange: 0x0097
 *		state: Identifies the state of the physical input being reported.
 *		state_min: minimum state value 
 *		state_max: maximum state value 
*/
void received_xcmp_puinput_msg_bdst(UINT16_T xnl_devid, xcmp_msg_pui_broadcast_t pui_bcast)
{
  if(xnl_devid == device_1_id)
  {
    switch(pui_bcast.id)
    {
      case PUI_ID_TOP_FRONT_ORANGE:
      {
        // This is received when the accessory orange button has been pressed.
        if(pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_PRESS && !host_in_emergency_state)	// 0x0097
        {
          host_in_emergency_state = TRUE;
          // Send the 2 dot button press
          two_dot_programable_button_handler(PRESSED);
        }
        else if (pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_RELEASE && host_in_emergency_state)
        {
          host_in_emergency_state = FALSE;
          // Send the 2 dot button release
          two_dot_programable_button_handler(RELEASED);
        }	
      }
      break;
      // Handle the volume request messages
      case PUI_ID_ROTATORY_VOLUME:
      {
        accy_vol = pui_bcast.state;
        //Send message to connection manager to send XCMP Volume broadcast out
        if( device_1_id != 0 ) SEND_MESSAGE_TO_CONNECTION_MNGR(SEND_XCMP_VOL_BCAST_NO_TONE, NULL, 0);
      /*  //Dont send it to SPP device to avoid extra traffic on BT link
        if( device_2_id != 0 ) SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_SEND_XCMP_VOL_BCAST, NULL, 0);        */
      }
      break;
      //Handle the task light button press/release as 1-dot if configured on WRSM 
      case PUI_ID_ACCY_1_DOT:
      case PUI_ID_PORTABLE_SIDE_MIDDLE:
      {
        if(XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_PRESS == pui_bcast.state)
        {
          one_dot_programable_button_handler(PRESSED);
        }
        else if(XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_RELEASE == pui_bcast.state)
        {
          one_dot_programable_button_handler(RELEASED);
        }
      }
      break;
      case PUI_ID_ACCY_2_DOT:
      case PUI_ID_PORTABLE_SIDE_BUTTON:
      {
        if(XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_PRESS == pui_bcast.state)
                one_dot_test_flag = TRUE;
        else if(XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_RELEASE == pui_bcast.state)
                one_dot_test_flag = FALSE;
      }    
      break;
      case PUI_ID_ACCY_NO_DOT:
      case PUI_ID_PORTABLE_SIDE_TOP:
      {
        if(XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_PRESS == pui_bcast.state)
                one_dot_test_flag = TRUE;
        else if(XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_RELEASE == pui_bcast.state)
                one_dot_test_flag = FALSE;
      }   
      break;
      // VOLUME
      case PUI_ID_VOL_UP: 
      {
        if(pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_PRESS)
        {
          //Calculate the new volume
          accy_vol = accy_vol + accy_step_size;
          if (accy_vol >= MAX_VOL)
          {  
            accy_vol = MAX_VOL;	
          }          
          //Send message to connection manager to send XCMP Volume broadcast out
          if( device_1_id != 0) SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_SEND_XCMP_VOL_BCAST, NULL, 0);        
        /*  if( device_2_id != 0) RETRY_XCMP_MSG_TIMER(SPP_VOL_BCAST_MSG, SPP_SEND_XCMP_VOL_BCAST, 25, NULL);*/
        }
        else if (pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_RELEASE)
        {
          /* No action required as per design */
        }  
      }
      break;
      case PUI_ID_VOL_DOWN:
      {
        if(pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_PRESS)
        {
          //Calculate the new volume
          accy_vol = accy_vol - accy_step_size;
          if((accy_vol <= MIN_VOL) || (accy_vol >= MAX_VOL))
          {  
            accy_vol = MIN_VOL;	
          }         
          //Send message to connection manager to send XCMP Volume broadcast out
          if( device_1_id != 0) SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_SEND_XCMP_VOL_BCAST, NULL, 0);        
        /*  if( device_2_id != 0) RETRY_XCMP_MSG_TIMER(SPP_VOL_BCAST_MSG, SPP_SEND_XCMP_VOL_BCAST, 25, NULL);*/
        }
        else if (pui_bcast.id == PUI_ID_VOL_DOWN && pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_RELEASE)
        {
          /* No action required as per design */
        }       
      }
      break;
      default:
        break;
    }
  }
  else if(xnl_devid == device_2_id)
  {
    switch(pui_bcast.id)
    {
      // VOLUME
      case PUI_ID_VOL_UP:
      {
        if(pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_PRESS)
        {
          accy_vol += accy_step_size;
          if (accy_vol >= MAX_VOL)
          {  
            accy_vol = MAX_VOL;	
          }          
          if( device_2_id != 0 ) SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_SEND_XCMP_VOL_BCAST, NULL, 0);        
        /*  if( device_1_id != 0 ) RETRY_XCMP_MSG_TIMER(SPP_VOL_BCAST_MSG, SPP_SEND_XCMP_VOL_BCAST, 25, NULL);*/
        }
        else if (pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_RELEASE)
        {
          /* No action required as per design */
        }
        break;
      }
      case PUI_ID_VOL_DOWN:
      {
        if(pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_PRESS)
        {
          accy_vol -= accy_step_size;
		  // As after the step size subtraction the value can wrap around, it is equated MIN & MAX VOL
          if((accy_vol <= MIN_VOL) || (accy_vol >= MAX_VOL))
          {  
            accy_vol = MIN_VOL;	
          }          
          if( device_2_id != 0 ) SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_SEND_XCMP_VOL_BCAST, NULL, 0);        
        /*  if( device_1_id != 0 ) RETRY_XCMP_MSG_TIMER(SPP_VOL_BCAST_MSG, SPP_SEND_XCMP_VOL_BCAST, 25, NULL);*/
        }
        else if (pui_bcast.state == XCMP_MSG_PUI_STATE_MOMENTARY_BUTTON_RELEASE)
        {
          /* No action required as per design */
      }  
        break;
      }
      default:
        break;
    }
  }
}
//------------------------------------------------------------------------------------------------------------
// Battery level
//------------------------------------------------------------------------------------------------------------
void received_xcmp_batt_lvl_msg_bdst(UINT16_T xnl_devid, xcmp_msg_batt_lvl_broadcast_t batt_lvl_bcast)
{
  if(xnl_devid == device_2_id)
  {
    if(batt_lvl_bcast.batt_state == XCMP_BATTERY_LOW)
    {
      //Send a message to Start a tone
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_BATT_LVL_LOW, NULL, 0);
    }
    else if(batt_lvl_bcast.batt_state == XCMP_BATTERY_OKAY)
    {
      //Send a message to Stop a tone
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_BATT_LVL_OKAY, NULL, 0);      
    }
  }
}

//------------------------------------------------------------------------------------------------------------
// Radio status
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends a Radio Status reply message. FRSM sends a Radio Status request message as
 *				a way to check the USB connection. So it really doesn't matter in what status the Host is.
 * Parameters:
 *		condition: Condition is the same parameter echoed back from the Radio_Status message. 
 *		status: Status is the state of the condition in the radio or the value indicating the Model Number, 
 *				Serial Number or ESN. 
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_radio_status_msg_rply(UINT16_T xnl_devid, UINT8_T result, UINT8_T condition, UINT8_T status)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_radio_status_reply(xnl_devid, result, condition, status) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }
}

/* Description: This function receives a Radio Status request message.
 * Parameters:
 *		condition: Specifies which parameter of the radio to read the status of.
*/
void received_xcmp_radio_status_msg_rqst(UINT16_T xnl_devid, xcmp_msg_radio_status_request_t radio_status_req)
{
  if(xnl_devid == device_1_id)
  {
    if(radio_status_req.condition == RAD_STAT_LOW_BATT)
    {
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_RADIO_STATUS_MSG, NULL, 0);
      /* Check if ACK feature is enabled as any ACK received restarts the ping monitoring timer */
      if(xnl_get_device_ack_feature(device_1_id) == TRUE)
      {
        hsp_ping_rcvd = TRUE;
        BT_SYSTEM_PING_TIMER(HSP_PING, HSP_SYSTEM_PING_TIMER_EXPIRED, SYSTEM_PING_TIME, NULL);
      }
    }
    else
    {
      hsp_invalid_msg_xcmp_opcode = XCMP_MSG_OPCODE_REQUEST_RADIO_STATUS;
      SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_RCVD_XCMP_INVALID_MSG, NULL, 0);      
    }    
  }
  else if(xnl_devid == device_2_id)
  {
    if(radio_status_req.condition == RAD_STAT_LOW_BATT)
    {
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_RADIO_STATUS_MSG, NULL, 0);
      /* Check if ACK feature is enabled as any ACK received restarts the ping monitoring timer */      
      if(xnl_get_device_ack_feature(device_2_id) == TRUE)
      {     
        spp_ping_rcvd = TRUE;
        BT_SYSTEM_PING_TIMER(SPP_PING, SPP_SYSTEM_PING_TIMER_EXPIRED, SYSTEM_PING_TIME, NULL);
      }
    }
    else
    {
      spp_invalid_msg_xcmp_opcode = XCMP_MSG_OPCODE_REQUEST_RADIO_STATUS;
      SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_RCVD_XCMP_INVALID_MSG, NULL, 0);
    }
  }
}
//------------------------------------------------------------------------------------------------------------
// Codeplug Read XCMP message
//------------------------------------------------------------------------------------------------------------
tx_result_t send_xcmp_cplg_read_rqst(UINT16_T xnl_devid, UINT16_T block_index, UINT16_T block_index_val, UINT16_T field_id, UINT8_T block_index_reqd)
{
  if(xnl_device_tx_status (xnl_devid) != TX_BUSY)
  {
    if (xcmp_msg_tx_cplg_read_request(xnl_devid, block_index, block_index_val, field_id, block_index_reqd) == SUCC)
            return TX_SUCCESS;
    else
            return TX_FAILURE;
  }
  else
  {
    return TX_BUSY;
  }  
}
void received_xcmp_cplg_read_msg_rply(UINT16_T xnl_devid, xcmp_msg_cplg_read_reply_t cplg_read_reply)
{
  UINT16_T field_id = 0;
  INT16_T field_id_size = 0;
  UINT32_T cplg_data = 0;
  INT8_T i;
  
  if(cplg_read_reply.result == XCMP_MSG_RESULT_SUCCESS)
  {
    field_id = cplg_read_reply.field_id; 
    field_id_size = (cplg_read_reply.field_id_val_size - 1); 

    for(i = 0; i <= (cplg_read_reply.field_id_val_size - 1); i++, field_id_size--)
    {
      cplg_data |= (cplg_read_reply.field_val[i] << (field_id_size*8));
    }   
    if(field_id == GW_TPT_DELAY_CPLG_FIELD)
    {     
      if(cplg_data != 0)
      {
        apply_tpt_issue_delay = cplg_data;
      }      
      SEND_MESSAGE_TO_CONNECTION_MNGR(GW_TPT_DELAY_CPLG_READ_REPLY_RCVD, NULL, 0);
    }
    else if (field_id == GW_PTT_PRESS_NIBBLER_TMO_CPLG_FIELD)
    {
      if(cplg_data != 0)
      {
        nibbler_ptt_press_timeout = cplg_data;
      }
      SEND_MESSAGE_TO_CONNECTION_MNGR(GW_PTT_TMO_CPLG_READ_REPLY_RCVD, NULL, 0);
    }
    else if (field_id == GW_USE_NEVER_REPAIR_FIELD)
    {
      if(cplg_data != 0)
      {
        gw_never_repair = true;
      }
      else
      {
        gw_never_repair = false;
      }
      SEND_MESSAGE_TO_CONNECTION_MNGR(GW_NEVER_REPAIR_CPLG_READ_REPLY_RCVD, NULL, 0);
    }    
  }
}
//------------------------------------------------------------------------------------------------------------
// Unsupported XCMP message
//------------------------------------------------------------------------------------------------------------
void receive_xcmp_unsupported_msg_rqst(UINT16_T xnl_devid, UINT16_T xcmp_opcode)
{
  if(xnl_devid == device_1_id)
  {
    hsp_unsupported_xcmp_opcode = xcmp_opcode;
    SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_XCMP_UNSUPPORTED_MSG, NULL, 0);
  }
  else if(xnl_devid == device_2_id)
  {
    spp_unsupported_xcmp_opcode = xcmp_opcode;
    SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_XCMP_UNSUPPORTED_MSG, NULL, 0);
  }
}

//------------------------------------------------------------------------------------------------------------
// Invalid XCMP message
//------------------------------------------------------------------------------------------------------------
void receive_xcmp_invalid_msg_rqst(UINT16_T xnl_devid, UINT16_T xcmp_opcode)
{
  if(xnl_devid == device_1_id)
  {
    hsp_invalid_msg_xcmp_opcode = xcmp_opcode;
    SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_RCVD_XCMP_INVALID_MSG, NULL, 0);
  }
  else if(xnl_devid == device_2_id)
  {
    spp_invalid_msg_xcmp_opcode = xcmp_opcode;
    SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_RCVD_XCMP_INVALID_MSG, NULL, 0);
  }  
}