#ifndef XCMP_CONFIG_H
#define XCMP_CONFIG_H

/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013-2015 MOTOROLA SOLUTIONS,INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: xcmp_config.h
#
# --------------------------- General Description -----------------------------
# The file includes configuration of XCMP functions.
# Modify this file to port to other project using XCMP functions.
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/

#include "xcmp.h"
#include "xcmp_msg_dev_init_status.h"
#include "xcmp_msg_vol_cntrl.h"
#include "xcmp_msg_pu_input.h"
#include "xcmp_msg_tone_cntrl.h"
#include "xcmp_msg_shutdown.h"
#include "xcmp_msg_ver_info.h"
#include "xcmp_msg_bt_config.h"
#include "xcmp_msg_ping.h"
#include "xcmp_msg_batt_lvl.h"
#include "xcmp_msg_radio_status.h"
#include "xcmp_msg_spkr_cntrl.h"
#include "xcmp_msg_cplg_read.h"
#include "utilities.h"
#include "xnl.h"
/************************************************************************************
*
*--------------------------------- Revision History ----------------------------------
*
*   AUTHOR              Date Modified   Tracking Number         Description
* Abhishek Trivedi      08/25/2013      CCMPD01827271     XNL-XCMP host
* Abhishek Trivedi      11/8/2013       CCMPD01834482     Updates for XNL-XCMP 
* Abhishek Trivedi      11/19/2013      CCMPD01836292     Add the BT Config Broadcast message to send 
* WRN637                01/07/2014      CCMPD01849957     Volume tone request support
* Abhishek Trivedi      03/18/2014	CCMPD01873731     Resolve 2 device connection issue
* Abhishek Trivedi      10/22/2014      CCMPD01938895     Audio Switch
* Abhishek Trivedi      11/02/2014      CCMPD01943851     Codeplug Read Request for Talk Permit Tone issue
* Abhishek Trivedi      01/14/2015      CCMPD01958122     GW supports infinte pairing
**--------------------------- End of History Template-----------------------------
* *********************************************************************************/

//------------------------------------------------------------------------------
//      Defines & Globals
//------------------------------------------------------------------------------
//Do not change this as it is a constant
#define ATTR_AND_VALUE  2
/////////////////////////

#define NUMBER_OF_ATTRIBUTES    		3
#define DEVICE_DESCRIPTOR_SIZE		((NUMBER_OF_ATTRIBUTES) * (ATTR_AND_VALUE))			// Max = 10*2

#define SEND_OPTIONAL_PARAM             0x01
#define NO_OPTIONAL_PARAM               0x00

#define DEFAULT_STEP_SIZE				0x20

//Gateway Codeplug Field in WRSM
#define GW_TPT_DELAY_CPLG_FIELD                     0x0000
#define GW_PTT_PRESS_NIBBLER_TMO_CPLG_FIELD         0x0001
#define GW_USE_NEVER_REPAIR_FIELD                   0x0002

typedef enum 
{
  ACCY_VER_REQ_NOT_SENT = 0,
  ACCY_HW_VER_MSG_SENT,
  ACCY_HW_VER_MSG_RCVD,
  ACCY_SW_VER_MSG_SENT,
  ACCY_SW_VER_MSG_RCVD
}ver_msg_state_t;

extern data_buffer_struct_t *ptr_to_xcmp_payload_glb;
extern UINT16_T ptr_new_message_rx_index;
//------------------------------------------------------------------------------
//      FUNCTION PROTOTYPES
//------------------------------------------------------------------------------
/*=============================================================================
	FUNCTION: xcmp_get_1byte()

	DESCRIPTION: Mimic function of buffer_read_and_increment_out_index()

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: UINT8_T
=============================================================================*/
#define xcmp_get_1byte()  (buffer_read_byte_no_increment(ptr_to_xcmp_payload_glb, ptr_new_message_rx_index++))
/*=============================================================================
	FUNCTION: xcmp_get_2bytes()

	DESCRIPTION: Mimic function of buffer_read_and_increment_out_index()

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: UINT16_T data
=============================================================================*/
UINT16_T xcmp_get_2bytes();
/*=============================================================================
	FUNCTION: xcmp_discard_buffer_bytes()

	DESCRIPTION: Mimic function of buffer_read_and_increment_out_index()

        ARGUMENTS PASSED: UINT16_T count: bytes to be discarded

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
#define xcmp_discard_buffer_bytes(count)  { (ptr_new_message_rx_index += count); buffer_read_byte_no_increment((ptr_to_xcmp_payload_glb), (ptr_new_message_rx_index)); }


//------------------------------------------------------------------------------
//    ENUMS &  STRUCTURES
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//      FUNCTION PROTOTYPES
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------------------
// PUI
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends Physical User Input broadcast message.
 * Parameters:
 *		type: Identifies the type of physical input being reported.
 *		pu_id: Identifies which physical input is being reported. 
 *			E.g. Vol_up: 0x0050, Vol_down: 0x0051.
 *		pu_state: Identifies the state of the physical input being reported.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_puinput_msg_bdst(UINT16_T xnl_devid, UINT8_T source, UINT8_T type, UINT16_T pu_id, UINT8_T pu_state);
						
//------------------------------------------------------------------------------------------------------------
// Volume control
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends Volume Control request message. The default Attenuator_Number is set to be
 *				ALL SPEAKERS.
 * Parameters:
 *		function: Specifies the function being requested. See volume_ctrl_func_t.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_volume_control_msg_rqst(UINT16_T xnl_devid, volume_ctrl_func_t function);

/* Description: This function sends Volume Control reply message. The default Attenuator_Number is set to be
 *				ALL SPEAKERS.
 * Parameters:
 *		result: XCMP reply result code. Success: 0x00, Failure: 0x01
 *		function: Specifies the function being requested. See volume_ctrl_func_t.
 *		volume_data: Contains the current setting of the specified volume attenuator.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_volume_control_msg_rply(UINT16_T xnl_devid, xcmp_reply_result_t result, volume_ctrl_func_t function, UINT8_T volume_data, UINT8_T send_opt_param);
/* Description: This function sends Volume Control broadcast message. The default Attenuator_Number is set to be
 *				ALL SPEAKERS.
 * Parameters:
 *		volume_data: Contains the current setting of the specified volume attenuator.
 *		audio_parameter: Contains Radio audio parameter setting. 
 *						 Radio Speaker Automatic Volume Control Enable/Disable: 0x01/0x00
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_volume_control_msg_bdst(UINT16_T xnl_devid, UINT8_T volume_data, UINT8_T audio_parameter);
//------------------------------------------------------------------------------------------------------------
// Speaker control
//------------------------------------------------------------------------------------------------------------
tx_result_t send_xcmp_speaker_control_msg_rqst(UINT16_T xnl_devid, UINT16_T speaker_number, UINT16_T function);
tx_result_t send_xcmp_speaker_control_msg_bdst(UINT16_T xnl_devid, UINT16_T speaker_number, UINT16_T function);
tx_result_t send_xcmp_speaker_control_msg_rply(UINT16_T xnl_devid, xcmp_reply_result_t result, UINT16_T speaker_number, UINT16_T function);
//------------------------------------------------------------------------------------------------------------
// Tone control
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends Tone Control Request message.
 * Parameters:
 *		control: Type of tone action to perform. Stop: 0x00, Start: 0x01
 *		volume: Specifies the volume. Current Volume: 0x00
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_logical_tone_control_msg_rqst(UINT16_T xnl_devid, UINT8_T control, UINT16_T identifier, UINT8_T volume);
/* Description: This function sends Tone Control broadcast message.
 * Parameters:
 *		control: Type of tone action to perform. Stop: 0x00, Start: 0x01
 *		volume: Specifies the volume. Current Volume: 0x00
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_tone_control_msg_bdst(UINT16_T xnl_devid, UINT8_T control, UINT8_T volume);

/* Description: This function sends Tone Control reply message.
 * Parameters:
 *		result: XCMP reply result code. Success: 0x00, Failure: 0x01
 *		duration: Value determined by repetition, timebase, and pattern (in ms); 0 means continuous
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_tone_control_msg_rply(UINT16_T xnl_devid, xcmp_reply_result_t result, UINT16_T tone_id, UINT16_T duration);

//------------------------------------------------------------------------------------------------------------
// Version info
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends a version info request message to accessory.
 * Parameters:
 *		type: The version type the Host supports. See ver_info_request_type_t.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_version_info_msg_rqst(UINT16_T xnl_devid, ver_info_request_type_t type);
tx_result_t send_xcmp_version_info_msg_rply(UINT16_T xnl_devid, UINT8_T result, UINT8_T* ver_ptr, UINT8_T size);

//------------------------------------------------------------------------------------------------------------
// Radio status
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends a Radio Status reply message. FRSM sends a Radio Status request message as
 *				a way to check the USB connection. So it really doesn't matter in what status the Host is.
 * Parameters:
 *		condition: Condition is the same parameter echoed back from the Radio_Status message. 
 *		status: Status is the state of the condition in the radio or the value indicating the Model Number, 
 *				Serial Number or ESN. 
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_radio_status_msg_rply(UINT16_T xnl_devid, UINT8_T result, UINT8_T condition, UINT8_T status);

/* Description: This function receives a Radio Status request message.
 * Parameters:
 *		condition: Specifies which parameter of the radio to read the status of.
*/
void received_xcmp_radio_status_msg_rqst(UINT16_T xnl_devid, xcmp_msg_radio_status_request_t condition);

//------------------------------------------------------------------------------------------------------------
// BlueTooth config
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends a BlueTooth Config reply message. Assuming the 'function' field of the message 
 *				is XCMP_MSG_BT_POWER_UP_CONFIG.
 * Parameters:
 *		result: XCMP reply result code. Success: 0x00, Failure: 0x01
 *		function: Functions to perform. Power Up Config: 0x04
 *		accy_type: Types of accessory. OPTION BOARD: 0x00, AUDIO ACCESSORY: 0x01,
 *					DATA ACCESSORY: 0x02, GENERIC / NON-AUDIO: 0x03
 *		pairing_timeout: Value in minutes. After which a disconnected accessory will be de-paired.
 *		out_of_range_timeout: Value in minutes. After which the link will be dropped for an accessory that is out of range.
 *		bt_state: N/A for Audio Accessory.
 *		pairing_type: N/A for Audio Accessory.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_bt_config_msg_rply(UINT16_T xnl_devid, xcmp_reply_result_t result, UINT8_T function, 
										UINT8_T accy_type, UINT8_T pairing_timeout, UINT8_T out_of_range_timeout, 
										UINT8_T bt_state, UINT8_T pairing_type);
/* Description: This function sends a BlueTooth Config Broadcast message. Assuming the 'function' field of the message 
 *				is XCMP_MSG_BT_POWER_UP_CONFIG.
 * Parameters:
 *		function: Functions to perform. Power Up Config: 0x04
 *		accy_type: Types of accessory. OPTION BOARD: 0x00, AUDIO ACCESSORY: 0x01,
 *					DATA ACCESSORY: 0x02, GENERIC / NON-AUDIO: 0x03
 *		pairing_timeout: Value in minutes. After which a disconnected accessory will be de-paired.
 *		out_of_range_timeout: Value in minutes. After which the link will be dropped for an accessory that is out of range.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_bt_config_msg_bcast(UINT16_T xnl_devid, UINT8_T function, 
					  UINT8_T accy_type, UINT8_T pairing_timeout, UINT8_T out_of_range_timeout, 
					   UINT8_T bt_state, UINT8_T pairing_type);

void receive_xcmp_bt_config_msg_rqst(UINT16_T xnl_devid, UINT8_T function, UINT8_T accy_type);

//------------------------------------------------------------------------------------------------------------
// Shutdown
//------------------------------------------------------------------------------------------------------------
/* Description: This function sends a Shutdown broadcast message.
 * Parameters:
 *		op: Describes the operation that needs to be performed on the device. See shutdown_operation_t.
 *		reason: Describes the reason for a request to shut down the device. Valid when Operation value is Shut 
 *				Down Device. See shutdown_reason_t.
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_shutdown_msg_bdst(UINT16_T xnl_devid, shutdown_operation_t op, shutdown_reason_t reason);

//------------------------------------------------------------------------------------------------------------
// Ping
//------------------------------------------------------------------------------------------------------------
/* Description: The function sends a Ping request message.
 * Parameters: 
 * Returns:
 *		tx_result_t: TX_SUCCESS, TX_FAILURE, or TX_BUSY
*/
tx_result_t send_xcmp_ping_msg_rqst(UINT16_T xnl_devid);
tx_result_t send_xcmp_ping_msg_rply(UINT16_T xnl_devid, UINT8_T result);
//------------------------------------------------------------------------------------------------------------
// Codeplug Read XCMP message
//------------------------------------------------------------------------------------------------------------
tx_result_t send_xcmp_cplg_read_rqst(UINT16_T xnl_devid, UINT16_T block_index, UINT16_T block_index_val, UINT16_T field_id, UINT8_T block_index_reqd);
//------------------------------------------------------------------------------------------------------------
// Unsupported XCMP message
//------------------------------------------------------------------------------------------------------------
void receive_xcmp_unsupported_msg_rqst(UINT16_T xnl_devid, UINT16_T xcmp_opcode);
void receive_xcmp_invalid_msg_rqst(UINT16_T xnl_devid, UINT16_T xcmp_opcode);
#endif //XCMP_CONFIG_H