/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013-2014 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: xcmp_msg_ver_info.c
#
# --------------------------- General Description -----------------------------
# The file includes handler function for XCMP Tone Control messages. 
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/
#include "xnl.h"
#include "xcmp_config.h"
/************************************************************************************
*
*--------------------------------- Revision History ----------------------------------
*
*   AUTHOR               Date Modified        Tracking Number         Description
* Abhishek Trivedi       09/06/2013          CCMPD01827271           XNL-XCMP Host
* Abhishek Trivedi       11/8/2013           CCMPD01834482       Updates for XNL-XCMP 
* Abhishek Trivedi      03/18/2014	     CCMPD01873731	  Rework for XNL-XCMP 
**--------------------------- End of History Template-----------------------------
* *********************************************************************************/

/* ***************************************************************************
                                DEFINES
   ************************************************************************** */

/* ***************************************************************************
                                ENUMS & STRUCTURES
   ************************************************************************** */

/* ***************************************************************************
                                GLOBAL VARIABLES
   ************************************************************************** */                                               

/* ***************************************************************************
                                EXTERNAL VARIABLES
   ************************************************************************** */

/* ***************************************************************************
                                EXTERNAL FUNCTIONS
   ************************************************************************** */

/* ***************************************************************************
                                FUNCTION PROTOTYPES
   ************************************************************************** */
/******************************************************************************************
 *  FUNCTION NAME: xcmp_msg_tx_tone_cntlr_request()
 *
 *  DESCRIPTION: This function will build the TONECTRLREQ message based on events  
 ******************************************************************************************/
success_failure_t xcmp_msg_tx_tone_cntlr_request(UINT16_T xnl_devid, UINT8_T control, UINT16_T identifier, UINT8_T vol_ctrl, UINT8_T * tone_spec, UINT8_T tone_spec_freq_size)
{	
  xcmp_msg_tone_control_request_t outgoing_msg;
  UINT8_T message_size = XCMP_MSG_TONE_REQUEST_SIZE;
  
  outgoing_msg.opcode         = XCMP_MSG_OPCODE_REQUEST_TONE_CNTRL;
  outgoing_msg.ToneControl    = control;
  outgoing_msg.ToneIdentifier = identifier;
  outgoing_msg.ToneVolumeCtrl = vol_ctrl;
  
  // Only send out the tone specifics if the logical tone identifier is 0.
  if((tone_spec != NULL) && (tone_spec_freq_size != 0) && identifier == XCMP_MSG_SPECIFIC_TONE)
  {
    outgoing_msg.SpecificTone.tone_specifics_repetition     = *tone_spec++;
    outgoing_msg.SpecificTone.tone_specifics_timebase       = *tone_spec++;
    outgoing_msg.SpecificTone.tone_specifics_priority       = *tone_spec++;
    outgoing_msg.SpecificTone.tone_specifics_numfrequencies = *tone_spec++;
    //Populated 4 bytes
    tone_spec_freq_size -=4;
    message_size +=4;
    //Populate the frequency structure
    for(UINT8_T i=0; i<tone_spec_freq_size && i<XCMP_MSG_TONE_CTRL_MAX_FRQ_NUMBER; i++, message_size++)
    {
      outgoing_msg.SpecificTone.frequency[i] = *tone_spec++;
    }
  }
  // XNL to send data
  return xnl_send_data_msg(xnl_devid, (UINT8_T *)&outgoing_msg, message_size, XNL_DATA_MSG_REQUEST);
}

/******************************************************************************************
 *  FUNCTION NAME: xcmp_msg_tx_tone_cntlr_broadcast()
 *
 *  DESCRIPTION: This function will build and send the TONE CTRL BROADCAST message  
 ******************************************************************************************/
success_failure_t xcmp_msg_tx_tone_cntlr_broadcast(UINT16_T xnl_devid, UINT8_T control, UINT16_T identifier, UINT8_T volume, UINT8_T * tone_spec, UINT8_T tone_spec_size)
{	
  xcmp_msg_tone_control_broadcast_t outgoing_msg;
  
  outgoing_msg.opcode = XCMP_MSG_OPCODE_BROADCAST_TONE_CNTRL;
  outgoing_msg.ToneControl = control;
  outgoing_msg.ToneIdentifier = identifier;
  outgoing_msg.ToneVolumeCtrl = volume;
  
  // Only send out the Tone specifics when the Tone Control is Start
  if((tone_spec != NULL) && (tone_spec_size >= XCMP_MSG_TONE_SPECIFICS_SIZE) && control == XCMP_MSG_TONE_CTRL_START)
  {
    for(UINT8_T i=0; i<tone_spec_size; i++)
    {
      outgoing_msg.ToneSpecifics[i] = *tone_spec++;
    }
  }
  else
  { //Fill 0's for Tone Specifics array
    for(UINT8_T i=0; i<XCMP_MSG_TONE_SPECIFICS_SIZE; i++)
    {
      outgoing_msg.ToneSpecifics[i] = 0;
    }    
  }
  
  // XNL to send data
  return xnl_send_data_msg(xnl_devid, (UINT8_T *)&outgoing_msg, (UINT8_T)XCMP_MSG_TONE_CTRL_BROADCAST_SIZE, XNL_DATA_MSG_BRDCST);
}

/******************************************************************************************
 *  FUNCTION NAME: xcmp_msg_tx_tone_cntlr_reply()
 *
 *  DESCRIPTION: This function will build the TONECTRLREQ message based on events  
 ******************************************************************************************/
success_failure_t xcmp_msg_tx_tone_cntlr_reply(UINT16_T xnl_devid, xcmp_reply_result_t result, UINT16_T identifier, UINT16_T duration)
{	
  xcmp_msg_tone_control_reply_t outgoing_msg;
  
  outgoing_msg.opcode           = XCMP_MSG_OPCODE_REPLY_TONE_CNTRL;
  outgoing_msg.result           = result;
  outgoing_msg.toneIdentifier   = identifier;
  outgoing_msg.duration         = duration;
  
  // XNL to send data
  return xnl_send_data_msg(xnl_devid, (UINT8_T *)&outgoing_msg, (UINT8_T)XCMP_MSG_TONE_CTRL_REPLY_SIZE, XNL_DATA_MSG_REPLY);
}

/*=============================================================================
	FUNCTION: xcmp_msg_rx_tone_cntlr_request()
	
	DESCRIPTION: This function processes the Tone Control request.
==============================================================================*/
xcmp_return_value_t xcmp_msg_rx_tone_cntlr_request(UINT16_T xnl_devid, UINT8_T payload_length)
{
  UINT8_T remaining_length = 0;
  UINT8_T no_of_freqs = 0;
  if (payload_length >= XCMP_MSG_TONE_CTRL_REQUEST_MIN_MSG_SIZE )
  {
    xcmp_msg_tone_control_request_t incoming_msg;
            
    // Decode the message			
    incoming_msg.opcode	            = XCMP_MSG_OPCODE_REQUEST_TONE_CNTRL;											
    incoming_msg.ToneControl        = xcmp_get_1byte();
    incoming_msg.ToneIdentifier     = xcmp_get_2bytes();
    incoming_msg.ToneVolumeCtrl     = xcmp_get_1byte();
    
    if(incoming_msg.ToneIdentifier == XCMP_MSG_SPECIFIC_TONE)
    {
      incoming_msg.SpecificTone.tone_specifics_repetition     = xcmp_get_1byte();
      incoming_msg.SpecificTone.tone_specifics_timebase       = xcmp_get_1byte();
      incoming_msg.SpecificTone.tone_specifics_priority       = xcmp_get_1byte();
      incoming_msg.SpecificTone.tone_specifics_numfrequencies = xcmp_get_1byte();   
      // Read 10 bytes of payload
      remaining_length = payload_length - 10;
      no_of_freqs = incoming_msg.SpecificTone.tone_specifics_numfrequencies;
      // Get the remaining bytes for the frequency structure
      if(no_of_freqs <= SPECIFIC_TONE_MAX_FREQS_ALLOWED)
      {
        for(UINT8_T i=0; (i < (no_of_freqs * SPECIFIC_TONE_FREQ_AND_PATTERN_SIZE)) && (i<remaining_length); i++)
        {
          incoming_msg.SpecificTone.frequency[i] = xcmp_get_1byte();
        }      
      }
      else
      {
        for(UINT8_T i=0; (i < XCMP_MSG_TONE_CTRL_MAX_FRQ_NUMBER) && (i<remaining_length); i++)
        {
          incoming_msg.SpecificTone.frequency[i] = xcmp_get_1byte();
        }         
      }
    }
    //Pass decoded message to user application
    received_xcmp_tone_control_msg_rqst(xnl_devid, incoming_msg);
            
    return XCMP_SUCCESS;
  }  
  else
  {
    return XCMP_INSUFFICIENT_DATA;
  }
} 

/*=============================================================================
	FUNCTION: xcmp_msg_rx_tone_cntlr_reply()
	
	DESCRIPTION: This function processes the Tone Control Reply.
==============================================================================*/
xcmp_return_value_t xcmp_msg_rx_tone_cntlr_reply(UINT16_T xnl_devid, UINT8_T payload_length)
{
  if (payload_length >= XCMP_MSG_TONE_CTRL_REPLY_MIN_MSG_SIZE )
  {
    xcmp_msg_tone_control_reply_t incoming_msg;
            
    // Decode the message			
    incoming_msg.opcode	            = XCMP_MSG_OPCODE_REPLY_TONE_CNTRL;		
    incoming_msg.result             = xcmp_get_1byte();
    incoming_msg.toneIdentifier     = xcmp_get_2bytes();
    incoming_msg.duration           = xcmp_get_2bytes();  
    //Pass decoded message to user application
    received_xcmp_tone_control_msg_rply(xnl_devid, incoming_msg);
            
    return XCMP_SUCCESS;    
  }
  else
  {
    return XCMP_INSUFFICIENT_DATA;
  }  
  
}
/* End of xcmp_msg_rx_tone_control() */

