
/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013-2014 MOTOROLA SOLUTIONS, INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: xnl.c
#
# --------------------------- General Description -----------------------------
#
# This file contains the XNL stack engine. The user should not have to modify any
# of the functionality written in this file. User will need to make appropriate
# API calls written in this function when integrating the XNL stack.
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/
#include "xnl.h"
#include "xnl_config.h"
#include "utilities.h"
#include "xcmp.h"
#include "xcmp_config.h"
/********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
*   AUTHOR              Date Modified         Tracking Number         Description
* Abhishek Trivedi      09/02/2013            CCMPD01827271       XNL Interfaces
* Abhishek Trivedi      11/8/2013             CCMPD01834482       Updates for XNL-XCMP 
* WRN637                20/12/2013            CCMPD01833277       WPTT state machine
* WRN637                01/22/2014            CCMPD01854392       Move XCMP related handling to HSP_XNL_XCMP_MANAGER_TASK to prevent multiple access to XCMP decoder at the same time    
* Abhishek Trivedi      03/18/2014	      CCMPD01873731	  Rework for XNL-XCMP 
* Abhishek Trivedi      04/18/2014            CCMPD01885148       System Ping Timeout
* Phong Tran            05/12/2014            CCMPD01885130       Modified to support XXT device
* Abhishek Trivedi      11/11/2014            CCMPD01945832       Edge of Range Support
**--------------------------- End of History Template-----------------------------
* *********************************************************************************/

//------------------------------------------------------------------------------
//      PRIVATE ENUMERATIONS (User is discouraged to modifying these)
//------------------------------------------------------------------------------


typedef
#if __ICCAVR32__
#pragma pack(1)
#endif
struct
#if (defined __GNUC__)
__attribute__((__packed__))
#endif
{
  UINT16_T opcode;
  UINT8_T  protocol_id;
  UINT8_T  xnl_flags;
  UINT16_T destination_address;
  UINT16_T source_address;
  UINT16_T transaction_id;
  UINT16_T payload_length;
} 
#if __ICCAVR32__
#pragma pack()
#endif 
xnl_header_t;

//------------------------------------------------------------------------------
//      PRIVATE GLOBAL VARIABLES (User is discouraged to modifying these)
//------------------------------------------------------------------------------
// XNL device information
UINT16_T address_generator_glb;
UINT16_T number_of_devices_glb;
UINT16_T relay_source = 0x0000;

xnl_device_struct_t *xnl_device[NUMBER_OF_DEVICES_SUPPORTED];

// Unencrypted and encrypted arrays
UINT16_T encrypted_auth_value_array_glb[4];
UINT32_T unencrypted_auth_value_array_glb[2];

//------------------------------------------------------------------------------
//      PRIVATE FUNCTIONS (User should not have to modify implementation of these)
//------------------------------------------------------------------------------
void xnl_initialize_structure(xnl_device_struct_t *this_device);
void xnl_init_buffer(void);
UINT16_T xnl_buffer_read_two_bytes(data_buffer_struct_t * ptr);
void xnl_buffer_store_retry_message(const UINT8_T index, const UINT8_T* header_ptr, const UINT8_T header_size, const UINT8_T* payload_ptr, const UINT16_T payload_size);
success_failure_t xnl_send_retry_message(const UINT8_T index);
success_failure_t xnl_send_master_status_brdcst_msg(const UINT8_T index);
success_failure_t xnl_send_data_msg_ack(const UINT8_T index);
success_failure_t xnl_send_device_auth_key_reply_msg(const UINT8_T index);
success_failure_t xnl_send_device_sysmap_brdcst_msg(const UINT8_T index);
success_failure_t xnl_send_device_conn_reply_msg(const UINT8_T index);
success_failure_t xnl_send_data_to_hardware(const UINT8_T index, const UINT8_T *header_ptr, const UINT8_T header_size, const UINT8_T *payload_ptr, const UINT16_T payload_size);
success_failure_t xnl_send_relayed_data_msg(const UINT16_T destination_address, const UINT16_T payload_length);
success_failure_t xnl_is_valid_opcode(const UINT8_T opcode);
success_failure_t xnl_is_valid_transaction_id (const UINT8_T index, const UINT16_T transaction_id_to_check);
success_failure_t xnl_verify_source_address (const UINT8_T index, const UINT16_T address_to_check);
success_failure_t xnl_verify_destination_address (const UINT16_T address_to_check);
success_failure_t xnl_is_valid_ack_flag (const UINT8_T index, const UINT8_T flag_to_check);
success_failure_t xnl_decode_message(const UINT16_T expected_opcode, const UINT8_T index, UINT16_T bytes_to_check);
success_failure_t xnl_check_unused_address(UINT16_T xnl_devid);
void xnl_state_machine_init(const UINT8_T index, const UINT16_T bytes_to_check);
void xnl_state_machine_idle(const UINT8_T index, const UINT16_T bytes_to_check);
void xnl_set_address(UINT8_T index);
BOOL_T xnl_get_unused_device_index (UINT8_T *index);
UINT16_T xnl_calculate_transaction_id (const UINT8_T index);
extern void xnl_device_report_failure (UINT16_T xnl_devid, xnl_error_type_t error);
/*=============================================================================
	FUNCTION: xnl_initialize_structure()

	DESCRIPTION: Initializes the members of a device structure
=============================================================================*/
void xnl_initialize_structure(xnl_device_struct_t *this_device)
{
  this_device->transaction_id_base = 0;
  this_device->transaction_id_counter = 0;
  this_device->transaction_id = 0;
  this_device->address = 0;
  this_device->device_type = 0;
  this_device->rx_flag = XNL_RX_FLAG_RESET;
  this_device->tx_flag = 0;
  this_device->auth_level = 0;
  this_device->retry_message_size = 0;
  this_device->retry_buffer_ptr = NULL;
  this_device->retry_counter = 0;
  this_device->is_ack_feature_enabled = FALSE;
  this_device->is_xnl_authentication_complete = FALSE;
  this_device->is_xcmp_authentication_complete = FALSE;
  this_device->is_ack_pending = FALSE;
  this_device->state = WAITING_FOR_DEVICE_KEY_REQUEST_MSG;
  this_device->hw_message_size = 0;
  this_device->hw_payload_counter = 0;
  this_device->hw_payload_size = 0;
  this_device->hw_state = WAITING_FOR_XNL_OPCODE_HI;
  this_device->xnl_receive_buffer_glb.data_buffer_array_ptr = this_device->xnl_receive_array_glb;
  this_device->xnl_receive_buffer_glb.buffer_size = XNL_RX_BUFFER_SIZE;
  this_device->xnl_receive_buffer_glb.in_buffer_index = 0;
  this_device->xnl_receive_buffer_glb.out_buffer_index = 0;
  this_device->received_message_size = 0;
}
/*=============================================================================
	FUNCTION: xnl_init_component()

    DESCRIPTION: Callback function the application should call only once at the
	very beginning to initialize all global variables and buffers.
=============================================================================*/
void xnl_init_component(void)
{
  //xnl_device = xnl_malloc(sizeof(xnl_device_struct_t*) * NUMBER_OF_DEVICES_SUPPORTED); //Only 2 supported currently HSP & SPP
  for(UINT8_T i=0; i<NUMBER_OF_DEVICES_SUPPORTED; i++)
  {
    xnl_device[i] = NULL;
  }
  number_of_devices_glb                             = 0;
  address_generator_glb                             = XNL_MIN_ADDRESS;
  unencrypted_auth_value_array_glb[0]               = 0x12345678;
  unencrypted_auth_value_array_glb[1]               = 0x98765432;

//  xnl_buffer_init();
  xnl_queue_init();
}
/*=============================================================================
	FUNCTION: xnl_xcmp_new_device_register()

    DESCRIPTION: Callback function used whenever a new device is connected.
	This function resets the XNL and XCMP states for the specified device.
=============================================================================*/
UINT16_T xnl_xcmp_new_device_register (xnl_init_parameters_t device_param)
{
//  xnl_device_struct_t *new_device = (xnl_device_struct_t *)xnl_malloc(sizeof(xnl_device_struct_t));
  // Adding a global space allocation instead of malloc as it is running out of memory on reconnection
  xnl_device_struct_t *new_device = (xnl_device_struct_t *)xnl_global_alloc(sizeof(xnl_device_struct_t));
  if(new_device)
  {
    xnl_initialize_structure(new_device);
    number_of_devices_glb++;
    UINT8_T index = 0xFF;
      
    if((number_of_devices_glb <= NUMBER_OF_DEVICES_SUPPORTED) && (xnl_get_unused_device_index(&index)))
    {
      // Assign an unused index 
      //index = number_of_devices_glb - 1;
      // Assign the pointer to the new device connection structure
      xnl_device[index] = new_device;
      // Assign the address to the new device connection
      xnl_set_address(index);
      // Copy the device features configured by the user
      xnl_device[index]->is_ack_feature_enabled = device_param.is_ack_feature_required;
      xnl_device[index]->auth_level = device_param.auth_level;

      // Return the newly created unique id. User will use this ID to refer to this device from now on
      return xnl_device[index]->address;	
    }	
    else
    {
      // XNL already supports the maximum connections
      xnl_device_report_failure(0, XNL_TOO_MANY_CONNECTIONS);
      return 0;
    }
  }
  // Should not reach here unless malloc fails
  xnl_device_report_failure(0, XNL_MEMORY_ALLOC_ERROR);
  return 0; 
}
/*=============================================================================
	FUNCTION: xnl_xcmp_data_parser()

    DESCRIPTION: Callback function called by the hardware interface, passes a
	pointer to the data and the size of how many bytes to read in. Must specify
	which device it came from when called.
=============================================================================*/
success_failure_t xnl_xcmp_data_parser(const UINT16_T xnl_devid, UINT8_T rx_byte) 
{
  success_failure_t return_value = FAILURE;  
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);

  // Converts the unique device id to its index to be used in array lookup
  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0)) 
  {
    xnl_device[index]->hw_message_size++;
    switch(xnl_device[index]->hw_state) 
    {
      case WAITING_FOR_XNL_OPCODE_HI: // MSB of opcode
        if (rx_byte == 0x00) 
        {
          if(xnl_copy_rxd_data_byte(&(xnl_device[index]->xnl_receive_buffer_glb), rx_byte))
          {
            xnl_device[index]->hw_state = WAITING_FOR_XNL_OPCODE_LO;
            xnl_start_parser_reset_timer(xnl_devid);
          }
        }
        else
        {
          //Invalid opcode byte, reset the message size
          xnl_device[index]->hw_message_size--;
        }
        break; 
      case WAITING_FOR_XNL_OPCODE_LO: // LSB of opcode
        if (xnl_is_valid_opcode(rx_byte)) 
        {
          if(xnl_copy_rxd_data_byte(&(xnl_device[index]->xnl_receive_buffer_glb), rx_byte))
          {
            xnl_device[index]->hw_state = WAITING_FOR_XNL_ADDRESSES;
          }
        }
        else 
        {
          xnl_cancel_parser_reset_timer(xnl_devid);         
          //Invalid opcode, reset the message size
          xnl_device[index]->hw_message_size = 0;          
          xnl_device[index]->hw_state = WAITING_FOR_XNL_OPCODE_HI;
        }
        break;
      case WAITING_FOR_XNL_ADDRESSES: // Id + flag + destination address + source address + transaction id = 8
        // We do not actually look at the values of the bytes here. The XNL stack does this later. We are
        // only interested in a complete message right now
        if(xnl_copy_rxd_data_byte(&(xnl_device[index]->xnl_receive_buffer_glb), rx_byte))
        {        
          xnl_device[index]->hw_payload_counter++;
          if (xnl_device[index]->hw_payload_counter == 8) 
          {
            xnl_device[index]->hw_payload_counter = 0;
            xnl_device[index]->hw_state = WAITING_FOR_XNL_PAYLOAD_SIZE_HI;
          }
        }
        break;				
      case WAITING_FOR_XNL_PAYLOAD_SIZE_HI: // MSB of payload size
        if(xnl_copy_rxd_data_byte(&(xnl_device[index]->xnl_receive_buffer_glb), rx_byte))
        {        
          xnl_device[index]->hw_payload_size = rx_byte;
          xnl_device[index]->hw_payload_size <<= 8;
          xnl_device[index]->hw_state = WAITING_FOR_XNL_PAYLOAD_SIZE_LO;
        }
        break;
      case WAITING_FOR_XNL_PAYLOAD_SIZE_LO: // LSB of payload size
        if(xnl_copy_rxd_data_byte(&(xnl_device[index]->xnl_receive_buffer_glb), rx_byte))
        {        
          xnl_device[index]->hw_payload_size |= rx_byte;
          xnl_device[index]->hw_payload_counter = xnl_device[index]->hw_payload_size;
          if (xnl_device[index]->hw_payload_counter == 0) // No payload, received a complete message
          {
            xnl_device[index]->hw_state = WAITING_FOR_XNL_OPCODE_HI;	
            xnl_device[index]->received_message_size = xnl_device[index]->hw_message_size;
            xnl_device[index]->hw_message_size = 0;
            return_value = SUCC;
          } 
          else  // There is a payload to wait for
          {
            if(xnl_device[index]->hw_payload_size >= (XNL_RX_BUFFER_SIZE - XNL_HEADER_SIZE))
            {
              xnl_device_report_failure(xnl_devid, XNL_RECEIVE_OVERFLOW_ERROR);
            }
            else
            {
              xnl_device[index]->hw_state = WAITING_FOR_XNL_COMPLETE_PAYLOAD;
            }
          }
        }
        break;
      case WAITING_FOR_XNL_COMPLETE_PAYLOAD: // Wait for the payload to completely arrive
        if(xnl_copy_rxd_data_byte(&(xnl_device[index]->xnl_receive_buffer_glb), rx_byte))
        {        
          xnl_device[index]->hw_payload_counter--;
          if (xnl_device[index]->hw_payload_counter == 0)
          {
            xnl_device[index]->hw_state = WAITING_FOR_XNL_OPCODE_HI;	
            xnl_device[index]->received_message_size = xnl_device[index]->hw_message_size;            
            xnl_device[index]->hw_message_size = 0;             
            return_value = SUCC;
          }				
        }
        break;
      default:
        break;
    }
    return return_value;    
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);    
    return FAILURE;
  }
}
/*=============================================================================
	FUNCTION: xnl_state_machine()

    DESCRIPTION: Highest level state machine.
=============================================================================*/
void xnl_state_machine(const UINT16_T xnl_devid)
{
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);
  
  // Converts the unique device id to its index to be used in array lookup
  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0)) 
  {  
    if (xnl_device[index]->is_xnl_authentication_complete)
    {
      xnl_state_machine_idle(index, xnl_device[index]->received_message_size);
    }		
    else // Authentication is not yet complete for this device
    {
      xnl_state_machine_init(index, xnl_device[index]->received_message_size);
    }		
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);    
  }
}
/*=============================================================================
	FUNCTION: xnl_state_machine_init()

	DESCRIPTION: State machine during authentication.
==============================================================================*/
void xnl_state_machine_init(const UINT8_T index, const UINT16_T bytes_to_check)
{
  switch (xnl_device[index]->state)
  {	
    case WAITING_FOR_DEVICE_KEY_REQUEST_MSG: 
      if (xnl_decode_message(XNL_DEVICE_AUTH_KEY_REQUEST_OPCODE_0004, index, bytes_to_check)) //0004
      {
        //Cancel the parsing timer as it is a valid message
        xnl_cancel_parser_reset_timer(xnl_device[index]->address);        
        xnl_send_device_auth_key_reply_msg(index); //0005
        xnl_device[index]->state = WAITING_FOR_DEVICE_CONN_REQUEST_MSG;
      } 
      break;
    case WAITING_FOR_DEVICE_CONN_REQUEST_MSG: 
      if (xnl_decode_message(XNL_DEVICE_CONN_REQUEST_OPCODE_0006, index, bytes_to_check)) //0006
      {
        //Cancel the parsing timer as it is a valid message
        xnl_cancel_parser_reset_timer(xnl_device[index]->address);        
        if(xnl_send_device_conn_reply_msg(index) == TRUE)
        {
          //XNL Authentication Complete
          xnl_device[index]->is_xnl_authentication_complete = TRUE;
          xnl_device[index]->state = XCMP_OVER_XNL_MESSAGING; 
          //Send the Sysmap broadcast listing the connected devices and their address
          xnl_send_device_sysmap_brdcst_msg(index); //0009 - Sysmap Broadcast
          xnl_init_complete(xnl_device[index]->address);
          //Send the XCMP Device Init message
          xcmp_msg_tx_device_init_status(xnl_device[index]->address);
          //Start a 1 second timer to receive XCMP Dev Init from the device
          xnl_start_xnl_xcmp_enum_timer(xnl_device[index]->address);
        }
        else  //Authentication or Hardware Failure
        {
          xnl_device[index]->state = WAITING_FOR_DEVICE_KEY_REQUEST_MSG;
        }
      }		
      break;
    default: 
      break;	
  }
}
/*=============================================================================
	FUNCTION: xnl_state_machine_idle()

	DESCRIPTION: State machine used after authentication.
==============================================================================*/
void xnl_state_machine_idle(const UINT8_T index, const UINT16_T bytes_to_check)
{
  success_failure_t decode_result;
 //xnl_device[index]->state == XCMP_OVER_XNL_MESSAGING 
  //Waiting for an ACK 
  if (xnl_device[index]->is_ack_feature_enabled && xnl_device[index]->is_ack_pending)
  {
    decode_result = xnl_decode_message(XNL_DATA_MSG_ACK_OPCODE_000C, index, bytes_to_check);
  }
  else  // Waiting for data msgs
  {
    decode_result = xnl_decode_message(XNL_DATA_MSG_OPCODE_000B, index, bytes_to_check);
  }
  
  if(decode_result == TRUE)
  {
    //Cancel the parsing timer as it is a valid message
    xnl_cancel_parser_reset_timer(xnl_device[index]->address);
  }
}
/*=============================================================================
	FUNCTION: xnl_decode_message()

	DESCRIPTION: Decodes data in the receive buffer for an XNL message. Assumes 
	a complete XNL message is in the receive buffer ready to be decoded.
=============================================================================*/
success_failure_t xnl_decode_message(const UINT16_T expected_opcode, const UINT8_T index, UINT16_T bytes_to_check)
{
    UINT16_T opcode = 0;
    UINT8_T  protocol_id = 0;
    UINT16_T destination_address = 0;
    UINT16_T source_address = 0;
    UINT16_T payload_length = 0;
    UINT8_T  receive_flag_counter = 0;
    UINT16_T transaction_id = 0;
    UINT8_T  device_type = 0;
    UINT8_T  auth_level = 0;
    UINT16_T bytes_read = 0;

    success_failure_t return_value = FAILURE;
    
    // Get XNL header info
    opcode = xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));	
    protocol_id = xnl_buffer_read_one_byte(&(xnl_device[index]->xnl_receive_buffer_glb));	
    receive_flag_counter = xnl_buffer_read_one_byte(&(xnl_device[index]->xnl_receive_buffer_glb));	
    destination_address = xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));
    source_address = xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));
    transaction_id = xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));	
    payload_length = xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));	
    bytes_read = XNL_HEADER_SIZE;

    switch(opcode)
    {
      case XNL_DEVICE_MASTER_QUERY_OPCODE_0003:
        if(protocol_id == XNL_PROTO_XNL_CTRL) 
        {
          // regardless of what "expected_opcode" is, must go back to sending MASTER STATUS BRDCST
          xnl_send_master_status_brdcst_msg(index); //0002
          xnl_device[index]->is_xnl_authentication_complete = FALSE;
          xnl_device[index]->state = WAITING_FOR_DEVICE_KEY_REQUEST_MSG;
          xnl_device[index]->rx_flag = XNL_RX_FLAG_RESET;
          xnl_device[index]->is_xcmp_authentication_complete = FALSE;
          received_xnl_reset(xnl_device[index]->address);
          // Return failure since we never actually expect a 0003 message and since it
          // also does not match the opcode from the parameter
          return_value = FAILURE; 
        } 
        break;
      case XNL_DEVICE_AUTH_KEY_REQUEST_OPCODE_0004:
        if((opcode == expected_opcode) && (protocol_id == XNL_PROTO_XNL_CTRL)) 
        {
          return_value = SUCC;
        } 
        break;    
      case XNL_DEVICE_CONN_REQUEST_OPCODE_0006:
        if((opcode == expected_opcode) && (protocol_id == XNL_PROTO_XNL_CTRL))
        {
          /* Check if Device Requested ACK disabled and if device hardware type allows it */
          if((xnl_device[index]->is_ack_feature_enabled == FALSE) && 
             (receive_flag_counter == XNL_FLAGS_ACK_NOT_REQUIRED))
          {   
            xnl_device[index]->is_ack_feature_enabled = FALSE; 
          }
          else
          {
            xnl_device[index]->is_ack_feature_enabled = TRUE;
          }          
          //preferred_address is not granted by the host implementation here so discarding the bytes
          xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));
          device_type = xnl_buffer_read_one_byte(&(xnl_device[index]->xnl_receive_buffer_glb));
          auth_level = xnl_buffer_read_one_byte(&(xnl_device[index]->xnl_receive_buffer_glb));
          xnl_device[index]->device_type = device_type;
          // Not supported for this product. Possibility to make this into a binary library call is TBD
          encrypted_auth_value_array_glb[0] = xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));
          encrypted_auth_value_array_glb[1] = xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));
          encrypted_auth_value_array_glb[2] = xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));
          encrypted_auth_value_array_glb[3] = xnl_buffer_read_two_bytes(&(xnl_device[index]->xnl_receive_buffer_glb));

          if (xnl_device[index]->auth_level != auth_level)
          {
            // Error: Not allowing third party right now
            xnl_device_report_failure(xnl_device[index]->address, XNL_INIT_ERROR);
          }	
#ifdef AUTHENTICATION_SUPPORTED
          // Check if authentication key matches to what we calculated  
          if(encrypted_auth_value_array_glb != calculated_encrypted_auth_value_array_glb)
            xnl_device_report_failure(xnl_device[index]->address, XNL_INIT_ERROR);  
#endif
          //Keep track of the bytes read
          bytes_read = bytes_read + XNL_DEVICE_CONN_REQUEST_PAYLOAD;
          return_value = SUCC;
        } 
        break;
        /* These message opcodes are bunched together as the device can send a data message before getting the message
          sent out from the XNL Host and thus that message needs to get processed instead of waiting for an ACK */
      case XNL_DATA_MSG_OPCODE_000B:
      case XNL_DATA_MSG_ACK_OPCODE_000C:
        /* Check if the message is a Data Message */        
        if(opcode == XNL_DATA_MSG_OPCODE_000B)
        {
          if (xnl_verify_source_address(index, source_address) && (protocol_id == XNL_PROTO_XCMP) &&
             (destination_address == XNL_MASTER_ADDRESS) && (xnl_device[index]->rx_flag != receive_flag_counter))
          {
            // Message intended for master host so pass it to the XCMP stack
            xnl_device[index]->rx_flag = receive_flag_counter;
            xnl_device[index]->transaction_id = transaction_id; 
            if (xnl_device[index]->is_ack_feature_enabled) 
            {
               xnl_send_data_msg_ack(index);
            }					
            if (xnl_device[index]->is_xcmp_authentication_complete)	 
            {
               xcmp_process_message(xnl_device[index]->address, &(xnl_device[index]->xnl_receive_buffer_glb), payload_length);						
            } 
            else
            {
               xcmp_process_init(index, xnl_device[index]->address, &(xnl_device[index]->xnl_receive_buffer_glb), payload_length);					
            }
            //Discard the complete message
            xnl_buffer_discard_bytes(&(xnl_device[index]->xnl_receive_buffer_glb), payload_length);    
            bytes_read = bytes_read + payload_length;
            return_value = SUCC;  	
          }
          else if (xnl_verify_source_address(index, source_address) && (protocol_id == XNL_PROTO_XCMP) &&
                  (xnl_verify_destination_address(destination_address)) && (xnl_device[index]->rx_flag != receive_flag_counter))
          {
            relay_source = source_address;
            xnl_device[index]->rx_flag = receive_flag_counter;
            xnl_device[index]->transaction_id = transaction_id; 
            if (xnl_device[index]->is_ack_feature_enabled) 
            {
               xnl_send_data_msg_ack(index);
            }		
            // Message intended for another connected device, so repackage and
            // relay to the appropriate connected device		
            xnl_send_relayed_data_msg(destination_address, payload_length);
            return_value = SUCC; 
          }
          else if ((xnl_verify_source_address(index, source_address)) && (protocol_id == XNL_PROTO_XCMP) && 
                   (xnl_device[index]->rx_flag == receive_flag_counter) &&
                   ((destination_address == XNL_MASTER_ADDRESS) || (xnl_verify_destination_address(destination_address))))
          {
            xnl_device[index]->rx_flag = receive_flag_counter;
            xnl_device[index]->transaction_id = transaction_id; 
            if (xnl_device[index]->is_ack_feature_enabled) 
            {
               xnl_send_data_msg_ack(index);
            }		
            return_value = SUCC; 
          }
        }
        else
        {
          /* Check if the message is Data message ACK */
          if(opcode == XNL_DATA_MSG_ACK_OPCODE_000C)
          {       
            if ((source_address == xnl_device[index]->address) && (protocol_id == XNL_PROTO_XCMP) &&
                 (xnl_is_valid_ack_flag(index, receive_flag_counter)) && (payload_length == XNL_NO_PAYLOAD) &&
                 ((destination_address == XNL_MASTER_ADDRESS) || (xnl_verify_destination_address(destination_address))))
            {
              if (xnl_device[index]->is_ack_feature_enabled && xnl_device[index]->is_ack_pending)
              {
                if((xnl_is_valid_transaction_id(index, transaction_id)) || 
                   (xnl_device[index]->transaction_id == transaction_id))
                {
                  // Cancel this timer so we don't send out a retry message
                  xnl_timer_cancel_data_message_ack(xnl_device[index]->address);
                  // Reset the ACK pending flag
                  xnl_device[index]->is_ack_pending = FALSE;
                  xnl_device[index]->retry_counter = 0;
                  xnl_device[index]->retry_message_size = 0;
                  // Inform the application that an expected ACK was received
                  xnl_ack_received(xnl_device[index]->address);
                }
              }
              return_value = SUCC;      																	
            }		
          }          
        }
        break;
    case XNL_DEVICE_SYSMAP_REQUEST_OPCODE_0008:
      if (xnl_verify_source_address(index, source_address) && (protocol_id == XNL_PROTO_XNL_CTRL) &&
         (destination_address == XNL_MASTER_ADDRESS))
      {      
        xnl_send_device_sysmap_brdcst_msg(index); //0009 - Sysmap Broadcast
      }
      break;
  }
  // Safety check to discard if any data bytes left
  if(bytes_read < bytes_to_check) 
  {
    xnl_buffer_discard_bytes(&(xnl_device[index]->xnl_receive_buffer_glb), (bytes_to_check - bytes_read));
  } 
  return return_value;
}
/*=============================================================================
	FUNCTION: xnl_verify_source_address()

	DESCRIPTION: Checks whether the parameter is a valid source address.
=============================================================================*/
success_failure_t xnl_verify_source_address (const UINT8_T index, const UINT16_T address_to_check)
{
  //0x0000 is a broadcast address, so return success in case of correct addr or broadcast addr
  if(address_to_check == xnl_device[index]->address || address_to_check == 0x0000)
  {
    return SUCC;
  }
  else
  {
    return FAILURE;
  }
}
/*=============================================================================
	FUNCTION: xnl_verify_destination_address()

	DESCRIPTION: Checks whether the parameter is a valid destination address.
=============================================================================*/
success_failure_t xnl_verify_destination_address (const UINT16_T address_to_check)
{
  for (UINT8_T i = 0; i < NUMBER_OF_DEVICES_SUPPORTED; i++)
  {
    if (xnl_device[i]->is_xnl_authentication_complete && address_to_check == xnl_device[i]->address)
    {
      return SUCC;
    }
  }	
  return FAILURE;
}
/*=============================================================================
	FUNCTION: xnl_calculate_transaction_id()

	DESCRIPTION: Calculates the next transaction id for a given device
=============================================================================*/
UINT16_T xnl_calculate_transaction_id (const UINT8_T index)
{
  UINT16_T transaction_id = (((xnl_device[index]->transaction_id_base & 0x00FFu ) << 8) | (xnl_device[index]->transaction_id_counter));
  (xnl_device[index]->transaction_id_counter)++;
  return transaction_id;
}
/*=============================================================================
	FUNCTION: xnl_master_status_brdcst_msg()

	DESCRIPTION: Sends out a message that the master device is present
=============================================================================*/
success_failure_t xnl_send_master_status_brdcst_msg(const UINT8_T index)
{
  UINT8_T payload_array[XNL_MASTER_STATUS_BRDCST_PAYLOAD] = {0, 0, 0, 0, 0 , 0 , 0};

  xnl_header_t master_status_brdcst_header;

  master_status_brdcst_header.opcode                  = XNL_MASTER_STATUS_BRDCST_OPCODE_0002;
  master_status_brdcst_header.protocol_id             = XNL_PROTO_XNL_CTRL;    
  master_status_brdcst_header.xnl_flags               = XNL_FLAGS_UNUSED;  
  master_status_brdcst_header.destination_address     = XNL_ADDRESS_NOT_ASSIGNED;     
  master_status_brdcst_header.source_address          = XNL_MASTER_ADDRESS;      
  master_status_brdcst_header.transaction_id          = XNL_TRANS_ID_NOT_REQUIRED;       
  master_status_brdcst_header.payload_length          = XNL_MASTER_STATUS_BRDCST_PAYLOAD;

  payload_array[0]  = (UINT8_T)((XNL_PROTOCOL_VERSION_NUMBER & 0xFF000000u) >> 24);
  payload_array[1]  = (UINT8_T)((XNL_PROTOCOL_VERSION_NUMBER & 0x00FF0000u) >> 16);
  payload_array[2]  = (UINT8_T)((XNL_PROTOCOL_VERSION_NUMBER & 0x0000FF00u) >> 8);
  payload_array[3]  = (UINT8_T)(XNL_PROTOCOL_VERSION_NUMBER & 0x000000FFu);
  payload_array[4]  = ((XNL_MASTER_LOGICAL_IDENTIFIER & 0xFF00u) >> 8);
  payload_array[5]  = (XNL_MASTER_LOGICAL_IDENTIFIER & 0x00FFu);
  payload_array[6]  = XNL_DATA_MSG_SENT;
  
  return xnl_send_data_to_hardware(index, (UINT8_T*)&master_status_brdcst_header, XNL_HEADER_SIZE, 
                                         (UINT8_T*)&payload_array, XNL_MASTER_STATUS_BRDCST_PAYLOAD);
}
/*=============================================================================
	FUNCTION: xnl_send_device_auth_key_reply_msg()

	DESCRIPTION: Creates the device authentication key reply.
=============================================================================*/
success_failure_t xnl_send_device_auth_key_reply_msg(const UINT8_T index)
{
  UINT8_T payload_array [XNL_DEVICE_AUTH_KEY_REPLY_PAYLOAD] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  xnl_header_t device_auth_key_reply_header;

  device_auth_key_reply_header.opcode               = XNL_DEVICE_AUTH_KEY_REPLY_OPCODE_0005; 
  device_auth_key_reply_header.protocol_id          = XNL_PROTO_XNL_CTRL;   
  device_auth_key_reply_header.xnl_flags            = XNL_FLAGS_UNUSED;     
  device_auth_key_reply_header.destination_address  = XNL_ADDRESS_NOT_ASSIGNED;       
  device_auth_key_reply_header.source_address       = XNL_MASTER_ADDRESS;         
  device_auth_key_reply_header.transaction_id       = XNL_TRANS_ID_NOT_REQUIRED;          
  device_auth_key_reply_header.payload_length       = XNL_DEVICE_AUTH_KEY_REPLY_PAYLOAD;      

  payload_array[0] = ((XNL_TEMP_ADDRESS & 0xFF00u) >> 8);
  payload_array[1] = (XNL_TEMP_ADDRESS & 0x00FFu);

  payload_array[2] = (UINT8_T)((unencrypted_auth_value_array_glb[0] & 0xFF000000u) >> 24);
  payload_array[3] = (UINT8_T)((unencrypted_auth_value_array_glb[0] & 0x00FF0000u) >> 16);
  payload_array[4] = (UINT8_T)((unencrypted_auth_value_array_glb[0] & 0x0000FF00u) >> 8);
  payload_array[5] = (UINT8_T)(unencrypted_auth_value_array_glb[0] & 0x000000FFu);
  payload_array[6] = (UINT8_T)((unencrypted_auth_value_array_glb[1] & 0xFF000000u) >> 24);
  payload_array[7] = (UINT8_T)((unencrypted_auth_value_array_glb[1] & 0x00FF0000u) >> 16);
  payload_array[8] = (UINT8_T)((unencrypted_auth_value_array_glb[1] & 0x0000FF00u) >> 8);
  payload_array[9] = (UINT8_T)(unencrypted_auth_value_array_glb[1] & 0x000000FFu);

  return xnl_send_data_to_hardware(index, (UINT8_T*)&device_auth_key_reply_header, XNL_HEADER_SIZE, 
                                         (UINT8_T*)&payload_array, XNL_DEVICE_AUTH_KEY_REPLY_PAYLOAD);
}
/*=============================================================================
	FUNCTION: xnl_send_device_conn_reply_msg()

	DESCRIPTION: Creates the device connection reply.
=============================================================================*/
success_failure_t xnl_send_device_conn_reply_msg(const UINT8_T index)
{
  UINT8_T payload_array [XNL_DEVICE_CONN_REPLY_PAYLOAD] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
  xnl_header_t device_conn_reply_header;

  device_conn_reply_header.opcode               = XNL_DEVICE_CONN_REPLY_OPCODE_0007;     
  device_conn_reply_header.protocol_id          = XNL_PROTO_XNL_CTRL;   
  device_conn_reply_header.source_address       = XNL_MASTER_ADDRESS;       
  device_conn_reply_header.destination_address  = XNL_TEMP_ADDRESS;
  device_conn_reply_header.transaction_id       = XNL_TRANS_ID_NOT_REQUIRED;     
  device_conn_reply_header.payload_length       = XNL_DEVICE_CONN_REPLY_PAYLOAD;
  //ACK feature reply
  if(xnl_device[index]->is_ack_feature_enabled)
  {
    device_conn_reply_header.xnl_flags = XNL_ACK_ENABLE;
  }
  else
  {
    device_conn_reply_header.xnl_flags = XNL_ACK_DISABLE;
  }
  xnl_device[index]->transaction_id_base = xnl_device[index]->address;
  //Check if the authentication key matches to what we calculated
  /* In regards to a discussion, we are not going to implement the authentication key
    for the Gateway. The Gateway is supposed to accept it in any case */
#ifdef AUTHENTICATION_SUPPORTED
  // Check if authentication key matches to what we calculated  
  if(encrypted_auth_value_array_glb != calculated_encrypted_auth_value_array_glb)  
#else
  //Always accept the connection as we are not checking the authentication key  
  if(TRUE)
#endif    
  {
    payload_array[0] = XNL_SUCCESS; 
    payload_array[1] = xnl_device[index]->transaction_id_base;
    payload_array[2] = ((xnl_device[index]->address & 0xFF00u) >> 8);
    payload_array[3] = (xnl_device[index]->address & 0x00FFu);
    payload_array[4] = (xnl_device[index]->device_type);
    payload_array[5] = (index + 1); //logical device ID
    payload_array[6] = (UINT8_T)((encrypted_auth_value_array_glb[0] & 0xFF00u ) >> 8);
    payload_array[7] = (UINT8_T)(encrypted_auth_value_array_glb[0] & 0x00FFu );
    payload_array[8] = (UINT8_T)((encrypted_auth_value_array_glb[1] & 0xFF00u ) >> 8);
    payload_array[9] = (UINT8_T)(encrypted_auth_value_array_glb[1] & 0x00FFu );
    payload_array[10] = (UINT8_T)((encrypted_auth_value_array_glb[2] & 0xFF00u ) >> 8);
    payload_array[11] = (UINT8_T)(encrypted_auth_value_array_glb[2] & 0x00FFu );
    payload_array[12] = (UINT8_T)((encrypted_auth_value_array_glb[3] & 0xFF00u ) >> 8);
    payload_array[13] = (UINT8_T)(encrypted_auth_value_array_glb[3] & 0x00FFu );
  
    return xnl_send_data_to_hardware(index, (UINT8_T*)&device_conn_reply_header, XNL_HEADER_SIZE, 
                                           (UINT8_T*)&payload_array, XNL_DEVICE_CONN_REPLY_PAYLOAD);    
  }
  else
  {
    // Send all 0's in the payload as the authentication failed
    xnl_send_data_to_hardware(index, (UINT8_T*)&device_conn_reply_header, XNL_HEADER_SIZE, 
                                    (UINT8_T*)&payload_array, XNL_DEVICE_CONN_REPLY_PAYLOAD);
    return FALSE;
  }
}
/*=============================================================================
	FUNCTION: xnl_send_device_sysmap_brdcst_msg()

	DESCRIPTION: Sends out a message containing all the address for all connected
	devices and their type.
=============================================================================*/
success_failure_t xnl_send_device_sysmap_brdcst_msg(const UINT8_T index)
{
  xnl_header_t xnl_device_sysmap_brdcst_header;
  
  // Get the total number of devices authenticated
  UINT8_T number_of_authenticated_devices = 0;
  for (UINT8_T i = 0; i < NUMBER_OF_DEVICES_SUPPORTED; i++)
  {
    if (xnl_device[i]->is_xnl_authentication_complete)
            number_of_authenticated_devices++;
  }

  xnl_device_sysmap_brdcst_header.opcode              = XNL_DEVICE_SYSMAP_BRDCST_OPCODE_0009;   
  xnl_device_sysmap_brdcst_header.protocol_id         = XNL_PROTO_XNL_CTRL;    
  xnl_device_sysmap_brdcst_header.xnl_flags           = XNL_FLAGS_UNUSED;         
  xnl_device_sysmap_brdcst_header.source_address      = XNL_MASTER_ADDRESS; 
  xnl_device_sysmap_brdcst_header.transaction_id      = XNL_TRANS_ID_NOT_REQUIRED;       
  xnl_device_sysmap_brdcst_header.payload_length      = XNL_DEVICE_SYSMAP_BRDCST_PAYLOAD + (5 * number_of_authenticated_devices);
  xnl_device_sysmap_brdcst_header.destination_address = xnl_device[index]->address;
  
  UINT8_T payload_array[22];
  UINT8_T offset;

  payload_array[0]   = ((number_of_authenticated_devices & 0xFF00) >> 8);
  payload_array[1]   = (number_of_authenticated_devices & 0x00FF);	
  
  offset = 2; // Figure out all of the connected devices and add their info into the payload
  for (UINT8_T i = 0; i < number_of_authenticated_devices; i++)
  {
    if (xnl_device[i]->is_xnl_authentication_complete)
    {
      payload_array[offset++] = xnl_device[i]->device_type;
      payload_array[offset++] = (i + 1);  //logical device id
      payload_array[offset++] = (xnl_device[i]->address & 0xFF00) >> 8;
      payload_array[offset++] = xnl_device[i]->address & 0x00FF;
      payload_array[offset++] = xnl_device[i]->auth_level;		
    }
  }		
  return xnl_send_data_to_hardware(index, (UINT8_T*)&xnl_device_sysmap_brdcst_header, XNL_HEADER_SIZE, 
                                         (UINT8_T*)&payload_array, xnl_device_sysmap_brdcst_header.payload_length);
}
/*=============================================================================
	FUNCTION: xnl_send_data_msg_ack()

	DESCRIPTION: Handshake between host and device to confirm that a data
	message from the host has been received.
=============================================================================*/
success_failure_t xnl_send_data_msg_ack(const UINT8_T index)
{
  if(xnl_device[index]->is_xnl_authentication_complete)
  {
    xnl_header_t data_msg_ack_header;
    
    data_msg_ack_header.opcode	            = XNL_DATA_MSG_ACK_OPCODE_000C;		
    data_msg_ack_header.protocol_id         = XNL_PROTO_XCMP;		
    data_msg_ack_header.source_address      = XNL_MASTER_ADDRESS;   
    data_msg_ack_header.payload_length      = XNL_NO_PAYLOAD;
    data_msg_ack_header.xnl_flags           = xnl_device[index]->rx_flag;
    data_msg_ack_header.transaction_id      = xnl_device[index]->transaction_id;
    data_msg_ack_header.destination_address = xnl_device[index]->address;
  
    /* Use a different interface as the ACK is sent out from within the task and 
       should use its own buffer */
    return xnl_send_data_ack_to_hardware(index, (UINT8_T*)&data_msg_ack_header);
  }
  else
  {
    return FAILURE;
  }
}
/*=============================================================================
	FUNCTION: xnl_send_data_msg()

	DESCRIPTION: Sends a data message to the device. The payload of this message
	contains the XCMP message to host will send to the device.
=============================================================================*/
success_failure_t xnl_send_data_msg(const UINT16_T xnl_devid, const UINT8_T * payload_data_ptr, const UINT16_T payload_length, UINT8_T msg_type)
{
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);
  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0)) 
  {
    // Check if the connected device has finished XNL Authentication
    if(xnl_device[index]->is_xnl_authentication_complete)
    {
      xnl_header_t xnl_data_msg_header;
      if (xnl_device[index]->is_ack_feature_enabled) 
      {
        if (xnl_device[index]->is_ack_pending)
        {
          return FAILURE;
        }
      }
      //Opcode and Protocol
      xnl_data_msg_header.opcode	    = XNL_DATA_MSG_OPCODE_000B;		
      xnl_data_msg_header.protocol_id 	    = XNL_PROTO_XCMP;				
      //Flag   
      if (xnl_device[index]->tx_flag > XNL_FLAG_MAX_VALUE)
      {
        xnl_device[index]->tx_flag = XNL_FLAG_MIN_VALUE; // Modular since flag ranges from 0 - 7   
      }
  
      // For Unsupported Opcode msgs, use the XNL flag value of the msg sent by host.
      if (XNL_UNSUPPORTED_REQ == msg_type)
      {
        xnl_data_msg_header.xnl_flags           = xnl_device[index]->rx_flag;
      }
      else
      {
        xnl_data_msg_header.xnl_flags           = (xnl_device[index]->tx_flag)++; 	
      }
      // Source and Destination Address
      if(msg_type == XNL_DATA_MSG_RELAY)
      {
        xnl_data_msg_header.source_address      = relay_source;
      }
      else
      {
        xnl_data_msg_header.source_address      = XNL_MASTER_ADDRESS;	
      }
      xnl_data_msg_header.destination_address   = xnl_device[index]->address;
      // Transaction ID
      if((msg_type == XNL_DATA_MSG_BRDCST) || (msg_type == XNL_DATA_MSG_REQUEST) || (msg_type == XNL_DATA_MSG_RELAY))
      {
        xnl_data_msg_header.transaction_id      = xnl_calculate_transaction_id(index);
      }
      else  //Reply and Unsupported Req
      {
        xnl_data_msg_header.transaction_id      = xnl_device[index]->transaction_id;
      }
      //Payload Size
      xnl_data_msg_header.payload_length	      = payload_length;	
  
      if( xnl_send_data_to_hardware(index, (UINT8_T*)&xnl_data_msg_header, XNL_HEADER_SIZE, payload_data_ptr, payload_length) == SUCC )
      {
        if (xnl_device[index]->is_ack_feature_enabled) // Only if the ACK feature is enabled, we care to store the retry message
        {
          if (!xnl_device[index]->is_ack_pending)
          {
            xnl_device[index]->is_ack_pending = TRUE;
            xnl_buffer_store_retry_message(index, (UINT8_T*)&xnl_data_msg_header, XNL_HEADER_SIZE, payload_data_ptr, payload_length);
            xnl_timer_start_data_message_ack(xnl_devid);
          }
        }
        return SUCC;
      }
      else
      {
        return FAILURE;
      }
    }
    else
    {
      return FAILURE;
    }
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);
    return FAILURE;
  }
} 
/*=============================================================================
	FUNCTION: xnl_send_relayed_data_msg()

	DESCRIPTION: Sends a data message from one device to another device connected
	to the host.
=============================================================================*/
success_failure_t xnl_send_relayed_data_msg(const UINT16_T destination_address, const UINT16_T payload_length)
{
  UINT8_T return_value = FAILURE;
  UINT16_T index;
  
  // Get id index by matching its address
  for (index = 0; index < NUMBER_OF_DEVICES_SUPPORTED; index++)
  {
    if (destination_address == xnl_device[index]->address) break;
  }	
  xnl_tx_clr_msg_relay_buffer(index);

  for (UINT8_T i = 0; i < payload_length; i++) // Transfer the payload
  {
    xnl_device[index]->xnl_relay_array_glb[i] = xnl_read_data_byte(&(xnl_device[index]->xnl_receive_buffer_glb));
  }
  // Send out the message to the hardware interface
  return_value = xnl_send_data_msg(index, &(xnl_device[index]->xnl_relay_array_glb[0]), payload_length, XNL_DATA_MSG_RELAY);

  return return_value;
}

/*=============================================================================
	FUNCTION: xnl_buffer_store_retry_message()

	DESCRIPTION: Stores the message to send out later during a retry.
=============================================================================*/
void xnl_buffer_store_retry_message(const UINT8_T index, const UINT8_T* header_ptr, const UINT8_T header_size, 
									const UINT8_T* payload_ptr, const UINT16_T payload_size) 
{
  UINT16_T i = 0; 
  xnl_device[index]->retry_buffer_ptr = (UINT8_T *)&(xnl_device[index]->retry_buffer_data[0]);

  if (xnl_device[index]->retry_buffer_ptr != NULL)
  {
    if(header_ptr != NULL) // Will lose previous retry message since it will overwrite
    {
      for(i = 0; i < header_size; i++) 
      {
        xnl_device[index]->retry_buffer_ptr[i] = *header_ptr++;
      }	
    
      if(payload_ptr != NULL)
      {
        for(i = header_size; i < header_size + payload_size; i++)
        {
          xnl_device[index]->retry_buffer_ptr[i] = *payload_ptr++;
        }			
      }
      // Since a write to the retry buffer overwrites the previous stored retry message,
      // reset the counter value to 1 as message is sent once and re-specify the retry message size
      xnl_device[index]->retry_counter = 1;
      xnl_device[index]->retry_message_size = header_size + payload_size;
    }
    else
    {
      // Header pointer is NULL
      xnl_device_report_failure(xnl_device[index]->address, XNL_INVALID_PARAMETER);
    }
  }
  else // Notify application there isn't enough memory to allocate the message
  {
    xnl_device_report_failure(xnl_device[index]->address, XNL_MEMORY_ALLOC_ERROR);
  }
}
/*=============================================================================
	FUNCTION: xnl_is_valid_opcode()

    DESCRIPTION: Checks whether the parameter is a valid XNL opcode.
=============================================================================*/
success_failure_t xnl_is_valid_opcode(const UINT8_T opcode)
{
  switch(opcode) 
  {
    case XNL_DEVICE_MASTER_QUERY_OPCODE_0003:
    case XNL_DEVICE_AUTH_KEY_REQUEST_OPCODE_0004:
    case XNL_DEVICE_CONN_REQUEST_OPCODE_0006:
    case XNL_DATA_MSG_OPCODE_000B:
    case XNL_DATA_MSG_ACK_OPCODE_000C:
            return SUCC;
    default:
            return FAILURE;
  }
}
/*=============================================================================
	FUNCTION: xnl_is_valid_transaction_id()

	DESCRIPTION: Checks whether a given transaction is its expected value
=============================================================================*/
success_failure_t xnl_is_valid_transaction_id (const UINT8_T index, const UINT16_T transaction_id_to_check)
{
  UINT8_T previous_transaction_id_counter = xnl_device[index]->transaction_id_counter - 1;
  if((((xnl_device[index]->transaction_id_base & 0x00FFu) << 8) | previous_transaction_id_counter) == transaction_id_to_check)
  {
    return SUCC;
  }
  else
  {
    return FAILURE;
  }
}
/*=============================================================================
	FUNCTION: xnl_is_valid_ack_flag()

    DESCRIPTION: Checks whether the parameter is a valid XNL flag value.
=============================================================================*/
success_failure_t xnl_is_valid_ack_flag (const UINT8_T index, const UINT8_T flag_to_check)
{
  // Since the transmit flag gets incremented by one every time we send a message
  // we check one less than the current transmit flag value
  if((flag_to_check == xnl_device[index]->tx_flag - 1) ||
     (flag_to_check == xnl_device[index]->rx_flag))
  {
    return SUCC;
  }
  else
  {
    return FAILURE;
  }
}
/*=============================================================================
    FUNCTION: xnl_device_tx_status()

    DESCRIPTION: Returns SUCC or FAILURE depending on if the transmit path
	is busy or not.
=============================================================================*/
tx_result_t xnl_device_tx_status (const UINT16_T xnl_devid)
{
  INT8_T index = xnl_get_device_index_from_id(xnl_devid);
  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0)) 
  {
    if(xnl_device[index]->is_ack_pending)
    {
      return TX_BUSY;
    }
    else
    {
      return TX_SUCCESS;
    }
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);
    return TX_FAILURE;
  }
}
/*=============================================================================
	FUNCTION: xnl_timer_expired()

    DESCRIPTION: Handles all XNL related timer expirations.
=============================================================================*/
void xnl_timer_expired (const UINT16_T xnl_devid, UINT8_T msg_id)
{
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);
  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0)) 
  {
    switch (msg_id)
    {
      case HSP_MASTER_STATUS_BCAST_TIMER_EXPIRED:
      case SPP_MASTER_STATUS_BCAST_TIMER_EXPIRED:        
        xnl_send_master_status_brdcst_msg(index);
        xnl_device[index]->state = WAITING_FOR_DEVICE_KEY_REQUEST_MSG;
        break;
      case HSP_DATA_MSG_ACK_TIMER_EXPIRED:
      case SPP_DATA_MSG_ACK_TIMER_EXPIRED:        
        // If timer expired waiting for an ACK
        if (xnl_device[index]->is_ack_pending) 
        {
          if (xnl_device[index]->retry_counter < 5)
          {
            xnl_send_retry_message(index);
            xnl_timer_start_data_message_ack(xnl_devid);
            (xnl_device[index]->retry_counter)++;
          } 
          else // More than 5 retries were used for one message the host sent out...
          {
            // Notify application that there is a connection error since the host did not
            // get an acknowledgment for 5 retry messages
            xnl_device_report_failure(xnl_devid, XNL_CONNECTION_ERROR);
          }	
        }
        break;
      case HSP_PARSER_SM_RESET_TIMER_EXPIRED:
      case SPP_PARSER_SM_RESET_TIMER_EXPIRED:
        //Reset the state machine variable and related variables
        xnl_device[index]->hw_state = WAITING_FOR_XNL_OPCODE_HI;
        xnl_device[index]->hw_message_size = 0;  
        xnl_device[index]->hw_payload_counter = 0;
        xnl_device[index]->hw_payload_size = 0;
        xnl_device[index]->received_message_size = 0;
        //Drop the bytes
        xnl_device[index]->xnl_receive_buffer_glb.out_buffer_index = xnl_device[index]->xnl_receive_buffer_glb.in_buffer_index;        
        break;
      // The 1 second timer expired waiting to receive the XCMP Dev Init message from the device
      case HSP_ENUM_TIMER_EXPIRED:
      case SPP_ENUM_TIMER_EXPIRED:
        xnl_device[index]->is_xnl_authentication_complete = FALSE;
        xnl_device[index]->state = WAITING_FOR_DEVICE_KEY_REQUEST_MSG;
        xnl_device[index]->rx_flag = XNL_RX_FLAG_RESET;
        xnl_device[index]->is_xcmp_authentication_complete = FALSE;    
        // Notify application that there is a xnl initialization error since the device did not respond with xcmp dev init msg
        xnl_device_report_failure(xnl_devid, XNL_INIT_ERROR);          
        break;

      default:
        break;
    }
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);
  }
}
/*=============================================================================
	FUNCTION: xnl_set_device_ack_feature()

    DESCRIPTION: Enable/Disable the XNL ACK feature.
=============================================================================*/
void xnl_set_device_ack_feature(const UINT16_T xnl_devid, BOOL_T flag)
{
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);
  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0)) 
  {
    xnl_device[index]->is_ack_feature_enabled = flag;
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);
  }
}
/*=============================================================================
	FUNCTION: xnl_get_device_ack_feature()

    DESCRIPTION: Check if the XNL ACK feature is enabled or disabled
=============================================================================*/
BOOL_T xnl_get_device_ack_feature(const UINT16_T xnl_devid)
{
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);
  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0)) 
  {
    return xnl_device[index]->is_ack_feature_enabled;
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);
    return FALSE;
  }
}
/*=============================================================================
	FUNCTION: xnl_get_device_index_from_id()

    DESCRIPTION: Obtains the index to use for a given device id
=============================================================================*/
UINT8_T xnl_get_device_index_from_id (const UINT16_T id_to_check)
{
  for (UINT8_T i = 0; i < NUMBER_OF_DEVICES_SUPPORTED; i++)
  {
    if (xnl_device[i]->address == id_to_check)
            return i;
  }
  return NUMBER_OF_DEVICES_SUPPORTED+1; // Error, no address matches the parameter...
}
/*=============================================================================
	FUNCTION: xnl_get_unused_device_index()

    DESCRIPTION: Obtains an available index to use to assign a new device. Returns
	true or false depending on if it was able to find an unused device index.
=============================================================================*/
BOOL_T xnl_get_unused_device_index (UINT8_T *index)
{
  for (UINT8_T i = 0; i < NUMBER_OF_DEVICES_SUPPORTED; i++)
  {
    if (xnl_device[i] == NULL)
    {
      *index = i;
      return TRUE;
    }
  }
  return FALSE; 
}
/*=============================================================================
	FUNCTION: xnl_check_unused_address()

    DESCRIPTION: Checks if the assigned address does not match to an address
                already assigned to a connected device.
=============================================================================*/
success_failure_t xnl_check_unused_address(UINT16_T addr_generated)
{
  for (UINT8_T i = 0; i < NUMBER_OF_DEVICES_SUPPORTED; i++)
  {
    if (xnl_device[i]->address == addr_generated)
    {
      return TRUE;
    }
  }
  return FALSE;   
}
/*=============================================================================
	FUNCTION: xnl_set_address()

    DESCRIPTION: Gets the address for the new connected device.
=============================================================================*/
void xnl_set_address(UINT8_T index)
{
  // Check if the same address is not used by other connected device
  while (xnl_check_unused_address(address_generator_glb))
  {
    address_generator_glb++;
    if (address_generator_glb >= XNL_MAX_ADDRESS) 
    {
      address_generator_glb = XNL_MIN_ADDRESS;
    }
  }
  xnl_device[index]->address = address_generator_glb++;
  if (address_generator_glb >= XNL_MAX_ADDRESS) 
  {
    address_generator_glb = XNL_MIN_ADDRESS;
  }
}
/*=============================================================================
	FUNCTION: xnl_xcmp_authentication_complete()

    DESCRIPTION: Sets a boolean to true indicating that authentication
	has completed for a device
=============================================================================*/
void xnl_xcmp_authentication_complete (const UINT16_T xnl_devid)
{
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);
  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0)) 
  {
    xnl_device[index]->is_xcmp_authentication_complete = TRUE;
    xnl_xcmp_init_complete(xnl_devid);
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);		
  }
}
/*=============================================================================
	FUNCTION: xnl_reset_device_structure()

	DESCRIPTION: Resets the variables of a device structure
=============================================================================*/
void xnl_reset_device_structure(xnl_device_struct_t *reset_device)
{
  reset_device->hw_state = WAITING_FOR_XNL_OPCODE_HI;  
  reset_device->state = WAITING_FOR_DEVICE_KEY_REQUEST_MSG;  
  reset_device->is_xnl_authentication_complete = FALSE;
  reset_device->is_xcmp_authentication_complete = FALSE;  
  reset_device->transaction_id_base = 0;
  reset_device->transaction_id_counter = 0;
  reset_device->transaction_id = 0;
  reset_device->address = 0;
  reset_device->device_type = 0;
  reset_device->rx_flag = XNL_RX_FLAG_RESET;
  reset_device->tx_flag = 0;
  reset_device->auth_level = 0;
  reset_device->retry_message_size = 0;
  reset_device->retry_counter = 0;
  reset_device->is_ack_feature_enabled = FALSE;
  reset_device->is_ack_pending = FALSE;
  reset_device->hw_message_size = 0;
  reset_device->hw_payload_counter = 0;
  reset_device->hw_payload_size = 0;
  reset_device->retry_buffer_ptr = NULL;
  reset_device->received_message_size = 0;
}
/*=============================================================================
	FUNCTION: xnl_reset_retry_params()

	DESCRIPTION: Resets the variables for XNL Retry mechanism
=============================================================================*/
void xnl_reset_retry_params(UINT16_T xnl_devid)
{
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);
  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0)) 
  {
    xnl_device[index]->is_ack_pending = FALSE;
    xnl_device[index]->retry_buffer_ptr = NULL; 
    xnl_device[index]->retry_message_size = 0;
    xnl_device[index]->retry_counter = 0;
  } 
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);
  }  
}
/*=============================================================================
	FUNCTION: xnl_device_disconnected()

    DESCRIPTION: Takes care of required overhead whenever a host disconnects a
	device.
=============================================================================*/
void xnl_device_disconnected (UINT16_T xnl_devid)
{
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);

  if((index < NUMBER_OF_DEVICES_SUPPORTED) && (number_of_devices_glb != 0))  
  {	
    number_of_devices_glb--;
    if (xnl_device[index]->state < WAITING_FOR_DEVICE_CONN_REQUEST_MSG)
    {
      xnl_timer_cancel_send_master_status(xnl_devid);
    }	
    // Reset the structure to reuse for new connection
    xnl_reset_device_structure(xnl_device[index]);
    // Free the allocated memory
    xnl_global_alloc_free(xnl_device[index]); //Use xnl_free in case of malloc
    xnl_device[index] = NULL;
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);
  }
}
