
/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2011,2013-2014 MOTOROLA SOLUTIONS,INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: xnl_config.c
#
# --------------------------- General Description -----------------------------
# This is the xnl configuration file. Project specific defines for using the
# general xnl engine in xnl.c goes here. Specify which XNL commands your project
# will support in the enum below.
#
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/
#include <string.h>
#include "xnl_config.h"
#include "xnl.h"
#include "osal.h"
#include "taskconfig.h"
#include "xcmp_config.h"
/************************************************************************************
*
*--------------------------------- Revision History ----------------------------------
*
*   AUTHOR          Date Modified       Tracking Number         Description
* Abhishek Trivedi  09/02/2013         CCMPD01827271       XNL Application Configurations
* Abhishek Trivedi  11/8/2013          CCMPD01834482       Updates for XNL-XCMP 
* WRN637            11/20/2013         CCMPD01838036       Resolve Gateway Reset issue
* WRN637            20/12/2013         CCMPD01833277       WPTT state machine
* WRN637            01/22/2014         CCMPD01854392       Move XCMP related handling to HSP_XNL_XCMP_MANAGER_TASK to prevent multiple access to XCMP decoder at the same time    
* Abhishek Trivedi  03/18/2014	       CCMPD01873731	   Rework for XNL-XCMP 
* Aneeket Patkar    03/28/2014         CCMPD01877631       Remove Volume ramp feature as per the updated design
* Abhishek Trivedi  04/04/2014         CCMPD01882874       Error Recovery 
* Abhishek Trivedi  04/18/2014         CCMPD01885148       System Ping Timeout
* Phong Tran        05/12/2014         CCMPD01885130       Modified to support XXT device
* Abhishek Trivedi  11/11/2014         CCMPD01945832       Edge of Range Support
*--------------------------- End of History Template-----------------------------
* *********************************************************************************/

/* ***************************************************************************
                                GLOBAL VARIABLES
   ************************************************************************** */
UINT8_T mem_block1[GLB_MEM_XNL_REQ_SIZE];
UINT8_T mem_block2[GLB_MEM_XNL_REQ_SIZE];
bool mem_block1_used = FALSE;
bool mem_block2_used = FALSE;

bool hsp_ping_rcvd = FALSE;
bool spp_ping_rcvd = FALSE;

xQueueHandle hsp_xnl_xcmp_data_rx_Q, spp_xnl_xcmp_data_rx_Q;

TimerDescriptorType Device1Timer[NUMBER_OF_HSP_XNL_TIMERS];
TimerDescriptorType Device2Timer[NUMBER_OF_SPP_XNL_TIMERS];

#define    DEVICE_1_TIMER(TIMER, MSG_ID, TIMER_VAL, DATA_PTR)      TaskCancelTimer(&Device1Timer[TIMER]);\
           Device1Timer[TIMER].TimerMatchValue = \
           SYSTEM_TIME + TIMER_VAL; \
           Device1Timer[TIMER].TaskToNotify = HSP_XNL_XCMP_TIMERS_TASK; \
           Device1Timer[TIMER].MessageID = MSG_ID; \
           Device1Timer[TIMER].Message = DATA_PTR; \
           Device1Timer[TIMER].Next = NULL; \
           ScheduleTimer(&Device1Timer[TIMER])

#define CANCEL_DEVICE_1_TIMER(TIMER)          TaskCancelTimer(&Device1Timer[TIMER])

#define    DEVICE_2_TIMER(TIMER, MSG_ID, TIMER_VAL, DATA_PTR)      TaskCancelTimer(&Device2Timer[TIMER]);\
           Device2Timer[TIMER].TimerMatchValue = \
           SYSTEM_TIME + TIMER_VAL; \
           Device2Timer[TIMER].TaskToNotify = HSP_XNL_XCMP_TIMERS_TASK; \
           Device2Timer[TIMER].MessageID = MSG_ID; \
           Device2Timer[TIMER].Message = DATA_PTR; \
           Device2Timer[TIMER].Next = NULL; \
           ScheduleTimer(&Device2Timer[TIMER])

#define CANCEL_DEVICE_2_TIMER(TIMER)          TaskCancelTimer(&Device2Timer[TIMER])

           
/* ***************************************************************************
                                EXTERNAL VARIABLES
   ************************************************************************** */
extern UINT16_T device_1_id;
extern UINT16_T device_2_id;
extern tc_flags_t tc_flags;
extern TimerDescriptorType bt_system_ping_timer[];
/* ***************************************************************************
                                EXTERNAL FUNCTIONS
   ************************************************************************** */
extern success_failure_t send_hsp_xnl_msg(UINT8_T *data_tx_ptr, UINT16_T data_tx_length);
extern success_failure_t send_spp_xnl_msg(UINT8_T *data_tx_ptr, UINT16_T data_tx_length);
extern void fatal_error_reset(void);
/* ***************************************************************************
                                FUNCTION PROTOTYPES
   ************************************************************************** */
void xnl_device_report_failure (UINT16_T xnl_devid, xnl_error_type_t error);
/* ***************************************************************************
                                FUNCTION DEFINITIONS
   ************************************************************************** */
/*=============================================================================
	FUNCTION: hsp_xnl_xcmp_manager_task

	DESCRIPTION: XNL XCMP task.

	ARGUMENTS PASSED: Pointer to parameters

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void hsp_xnl_xcmp_manager_task(void *pvParameters)
{
  UINT8_T rxd_byte = 0; 
  for(;;)
  {    
    while ( xQueueReceive( hsp_xnl_xcmp_data_rx_Q, &rxd_byte, portMAX_DELAY) == TRUE)
    {
      if (xnl_xcmp_data_parser(device_1_id, rxd_byte) == SUCC) 
      {
        // Begin processing once a complete message is received
        xnl_state_machine(device_1_id);
      }      
    }
  }
}
/*=============================================================================
	FUNCTION: spp_xnl_xcmp_manager_task

	DESCRIPTION: XNL XCMP task.

	ARGUMENTS PASSED: Pointer to parameters

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void spp_xnl_xcmp_manager_task(void *pvParameters)
{
  UINT8_T rxd_byte = 0;
  
  for(;;)
  {    
    while ( xQueueReceive( spp_xnl_xcmp_data_rx_Q, &rxd_byte, portMAX_DELAY) == TRUE)
    {
      if (xnl_xcmp_data_parser(device_2_id, rxd_byte) == SUCC) 
      {
        // Begin processing once a complete message is received
        xnl_state_machine(device_2_id);
      }      
    }
  }
}
/*=============================================================================
	FUNCTION: hsp_xnl_xcmp_timers_task

	DESCRIPTION: XNL XCMP timers task.

	ARGUMENTS PASSED: Pointer to parameters

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
void hsp_xnl_xcmp_timers_task(void *pvParameters)
{

  for(;;)
  {  
    unsigned char message_id;
    ms_task_msg_t *msg;
    msg = (ms_task_msg_t *)Get_Msg_w_Time_Out(SUSPEND_INDEFFINATELY);
  
    if(msg != NULL)
    {
      message_id = msg->ros_msg.radio_if_msg.sub_opcode;
      OS_free(msg);
  
      switch (message_id)
      {
        case HSP_MASTER_STATUS_BCAST_TIMER_EXPIRED:
          xnl_timer_expired(device_1_id, message_id);
          break;
        case HSP_DATA_MSG_ACK_TIMER_EXPIRED:
          xnl_timer_expired(device_1_id, message_id);
          break;
        case HSP_PARSER_SM_RESET_TIMER_EXPIRED:
          xnl_timer_expired(device_1_id, message_id);
          break;   
        case HSP_ENUM_TIMER_EXPIRED:
          xnl_timer_expired(device_1_id, message_id);
          break; 
        case SPP_MASTER_STATUS_BCAST_TIMER_EXPIRED:
          xnl_timer_expired(device_2_id, message_id);
          break;
        case SPP_DATA_MSG_ACK_TIMER_EXPIRED:
          xnl_timer_expired(device_2_id, message_id);
          break;
        case SPP_PARSER_SM_RESET_TIMER_EXPIRED:
          xnl_timer_expired(device_2_id, message_id);
          break;   
        case SPP_ENUM_TIMER_EXPIRED:
          xnl_timer_expired(device_2_id, message_id);
          break;            
        default:
          break;
      }
    }
  }
}
/*=============================================================================
	FUNCTION: spp_xnl_xcmp_timers_task

	DESCRIPTION: XNL XCMP timers task.

	ARGUMENTS PASSED: Pointer to parameters

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
==============================================================================*/
/*void spp_xnl_xcmp_timers_task(void *pvParameters)
{
  for(;;)
  {  
    unsigned char message_id;
    ms_task_msg_t *msg;
  
    msg = (ms_task_msg_t *)Get_Msg_w_Time_Out(SUSPEND_INDEFFINATELY);
  
    if(msg != NULL)
    {
      message_id = msg->ros_msg.radio_if_msg.sub_opcode;
      OS_free(msg);
  
      switch (message_id)
      {
       
        default:
          break;
      }
    }
  }
}*/
/*=============================================================================
	FUNCTION: xnl_schedule_timer()

    DESCRIPTION: This function is used to schedule a timer for two devices
=============================================================================*/
void xnl_schedule_timer(UINT16_T xnl_devid, UINT8_T msg_id, int delay)
{
  if( xnl_devid == device_1_id )
  {
    switch(msg_id)
    {
      case MASTER_STATUS_BCAST_TIMER:
        DEVICE_1_TIMER(MASTER_STATUS_BCAST, HSP_MASTER_STATUS_BCAST_TIMER_EXPIRED, delay, NULL);        
        break;
      case DATA_MSG_ACK_TIMER:
        DEVICE_1_TIMER(DATA_MSG_ACK, HSP_DATA_MSG_ACK_TIMER_EXPIRED, delay, NULL);        
        break;
      case PARSER_SM_RESET_TIMER:
        DEVICE_1_TIMER(PARSER_SM_RESET, HSP_PARSER_SM_RESET_TIMER_EXPIRED, delay, NULL);        
        break;
      case XNL_XCMP_ENUM_TIMER:
        DEVICE_1_TIMER(XNL_XCMP_ENUM, HSP_ENUM_TIMER_EXPIRED, delay, NULL);        
        break;        
      default:
        break;
    }
  }
  else if( xnl_devid == device_2_id )
  {
    switch(msg_id)
    {
      case MASTER_STATUS_BCAST_TIMER:
        DEVICE_2_TIMER(SPP_MASTER_STATUS_BCAST, SPP_MASTER_STATUS_BCAST_TIMER_EXPIRED, delay, NULL);        
        break;
      case DATA_MSG_ACK_TIMER:
        DEVICE_2_TIMER(SPP_DATA_MSG_ACK, SPP_DATA_MSG_ACK_TIMER_EXPIRED, delay, NULL);            
        break;
      case PARSER_SM_RESET_TIMER:
        DEVICE_2_TIMER(SPP_PARSER_SM_RESET, SPP_PARSER_SM_RESET_TIMER_EXPIRED, delay, NULL);    
        break;
      case XNL_XCMP_ENUM_TIMER:
        DEVICE_2_TIMER(SPP_XNL_XCMP_ENUM, SPP_ENUM_TIMER_EXPIRED, delay, NULL);        
        break;        
      default:
        break;
    }    
  }
  else
  {
    xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);
  }
}


/*=============================================================================
	FUNCTION: xnl_cancel_timer()

    DESCRIPTION: This function is used to cancel a scheduled timer for two devices
=============================================================================*/
void xnl_cancel_timer(UINT16_T xnl_devid, UINT8_T msg_id)
{
  if( xnl_devid == device_1_id )
  {
    switch(msg_id)
    {
      case MASTER_STATUS_BCAST:
        CANCEL_DEVICE_1_TIMER(MASTER_STATUS_BCAST);        
        break;
      case DATA_MSG_ACK:
        CANCEL_DEVICE_1_TIMER(DATA_MSG_ACK);          
        break;
      case PARSER_SM_RESET:
        CANCEL_DEVICE_1_TIMER(PARSER_SM_RESET);          
        break;
      case XNL_XCMP_ENUM:
        CANCEL_DEVICE_1_TIMER(XNL_XCMP_ENUM);         
        break;
      default:
        break;
    }    
  }
  else if( xnl_devid == device_2_id )
  {
    switch(msg_id)
    {
      case MASTER_STATUS_BCAST:
        CANCEL_DEVICE_2_TIMER(SPP_MASTER_STATUS_BCAST);        
        break;
      case DATA_MSG_ACK:
        CANCEL_DEVICE_2_TIMER(SPP_DATA_MSG_ACK);          
        break;
      case PARSER_SM_RESET:
        CANCEL_DEVICE_2_TIMER(SPP_PARSER_SM_RESET);          
        break;
      case XNL_XCMP_ENUM:
        CANCEL_DEVICE_2_TIMER(XNL_XCMP_ENUM);         
        break;        
      default:
        break;
    }  
  }
  else
  {
      xnl_device_report_failure(xnl_devid, XNL_INVALID_PARAMETER);
  }
}
/*=============================================================================
	FUNCTION: xnl_copy_rxd_data_byte()

	DESCRIPTION: Copies data byte for future processing.

	ARGUMENTS PASSED: data byte

	RETURN VALUE: none
=============================================================================*/
success_failure_t xnl_copy_rxd_data_byte(data_buffer_struct_t * ptr, UINT8_T data)
{
  buffer_write_and_increment_in_index(ptr, data);
  //Need to change the buffer utilites to return value
  return TRUE;
}
/*=============================================================================
	FUNCTION: xnl_send_retry_message()

    DESCRIPTION: This function is used whenever the XNL stack needs to send a
	retry XCMP message to the hardware interface.
=============================================================================*/
success_failure_t xnl_send_retry_message(const UINT8_T index)
{
  return xnl_send_msg(xnl_device[index]->address, xnl_device[index]->retry_buffer_ptr, xnl_device[index]->retry_message_size);
}
/*=============================================================================
	FUNCTION: xnl_send_data_to_hardware()

    DESCRIPTION: This function is used whenever the XNL stack needs to send an
	XCMP message to the hardware interface.
=============================================================================*/
success_failure_t xnl_send_data_to_hardware(const UINT8_T index, const UINT8_T* header_ptr, const UINT8_T header_size, 
							   const UINT8_T* payload_ptr, const UINT16_T payload_size) 
{
  UINT8_T return_value = FAILURE;
  UINT16_T i = 0;

  if(header_ptr != NULL)
  {
    for(i = 0; i < header_size; i++) 
    {
      xnl_device[index]->xnl_transmit_array_glb[i] = *header_ptr++;
    }
    for(i = header_size; i < header_size + payload_size; i++)
    {
      xnl_device[index]->xnl_transmit_array_glb[i] = *payload_ptr++;
    }
    //Send out the message to the hardware interface
    return_value = xnl_send_msg(xnl_device[index]->address, &(xnl_device[index]->xnl_transmit_array_glb[0]), header_size + payload_size);	
  }
  else
  {
    // Header pointer is NULL
    xnl_device_report_failure(xnl_device[index]->address, XNL_INVALID_PARAMETER);
  }
  return return_value;	
}
/*=============================================================================
	FUNCTION: xnl_send_data_ack_to_hardware()

    DESCRIPTION: This function is used whenever the XNL stack needs to send an
	XNL Ack to the hardware interface. This is to avoid using the same 
        buffer as the data message
=============================================================================*/
success_failure_t xnl_send_data_ack_to_hardware(const UINT8_T index, const UINT8_T* header_ptr)
{
  UINT8_T return_value = FAILURE;
  UINT16_T i = 0;

  if(header_ptr != NULL)
  {
    for(i = 0; i < XNL_HEADER_SIZE; i++) 
    {
      xnl_device[index]->xnl_ack_array[i] = *header_ptr++;
    }

    //Send out the message to the hardware interface
    return_value = xnl_send_msg(xnl_device[index]->address, &(xnl_device[index]->xnl_ack_array[0]), XNL_HEADER_SIZE);	
  }
  else
  {
    // Header pointer is NULL
    xnl_device_report_failure(xnl_device[index]->address, XNL_INVALID_PARAMETER);
  }
  return return_value;	  
}
/*=============================================================================
	FUNCTION: xnl_send_msg()

        DESCRIPTION: Send the message to HSP or SPP stream handler

	ARGUMENTS PASSED: XNL Device ID and length  

	REFERENCE ARGUMENTS PASSED: data pointer

	RETURN VALUE: TRUE/FALSE
=============================================================================*/
success_failure_t xnl_send_msg(const UINT16_T xnl_devid, const UINT8_T *data_tx_ptr, const UINT16_T data_tx_length) 
{
  if (xnl_devid == device_1_id) 
  {
    if (tc_flags.simulator_enable && tc_flags.simulator_type == TC_SIM_DEVICE)
    {
      return send_data_to_pc((UINT8_T *)data_tx_ptr, data_tx_length);
    }
    else
    {
      return send_hsp_xnl_msg((UINT8_T *)data_tx_ptr, data_tx_length);
    }
  }			
  else if (xnl_devid == device_2_id)
  {
    return send_spp_xnl_msg((UINT8_T *)data_tx_ptr, data_tx_length);
  }			
  else
    return FAILURE;
}
/*=============================================================================
	FUNCTION: xnl_queue_init()

	DESCRIPTION: This will initialize the XNL Queues.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_queue_init(void)
{
  /* Initialize the Conrols - Radio Interface Q */
  hsp_xnl_xcmp_data_rx_Q  = xQueueCreate( HSP_DATA_Q_LENGTH, ( unsigned portBASE_TYPE ) sizeof( UINT8_T ) );
  spp_xnl_xcmp_data_rx_Q  = xQueueCreate( SPP_DATA_Q_LENGTH, ( unsigned portBASE_TYPE ) sizeof( UINT8_T ) );  
}
/*=============================================================================
	FUNCTION: xnl_buffer_init()

	DESCRIPTION: This will initialize the XNL buffers

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_buffer_init(void)
{
 /* // Initialize XNL Rx Buffer
  xnl_receive_buffer_glb.data_buffer_array_ptr      = &xnl_receive_array_glb[0];
  xnl_receive_buffer_glb.buffer_size                = XNL_RX_BUFFER_SIZE;
  xnl_receive_buffer_glb.in_buffer_index            = 0;
  xnl_receive_buffer_glb.out_buffer_index           = 0;
  
  // Initialize XNL Tx Buffer
  xnl_transmit_buffer_glb.data_buffer_array_ptr  = &xnl_transmit_array_glb[0];
  xnl_transmit_buffer_glb.buffer_size            = XNL_TX_BUFFER_SIZE;
  xnl_transmit_buffer_glb.in_buffer_index        = 0;
  xnl_transmit_buffer_glb.out_buffer_index       = 0;

    // Initialize XNL Retry Buffer
  xnl_relay_buffer_glb.data_buffer_array_ptr  = &xnl_relay_array_glb[0];
  xnl_relay_buffer_glb.buffer_size            = XNL_RELAY_BUFFER_SIZE;
  xnl_relay_buffer_glb.in_buffer_index        = 0;
  xnl_relay_buffer_glb.out_buffer_index       = 0;
  
  // Set all elements in these buffers to initial value of 0x00
  memset(xnl_receive_array_glb, 0, sizeof(xnl_receive_array_glb));
  memset(xnl_transmit_array_glb, 0, sizeof(xnl_transmit_array_glb));
  memset(xnl_relay_array_glb, 0, sizeof(xnl_relay_array_glb)); */
} 
/*=============================================================================
	FUNCTION: xnl_buffer_read_two_bytes()

	DESCRIPTION: Gets two bytes from the receive buffer.
=============================================================================*/
UINT16_T xnl_buffer_read_two_bytes(data_buffer_struct_t * ptr)
{
  UINT16_T data = 0;
  data = xnl_buffer_read_one_byte(ptr);        
  data <<= 8;
  data |= xnl_buffer_read_one_byte(ptr);
  return data;
} 
/*=============================================================================
	FUNCTION: xnl_rx_check2bytes()

	DESCRIPTION: Checks if two bytes in the XNL Rx buffer are the same as
                     the parameter.

	ARGUMENTS PASSED: value

	RETURN VALUE: TRUE or FALSE
=============================================================================*/
BOOL_T xnl_rx_check2bytes(data_buffer_struct_t * ptr, UINT16_T value)
{
  UINT16_T rxbyte = 0;

  rxbyte = xnl_buffer_read_two_bytes(ptr);

  if(rxbyte == value)
    return TRUE;
  else
    return FALSE;
} 
/*=============================================================================
	FUNCTION: xnl_tx_clr_msg_retry_buffer()

        DESCRIPTION: Clears the XNL Retry buffer

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE: none
=============================================================================*/
void xnl_tx_clr_msg_relay_buffer(UINT8_T index)
{
  memset(xnl_device[index]->xnl_relay_array_glb, 0, sizeof(xnl_device[index]->xnl_relay_array_glb));
}
/*=============================================================================
	FUNCTION: xnl_xcmp_init_complete()

        DESCRIPTION: Callback function for User to send XNL-XCMP Init complete to system.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_xcmp_init_complete(UINT16_T xnl_devid)
{
  UINT16_T *ptr = NULL;
  ptr = xnl_malloc(sizeof(UINT16_T));

  if(ptr == NULL)
  {
    // Error, no memory space
    xnl_device_report_failure(xnl_devid, XNL_MEMORY_ALLOC_ERROR);
  }
  *ptr = xnl_devid;
  SEND_MESSAGE_TO_CONNECTION_MNGR(XNL_XCMP_INITIALIZATION_COMPLETE, ptr, 2);
}
/*=============================================================================
	FUNCTION: xnl_init_complete()

        DESCRIPTION: Callback function for User to send XNL Init complete to system.

	ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_init_complete(UINT16_T xnl_devid)
{

}
/*=============================================================================
	FUNCTION: received_xnl_reset()

        DESCRIPTION: Callback function for User to send XNL reset to system.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void received_xnl_reset(UINT16_T xnl_devid)
{

}
/*=============================================================================
	FUNCTION: xnl_ack_received()

        DESCRIPTION: Callback function for User to perform an action when 
                     xnl-xcmp data message ack was received.

	ARGUMENTS PASSED: address of the device the ack was recevied for

	RETURN VALUE: none
=============================================================================*/
void xnl_ack_received(UINT16_T address)
{
  if(address == device_1_id)
  {
    if(hsp_ping_rcvd == TRUE)
    {
      CANCEL_BT_SYSTEM_PING_TIMER(HSP_PING);
      BT_SYSTEM_PING_TIMER(HSP_PING, HSP_SYSTEM_PING_TIMER_EXPIRED, SYSTEM_PING_TIME, NULL);
    }
  }
  else if (address == device_2_id)
  {
    if(spp_ping_rcvd == TRUE)
    {
      CANCEL_BT_SYSTEM_PING_TIMER(SPP_PING);
      BT_SYSTEM_PING_TIMER(SPP_PING, SPP_SYSTEM_PING_TIMER_EXPIRED, SYSTEM_PING_TIME, NULL);
    }    
  }
}
/*=============================================================================
	FUNCTION: disable_system_ping_feature()

        DESCRIPTION: Cancel system ping timer for respective device and reset 
                     the variable

	ARGUMENTS PASSED: address of the device that disconnected

	RETURN VALUE: none
=============================================================================*/
void disable_system_ping_feature(UINT16_T address)
{
  if(address == device_1_id)
  {
    hsp_ping_rcvd = FALSE;
    CANCEL_BT_SYSTEM_PING_TIMER(HSP_PING);
  }
  else if (address == device_2_id)
  {
    spp_ping_rcvd = FALSE;
    CANCEL_BT_SYSTEM_PING_TIMER(SPP_PING);
  }  
}
/*=============================================================================
	FUNCTION: xnl_data_msg_retries_expired()

        DESCRIPTION: Callback function for User to perform an action when 
                     xnl-xcmp data message retries expired.

	ARGUMENTS PASSED: XNL Device ID

	RETURN VALUE: none
=============================================================================*/
void xnl_data_msg_retries_expired(UINT16_T xnl_devid)
{
  if((xnl_devid == device_1_id) && (device_1_id != 0))
  {
    SEND_MESSAGE_TO_CONNECTION_MNGR(HSP_DATA_COMM_ERROR, NULL, 0);
  }
  else if ((xnl_devid == device_2_id) && (device_2_id != 0))
  {
    SEND_MESSAGE_TO_CONNECTION_MNGR(SPP_DATA_COMM_ERROR, NULL, 0);
  }
/*
  UINT8_T i;
  UINT8_T index = xnl_get_device_index_from_id(xnl_devid);

  xnl_reset_device_structure(xnl_device[index]);
  xnl_device[index]->is_ack_pending = FALSE;
  for( i = 0; i < NUMBER_OF_XNL_TX_SUPPORTED_MSGS ; i++) 
  {
    CANCEL_RETRY_XCMP_MSG_TIMER(i);
  }*/
}
/*=============================================================================
	FUNCTION: xnl_reset_xcmp_retry_failures()

        DESCRIPTION: Callback function for User to perform an action when 
                     xnl retry parameters need to be reset.

	ARGUMENTS PASSED: XNL Device ID

	RETURN VALUE: none
=============================================================================*/
void xnl_reset_xcmp_retry_failures(UINT16_T xnl_devid)
{
  xnl_reset_retry_params(xnl_devid);
}
/*=============================================================================
	FUNCTION: xnl_write_system_log

        DESCRIPTION: Callback function for XNL to write logs.

	ARGUMENTS PASSED: none

        RETURN VALUE: none
=============================================================================*/
void xnl_write_system_log(UINT8_T log)
{
  //SystemLogWrite(TASK_XNL_XCMP_MANAGER_ID,log);
}

/*=============================================================================
	FUNCTION: disable_interrupts() & enable_interrupts()

        DESCRIPTION: Callback function for User to enable and disable interrupts.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void disable_interrupts(void)
{
  Disable_global_interrupt();
}
void enable_interrupts(void)
{
  Enable_global_interrupt();
}
/*=============================================================================
	FUNCTION: xnl_malloc() & xnl_free()

        DESCRIPTION: Callback function for User to malloc and free.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void* xnl_malloc(int size)
{
  void * ptr;
  ptr = OS_malloc(size);
  return ptr;  
}
void xnl_free(void *ptr)
{
  OS_free(ptr);  
}
/*=============================================================================
	FUNCTION: xnl_global_alloc() & xnl_global_alloc_free()

        DESCRIPTION: Callback function for User to assign global memory.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void* xnl_global_alloc(void)
{
  void * ptr;
  if(!mem_block1_used)
  {
    ptr = &mem_block1;
    mem_block1_used = TRUE;
    return ptr;
  }
  else if(!mem_block2_used)
  {
    ptr = &mem_block2;
    mem_block2_used = TRUE;  
    return ptr;
  }
  else
  {
    return NULL;
  }
}
void xnl_global_alloc_free(void *ptr)
{
  if(ptr == &mem_block1)
  {
    mem_block1_used = FALSE;
  }
  else if(ptr == &mem_block2)
  {
    mem_block2_used = FALSE; 
  }
}
/*=============================================================================
	FUNCTION: xnl_device_report_failure

        DESCRIPTION: Callback function for XNL to report failures to the system.

	ARGUMENTS PASSED: error type for the XNL device connected

        RETURN VALUE: none
=============================================================================*/
void xnl_device_report_failure (UINT16_T xnl_devid, xnl_error_type_t error)
{
  switch(error)
  {
    case XNL_CONNECTION_ERROR:
            xnl_data_msg_retries_expired(xnl_devid);
            break;
    case XNL_MEMORY_ALLOC_ERROR:
            // Memory not available. Do a watchdog reset
            fatal_error_reset();
    case XNL_RECEIVE_OVERFLOW_ERROR:
            // Take appropriate action here...
            break;
    case XNL_INVALID_PARAMETER:
    case XNL_INIT_ERROR:
            break;
    default:
            break;
  }
}
