#ifndef XNL_CONFIG_H
#define XNL_CONFIG_H
/*################################################################################
#
#                  F U N C T I O N S P E C I F I C A T I O N
#             COPYRIGHT 2013 MOTOROLA SOLUTIONS,INC. ALL RIGHTS RESERVED.
#                    MOTOROLA CONFIDENTIAL RESTRICTED
#
#################################################################################
#
# FILE NAME: xnl_config.h
#
# --------------------------- General Description -----------------------------
# This is the xnl configuration file. Project specific defines for using the
# general xnl engine in xnl.c goes here. Specify which XNL commands your project
# will support in the enum below.
#
#
# --------------------------- HEADER FILE INCLUDES ----------------------------*/
/* Project specific support */
#include "config.h"
#include "utilities.h"
#include "timers.h"
/********************************************************************************
*
*--------------------------- Revision History ----------------------------------
*
*   AUTHOR              Date Modified         Tracking Number         Description
* Abhishek Trivedi      09/02/2013            CCMPD01827271       XNL Application Configurations
* Abhishek Trivedi      11/8/2013             CCMPD01834482       Updates for XNL-XCMP 
* WRN637                01/22/2014            CCMPD01854392       Move XCMP related handling to HSP_XNL_XCMP_MANAGER_TASK to prevent multiple access to XCMP decoder at the same time    
* Abhishek Trivedi      03/18/2014	      CCMPD01873731	  Rework for XNL-XCMP 
* Abhishek Trivedi      04/18/2014            CCMPD01885148       System Ping Timeout
* Abhishek Trivedi      11/11/2014            CCMPD01945832       Edge of Range Support
**--------------------------- End of History Template-----------------------------
* *********************************************************************************/
//------------------------------------------------------------------------------
//      Defines
//------------------------------------------------------------------------------
// Authentication using the encrypted value is not currently supported for this product
//#define AUTHENTICATION_SUPPORTED

/* XNL Buffer Sizes */
#define XNL_RX_BUFFER_SIZE            255
#define XNL_TX_BUFFER_SIZE            50
#define XNL_RELAY_BUFFER_SIZE         50
#define DEVICE_RX_BUFFER_SIZE         255

#define HSP_DATA_Q_LENGTH             XNL_RX_BUFFER_SIZE
#define SPP_DATA_Q_LENGTH             XNL_RX_BUFFER_SIZE

#define NUMBER_OF_DEVICES_SUPPORTED   2
#define GLB_MEM_XNL_REQ_SIZE          500

#define XNL_TIMER_MASTER_STATUS_BRDCST_DELAY   100 
#define XNL_TIMER_PARSER_SM_RESET_DELAY        250       

#define MAX_VOL                             0xFF
#define MIN_VOL                             0x00
#define VOL_RAMPING_TIME                    1000 
#define VOL_TIMEOUT_TIME                    5000

#define DATA_MSG_ACK_TIMER        1
#define MASTER_STATUS_BCAST_TIMER 2
#define PARSER_SM_RESET_TIMER     3
#define XNL_XCMP_ENUM_TIMER       4
// Macros interfacing for timers
#define xnl_timer_cancel_data_message_ack(xnl_devid)             xnl_cancel_timer(xnl_devid, DATA_MSG_ACK_TIMER)
#define xnl_timer_cancel_send_master_status(xnl_devid)           xnl_cancel_timer(xnl_devid, MASTER_STATUS_BCAST_TIMER)
#define xnl_cancel_parser_reset_timer(xnl_devid)                 xnl_cancel_timer(xnl_devid, PARSER_SM_RESET_TIMER)
#define xnl_cancel_enum_timer(xnl_devid)                         xnl_cancel_timer(xnl_devid, XNL_XCMP_ENUM_TIMER)

#define xnl_timer_start_data_message_ack(xnl_devid)              xnl_schedule_timer((xnl_devid), DATA_MSG_ACK_TIMER, (XNL_TIMER_DATA_MESSAGE_ACK_VALUE))
#define xnl_timer_start_send_master_status_delay(xnl_devid)      xnl_schedule_timer((xnl_devid), MASTER_STATUS_BCAST_TIMER, (XNL_TIMER_MASTER_STATUS_BRDCST_DELAY))
#define xnl_start_parser_reset_timer(xnl_devid)                  xnl_schedule_timer((xnl_devid), PARSER_SM_RESET_TIMER, (XNL_TIMER_PARSER_SM_RESET_DELAY))
#define xnl_start_xnl_xcmp_enum_timer(xnl_devid)                 xnl_schedule_timer((xnl_devid), XNL_XCMP_ENUM_TIMER, (XNL_TIMER_ENUM_DELAY))

#define SYSTEM_PING_TIME        40000

#define    BT_SYSTEM_PING_TIMER(TIMER, MSG_ID, TIMER_VAL, DATA_PTR)      TaskCancelTimer(&bt_system_ping_timer[TIMER]);\
           bt_system_ping_timer[TIMER].TimerMatchValue = \
           SYSTEM_TIME + TIMER_VAL; \
           bt_system_ping_timer[TIMER].TaskToNotify = CONNECTION_MANAGER; \
           bt_system_ping_timer[TIMER].MessageID = MSG_ID; \
           bt_system_ping_timer[TIMER].Message = DATA_PTR; \
           bt_system_ping_timer[TIMER].Next = NULL; \
           ScheduleTimer(&bt_system_ping_timer[TIMER])

#define CANCEL_BT_SYSTEM_PING_TIMER(TIMER)          TaskCancelTimer(&bt_system_ping_timer[TIMER])

//------------------------------------------------------------------------------
//      GLOBAL VARIABLES
//------------------------------------------------------------------------------
extern TimerDescriptorType xnl_msg_retry_timer[];
extern TimerDescriptorType vol_ramp_timer[];

//------------------------------------------------------------------------------
//    ENUMS &  STRUCTURES
//------------------------------------------------------------------------------
typedef enum hsp_xnl_timer_msgs
{
  DATA_MSG_ACK = 0,
  MASTER_STATUS_BCAST,
  PARSER_SM_RESET,
  XNL_XCMP_ENUM,
  //Keep adding here
  NUMBER_OF_HSP_XNL_TIMERS
} hsp_xnl_timer_msgs_t;

typedef enum spp_xnl_timer_msgs
{
  SPP_DATA_MSG_ACK = 0,
  SPP_MASTER_STATUS_BCAST,
  SPP_PARSER_SM_RESET,
  SPP_XNL_XCMP_ENUM,
  //Keep adding here
  NUMBER_OF_SPP_XNL_TIMERS
} spp_xnl_timer_msgs_t;

typedef enum xnl_timers_expired_msgs
{
  HSP_MASTER_STATUS_BCAST_TIMER_EXPIRED = 0,
  HSP_DATA_MSG_ACK_TIMER_EXPIRED,
  HSP_PARSER_SM_RESET_TIMER_EXPIRED,
  SPP_MASTER_STATUS_BCAST_TIMER_EXPIRED,
  SPP_DATA_MSG_ACK_TIMER_EXPIRED,
  SPP_PARSER_SM_RESET_TIMER_EXPIRED,
  HSP_ENUM_TIMER_EXPIRED,
  SPP_ENUM_TIMER_EXPIRED
}xnl_timers_expired_msgs_t;

typedef enum ping_timers
{
  HSP_PING = 0,
  SPP_PING,
  //Keep adding here
  NUMBER_OF_PING_TIMERS
} ping_timers_t;
//------------------------------------------------------------------------------
//      FUNCTION PROTOTYPES
//------------------------------------------------------------------------------
/*=============================================================================
    FUNCTION: xnl_buffer_transfer_to_receive()

    DESCRIPTION: Transfers a complete message from the hardware buffer to the
	XNL receive buffer. This function should only be called when it is guaranteed
	that the XNL stack got a complete message from the hardware interface.
=============================================================================*/
void xnl_buffer_transfer_to_receive (const UINT8_T index);
/*=============================================================================
	FUNCTION: xnl_buffer_discard_bytes()

	DESCRIPTION: Bypasses a number of bytes.

	ARGUMENTS PASSED: Number of bytes to skip.

	REFERENCE ARGUMENTS PASSED: address of pointer to buffer
=============================================================================*/
#define xnl_buffer_discard_bytes(ptr, count)	(buffer_increment_out_index((ptr), (count)))
/*=============================================================================
	FUNCTION: xnl_buffer_read_one_byte()

	DESCRIPTION: Read a byte from the XNL Receive buffer.

	REFERENCE ARGUMENTS PASSED: address of pointer to buffer
=============================================================================*/
#define xnl_buffer_read_one_byte(ptr)	(buffer_read_and_increment_out_index(ptr))

#define xnl_read_data_byte(ptr)         (buffer_read_and_increment_out_index(ptr))
#define xnl_get_buffer_size(ptr)        (buffer_get_size_of_new_data(ptr))

success_failure_t xnl_copy_rxd_data_byte(data_buffer_struct_t * ptr, UINT8_T data);
/*=============================================================================
	FUNCTION: xnl_tx_buffer()

	DESCRIPTION: Copies the XNL message bytes to output buffer.

	ARGUMENTS PASSED: Source pointer, message size

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_tx_buffer(UINT8_T* xnl_msg_ptr, UINT8_T* payload_ptr, UINT8_T xnl_msg_size, UINT16_T payload_size);
/*=============================================================================
	FUNCTION: xnl_rx_check2bytes()

	DESCRIPTION: Checks if two bytes in the XNL Rx buffer are the same as
                     the parameter.

	ARGUMENTS PASSED: value

	RETURN VALUE: TRUE or FALSE
=============================================================================*/
BOOL_T xnl_rx_check2bytes(data_buffer_struct_t * ptr, UINT16_T value);
/*=============================================================================
	FUNCTION: xnl_buffer_init()

	DESCRIPTION: This will initialize the XNL buffer.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_buffer_init(void);
/*=============================================================================
	FUNCTION: xnl_queue_init()

	DESCRIPTION: This will initialize the XNL Queues.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_queue_init(void);
/*=============================================================================
	FUNCTION: xnl_tx_clr_msg_relay_buffer()

        DESCRIPTION: Clears the buffer - used for retry here

	ARGUMENTS PASSED: buffer pointer

	REFERENCE ARGUMENTS PASSED:

	RETURN VALUE: none
=============================================================================*/
void xnl_tx_clr_msg_relay_buffer(UINT8_T index);
/*=============================================================================
	FUNCTION: xnl_xcmp_init_complete()

        DESCRIPTION: Callback function for User to send XNL-XCMP Init complete to system.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_xcmp_init_complete(UINT16_T xnl_devid);
/*=============================================================================
	FUNCTION: xnl_init_complete()

        DESCRIPTION: Callback function for User to send XNL Init complete to system.

	ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_init_complete(UINT16_T xnl_devid);
/*=============================================================================
	FUNCTION: received_xnl_reset()

        DESCRIPTION: Callback function for User to send XNL reset to system.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void received_xnl_reset(UINT16_T xnl_devid);
/*=============================================================================
	FUNCTION: xnl_data_msg_retries_expired()

        DESCRIPTION: Callback function for User to perform an action when 
                     xnl-xcmp data message retries expired.

	ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void xnl_data_msg_retries_expired(UINT16_T xnl_devid);
/*=============================================================================
	FUNCTION: disable_interrupts() & enable_interrupts()

        DESCRIPTION: Callback function for User to enable and disable interrupts.

	ARGUMENTS PASSED: none

	REFERENCE ARGUMENTS PASSED: none

	RETURN VALUE: none
=============================================================================*/
void disable_interrupts(void);
void enable_interrupts(void);
/*=============================================================================
	FUNCTION: xnl_write_system_log

        DESCRIPTION: Callback function for XNL to write logs.

	ARGUMENTS PASSED: none

        RETURN VALUE: none
=============================================================================*/
void xnl_write_system_log(UINT8_T log);
/*=============================================================================
	FUNCTION: XNL Malloc and Free

        DESCRIPTION: Callback function for memory or global space allocation.

	ARGUMENTS PASSED: pointer

        RETURN VALUE: none
=============================================================================*/
void* xnl_malloc(int size);
void* xnl_global_alloc();
void xnl_free(void *ptr);
void xnl_global_alloc_free(void *ptr);

void xnl_schedule_timer(UINT16_T xnl_devid, UINT8_T msg_id, int delay);
void xnl_cancel_timer(UINT16_T xnl_devid, UINT8_T msg_id);

success_failure_t xnl_send_retry_message(const UINT8_T index);
success_failure_t xnl_send_msg(const UINT16_T xnl_devid, const UINT8_T *data_tx_ptr, const UINT16_T data_tx_length);
success_failure_t xnl_send_data_to_hardware(const UINT8_T index, const UINT8_T* header_ptr, const UINT8_T header_size, 
							   const UINT8_T* payload_ptr, const UINT16_T payload_size);
success_failure_t xnl_send_data_ack_to_hardware(const UINT8_T index, const UINT8_T* header_ptr);

void xnl_reset_xcmp_retry_failures(UINT16_T xnl_devid);
#endif // XNL_CONFIG_H

