#!/bin/bash

SLEEP_INTERVAL=${1:-900}
MAX_FILES=${2:-384}
ENABLE_FTP=${3:-no}
FTP_SERVER_IP=${4:-}
FTP_USER=${5:-}
FTP_PASSWD=${6:-}
FTP_DEST_PATH=${7:-}
ENABLE_DEBUG=${8:-no}

# Monitor all the processes under /opt/trs/bin and strongswan
processList=`ls /opt/trs/bin && ls /opt/timSim/bin/`
processList="$processList charon"
uhz=`getconf CLK_TCK`
# It was reported once that getconf is not available on some FZM boards
if [ $? -gt 1 ]; then
  uhz=100
fi

running_num=$MAX_FILES

function usage()
{
  echo "Usage: $0 <SleepInterval MaxFiles EnableFtp FtpServerIp FtpUser FtpPasswd FtpDestPath EnableDebug>"
  echo
  echo "Usage: $0        Run the script with default config"
  echo
  echo "Usage: $0 [-h]   Show this help"
  echo
  echo "Sleep_Interval =  Time interval of dump collection in seconds, default 900(96 hours data collected on 15m interval)"
  echo "Max_Files =       Generate Max_Files number of files and then rotate, default 384(96h data collected on 15m interval). Max_Files could be as big as max file name limit permits(usually 255 chars)"
  echo "Enable_Ftp =      <yes/no>Transfer the dump files to ftp server or not, default is "no" (copy to /var/log/procCaptureLogs on the target)"
  echo "Ftp_Server_Ip =   Ftp server IP"
  echo "Ftp_User =        User name of the ftp server"
  echo "Ftp_Passwd =      Password of user@ftpServer"
  echo "Ftp_Dest_Path =   Destination folder on ftp server"
  echo "Enable_Debug =    <yes/no>Enable debug traces to stdout, default is "no" (to logfile ProcCapture.log)"
  echo
  echo "ProcCapture.sh when launched on the target, periodically dumps partial contents of the procfs for TRS processes in given interval. Each time it runs, it writes a compressed dump-file named procCapture_<n>.log.xz under /var/log/procCaptureLogs if ftp is not enabled. In these filenames, <n> is a running number, where higher numbers correspond to older dumps). "
  echo
}

if [ "$1" == "-h" ]; then
  usage
  exit 0
fi

if [ "$1" == "--help" ]; then
  usage
  exit 0
fi

if [ "$ENABLE_DEBUG" == "yes" ]; then
  echo "Script started at `date`"
  echo "Using below config for procCapture"
  echo "SLEEP_INTERVAL: $SLEEP_INTERVAL"
  echo "MAX_FILES:      $MAX_FILES"
  echo "ENABLE_FTP:     $ENABLE_FTP"
  echo "FTP_SERVER_IP:  $FTP_SERVER_IP"
  echo "FTP_USER:       $FTP_USER"
  echo "FTP_PASSWD:     $FTP_PASSWD"
  echo "FTP_DEST_PATH:  $FTP_DEST_PATH"
  echo "ENABLE_DEBUG:   $ENABLE_DEBUG"
else
  echo "##########################################################" >> /var/log/ProcCapture.log
  echo "Script started at `date`  " >> /var/log/ProcCapture.log
  echo "Using below config for procCapture" >> /var/log/ProcCapture.log
  echo "SLEEP_INTERVAL: $SLEEP_INTERVAL" >> /var/log/ProcCapture.log
  echo "MAX_FILES:      $MAX_FILES" >> /var/log/ProcCapture.log
  echo "ENABLE_FTP:     $ENABLE_FTP" >> /var/log/ProcCapture.log
  echo "FTP_SERVER_IP:  $FTP_SERVER_IP" >> /var/log/ProcCapture.log
  echo "FTP_USER:       $FTP_USER" >> /var/log/ProcCapture.log
  echo "FTP_PASSWD:     $FTP_PASSWD" >> /var/log/ProcCapture.log
  echo "FTP_DEST_PATH:  $FTP_DEST_PATH" >> /var/log/ProcCapture.log
  echo "ENABLE_DEBUG:   $ENABLE_DEBUG" >> /var/log/ProcCapture.log
fi

GetBuildInfo()
{
  /opt/trs/bin/objcli show_version >>/tmp/buildInfo.txt
  build_ver=`cat /tmp/buildInfo.txt|head -n 4| tail -n 1| awk -F ': ' '{print $2}'`
  rm -rf /tmp/buildInfo.txt
}

DumpHeader()
{
  fpa_pool=0
  max_fpa_pools=8
  time_stamp=`date +"%d-%m-%Y %H:%M:%S"`
  file_extn=$(printf "%03d\n" $running_num)

  echo "build_ver=$build_ver">>/tmp/procCapture_$file_extn.log
  echo "date=$time_stamp" >> /tmp/procCapture_$file_extn.log

  echo "@uptime {">>/tmp/procCapture_$file_extn.log
  cat /proc/uptime>>/tmp/procCapture_$file_extn.log
  echo "}@">>/tmp/procCapture_$file_extn.log

  echo "@meminfo {">>/tmp/procCapture_$file_extn.log
  cat /proc/meminfo>>/tmp/procCapture_$file_extn.log
  echo "}@">>/tmp/procCapture_$file_extn.log

  # In case of FSMR3 collect fpa availability
  if [ -f /sys/kernel/fpapool/fpa_pool ]; then
    echo "@fpainfo {">>/tmp/procCapture_$file_extn.log
    while [ $fpa_pool -lt $max_fpa_pools ]
    do
      echo "fpa$fpa_pool:" `head -$((fpa_pool + 2)) /sys/kernel/fpapool/fpa_pool | tail -1 | awk '{print $2}'` >>/tmp/procCapture_$file_extn.log
      fpa_pool=$((fpa_pool + 1))
    done
    echo "}@">>/tmp/procCapture_$file_extn.log
  fi

  if [ $running_num -ge 2 ]; then
    running_num=$(( $running_num - 1 ))
  else
    running_num=$MAX_FILES
  fi
}

DumpProc()
{
    utime=`cat /proc/$1/stat | cut -d " " -f14`
    stime=`cat /proc/$1/stat | cut -d " " -f15`
    cpu=$(((utime + stime) / $uhz))

    heap=`grep -A1 heap /proc/$1/smaps | grep Size | awk '{print $2}'`
    # more than one, loop thru
    if [ `echo $heap | wc -w` -gt 1 ]; then
      sum=0
      for heapSz in $heap
      do
        sum=$((sum+heapSz))
      done
      heap=$sum
    fi

	echo "@$1/stat {">>/tmp/procCapture_$file_extn.log
    cat /proc/$1/statm | cut -d " " -f1,2,3 |awk -v procss="$process" -v pid="$1" -v cpu="$cpu" -v heap="$heap" '{print pid,"("procss,") "cpu" "heap" "$1*4" "$2*4,$3*4,(($2*4)-($3*4))}' >> /tmp/procCapture_$file_extn.log
	echo "}@">>/tmp/procCapture_$file_extn.log
}

GetBuildInfo

while true
do
  DumpHeader

  # Collect process level information
  for process in $processList
  do
    # pid (process), cpu, heap, vsz, RSS, Shared, private
    # cpu = user ticks -> s
    # rss = nr.pages   -> kB
    pid="$(pidof $process)"

    # no pid, check for memcheck tool
    if [ -z "$pid" ]; then
      pid="$(pgrep -f tool=memcheck)"
      continue
    fi

    # still no pid, continue
    if [ -z "$pid" ]; then
      continue
    fi

    # more than one pid, loop thru
    if [ `echo $pid | wc -w` -gt 1 ]; then
      for mpid in $pid
      do
        DumpProc $mpid
      done
    else
      DumpProc $pid
    fi
  done

  if [ $ENABLE_FTP == 'yes' ]; then
    # Move files to FTP server
    xz /tmp/procCapture_$file_extn.log
    busybox ftpput -u $FTP_USER -p $FTP_PASSWD $FTP_SERVER_IP $FTP_DEST_PATH/procCapture_$file_extn.log.xz /tmp/procCapture_$file_extn.log.xz
    rm -rf /tmp/procCapture_$file_extn.log.xz
  else
    # Move files to /var/log to be collected later as part of extended/snaphot logs
    if [ ! -d /var/log/procCapture ]; then
      mkdir -p /var/log/procCapture
    fi
    # Check for number of file under /var/log/procCapture to rotate logs if required
    total_files=`ls -lrt /var/log/procCapture/proc* | wc -l`
    if [ $total_files -ge $MAX_FILES ]; then
      file_name=`ls -lrt /var/log/procCapture/proc* | head -n 1 | awk '{print $9}'`
      rm -rf $file_name
    fi
    xz /tmp/procCapture_$file_extn.log
    mv /tmp/procCapture_$file_extn.log.xz /var/log/procCapture/.
  fi
  sleep $SLEEP_INTERVAL
done
