#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright 2016 Nokia Solutions and Networks. All rights reserved


import socket
import threading
import log


class LoaderSocket():
    def __init__(self, socket_family, message_queue):
        self.queue = message_queue
        self.is_running = True
        self.sock = socket.socket(socket_family, socket.SOCK_STREAM)

    def send(self, message):
        message_length = len(message)
        netstring_to_send = str(message_length) + ':' + message + ','
        log.logger.info("socket netstring_to_send: %s" % netstring_to_send)

        try:
            self.sock.sendall(netstring_to_send)
        except socket.error as msg:
            log.logger.error("LoaderSocket:: failed to send message:(%s)" % msg.strerror)
            return False
        else:
            log.logger.debug("LoaderSocket:: message send ok")
            return True

    def get_message_data(self):
        data_length = '0'
        is_length_obtained = False
        is_length_started = False
        while not is_length_obtained:
            # NetString format is length:payload, for example 3:abc,
            # Read 1 byte each time to get the msg length
            content = self.sock.recv(1)
            if not content:
                log.logger.error("LoaderSocket:: socket is already closed by remote side")
                raise Exception("socket error")

            if not is_length_started and not content.isdigit() and content != ':':
                continue
            elif content == ':':
                is_length_obtained = True
            else:
                is_length_started = True
                data_length += content

        log.logger.info("data length: %s" % data_length)
        if data_length.isdigit() and int(data_length) < 20000:
            # Read data with length and a comma
            # Do not read one more byte due to not end with ',' yet
            # data = self.sock.recv(int(data_length) + 1)
            data = self.sock.recv(int(data_length))
            if not data:
                log.logger.error("LoaderSocket:: socket is already closed by remote side")
                raise Exception("socket error")
                
            return data
        else:
            log.logger.error("data length format is wrong or length is out of range")
            return None

    def handle_received_data(self, data):
        if data is None:
            log.logger.error("None data received from socket")
            return
        # if data[-1:] != ',':
        #     log.logger.error("Data NOT ends with comma")
        #     return

        # data_without_comma = data[:-1]
        data_without_comma = data
        log.logger.info("LoaderSocket:: received message, length = %d" % len(data_without_comma))
        self.queue.put(data_without_comma)

    def receive(self):
        try:
            while self.is_running:
                data = self.get_message_data()
                self.handle_received_data(data)
        except Exception as e:
            log.logger.info("Exception: %s" % str(e))
        finally:
            log.logger.info("self.is_running = %d" % self.is_running)
            log.logger.info("LoaderSocket close")
            self.sock.close()
            log.logger.info("LoaderSocket: terminate loader")
            signal.alarm(1)

    def start(self, address, threadName):
        try:
            self.sock.connect(address)
        except socket.error as msg:
            log.logger.error("LoaderSocket:: failed to create socket:(%s)" % msg.strerror)
            return False

        log.logger.info("LoaderSocket client connect OK")

        self.server_thread = threading.Thread(target=self.receive, name=threadName)
        self.server_thread.setDaemon(True)
        self.server_thread.start()
        return True

    def stop(self):
        log.logger.info("LoaderSocket stop")
        self.is_running = False
