"use strict";
var sp = require('sprintf-js');
var strftime = require("strftime");
var util = require("util");
var logstream = require("logrotate-stream");
var fs = require("fs");
var path = require("path");
var os=require("os");
if(os.arch() == 'mips64') {
  var platformLogger = require("./aasyslog");
} else {
  var platformLogger = require("./aasyslog_dummy");
}

var toLogFile = null;
var appenders = [];
var LOG_DIR_NAME = "logs";
var platformLoggerInitDone = false;

var appenderFactory = {
    "console": function() {
        return defaultLogAppender;
    },
    "syslog": function(opts) {
        return syslogAppender(opts.host, opts.port);
    },
    "file": function(opts) {
        return fileAppender(opts.filename, opts.maxSize, opts.rotationCount);
    },
    "platform": function() {
        return platformAppender;
    }
};

exports.initialize = function(list) {
    appenders = list.map(function(a) {
        if(! appenderFactory.hasOwnProperty(a.name)) {
            throw new Error("Unknown log appender: "+a.name);
        }
        return appenderFactory[a.name](a.opts);
    });
};

exports.setAppenders = function(list) {
    appenders = list;
};

/**
 * Asynchronous (returns immediately) appender functions,
 * iterates (synchronously) over all appenders.
 */
exports.append=function(logMessage) {
    setImmediate(function() {
        appenders.forEach(function(appender) {
            appender(logMessage);
        });
    });
};

function toSimpleLogLine(entry) {
    var stamp = strftime('%H:%M:%S.%L', new Date(entry.time));
    var source = entry.lvl;
    if(entry.component) {
        source = source + "/" + entry.component;
    }
    if(entry.pkg) {
        source = source + "/" + entry.pkg;
    }
    return util.format("<%s> %s:%s\n", stamp, source, entry.msg);
}

function redirectLogsToFile(logMessage) {
    var test_file_name = path.basename(process.argv[6], ".js");
    fs.appendFile(path.join(LOG_DIR_NAME, test_file_name + ".log"), logMessage);
}

/**
 * Console (stdout) appender.
 */
function defaultLogAppender(entry) {
     var logLine = toSimpleLogLine(entry);

     if (process.env.REDIRECT_LOGS) {
         redirectLogsToFile(logLine);
     }
     else {
         process.stdout.write(logLine);
     }
 }


/**
 * Syslog appender. Will use given host and port to send UDP log strings.
 * Uses module scoped socket and unrefs it to allow process exit.
 */
var syslogSocket;
var MAX_SYSLOG_SIZE_MATCHER = /.{1,512}/g;
function syslogAppender(host, port) {
    var counter = 0;
    var getCounterValue = function() {
        if(counter === 256) {
            counter = 0;
        }
        var str = counter.toString(16);
        counter++;
        if(str.length === 1) {
            return "0"+str;
        }
        return str;
    };

    var generateFormalLogLine = function(entry) {
        /* Add formal stuff around actual log line for btslog-tool to recognize it as log
         * example:
         * ff FCT-1011 <2004-01-01T00:06:24.475Z> INF/NODEOAM SocketMescom sendCommand...
         */

        var board = "FCT-1011";

        var lineStart = getCounterValue() + " " + board;
        var date = new Date(entry.time);
        var datePrint = date.toISOString();

        var logLevel = util.format("%s/%s", entry.lvl, entry.component);
        if(entry.pkg) {
            logLevel = util.format("%s/%s", logLevel, entry.pkg);
        }

        var eol = "\r\n";

        return util.format("%s <%s> %s %s%s",
                           lineStart,
                           datePrint,
                           logLevel,
                           entry.msg,
                           eol);
    };
    var createSocket = function() {
        if(syslogSocket) {
            syslogSocket.close();
        }
        syslogSocket = require("dgram").createSocket("udp4");
        syslogSocket.unref();
        syslogSocket.on("error", function(err) {
            process.stderr.write("ERROR: Logger syslog appender failed: "+err+"\n");
            createSocket();
        });
    };
    createSocket();
    return function(entry) {
        entry.msg.match(MAX_SYSLOG_SIZE_MATCHER).forEach(function(line) {
            var logline = generateFormalLogLine({
                counter: entry.counter,
                lvl: entry.lvl,
                time: entry.time,
                component: entry.component,
                pkg: entry.pkg,
                msg: line});
            var b = new Buffer(logline);
            syslogSocket.send(b, 0, b.length, port, host);
        });
    };
}

function fileAppender(filename, maxSize, rotationCount) {
    filename = require("path").resolve(filename);
    try {
        fs.writeFileSync(filename, "");
        if (toLogFile === null) {
            toLogFile = logstream({file: filename, size: maxSize || "20MB", keep: rotationCount || 3, compress: true});
        }
        else {
            toLogFile.constructor({file: filename, size: maxSize || "20MB", keep: rotationCount || 3, compress: true});
        }
    } catch(err) {
        process.stderr.write("ERROR: file appender setup failed: "+err+"\n");
    }

    return function(entry) {
        try {
            toLogFile.write(toSimpleLogLine(entry));
        } catch(err) {
            process.stderr.write("ERROR: write to" + filename + " failed: "+err+"\n");
        }

    };
}


/**
 * Platform appender. Forwards logging to AaSyslog
 */
function platformAppender(entry) {
    var message = "";

    if(platformLoggerInitDone == false) {
        platformLogger.AaSyslogInit();
        platformLoggerInitDone = true;
    }

    if(entry.component) { message = entry.component; }
    if(entry.pkg) { message = message + "/" + entry.pkg; }
    if(entry.msg) { message = message + "/" + entry.msg; }

    platformLogger.AaSyslogPrint(entry.lvl, message);
 }
