"use strict";
var appenders = require("./appenders");
var util = require("util");

/**
 * Logger class, represents a logger with given package and level.
 * private class, factory method is introduced below to create instances.
 */
function Logger(pkg, level) {
    if(pkg === undefined) {
        throw new Error("logging package or level not given");
    }
    this.pkg = pkg;
    this.setLogLevel(level);
}
Logger.prototype.log = function() {};
Logger.prototype.levels = {
    SILENT: 0,
    TRC: 1,
    DBG: 2,
    INF: 3,
    WRN: 4,
    ERR: 5
};
Logger.prototype.trace = loggerFunction(Logger.prototype.levels.TRC);
Logger.prototype.debug = loggerFunction(Logger.prototype.levels.DBG);
Logger.prototype.info = loggerFunction(Logger.prototype.levels.INF);
Logger.prototype.warn = loggerFunction(Logger.prototype.levels.WRN);
Logger.prototype.error = loggerFunction(Logger.prototype.levels.ERR);

Logger.prototype.setLogLevel = function(level) {
    var levelList = Object.keys(this.levels);
    if (level === undefined || typeof level !== "string") {
        throw new Error("Log level must be one of " +
                        JSON.stringify(Object.keys(this.levels)));
    } else if(levelList.indexOf(level) === -1) {
        throw new Error("Invalid logging level " + level +
                        ", must be one of " + JSON.stringify(Object.keys(this.levels)));
    }
    this.logLevel = levelList.indexOf(level);
    config.levels[this.pkg] = level;
};
Logger.prototype.getLogLevel = function() {
    return Object.keys(this.levels)[this.logLevel];
};
/* Logger class ends */

var component_name = "";
var loggers = {};
var config = {
    "levels":{
        "": "DBG"
    }
};

/**
 * The actual logging function that pushes logged lines to appenders.
 */
function loggerFunction(level) {
    var levels = Object.keys(Logger.prototype.levels);
    return function() {
        if(this.logLevel === this.levels.SILENT) {
            return;
        }
        if (level < this.logLevel) {
            return;
        }
        var args = [].slice.call(arguments);
        var text = args[0];
        if(args.length > 1 || typeof text !== 'string') {
            text = util.format.apply(this, args);
        }
        appenders.append({
            component: component_name,
            time: Date.now(),
            pkg: this.pkg,
            lvl: levels[level],
            msg: text
        });
    };
}

function createOrGetLogger(pkg) {
    if (pkg === undefined) {
        throw new Error("Parameter pkg is undefined");
    }
    if (!pkg.match(/\w+/)) {
        throw new Error("Package name not provided for logger");
    }
    if(loggers[pkg] !== undefined) {
        return loggers[pkg];
    }
    var level = config.levels[pkg] || config.levels[""];
    var logger = loggers[pkg] = new Logger(pkg, level);
    return logger;
}

function changeAllLogLevels(newLevels) {
    Object.keys(newLevels).forEach(function(key) {
        if(loggers[key]) {
            loggers[key].setLogLevel(newLevels[key]);
        }
        config.levels[key] = newLevels[key];
    });
}

function configuration() {
    return config;
}

function configure(newConfig) {
    if(typeof newConfig !== "object") {
        throw new Error("Logging configure needs an object parameter, eg. {levels: {\"\": \"DEBUG\"}}");
    }
    if(newConfig.hasOwnProperty("component")) {
        component_name = newConfig.component;
    }
    var newLevels;
    if(newConfig.hasOwnProperty("levels")) {
        newLevels = newConfig.levels;
    } else {
        newLevels = newConfig;
    }
    Object.keys(newLevels).forEach(function(key) {
        if(loggers[key]) {
            loggers[key].setLogLevel(newLevels[key]);
        }
        config.levels[key] = newLevels[key];
    });
    var newAppenders = newConfig.appenders || [
        {name: "console", opts: {}},
        {name: "syslog", opts: {port: 51000, host: "127.0.0.1"}}
    ];
    appenders.initialize(newAppenders);
}

module.exports.forPackage = createOrGetLogger;
module.exports.configure = configure;
module.exports.configuration = configuration;
module.exports.changeAllLogLevels = changeAllLogLevels;
