"use strict";

var util = require("util");

var _ = require("underscore");
var deepcopy = require("deepcopy");

var Base = require("./Base");
var constants = require("./constants");
var utils = require("./utils");

var any = constants.any;
var anyOrNone = constants.anyOrNone;
var anyOrNoneToEnd = constants.anyOrNoneToEnd;

var validate = require("./validate").expectation;

function Expectation(mock) {
    Base.call(this);

    this.mock = mock;

    this.expectedArgs = [anyOrNoneToEnd];
    this.expectedArgsIsSet = false;
    this.predicateContext = this;

    this.actualCallCount = 0;

    this.expectedMinCallCount = 0;
    this.expectedMaxCallCount = Infinity;

    this.expectedMaxCallCountIsSet = false;
    this.expectedMinCallCountIsSet = false;
    this.expectedCallCountIsSet = false;

    this.resolves = [];

    this.after = [];

    this.saveStacks();
}

util.inherits(Expectation, Base);

Expectation.prototype.saveStacks = function() {
    Error.captureStackTrace(this);

    // First, remove 'Error' so it's starts with 'at' and then shift it 2 spaces to right
    this.stack = this.stack.replace(/^Error\n/, "").replace(/ {4}/g, "      ");

    // todo There is a bug in mocha that makes the error message busted, remove these hacks when a fix is ready

    // Due to the bug, slashed must be doubled
    this.stack = this.stack.replace(/\//g, "//");

    // If the error message is as we expect we should find the essential part of it which contains word 'Context'
    var contextPart = this.stack.match(/(.*Context.*\n)/);

    if (Array.isArray(contextPart)) {
        this.shortStack = contextPart[0];
    }
    else {
        // As we cannot have a short stack, the bug will still linger if we don't replace colons with semicolons.
        // Also adding some new lines for readability.
        this.shortStack = this.stack.replace(/:/g, ";") + "\n\n";
    }
};

["toBeCalled", "times", "at", "with", "set", "and", "but", "which"]
    .forEach(function(chainName) {
        utils.createChain(Expectation.prototype, chainName, function() { return this; });
    });

Expectation.prototype.least = function(count) {
    validate.least.callability(this);
    validate.least.args(this, count);

    this.expectedMinCallCount = count;
    this.expectedMinCallCountIsSet = true;
    this.expectedCallCountIsSet = true;
    return this;
};

utils.createChain(Expectation.prototype, "unlimitedly", function() {
    return this.least(0);
});

Expectation.prototype.most = function(count) {
    validate.most.callability(this);
    validate.most.args(this, count);

    this.expectedMaxCallCount = count;
    this.expectedMaxCallCountIsSet = true;
    this.expectedCallCountIsSet = true;
    return this;
};

Expectation.prototype.between = function(from, to) {
    validate.between.callability(this);
    validate.between.args(this, from, to);

    return this.least(from).most(to);
};

Expectation.prototype.exactly = function(count) {
    validate.exactly.callability(this);
    validate.exactly.args(this, count);

    return this.between(count, count);
};

utils.createChain(Expectation.prototype, "never", function() {
    return this.exactly(0);
});

utils.createChain(Expectation.prototype, "once", function() {
    return this.exactly(1);
});

utils.createChain(Expectation.prototype, "twice", function() {
    return this.exactly(2);
});

utils.createChain(Expectation.prototype, "thrice", function() {
    return this.exactly(3);
});

Expectation.prototype.args = function() {
    var expectedArgs = _.toArray(arguments);

    validate.args.callability(this);
    validate.args.args(expectedArgs); // Validate arguments for method 'args'

    this.expectedArgs = expectedArgs.map(function(arg) {
        return _.contains([any, anyOrNone, anyOrNoneToEnd], arg) ? arg : deepcopy(arg);
    });

    this.expectedArgsIsSet = true;
    return this;
};

Expectation.prototype.argsWhich = function(predicate, context) {
    validate.argsWhich.callability(this);
    validate.argsWhich.args(predicate);

    this.expectedArgs = predicate;
    if (arguments.length > 1) {
        this.predicateContext = context;
    }

    this.expectedArgsIsSet = true;
    return this;
};

Expectation.prototype.in = function() {
    var args = _.toArray(arguments);

    validate.in.callability(this);
    validate.in.args(args);

    args.forEach(function(sequence) {
        sequence.extend(this, 1, this.expectedMinCallCount);
    }, this);

    return this;
};

Expectation.prototype.toPromise = function(calls) {
    validate.toPromise.callability(this);
    validate.toPromise.args(calls);

    if (Array.isArray(calls)) {
        return calls.map(function(call) {
            return utils.callToPromise(this, call);
        }, this);
    }

    return utils.callToPromise(this, (calls === undefined) ? this.expectedMinCallCount : calls);
};

Expectation.prototype.take = function(beginCall, endCall) {
    validate.take.callability(this);
    validate.take.args(this, beginCall, endCall);

    return new ExpectationTake(this, beginCall, (endCall === undefined) ? beginCall : endCall);
};

function ExpectationTake(expectation, beginCall, endCall) {
    this.expectation = expectation;
    this.beginCall = beginCall;
    this.endCall = endCall;
}

ExpectationTake.prototype.in = function() {
    var args = _.toArray(arguments);

    validate.in.callability(this.expectation);
    validate.in.args(args);

    args.forEach(function(sequence) {
        sequence.extend(this.expectation, this.beginCall, this.endCall);
    }, this);

    return this;
};

ExpectationTake.prototype.giveBack = function() {
    return this.expectation;
};

module.exports = Expectation;
