"use strict";

var _ = require("underscore");

var Promise = require("bluebird");
var chalk = require("chalk");

var state = require("./state");
var constants = require("./constants");
var defaultComparator = require("./defaultComparator");

var any = constants.any;
var anyOrNone = constants.anyOrNone;
var anyOrNoneToEnd = constants.anyOrNoneToEnd;

module.exports.isInt = function(number) {
    return number === parseInt(number);
};

module.exports.getMinArgsCount = function(expectedArgs) {
    return _.findLastIndex(expectedArgs, function(arg) {
        return (arg !== anyOrNone && arg !== anyOrNoneToEnd);
    }) + 1;
};

module.exports.getMaxArgsCount = function(expectedArgs) {
    if (_.last(expectedArgs) === anyOrNoneToEnd) {
        return Infinity;
    }
    return expectedArgs.length;
};

module.exports.findWrongArgIndex = function(expectation, actualArgs, expectedArgs) {
    var comparator = module.exports.getComparator(expectation);

    return _.findIndex(actualArgs, function(actualArg, index) {
        var expectedArg = expectedArgs[index];
        if (index >= expectedArgs.length || _.contains([any, anyOrNone, anyOrNoneToEnd], expectedArg)) {
            return false;
        }

        var result = comparator.fn.call(comparator.context, actualArg, expectedArg, index);
        if (typeof result !== "boolean") {
            throw new Error("Comparator returned " + typeof result + " " + result + " instead of boolean.");
        }
        return !result;
    });
};

module.exports.callToPromise = function(expectation, call) {
    return new Promise(function(resolve) {
        if (Array.isArray(expectation.resolves[call]) === false) {
            expectation.resolves[call] = [];
        }
        expectation.resolves[call].push(resolve);
    });
};

module.exports.createChain = function(prototype, chainName, callback) {
    prototype["_" + chainName] = callback;

    Object.defineProperty(prototype, chainName, {
        enumerable: true,
        get: callback
    });
};

module.exports.getComparator = function(expectation) {
    var fromThisTake = _.find([expectation, expectation.mock], function(obj) {
        return obj.comparatorFunction !== undefined;
    });

    if (fromThisTake !== undefined) {
        return {
            fn: fromThisTake.comparatorFunction,
            context: fromThisTake.comparatorContextIsSet ? fromThisTake.comparatorContext : expectation
        };
    }

    return {
        fn: defaultComparator,
        context: expectation
    };
};

module.exports.getReturnValue = function(expectation) {
    if (expectation.returnValueIsSet) {
        return expectation.returnValue;
    }
    else if (expectation.mock.returnValueIsSet) {
        return expectation.mock.returnValue;
    }
    return expectation.mock.defaultReturnValue;
};

module.exports.isReturnValueSet = function(expectation) {
    return expectation.returnValueIsSet || expectation.mock.returnValueIsSet || expectation.mock.defaultReturnValueIsSet;
};

module.exports.getCallback = function(expectation, originalContext) {
    if (expectation.callbackFunction !== undefined) {
        return {
            fn: expectation.callbackFunction,
            context: expectation.callbackContextIsSet ? expectation.callbackContext : originalContext
        };
    }
    else if (expectation.mock.callbackFunction !== undefined) {
        return {
            fn: expectation.mock.callbackFunction,
            context: expectation.mock.callbackContextIsSet ? expectation.mock.callbackContext : originalContext
        };
    }
    else if (expectation.mock.defaultCallbackFunction !== undefined) {
        return {
            fn: expectation.mock.defaultCallbackFunction,
            context: expectation.mock.defaultCallbackContextIsSet ? expectation.mock.defaultCallbackContext : originalContext
        };
    }

    return undefined;
};

module.exports.getVerbalMockName = function(mock) {
    return (mock.mockedMethodName === undefined) ? "anonymous mock function" : chalk.cyan(mock.mockedMethodName);
};

module.exports.getUnfulfilledExpectations = function(mock) {
    return mock.expectations.filter(function(expectation) {
        return expectation.expectedCallCountIsSet
            && expectation.actualCallCount < expectation.expectedMinCallCount;
    });
};

module.exports.getFulfilledExpectations = function(mock) {
    return mock.expectations.filter(function(expectation) {
        return expectation.expectedCallCountIsSet
            && expectation.actualCallCount >= expectation.expectedMinCallCount;
    });
};

module.exports.getRepletedExpectations = function(mock) {
    return mock.expectations.filter(function(expectation) {
        return expectation.expectedCallCountIsSet
            && expectation.actualCallCount === expectation.expectedMaxCallCount;
    });
};

module.exports.getFulfilledButNotRepletedExpectations = function(mock) {
    return mock.expectations.filter(function(expectation) {
        return expectation.expectedCallCountIsSet
            && expectation.actualCallCount >= expectation.expectedMinCallCount
            && expectation.actualCallCount < expectation.expectedMaxCallCount;
    });
};

module.exports.getUnspecifiedExpectations = function(mock) {
    return mock.expectations.filter(function(expectation) {
        return expectation.expectedCallCountIsSet === false;
    });
};

module.exports.getMock = function(base) {
    return (base.mock !== undefined) ? base.mock : base;
};

module.exports.isMock = function(base) {
    return (base.mock === undefined);
};

module.exports.addToReset = function(mock) {
    if (mock.originalMethod === undefined) { // check if it's a NonMethodMock
        state.registeredMocksToReset.add(mock);
    }
};
