#!/bin/sh

# Sample script that checks if HWID is supported by installation package
#
# Returns: 0 on succes, any other value on failure


#these seven variables are modified in build time to point to valid files form installation package
BL_IMG="u-boot-64_0_0_0.bin"
FIT_IMG="rfsw-image-nahka-20161201.153842.7715041_multi.itb"
WD_PERIOD="0x3FFFFFFE"
SQUASHFS_IMG="rfsw-image-nahka-20161201.153842.7715041.rootfs.squashfs-xz"
ITS_IMG="fit-multi-image.its"
DTB_IMG="zImage-nahka_HWID_CODE.dtb"
ZIMAGE_IMG="zImage"

CMD=$1
HWID=$2
WDIR=$3
PKG=$4

#error codes
OK=0
NOT_SUPPORTED=1
SYNTAX_ERROR=2
INTERNAL_ERROR=3
HWID_NOT_SUPPORTED=4
FILE_GENERATION_FAILED=5
NO_WORKING_DIR=6

LOGF="${WDIR}/swman_log.log"
HWID_FILE_PATH="${WDIR}/hwids.txt"
SELECTIVE_HWID_FILE="selective_hwids.txt"
VARIANT_SPECIFIC_DIR_NAME="variant_specific"
IMAGE_FILES_DIR_NAME="image_files"
TOOLS_DIR_NAME="tools"

BL_IMG_PATH=${WDIR}/${BL_IMG}
FIT_IMG_PATH=${WDIR}/${FIT_IMG}
GEN=0
VAR=0
REV=0
EXP=0
SELECTED_HWID="0_0_0_0"
DEFAULT_PATH=$PATH
TOOLS_PATH="${WDIR}/${TOOLS_DIR_NAME}:$PATH"

extract_hwid() {
    HWID=$1
    # extract HWID values from the parameter
    GEN=$(echo "${HWID}" | cut -d _ -f 1)
    VAR=$(echo "${HWID}" | cut -d _ -f 2)
    REV=$(echo "${HWID}" | cut -d _ -f 3)
    EXP=$(echo "${HWID}" | cut -d _ -f 4)
}

extract_rev_of_dtb() {
    # explicit of the board <= revision of SWI  <= revision of the board
    # get all revisions supported by the package
    MY_REVS=$(grep -E "^${GEN}_${VAR}_" "${HWID_FILE_PATH}" | cut -d _ -f 3 | sort -nr)

    # check if there is a revision satisfying the above condition
    for rev in $MY_REVS
    do
        if [ "$rev" -le "$REV" ] && [ "$rev" -ge "$EXP" ]
        then
            REV_OF_DTB=$rev
            return
        fi
    done
}

generate_hex_hwid() {
    GEN_0=$(printf '%x' $((GEN / 0x100)))
    GEN_1=$(printf '%x' $((GEN & 0xff)))
    VAR_0=$(printf '%x' $((VAR / 0x100)))
    VAR_1=$(printf '%x' $((VAR & 0xff)))
    REV_0=$(printf '%x' $((REV / 0x100)))
    REV_1=$(printf '%x' $((REV & 0xff)))
    EXP_0=$(printf '%x' $((EXP / 0x100)))
    EXP_1=$(printf '%x' $((EXP & 0xff)))
    #REV_OF_DTB_0=$(printf '%x' $((REV_OF_DTB / 0x100)))
    #REV_OF_DTB_1=$(printf '%x' $((REV_OF_DTB & 0xff)))
}

generate_hex_wd() {
    WD=${WD_PERIOD:-"0"}
    WD_3=$(printf '%x' $((WD & 0xff)))
    WD_2=$(printf '%x' $(((WD / 0x100) & 0xff)))
    WD_1=$(printf '%x' $(((WD / 0x10000) & 0xff)))
    WD_0=$(printf '%x' $(((WD / 0x1000000) & 0xff)))
}

# add HWID check here
hwid() {
    HWID=$1
    # validate parameter
    # check if HWID has a right syntax
    if ! echo "$HWID" | grep -Eq  '^[0-9]+_[0-9]+_[0-9]+_[0-9]+$'
    then
        echo "$HWID"
        echo "Invalid HWID -- format is GEN_VAR_REV_EXP -- all decimals"
        return ${SYNTAX_ERROR}
    fi

    extract_hwid "$HWID"

    # check if hwids.txt file exists
    if [ ! -f "${HWID_FILE_PATH}" ]
    then
        echo "Installer error: incomplete installation package: missing hwids.txt"
        return ${INTERNAL_ERROR}
    fi

    # check generation and variant
    if ! grep -Eq "^${GEN}_${VAR}_" "${HWID_FILE_PATH}"
    then
        echo "Installer: incompatible HWID generation and/or variant"
        echo "Supported HWIDs:"
        cat "${HWID_FILE_PATH}"
        return "${HWID_NOT_SUPPORTED}"
    fi

    extract_rev_of_dtb
    if [ -n "${REV_OF_DTB}" ]
    then
        echo "${GEN}_${VAR}_${REV_OF_DTB}"
        return ${OK}
    fi
    # no valid hwid found
    echo "Installer error: incompatible installation package: HWID passed: ${HWID}"
    echo "Supported HWIDs:"
    cat "${HWID_FILE_PATH}"
    return ${HWID_NOT_SUPPORTED}
}

boot() {
    if [ -f "$BL_IMG_PATH" ]; then
        echo "$BL_IMG_PATH"
        return ${OK}
    fi
    if tar -C "${WDIR}" -xf "${PKG}" "${BL_IMG}.xz" && xz --decompress "${BL_IMG_PATH}.xz"
    then
        echo "${BL_IMG_PATH}"
        return ${OK}
    fi
    if tar -C "${WORKING_DIRECTORY}" -xf "${PKG}" "${BL_IMG}.lzma" && xz --decompress "${BL_IMG_PATH}.lzma"
    then
        echo "${BL_IMG_PATH}"
        return ${OK}
    fi

    return ${FILE_GENERATION_FAILED}
}

set_selected_hwid() {
    if ! hwid "${HWID}" >> "${LOGF}"; then
        return $?
    fi
    SELECTED_HWID=$(grep "${GEN}_${VAR}_${REV_OF_DTB}_" "${HWID_FILE_PATH}")
    if ! echo "$SELECTED_HWID" | grep -Eq  '^[0-9]+_[0-9]+_[0-9]+_[0-9]+$'
    then
        return 1
    fi
}

switch_path_to_default() {
    PATH="${DEFAULT_PATH}"
}

switch_path_to_tools() {
    PATH="${TOOLS_PATH}"
}

prepare_variant_specific_files() {
    if tar -C "${WDIR}" -xf "${PKG}" "${SELECTIVE_HWID_FILE}"; then
        SELECTIVE_HWID=$(grep "${SELECTED_HWID}" "${WDIR}/${SELECTIVE_HWID_FILE}")
        echo "$SELECTIVE_HWID" | tr " " "\n" | while read -r variant_specific_file
        do
            if [ "${SELECTED_HWID}" = "${variant_specific_file}" ]; then
                continue
            fi
            if ! tar -C "${WDIR}" -xf "${PKG}" "${VARIANT_SPECIFIC_DIR_NAME}/${variant_specific_file}"; then
                return 1
            fi
        done
    else
        return 1
    fi

    return 0
}

extract_image_files() {
    DTB_FILE_XZ="$(echo $DTB_IMG | sed "s/HWID_CODE/${SELECTED_HWID}/g").xz"
    ZIMAGE_FILE_XZ="${ZIMAGE_IMG}.xz"
    if ! tar -C "${WDIR}" -xf "${PKG}" "${IMAGE_FILES_DIR_NAME}/${DTB_FILE_XZ}"; then
        return 1
    fi
    if ! unxz "${WDIR}/${IMAGE_FILES_DIR_NAME}/${DTB_FILE_XZ}"; then
        return 1
    fi

    if ! tar -C "${WDIR}" -xf "${PKG}" "${IMAGE_FILES_DIR_NAME}/${ZIMAGE_FILE_XZ}"; then
        return 1
    fi
    if ! unxz "${WDIR}/${IMAGE_FILES_DIR_NAME}/${ZIMAGE_FILE_XZ}"; then
        return 1
    fi

    if ! tar -C "${WDIR}" -xf "${PKG}" "${IMAGE_FILES_DIR_NAME}/${SQUASHFS_IMG}"; then
        return 1
    fi

    if ! tar -C "${WDIR}" -xf "${PKG}" "${IMAGE_FILES_DIR_NAME}/${ITS_IMG}"; then
        return 1
    fi

    return 0
}

generate_ramdisk() {
    IMAGE_SQUASH_FS="${WDIR}/${IMAGE_FILES_DIR_NAME}/${SQUASHFS_IMG}"
    if [ -d "${WDIR}/${VARIANT_SPECIFIC_DIR_NAME}" ]; then
        if [ -e "${IMAGE_SQUASH_FS}" ]; then
            switch_path_to_tools
            if ! mksquashfs "${WDIR}/${VARIANT_SPECIFIC_DIR_NAME}" "${IMAGE_SQUASH_FS}" -keep-as-directory -no-recovery >> "${LOGF}"; then
                switch_path_to_default
                if ! mksquashfs "${WDIR}/${VARIANT_SPECIFIC_DIR_NAME}" "${IMAGE_SQUASH_FS}" -keep-as-directory -no-recovery >> "${LOGF}"; then
                    return 1
                else
                    return 0
                fi
            else
                switch_path_to_default
                return 0
            fi
        else
            return 1
        fi
    fi
}

remove_unused_files() {
    rm -rf "${WDIR:?}/${VARIANT_SPECIFIC_DIR_NAME}"
    rm -f "${PKG}"
    return 0
}

unpack_tools() {
    if ! tar -C "${WDIR}" -xf "${PKG}" "${TOOLS_DIR_NAME}"; then
        echo "No tools in installation package" >> "${LOGF}"
        return 0
    fi

    find "${WDIR}/${TOOLS_DIR_NAME}" -name "*.xz" | while read -r tool_binary_xz
    do
        unxz "$tool_binary_xz"
    done

    return 0
}

generate_fit() {
    ITS_FILE_PATH="${WDIR}/${IMAGE_FILES_DIR_NAME}/$ITS_IMG"
    if [ -e "${ITS_FILE_PATH}" ]; then
        if ! sed -i "s/HWID_CODE/${SELECTED_HWID}/g" "${ITS_FILE_PATH}"; then
            return 1
        fi
        switch_path_to_tools
        if mkimage -D "-I dts -O dtb -p 4096" -v -f "${ITS_FILE_PATH}" "${FIT_IMG_PATH}" >> "${LOGF}"; then
            switch_path_to_default
            return 0
        else
            switch_path_to_default
            if mkimage -D "-I dts -O dtb -p 4096" -v -f "${ITS_FILE_PATH}" "${FIT_IMG_PATH}" >> "${LOGF}"; then
                return 0
            else
                return 1
            fi
        fi
    fi
}

fit() {
    if [ -f "$FIT_IMG_PATH" ]; then
    echo "$FIT_IMG_PATH"
        exit 0
    fi
    if (set_selected_hwid && unpack_tools && prepare_variant_specific_files && extract_image_files && generate_ramdisk && boot >> "${LOGF}" && remove_unused_files && generate_fit); then
        echo "${FIT_IMG_PATH}"
        exit ${OK}
    fi
    if tar -C "${WDIR}" -xf "${PKG}" "${FIT_IMG}.xz" && xz --decompress "${FIT_IMG_PATH}.xz"
    then
        echo "${FIT_IMG_PATH}"
        exit ${OK}
    fi
    if tar -C "${WDIR}" -xf "${PKG}" "${FIT_IMG}.lzma" && xz --decompress "${FIT_IMG_PATH}.lzma"
    then
        echo "${FIT_IMG_PATH}"
        exit ${OK}
    fi
    exit ${FILE_GENERATION_FAILED}
}

opaque() {
    HWID=$1
    WDIR=$2
    extract_hwid "$HWID"
    OPAQUE_FILE="${WDIR}/opaque-${HWID}"
    dd of="${OPAQUE_FILE}" if=/dev/zero count=32 bs=1
    #set opaque ID
    printf '\x1' | dd of="${OPAQUE_FILE}" bs=1 seek=0
    extract_rev_of_dtb
    generate_hex_hwid
    BIG_HWID="\x${GEN_1}\x${GEN_0}\x${VAR_1}\x${VAR_0}\x${REV_1}\x${REV_0}\x${EXP_1}\x${EXP_0}"
    #SMALL_HWID="\x0\x0\x${VAR_1}\x${VAR_0}\x${REV_OF_DTB_1}\x${REV_OF_DTB_0}"
    printf "%b" "$BIG_HWID" | dd of="${OPAQUE_FILE}" bs=1 seek=4
    generate_hex_wd
    #watchdog
    WD_FRM="\x${WD_3}\x${WD_2}\x${WD_1}\x${WD_0}"
    printf "%b" "$WD_FRM" | dd of="${OPAQUE_FILE}" bs=1 seek=16
    if [ -f "$OPAQUE_FILE" ]; then
        echo "$OPAQUE_FILE"
        exit 0
    fi
    exit ${FILE_GENERATION_FAILED}
}

cleanup() {
    rm -f "${HWID_FILE_PATH}"
    rm -f "${BL_IMG_PATH}"
    rm -f "${FIT_IMG_PATH}"
    rm -f "${BL_IMG_PATH}.xz"
    rm -f "${FIT_IMG_PATH}.xz"
    rm -f "${BL_IMG_PATH}.lzma"
    rm -f "${FIT_IMG_PATH}.lzma"
    rm -f "${WDIR}/${SELECTIVE_HWID_FILE}"
    rm -rf "${WDIR:?}/${VARIANT_SPECIFIC_DIR_NAME}"
    rm -rf "${WDIR:?}/${IMAGE_FILES_DIR_NAME}"
    rm -rf "${WDIR:?}/${TOOLS_DIR_NAME}"
    rm -f "${LOGF}"
    exit ${OK}
}

PARAMS=$#

# print help
if [ "$PARAMS" -eq 0 ]
    then
    echo "Usage: rfsw-installer.sh ACTION HWID WORKING_DIRECTORY [INSTALLATION_PACKAGE] [...]"
    echo "Perform installation package actions."
    echo
    echo "Mandatory ACTIONs:"
    echo "  -h, --is-hwid-supported              check if HWID is supported and return GENERATION_VARIANT_REVISION string"
    echo "  -b, --generate-bootloader            generate bootloader image for HWID"
    echo "  -f, --generate-fit                   generate FIT for HWID"
    echo "  -c, --clean-up                       remove all temporary files; the installer is not removed"
    echo "  -o, --opaque                         generate opaque buffer file"
    echo
    echo "HWID                               HWID of radio module in format GENERATION_VARIANT_REVISION_EXPLICIT in decimal"
    echo "WORKING_DIRECTORY                  absolute path to the directory where the installation package is extracted"
    echo "INSTALLATION_PACKAGE               absolute path to the installation package file"
    echo "...                                ignored params"
    echo
    echo "Returns 0 on success. Return value not equal to 0 means failure."
    exit ${OK}
fi
if [ "$PARAMS" -lt 4 ]
    then
    echo ERROR: invalid number of parameters.
    exit ${SYNTAX_ERROR}
fi

# check if directory
if [ ! -d "$WDIR" ] ; then
    echo ERROR: WORKING_DIRECTORY does not exist
    exit ${NO_WORKING_DIR}
fi

#extract hwids.txt if needed
if [ "$PARAMS" -eq 4 ]
then
    if [ ! -e "${WDIR}/hwids.txt" ]
    then
        if ! tar -C "${WDIR}" -xf "${PKG}" hwids.txt
        then
            exit ${INTERNAL_ERROR}
        fi
    fi
fi

case "$CMD" in
"--is-hwid-supported" | "-h")
hwid "$HWID"
exit $?
;;
"--generate-bootloader" | "-b")
boot "$HWID" "$WDIR"
exit $?
;;
"--generate-fit" | "-f")
fit "$HWID" "$WDIR"
exit $?
;;
"--opaque" | "-o")
opaque "$HWID" "$WDIR"
exit $?
;;
"--clean-up" | "-c")
cleanup "$HWID" "$WDIR"
exit $?
;;
*)
echo ERROR: invalid parameter
exit ${NOT_SUPPORTED}
esac

exit ${OK}

