# -*- coding: utf-8 -*-
# Copyright 2016 Nokia Solutions and Networks. All rights reserved


from google.protobuf import message
import log
import signal
import threading
import subprocess
import rpc_request_pb2
import rpc_response_pb2
import rpc_message_pb2
import psm_rpc_mesasge_pb2
import psm_rpc_response_pb2
import interface_version_pb2


class CommandHandler():

    def __init__(self, psm_socket):
        self.psm_socket = psm_socket

    def handle(self, msg):
        log.logger.info("CommandHandler: handle")
        received_msg = psm_rpc_mesasge_pb2.LoaderRPC()
        try:
            received_msg.ParseFromString(msg)
        except message.DecodeError:
            log.logger.info("CommandHandler: decode failure, it's not a PSM mesasge")
            return False

        log.logger.info("Received psm_rpc_message from PSM:\n%s" % received_msg)
        if received_msg.HasField("request"):
            log.logger.info("CommandHandler: handle cmd from PSM")
            if received_msg.request.service_name == "TerminateLoader":
                log.logger.info("Cmd: terminate loader")
                signal.alarm(1)
                return True
            self.run(received_msg.request)
            return True

        log.logger.info("This PSM message can not be handled by CommandHandler, try PsmMessageHandler")
        return False

    def command_executor(self, command_request):
        command_id = command_request.transaction_id
        command_name = command_request.service_name
        p = subprocess.Popen(command_request.request,
                             shell=True, stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE)

        stdoutput, erroutput = p.communicate()
        response = psm_rpc_mesasge_pb2.LoaderRPC()
        execution_result = psm_rpc_response_pb2.psm_rpc_response()
        execution_result.transaction_id = command_id
        execution_result.service_name = command_name
        log.logger.info("Cmd result:")
        log.logger.info("StdOutput:\n%s" % stdoutput)
        log.logger.info("ErrOutput:\n%s" % erroutput)
        if not erroutput.strip():
            execution_result.rpc_response = "OK"
        elif "ChecksumFail" in erroutput:
            execution_result.rpc_response = "ChecksumFail"
        else:
            execution_result.rpc_response = "NOK"
        response.response.MergeFrom(execution_result)
        log.logger.info("Cmd response:\n%s" % response)
        self.psm_socket.send(response.SerializeToString())

    def run(self, request):
        server_thread = threading.Thread(
            target=self.command_executor,
            args=(request,),
            name="commandHandler")
        server_thread.setDaemon(True)
        server_thread.start()


class PsmMessageHandler():
    def __init__(self, esm_socket):
        self.esm_socket = esm_socket
        self.isinitialized = False

    def handle(self, msg):
        log.logger.info("PsmMessageHandler: handle")
        received_msg = psm_rpc_mesasge_pb2.LoaderRPC()
        try:
            received_msg.ParseFromString(msg)
        except message.DecodeError:
            log.logger.info("PsmMessageHandler: message decode failure, it is not a psm message")
            return False

        log.logger.info("Received psm_rpc_message from psm:\n%s" % received_msg)
        if received_msg.HasField("rpcMessage"):
            log.logger.info("PsmMessageHandler: it is rpc message, just forward it to ESM")
            try:
                self.esm_socket.send(received_msg.rpcMessage.SerializeToString())
                return True
            except:
                log.logger.error("Forward msg to ESM failure, there is something wrong!!!")
                return False

        log.logger.info("This PSM message can not be handled by PsmMessageHandler, try CommandHandler")
        return False


class EsmMessageHandler():
    def __init__(self, psm_socket, esm_socket):
        self.psm_socket = psm_socket
        self.esm_socket = esm_socket
        self.isinitialized = False

    def get_next_message_id(self, last_message_id):
        self.message_id = last_message_id + 1
        return self.message_id

    def initialize(self):
        first_message_to_esm = rpc_message_pb2.RPCMessage()
        rpc_request_message = rpc_request_pb2.RPCRequest()
        rpc_request_message.id = "257"
        rpc_request_message.method = 1
        supported_interface_version = interface_version_pb2.InterfaceVersion()
        supported_interface_version.majorVersion = 1
        supported_interface_version.minorVersion = 0
        rpc_request_message.niv_params.supportedVersions.extend(
            [supported_interface_version])
        first_message_to_esm.request.MergeFrom(rpc_request_message)
        log.logger.info("niv msg:\n%s" % first_message_to_esm)
        self.esm_socket.send(first_message_to_esm.SerializeToString())
        log.logger.info("EsmMessageHandler: niv_params is sent with id %s" % rpc_request_message.id)

    def is_interface_version_negotiateon_ok(self, supported_version):
        # TODO(jiemo):waiting for version verification.
        return True

    def get_psm_ipaddress(self, message_id):
        message_to_get_psm_ipaddress = rpc_message_pb2.RPCMessage()
        rpc_request_message = rpc_request_pb2.RPCRequest()
        rpc_request_message.id = "258"
        rpc_request_message.method = 2
        message_to_get_psm_ipaddress.request.MergeFrom(rpc_request_message)
        log.logger.info("gmmi msg:\n%s" % message_to_get_psm_ipaddress)
        self.esm_socket.send(message_to_get_psm_ipaddress.SerializeToString())
        log.logger.info("EsmMessageHandler: gmmi_data is sent with id %s" % rpc_request_message.id)

    def handle_init_message(self, received_msg):
        # check if received msg has response or request(req msg is keep alive sent from ESM)
        if not (received_msg.HasField("response") or received_msg.HasField("request")):
            log.logger.error("EsmMessageHandler: unexpected ESM message")
            return False

        if received_msg.HasField("request"):
            # method == 8 means keep alive msg
            if received_msg.request.method == 8:
                log.logger.debug("Receive keep alive from ESM before connecting to PSM")
                self.send_keep_alive_ack_to_esm(received_msg.request.id)

                # Workaround: get IP address after keepalive.
                self.get_psm_ipaddress(received_msg.request.id)

                return True
            else:
                log.logger.error("Unexpected ESM msg, request from ESM should be keep alive.")
                return False
        else:
            response = received_msg.response
            if response.status != 0:
                log.logger.error("EsmMessageHandler: message status has error")
                return False

            if response.HasField("niv_data"):
                log.logger.info("EsmMessageHandler: receive response for niv_data message")
                selected_version = response.niv_data.selectedVersion
                if self.is_interface_version_negotiateon_ok(selected_version):
                    # Workaround
                    # self.get_psm_ipaddress(response.id)
                    return True
                else:
                    log.logger.error("EsmMessageHandler: Interface negotiation failed")
                    return False

            elif response.HasField("gmmi_data"):
                log.logger.info("EsmMessageHandler: receive response for gmmi_data message")
                self.remote_ipaddress = response.gmmi_data.remoteIpAddress.encode('utf-8')
                self.local_ipaddress = response.gmmi_data.localIpAddress.encode('utf-8')
                log.logger.info("EsmMessageHandler: remote_ipaddress is %s" % self.remote_ipaddress)
                log.logger.info("EsmMessageHandler: local_ipaddress is %s" % self.local_ipaddress)
                self.psm_socket.start((self.remote_ipaddress, 2050), "psmSocket")
                self.isinitialized = True
                return True
            else:
                log.logger.error("EsmMessageHandler: Unexpected message during ESM initialization")
                return False

    def send_keep_alive_ack_to_esm(self, ack_id):
        keep_alive_ack = rpc_message_pb2.RPCMessage()
        rpc_response_message = rpc_response_pb2.RPCResponse()
        rpc_response_message.id = ack_id
        rpc_response_message.status = 0
        keep_alive_ack.response.MergeFrom(rpc_response_message)
        log.logger.info("keep alive ack:\n%s" % keep_alive_ack)
        self.esm_socket.send(keep_alive_ack.SerializeToString())
        log.logger.info("Keep alive ack is sent to ESM before socket created with PSM, ack_id = %s" % ack_id)

    def forward_rpcmessage_to_psm(self, message):
        # ESM message is sent to PSM directly without PSM message header.
        # message_to_psm = psm_rpc_mesasge_pb2.LoaderRPC()
        # message_to_psm.rpcMessage.MergeFrom(message)
        try:
            self.psm_socket.send(message.SerializeToString())
            return True
        except:
            log.logger.error("Failed to forward message to PSM, there is something wrong!!!")
            return False

    def handle_rpc_messgage(self, message):
        if not self.isinitialized:
            log.logger.info("EsmMessageHandler: handle_init_message")
            return self.handle_init_message(message)
        else:
            log.logger.info("EsmMessageHandler: forward rpc_message to PSM")
            return self.forward_rpcmessage_to_psm(message)

    def handle(self, messagestring):
        log.logger.info("EsmMessageHandler: handle")
        received_msg = rpc_message_pb2.RPCMessage()
        try:
            received_msg.ParseFromString(messagestring)
        except message.DecodeError:
            # After socket with PSM is created, loader can receive msg from PSM.
            # Msg will be handled by EsmMessageHandler first, but msg should be parsed failed.
            # Then next handler 'PsmMessageHandler' will be used to handle it. So following error log may come from here.
            log.logger.info("EsmMessageHandler: message decode failure, it is not a ESM message, try PSM handler")
            return False

        log.logger.info("Received rpc message from ESM:\n%s" % received_msg)

        try:
            # I can't remember why it should return the result of function handle_rpc_messgage.
            # Take care when you want to change this behavior, at least a full test is necessary if you change it.
            return self.handle_rpc_messgage(received_msg)
        except:
            log.logger.error("EsmMessageHandler: exception is detectd while handling ESM message")
            return False
