#!/bin/bash
##########################################################################
# core_pattern
# |/opt/CCS/crashHandler.sh 1011 %e %p %s %t %c <CoreSize> <CorePath> <MultiCoreEnabled> <MultiCoreLockTimeout>
#-------------------------------------------------------------------------
# Input parameters:
readonly NODEID=$1                      # <NodeID>                NodeID on which the crash occurred.
readonly PROCPATH=$2                    # %E                      Pathname of executable,with slashes ('/') replaced by exclamation marks ('!').
readonly PID=$3                         # %p                      PID of dumped process, as seen in the PID namespace in which the process resides.
readonly SIGNALNO=$4                    # %s                      Number of signal causing dump.
readonly TIME=$5                        # %t                      Time of dump, expressed as seconds since the Epoch, 1970-01-01 00:00:00 +0000 (UTC).
readonly SOFTLIMIT=$6                   # %c                      Core file size soft resource limit of crashing process (since Linux 2.6.24).
         CORESIZE=$7                    # <CoreSize>              Maximum size of the compressed core file.
readonly COREPATH=${8}/${NODEID}        # <CorePath>              Path where the corefile and bt-coreinfo will be stored.
readonly MULTICORE_ENABLED=$9           # <MultiCoreEnabled>      Flag to enable multi core dumping; value 0 disables otherwise enables multicore dumping.
readonly MULTICORE_LOCK_TIMEOUT=${10}   # <MultiCoreLockTimeout>  Lock timeout in sec for multi core dumping.
#
# Truncate Variables to specified length:
readonly PROCNAME="$(basename "$(echo "${PROCPATH}" | tr '!' '/')")"
readonly PROCNAME_TRUNC="$(echo -e "${PROCNAME}" | tr -d '[:space:]' | cut -c -16)"
readonly PID_TRUNC="$(echo -e "${PID}" | cut -c -5)"
readonly SIGNALNO_TRUNC="$(echo -e "${SIGNALNO}" | cut -c -3)"

#
# Path and Filename:
readonly CRASHINFO="${NODEID}_${PROCNAME_TRUNC}_${PID_TRUNC}_${SIGNALNO_TRUNC}_${TIME}"
         COREFILE=$COREPATH/${CRASHINFO}.core.xz
readonly BTCOREINFO=$COREPATH/bt-coreinfo.txt
#
# Binary paths
readonly CRASHHANDLER_LEGACY="/usr/bin/crashHandler"
readonly CRASHHANDLER="/usr/bin/crashinfo"
readonly OPT_CORENOTIFIER="/opt/CCS/CoreNotifier"
         CORENOTIFIER=""
#
# Lockfile setting
readonly LOCKPATH=$COREPATH
readonly SCRIPTNAME=$(basename "$0")
readonly LOCK_FD=200
readonly SOFTLIMIT_MAGIC_NUMBER=13
#
#HW type
readonly HW_TYPE=$(cat /etc/mbosArch)

#
# Truncation maximum file size
readonly MAX_BTCOREINFO_SIZE=1048576 # 1MB
#
##########################################################################

coreEnabled() {
    if [ "$SOFTLIMIT" == "$SOFTLIMIT_MAGIC_NUMBER" ]; then
        CORESIZE=0
    fi

    if [ "$CORESIZE" == "0" ]; then
        COREFILE=/dev/null
        ISCORE_ENABLED=0
    else
        ISCORE_ENABLED=1
    fi
}

truncate() {
    if [ -e "$BTCOREINFO" ]; then
        BTCOREINFO_SIZE=$(wc -c "$BTCOREINFO" | awk '{ print $1 }')

        if [ "$BTCOREINFO_SIZE" -gt "$MAX_BTCOREINFO_SIZE" ]; then
            dd if=/dev/null of="$BTCOREINFO" seek="$MAX_BTCOREINFO_SIZE" bs=1 count=1
        fi
    fi
}

# does the actual core dumping and generates the crash info file
dumpViaCrashInfo()
{
    local isCoreDumpEnabled=$1
    if [ "$isCoreDumpEnabled" -gt 0 ]; then
        local CRASHINFO_PARAMS="-P ${PID} -o log_info=info -o core_exists=keep -o core_filter=xz -T0 -0 -o core_filter=head -c $((CORESIZE*1024*1024)) \
        -o core_filter=openssl smime -encrypt -aes256 -outform DER /etc/keys.d/certs/coreencrcert.pem -o core_output=${COREFILE} -o info_output=${BTCOREINFO}"
        logger -t CCS "${FUNCNAME[0]} ${CRASHHANDLER} ${CRASHINFO_PARAMS}"

        ${CRASHHANDLER} -P ${PID} -o "log_info=info" -o "core_exists=keep" -o "core_filter=xz -T0 -0" -o "core_filter=head -c $(($CORESIZE*1024*1024))" \
         -o "core_filter=openssl smime -encrypt -aes256 -binary -outform DER /etc/keys.d/certs/coreencrcert.pem" -o "core_output=${COREFILE}" -o "info_output=${BTCOREINFO}"
    else
        local CRASHINFO_PARAMS="-P ${PID} -o log_info=info -o core_exists=keep -o core_filter=~ -o core_output=~ -o info_output=${BTCOREINFO}"
        logger -t CCS "${FUNCNAME[0]} ${CRASHHANDLER} ${CRASHINFO_PARAMS}"

        ${CRASHHANDLER} -P ${PID} -o "log_info=info" -o "core_exists=keep" -o "core_filter=~" -o "core_output=~" -o "info_output=${BTCOREINFO}"
    fi

    _ERR="$?"
    if [ ${_ERR} -ne 0 ]; then
        logger -t CCS "failed to execute ${CRASHHANDLER}, _ERR=${_ERR}"
    fi
}

coreDumping() {
    if [ -e ${CRASHHANDLER} ]; then
        if [ $CORESIZE -ne 0 ]; then
            COREFILE=$COREFILE.p7
        fi
        dumpViaCrashInfo $ISCORE_ENABLED
    elif [ -e ${CRASHHANDLER_LEGACY} ]; then
        if [ $CORESIZE -ne 0 ]; then
            COREFILE=$COREFILE.p7
        fi
        logger -t CCS " ${CRASHHANDLER_LEGACY} -dk -s $CORESIZE -o $COREFILE -l $BTCOREINFO $PROCNAME"
        ${CRASHHANDLER_LEGACY} -dk -s $CORESIZE -o "$COREFILE" -l "$BTCOREINFO" "$PROCNAME"
    else
        if [ $CORESIZE -ne 0 ]; then
            CORESIZE=$(($CORESIZE*1024))
            logger -t CCS "xz -0 -c | dd of=$COREFILE bs=1024 count=$CORESIZE"
            xz -0 -c | dd of=$COREFILE bs=1024 count=$CORESIZE
        else
            cat > /dev/null
        fi
    fi
}

lock() {
    lock_file=$COREPATH/.$SCRIPTNAME.lock

    eval "exec $LOCK_FD>$lock_file"

    if [[ $HW_TYPE == "fzm" && $PROCNAME == "CCSLiteExe" ]]; then
        flock -w "$MULTICORE_LOCK_TIMEOUT" $LOCK_FD
        return 0
    elif [[ $MULTICORE_ENABLED != 0 ]]; then
        flock -w "$MULTICORE_LOCK_TIMEOUT" $LOCK_FD
        return 0
    else
        flock -n $LOCK_FD \
            && return 0 \
            || return 1
    fi
}
notify() {
    if [ "${CORENOTIFIER}" != "" ]; then
        logger -t CCS  "${CORENOTIFIER} -m $1 -e $PROCNAME -p $PID -s $SIGNALNO -t $TIME -d $COREPATH -x $ISCORE_ENABLED -n $NODEID"
        ${CORENOTIFIER} -m "$1" -e "$PROCNAME" -p "$PID" -s "$SIGNALNO" -t "$TIME" -d "$COREPATH" -x "$ISCORE_ENABLED" -n "$NODEID"
    else
        logger -t CCS "Crash handling $1"
    fi
}

configureNotifier() {
    if [ -e ${OPT_CORENOTIFIER} ];then
        CORENOTIFIER=$OPT_CORENOTIFIER
    fi
}

#main function
main() {
    mkdir -p -m 1750 "$COREPATH"
    lock || exit 1
    umask 0037
    logger -t CCS "OWNPID=$$ starting core script NODEID=$NODEID PROCNAME=$PROCNAME SIGNALNO=$SIGNALNO TIME=$TIME SOFTLIMIT=$SOFTLIMIT CORESIZE=$CORESIZE COREPATH=$COREPATH"

    local NUMBER_OF_CORES=0
    NUMBER_OF_CORES=$(find "$COREPATH" -name "*core.xz*" | wc -l )

    if [ "${MULTICORE_ENABLED}" -eq 0 ] && [ "$NUMBER_OF_CORES" -gt 0 ]; then
        CORESIZE=0
        logger -t CCS "OWNPID=$$ core file exists"
    fi;
    configureNotifier
    coreEnabled
    notify "STARTED" & #detach notification to start coreDumping immediately
    coreDumping
    truncate
    notify "COMPLETED"
    rm "$lock_file"
}
# entry point
main
