#!/bin/bash
#
# CCS-RT specific crashHandler.sh for containers
# Based on MCU CCS crashHandler.sh with the following differences:
#   - CoreSize parameter is ignored and is instead hardcoded to unlimited
#   - CorePath parameter is ignored and is instead hardcoded to /tmp
#   - Make process/PID-specific bt-coreinfo.txt files and remove global
#     lock to allow concurrent core/bt-coreinfo.txt generation
#   - Don't remove any core files (MCU HWAPI responsible for cleanup)
##########################################################################
# core_pattern
# |/opt/CCS/crashHandler.sh 1011 %e %p %s %t %c <CoreSize> <CorePath>
#-------------------------------------------------------------------------
# Input parameters:
readonly NODEID=$1          # <NodeID>      NodeID on which the crash occured
readonly PROCNAME=$2        # %e            executable filename (without path prefix)
readonly PID=$3             # %p            PID of dumped process, as seen in the PID namespace in which the process resides
readonly SIGNALNO=$4        # %s            number of signal causing dump
readonly TIME=$5            # %t            time of dump, expressed as seconds since the Epoch, 1970-01-01 00:00:00 +0000 (UTC)
readonly SOFTLIMIT=$6       # %c            core file size soft resource limit of crashing process (since Linux 2.6.24)
         CORESIZE=200       # 200MB         Maximum size of the compressed core file (ignore)
readonly COREPATH=/tmp      #               Path where the corefile and bt-coreinfo will be stored

#
# Truncate Variables to specified length:
readonly PROCNAME_TRUNC="$(tr -d '[:space:]' <<< "$PROCNAME" | cut -c -16)"
readonly PID_TRUNC=${PID::5}
readonly SIGNALNO_TRUNC=${SIGNALNO::3}

#
# Path and Filename:
readonly CRASHINFO="${NODEID}_${PROCNAME_TRUNC}_${PID_TRUNC}_${SIGNALNO_TRUNC}_${TIME}"
         COREFILE=$COREPATH/${CRASHINFO}.core.xz
readonly BTCOREINFO="$COREPATH/${CRASHINFO}.bt-coreinfo.txt"
readonly INFOFILE=$COREPATH/.crashInfo
#
# Binary paths
readonly CRASHHANDLER=/usr/bin/crashHandler
readonly CORENOTIFIER=/opt/CCS/CoreNotifier
#
# Lockfile setting
readonly LOCKPATH=$COREPATH
readonly SCRIPTNAME=$(basename "$0")
readonly LOCK_FD=200
#
# Truncation maximum file size
readonly MAX_BTCOREINFO_SIZE=1048576 # 1MB
#
# Debug settings
DEBUGLOG=/dev/null
##########################################################################

coreExists() {
    [ -e "$1" ] ;
}

coreEnabled() {
    if [ "$SOFTLIMIT" == "0" ]; then
        CORESIZE=0
    fi

    if [ "$CORESIZE" == "0" ]; then
        COREFILE=/dev/null
        ISCORE=0
    else
        ISCORE=1
    fi
}

truncate() {
    if [ -e "$BTCOREINFO" ]; then
        BTCOREINFO_SIZE=$(wc -c "$BTCOREINFO" | awk '{ print $1 }')

        if [ "$BTCOREINFO_SIZE" -gt "$MAX_BTCOREINFO_SIZE" ]; then
            dd if=/dev/null of="$BTCOREINFO" seek="$MAX_BTCOREINFO_SIZE" bs=1 count=1
        fi
    fi
}

coreDumping() {
    if [ -e ${CRASHHANDLER} ];then
        if [ $CORESIZE -ne 0 ]; then
            COREFILE=$COREFILE.p7
        fi
        if [[ -e "/tmp/node_$NODEID" ]]; then
            echo "[Running within $NODEID namespace] ${CRASHHANDLER} -dk -s $CORESIZE -o $COREFILE -l $BTCOREINFO $PROCNAME" >> $DEBUGLOG
            /usr/bin/nsenter --target "$(/bin/machinectl show -p Leader "$NODEID" | cut -d= -f2-)" --mount --uts --ipc --net --pid -- "${CRASHHANDLER}" -dk -s "$CORESIZE" -o "$COREFILE" -l "$BTCOREINFO" "$PROCNAME"
        else
            echo " ${CRASHHANDLER} -dk -s $CORESIZE -o $COREFILE -l $BTCOREINFO $PROCNAME" >> $DEBUGLOG
            ${CRASHHANDLER} -dk -s $CORESIZE -o "$COREFILE" -l "$BTCOREINFO" "$PROCNAME"
        fi
    else
        if [ $CORESIZE -ne 0 ]; then
            CORESIZE=$((CORESIZE*1024))
            echo "xz -0 -c | dd of=$COREFILE bs=1024 count=$CORESIZE" >> $DEBUGLOG
            xz -0 -c | dd of=$COREFILE bs=1024 count=$CORESIZE
        else
            cat > /dev/null
        fi
    fi
}

lock() {
    lock_file=$COREPATH/.$SCRIPTNAME.lock

    eval "exec $LOCK_FD>$lock_file"

    flock -n $LOCK_FD \
        && return 0 \
        || return 1
}

notify() {
    echo "${CORENOTIFIER} -m $1 -e $PROCNAME -p $PID -s $SIGNALNO -t $TIME -d $COREPATH -x $ISCORE -n $NODEID" >> $DEBUGLOG
    "${CORENOTIFIER}" -m "$1" -e "$PROCNAME" -p "$PID" -s "$SIGNALNO" -t "$TIME" -d "$COREPATH" -x "$ISCORE" -n "$NODEID"
}

#main function
main() {
    mkdir -p "$COREPATH"
    # Note: no lock taken to allow concurrent core/bt-coreinfo generation
    umask 0037
    echo "OWNPID=$$ starting core script NODEID=$NODEID PROCNAME=$PROCNAME SIGNALNO=$SIGNALNO TIME=$TIME SOFTLIMIT=$SOFTLIMIT CORESIZE=$CORESIZE COREPATH=$COREPATH" >> $DEBUGLOG

    # Note: any generated core/bt-coreinfo files do not get removed
    # by this script. 

    coreEnabled
    notify "STARTED" & #detach notification to start coreDumping immediately
    coreDumping
    truncate
    notify "COMPLETED"
    rm "$lock_file"
}
# entry point
main
