<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	RTTHandler.php
//
// Description:	This file handles an asyncronous request from the client for
//		RTT measurements. The request is XML encoded and
//		sent to the WebUI Agent for processing. Any response from the
//		agent is JSON encoded and sent as-is to the client (pass-thru).
//
// ****************************************************************************

include_once "../../phplib/MessageHandler.php";
include_once "../../phplib/XmlMessageParser.php";
include_once "../../phplib/FailureResponse.php";
include_once "../../phplib/messages/RTTRequest.php";
include_once "../../phplib/messages/RTTResponse.php";

openlog( "WebUI CGI: RTT", LOG_ODELAY, LOG_USER );

header("Content-type: application/json; charset=utf-8");
header("Cache-control: no-store");
header("Pragma: no-cache");

// Read session cookie sent by client. Should always be available unless user
// has disabled cookies (would cause unavailability of WebUI service).
if( !isset( $_COOKIE[ "FZAPWebUISession" ] ) )
{
	// Unexpected. User should not have been able to access WebUI page with
	// cookies disabled. Log error and send failure indication.
	syslog( LOG_ERR, "No session cookie" );

	FailureResponse::SendFailureResponse( "CGI - Client cookie not received" );

	closelog();
	exit();
}

// Session cookie received from client. Retrieve user session ID. Needs to be
// sent to the WebUIAgent for validation (with every request).
$sessionId = $_COOKIE[ "FZAPWebUISession" ];


// Check if post data is available and target IP address has been populated.
if( !(isset( $_POST["targetIpAddress"] )) || ($_POST["targetIpAddress"] === "") )
{
	// No post'ed data, or missing IP address. Return failure indication.
	syslog( LOG_ERR, "Missing target IP address" );

	FailureResponse::SendFailureResponse( "CGI - Missing target IP address" );

	closelog();
	exit();
}

$targetIpAddress = $_POST["targetIpAddress"];

if( !(isset( $_POST["csrfToken"] )) || ($_POST["csrfToken"] === "") )
{
	// No post'ed data, or missing IP address. Return failure indication.
	syslog( LOG_ERR, "Missing CSRF Token" );

	FailureResponse::SendFailureResponse( "CGI - Missing CSRF Token" );

	closelog();
	exit();
}

$csrfToken = $_POST["csrfToken"];

if( !(filter_var( $targetIpAddress, FILTER_VALIDATE_IP )))
{
	syslog( LOG_ERR, "Invalid target IP address" );

	FailureResponse::SendFailureResponse( "CGI - Invalid target IP address" );

	closelog();
	exit();
}


$rttRequest = new RTTRequest( $sessionId, $csrfToken );
if( is_null( $rttRequest ) )
{
	syslog( LOG_ERR, "Failed to allocate memory for RTTRequest message" );

	FailureResponse::SendFailureResponse( "CGI - Memory allocation failure" );

	closelog();
	exit();
}

// Set target IP address entered by the user.
$rttRequest->SetIpAddress( $targetIpAddress );


// Send the message. The MessageHandler will wait for and return the response,
// if any.
$response = MessageHandler::SendAndReceive( $rttRequest->XmlMsg() );
if( is_null( $response ) )
{
	syslog( LOG_ERR, "Failed to send RTTRequest message and/or receive response" );

	FailureResponse::SendFailureResponse( "CGI - Message send or receive failure" );

	closelog();
	exit();
}


// Parse the received XML message response.
$msg = XmlMessageParser::ParseMsg( $response );

// Validate response.
if( (is_null( $msg )) || ($msg->OK() !== true) || ($msg->GetMessageType() !== "RTTResponse") )
{
	// Remove line breaks from response string and print to syslog.
	$msgString = str_replace( "\n", null, $response );
	syslog( LOG_ERR, "Invalid or null RTTResponse message: " . $msgString );

	// Internal error (e.g., protocol error between agent and HTTP server).
	FailureResponse::SendFailureResponse( "CGI - Invalid or null RTTResponse" );

	closelog();
	exit();
}


// No need to check success or failure; will be handled by client. Encode response message
// as JSON and send to client.
$jsonString = json_encode( $msg );

// Check encoding result
if( $jsonString === false )
{
	// JSON encoding failed. Received message must have been invalid.
	syslog( LOG_ERR, "JSON encoding failed" );
	FailureResponse::SendFailureResponse( "CGI - JSON encoding failure" );

	closelog();
	exit();
}


// Output encoded JSON string to client.
echo $jsonString;

closelog();

?>

