<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2014, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	SiteIdHandler.php
//
// Description:	This file handles an asyncronous request from the client to
//		configure the auto-connection site ID. The site ID entered by
//		the user is retrieved from post data, and sent to the WebUI
//		Agent for processing.
//
// ****************************************************************************

include_once "../../phplib/MessageHandler.php";
include_once "../../phplib/XmlMessageParser.php";
include_once "../../phplib/messages/ConfigureSiteIdRequest.php";
include_once "../../phplib/messages/ConfigureSiteIdResponse.php";

openlog( "WebUI CGI: ConfigureSiteId", LOG_ODELAY, LOG_USER );

header("Content-type: text/plain; charset=utf-8");
header("Cache-control: no-store");
header("Pragma: no-cache");

// Constants for return values to client. Any positive integer indicates success.
const SUCCESS = 1;
const INVALID_SESSION = -1;
const INVALID_REQUEST = -2;
const INTERNAL_ERROR = -3;
const INVALID_STATE = -4;
const INVALID_INPUT = -5;
const INVALID_TOKEN = -6;


// Read session cookie sent by client. Should always be available unless user
// has disabled cookies.
if( !isset( $_COOKIE[ "FZAPWebUISession" ] ) )
{
	// Unexpected. User should not have been able to access WebUI page with
	// cookies disabled. Log error and send failure indication.
	syslog( LOG_ERR, "No session cookie" );
	echo INVALID_SESSION;

	closelog();
	exit();
}

// Session cookie received from client. Retrieve user session ID. Needs to be
// sent to the WebUIAgent for validation (with every request).
$sessionId = $_COOKIE[ "FZAPWebUISession" ];


// Check if post data is available and site ID and csrfToken have been populated.
if( (isset( $_POST[ "siteId" ] )) && ($_POST[ "siteId" ] !== "") &&
    (isset( $_POST[ "ctkn" ] )) && ($_POST[ "ctkn" ] !== "")  )
{
	$siteId = $_POST[ "siteId" ];
	$csrfToken = $_POST[ "ctkn" ];

	// Validate site ID. Per the PDDB, ASCII characters in the range from
	// 32 (space) to 126 (~) are valid, with up to 50 characters.
	// Match to pattern based on PDDB string validation rule.
	$pattern = "/^([\x21-\x7E]([\x20-\x7E]*[\x21-\x7E])?)?$/";
	$siteIdLength = strlen( $siteId );

	if( (!preg_match( $pattern, $siteId )) || ($siteIdLength < 1) || ($siteIdLength > 50) )
	{
		syslog( LOG_ERR, "Invalid user input" );

		echo INVALID_INPUT;

		closelog();
		exit();
	}

	$configureSiteIdRequest = new configureSiteIdRequest( $sessionId, $csrfToken );
	if( is_null( $configureSiteIdRequest ) )
	{
		syslog( LOG_ERR, "Failed to allocate memory for ConfigureSiteIdRequest message" );

		// Treat as internal failure.
		echo INTERNAL_ERROR;

		closelog();
		exit();
	}

	// Set the new site ID entered by the user.
	$configureSiteIdRequest->SetSiteId( $siteId );

	// Send the message. The MessageHandler will wait for and return the
	// response, if any.
	$response = MessageHandler::SendAndReceive( $configureSiteIdRequest->XmlMsg() );
	if( is_null( $response ) )
	{
		syslog( LOG_ERR, "Failed to send ConfigureSiteIdRequest message and/or receive response" );

		// Treat as internal failure.
		echo INTERNAL_ERROR;

		closelog();
		exit();
	}

	// Parse the received XML message response.
	$msg = XmlMessageParser::ParseMsg( $response );

	// Validate response.
	if( (is_null( $msg )) || ($msg->OK() !== true) || ($msg->GetMessageType() !== "ConfigureSiteIdResponse") )
	{
		// Remove line breaks from response string and print to syslog.
		$msgString = str_replace( "\n", null, $response );
		syslog( LOG_ERR, "Invalid or null ConfigureSiteIdResponse message: " . $msgString );

		// Internal error (e.g., protocol error between agent and HTTP server).
		echo INTERNAL_ERROR;

		closelog();
		exit();
	}


	// Check the result.
	if( $msg->GetStatus() === "Success" )
	{
		// Configuration successful.
		echo SUCCESS;
	}
	else
	{
		// Configuration failed. Check the received failure reason (optional
		// element).
		if( $msg->GetFailureReason() === "invalidSession" )
		{
			// Session has expired. The user will need to login again.
			echo INVALID_SESSION;
		}
		if( $msg->GetFailureReason() === "invalidSecurityToken" )
		{
			// Session has expired. The user will need to login again.
			echo INVALID_TOKEN;
		}
		else if( $msg->GetFailureReason() === "Site is Commissioned" )
		{
			// Can't change site ID once the site has been commissioned.
			echo INVALID_STATE;
		}
		else
		{
			// Other execution error has occurred.
			syslog( LOG_ERR, "Configure site ID failure reason: " . $msg->GetFailureReason() );
			echo INTERNAL_ERROR;
		}
	}
}
else
{
	// No post'ed data, or incorrect format. Return failure indication.
	echo INVALID_REQUEST;
}

closelog();

?>

