<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2016, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	SnapshotRequestHandler.php
//
// Description:	This file handles an asyncronous request from the client to
//		start snapshot collection. The request is XML encoded and sent
//		to the WebUI Agent for processing. Any response from the agent
//		is JSON encoded and sent as-is to the client (pass-thru).
//
// ****************************************************************************

include_once "../../phplib/MessageHandler.php";
include_once "../../phplib/XmlMessageParser.php";
include_once "../../phplib/FailureResponse.php";
include_once "../../phplib/messages/SnapshotRequest.php";
include_once "../../phplib/messages/SnapshotResponse.php";

openlog( "WebUI CGI: Snapshot", LOG_ODELAY, LOG_USER );

header("Content-type: application/json; charset=utf-8");
header("Cache-control: no-store");
header("Pragma: no-cache");

// Read session cookie sent by client. Should always be available unless user
// has disabled cookies (would cause unavailability of WebUI service).
if( !isset( $_COOKIE[ "FZAPWebUISession" ] ) )
{
	// Unexpected. User should not have been able to access WebUI page with
	// cookies disabled. Log error and send failure indication.
	syslog( LOG_ERR, "No session cookie" );

	FailureResponse::SendFailureResponse( "CGI - Client cookie not received" );

	closelog();
	exit();
}

// Session cookie received from client. Retrieve user session ID. Needs to be
// sent to the WebUIAgent for validation (with every request).
$sessionId = $_COOKIE[ "FZAPWebUISession" ];


// Retrieve query string from URL of page request.
$logCoverage = "";
if( isset( $_SERVER["QUERY_STRING"] ) )
{
	$logCoverage = $_SERVER[ "QUERY_STRING" ];
}

// Validate log coverage level.
if( ($logCoverage !== "full") && ($logCoverage !== "minor") )
{
	syslog( LOG_ERR, "Invalid log coverage level: " . $logCoverage );

	FailureResponse::SendFailureResponse( "CGI - Invalid log coverage level" );

	closelog();
	exit();
}


$snapshotRequest = new SnapshotRequest( $sessionId );
if( is_null( $snapshotRequest ) )
{
	syslog( LOG_ERR, "Failed to allocate memory for SnapshotRequest message" );

	FailureResponse::SendFailureResponse( "CGI - Memory allocation failure" );

	closelog();
	exit();
}

$snapshotRequest->SetLogCoverage( $logCoverage );


// Send the message. The MessageHandler will wait for and return the response,
// if any.
$response = MessageHandler::SendAndReceive( $snapshotRequest->XmlMsg() );
if( is_null( $response ) )
{
	syslog( LOG_ERR, "Failed to send SnapshotRequest message and/or receive response" );

	FailureResponse::SendFailureResponse( "CGI - Message send or receive failure" );

	closelog();
	exit();
}


// Parse the received XML message response.
$msg = XmlMessageParser::ParseMsg( $response );

// Validate response.
if( (is_null( $msg )) || ($msg->OK() !== true) || ($msg->GetMessageType() !== "SnapshotResponse") )
{
	// Remove line breaks from response string and print to syslog.
	$msgString = str_replace( "\n", null, $response );
	syslog( LOG_ERR, "Invalid or null SnapshotResponse message: " . $msgString );

	// Internal error (e.g., protocol error between agent and HTTP server).
	FailureResponse::SendFailureResponse( "CGI - Invalid or null SnapshotResponse" );

	closelog();
	exit();
}


// No need to check success or failure; will be handled by client. Encode response message
// as JSON and send to client.
$jsonString = json_encode( $msg );

// Check encoding result
if( $jsonString === false )
{
	// JSON encoding failed. Received message must have been invalid.
	syslog( LOG_ERR, "JSON encoding failed" );
	FailureResponse::SendFailureResponse( "CGI - JSON encoding failure" );

	closelog();
	exit();
}


// Output encoded JSON string to client.
echo $jsonString;

closelog();

?>

