<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2016, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	index.php
//
// Description:	This file is invoked by the HTTP web server when the client
//		requests the WebUI main page. The code first checks if the user
//		has a valid session. If valid, the HTML page is displayed. If
//		not, the user is directed to login. Actual session validation
//		is performed by the WebUIAgent. Communication with the WebUIAgent
//		process is done via the included MessageHandler class.
//
// ****************************************************************************

include_once "../phplib/MessageHandler.php";
include_once "../phplib/XmlMessageParser.php";
include_once "../phplib/messages/ValidateSessionRequest.php";
include_once "../phplib/messages/ValidateSessionResponse.php";

openlog( "WebUI CGI: Index", LOG_ODELAY, LOG_USER );
header("Content-type: text/html; charset=utf-8");
header( "X-Frame-Options: DENY" );

// Page request for index.php. Check if session cookie received from client.
// If not, this is a new access to the page; redirect to login.
if( !isset( $_COOKIE[ "FZAPWebUISession" ] ) )
{
	// User not logged in; redirect to login page.
	header( "Location: login.php" );

	closelog();
	exit();
}


// Session cookie received from client. Retrieve user session ID.
$sessionId = $_COOKIE[ "FZAPWebUISession" ];

// Validate session ID by checking with the WebUIAgent. All session management
// is done by the agent.

$validateSessionRequest = new ValidateSessionRequest( $sessionId);
if( is_null( $validateSessionRequest ) )
{
	syslog( LOG_ERR, "Failed to allocate memory for ValidateSessionRequest message" );

	// Something must be seriously wrong. Try to recover by redirecting
	// to login page with internal error indication.
	header( "Location: login.php?internal_error" );

	closelog();
	exit();
}

// Send message to WebUIAgent. The MessageHandler will wait for, and return,
// the response, if any.
$response = MessageHandler::SendAndReceive( $validateSessionRequest->XmlMsg() );
if( is_null( $response ) )
{
	syslog( LOG_ERR, "Failed to send ValidateSessionRequest message and/or receive response" );

	// Failed to send and/or receive a response. Assume session invalid.
	// Try to recover by redirecting to login.
	header( "Location: login.php?message_error" );

	closelog();
	exit();
}

// Parse the received XML message response.
$msg = XmlMessageParser::ParseMsg( $response );

// Validate response.
if( (is_null( $msg )) || ($msg->OK() !== true) || ($msg->GetMessageType() !== "ValidateSessionResponse") )
{
	// Remove line breaks from response string and print to syslog.
	$msgString = str_replace( "\n", null, $response );
	syslog( LOG_ERR, "Invalid or null ValidateSessionResponse message: " . $msgString );

	// Hmm... Try to recover by redirecting to login.
	header( "Location: login.php?parsing_error" );

	closelog();
	exit();
}


// Received a valid response from the WebUIAgent. Now check if the session ID was valid.
// Status = Success indicates valid session (Failure indicates invalid session).
if( $msg->GetStatus() !== "Success" )
{
	// Expired or invalid session; delete expired session cookie from client by setting
	// past due expiration time.
	setcookie( "FZAPWebUISession", "", time() - 86400, "/", "", true, true );	// yesterday
	
	// Redirect to login page with session expired indication
	header( "Location: login.php?expired" );
	
	closelog();
	exit();
}


// Save off user name for use in the HTML page. The user name should be included in the
// validate session response message if status indicates success. Even if not included
// (message format error), Get method should return empty string (okay to display without
// checking).
$username = $msg->GetUserName();
$csrftoken = $msg->GetCsrfToken();

closelog();

// If we got here, the session was valid. Serve up the index page (fall through to HTML).
header( "Cache-Control: no-cache, must-revalidate" );

?>


<!DOCTYPE html>
<html>
	<head>
		<meta charset="utf-8">
		<meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1" />
		<meta name="viewport" content="width=device-width, minimum-scale=1.0, initial-scale=1.0, user-scalable=yes">
		<meta name="msapplication-tap-highlight" content="no" />
		<title>Flexi Zone Web Access</title>
		<link rel = "stylesheet" type = "text/css" href = "/OceanTouch/css/styles.css">

		<style>

			.headerPanel {
				position: fixed;
				top: 0px;
				left: 0px;
				width: 100%;
				height: 84px;
				min-width: 550px;
				background-image: url( "/images/header-panel.png" );
				background-repeat: repeat-x;
				background-size: 200px 85px;
				background-position: 0px 0px;
				box-shadow: 0px 0.3em 0.5em #444444;
				z-index: 100;
			}

			.headerCaption {
				padding-right: 30px;
				position: absolute;
				top: 26px;
				right: 0px;
				color: #FFFFFF;
			}

			.headerCaption h1
			{
				font-size: 24px;
				line-height: 32px;
				font-weight: bold;
				color: #fff;
				text-shadow: rgba(0,0,0,0.45) 1px 1px 0px;
			}

			.headerCaption h1 .alt
			{
				color: #ccc;
			}


			.headerLogo {
				position: absolute;
				top: 0px;
				left: 0px;
				height: 100%;
				width: auto;
			}

			.menu-link {
				position: absolute;
				top: 7em;
				left: 30px;
			}

			.menu-link > span.icon
			{
				display: inline-block;
				width: 13px;
				height: 10px;
				margin-right: 3px;
				background: transparent url( /OceanTouch/images/icon-list.png ) no-repeat left top;
			}

			.step.current > .icon
			{
				width: 20px;
				height: 20px;
				margin-right: 1px;
				margin-left: -2px;
				background: transparent url(/images/spinner.gif) no-repeat left top;
				background-size: 20px 20px;
			}

			div.spinner-container
			{
				position: relative;
			}

			div.spinner-container .center-spinner
			{
				position: absolute;
				z-index: 1;
			}

			div.spinner-container .center-spinner > .step.current > .icon
			{
				width: 40px;
				height: 40px;
				margin: 0px;
				padding: 0px;
				background: transparent url(/images/spinner.gif) no-repeat left top;
				background-size: 40px 40px;
			}

			.dialog.dim.error
			{
				z-index: 50;
			}

			.dialog.dim.error.active
			{
				z-index: 52;
			}

			.dialog.dim.warning
			{
				z-index: 30;
			}

			.dialog.dim.warning.active
			{
				z-index: 32;
			}

			.dialog.dim
			{
				z-index: 20;
			}

			.dialog.dim.active
			{
				z-index: 22;
			}

			.dialog
			{
				z-index: 10;
			}

			.dialog.active
			{
				z-index: 12;
			}

			.dialog-dimmer.error
			{
				z-index: 48;
				background-color: rgba( 0, 0, 0, 0.5 );
			}

			.dialog-dimmer.warning
			{
				z-index: 28;
			}

			.dialog-dimmer
			{
				z-index: 18;
			}

			.dialog.notification
			{
				min-width: 100px;
				position: fixed;
				top: 7em;
				right: 1em;
			}

			.dialog.notification > .content
			{
				padding-top: 10px;
				padding-right: 25px;
				padding-bottom: 0px;
				padding-left: 0px;
			}

			.step.upcomming
			{
				color:#999999;
			}

			.content-display-panel
			{
				position: absolute;
				top: 160px;
				left: 300px;
				min-width: 350px;
				z-index: 1;
			}

			.content-display-panel.left-aligned
			{
				left: 30px;
			}

			#menuPanel
			{
				position: absolute;
				top: 160px;
				left: 30px;
				z-index: 2;
			}

			.menu-rule
			{
				margin-top: -8px;
				margin-bottom: 8px;
			}

			.user-role h3
			{
				margin-right: 8px;
				text-overflow: ellipsis;
				overflow: hidden;
			}

			.multi-column-content
			{
				-webkit-column-count: 2;
				-moz-column-count: 2;
				column-count: 2;

				-webkit-column-gap: 60px;
				-moz-column-gap: 60px;
				column-gap: 60px;

				-webkit-column-rule: 2px solid #d8d8d8;
				-moz-column-rule: 2px solid #d8d8d8;
				column-rule: 2px solid #d8d8d8;

				padding-bottom: 4px;
			}

			@media only screen and (max-width: 700px) {

				.multi-column-content
				{
					-webkit-column-count: 1;
					-moz-column-count: 1;
					column-count: 1;
				}
			}

			.alarm-fault-name
			{
				text-overflow: ellipsis;
				overflow: hidden;
				max-width: 600px;
			}
			.alarm-time
			{
				text-overflow: ellipsis;
				overflow: hidden;
				max-width: 168px;
			}
			.alarm-id
			{
				text-overflow: ellipsis;
				overflow: hidden;
				max-width: 84px;
			}
			.alarm-source
			{
				text-overflow: ellipsis;
				overflow: hidden;
				max-width: 224px;
			}

			.certificate-subject
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 210px;
			}
			.certificate-issuer
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 112px;
			}
			.certificate-type
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 56px;
			}
			.certificate-validfrom
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 182px;
			}
			.certificate-validto
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 182px;
			}
			.certificate-serialnum
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 70px;
			}
			.certificate-fingerprint
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 210px;
			}
			.certificates-panel-dimmer
			{
				position: absolute;
				top: 0px;
				left: 0px;
				width: 100%;
				height: 100%;
				z-index: 3;
				margin-top: -5px;
				margin-left: -5px;
				padding-bottom: 10px;
				padding-right: 7px;
				background-color: rgba(102, 204, 255, 0.25);
				}

			.antenna-id
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 100px;
			}
			.antenna-site-id
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 300px;
			}
			.antenna-ip-address
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 110px;
			}
			.sfn-group-1
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 100px;
			}
			.sfn-group-2
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 100px;
			}
			table.standard > tbody > tr > td.sfn-group-member
			{
				text-align: center;
			}
			table.standard > tbody > tr > td.sfn-group-member > .icon
			{
				display: inline-block;
				width: 16px;
				height: 17px;
				margin-right: 3px;
				vertical-align: -3px;
				background: transparent url(/OceanTouch/images/alert-icons.png) no-repeat left top;
				background-position: -48px;
			}
			.antenna-operational-state
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 125px;
			}
			.antenna-blocking-state
			{
				overflow: hidden;
				text-overflow: ellipsis;
				max-width: 125px;
			}

			.variable-width-panel-section
			{
				width: inherit;
			}

			.variable-width-panel-section .progress-indicator
			{
				width: 60%;
				float: left;
				overflow: hidden;
				margin-top: -2px;
			}

			.variable-width-panel-section .buttonbar
			{
				width: 40%;
				float: left;
			}

			#bubble
			{
				z-index: 40;
				max-width: 380px;
			}
		</style>

	</head>


	<body>

		<div class = "headerPanel">
			<div class = "headerCaption">
				<h1>Flexi Zone&#8482; <span class = "alt">Web Access</span></h1>
			</div>
			<img class = "headerLogo" src = "/images/NOKIA_LOGO_WHITE.png">
		</div>

		<a id = "menuLink" href = "#" class = "menu-link text">
			<span class = "icon"></span>
			Menu
		</a>

		<div id = "menuPanel" class="panel vertical-menu span-6">

			<ul class = "tree vpush-1">
				<li>
					<a href = "#">Auto-Connection</a>
					<ul>
						<li>
							<a id = "monitorPnPLink" class = "content-panel-link" href = "#">
								Display Plug &amp; Play Progress
							</a>
						</li>
						<li>
							<a id = "configureSiteIdLink" class = "dialog-link" href="#">
								Configure Site ID
							</a>
						</li>
					</ul>
				</li>
				<li>
					<a href = "#">Maintenance</a>
					<ul>
						<li>
							<a id = "resetSiteLink" class = "dialog-link" href = "#">
                                                                Reset Small Cell
							</a>
						</li>
						<li>
							<a id = "configResetLink" class = "dialog-link" href = "#">
								Configuration Reset
							</a>
						</li>
						<li>
							<a id = "blockSiteLink" class = "dialog-link" href = "#">
								Block Small Cell
							</a>
						</li>
						<li>
							<a id = "unblockSiteLink" class = "dialog-link" href = "#">
								Unblock Small Cell
							</a>
						</li>
						<li>
							<a id = "locationLockLink" class = "dialog-link" href = "#">
								Location Lock Management
							</a>
						</li>
						<li>
							<a id = "changeBandClassLink" class = "dialog-link" href = "#">
								Modify HW Variant
							</a>
						</li>
					</ul>
				</li>
				<li>
					<a id = "displaySiteInfoLink" class = "content-panel-link" href = "#">
						Display Site Information
					</a>
				</li>
				<li>
					<a id = "displayCellInfoLink" class = "content-panel-link" href = "#">
						Display Cell Information
					</a>
				</li>
				<li>
					<a id = "displayAlarmsLink" class = "content-panel-link" href = "#">
						Display Active Alarms
					</a>
				</li>
				<li>
					<a id = "displayAntennaInfoLink" class = "content-panel-link" href = "#">
						SFN Configuration
					</a>
				</li>
				<li>
					<a id = "snapshotLink" class = "dialog-link" href = "#">
						Snapshot Collection
					</a>
				</li>
				<li>
					<a id = "displayBluetoothLink" class = "content-panel-link" href = "#">
						Bluetooth Configuration
					</a>
				</li>
				<li>
					<a id = "displayIpSecLink" class = "content-panel-link" href = "#">
						IP Security Configuration
					</a>
				</li>
				<li>
					<a id = "rttLink" class = "content-panel-link" href = "#">
						IP Connectivity Test
					</a>
				</li>
				<li> 
					<a id = "certificateManagementLink" class = "content-panel-link" href = "#">
						Display Certificate Information
					</a>
				</li>

				<div class = "ruled menu-rule"></div>

				<li>
					<a href = "#">About</a>
					<ul>
						<li>
							<a id = "browsersLink" class = "dialog-link" href = "#">
								Recommended Browsers
							</a>
						</li>
					</ul>
				</li>
			</ul>

			<div class = "ruled menu-rule"></div>

			<div class = "vpush-2 user-role" style = "margin-left: 16px;">
				<img src="/OceanTouch/images/profile.png" width="42" height="42" class="left" />
				<h3>
					<?php echo $username; ?>
				</h3>
				<a id = "logoutLink" href = "#">Log Out</a>
			</div>

			<div style = "clear: both;"></div>
                
		</div>


		<div id = "siteInfoPanel" class = "content-display-panel">
			<div class = "panel-portal highlight-top">

				<h1 class = "titlebar ruled">
					Site Information
				</h1>

				<div class = "grid vpush-1 span-18 lead-tab">
					<div class="span-8 text">Site name:</div>
					<div id = "siteName" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">BTS type:</div>
					<div id = "btsType" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">AP Mod ID:</div>
					<div id = "apModId" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">ANT Mod ID:</div>
					<div id = "antModId" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">SFN AP Master ID:</div>
					<div id = "sfnApMasterId" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">Auto-Connection Site ID:</div>
					<div id = "siteId" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">BTS ID:</div>
					<div id = "btsId" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">HW type:</div>
					<div id = "unitName" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">Serial number:</div>
					<div id = "hwSerialNumber" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">Managed object in SCF:</div>
					<div id = "managedObject" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">BTS address (mplane):</div>
					<div id = "btsAddress" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">SW release version:</div>
					<div id = "swVersion" class="span-10 text">Not available</div>
				</div>

				<h4 class = "vpush-1"><strong>Status</strong></h4>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">Operational state</div>
					<div id = "opState" class="span-10 text">Not available</div>
				</div>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">Blocking state</div>
					<div id = "blockState" class="span-10 text">Not available</div>
				</div>

				<h4 class = "vpush-1"><strong>Synchronization</strong></h4>
				<div class = "grid span-18 lead-tab">
					<div class="span-8 text">Synchronization source:</div>
					<div id = "syncSource" class="span-10 text">Not available</div>
				</div>
				<div style = "clear: both;"></div>
			</div>
		</div>

		<div id = "cellInfoPanel" class = "content-display-panel">
			<div class = "panel-portal highlight-top">

				<h1 class = "titlebar ruled">
					Cell Information
				</h1>

				<div id = "cellInfoMissingDescription" class = "span-18">
					<p class = "text">
					Cell information is not available.
					</p>
				</div>

				<div id="cell_0">
					<h4 class = "vpush-1"><strong>Cell</strong></h4>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Global cell ID(LNCEL ID):</div>
						<div  class="global-cell-id span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Physical cell ID:</div>
						<div  class="phy-cell-id span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Downlink(TX) carrier frequency:</div>
						<div class="dl-carrier-frequency span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Uplink(RX) carrier frequency:</div>
						<div class="ul-carrier-frequency span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Downlink channel bandwidth:</div>
						<div class="dl-channel-bandwidth span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Uplink channel bandwidth:</div>
						<div class="ul-channel-bandwidth span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Channel bandwidth:</div>
						<div class="channel-bandwidth span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">MIMO:</div>
						<div class="mimo-type span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Operational state:</div>
						<div class="cell-op-state span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Blocking state:</div>
						<div class="cell-block-state span-10 text">Not available</div>
					</div>
					<div style = "clear: both;"></div>
				</div>
				<div id="cell_1">
					<h4 class = "vpush-1"><strong>Cell</strong></h4>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Global cell ID(LNCEL ID):</div>
						<div class="global-cell-id span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Physical cell ID:</div>
						<div  class="phy-cell-id span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Downlink(TX) carrier frequency:</div>
						<div class="dl-carrier-frequency span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Uplink(RX) carrier frequency:</div>
						<div class="ul-carrier-frequency span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Downlink channel bandwidth:</div>
						<div class="dl-channel-bandwidth span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Uplink channel bandwidth:</div>
						<div class="ul-channel-bandwidth span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Channel bandwidth:</div>
						<div class="channel-bandwidth span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">MIMO:</div>
						<div class="mimo-type span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Operational state:</div>
						<div class="cell-op-state span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Blocking state:</div>
						<div class="cell-block-state span-10 text">Not available</div>
					</div>
					<div style = "clear: both;"></div>
				</div>

				<div id="cell_2">
					<h4 class = "vpush-1"><strong>Cell</strong></h4>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Global cell ID(LNCEL ID):</div>
						<div class="global-cell-id span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Physical cell ID:</div>
						<div  class="phy-cell-id span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Downlink(TX) carrier frequency:</div>
						<div class="dl-carrier-frequency span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Uplink(RX) carrier frequency:</div>
						<div class="ul-carrier-frequency span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Downlink channel bandwidth:</div>
						<div class="dl-channel-bandwidth span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Uplink channel bandwidth:</div>
						<div class="ul-channel-bandwidth span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Channel bandwidth:</div>
						<div class="channel-bandwidth span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">MIMO:</div>
						<div class="mimo-type span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Operational state:</div>
						<div class="cell-op-state span-10 text">Not available</div>
					</div>
					<div class = "grid span-18 lead-tab">
						<div class="span-8 text">Blocking state:</div>
						<div class="cell-block-state span-10 text">Not available</div>
					</div>
					<div style = "clear: both;"></div>
				</div>
			</div>
		</div>

		<div id = "PnPPanel" class = "content-display-panel">
			<div class = "panel">

				<h1 class = "titlebar ruled">
					Auto Connection &amp; Configuration
				</h1>

				<div class = "multi-column-content vpush-1">

					<h4><strong>Auto Connection Status</strong></h4>

					<div class = "ruled span-9 vpush-1 lead-tab">
						<span id = "AutoConnectionIpState" class="step upcomming">
							<span class="icon"></span>
							IP Connectivity
						</span>
					</div>

					<div class = "ruled span-9 vpush-1 lead-tab">
						<span id = "AutoConnectionDhcpState" class="step upcomming">
							<span class="icon"></span>
							DHCP Server Connection
						</span>
					</div>

					<div class = "ruled span-9 vpush-1 lead-tab">
						<span id = "AutoConnectionFzcDiscovery" class="step upcomming">
							<span class="icon"></span>
							Connection to dedicated O&M mediator
						</span>
					</div>

					<div class = "vpush-1 span-9 lead-tab">
						<span id = "AutoConnectionFzcConnection" class="step upcomming">
							<span class="icon"></span>
							Connection to final O&M mediator
						</span>
					</div>


					<h4 class = "vpush-2"><strong>Software Download &amp; Configuration</strong></h4>

					<div class = "vpush-1 span-9 lead-tab ruled">
						<span id = "SWDownload" class="step upcomming">
							<span class="icon"></span>
							Software Download
						</span>
					</div>

					<div class = "vpush-1 span-9 lead-tab ruled">
						<span id = "SWActivation" class="step upcomming">
							<span class="icon"></span>
							Software Activation
						</span>
					</div>

					<div class = "vpush-1 span-9 lead-tab ruled">
						<span id = "SWPreActivation" class="step upcomming">
							<span class="icon"></span>
							Software Pre-Activation
						</span>
					</div>

					<div class = "vpush-1 span-9 lead-tab ruled">
						<span id = "ConfFileDownload" class="step upcomming">
							<span class="icon"></span>
							Configuration File Download
						</span>
					</div>

					<div class = "vpush-1 span-9 lead-tab">
						<span id = "ConfFileActivation" class="step upcomming">
							<span class="icon"></span>
							Configuration File Activation
						</span>
					</div>

					<div class = "vpush-1 span-9 lead-tab">
						<span id = "SWAndConfigActivation" class="step upcomming">
							<span class="icon"></span>
							Software and Configuration Activation
						</span>
					</div>

					<div style = "clear: both;"></div>

				</div>
			</div>
		</div>

		<div id = "AntennaPanel" class = "content-display-panel">
			<div class = "panel">
				<div id = "AntennaPanelDimmer" class = "certificates-panel-dimmer"> </div>
				<h1 class = "titlebar ruled">
					SFN Configuration
				</h1>
				<p class = "vpush-1-half text">
					SFN Group 1:
					<span id = "sfnGroup1CellInfo" class = "lead-tab">Not available</span>
					<br>
					SFN Group 2:
					<span id = "sfnGroup2CellInfo" class = "lead-tab">Not available</span>
				</p>

				<table id = "antennaTable" class="vpush-1 standard row-selection span-25">
					<thead>
						<tr>
							<th>Antenna ID</th>
							<th>Antenna Site ID</th>
							<th>IP Address</th>
							<th>SFN Group 1</th>
							<th>SFN Group 2</th>
							<th>Operational State</th>
							<th>Blocking State</th>
						</tr>
					</thead>

					<tbody class="selectable">
						<tr>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
						</tr>
						<tr>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
						</tr>
					</tbody>
				</table>

				<div class="vpush-1-half text-right buttonbar">
					<button id = "antennaDetailsButton" class = "button small">Details</button>
				</div>

				<div class = ruled></div>
				<div class="text-left buttonbar vpush-1-half">
					<button id = "blockAntennaButton" class = "button small">Block Antenna</button>
					<button id = "unblockAntennaButton" class = "button small">Unblock Antenna</button>
					<button id = "resetAntennaButton" class = "button small">Reset Antenna</button>
				</div>
				<div style = "clear: both;"></div>
			</div>
		</div>

		<!-- alarms table -->
		<div class = "content-display-panel">
			<table id = "activeAlarmsTable" class="standard row-selection span-25">

				<thead>
					<tr>
						<th class = "sortable">Severity</th>
						<th class = "sortable">Alarm Time GMT -06:00</th>

						<th>Alarm ID</th>
						<th>Fault Name</th>
						<th>Source</th>
					</tr>
				</thead>

				<tbody class="selectable">
					<tr>
						<td></td>
						<td>&nbsp;</td>
						<td>&nbsp;</td>
						<td>&nbsp;</td>
						<td>&nbsp;</td>
					</tr>
					<tr>
						<td></td>
						<td>&nbsp;</td>
						<td>&nbsp;</td>
						<td>&nbsp;</td>
						<td>&nbsp;</td>
					</tr>
					<tr>
						<td></td>
						<td>&nbsp;</td>
						<td>&nbsp;</td>
						<td>&nbsp;</td>
						<td>&nbsp;</td>
					</tr>
				</tbody>

				<tfoot>
					<tr>
						<td colspan = "2">Active Alarms: 0</td>
						<td colspan = "3">&nbsp;</td>
					</tr>
				</tfoot>

			</table>

			<div style = "clear: both;"></div>
		</div>


		<div id = "bluetoothPanel" class = "content-display-panel">
			<div class = "panel">

				<h1 class = "titlebar ruled">
					Bluetooth Configuration
				</h1>

				<p class = "vpush-2 text">
					Bluetooth Interface Support:
					<span id = "bluetoothStatus" class = "lead-tab">Not available</span>
				</p>

				<table id = "pairingsTable" class="standard vpush-2 span-18">
					<thead>
						<tr>
							<th class = "sortable">Order Number</th>
							<th class = "sortable">Device Address</th>
							<th class = "sortable">Device Status</th>
						</tr>
					</thead>

					<tbody>
						<tr>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
						</tr>
						<tr>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
						</tr>
						<tr>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
						</tr>
					</tbody>
				</table>

				<p class = "ruled"></p>

				<div class = "grid variable-width-panel-section">
					<div class = "vspan-2 progress-indicator">
						<div id = "bluetoothProgressIndicator" class = "step">
							<span class = "icon"></span>
							<span class = "text"></span>
						</div>
					</div>
					<div class = "text-right buttonbar">
						<button id = "refreshBluetoothButton" class = "button small">Refresh</button>
					</div>
				</div>

				<div style = "clear: both;"></div>
			</div>
		</div>


		<div id = "ipSecPanel" class = "content-display-panel">
			<div class = "panel">

				<h1 class = "titlebar ruled">
					IP Security
				</h1>

				<p class = "vpush-2 text">
					IpSecEnabled:
					<span id = "ipSecStatus" class = "lead-tab">Not available</span>
				</p>

				<span class = "vpush-1 text">
					Internet Key Exchange (IKE) Associations:
				</span>

				<table id = "associationsTable" class="standard span-18">
					<thead>
						<tr>
							<th class = "sortable">Remote Tunnel Endpoint</th>
							<th class = "sortable">Peer State</th>
						</tr>
					</thead>

					<tbody>
						<tr>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
						</tr>
						<tr>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
						</tr>
					</tbody>
				</table>

				<p class = "ruled"></p>

				<div class = "grid variable-width-panel-section">
					<div class = "vspan-2 progress-indicator">
						<div id = "ipSecProgressIndicator" class = "step">
							<span class = "icon"></span>
							<span class = "text"></span>
						</div>
					</div>
					<div class = "text-right buttonbar">
						<button id = "refreshIpSecButton" class = "button small">Refresh</button>
					</div>
				</div>

				<div style = "clear: both;"></div>
			</div>
		</div>


		<div id = "rttPanel" class = "content-display-panel">
			<div class = "panel">

				<h1 class = "titlebar ruled">
					IP Connectivity
				</h1>

				<p class = "vpush-2 span-16 text">
					Perform IP Connectivity test between Flexi Zone&#8482 small cell and target host.
				</p>

				<form action = "#">
					<div class = "grid span-16 vpush-2">
						<div class = "span-3 vspan-2"><label>Target IP:</label></div>
						<div class = "span-7">
							<input id = "targetIpField" type = "text" class = "text required" placeholder = "127.0.0.1" required = "true" title = "Enter host IP address; format: 255.255.255.255"></input>
							<input type="hidden" name="csrfToken" value="<?php echo $csrftoken;?>">
						</div>
						<div class = "span-6 text-right">
							<button id = "rttButton" class = "call-to-action" type = "submit" value = "Test">
								Test
							</button>
						</div>
					</div>
				</form>

				<p class = "ruled"></p>

				<div class = "vpush-1 spinner-container">
					<div id = "rttSpinner" class = "center-spinner">
						<span class = "step current">
							<span class = "icon"></span>
						</span>
					</div>

					<textarea id = "rttOutputArea" rows = "11" readonly></textarea>
				</div>

				<div style = "clear: both;"></div>
			</div>
		</div>

		<div id = "CertificatePanel" class = "content-display-panel">
			<div class = "panel">
				<div id = "CertificatePanelDimmer" class = "certificates-panel-dimmer"> </div>
				<h1 class = "titlebar ruled">
					Certificate Management
				</h1>

				<table id = "certificatesTable" class="vpush-1 standard row-selection span-25">
					<thead>
						<tr>
							<th>Issued to</th>
							<th>Issued by</th>
							<th>Type</th>
							<th>Valid from</th>
							<th>Valid to</th>
							<th>Serial number</th>
							<th>Fingerprint</th>
						</tr>
					</thead>

					<tbody class="selectable">
						<tr>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
						</tr>
						<tr>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
							<td>&nbsp;</td>
						</tr>
					</tbody>
				</table>

				<div class="vpush-1-half text-right buttonbar">
					<button id = "certificateDetailsButton" class = "button small">Details</button>
				</div>

				<p class = "vpush-1-half text">
					<strong>CMP Server: </strong>
					<strong id = "CmpServerStatus">Not available</strong>
				</p>
				<div class="text-left buttonbar">
					<button id = "CmpKeyUpdateButton" class = "button small">CMP Key Update</button>
					<button id = "initializeCertificatesButton" class = "button small">Initialize Certificates</button>
				</div>
				<div class = "span-10 vspan-2 vpush-1-half">
					<div id = "Certificate_Indicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</div>
				</div>
				<div class = ruled></div>

				<div class = "grid vpush-1 variable-width-panel-section">
					<div class = "vspan-2 progress-indicator">
						<div id = "certificatesRefreshIndicator" class = "step">
							<span class = "icon"></span>
							<span class = "text"></span>
						</div>
					</div>
					<div class = "text-right buttonbar">
						<button id = "refreshCertificatesButton" class = "button small">Refresh</button>
					</div>
				</div>

				<div style = "clear: both;"></div>
			</div>
		</div>

		<!-- Block site pop-up dialog -->
		<div id = "blockDialog" class = "dialog warning dim block-form">
			<div class = "title">
				<h1>Block Small Cell</h1>
			</div>

			<p class = "ruled span-18">
				Blocking the site will interrupt service in all cells. Do you want to proceeed?
			</p>

			<div class = "grid span-18">
				<div class = "span-10 vspan-2">
					<div id = "blockProgressIndicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</div>
				</div>
				<div class = "span-8 text-right buttonbar">
					<button id = "blockConfirmButton" class = "call-to-action">Confirm</button>
					<button id = "blockCloseButton">Close</button>
				</div>
			</div>
		</div>

		<!-- Block antenna pop-up dialog -->
		<div id = "blockAntennaDialog" class = "dialog warning dim block-form">
			<div class = "title">
				<h1>Block Antenna</h1>
			</div>

			<p class = "ruled span-18">
				Blocking the antenna will interrupt service for this antenna. Do you want to proceeed?
			</p>

			<div class = "grid span-18">
				<div class = "span-10 vspan-2">
					<div id = "blockAntennaProgressIndicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</div>
				</div>
				<div class = "span-8 text-right buttonbar">
					<button id = "blockAntennaConfirmButton">Confirm</button>
					<button id = "blockAntennaCloseButton">Close</button>
				</div>
			</div>
		</div>

		<!-- Unblock site pop-up dialog -->
		<div id = "unblockDialog" class = "dialog warning dim block-form">
			<div class = "title">
				<h1>Unblock Small Cell</h1>
			</div>

			<p class = "ruled span-18">
				Do you want to proceeed?
			</p>

			<div class = "grid span-18">
				<div class = "span-10 vspan-2">
					<div id = "unblockProgressIndicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</div>
				</div>
				<div class = "span-8 text-right buttonbar">
					<button id = "unblockConfirmButton" class = "call-to-action">Confirm</button>
					<button id = "unblockCloseButton">Close</button>
				</div>
			</div>
		</div>

		<!-- Unblock antenna pop-up dialog -->
		<div id = "unblockAntennaDialog" class = "dialog warning dim block-form">
			<div class = "title">
				<h1>Unblock Antenna</h1>
			</div>

			<p class = "ruled span-18">
				Do you want to proceeed?
			</p>

			<div class = "grid span-18">
				<div class = "span-10 vspan-2">
					<div id = "unblockAntennaProgressIndicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</div>
				</div>
				<div class = "span-8 text-right buttonbar">
					<button id = "unblockAntennaConfirmButton">Confirm</button>
					<button id = "unblockAntennaCloseButton">Close</button>
				</div>
			</div>
		</div>

		<!-- Site ID pop-up dialog -->
		<div id = "siteIdDialog" class = "dialog">

			<div class="title">
				<h1>Configure Site ID</h1>
			</div>

			<form action = "#">
				<div class="grid span-15 vlead-2">
					<div class="span-6 vspan-2"><label>Auto-connection Site ID:</label></div>
					<div class="span-9">
						<input id = "siteIdField" type = "text" class = "text required" placeholder = "Site ID" required = "true" title = "Enter Auto-Connection Site ID: [1..50] ASCII characters"></input>
						<input type="hidden" name="csrfToken" value="<?php echo $csrftoken;?>">
					</div>
				</div>

				<p class="ruled"></p>

				<div class = "grid span-15">
					<div class = "span-8 vspan-2">
						<span id = "siteIdStatus" class = "step">
							<span class = "icon"></span>
							<span class = "text"></span>
						</span>
					</div>

					<div class = "span-7 text-right buttonbar">
						<button id = "sendButton" class = "call-to-action" type = "submit" value = "Configure">
							Configure
						</button>
						<button id = "closeButton" value = "Close">Close</button>
					</div>
				</div>

				<div id = "siteIdErrorDescription" class = "span-15">
					<p class = "text">
						Auto-connection Site ID cannot be changed after site has been commissioned.
					</p>
				</div>
			</form>
		</div>
		
		<!-- Location Lock pop-up dialog -->
		<div id = "locationLockDialog" class = "dialog block-form">

			<div class="title">
				<h1>Location Lock Management</h1>
			</div>
		
			<div class = "grid span-17 ruled">
				<div class = "span-10 vpush-1" id = "locationLockDisplay" >
					<p class = "text">
						<div><strong>Toggle location lock:</strong></div>
						<div>Current actLocationLock value:  <span id="locationLockCurrent"></span></div>
					</p>
					<p class = "span-8">
						<label>
							<input type = "radio" name = "locationLockUpdate" checked = "true" value = "false">
								False (unlocked)
							</input>
						</label>
						<label>
							<input type = "radio" name = "locationLockUpdate" value = "true">
								True (locked)
							</input>
						</label>
					</p>
				</div>
				<div class = "span-10 vpush-1" id = "locationLockStatusUnavailable">
					<p class = "text">Location lock status currently unavailable.</p>
				</div>
			</div>

			<div class = "grid span-17">
				<div class = "span-10 vspan-2">
					<span id = "locationLockProgressIndicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</span>
				</div>

				
				<div class="text-right span-7 buttonbar">
					<button id = "locationLockStartButton" class = "call-to-action">Confirm</button>
					<button id = "locationLockCloseButton">Close</button>
				</div>
			</div>
		</div>

		<!-- Reset site pop-up dialog -->
		<div id = "resetSiteDialog" class = "dialog warning dim block-form">
			<div class = "title">
				<h1>Reset Small Cell</h1>
			</div>

			<p class = "ruled span-15">Resetting the site will cause an interruption in service. Do you want to proceeed?
			</p>

			<div class = "grid span-15">
				<div class = "span-8 vspan-2">
					<div id = "resetSiteProgressIndicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</div>
				</div>
				<div class = "span-7 text-right buttonbar">
					<button id = "resetSiteConfirm">Confirm</button>
					<button id = "resetSiteCancel" class = "call-to-action">Close</button>
				</div>
			</div>
			<div id = "resetSiteErrorDescription" class = "span-15">
				<p class = "text">
				    Site cannot be reset, please try again later
				</p>
			</div>
		</div>

		<!-- Reset antenna pop-up dialog -->
		<div id = "resetAntennaDialog" class = "dialog warning dim block-form">
			<div class = "title">
				<h1>Reset Antenna</h1>
			</div>
			<p class = "ruled span-15">Resetting the antenna will cause an interruption in service. Do you want to proceeed?
			</p>
			<div class = "grid span-15">
				<div class = "span-8 vspan-2">
					<div id = "resetAntennaProgressIndicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</div>
				</div>
				<div class = "span-7 text-right buttonbar">
					<button id = "resetAntennaConfirmButton">Confirm</button>
					<button id = "resetAntennaCloseButton">Close</button>
				</div>
			</div>
		</div>

		<!-- Config Reset pop-up dialog -->
		<div id = "configResetDialog" class="dialog warning dim block-form">
			<div class="title">
				<h1>Configuration Reset</h1>
			</div>
			<p class = "ruled span-18">The site will be reset and all existing configuration data will be removed except auto-connection site ID and security credentials. Auto-connection will start from the beginning. Please make sure pre-planned configuration data is available if auto-configuration is expected to be performed.
			</p>
			<div class = "grid span-18 ruled">
				<span class = "text">
					<strong>Additional Options</strong>
				</span>
				<div class="span-18 vpush-1-half">
					<label>
						<input type = "checkbox" id = "removeSiteIdCheckbox" value = "removeSiteId">
							Remove Auto-Connection Site ID
						</input>
					</label>
				</div>
				<div class="span-18 vpush-1-half">
					<label>
						<input type = "checkbox" id = "removeSecCredCheckbox" value = "removeSecCred" class = "checkbox-with-bubble" warning-text = "Security infrastructure needs to be in place when security credentials are removed and autoconnection procedure shall be used to re-establish secure connection">
							Remove Security Credentials
						</input>
					</label>
                                </div>
                        </div>
			<div class = "grid span-18">
				<div class = "span-11 vspan-2">
					<span id = "configResetProgressIndicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</span>
				</div>

				<div class="text-right span-7 buttonbar">
					<button id = "configResetConfirmButton" class = "call-to-action">Confirm</button>
					<button id = "configResetCancelButton">Cancel</button>
				</div>
			</div>
			<div id = "configResetErrorDescription" class = "span-18">
				<p class = "text">
				    Site cannot be reset, please try again later
				</p>
			</div>

		</div>

		<!-- Snapshot pop-up dialog -->
		<div id = "snapshotDialog" class="dialog block-form">
			<div class="title">
				<h1>Snapshot Collection</h1>
			</div>

			<div class = "grid span-15 ruled">
				<div class = "span-15 vpush-1">
					<p class = "text">
						<strong>Log coverage level:</strong>
					</p>
					<label>
						<input type = "radio" name = "logCoverage" checked = "true" value = "full">
							Full: All available data collected
						</input>
					</label>
					<label>
						<input type = "radio" name = "logCoverage" value = "minor">
							Partial: Most important data collected
						</input>
					</label>
				</div>

				<div class = "span-15 vpush-1">
					<a id = "downloadSnapshotLink" class = "button icon save" href = "#">
						<span class = "icon"></span>
						Download
					</a>
				</div>
			</div>

			<div class = "grid span-15">
				<div class = "span-9 vspan-2">
					<span id = "snapshotProgressIndication" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</span>
				</div>

				<div class="text-right span-6 buttonbar">
					<button id = "snapshotStartButton" class = "call-to-action">Start</button>
					<button id = "snapshotCancelButton">Cancel</button>
					<button id = "snapshotCloseButton">Close</button>
				</div>
			</div>
		</div>

		<!-- BandClass pop-up dialog -->
		<div id = "bandClassDialog" class="dialog warning dim block-form">
			<div class="title">
				<h1>Modify HW Variant</h1>
			</div>
			<p class = "span-18 ruled" style="margin-bottom: 10px;">Update HW Variant to match installed filter.  This will cause a site reset.<br>Make sure the new RF filter has been installed before proceeding.
			</p>
			<div class = "grid span-18">
				<div class = "span-9">
					<p class = "text">
						<div><strong>Select new HW Variant</strong></div>
						<div>Current set Variant:  <span id="bandClassCurrent"></span></div>
					</p>
					<label>
						<input type = "radio" name = "filterType" id="fwhr" checked = "true" value = "FWHR">
                                                        FWHR:   Full Band
						</input>
					</label>
					<label>
						<input type = "radio" name = "filterType" id="fwhrhb" value = "FWHRHB">
                                                        FWHRHB: High Band
						</input>
					</label>
					<label>
						<input type = "radio" name = "filterType" id="fwhrlb" value = "FWHRLB">
                                                        FWHRLB: Low Band
						</input>
					</label>
				</div>
			</div>
			<div class = "grid span-18 ruled">
				<div class="span-18 text-right" id="bandClassResetWarning"><strong>A site reset will cause an interruption in service.  Do you want to proceed?</strong></div>
			</div>
			<div class = "grid span-18">
				<div class = "span-11 vspan-2">
					<span id = "bandClassProgressIndicator" class = "step">
						<span class = "icon"></span>
						<span class = "text"></span>
					</span>
				</div>
				
				<div class="text-right span-7 buttonbar">
					<button id = "bandClassStartButton" class = "call-to-action">Confirm</button>
					<button id = "bandClassCloseButton">Close</button>
				</div>
			</div>
		</div>


		<!-- Link loss pop-up dialog -->
		<div id = "linkLossDialog" class = "dialog error dim block-form">
			<div class = "title">
				<h1>Connection Lost</h1>
			</div>

			<p class = "ruled span-15">
				Connection to site has been lost. Attempting to reconnect.
				<br>Reconnecting should not take more than 10 minutes.
			</p>

			<div class = "grid span-15">
				<div class = "span-10 vspan-2">
					<div class = "step current">
						<span class = "icon"></span>
						<span class = "text">Reconnecting</span>
					</div>
				</div>
				<div class = "text-right span-5 buttonbar">
					<button id = "linkLossCloseButton" class = "call-to-action">Close</button>
				</div>
			</div>
		</div>


		<!-- Recommended browsers pop-up dialog -->
		<div id = "browsersDialog" class = "dialog block-form">

			<p class = "vpush-2 span-15">
				<strong>Recommended web browsers:</strong>
				<br>For optimal user experience, the following browsers are recommended: Firefox 32 or above, Chrome 37 or above, Internet Explorer 10 or above. Information on the page may not display correctly in other browsers or browser versions.
			</p>
			<p class = "ruled span-15">
				<strong>Note:&nbsp;</strong>
				Javascript and internet cookies must be enabled to access this web page.
			</p>

			<div class = "text-right span-15 buttonbar">
				<button id = "browsersCloseButton">Okay</button>
			</div>
		</div>


		<!-- Notification area -->
		<div id = "notificationDialog" class = "dialog notification modal"
			<div class = "grid">
				<div class = "vspan-2">
					<div class = "step current">
						<span class = "icon"></span>
						<span class = "text">Reconnecting</span>
					</div>
				</div>
			</div>
		</div>

		<!-- Certificate Details pop-up dialog -->
		<div id = "certificateDetailsDialog" class="dialog">
			<div class="title">
				<h1>Certificate Details</h1>
			</div>
			<div class = "span-20">
				<div class = "ruled vpush-1">
					<p class="vspan-1">Certificate Info</p>
					<textarea id="certificateDetailsText" rows = "7"readonly></textarea>
				</div>
				<div class = "vpush-1">
					<p class = "vspan-1">Certificate Extensions Info</p>
					<textarea id="extensionDetailsText" rows = "10" readonly></textarea>
				</div>
			</div>
		</div>

		<!-- Antenna Details pop-up dialog -->
		<div id = "antennaDetailsDialog" class="dialog">
			<div class="title">
				<h1>Antenna Details</h1>
			</div>
			<div class = "span-20">
				<div class = "ruled vpush-1">
					<p class="vspan-1">Antenna Info</p>
					<textarea id="antennaDetailsText" rows = "7"readonly></textarea>
				</div>
				<div class = "vpush-1">
					<p class = "vspan-1">Antenna Additional Info</p>
					<textarea id="antennaExtensionDetailsText" rows = "4" readonly></textarea>
				</div>
			</div>
		</div>

		<!-- Alarm Details pop-up dialog -->
		<div id = "alarmDetailsDialog" class="dialog dim block-form">
			<div class="title">
				<h1>Alarm Details</h1>
			</div>
			<div class = "span-20">
				<div class = "vpush-1">
					<textarea id="alarmDetailsText" rows = "10"readonly></textarea>
				</div>
			</div>
		</div>

		<div class="footer small">
			&copy; 2016 Nokia Networks. All rights reserved.
		</div>

		<script type="text/javascript" src="/OceanTouch/js/jquery.min.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/nsnkit.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/backgrounds.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/buttons.js"></script>
		<script type="text/javascript" src="/js/webuiDialogs.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/textinput.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/textarea.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/panel.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/tree.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/pulldowns.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/tooltip.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/radiobutton.js"></script>
		<script type="text/javascript" src="/js/WebuiBubble.js"></script>
		<script type="text/javascript" src="/js/WebuiCheckbox.js"></script>

		<script src = "/js/WebuiUtilities.js" type = "text/javascript"></script>
		<script src = "/js/ConfigureSiteId.js" type = "text/javascript"></script>
		<script src = "/js/Logout.js" type = "text/javascript"></script>
		<script src = "/js/MonitorPnP.js" type = "text/javascript"></script>
		<script src = "/js/ResetSite.js" type = "text/javascript"></script>
		<script src = "/js/ConfigReset.js" type = "text/javascript"></script>
		<script src = "/js/StandardTable.js" type = "text/javascript"></script>
		<script src = "/js/DisplayActiveAlarms.js" type = "text/javascript"></script>
		<script src = "/js/DisplaySiteInformation.js" type = "text/javascript"></script>
		<script src = "/js/DisplayCellInformation.js" type = "text/javascript"></script>
		<script src = "/js/AntennaManagement.js" type = "text/javascript"></script>
		<script src = "/js/Menu.js" type = "text/javascript"></script>
		<script src = "/js/CollectSnapshot.js" type = "text/javascript"></script>
		<script src = "/js/DisplayBluetoothConfiguration.js" type = "text/javascript"></script>
		<script src = "/js/DisplayIpSecConfiguration.js" type = "text/javascript"></script>
		<script src = "/js/MeasureRTT.js" type = "text/javascript"></script>
		<script src = "/js/BlockSite.js" type = "text/javascript"></script>
		<script src = "/js/UnblockSite.js" type = "text/javascript"></script>
		<script src = "/js/LinkAndSessionManagement.js" type = "text/javascript"></script>
		<script src = "/js/CertificateManagement.js" type = "text/javascript"></script>
		<script src = "/js/LocationLockManagement.js" type = "text/javascript"></script>
		<script src = "/js/BandFilter.js" type = "text/javascript"></script>
		<script src = "/js/About.js" type = "text/javascript"></script>
		<script src = "/js/Preload.js" type = "text/javascript"></script>
		<!-- <script src = "/js/EventDetection.js" type = "text/javascript"></script> -->

	</body>
</html>
