// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2016, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		AntennaManagement.js
//
// ****************************************************************************

(function( $ )
{
	var ANTENNA_INFO_REQUEST_TIMER_INTERVAL = 10000;
	var timerId;
	var NUM_ADD_OR_DELETE_ROWS = 1;
	var MAX_ANTENNAS = 12;
	var MAX_CELLS = 2;
	var MIN_NUM_TABLE_ROWS = 12;
	var antennaXmlHttp;
	var xmlHttp;
	var antennaArray = [];
	var initInProgress = false;
	var strNewLine = "\n";

	// AntennaEntry object
	var AntennaEntry = function( AntennaId, AntennaSiteId, AntennaIpAddress, SfnGroup1, SfnGroup2, OpState,
								SerialNumber, HwType, WorkingState, BlockState )
	{
		this.AntennaId = initializeStringValue( AntennaId );
		this.AntennaSiteId = initializeStringValue( AntennaSiteId );
		this.AntennaIpAddress = initializeStringValue( AntennaIpAddress );
		this.SfnGroup1 = SfnGroup1;
		this.SfnGroup2 = SfnGroup2;
		this.OpState = initializeStringValue( OpState );
		this.SerialNumber = initializeStringValue( SerialNumber );
		this.HwType = initializeStringValue( HwType );
		this.WorkingState = initializeStringValue( WorkingState );
		this.BlockState = initializeStringValue( BlockState );
	}

	var displayDetails = function( index )
	{
		var $dialog = $( "#antennaDetailsDialog" );

		if( $dialog.is( ":visible" ) )
		{
			return;
		}

		var text_val = "";
		var extension_text_val="";

		text_val += "Antenna ID : ";
		text_val += antennaArray[index].AntennaId;
		text_val += strNewLine;

		text_val += "Antenna Site ID : ";
		text_val += antennaArray[index].AntennaSiteId;
		text_val += strNewLine;

		text_val += "Antenna Ip Address : ";
		text_val += antennaArray[index].AntennaIpAddress;
		text_val += strNewLine;

		text_val += "SFN Group 1 Member: ";
		antennaArray[index].SfnGroup1 ? text_val += "Yes" : text_val += "No";
		text_val += strNewLine;

		text_val += "SFN Group 2 Member: ";
		antennaArray[index].SfnGroup2 ? text_val += "Yes" : text_val += "No";
		text_val += strNewLine;

		text_val += "Operational State : ";
		text_val += antennaArray[index].OpState;
		text_val += strNewLine;

		text_val += "Blocking State : ";
		text_val += antennaArray[index].BlockState;

		extension_text_val += "Serial Number : ";
		extension_text_val += antennaArray[index].SerialNumber;
		extension_text_val += strNewLine;

		extension_text_val += "HW Type : ";
		extension_text_val += antennaArray[index].HwType;
		extension_text_val += strNewLine;

		extension_text_val += "Working State : ";
		extension_text_val += antennaArray[index].WorkingState;

		$( "#antennaDetailsText" ).val(text_val);
		$( "#antennaExtensionDetailsText" ).val(extension_text_val);
		$dialog.trigger("open");
		$( "#AntennaPanelDimmer" ).show();
	}

	NSNReady( function()
	{
		// Setup antenna table.
		var $table = $( "#antennaTable" );
		$table.standard_table( MIN_NUM_TABLE_ROWS, MAX_ANTENNAS );
		setupTableCells( $table );

		$( "#antennaDetailsDialog" ).bind("close", function()
		{
			$( "#antennaDetailsText" ).val("");
			$( "#antennaExtensionDetailsText" ).val("");

			$( "#AntennaPanelDimmer" ).hide();
		});

		$( "#displayAntennaInfoLink" ).click( function() {

			if( $( "#AntennaPanel" ).is( ":visible" ) )
			{
				return;
			}

			// Hide any open panel(s) in the display area.
			$( ".content-display-panel" ).not( "#AntennaPanel" ).trigger( "closePanel" );

			// Deselect all rows
			$('#AntennaPanel tbody tr').removeClass('selected');

			// Show the antenna panel in the display area.
			$( "#AntennaPanel" ).show();
			$( "#blockAntennaButton" ).addClass( "disabled" );
			$( "#unblockAntennaButton" ).addClass( "disabled" );
			$( "#resetAntennaButton" ).addClass( "disabled" );
			$( "#antennaDetailsButton" ).addClass( "disabled" );

			//bind antenna detail button to selectionEvent
			$table.bind("selectionEvent", function()
			{
				var $selected_row = $table.find( "tbody tr.selected" );
				if( $selected_row.length > 0 )
				{
					var rowIndex = $selected_row.index();
					if( ( rowIndex >= 0 )  && ( rowIndex < antennaArray.length ) )
					{
						if(( antennaArray[rowIndex].BlockState !== "blocked" ) &&
						   ( antennaArray[rowIndex].BlockState !== "unblocked" ))
						{
							$( "#blockAntennaButton" ).addClass( "disabled" );
							$( "#unblockAntennaButton" ).addClass( "disabled" );
						}
						else
						{
							if( antennaArray[rowIndex].BlockState == "blocked" )
								$( "#blockAntennaButton" ).addClass( "disabled" );
							else
								$( "#blockAntennaButton" ).removeClass( "disabled" );

							if( antennaArray[rowIndex].BlockState == "unblocked" )
								$( "#unblockAntennaButton" ).addClass( "disabled" );
							else
								$( "#unblockAntennaButton" ).removeClass( "disabled" );
						}
						$( "#resetAntennaButton" ).removeClass( "disabled" );
						$( "#antennaDetailsButton" ).removeClass( "disabled" );
					}
					else
					{
						$( "#blockAntennaButton" ).addClass( "disabled" );
						$( "#unblockAntennaButton" ).addClass( "disabled" );
						$( "#resetAntennaButton" ).addClass( "disabled" );
						$( "#antennaDetailsButton" ).addClass( "disabled" );
					}
				}
				else
				{
					$( "#blockAntennaButton" ).addClass( "disabled" );
					$( "#unblockAntennaButton" ).addClass( "disabled" );
					$( "#resetAntennaButton" ).addClass( "disabled" );
					$( "#antennaDetailsButton" ).addClass( "disabled" );
				}
			});

			$table.bind("openEvent", function()
			{
				var $selected_row = $table.find( "tbody tr.selected" );
				if( $selected_row.length > 0 )
				{
					var rowIndex = $selected_row.index();
					if( ( rowIndex >= 0 )  && ( rowIndex < antennaArray.length ) )
					{
						displayDetails(rowIndex);
					}
				}
			});

			$( "#AntennaPanelDimmer" ).hide();

			SendAntennaInfoRequest();
		});

		$( "#antennaDetailsButton" ).click( function() {
			if( $( this ).is( ".disabled" ) )
			{
				return;
			}
			var rowIndex = $table.find( "tbody tr.selected" ).index();
			if( (rowIndex >= 0 ) && (rowIndex < antennaArray.length) )
			{
				displayDetails(rowIndex);
			}
		});

		$( "#blockAntennaButton" ).click( function() {
			if( $( this ).is( ".disabled" ) )
			{
				return;
			}
			var rowIndex = $table.find( "tbody tr.selected" ).index();
			if( (rowIndex >= 0 ) && (rowIndex < antennaArray.length) )
			{
				antennaMaintenance("block", antennaArray[rowIndex].AntennaId);
			}
		});

		$( "#unblockAntennaButton" ).click( function() {

			if( $( this ).is( ".disabled" ) )
			{
				return;
			}
			var rowIndex = $table.find( "tbody tr.selected" ).index();
			if( (rowIndex >= 0 ) && (rowIndex < antennaArray.length) )
			{
				antennaMaintenance("unblock", antennaArray[rowIndex].AntennaId);
			}
		});

		$( "#resetAntennaButton" ).click( function() {

			if( $( this ).is( ".disabled" ) )
			{
				return;
			}
			var rowIndex = $table.find( "tbody tr.selected" ).index();
			if( (rowIndex >= 0 ) && (rowIndex < antennaArray.length) )
			{
				antennaMaintenance("reset", antennaArray[rowIndex].AntennaId);
			}

		});

		$( "#AntennaPanel" ).bind( "closePanel", function()
		{
			if( typeof antennaXmlHttp !== "undefined" )
			{
				antennaXmlHttp.removeEventListener( "readystatechange", processAntennaInfoResponse, false );
				antennaXmlHttp.abort();
			}

			$( "#antennaDetailsDialog" ).trigger("close");
			$( "#AntennaPanel" ).hide();
		});

		// Antenna panel is hidden until opened by user.
		$( "#AntennaPanel" ).hide();
	});


	var SendAntennaInfoRequest = function()
	{
		// Create and send XMLHttpRequest to server
		try
		{
			antennaXmlHttp = new XMLHttpRequest();

			antennaXmlHttp.addEventListener( "readystatechange", processAntennaInfoResponse, false );
			antennaXmlHttp.open( "GET", "cgi-bin/AntennaInfoHandler.php", true );
			antennaXmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous get Antenna info request failed" );

			// Set timer and try again later.
			startAntennaInfoRequestTimer();
			return;
		}
	}


	var processAntennaInfoResponse = function()
	{
		if( (antennaXmlHttp.readyState == 4) && (antennaXmlHttp.status == 200) )
		{
			var AntennaInfoResponse;

			// Regardless of success or failure, set timer to send next request
			// after a wait interval.
			startAntennaInfoRequestTimer();

			try
			{
				// A successful response should contain a JSON object.
				AntennaInfoResponse = JSON.parse( antennaXmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processAntennaInfoResponse():\n" + exception );
				console.log( antennaXmlHttp.responseText );
				return;
			}

			if( !("statusInd" in AntennaInfoResponse) ||
				 (typeof AntennaInfoResponse.statusInd !== "string") )
			{
				console.log( "Invalid get antenna info response; missing status indicator" );
				return;
			}
			if( AntennaInfoResponse.statusInd === "Success" )
			{
				updateAntennas( AntennaInfoResponse );
			}
			else
			{

				// Failure has occurred. Read failure reason.
				if( !("_failReason" in AntennaInfoResponse) ||
					(typeof AntennaInfoResponse._failReason !== "string") )
				{
					console.log( "Invalid  AntennaInfoResponse; missing failure reason" );
					return;
				}

				console.log( "get antenna info failure reason: " + AntennaInfoResponse._failReason );

				if( AntennaInfoResponse._failReason === "invalidSession" )
				{
					stopAntennaInfoRequestTimer();

					// Configuration failed because the user session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}
			}
		}

		else if( antennaXmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive AntennaInfoResponse from server" );

			// Set timer and try again later.
			startAntennaInfoRequestTimer();
		}
	}

	var startAntennaInfoRequestTimer = function()
	{
		if( $( "#antennaTable" ).is( ":visible" ) )
		{
			timerId = setTimeout( function() {

				// Reset timer ID to undefined
				timerId = void( 0 );

				// Invoke call back function
				SendAntennaInfoRequest();

			}, ANTENNA_INFO_REQUEST_TIMER_INTERVAL );
		}

		// Else, panel has been closed while waiting for response from server; don't
		// start timer.
	}

	var stopAntennaInfoRequestTimer = function()
	{
		if( typeof timerId !== "undefined" )
		{
			clearTimeout( timerId );

			// Reset to undefined
			timerId = void( 0 );
		}
	}

	var updateAntennas = function( AntennaInfoResponse )
	{
		var sfnGroup1Cell = "";
		var sfnGroup2Cell = "";
		var sfnGroup1 = false;
		var sfnGroup2 = false;
		var sfnGroupInfoArray = [];
		sfnGroupInfoArray[0] = "N/A"
		sfnGroupInfoArray[1] = "N/A"

		if( AntennaInfoResponse == null )
		{
			return;
		}

		if( "CellInfo" in AntennaInfoResponse )
		{
			if( $.isArray( AntennaInfoResponse.CellInfo ) )
			{
				for( var i = 0; (i < AntennaInfoResponse.CellInfo.length && i < MAX_CELLS); i++ )
				{
					sfnGroupInfoArray[i] = "Cell ID=" + AntennaInfoResponse.CellInfo[i].CellID + ", " +
							"Physical cell ID (PCI)=" + AntennaInfoResponse.CellInfo[i].Pci + ", " +
							"EARFCN=" + AntennaInfoResponse.CellInfo[i].Earfcn;
					if (i == 0)
						sfnGroup1Cell = AntennaInfoResponse.CellInfo[i].CellID;
					else if (i == 1)
						sfnGroup2Cell = AntennaInfoResponse.CellInfo[i].CellID;
				}
			}
			else
			{
				sfnGroupInfoArray[0] = "Cell ID=" + AntennaInfoResponse.CellInfo.CellID + ", " +
						"Physical cell ID (PCI)=" + AntennaInfoResponse.CellInfo.Pci + ", " +
						"EARFCN=" + AntennaInfoResponse.CellInfo.Earfcn;

				sfnGroup1Cell = AntennaInfoResponse.CellInfo.CellID;
			}
		}

		// Clear out existing antenna array.
		while( antennaArray.length > 0 )
		{
			antennaArray.pop();
		}

		if( "AntennaInfo" in AntennaInfoResponse )
		{
			if( $.isArray( AntennaInfoResponse.AntennaInfo ) )
			{
				for( var i = 0; (i < AntennaInfoResponse.AntennaInfo.length && i < MAX_ANTENNAS); i++ )
				{
					sfnGroup1 = false;
					sfnGroup2 = false;

					if (AntennaInfoResponse.AntennaInfo[i].CellID1 != "") {
						if( AntennaInfoResponse.AntennaInfo[i].CellID1 == sfnGroup1Cell)
							sfnGroup1 = true;
						else if( AntennaInfoResponse.AntennaInfo[i].CellID1 == sfnGroup2Cell)
							sfnGroup2 = true;
					}
					if (AntennaInfoResponse.AntennaInfo[i].CellID2 != "") {
						if( AntennaInfoResponse.AntennaInfo[i].CellID2 == sfnGroup1Cell)
							sfnGroup1 = true;
						else if( AntennaInfoResponse.AntennaInfo[i].CellID2 == sfnGroup2Cell)
							sfnGroup2 = true;
					}

					var tmpAntennaEntry = new AntennaEntry(
						AntennaInfoResponse.AntennaInfo[i].AntennaId,
						AntennaInfoResponse.AntennaInfo[i].AntennaSiteId,
						AntennaInfoResponse.AntennaInfo[i].AntennaIpAddress,
						sfnGroup1,
						sfnGroup2,
						AntennaInfoResponse.AntennaInfo[i].OpState,
						AntennaInfoResponse.AntennaInfo[i].SerialNumber,
						AntennaInfoResponse.AntennaInfo[i].HwType,
						AntennaInfoResponse.AntennaInfo[i].WorkingState,
						AntennaInfoResponse.AntennaInfo[i].BlockState);
					antennaArray.push( tmpAntennaEntry );
				}
			}

			else
			{
				// Single, non-array type object.

				if (AntennaInfoResponse.AntennaInfo.CellID1 != "") {
					if( AntennaInfoResponse.AntennaInfo.CellID1 == sfnGroup1Cell)
						sfnGroup1 = true;
					else if( AntennaInfoResponse.AntennaInfo.CellID1 == sfnGroup2Cell)
						sfnGroup2 = true;
				}
				if (AntennaInfoResponse.AntennaInfo.CellID2 != "") {
					if( AntennaInfoResponse.AntennaInfo.CellID2 == sfnGroup1Cell)
						sfnGroup1 = true;
					else if( AntennaInfoResponse.AntennaInfo.CellID2 == sfnGroup2Cell)
						sfnGroup2 = true;
				}

				var tmpAntennaEntry = new AntennaEntry(
					AntennaInfoResponse.AntennaInfo.AntennaId,
					AntennaInfoResponse.AntennaInfo.AntennaSiteId,
					AntennaInfoResponse.AntennaInfo.AntennaIpAddress,
					sfnGroup1,
					sfnGroup2,
					AntennaInfoResponse.AntennaInfo.OpState,
					AntennaInfoResponse.AntennaInfo.SerialNumber,
					AntennaInfoResponse.AntennaInfo.Hwtype,
					AntennaInfoResponse.AntennaInfo.WorkingState,
					AntennaInfoResponse.AntennaInfo.BlockState);

				antennaArray.push( tmpAntennaEntry );
			}
		}

		$( "#AntennaPanel" ).find( "#sfnGroup1CellInfo" ).text( sfnGroupInfoArray[0] );
		$( "#AntennaPanel" ).find( "#sfnGroup2CellInfo" ).text( sfnGroupInfoArray[1] );

		// Display antennas in table; note: empty table displayed if no antenna
		displayAntennasTable();
	}

	var displayAntennasTable = function()
	{

		$table = $( "#antennaTable" );

		var rowsInTable = $table.find( "tbody > tr" ).length;

		// Clear out display table
		for( var i = 1; i <= rowsInTable; i++ )
		{
			var $row = $table.find( "tbody > tr:nth-of-type(" + i + ")" );
			clearAntennaRow( $row );
		}

		for( var i = 0; (i < antennaArray.length && i < rowsInTable); i++ )
		{
			var AntennaId_str ="";

			var $row = $table.find( "tbody > tr:nth-of-type(" + (i+1) + ")" );
			populateAntennaRow( $row, antennaArray[i] );
		}

		$( "#menuPanel" ).trigger( "contentPanelResized" );
	}

	var setupTableCells = function( $table )
	{
		var rowsInTable = $table.find( "tbody > tr" ).length;
		for( var i = 1; i <= rowsInTable; i++ )
		{
			var $row = $table.find( "tbody > tr:nth-of-type(" + i + ")" );
			$row.find( "td:first-child" ).addClass( "antenna-id cell-value" )
				.next().addClass( "antenna-site-id cell-value" )
				.next().addClass( "antenna-ip-address cell-value" )
				.next().addClass( "sfn-group-1" ).append( E( "div", "icon" ) )
				.next().addClass( "sfn-group-2" ).append( E( "div", "icon" ) )
				.next().addClass( "antenna-operational-state cell-value")
				.next().find( ".cell-value").addClass( "antenna-blocking-state" );
		}
	}

	var clearAntennaRow = function( $row )
	{
		$row.find( ".cell-value" ).text( "" );
		$row.find( ".sfn-group-member" ).removeClass( "sfn-group-member" );
	}

	var populateAntennaRow = function( $row, antennaEntry )
	{
		$row.find( ".antenna-id" ).text( antennaEntry.AntennaId );
		$row.find( ".antenna-site-id" ).text( antennaEntry.AntennaSiteId );
		$row.find( ".antenna-ip-address" ).text( antennaEntry.AntennaIpAddress );
		$row.find( ".antenna-operational-state" ).text( antennaEntry.OpState );
		$row.find( ".antenna-blocking-state" ).text( antennaEntry.BlockState );

		if( antennaEntry.SfnGroup1 )
		{
			$row.find( ".sfn-group-1" ).addClass( "sfn-group-member" );
		}

		if( antennaEntry.SfnGroup2 )
		{
			$row.find( ".sfn-group-2" ).addClass( "sfn-group-member" );
		}
	}

	var updateAndShowProgressIndicator = function( operation, stepName, displayText )
	{
		// displayText argument is "optional". Expected to be explicitly passed
		// by calling function for info and warning step indicators.
		if( typeof displayText === "undefined" )
		{
			switch( stepName )
			{
				case "current":
					displayText = "Performing operation";
					break;

				case "error":
					displayText = "Operation failed";
					break;

				case "success":
					displayText = "Operation complete";
					break;

				default:
					// Explicit display text should have been provided.
					console.log( "Unrecognized step: " + stepName );
					return;
			}
		}

		var classList = "step " + stepName;
		var $progressIndicator;

		if( operation == "block" )
			$progressIndicator = $( "#blockAntennaProgressIndicator" )
		else if( operation == "unblock" )
			$progressIndicator = $( "#unblockAntennaProgressIndicator" )
		else if( operation == "reset" )
			$progressIndicator = $( "#resetAntennaProgressIndicator" )
		else {
			console.log( "Invalid operation: " + operation );
			return;
		}

		$progressIndicator.removeClass()
		$progressIndicator.addClass( classList )
		$progressIndicator.find( ".text" ).text( displayText ).end()
		$progressIndicator.show();
	}

	var sendBlockingStateChangeOrResetRequest = function( operation, antennaId )
	{
		try
		{
			xmlHttp = new XMLHttpRequest();
			var csrfToken = $( "#rttPanel" ).find( "input[name='csrfToken']" ).val();
			var csrfTokenEncoded = encodeURIComponent( csrfToken );
			var postData = "";
			var url = "";

			if( operation == "block" ) {
				var state = "blocked";
				xmlHttp.addEventListener( "readystatechange", processBlockStateChangeResponse, false );
				postData = "bs=" + state + "&antModId=" + antennaId + "&ctkn=" + csrfTokenEncoded;
				url = "cgi-bin/BlockUnblockHandler.php";
			}
			else if( operation == "unblock") {
				var state = "unblocked";
				xmlHttp.addEventListener( "readystatechange", processUnblockStateChangeResponse, false );
				postData = "bs=" + state + "&antModId=" + antennaId + "&ctkn=" + csrfTokenEncoded;
				url = "cgi-bin/BlockUnblockHandler.php";
			}
			else if( operation == "reset") {
				xmlHttp.addEventListener( "readystatechange", processAntennaResetRequestResponse, false );
				postData = "antModId=" + antennaId + "&ctkn=" + csrfTokenEncoded;
				url = "cgi-bin/ResetSite.php";
			}
			else {
				console.log( "Invalid operation: " + operation );
				return;
			}

			xmlHttp.open( "POST", url, true );
			xmlHttp.setRequestHeader( "Content-Type", "application/x-www-form-urlencoded" );
			xmlHttp.setRequestHeader( "Content-length", postData.length );

			xmlHttp.send(postData);
		}
		catch( exception )
		{
			console.log( "Asynchronous state change request failed for " + operation );
			updateAndShowProgressIndicator(operation, "error" );
			updateButtonsOnCompletion( operation );

			return;
		}
	}

	var updateButtonsOnCompletion = function( operation )
	{
		var $confirmButton;
		var $closeButton;

		if( operation == "block") {
			$confirmButton = $( "#blockAntennaConfirmButton" );
			$closeButton = $( "#blockAntennaCloseButton" );
		}
		else if( operation == "unblock") {
			$confirmButton = $( "#unblockAntennaConfirmButton" );
			$closeButton = $( "#unblockAntennaCloseButton" );
		}
		else if( operation == "reset") {
			$confirmButton = $( "#resetAntennaConfirmButton" );
			$closeButton = $( "#resetAntennaCloseButton" );
		}

		$confirmButton.hide();
		$closeButton.removeClass( "disabled" );
		$closeButton.find( ".button-content" ).text( "Close" );
		$closeButton.focus();
	}

	var processBlockStateChangeResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			var stateChangeResponse;
			try
			{
				// A valid response should contain a JSON object.
				stateChangeResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processBlockStateChangeResponse():\n" + exception );
				console.log( xmlHttp.responseText );

				updateAndShowProgressIndicator( "block", "error" );
				updateButtonsOnCompletion( "block" );
				return;
			}

			if( !("statusInd" in stateChangeResponse) ||
				 (typeof stateChangeResponse.statusInd !== "string") )
			{
				console.log( "Invalid block state change response; missing status indicator " );
				updateAndShowProgressIndicator( "block", "error" );
				updateButtonsOnCompletion( "block" );
				return;
			}

			if( stateChangeResponse.statusInd === "Success" )
			{
				// Update progress indication.
				updateAndShowProgressIndicator( "block", "success" );
				updateButtonsOnCompletion( "block" );
			}
			else
			{
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in stateChangeResponse) ||
					 (typeof stateChangeResponse._failReason !== "string") )
				{
					console.log( "Invalid block state change response; missing failure reason" );
					updateAndShowProgressIndicator( "block", "error" );
					updateButtonsOnCompletion( "block" );
					return;
				}

				console.log( "Block failure reason: " + stateChangeResponse._failReason );

				if( (stateChangeResponse._failReason === "invalidSession"))
				{
						updateAndShowProgressIndicator( "block", "warning", "Session expired" );

					// User session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}

				if(stateChangeResponse._failReason === "invalidSecurityToken")
				{
					updateAndShowProgressIndicator( "block", "warning", "Security Token expired" );

					// User session has expired.
					window.alert( "Security token has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}

				// Procedure failed.
				updateAndShowProgressIndicator( "block", "error" );
				updateButtonsOnCompletion( "block" );
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive block state change response from server" );

			// Update progress indication.
			updateAndShowProgressIndicator( "block", "error" );
			updateButtonsOnCompletion( "block" );
		}
	}

	var processUnblockStateChangeResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			var stateChangeResponse;
			try
			{
				// A valid response should contain a JSON object.
				stateChangeResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processUnblockStateChangeResponse():\n" + exception );
				console.log( xmlHttp.responseText );

				updateAndShowProgressIndicator( "unblock", "error" );
				updateButtonsOnCompletion( "unblock" );
				return;
			}

			if( !("statusInd" in stateChangeResponse) ||
				 (typeof stateChangeResponse.statusInd !== "string") )
			{
				console.log( "Invalid unblock state change response; missing status indicator " );
				updateAndShowProgressIndicator( "unblock", "error" );
				updateButtonsOnCompletion( "unblock" );
				return;
			}

			if( stateChangeResponse.statusInd === "Success" )
			{
				// Update progress indication.
				updateAndShowProgressIndicator( "unblock", "success" );
				updateButtonsOnCompletion( "unblock" );
			}
			else
			{
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in stateChangeResponse) ||
					 (typeof stateChangeResponse._failReason !== "string") )
				{
					console.log( "Invalid unblock state change response; missing failure reason" );
					updateAndShowProgressIndicator( "unblock", "error" );
					updateButtonsOnCompletion( "unblock" );
					return;
				}

				console.log( "Unblock failure reason: " + stateChangeResponse._failReason );

				if( (stateChangeResponse._failReason === "invalidSession"))
				{
						updateAndShowProgressIndicator( "unblock", "warning", "Session expired" );

					// User session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}

				if(stateChangeResponse._failReason === "invalidSecurityToken")
				{
					updateAndShowProgressIndicator( "unblock", "warning", "Security Token expired" );

					// User session has expired.
					window.alert( "Security token has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}

				// Procedure failed.
				updateAndShowProgressIndicator( "unblock", "error" );
				updateButtonsOnCompletion( "unblock" );
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive unblock state change response from server" );

			// Update progress indication.
			updateAndShowProgressIndicator( "unblock", "error" );
			updateButtonsOnCompletion( "unblock" );
		}
	}

	var processAntennaResetRequestResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			var resetRequestResponse;
			try
			{
				resetRequestResponse = JSON.parse(xmlHttp.responseText);
			}
			catch (exception)
			{
				console.log( "JSON parsing error in processAntennaResetRequestResponse():\n" + exception );
				console.log( xmlHttp.responseText );
				updateAndShowProgressIndicator( "reset", "error" );
				updateButtonsOnCompletion( "reset" );
				return;
			}

			if(!("statusInd" in resetRequestResponse) ||
				(typeof resetRequestResponse.statusInd !== "string"))
			{
				console.log( "Invalid antenna reset site response; missing status indicator" );
				updateAndShowProgressIndicator( "reset", "error" );
				updateButtonsOnCompletion( "reset" );
				return;
			}
			if(resetRequestResponse.statusInd === "Success")
			{
				updateAndShowProgressIndicator( "reset", "success", "Reset initiated" );
				updateButtonsOnCompletion( "reset" );
			}
			else
			{
				if( !("_failReason" in resetRequestResponse) ||
					(typeof resetRequestResponse._failReason !== "string") )
				{
					console.log( "Invalid antenna reset response; missing failure reason" );
					updateAndShowProgressIndicator( "reset", "error" );
					updateButtonsOnCompletion( "reset" );
					return;
				}

				console.log("Antenna reset req failure reason: " + resetRequestResponse._failReason);

				if(resetRequestResponse._failReason === "invalidSession"){
					updateAndShowProgressIndicator( "reset", "warning", "Session expired" );
					window.alert( "Session has expired - Please login again" );
					location.assign( "login.php?expired" );
					return;
				}

				if(resetRequestResponse._failReason === "invalidSecurityToken")
				{
					updateAndShowProgressIndicator( "reset", "warning", "Security Token expired" );
					window.alert( "Security token has expired - Please login again" );
					location.assign( "login.php?expired" );
					return;
				}

				if(resetRequestResponse._failReason === "busy")
				{
					updateAndShowProgressIndicator( "reset", "error", "Site busy");
					$( "#resetAntennaConfirmButton" ).removeClass( "disabled" ).show();
					updateButtonsOnCompletion( "reset" );
				}
				// Procedure failed.
				else
				{
					updateButtonsOnCompletion( "reset" );
					updateAndShowProgressIndicator( "reset", "error" );
				}
			}
		}
		else if( xmlHttp.readyState == 4 )
		{
			console.log( "Failed to receive antenna reset response from server" );
			updateAndShowProgressIndicator( "reset", "error" );
			updateButtonsOnCompletion( "reset" );
		}
	}

	var antennaMaintenance =  function( operation, antennaId )
	{
		var $operationDialog;
		var $closeButton;
		var $confirmButton;
		var $progressIndicator;

		if( operation == "block")
		{
			$operationDialog = $( "#blockAntennaDialog" );
			$closeButton = $("#blockAntennaCloseButton");
			$confirmButton = $("#blockAntennaConfirmButton");
			$progressIndicator = $("#blockAntennaProgressIndicator");
		}
		else if( operation == "unblock")
		{
			$operationDialog = $( "#unblockAntennaDialog" );
			$closeButton = $("#unblockAntennaCloseButton");
			$confirmButton = $("#unblockAntennaConfirmButton");
			$progressIndicator = $("#unblockAntennaProgressIndicator");
		}
		else if( operation == "reset")
		{
			$operationDialog = $( "#resetAntennaDialog" );
			$closeButton = $("#resetAntennaCloseButton");
			$confirmButton = $("#resetAntennaConfirmButton");
			$progressIndicator = $("#resetAntennaProgressIndicator");
		}
		else
		{
			console.log( "Error, unexpected antenna operation: " + operation);
            return;
		}

		if( $operationDialog.is( ":visible" ) )
		{
			return;
		}

		$progressIndicator.hide();
		$confirmButton.removeClass( "disabled" ).show();

		$closeButton.removeClass("disabled");
		$closeButton.find(".button-content" ).text( "Cancel" );

		$operationDialog.trigger( "open" );

		// Set focus to close/cancel button once dialog has opened.
		setTimeout( function()
		{
			$closeButton.focus();
		}, 100 );


		$confirmButton.off("click");
		$confirmButton.click( function()
		{
			if( $( this ).is( ".disabled" ) )
			{
				return;
			}

			$closeButton.addClass("disabled");
			$( this ).addClass( "disabled" );
			updateAndShowProgressIndicator( operation, "current" );
			sendBlockingStateChangeOrResetRequest( operation, antennaId);

		});

		$closeButton.click( function()
		{
			if( $( this ).is( ".disabled" ) )
			{
				return;
			}
			$operationDialog.trigger( "close" );
		});

		$operationDialog.bind( "close", function()
		{
			// Stop any ongoing request.
			// Note, the dialog window itself is closed by handler in webuiDialogs.js
			if( typeof xmlHttp !== "undefined" )
			{
				xmlHttp.removeEventListener( "readystatechange", processBlockingStateChangeResponse, false );
				xmlHttp.abort();
			}
		});
	}

})( jQuery )
