// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		CertificateManagement.js
//
// ****************************************************************************

(function( $ )
{
	var TIMER_INTERVAL = 10000;
	var timerId;	
	var NUM_ADD_OR_DELETE_ROWS = 1;
	var MAX_CERTIFICATES = 17;
	var MIN_NUM_TABLE_ROWS = 2;
	var certXmlHttp;
	var xmlHttp;
	var certificatesArray = [];
	var updateInProgress = false;
	var initInProgress = false;
        var strNewLine = "\n";

	var startProgressIndicatorTimer = function()
	{
		timerId = setTimeout( function() {

			// Reset timer ID to undefined
			timerId = void( 0 );
			// hide the progress indicator
			$( "#Certificate_Indicator" ).parent().hide();	

		}, TIMER_INTERVAL );

	}

	var hideProgressIndicator = function()
	{
		if( typeof timerId !== "undefined" )
		{
			clearTimeout( timerId );
			$( "#Certificate_Indicator" ).parent().hide();	
			// Reset to undefined
			timerId = void( 0 );
		}
	}

	//CertificateExtensionInfo
	var CertificateExtensionInfo = function (ExtensionName, ExtensionValue)
	{
		this.ExtensionName = initializeStringValue( ExtensionName );
		this.ExtensionValue = initializeStringValue( ExtensionValue );
	}

	// CertificateEntry object
	var CertificateEntry = function( Index, Subject, Issuer, Type, ValidFrom, ValidTo, SerialNumber, Fingerprint, CertificateExtensionInfo )
	{
		this.Index = initializeStringValue( Index );
		this.Subject = initializeStringValue( Subject );
		this.Issuer = initializeStringValue( Issuer );
		this.Type = initializeStringValue( Type );
		this.ValidFrom = initializeStringValue( ValidFrom );
		this.ValidTo = initializeStringValue( ValidTo );
		this.SerialNumber = initializeStringValue( SerialNumber );
		this.Fingerprint = initializeStringValue( Fingerprint );

		this.CertificateExtensionInfo = CertificateExtensionInfo;
	}

	//Helper function to proccess Certificate Detials Array
	var getCertificateExtensionInfo =  function(ExtensionInfo)
	{
		var tmpCertificateExtensionInfo = [];
		if( $.isArray(ExtensionInfo) )
		{
			for( var i = 0; i < ExtensionInfo.length ; i++ )
			{
				var tmpDetails = new CertificateExtensionInfo(ExtensionInfo[i].ExtensionName,
									          ExtensionInfo[i].ExtensionValue);
				tmpCertificateExtensionInfo.push(tmpDetails);
			}
		}
		else
		{
                    	var tmpDetails = new CertificateExtensionInfo(ExtensionInfo.ExtensionName,
                                                                          ExtensionInfo.ExtensionValue);
                     	tmpCertificateExtensionInfo.push(tmpDetails);

		}
		return tmpCertificateExtensionInfo;
	}


	var updateAndShowRefreshIndicator = function( stepName, displayText )
	{
		// displayText argument is "optional". Expected to be explicitly passed
		// by calling function for info and warning step indicators.
		if( typeof displayText === "undefined" )
		{
			switch( stepName )
			{
				case "current":
					displayText = "";
					break;

				case "error":
					displayText = "Failed to retrieve data from server";
					break;

				case "success":
					displayText = "";
					break;

				default:
					// Explicit display text should have been provided.
					console.log( "Unrecognized step: " + stepName );
					return;
			}
		}

		var classList = "step " + stepName;

		$( "#certificatesRefreshIndicator" )
			.removeClass()
			.addClass( classList )
			.find( ".text" ).text( displayText ).end()
			.show();
	}


	var hideRefreshIndicator = function()
	{
		$( "#certificatesRefreshIndicator" ).hide();
	}


	var updateAndShowProgressIndicator = function( stepName, displayText )
	{
		// displayText argument is "optional". Expected to be explicitly passed
		// by calling function for info and warning step indicators.
		if( typeof displayText === "undefined" )
		{
			switch( stepName )
			{
				case "current":
					if(updateInProgress)
					{
						displayText = "Updating";
					}
					else
					{
						displayText = "Initializing";
					}
					break;
				case "success":
					if(updateInProgress)
					{
						displayText = "Update complete";
					}
					else
					{
						displayText = "Initialization complete";
					}
					startProgressIndicatorTimer();
					break;
				case "error":
					if(updateInProgress)
					{
						displayText = "Update failed";
					}
					else
					{
						displayText = "Initialization failed";
					}
					startProgressIndicatorTimer();
					break;
				default:
					// Explicit display text should have been provided.
					console.log( "Unrecognized step: " + stepName );
					return;
			}
		}

		var classList = "step " + stepName;

		$( "#Certificate_Indicator" )
			.removeClass()
			.addClass( classList )
			.find( ".text" ).text( displayText ).end().parent()
			.show();
	}

	var displayDetails = function( index )
	{
		var $dialog = $( "#certificateDetailsDialog" );

		if( $dialog.is( ":visible" ) )
		{
			return;
		}

		var text_val = "";
		var extension_text_val="";

		text_val += "Subject : ";
		text_val += certificatesArray[index].Subject;
		text_val += strNewLine;

		text_val += "Issuer : ";
		text_val += certificatesArray[index].Issuer;
		text_val += strNewLine;

		text_val += "Type : ";
		text_val += certificatesArray[index].Type;
		text_val += strNewLine;

		text_val += "Valid From : ";
		text_val += certificatesArray[index].ValidFrom;
		text_val += strNewLine;

		text_val += "Valid To : ";
		text_val += certificatesArray[index].ValidTo;
		text_val += strNewLine;

		text_val += "Serial Number : ";
		text_val += certificatesArray[index].SerialNumber;
		text_val += strNewLine;

		text_val += "Fingerprint : ";
		text_val += certificatesArray[index].Fingerprint;

		for( var i = 0; i < certificatesArray[index].CertificateExtensionInfo.length; i++ )
		{
			var extension_name = certificatesArray[index].CertificateExtensionInfo[i].ExtensionName.replace(/^\s*[\r\n]/gm, "");
			extension_text_val += extension_name;
			extension_text_val += strNewLine;
			var extension_val = certificatesArray[index].CertificateExtensionInfo[i].ExtensionValue.replace(/^\s*[\r\n]/gm, "");
			extension_text_val += extension_val;
			if( i < (certificatesArray[index].CertificateExtensionInfo.length - 1) )
			{
				extension_text_val += strNewLine;
				extension_text_val += strNewLine;
			}
		}
		$( "#certificateDetailsText" ).val(text_val);
		$( "#extensionDetailsText" ).val(extension_text_val);
		$dialog.trigger("open");
		$( "#CertificatePanelDimmer" ).show();
	}

	NSNReady( function()
	{
		// Setup certificates table.
		var $table = $( "#certificatesTable" );
		$table.standard_table( MIN_NUM_TABLE_ROWS, MAX_CERTIFICATES );

		$( "#certificateDetailsDialog" ).bind("close", function()
		{
			$( "#certificateDetailsText" ).val("");
			$( "#extensionDetailsText" ).val("");
			
			$( "#CertificatePanelDimmer" ).hide();
		});

		$( "#certificateManagementLink" ).click( function() {

			if( $( "#CertificatePanel" ).is( ":visible" ) )
			{
				return;
			}

			// Hide any open panel(s) in the display area.
			$( ".content-display-panel" ).not( "#CertificatePanel" ).trigger( "closePanel" );

			// Show the Certificate panel in the display area.
			$( "#CertificatePanel" ).show();
			$( "#certificateDetailsButton" ).addClass( "disabled" );
			$( "#CmpKeyUpdateButton" ).addClass( "disabled" );
			$( "#initializeCertificatesButton" ).addClass( "disabled" );
			$( "#refreshCertificatesButton" ).addClass( "disabled" );

			//bind certificate detail button to selectionEvent
			$table.bind("selectionEvent", function()
			{
				var $selected_row = $table.find( "tbody tr.selected" );
				if( $selected_row.length > 0 )
				{
					var rowIndex = $selected_row.index();
					if( ( rowIndex >= 0 )  && ( rowIndex < certificatesArray.length ) )
					{
						$( "#certificateDetailsButton" ).removeClass( "disabled" );
					}
					else
					{
						$( "#certificateDetailsButton" ).addClass( "disabled" );
					}
				}
				else
				{
					$( "#certificateDetailsButton" ).addClass( "disabled" );
				}
			});

                        $table.bind("openEvent", function()
                        {
                                var $selected_row = $table.find( "tbody tr.selected" );
                                if( $selected_row.length > 0 )
                                {
					var rowIndex = $selected_row.index();
					if( ( rowIndex >= 0 )  && ( rowIndex < certificatesArray.length ) )
					{
						displayDetails(rowIndex);
					}
                                }
                        });

			$( "#CertificatePanelDimmer" ).hide();

			hideProgressIndicator();
			updateAndShowRefreshIndicator( "current" );
			CertificateInfoRequest();
		});

		$( "#CertificatePanel" ).find( "#Certificate_Indicator" ).parent().hide();

		$( "#CmpKeyUpdateButton" ).click( function() {

			if( $( this ).is( ".disabled" ) )
			{
				return;
			}

			//hide the indicator if the previous result is still there
			hideProgressIndicator();
			$( this ).addClass( "disabled" );
			$( "#initializeCertificatesButton" ).addClass("disabled");
			$( "#refreshCertificatesButton" ).addClass("disabled");
			updateInProgress = true;
			updateAndShowProgressIndicator("current");
			CmpKeyUpdateRequest();
		});

		$( "#initializeCertificatesButton" ).click( function() {

			if( $( this ).is( ".disabled" ) )
			{
				return;
			}

			//hide the indicator if the previous result is still there
			hideProgressIndicator();
			$( this ).addClass( "disabled" );
			$( "#CmpKeyUpdateButton" ).addClass( "disabled" );
			$( "#refreshCertificatesButton" ).addClass("disabled");
			initInProgress = true;
			updateAndShowProgressIndicator("current");
			InitCertsRequest();
		});

		$( "#refreshCertificatesButton" ).click( function() {

			if( $( this ).is( ".disabled" ) )
			{
				return;
			}

			$( this ).addClass( "disabled" );
			if( $( "#CmpServerStatus" ).text() == "Configured"  )
			{
				$( "#CmpKeyUpdateButton" ).addClass( "disabled" );
				$( "#initializeCertificatesButton" ).addClass("disabled");
			}

			hideProgressIndicator();
			updateAndShowRefreshIndicator( "current" );
			CertificateInfoRequest();
		});

		$( "#certificateDetailsButton" ).click( function() {

			if( $( this ).is( ".disabled" ) )
			{
				return;
			}
			var rowIndex = $table.find( "tbody tr.selected" ).index();
			if( (rowIndex >= 0 ) && (rowIndex < certificatesArray.length) )
			{
			 	displayDetails(rowIndex);
			}

		});

		$( "#CertificatePanel" ).bind( "closePanel", function()
		{
			if( typeof certXmlHttp !== "undefined" )
			{
				certXmlHttp.removeEventListener( "readystatechange", processCertificateInfoResponse, false );
				certXmlHttp.abort();
			}
			if( typeof xmlHttp !== "undefined" )
			{
				xmlHttp.removeEventListener( "readystatechange", processCmpKeyUpdateResponse, false );
				xmlHttp.removeEventListener( "readystatechange", processInitCertsResponse, false );
				xmlHttp.abort();
			} 

			$( "#certificateDetailsDialog" ).trigger("close");
			$( "#CertificatePanel" ).hide();
		});

		// Certificate panel is hidden until opened by user.
		$( "#CertificatePanel" ).hide();
	});


	var CertificateInfoRequest = function()
	{
		// Create and send XMLHttpRequest to server
		try
		{
			certXmlHttp = new XMLHttpRequest();

			// Register event listener for response
			certXmlHttp.addEventListener( "readystatechange", processCertificateInfoResponse, false );

			// GET information asynchronously from server
			certXmlHttp.open( "GET", "cgi-bin/CertificateInfoHandler.php", true );
			certXmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous get Certificates connection request failed" );
			$( "#refreshCertificatesButton" ).removeClass( "disabled" );
			if( $( "#CmpServerStatus" ).text() == "Configured"  )
			{
				$( "#CmpKeyUpdateButton" ).removeClass( "disabled" );
				$( "#initializeCertificatesButton" ).removeClass("disabled");
			}
			updateAndShowRefreshIndicator( "error" );
			return;
		}
	}


	var processCertificateInfoResponse = function()
	{
		if( (certXmlHttp.readyState == 4) && (certXmlHttp.status == 200) )
		{
			$( "#refreshCertificatesButton" ).removeClass( "disabled" );
                        if( $( "#CmpServerStatus" ).text() == "Configured"  )
                        {
                                $( "#CmpKeyUpdateButton" ).removeClass( "disabled" );
                                $( "#initializeCertificatesButton" ).removeClass("disabled");
                        }

			hideRefreshIndicator();

			var CertificateInfoResponse;

			try
			{
				// A successful response should contain a JSON object.
				CertificateInfoResponse = JSON.parse( certXmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processCertificateInfoResponse():\n" + exception );
				console.log( certXmlHttp.responseText );
				updateAndShowRefreshIndicator( "error" );
				return;
			}


			if( !("statusInd" in CertificateInfoResponse) ||
			     (typeof CertificateInfoResponse.statusInd !== "string") )
			{
				console.log( "Invalid get Certificates response; missing status indicator" );
				updateAndShowRefreshIndicator( "error" );
				return;
			}

			if( CertificateInfoResponse.statusInd === "Success" )
			{
				updateCertificates( CertificateInfoResponse );
			}
			else
			{
				updateAndShowRefreshIndicator( "error" );

				// Failure has occurred. Read failure reason.
				if( !("_failReason" in CertificateInfoResponse) ||
				     (typeof CertificateInfoResponse._failReason !== "string") )
				{
					console.log( "Invalid  CertificateInfoResponse; missing failure reason" );
					return;
				}

				console.log( "get Certificates failure reason: " + CertificateInfoResponse._failReason );

				if( CertificateInfoResponse._failReason === "invalidSession" )
				{
					// Configuration failed because the user session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}
			}
		}

		else if( certXmlHttp.readyState == 4 )
		{
			$( "#refreshCertificatesButton" ).removeClass( "disabled" );
			if( $( "#CmpServerStatus" ).text() == "Configured"  )
			{
                                $( "#CmpKeyUpdateButton" ).removeClass( "disabled" );
                                $( "#initializeCertificatesButton" ).removeClass("disabled");
			}
			// HTTP level failure has occurred.
			console.log( "Failed to receive CertificateInfoResponse from server" );
			updateAndShowRefreshIndicator( "error" );
		}
	}

	var CmpKeyUpdateRequest = function()
	{
		// Create and send XMLHttpRequest to server
		try
		{
			xmlHttp = new XMLHttpRequest();

			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processCmpKeyUpdateResponse, false );

			// GET information asynchronously from server
			xmlHttp.open( "GET", "cgi-bin/CmpKeyUpdateHandler.php", true );
			xmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous CmpKeyUpdate connection request failed" );
                	$( "#CmpKeyUpdateButton" ).removeClass( "disabled" );
                	$( "#initializeCertificatesButton" ).removeClass( "disabled" );
			$( "#refreshCertificatesButton" ).removeClass("disabled");
			updateAndShowProgressIndicator("error");
                	updateInProgress = false;	
			return;
		}
	}

	var processCmpKeyUpdateResponse = function()
	{

		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
                	$( "#CmpKeyUpdateButton" ).removeClass( "disabled" );
               	 	$( "#initializeCertificatesButton" ).removeClass( "disabled" );
			$( "#refreshCertificatesButton" ).removeClass("disabled");

			var CmpKeyUpdateResponse;

			try
			{
				// A successful response should contain a JSON object.
				CmpKeyUpdateResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processCmpKeyUpdateResponse():\n" + exception );
				console.log( xmlHttp.responseText );
				updateAndShowProgressIndicator("error");
				updateInProgress = false;
				return;
			}


			if( !("statusInd" in CmpKeyUpdateResponse) ||
			     (typeof CmpKeyUpdateResponse.statusInd !== "string") )
			{
				console.log( "Invalid CmpKeyUpdate response; missing status indicator" );
				updateAndShowProgressIndicator("error");
				updateInProgress = false;
				return;
			}

			if( CmpKeyUpdateResponse.statusInd === "Success" )
			{
				updateAndShowProgressIndicator("success");
				updateInProgress = false;
			}
			else
			{
				updateAndShowProgressIndicator("error");
				updateInProgress = false;
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in CmpKeyUpdateResponse) ||
				     (typeof CmpKeyUpdateResponse._failReason !== "string") )
				{
					console.log( "Invalid  CmpKeyUpdateResponse; missing failure reason" );
					return;
				}

				console.log( "CmpKey Update failure reason: " + CmpKeyUpdateResponse._failReason );

				if( CmpKeyUpdateResponse._failReason === "invalidSession" )
				{
					// Configuration failed because the user session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}
			}
		}

		else if( xmlHttp.readyState == 4 )
		{

                	$( "#CmpKeyUpdateButton" ).removeClass( "disabled" );
               	 	$( "#initializeCertificatesButton" ).removeClass( "disabled" );
			$( "#refreshCertificatesButton" ).removeClass("disabled");
			updateAndShowProgressIndicator("error");
			updateInProgress = false;
			// HTTP level failure has occurred.
			console.log( "Failed to receive CmpKeyUpdateResponse from server" );
		}
	}

	var InitCertsRequest = function()
	{
		// Create and send XMLHttpRequest to server
		try
		{
			xmlHttp = new XMLHttpRequest();

			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processInitCertsResponse, false );

			// GET information asynchronously from server
			xmlHttp.open( "GET", "cgi-bin/InitCertsHandler.php", true );
			xmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous Initialize Certificates request failed" );
                	$( "#CmpKeyUpdateButton" ).removeClass( "disabled" );
                	$( "#initializeCertificatesButton" ).removeClass( "disabled" );
			$( "#refreshCertificatesButton" ).removeClass("disabled");
			updateAndShowProgressIndicator("error");
                	initInProgress = false;
			return;
		}
	}

	var processInitCertsResponse = function()
	{

		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
                	$( "#CmpKeyUpdateButton" ).removeClass( "disabled" );
                	$( "#initializeCertificatesButton" ).removeClass( "disabled" );
			$( "#refreshCertificatesButton" ).removeClass("disabled");
			var InitCertsResponse;

			try
			{
				// A successful response should contain a JSON object.
				InitCertsResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processInitCertsResponse():\n" + exception );
				console.log( xmlHttp.responseText );
				updateAndShowProgressIndicator("error");
				initInProgress = false;
				return;
			}


			if( !("statusInd" in InitCertsResponse) ||
			     (typeof InitCertsResponse.statusInd !== "string") )
			{
				console.log( "Invalid InitCerts response; missing status indicator" );
				updateAndShowProgressIndicator("error");
				initInProgress = false;
				return;
			}

			if( InitCertsResponse.statusInd === "Success" )
			{
				updateAndShowProgressIndicator("success");
				initInProgress = false;
			}
			else
			{
				updateAndShowProgressIndicator("error");
				initInProgress = false;
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in InitCertsResponse) ||
				     (typeof InitCertsResponse._failReason !== "string") )
				{
					console.log( "Invalid  InitCertsResponse; missing failure reason" );
					return;
				}

				console.log( "Initialize Certificates failure reason: " + InitCertsResponse._failReason );

				if( InitCertsResponse._failReason === "invalidSession" )
				{
					// Configuration failed because the user session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
                	$( "#CmpKeyUpdateButton" ).removeClass( "disabled" );
                	$( "#initializeCertificatesButton" ).removeClass( "disabled" );
			$( "#refreshCertificatesButton" ).removeClass("disabled");
			updateAndShowProgressIndicator("error");
               	 	initInProgress = false;
			// HTTP level failure has occurred.
			console.log( "Failed to receive InitCertsResponse from server" );
		}

	}

	var updateCertificates = function( CertificateInfoResponse )
	{
		if( CertificateInfoResponse == null )
		{
			return;
		}

		if( !("CmpServerStatus" in CertificateInfoResponse) )
		{
			console.log( "Invalid CertificateInfoResponse; missing CmpServerStatus flag" );
			updateAndShowRefreshIndicator( "error" );
			return;
		}

		displayCmpServerStatus( CertificateInfoResponse.CmpServerStatus );

		// Clear out existing certificates array.
		while( certificatesArray.length > 0 )
		{
			certificatesArray.pop();
		}

		if( "CertificateInfo" in CertificateInfoResponse )
		{
			if( $.isArray( CertificateInfoResponse.CertificateInfo ) )
			{
				for( var i = 0; (i < CertificateInfoResponse.CertificateInfo.length && i < MAX_CERTIFICATES); i++ )
				{
					var tmpExtensionInfo = new Array();
					if("ExtensionInfo" in CertificateInfoResponse.CertificateInfo[i] )
					{
						tmpExtensionInfo = getCertificateExtensionInfo(CertificateInfoResponse.CertificateInfo[i].ExtensionInfo)
					}

					var tmpCertificateEntry = new CertificateEntry(
						CertificateInfoResponse.CertificateInfo[i].Index,
						CertificateInfoResponse.CertificateInfo[i].Subject,
						CertificateInfoResponse.CertificateInfo[i].Issuer,
						CertificateInfoResponse.CertificateInfo[i].Type,
						CertificateInfoResponse.CertificateInfo[i].ValidFrom,
						CertificateInfoResponse.CertificateInfo[i].ValidTo,
						CertificateInfoResponse.CertificateInfo[i].SerialNum,
						CertificateInfoResponse.CertificateInfo[i].FingerPrint,
						tmpExtensionInfo);
					certificatesArray.push( tmpCertificateEntry );
				}

			}

			else
			{
				// Single, non-array type object.
				var tmpExtensionInfo = new Array();
				if("ExtensionInfo" in CertificateInfoResponse.CertificateInfo)
				{
					tmpExtensionInfo = getCertificateExtensionInfo(CertificateInfoResponse.CertificateInfo.ExtensionInfo)
				}

				var tmpCertificateEntry = new CertificateEntry(
					CertificateInfoResponse.CertificateInfo.Index,
					CertificateInfoResponse.CertificateInfo.Subject,
					CertificateInfoResponse.CertificateInfo.Issuer,
					CertificateInfoResponse.CertificateInfo.Type,
					CertificateInfoResponse.CertificateInfo.ValidFrom,
					CertificateInfoResponse.CertificateInfo.ValidTo,
					CertificateInfoResponse.CertificateInfo.SerialNum,
					CertificateInfoResponse.CertificateInfo.FingerPrint,
					tmpExtensionInfo);

				certificatesArray.push( tmpCertificateEntry );
			}
		}

		// Display Certificates in table; note: empty table displayed if no certificates
		displayCertificatesTable();
	}


	var displayCmpServerStatus = function( CmpServerStatus )
	{

		// Based on CmpServerStatus flag to Enable/disable "CMP Key Update" and "Initilize Certificates" buttons
		if( typeof CmpServerStatus !== "string" )
		{
			CmpServerStatus = "notconfigured";
		} 

		switch( CmpServerStatus.toLowerCase() )
		{
			case "1":
			case "configured":
				$( "#CmpServerStatus" ).text("Configured");
				$( "#CmpKeyUpdateButton" ).removeClass("disabled");
				$( "#initializeCertificatesButton" ).removeClass("disabled");
				break;

			case "0":
			case "notconfigured":
				$( "#CmpServerStatus" ).text("Not configured");
				$( "#CmpKeyUpdateButton" ).addClass("disabled");
				$( "#initializeCertificatesButton" ).addClass("disabled");
				break; 

			default:
				$( "#CmpServerStatus" ).text("Not configured");
				$( "#CmpKeyUpdateButton" ).addClass("disabled");
				$( "#initializeCertificatesButton" ).addClass("disabled");
				break;
		}

	}

	var displayCertificatesTable = function()
	{

		$table = $( "#certificatesTable" );

		var rowsInTable = $table.find( "tbody > tr" ).length;

		// Clear out display table
		for( var i = 1; i <= rowsInTable; i++ )
		{
			var $row = $table.find( "tbody > tr:nth-of-type(" + i + ")" );
			clear_certificate_row( $row );
		}

		// Check if we need to add rows.
		while( (certificatesArray.length > rowsInTable) && (rowsInTable < MAX_CERTIFICATES) )
		{
			// Add groups of rows
			$table.add_rows_to_table( NUM_ADD_OR_DELETE_ROWS, MAX_CERTIFICATES );
			rowsInTable = $table.find( "tbody > tr" ).length;
		}

		for( var i = 0; (i < certificatesArray.length && i < rowsInTable); i++ )
		{
			var Subject_str ="";
			
			var $row = $table.find( "tbody > tr:nth-of-type(" + (i+1) + ")" );
			populate_certificate_row( $row,
						certificatesArray[i].Index,
						certificatesArray[i].Subject,
                                                certificatesArray[i].Issuer,
                                                certificatesArray[i].Type,
                                                certificatesArray[i].ValidFrom,
                                                certificatesArray[i].ValidTo,
                                                certificatesArray[i].SerialNumber,
						certificatesArray[i].Fingerprint);
		}

		var numCertificates = certificatesArray.length;
		if( numCertificates < MIN_NUM_TABLE_ROWS )
		{
			numCertificates = MIN_NUM_TABLE_ROWS;
		}

		if( numCertificates > MAX_CERTIFICATES )
		{
			numCertificates = MAX_CERTIFICATES;
		}
		// Check if we can delete rows
		while( rowsInTable >= (numCertificates + NUM_ADD_OR_DELETE_ROWS) )
		{
			$table.remove_rows_from_table( NUM_ADD_OR_DELETE_ROWS, MIN_NUM_TABLE_ROWS );
			rowsInTable = $table.find( "tbody > tr" ).length;
		}

		// The width of the certificate table may have changed, which may impact the
		// panel width. Hide or show the menu panel accordingly.
		$( "#menuPanel" ).trigger( "contentPanelResized" );
	}

	var clear_certificate_row = function( $row )
	{
		$row.find( "td:not( :last-child )" ).text( "" );
		$row.find( "td:last-child" ).find( ".cell-value" ).text( "" );
	}


	var populate_certificate_row = function( $row,
						 Index,
						 Subject,
						 Issuer,
						 Type,
						 ValidFrom,
						 ValidTo,
						 SerialNumber,
						 Fingerprint )
	{

		var indent_format = Index + "em";

		$row.find( "td:first-child" ).addClass( "certificate-subject" ).text( Subject )
			.next().addClass( "certificate-issuer" ).text( Issuer )
			.next().addClass( "certificate-type" ).text( Type )
			.next().addClass( "certificate-validfrom" ).text( ValidFrom)
			.next().addClass( "certificate-validto").text( ValidTo)
			.next().addClass( "certificate-serialnum").text( SerialNumber)
			.next().find( ".cell-value").addClass( "certificate-fingerprint" ).text( Fingerprint );

		//add indent to certificate-subject
		$row.find( "td:first-child" ).css("text-indent", indent_format); 
	}

})( jQuery )
