// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		CollectSnapshot.js
//
// ****************************************************************************

( function( $ )
{

	var REQUEST_TIMER_INTERVAL = 10000;
	var xmlHttp;
	var timerId;
	var collectionInProgress = false;


	var startRequestTimer = function()
	{
		if( $( "#snapshotDialog" ).is( ":visible" ) )
		{
			timerId = setTimeout( function() {

				// Reset timer ID to undefined
				timerId = void( 0 );

				// Invoke call back function
				sendSnapshotDataRequest();

			}, REQUEST_TIMER_INTERVAL );
		}

		// Else, panel has been closed while waiting for response from server; don't
		// start timer.
	}

	var stopRequestTimer = function()
	{
		if( typeof timerId !== "undefined" )
		{
			clearTimeout( timerId );

			// Reset to undefined
			timerId = void( 0 );
		}
	}


	var sendSnapshotRequest = function()
	{
		// Create and send XMLHttpRequest to server.
		// This is the initial request to start collection.
		try
		{
			xmlHttp = new XMLHttpRequest();

			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processSnapshotResponse, false );

			var url = "cgi-bin/SnapshotRequestHandler.php";

			// Read selected log coverage level from user interface
			var logCoverage = $( "#snapshotDialog input[name=logCoverage]:checked" ).val();

			url += "?" + logCoverage;

			// GET information asynchronously from server
			xmlHttp.open( "GET", url, true );
			xmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous snapshot request failed" );
			handleFailure();
			return;
		}
	}


	var processSnapshotResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			var snapshotResponse;

			try
			{
				// A valid response should contain a JSON object.
				snapshotResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processSnapshotResponse():\n" + exception );
				console.log( xmlHttp.responseText );

				handleFailure();
				return;
			}


			if( !("statusInd" in snapshotResponse) ||
			     (typeof snapshotResponse.statusInd !== "string") )
			{
				console.log( "Invalid snapshot response; missing status indicator" );
				handleFailure();
				return;
			}

			if( snapshotResponse.statusInd === "Success" )
			{
				// Update progress indication.
				updateAndShowProgressIndicator( "current" );

				// Start periodic pings for updates.
				startRequestTimer();
			}
			else
			{
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in snapshotResponse) ||
				     (typeof snapshotResponse._failReason !== "string") )
				{
					console.log( "Invalid snapshot response; missing failure reason" );
					handleFailure();
					return;
				}

				console.log( "Snapshot failure reason: " + snapshotResponse._failReason );

				if( snapshotResponse._failReason === "invalidSession" )
				{
					// User session has expired.
					handleSessionTimeout();
					return;
				}

				else if( snapshotResponse._failReason === "rqstPending" )
				{
					// Flash warning message for 3s then display spinner and
					// start periodic pings for updates. This situation can
					// occur if, for example, one-shot stop request fails to
					// reach server.

					updateAndShowProgressIndicator( "warning", "Collection in progress; continuing" );

					setTimeout( function() {

						// Check if collection is still in progress; could have been
						// canceled while flashing warning.
						if( collectionInProgress )
						{
							updateAndShowProgressIndicator( "current" );
							startRequestTimer();
						}
					}, 3000 );

					return;
				}

				// Procedure failed.
				handleFailure();
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive snapshot response from server" );

			// Update progress indication and UI buttons.
			handleFailure();
		}
	}


	var sendSnapshotDataRequest = function()
	{
		// Create and send XMLHttpRequest to server. This is the periodic data request
		// sent continuously until snapshot collection completes.
		try
		{
			xmlHttp = new XMLHttpRequest();

			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processSnapshotDataResponse, false );

			// GET information asynchronously from server
			xmlHttp.open( "GET", "cgi-bin/SnapshotDataRequestHandler.php", true );
			xmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous snapshot data request failed" );

			// Set timer and try again later.
			startRequestTimer();
			return;
		}
	}


	var processSnapshotDataResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			var snapshotDataResponse;

			try
			{
				// A valid response should contain a JSON object.
				snapshotDataResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processSnapshotDataResponse():\n" + exception );
				console.log( xmlHttp.responseText );

				// Unexpected (and hopefully transient) failure. Set timer and try again later.
				startRequestTimer();
				return;
			}


			if( !("statusInd" in snapshotDataResponse) ||
			     (typeof snapshotDataResponse.statusInd !== "string") )
			{
				console.log( "Invalid snapshot data response; missing status indicator" );

				// Unlikely interface failure. Set timer and try again later.
				startRequestTimer();
				return;
			}


			if( snapshotDataResponse.statusInd === "Success" )
			{
				if( !("FileName" in snapshotDataResponse) ||
				     (typeof snapshotDataResponse.FileName !== "string") )
				{
					console.log( "Invalid snapshot data response; missing file name" );

					// Unlikely interface failure. Set timer and try again later.
					startRequestTimer();
					return;
				}

				// Collection is complete and snapshot file is ready for download.
				handleCollectionComplete( snapshotDataResponse.FileName );
			}

			else if( snapshotDataResponse.statusInd === "InProgress" )
			{
				// Snapshot collection still in progress. Keep waiting.
				startRequestTimer();
			}

			else
			{
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in snapshotDataResponse) ||
				     (typeof snapshotDataResponse._failReason !== "string") )
				{
					console.log( "Invalid snapshot data response; missing failure reason" );

					// Probably non-transient (compound) failure. End procedure.
					handleFailure();
					return;
				}

				console.log( "Snapshot data response failure reason: " + snapshotDataResponse._failReason );

				if( snapshotDataResponse._failReason === "invalidSession" )
				{
					// User session has expired.
					handleSessionTimeout();
					return;
				}

				// Non-transient failure. End procedure.
				handleFailure();
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive snapshot data response from server" );

			// Possibly transient failure. Try again next interval.
			startRequestTimer();
		}
	}


	var sendSnapshotStopRequest = function()
	{
		// Create and send XMLHttpRequest to server. This is a request to cancel an
		// ongoing snapshot data collection procedure.
		try
		{
			xmlHttp = new XMLHttpRequest();

			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processSnapshotStopResponse, false );

			// GET information asynchronously from server
			xmlHttp.open( "GET", "cgi-bin/SnapshotStopRequestHandler.php", true );
			xmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous snapshot stop request failed" );

			// Treat as success case (snapshot canceled).
			handleStopComplete();
			return;
		}
	}


	var processSnapshotStopResponse = function()
	{
		if( xmlHttp.readyState == 4 )
		{
			// Don't care about success or failure; treat everything as success case
			// (snapshot canceled).
			handleStopComplete();
		}
	}


	var handleCollectionComplete = function( url )
	{
		// Collection is complete and snapshot file is ready for download.
		updateAndShowProgressIndicator( "success" );
		$( "#downloadSnapshotLink" ).attr( "href", url ).parent().show();

		// Replace start and cancel buttons with close button. Note, close button
		// is disabled until download link is clicked to avoid accidental closure
		// of dialog window (would require new snapshot collection).
		$( "#snapshotStartButton" ).hide();
		$( "#snapshotCancelButton" ).addClass( "disabled" ).hide();
		$( "#snapshotCloseButton" ).addClass( "call-to-action" )
			.addClass( "disabled" )
			.show();

		collectionInProgress = false;
	}


	var handleFailure = function()
	{
		updateAndShowProgressIndicator( "error" );
		$( "#snapshotCancelButton" ).addClass( "disabled" ).hide();
		$( "#snapshotStartButton" ).removeClass( "disabled" ).focus();
		$( "#snapshotCloseButton" ).show();
		collectionInProgress = false;
	}


	var handleSessionTimeout = function()
	{
		updateAndShowProgressIndicator( "warning", "Session expired" );
		$( "#snapshotCancelButton" ).addClass( "disabled" ).hide();
		$( "#snapshotStartButton" ).removeClass( "disabled" );
		$( "#snapshotCloseButton" ).show();
		collectionInProgress = false;

		// User session has expired.
		window.alert( "Session has expired - Please login again" );

		// Open login page with session expired indication.
		location.assign( "login.php?expired" );
	}


	var handleStopComplete = function()
	{
		updateAndShowProgressIndicator( "info", "Snapshot collection aborted" );
		$( "#snapshotCancelButton" ).hide();
		$( "#snapshotStartButton" ).removeClass( "disabled" ).focus();
		$( "#snapshotCloseButton" ).show();
	}


	var updateAndShowProgressIndicator = function( stepName, displayText )
	{
		// displayText argument is "optional". Expected to be explicitly passed
		// by calling function for info and warning step indicators.
		if( typeof displayText === "undefined" )
		{
			switch( stepName )
			{
				case "current":
					displayText = "Collection in progress";
					break;

				case "error":
					displayText = "Snapshot collection failed";
					break;

				case "success":
					displayText = "Snapshot collection complete";
					break;

				default:
					// Explicit display text should have been provided.
					console.log( "Unrecognized step: " + stepName );
					return;
			}
		}

		var classList = "step " + stepName;

		$( "#snapshotProgressIndication" )
			.removeClass()
			.addClass( classList )
			.find( ".text" ).text( displayText ).end()
			.show();
	}


	NSNReady( function()
	{

		$( "#snapshotLink" ).click( function() {

			if( $( "#snapshotDialog" ).is( ":visible" ) )
			{
				// Dialog window already open.
				return;
			}

			// Initial settings. Progress indication is shown once snapshot collection is
			// initiated. The cancel button is hidden and disabled until collection starts.
			// The download link is shown after collection completes. The close button is
			// initially labeled "Cancel" to indicate the dialog can be closed without starting
			// the procedure (not to be confused with the cancel button).

			$( "#downloadSnapshotLink" ).parent().hide();
			$( "#snapshotProgressIndication" ).hide();

			$( "#snapshotStartButton" ).removeClass( "disabled" )
				.addClass( "call-to-action" )
				.show();

			$( "#snapshotCancelButton" ).addClass( "disabled" ).hide();

			var $closeButton = $( "#snapshotCloseButton" );
			$closeButton.removeClass( "disabled" );
			$closeButton.removeClass( "call-to-action" );
			$closeButton.find( ".button-content" ).text( "Cancel" );
			$closeButton.show();

			collectionInProgress = false;

			// Open the dialog
			$( "#snapshotDialog" ).trigger( "open" );

			// Set focus to start button once dialog has opened.
			setTimeout( function()
			{
				$( "#snapshotStartButton" ).focus();
			}, 100 );


			$( "#snapshotStartButton" ).click( function() {

				if( $( this ).is( ".disabled" ) )
				{
					return;
				}

				$( this ).addClass( "disabled" );

				$closeButton.find( ".button-content" ).text( "Close" ).end().hide();
				$( "#snapshotCancelButton" ).removeClass( "disabled" ).show();

				collectionInProgress = true;

				updateAndShowProgressIndicator( "info", "Starting snapshot collection" );
				sendSnapshotRequest();
			});

			$( "#snapshotCancelButton" ).click( function() {

				if( $( this ).is( ".disabled" ) )
				{
					return;
				}

				$( this ).addClass( "disabled" );

				// Stop periodic timer if active.
				stopRequestTimer();

				// Stop ongoing request, if any.
				if( typeof xmlHttp !== "undefined" )
				{
					// One of these should be registered depending on current step in the procedure;
					// don't care which one. Remove so that the listener doesn't fire on abort().
					xmlHttp.removeEventListener( "readystatechange", processSnapshotResponse, false );
					xmlHttp.removeEventListener( "readystatechange", processSnapshotDataResponse, false );
					xmlHttp.abort();
				}

				sendSnapshotStopRequest();
				collectionInProgress = false;
			});

			$closeButton.click( function() {

				if( $( this ).is( ".disabled" ) )
				{
					return;
				}

				$( this ).addClass( "disabled" );

				// Close the dialog window.
				$( "#snapshotDialog" ).trigger( "close" );
			});

			$( "#downloadSnapshotLink" ).click( function() {

				// Browser default functionality will handle the file download.
				// Just need to enable close button.
				$closeButton.removeClass( "disabled" ).focus();
			});

			$( "#snapshotDialog" ).find( ".radiobutton-container" ).click( function()
			{
				// Reset focus to whichever button is supposed to be active.
				$( "#snapshotDialog" ).find( "button.call-to-action" ).focus();
			});
		});

		$( "#snapshotDialog" ).bind( "close", function() {

			// Programmatically click the cancel button to stop any active requests.
			// Note, the dialog window itself is closed by handler in webuiDialogs.js
			if( collectionInProgress )
			{
				$( "#snapshotCancelButton" ).click();
			}
		});
	});

})( jQuery )
