// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		DisplayActiveAlarms.js
//
// ****************************************************************************

( function( $ )
{

	var ALARMS_REQUEST_TIMER_INTERVAL = 4000;
	var MAX_ALARMS = 500;
	var MIN_NUM_ROWS = 10;
	var NUM_ADD_OR_DELETE_ROWS = 5;
	var strNewLine = "\n";

	var xmlHttp;
	var timerId;
	var alarmsArray = [];
	var $sortColumn;


	// Alarm object.
	var Alarm = function( severity, alarmTime, alarmId, faultName, source )
	{
		this.severity = initializeStringValue( severity );
		this.alarmId = initializeStringValue( alarmId );
		this.faultName = initializeStringValue( faultName );
		this.source = initializeStringValue( source );

		this.translateSeverityToNumerical = function()
		{
			var severityMap = [
				"Critical",
				"Major",
				"Minor",
				"Notice",
				"Clear",
				"Invalid"
			];

			var index = -1;

			if( Array.prototype.indexOf )
			{
				index = severityMap.indexOf( this.severity );
			}
			else
			{
				// Internet ExploDer 8 or lower...
				for( var i = 0; i < severityMap.length; i++ )
				{
					if( severityMap[i] === this.severity )
					{
						index = i;
					}
				}
			}

			if( index == -1 )
			{
				// Not found
				index = 1000;
			}

			return index;
		}

		if( typeof alarmTime === "string" )
		{
			// Input format: 20141112182630+0000
			var year = alarmTime.substr( 0, 4 );
			var month = alarmTime.substr( 4, 2 );
			var day = alarmTime.substr( 6, 2 );
			var hour = alarmTime.substr( 8, 2 );
			var minute = alarmTime.substr( 10, 2 );
			var second = alarmTime.substr( 12, 2 );

			// offset hour includes +/- sign
			var offsetHour = alarmTime.substr( 14, 3 );
			var offsetMinute = alarmTime.substr( 17, 2 );

			// Format: YYYY-MM-DDThh:mm:ssTZD (eg 2014-11-12T18:26:30+00:00)
			var dateString = year + "-" + month + "-" + day;
			dateString += "T" + hour + ":" + minute + ":" + second;
			dateString += offsetHour + ":" + offsetMinute;

			this.alarmTime = new Date( dateString );
		}
		else
		{
			console.log( "Missing alarm time" );

			// Use current time.
			this.alarmTime = new Date();
		}
	};

	var sortAlarms = function()
	{
		if( typeof $sortColumn === "undefined" )
		{
			// No sort order selected.
			return;
		}

		$elem = $sortColumn;

		if( $elem.hasClass( "first-child" ) )
		{
			// Severity column
			alarmsArray.sort( function( a, b ) {

				var severityA = a.translateSeverityToNumerical();
				var severityB = b.translateSeverityToNumerical();

				if( $elem.hasClass( "up" ) )
				{
					return severityB - severityA;
				}
				else
				{
					return severityA - severityB;
				}
			});
		}

		else
		{
			// Alarm time column
			alarmsArray.sort( function( a, b )
			{
				if( $elem.hasClass( "up" ) )
				{
					return b.alarmTime.getTime() - a.alarmTime.getTime();
				}
				else
				{
					return a.alarmTime.getTime() - b.alarmTime.getTime();
				}
			});
		}
	}


	var startAlarmsRequestTimer = function()
	{
		if( $( "#activeAlarmsTable" ).parent().is( ":visible" ) )
		{
			timerId = setTimeout( function() {

				// Reset timer ID to undefined
				timerId = void( 0 );

				// Invoke call back function
				sendAlarmsRequest();

			}, ALARMS_REQUEST_TIMER_INTERVAL );
		}

		// Else, panel has been closed while waiting for response from server; don't
		// start timer.
	}

	var stopAlarmsRequestTimer = function()
	{
		if( typeof timerId !== "undefined" )
		{
			clearTimeout( timerId );

			// Reset to undefined
			timerId = void( 0 );
		}
	}


	var findSeverity = function( $alarmRow )
	{
		var severity = "";
		if( $alarmRow.hasClass( "critical" ) )
		{
			severity = "Critical";
		}
		else if( $alarmRow.hasClass( "major" ) )
		{
			severity = "Major";
		}
		else if( $alarmRow.hasClass( "minor" ) )
		{
			severity = "Minor";
		}
		else if( $alarmRow.hasClass( "clear" ) )
		{
			severity = "Clear";
		}
		else if( $alarmRow.hasClass( "notice" ) )
		{
			severity = "Notice";
		}
		else if( $alarmRow.hasClass( "info" ) )
		{
			severity = "Info";
		}
		else
		{
			severity = "Invalid";
		}
		
		return severity;
	}
	
	$.fn.setup_alarm_cells = function()
	{
		return this.each( function()
		{
			// Severity is associated with the parent row of each alarm cell.
			var severity = findSeverity( $( this ).parent() );
			
			if( severity !== "Invalid" )
			{
				$( this ).append( E( "div", "alarm-container",
					E( "div", "strip" ),
					E( "span", "icon" ),
					E( "span", T( severity ) )
				));
			}
		} );
	}

	$.fn.clear_alarm_row = function()
	{
		var severityText = findSeverity( this );
		if( severityText !== "Invalid" )
		{
			var severityClass = severityText.toLowerCase();
			this.removeClass( severityClass );
		}

		this.find( "td.alarm" ).removeClass( "alarm" );
		this.find( "td:not( :last-child )" ).text( "" );
		this.find( "td:last-child" ).find( ".cell-value" ).text( "" );
	}

	$.fn.populate_alarm_row = function( severityText, alarmTime, alarmId, faultName, source )
	{
		var severityClass = severityText.toLowerCase();
		this.addClass( severityClass );

                this.find( "td:first-child" ).addClass( "alarm" )
                        .append( E( "div", "alarm-container",
                                E( "div", "strip" ),
                                E( "span", "icon" ),
                                E( "span", T( severityText ) ) ) )
                        .next().addClass( "alarm-time" ).text( alarmTime )
                        .next().addClass( "alarm-id" ).text( alarmId )
                        .next().addClass( "alarm-fault-name" ).text( faultName )
			.next().find( ".cell-value" ).addClass( "alarm-source" ).text( source );
	}

	var displayDetails = function( index )
	{
		var $dialog = $( "#alarmDetailsDialog" );

		if( $dialog.is( ":visible" ) )
		{
			return;
		}

		var text_val = "";

		text_val += "Severity :  ";
		text_val += alarmsArray[index].severity;
		text_val += strNewLine;
                text_val += strNewLine;

                text_val += "Time :  ";
                text_val += alarmsArray[index].alarmTime.toLocaleString();
                text_val += strNewLine;
                text_val += strNewLine;

                text_val += "Alarm ID :  ";
                text_val += alarmsArray[index].alarmId;
                text_val += strNewLine;
                text_val += strNewLine;

                text_val += "Fault name :  ";
                text_val += alarmsArray[index].faultName;
                text_val += strNewLine;
                text_val += strNewLine;

                text_val += "Source :  ";
                text_val += alarmsArray[index].source;

		$( "#alarmDetailsText" ).val(text_val);
		$dialog.trigger("open");
	}

	NSNReady( function()
	{
		// Setup alarms table.
		var $table = $( "#activeAlarmsTable" );
		$table.find( "thead tr th:nth-of-type(2)" )
			.text( "Alarm Time GMT " + determineLocalTimeOffset() );

		$table.standard_table( MIN_NUM_ROWS, MAX_ALARMS );

		$table.find( "thead th.sortable" ).each( function()
		{
			var $headerCell = $( this );

			$headerCell.bind( "sortNeeded", function()
			{
				$sortColumn = $headerCell;
				sortAlarms();
				displayAlarms();
			});
		});


		var alarmCellsInitialized = false;

		$( "#displayAlarmsLink" ).click( function()
		{
			if( $table.parent().is( ":visible" ) )
			{
				return;
			}

			// Hide any open panel(s) in the display area.
			$( ".content-display-panel" ).not( $table.parent() ).trigger( "closePanel" );

			$table.parent().show();

			if( !alarmCellsInitialized )
			{
				initializeAlarmCells( $table );
				alarmCellsInitialized = true;
			}

			sendAlarmsRequest();
		});

		$table.parent().bind( "closePanel", function()
		{
			// Stop periodic timer if active.
			stopAlarmsRequestTimer();

			if( typeof xmlHttp !== "undefined" )
			{
				xmlHttp.removeEventListener( "readystatechange", processAlarmsResponse, false );
				xmlHttp.abort();
			}

			$table.parent().hide();
		});

                $table.bind("openEvent", function()
                {
                        var $selected_row = $table.find( "tbody tr.selected" );
                        if( $selected_row.length > 0 )
                        {
				var rowIndex = $selected_row.index();
				if( ( rowIndex >= 0 )  && ( rowIndex < alarmsArray.length ) )
				{
					displayDetails(rowIndex);
				}
                        }
                });

		// Hide the table initially
		$table.parent().hide();
	});


	var initializeAlarmCells = function( $table )
	{
		$alarmCells = $table.find( "td.alarm" );
		if( $alarmCells.length > 0 )
		{
			$alarmCells.setup_alarm_cells();
		}
	}


	var sendAlarmsRequest = function()
	{
		// Create and send XMLHttpRequest to server
		try
		{
			xmlHttp = new XMLHttpRequest();

			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processAlarmsResponse, false );

			// GET information asynchronously from server
			xmlHttp.open( "GET", "cgi-bin/AlarmsHandler.php", true );
			xmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous alarms request failed" );

			// Set timer and try again later.
			startAlarmsRequestTimer();
			return;
		}
	}


	var processAlarmsResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			// Regardless of success or failure, set timer to send next request
			// after a wait interval.
			startAlarmsRequestTimer();

			var alarmsResponse;

			try
			{
				// A successful response should contain a JSON object.
				alarmsResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processAlarmsResponse():\n" + exception );
				console.log( xmlHttp.responseText );
				return;
			}


			if( !("statusInd" in alarmsResponse) ||
			     (typeof alarmsResponse.statusInd !== "string") )
			{
				console.log( "Invalid alarms response; missing status indicator" );
				return;
			}

			if( alarmsResponse.statusInd === "Success" )
			{
				updateAlarmsTable( alarmsResponse );
			}
			else
			{
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in alarmsResponse) ||
				     (typeof alarmsResponse._failReason !== "string") )
				{
					console.log( "Invalid alarms response; missing failure reason" );
					return;
				}

				console.log( "Alarms failure reason: " + alarmsResponse._failReason );

				if( alarmsResponse._failReason === "invalidSession" )
				{
					stopAlarmsRequestTimer();

					// Configuration failed because the user session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive alarms response from server" );

			// Try again next interval
			startAlarmsRequestTimer();
		}
	}

	var updateAlarmsTable = function( alarmsResponse )
	{
		if( alarmsResponse == null )
		{
			return;
		}

		// Clear out existing alarms array.
		while( alarmsArray.length > 0 )
		{
			alarmsArray.pop();
		}

		if( "AlarmInfo" in alarmsResponse )
		{

			if( $.isArray( alarmsResponse.AlarmInfo ) )
			{
				for( var i = 0; (i < alarmsResponse.AlarmInfo.length && i < MAX_ALARMS); i++ )
				{
					var alarm = new Alarm( alarmsResponse.AlarmInfo[i].Severity,
								alarmsResponse.AlarmInfo[i].ObservationTime,
								alarmsResponse.AlarmInfo[i].AlarmId,
								alarmsResponse.AlarmInfo[i].FaultName,
								alarmsResponse.AlarmInfo[i].Source );

					alarmsArray.push( alarm );
				}

				sortAlarms();
			}

			else
			{
				// Single, non-array type object.
				var alarm = new Alarm( alarmsResponse.AlarmInfo.Severity,
							alarmsResponse.AlarmInfo.ObservationTime,
							alarmsResponse.AlarmInfo.AlarmId,
							alarmsResponse.AlarmInfo.FaultName,
							alarmsResponse.AlarmInfo.Source );

				alarmsArray.push( alarm );
			}
		}

		// Display alarms in table; note: empty table displayed if no AlarmInfo instances
		// received in AlarmsResponse.
		displayAlarms();
	}

	var displayAlarms = function()
	{

		$table = $( "#activeAlarmsTable" );

		var rowsInTable = $table.find( "tbody > tr" ).length;

		// Clear out display table
		for( var i = 1; i <= rowsInTable; i++ )
		{
			$table.find( "tbody > tr:nth-of-type(" + i + ")" ).clear_alarm_row();
		}

		// Check if we need to add rows.
		while( (alarmsArray.length > rowsInTable) && (rowsInTable < MAX_ALARMS) )
		{
			// Add groups of rows
			$table.add_rows_to_table( NUM_ADD_OR_DELETE_ROWS, MAX_ALARMS );
			rowsInTable = $table.find( "tbody > tr" ).length;
		}

		var numActiveAlarms = 0;
		for( var i = 0; (i < alarmsArray.length && i < rowsInTable); i++ )
		{
			$table.find( "tbody > tr:nth-of-type(" + (i+1) + ")" )
				.populate_alarm_row( alarmsArray[i].severity,
						alarmsArray[i].alarmTime.toLocaleString(),
						alarmsArray[i].alarmId,
						alarmsArray[i].faultName,
						alarmsArray[i].source );

			if( alarmsArray[i].severity !== "Clear" )
			{
				numActiveAlarms++;
			}
		}

		var numAlarmRows = alarmsArray.length;
		if( numAlarmRows < MIN_NUM_ROWS )
		{
			numAlarmRows = MIN_NUM_ROWS;
		}

		if( numAlarmRows > MAX_ALARMS )
		{
			numAlarmRows = MAX_ALARMS;
		}
		// Check if we can delete rows
		while( rowsInTable >= (numAlarmRows + NUM_ADD_OR_DELETE_ROWS) )
		{
			$table.remove_rows_from_table( NUM_ADD_OR_DELETE_ROWS, MIN_NUM_ROWS );
			rowsInTable = $table.find( "tbody > tr" ).length;
		}

		$table.find( "tfoot > tr > td:first-child" ).text( "Active Alarms: " + numActiveAlarms );

		// The width of the alarms table may have changed. Hide or show the menu panel
		// accordingly.
		$( "#menuPanel" ).trigger( "contentPanelResized" );
	}

})( jQuery )
