// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		DisplayBluetoothConfiguration.js
//
// ****************************************************************************

(function( $ )
{
	var NUM_ADD_OR_DELETE_ROWS = 1;
	var MAX_PAIRINGS = 50;
	var MIN_NUM_TABLE_ROWS = 3;

	var xmlHttp;
	var pairingsArray = [];
	var $sortColumn;


	// DevicePairing helper object.
	var DevicePairing = function( pairingOrderNum, deviceAddress, deviceStatus )
	{
		this.pairingOrderNum = initializeStringValue( pairingOrderNum );
		this.deviceAddress = initializeStringValue( deviceAddress );
		this.deviceStatus = initializeStringValue( deviceStatus );

		this.translateStatusToNumerical = function()
		{
			var statusMap = [
				"whitelisted",
				"blacklisted",
				"neutralentry",
				"Invalid"
			];

			var index = -1;

			if( Array.prototype.indexOf )
			{
				index = statusMap.indexOf( this.deviceStatus.toLowerCase() );
			}
			else
			{
				// Internet ExploDer 8 or lower...
				for( var i = 0; i < statusMap.length; i++ )
				{
					if( statusMap[i] === this.deviceStatus.toLowerCase() )
					{
						index = i;
					}
				}
			}

			if( index == -1 )
			{
				// Not found
				index = 1000;
			}

			return index;
		}
	};


	NSNReady( function()
	{
		// Setup pairings table.
		var $table = $( "#pairingsTable" );
		$table.standard_table( MIN_NUM_TABLE_ROWS, MAX_PAIRINGS );

		$table.find( "thead th.sortable" ).each( function()
		{
			var $headerCell = $( this );

			$headerCell.bind( "sortNeeded", function()
			{
				$sortColumn = $headerCell;
				sortPairingsTable();
				displayPairingsTable();
			});
		});


		$( "#displayBluetoothLink" ).click( function() {

			if( $( "#bluetoothPanel" ).is( ":visible" ) )
			{
				return;
			}

			// Hide any open panel(s) in the display area.
			$( ".content-display-panel" ).not( "#bluetoothPanel" ).trigger( "closePanel" );

			// Show the site info panel in the display area.
			$( "#bluetoothPanel" ).show();

			$( "#refreshBluetoothButton" ).addClass( "disabled" );
			updateAndShowProgressIndicator( "current" );
			sendBluetoothDataRequest();
		});

		$( "#refreshBluetoothButton" ).click( function() {

			var $refreshButton = $( this );
			if( $refreshButton.is( ".disabled" ) )
			{
				return;
			}

			$refreshButton.addClass( "disabled" );
			updateAndShowProgressIndicator( "current" );
			sendBluetoothDataRequest();
		});

		$( "#bluetoothPanel" ).bind( "closePanel", function()
		{
			if( typeof xmlHttp !== "undefined" )
			{
				xmlHttp.removeEventListener( "readystatechange", processBluetoothDataResponse, false );
				xmlHttp.abort();
			}

			$( "#bluetoothPanel" ).hide();
		});

		// Bluetooth panel is hidden until opened by user.
		$( "#bluetoothPanel" ).hide();
	});


	var sendBluetoothDataRequest = function()
	{
		// Create and send XMLHttpRequest to server
		try
		{
			xmlHttp = new XMLHttpRequest();

			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processBluetoothDataResponse, false );

			// GET information asynchronously from server
			xmlHttp.open( "GET", "cgi-bin/BluetoothHandler.php", true );
			xmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous bluetooth data request failed" );
			$( "#refreshBluetoothButton" ).removeClass( "disabled" );
			updateAndShowProgressIndicator( "error" );
			return;
		}
	}


	var processBluetoothDataResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			$( "#refreshBluetoothButton" ).removeClass( "disabled" );
			$( "#bluetoothProgressIndicator" ).hide();

			var bluetoothDataResponse;

			try
			{
				// A successful response should contain a JSON object.
				bluetoothDataResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processBluetoothDataResponse():\n" + exception );
				console.log( xmlHttp.responseText );
				updateAndShowProgressIndicator( "error" );
				return;
			}


			if( !("statusInd" in bluetoothDataResponse) ||
			     (typeof bluetoothDataResponse.statusInd !== "string") )
			{
				console.log( "Invalid bluetooth data response; missing status indicator" );
				updateAndShowProgressIndicator( "error" );
				return;
			}

			if( bluetoothDataResponse.statusInd === "Success" )
			{
				updateBluetoothData( bluetoothDataResponse );
			}
			else
			{
				updateAndShowProgressIndicator( "error" );

				// Failure has occurred. Read failure reason.
				if( !("_failReason" in bluetoothDataResponse) ||
				     (typeof bluetoothDataResponse._failReason !== "string") )
				{
					console.log( "Invalid bluetooth data response; missing failure reason" );
					return;
				}

				console.log( "Bluetooth failure reason: " + bluetoothDataResponse._failReason );

				if( bluetoothDataResponse._failReason === "invalidSession" )
				{
					// Configuration failed because the user session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive bluetooth data response from server" );
			$( "#refreshBluetoothButton" ).removeClass( "disabled" );
			updateAndShowProgressIndicator( "error" );
		}
	}


	var updateBluetoothData = function( bluetoothDataResponse )
	{
		if( bluetoothDataResponse == null )
		{
			return;
		}

		if( !("bluetoothStatus" in bluetoothDataResponse) ||
		     (typeof bluetoothDataResponse.bluetoothStatus !== "string") )
		{
			console.log( "Invalid bluetooth data response; missing bluetooth status" );
			updateAndShowProgressIndicator( "error" );
			return;
		}

		displayBluetoothStatus( bluetoothDataResponse.bluetoothStatus );


		// Clear out existing pairings array.
		while( pairingsArray.length > 0 )
		{
			pairingsArray.pop();
		}

		if( "DevicePairing" in bluetoothDataResponse )
		{
			if( $.isArray( bluetoothDataResponse.DevicePairing ) )
			{
				for( var i = 0; (i < bluetoothDataResponse.DevicePairing.length && i < MAX_PAIRINGS); i++ )
				{
					var devicePairing = new DevicePairing(
						bluetoothDataResponse.DevicePairing[i].pairingOrderNum,
						bluetoothDataResponse.DevicePairing[i].deviceAddress,
						bluetoothDataResponse.DevicePairing[i].deviceStatus );

					pairingsArray.push( devicePairing );
				}

				sortPairingsTable();
			}

			else
			{
				// Single, non-array type object.
				var devicePairing = new DevicePairing(
					bluetoothDataResponse.DevicePairing.pairingOrderNum,
					bluetoothDataResponse.DevicePairing.deviceAddress,
					bluetoothDataResponse.DevicePairing.deviceStatus );

				pairingsArray.push( devicePairing );
			}
		}

		// Display device pairings in table; note: empty table displayed if no DevicePairing
		// instances received in bluetooth data response.
		displayPairingsTable();
	}


	var displayBluetoothStatus = function( bluetoothStatus )
	{
		var uiBluetoothStatus = "Invalid";

		// Format device status for UI display
		switch( bluetoothStatus.toLowerCase() )
		{
			case "enabled":
				uiBluetoothStatus = "Enabled";
				break;

			case "disabled":
				uiBluetoothStatus = "Disabled";
				break;

			case "autofallback":
				uiBluetoothStatus = "Auto fallback";
				break;

			default:
				// UI format remains as Invalid
				break;
		}

		$( "#bluetoothPanel" ).find( "#bluetoothStatus" ).text( uiBluetoothStatus );
	}


	var sortPairingsTable = function()
	{
		if( typeof $sortColumn === "undefined" )
		{
			// No sort order selected.
			return;
		}

		if( $sortColumn.hasClass( "first-child" ) )
		{
			// Order number (first) column. The order number is a regular integer
			// value that can be sorted numerically.
			pairingsArray.sort( function( a, b ) {

				if( $sortColumn.hasClass( "up" ) )
				{
					return b.pairingOrderNum - a.pairingOrderNum;
				}
				else
				{
					return a.pairingOrderNum - b.pairingOrderNum;
				}
			});
		}

		else if( $sortColumn.hasClass( "last-child" ) )
		{
			// device status (last) column
			pairingsArray.sort( function( a, b ) {

				var statusA = a.translateStatusToNumerical();
				var statusB = b.translateStatusToNumerical();

				if( $sortColumn.hasClass( "up" ) )
				{
					return statusB - statusA;
				}
				else
				{
					return statusA - statusB;
				}
			});
		}

		else
		{
			// Device address (middle) column. Use alphabetic sort.
			pairingsArray.sort( function( a, b )
			{
				if( $sortColumn.hasClass( "up" ) )
				{
					if( b.deviceAddress < a.deviceAddress )
					{
						return -1;
					}
					else if( b.deviceAddress == a.deviceAddress )
					{
						return 0;
					}
					else
					{
						return 1;
					}
				}

				else
				{
					if( a.deviceAddress < b.deviceAddress )
					{
						return -1;
					}
					else if( a.deviceAddress == b.deviceAddress )
					{
						return 0;
					}
					else
					{
						return 1;
					}
				}
			});
		}
	}


	var displayPairingsTable = function()
	{

		$table = $( "#pairingsTable" );

		var rowsInTable = $table.find( "tbody > tr" ).length;

		// Clear out display table
		for( var i = 1; i <= rowsInTable; i++ )
		{
			var $row = $table.find( "tbody > tr:nth-of-type(" + i + ")" );
			clear_pairing_row( $row );
		}

		// Check if we need to add rows.
		while( (pairingsArray.length > rowsInTable) && (rowsInTable < MAX_PAIRINGS) )
		{
			// Add groups of rows
			$table.add_rows_to_table( NUM_ADD_OR_DELETE_ROWS, MAX_PAIRINGS );
			rowsInTable = $table.find( "tbody > tr" ).length;
		}

		for( var i = 0; (i < pairingsArray.length && i < rowsInTable); i++ )
		{
			var $row = $table.find( "tbody > tr:nth-of-type(" + (i+1) + ")" );
			populate_pairing_row( $row,
						pairingsArray[i].pairingOrderNum,
						pairingsArray[i].deviceAddress,
						pairingsArray[i].deviceStatus );
		}

		var numPairingRows = pairingsArray.length;
		if( numPairingRows < MIN_NUM_TABLE_ROWS )
		{
			numPairingRows = MIN_NUM_TABLE_ROWS;
		}

		if( numPairingRows > MAX_PAIRINGS )
		{
			numPairingRows = MAX_PAIRINGS;
		}
		// Check if we can delete rows
		while( rowsInTable >= (numPairingRows + NUM_ADD_OR_DELETE_ROWS) )
		{
			$table.remove_rows_from_table( NUM_ADD_OR_DELETE_ROWS, MIN_NUM_TABLE_ROWS );
			rowsInTable = $table.find( "tbody > tr" ).length;
		}

		// The width of the device pairings table may have changed, which may impact the bluetooth
		// panel width. Hide or show the menu panel accordingly.
		$( "#menuPanel" ).trigger( "contentPanelResized" );
	}


	var clear_pairing_row = function( $row )
	{
		$row.find( "td:not( :last-child )" ).text( "" );
		$row.find( "td:last-child" ).find( ".cell-value" ).text( "" );
	}


	var populate_pairing_row = function( $row, pairingOrderNum, deviceAddress, deviceStatus )
	{
		var uiDeviceStatus = "Invalid";

		// Format device status for UI display
		switch( deviceStatus.toLowerCase() )
		{
			case "whitelisted":
				uiDeviceStatus = "White listed";
				break;

			case "blacklisted":
				uiDeviceStatus = "Black listed";
				break;

			case "neutralentry":
				uiDeviceStatus = "Neutral entry";
				break;

			default:
				// UI format remains as Invalid
				break;
		}

		$row.find( "td:first-child" ).text( pairingOrderNum )
			.next().text( deviceAddress )
			.next().find( ".cell-value" ).text( uiDeviceStatus );
	}


	var updateAndShowProgressIndicator = function( stepName, displayText )
	{
		// displayText argument is "optional". Expected to be explicitly passed
		// by calling function for info and warning step indicators.
		if( typeof displayText === "undefined" )
		{
			switch( stepName )
			{
				case "current":
					displayText = "";
					break;

				case "error":
					displayText = "Failed to retrieve data from server";
					break;

				case "success":
					displayText = "";
					break;

				default:
					// Explicit display text should have been provided.
					console.log( "Unrecognized step: " + stepName );
					return;
			}
		}

		var classList = "step " + stepName;

		$( "#bluetoothProgressIndicator" )
			.removeClass()
			.addClass( classList )
			.find( ".text" ).text( displayText ).end()
			.show();
	}

})( jQuery )
