// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		LinkAndSessionManagement.js
//
// ****************************************************************************

( function( $ )
{

	var SUPERVISION_INTERVAL = 2000;
	var LINK_LOSS_DETECTION_THRESHOLD = 15000;
	var GUARD_TIMER_DURATION = 3000;

	var xmlHttp;
	var requestTimerId;
	var guardTimerId;
	var userActivity;
	var lastResponseTime;
	var resetInitiated;
	var linkState;


	NSNReady( function()
	{
		userActivity = false;
		lastResponseTime = new Date().getTime();
		resetInitiated = false;
		linkState = true;

		startRequestTimer();

		$( "#linkLossCloseButton" ).click( function()
		{
			$( "#linkLossDialog" ).trigger( "close" );
			$( "#notificationDialog" ).trigger( "open" );
		});

		$( "#linkLossDialog" ).find( "a.close" ).click( function()
		{
			$( "#linkLossDialog" ).trigger( "close" );
			$( "#notificationDialog" ).trigger( "open" );
		});

		$( document ).bind( "resetInitiated", function()
		{
			resetInitiated = true;
		});

		$( "a, button" ).click( function()
		{
			userActivity = true;
		});

		$( window ).resize( function()
		{
			userActivity = true;
		});

		$( "#linkLossDialog" ).bind( "open", function()
		{
			// Set focus to the close button once dialog has opened.
			setTimeout( function()
			{
				$( "#linkLossCloseButton" ).focus();
			}, 200 );
		});

		$( document ).bind( "stopLinkManagement", function()
		{
			// Stop ongoing request, if any.
			if( typeof xmlHttp !== "undefined" )
			{
				// Remove registered event listener so it doesn't fire on abort().
				xmlHttp.removeEventListener( "readystatechange", processLinkManagementResponse, false );
				xmlHttp.abort();
			}

			// Cancel timers, if active.
			stopRequestTimer();
			stopGuardTimer();
		});
	});


	var startRequestTimer = function()
	{
		requestTimerId = setTimeout( function() {

			// Reset timer ID to undefined
			requestTimerId = void( 0 );

			// Invoke call back function
			sendLinkManagementRequest();

		}, SUPERVISION_INTERVAL );
	}

	var stopRequestTimer = function()
	{
		if( typeof requestTimerId !== "undefined" )
		{
			clearTimeout( requestTimerId );

			// Reset to undefined
			requestTimerId = void( 0 );
		}
	}


	var startGuardTimer = function()
	{
		guardTimerId = setTimeout( function()
		{
			// Reset timer ID to undefined
			guardTimerId = void( 0 );

			handleGuardTimerExpiration();

		}, GUARD_TIMER_DURATION );
	}

	var stopGuardTimer = function()
	{
		if( typeof guardTimerId !== "undefined" )
		{
			clearTimeout( guardTimerId );

			// Reset to undefined
			guardTimerId = void( 0 );
		}
	}


	var handleGuardTimerExpiration = function()
	{
		console.log( "Timeout waiting for link management response" );

		linkManagement();

		// Remove registered event listener and abort ongoing request (no response).
		xmlHttp.removeEventListener( "readystatechange", processLinkManagementResponse, false );
		xmlHttp.abort();

		// Immediately send new request.
		sendLinkManagementRequest();
	}


	var sendLinkManagementRequest = function()
	{
		// Create and send XMLHttpRequest to server. This is the periodic data request
		// sent continuously until snapshot collection completes.
		try
		{
			xmlHttp = new XMLHttpRequest();
		
			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processLinkManagementResponse, false );

			var url = "cgi-bin/LinkAndSessionManagementHandler.php";
			if( userActivity === true )
			{
				url += "?userActivity";
				userActivity = false;
			}

			// GET information asynchronously from server
			xmlHttp.open( "GET", url, true );
			xmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous link management request failed" );

			linkManagement();

			// Set timer and try again later.
			startRequestTimer();
			return;
		}

		// Successfully sent request. Start guard timer waiting for response.
		startGuardTimer();
	}


	var processLinkManagementResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			// Received a response. Cancel guard timer.
			stopGuardTimer();

			var linkManagementResponse;

			try
			{
				// A valid response should contain a JSON object.
				linkManagementResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processLinkManagementResponse():\n" + exception );
				console.log( xmlHttp.responseText );

				linkManagement();
				startRequestTimer();
				return;
			}


			if( !("statusInd" in linkManagementResponse) ||
			     (typeof linkManagementResponse.statusInd !== "string") )
			{
				console.log( "Invalid link management response; missing status indicator" );

				// Unlikely interface failure. Try again next interval.
				linkManagement();
				startRequestTimer();
				return;
			}


			if( linkManagementResponse.statusInd === "Success" )
			{
				// A response was received from the server, indicating the link is up.
				linkUp();

				updateOpAndBlockingState( linkManagementResponse );
				startRequestTimer();
			}

			else
			{
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in linkManagementResponse) ||
				     (typeof linkManagementResponse._failReason !== "string") )
				{
					console.log( "Invalid link management response; missing failure reason" );

					// Unexpected (compound) failure. Hopefully transient.
					linkManagement();
					startRequestTimer();
					return;
				}

				console.log( "Link management response failure reason: " + linkManagementResponse._failReason );

				if( linkManagementResponse._failReason === "invalidSession" )
				{
					// User session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}

				linkManagement();
				startRequestTimer();
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// Request complete. Cancel guard timer.
			stopGuardTimer();

			// HTTP level failure has occurred.
			console.log( "Failed to receive link management response from server" );

			linkManagement();
			startRequestTimer();
		}
	}


	var linkUp = function()
	{
		lastResponseTime = new Date().getTime();

		if( !linkState )
		{
			// Link re-established.
			linkState = true;

			var $linkLossDialog = $( "#linkLossDialog" );
			if( $linkLossDialog.is( ":visible" ) )
			{
				$linkLossDialog.trigger( "close" );
			}

			var $notificationDialog = $( "#notificationDialog" );
			if( $notificationDialog.is ( ":visible" ) )
			{
				$notificationDialog.trigger( "close" );
			}
		}
	}


	var linkManagement = function()
	{
		if( linkState )
		{
			// Link currently considered connected, but failed to get response to
			// link management request. If reset was initiated through WebUI (i.e.,
			// link failure is expected), immediately declare loss of link. Otherwise,
			// wait until threshold has been exceeded.

			if( resetInitiated )
			{
				$( "#linkLossDialog" ).trigger( "open" );
				linkState = false;
				resetInitiated = false;

				// Since reset was initiated through WebUI, either the Reset
				// dialog or Unblock dialog is probably open (unless manually
				// closed). Can be closed now that the link loss dialog is
				// open.

				if( $( "#resetSiteDialog" ).is( ":visible" ) )
				{
					$( "#resetSiteDialog" ).trigger( "close" );
				}

				if( $( "#unblockDialog" ).is( ":visible" ) )
				{
					$( "#unblockDialog" ).trigger( "close" );
				}

				if( $( "#configResetDialog" ).is( ":visible" ) )
				{
					$( "#configResetDialog" ).trigger( "close" );
				}
				

				return;
			}

			var currentTime = new Date().getTime();
			if( (currentTime - lastResponseTime) > LINK_LOSS_DETECTION_THRESHOLD )
			{
				// Exceeded threshold. Declare loss of link.
				$( "#linkLossDialog" ).trigger( "open" );
				linkState = false;
			}
		}
	}


	var updateOpAndBlockingState = function( linkManagementResponse )
	{
		if( linkManagementResponse == null )
		{
			return;
		}

		var $siteInfoPanel = $( "#siteInfoPanel" );
                var $cellInfoPanel = $( "#cellInfoPanel" );
  
		if( ("_unitName" in linkManagementResponse) &&
		    (typeof linkManagementResponse._unitName === "string") )
                {
					$siteInfoPanel.find("#unitName").text( linkManagementResponse._unitName );
                    if(linkManagementResponse._unitName != "FWHR" &&
                       linkManagementResponse._unitName != "FWHRLB" &&
                       linkManagementResponse._unitName != "FWHRHB")
                    {
                        $("#changeBandClassLink").parent().hide();
                    }
                    else
                    {
                        $("#changeBandClassLink").parent().show();
                    }                        
                }
		if( ("_operationalStatus" in linkManagementResponse) &&
		    (typeof linkManagementResponse._operationalStatus === "string") )
		{
			$siteInfoPanel.find( "#opState" ).text( linkManagementResponse._operationalStatus );

                        //Enable or disable Display Cell Infomation link according to site operational State
                        var operationalState = linkManagementResponse._operationalStatus.toLowerCase();
                        if( operationalState === "notcommissioned" )
                        {
                                $( "#menuPanel" ).find( "#displayCellInfoLink" ).parent().addClass( "disabled" );
                                $cellInfoPanel.trigger("uncommissionedEvent");
				$("#menuPanel").find("#locationLockLink").parent().addClass("disabled");
                        }
                        else
                        {
				$("#menuPanel").find("#locationLockLink").parent().removeClass("disabled");
                                $( "#menuPanel" ).find( "#displayCellInfoLink" ).parent().removeClass( "disabled" );
                        }
		}
		else
		{
			$siteInfoPanel.find( "#opState" ).text( "Not available" );
		}

		if( ("_blockingState" in linkManagementResponse) &&
		    (typeof linkManagementResponse._blockingState === "string") )
		{
			$siteInfoPanel.find( "#blockState" ).text( linkManagementResponse._blockingState );

			// Enable or disable Block and Unblock Site links according to the current
			// blocking state.
			var blockingState = linkManagementResponse._blockingState.toLowerCase();
			if( blockingState === "unblocked" )
			{
				$( "#menuPanel" ).find( "#blockSiteLink" ).parent().removeClass( "disabled" );
				$( "#menuPanel" ).find( "#unblockSiteLink" ).parent().addClass( "disabled" );
			}
			else if( blockingState === "blocked" )
			{
				$( "#menuPanel" ).find( "#blockSiteLink" ).parent().addClass( "disabled" );
				$( "#menuPanel" ).find( "#unblockSiteLink" ).parent().removeClass( "disabled" );
			}
		}
		else
		{
			$siteInfoPanel.find( "#blockState" ).text( "Not available" );
		}
	}

})( jQuery )
