// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		MeasureRTT.js
//
// ****************************************************************************

(function( $ )
{
	var MAX_RTT_DATA = 4096;
	var xmlHttp;

	NSNReady( function()
	{

		$( "#rttLink" ).click( function() {

			if( $( "#rttPanel" ).is( ":visible" ) )
			{
				return;
			}

			// Hide any open panel(s) in the display area.
			$( ".content-display-panel" ).not( "#rttPanel" ).trigger( "closePanel" );

			// Show the RTT panel in the display area.
			$( "#rttPanel" ).show();

			// Initial state: spinner hidden, ping button enabled.
			$( "#rttSpinner" ).hide();
			$( "#rttButton" ).removeClass( "disabled" );

			// Set focus to target IP field and select contents.
			$( "#targetIpField" ).select();
			$( "#targetIpField" ).focus();

			// Clear output area.
			$( "#rttOutputArea" ).val( "" );
		});

		$( "#rttButton" ).click( function() {

			var $rttButton = $( this );
			if( $rttButton.is( ".disabled" ) )
			{
				return false;
			}

			$rttButton.addClass( "disabled" );

			$( "#rttOutputArea" ).val( "" );

			var $spinner = $( "#rttSpinner" );
			$spinner.show();
			positionSpinner( $spinner );

			sendRTTRequest();

			return false;
		});


		$( "#rttPanel" ).bind( "closePanel", function()
		{
			if( typeof xmlHttp !== "undefined" )
			{
				xmlHttp.removeEventListener( "readystatechange", processRTTResponse, false );
				xmlHttp.abort();
			}

			$( "#rttSpinner" ).hide();
			$( "#rttButton" ).removeClass( "disabled" );
			$( "#rttPanel" ).hide();
		});

		// RTT panel is hidden until opened by user.
		$( "#rttPanel" ).hide();
	});


	var positionSpinner = function( $spinner )
	{
		// Position the spinner in the center of the parent container.
		var topPos = ($spinner.parent().height() - $spinner.outerHeight()) / 2;
		var leftPos = ($spinner.parent().width() - $spinner.outerWidth()) / 2;

		$spinner.css(
		{
			top: topPos + "px",
			left: leftPos + "px"
		});
	}


	var sendRTTRequest = function()
	{
		// Read user data from target IP field.
		var ipAddress = document.getElementById( "targetIpField" ).value;
		var csrfToken = $( "#rttPanel" ).find( "input[name='csrfToken']" ).val();
				
		// Make sure input is not empty
		if( ipAddress.length == 0 )
		{
			$( "#rttButton" ).removeClass( "disabled" );
			$( "#rttSpinner" ).hide();
			return;
		}

		// Escape special characters
		var ipAddressEncoded = encodeURIComponent( ipAddress );
		var csrfTokenEncoded = encodeURIComponent( csrfToken );

		// Create post data
		var postData = "targetIpAddress=" + ipAddressEncoded + "&csrfToken=" + csrfTokenEncoded;

		// Create and send XMLHttpRequest to server
		try
		{
			xmlHttp = new XMLHttpRequest();

			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processRTTResponse, false );

			// POST information to server
			xmlHttp.open( "POST", "cgi-bin/RTTHandler.php", true );

			// Set encoding used for the request body (HTML form) and length of contents.
			xmlHttp.setRequestHeader( "Content-Type", "application/x-www-form-urlencoded" );
			xmlHttp.setRequestHeader( "Content-length", postData.length );
			xmlHttp.send( postData );
		}
		catch( exception )
		{
			console.log( "Asynchronous RTT request failed" );
			$( "#rttButton" ).removeClass( "disabled" );
			$( "#rttSpinner" ).hide();
			$( "#rttOutputArea" ).val( "Request failed." );
			return;
		}
	}


	var processRTTResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			$( "#rttButton" ).removeClass( "disabled" );
			$( "#rttSpinner" ).hide();

			var rttResponse;

			try
			{
				// A successful response should contain a JSON object.
				rttResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processRTTResponse():\n" + exception );
				console.log( xmlHttp.responseText );
				return;
			}


			if( !("statusInd" in rttResponse) ||
			     (typeof rttResponse.statusInd !== "string") )
			{
				console.log( "Invalid RTT response; missing status indicator" );
				return;
			}

			if( rttResponse.statusInd === "Success" )
			{
				displayRTTMeasurements( rttResponse );

				// Return focus to target IP field and select contents.
				$( "#targetIpField" ).select();
				$( "#targetIpField" ).focus();
			}
			else
			{
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in rttResponse) ||
				     (typeof rttResponse._failReason !== "string") )
				{
					console.log( "Invalid RTT response; missing failure reason" );
					return;
				}

				console.log( "RTT failure reason: " + rttResponse._failReason );

				if( rttResponse._failReason === "invalidSession" )
				{
					// Measurement failed because the user session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}

				if( rttResponse._failReason === "invalidSecurityToken")
				{
					// Measurement failed because the security token has expired.
					updateAndShowProgressIndicator( "warning", "Security Token expired" );

					// Security Token has expired.
					window.alert( "Security token has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}

				$( "#rttOutputArea" ).val( "Request failed." );

				// Return focus to target IP field and select contents.
				$( "#targetIpField" ).select();
				$( "#targetIpField" ).focus();
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive RTT response from server" );
			$( "#rttOutputArea" ).val( "Request failed." );
			$( "#rttButton" ).removeClass( "disabled" );
			$( "#rttSpinner" ).hide();

			// Return focus to target IP field and select contents.
			$( "#targetIpField" ).select();
			$( "#targetIpField" ).focus();
		}
	}


	var displayRTTMeasurements = function( rttResponse )
	{
		if( rttResponse == null )
		{
			return;
		}

		if( !("RTTData" in rttResponse) ||
		     (typeof rttResponse.RTTData !== "string") )
		{
			console.log( "Invalid RTT response; missing RTTData" );
			return;
		}

		// Limit data to avoid excessive output in display area.
		var rttData = rttResponse.RTTData.substr( 0, MAX_RTT_DATA );
		$( "#rttOutputArea" ).val( rttData );
	}

})( jQuery )
