// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2014, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		MonitorPnP.js
//
// ****************************************************************************

var pnpTimerId;
var PNP_TIMER_INTERVAL = 3000;

function startPnPDataRequestTimer()
{
	if( $( "#PnPPanel" ).is( ":visible" ) )
	{
		pnpTimerId = setTimeout( function() {

			// Reset timer ID to undefined
			pnpTimerId = void( 0 );

			// Invoke callback function
			sendPeriodicDataRequest();

		}, PNP_TIMER_INTERVAL );
	}
	// Else, panel has been closed while waiting for response from server; don't
	// start timer.
}

function stopPnPDataRequestTimer()
{
	if( typeof pnpTimerId !== "undefined" )
	{
		clearTimeout( pnpTimerId );

		// Reset to undefined
		pnpTimerId = void( 0 );
	}
}

NSNReady( function() {

	// Hide the PnP panel; should have been setup already.
	$( "#PnPPanel" ).hide();

	$( "#monitorPnPLink" ).click( function() {
	
		if( $( "#PnPPanel" ).is( ":visible" ) )
		{
			// Panel is already open
			return;
		}

		// Hide any open panel(s) in the display area.
		$( ".content-display-panel" ).not( "#PnPPanel" ).trigger( "closePanel" );

		// Show the PnP panel in the display area.
		$( "#PnPPanel" ).show();
                $("#SWPreActivation").parent().hide();
                $("#SWAndConfigActivation").parent().hide();
		sendPeriodicDataRequest();
	});
	
	$( "#PnPPanel" ).bind( "closePanel", function()
	{
		// Stop periodic timer if active.
		stopPnPDataRequestTimer();

		$( "#PnPPanel" ).hide();
	});

});

function sendPeriodicDataRequest()
{
	// Create and send XMLHttpRequest to server
	try
	{
		var xmlHttp = new XMLHttpRequest();
		
		// Register event listener for response
		xmlHttp.addEventListener( "readystatechange",
			function() { processPnPDataResponse( xmlHttp ); }, false );
		
		// GET information asynchronously from server
		xmlHttp.open( "GET", "cgi-bin/PnPDataRequestHandler.php", true );
		xmlHttp.send();
	}
	catch( exception )
	{
		console.log( "Asynchronous PnP data request failed" );

		// Probably means server is resetting as part of the PnP procedure.
		// Set timer and try again later.
		startPnPDataRequestTimer();
		return;
	}
}


function processPnPDataResponse( xmlHttp )
{
	if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
	{
		// Regardless of success or failure, set timer to send next request
		// after a wait interval.
		startPnPDataRequestTimer();

		var dataResponse;

		try
		{
			// A successful response should contain a JSON object.
			dataResponse = JSON.parse( xmlHttp.responseText );
		}
		catch( exception )
		{
			// Parsing failure
			console.log( "JSON parsing error in processPnPDataResponse():\n" + exception );
			console.log( xmlHttp.responseText );
			return;
		}
                

		if( !("statusInd" in dataResponse) ||
		     (typeof dataResponse.statusInd !== "string") )
		{
			console.log( "Invalid PnP data response; missing status indicator" );
			return;
		}

		if( dataResponse.statusInd === "Success" )
		{
			updatePnPStatusFields( dataResponse );
		}
		else
		{
			// Failure has occurred. Read failure reason.
			if( !("_failReason" in dataResponse) ||
			     (typeof dataResponse._failReason !== "string") )
			{
				console.log( "Invalid PnP data response; missing failure reason" );
				return;
			}

			console.log( "PnP failure reason: " + dataResponse._failReason );

			if( dataResponse._failReason === "invalidSession" )
			{
				stopPnPDataRequestTimer();

				// Configuration failed because the user session has expired.
				window.alert( "Session has expired - Please login again" );

				// Open login page with session expired indication.
				location.assign( "login.php?expired" );
				return;
			}
		}
	}

	else if( xmlHttp.readyState == 4 )
	{
		// HTTP level failure has occurred. Maybe because the server is resetting
		// as part of the PnP procedure.
		console.log( "Failed to receive Plug and Play data response from server" );

		// Try again next interval
		startPnPDataRequestTimer();
	}
}

function validatePnPResponse(dataObject)
{
    //Mandatory fields
    var mandatoryPnPFields = ["AutoConnectionIpState", "AutoConnectionDhcpState", "AutoConnectionFzcDiscovery", "AutoConnectionFzcConnection",
                              "SWDownload", "ConfFileDownload"];
    var numOfMandatoryFields = mandatoryPnPFields.length;
    for (var i = 0; i < numOfMandatoryFields; i++)
    {
        if(!(mandatoryPnPFields[i] in dataObject))
        {
	    console.log("invalid PlugAndPlayDataResponse: missing " + mandatoryPnPFields[i]);
            return false;
        }
    }
    //Optional fields
    if(("SWActivation" in dataObject) || ("SWPreActivation" in dataObject))
    {
        if(("SWActivation" in dataObject) && ("SWPreActivation" in dataObject))
        {
	    console.log("Both sw act fields are present in the message");
            return false;  
        }
    }
    else
    {
        console.log("Missing sw activation field");
        return false; //Missing one of the activation fields
    }
    if(("ConfFileActivation" in dataObject) || ("SWAndConfigActivation" in dataObject))
    {
        if(("ConfFileActivation" in dataObject) && ("SWAndConfigActivation" in dataObject))
        {
	    console.log("Both conf act fields are present in the message");
            return false;
        }
    }
    else 
    {
        console.log("Missing conf activation field");
        return false;
    }
    return true;
}
function switchAlternatePnPStatusFields(dataObject)
{
    if("SWActivation" in dataObject)
    {
        $("#SWPreActivation" ).parent().hide();
        $("#SWActivation" ).parent().show();
    }
    else
    {
        $("#SWPreActivation" ).parent().show();
        $("#SWActivation" ).parent().hide();
    }

    if("ConfFileActivation" in dataObject)
    {
	$("#ConfFileActivation" ).parent().show();
        $("#SWAndConfigActivation" ).parent().hide();
    }
    else
    {
        $( "#SWAndConfigActivation" ).parent().show();
        $( "#ConfFileActivation" ).parent().hide();
    }
}
            
function updatePnPStatusFields( dataObject )
{
	if( dataObject == null )
	{
		return;
	}

        if(!validatePnPResponse(dataObject))
        {
            return;
        }

        //update table based on message
        switchAlternatePnPStatusFields(dataObject);

        //update Status fields
        for (var property in dataObject){
            if(typeof dataObject[property] !== "string"){
                console.log ("Invalid value for " + property);
                continue;
            }
            updatePnPStep(property, dataObject[property]);
        }

        //handle reset informs
        if("SWActivation" in dataObject)
        {
            checkAndInformLinkManagementOfReset( dataObject.SWActivation );
        }
        if("ConfFileActivation" in dataObject)
        {     //confFileActivation
            checkAndInformLinkManagementOfReset( dataObject.ConfFileActivation );
        }
        if("SWAndConfigActivation" in dataObject)
        { 
            checkAndInformLinkManagementOfReset( dataObject.SWAndConfigActivation );
        }
}

function updatePnPStep( htmlId, stepStatus )
{
	// Don't care about existing status. Simply overwrite with new status.
	var newClass = "";

	switch( stepStatus )
	{
		case "NotStarted":
			newClass = "upcomming";
			break;

		case "InProgress":
			newClass = "current";
			break;

		case "Success":
			newClass = "success";
			break;

		case "Failure":
			newClass = "error";
			break;

		default:
			// Leave unchanged.
			return;
	}

	// Space separated list of classes. All steps have the "step" class.
	var classList = "step " + newClass;

	// Check for existence of "ruled" class
	if( $( "#" + htmlId ).hasClass( "ruled" ) )
	{
		classList += " ruled";
	}
        $("#PnPPanel").find("#" + htmlId).removeClass().addClass(classList);
}

function checkAndInformLinkManagementOfReset( stepStatus )
{
	if( stepStatus === "InProgress" )
	{
		// Let link management functionality know that a reset is expected.
		$( document ).trigger( "resetInitiated" );
	}
}
