<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2016, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	login.php
//
// Description:	This file serves the WebUI login HTML page. User credentials
//		posted by the client are sent to the WebUIAgent C++ process
//		for authentication. On successful authentication, the user is
//		redirected to the WebUI main page (index.php). Communication
//		with the WebUIAgent is done via the included MessageHandler
//		class.
//
// ****************************************************************************

include_once "../phplib/SessionHelper.php";
include_once "../phplib/MessageHandler.php";
include_once "../phplib/XmlMessageParser.php";
include_once "../phplib/messages/LoginRequest.php";
include_once "../phplib/messages/LoginResponse.php";
include_once "../phplib/messages/TokenRequest.php";
include_once "../phplib/messages/TokenResponse.php";

openlog( "WebUI CGI: Login", LOG_ODELAY, LOG_USER );

header("Content-type: text/html; charset=utf-8");
header( "X-Frame-Options: DENY" );

$sessionId = "";
$CSRFtoken = "";

$loginReason = "";
$typeOfMessage = "none";

$loginSuccess = false;
$lastSuccessfulLoginBts = "Not available";
$lastSuccessfulLoginNetwork = "Not available";
$failedAttemptCountBts = 0;
$failedAttemptCountNetwork = 0;
$currentAccount = "";

function retrieveLoginHistory( LoginResponse $msg )
{
	global $lastSuccessfulLoginBts;
	global $lastSuccessfulLoginNetwork;
	global $failedAttemptCountBts;
	global $failedAttemptCountNetwork;

	if( $msg->HasLoginHistory() )
	{
		if( $msg->GetLastSuccessfulLoginTimeOnBts() !== "" )
		{
			$lastSuccessfulLoginBts = $msg->GetLastSuccessfulLoginTimeOnBts();
		}

		if( $msg->GetLastSuccessfulLoginTimeOnNetwork() !== "" )
		{
			$lastSuccessfulLoginNetwork = $msg->GetLastSuccessfulLoginTimeOnNetwork();
		}

		$failedAttemptCountBts = $msg->GetFailedCountSinceLastSuccessOnBts();
		$failedAttemptCountNetwork = $msg->GetFailedCountSinceLastSuccessOnNetwork();
	}
}


// Check if user already has a valid session. The login page could have been
// requested accidentally by clicking the back button, or similar.
if( isset( $_COOKIE["FZAPWebUISession"] ) )
{
	// Session cookie received from client. Retrieve user session ID.
	$sessionId = $_COOKIE["FZAPWebUISession"];

	$retval = SessionHelper::ValidateSession( $sessionId);
	if( $retval == SessionHelper::SESSION_VALID )
	{
		// User is already authenticated with valid session. Open index page.
	    header( "Location: /index.php" );

	    closelog();
	    exit();
	}

	else{
		//If non null token is returned
		setcookie( "FZAPWebUISession", "", time() - 86400, "/", "", true, true );
	}
}

// Check if post data is available.
if( (isset( $_POST[ "username" ] )) && (isset( $_POST[ "password" ] )) && (isset( $_POST[ "csrfToken"])))
{
	// This means the user submitted the login form. Send LoginRequest message
	// to the WebUIAgent for user authentication.

	$username = $_POST[ "username" ];
	$password = $_POST[ "password" ];
	$loginToken = $_POST["csrfToken"];

	// Create and populate login request message.
	$loginRequest = new LoginRequest( $username, $password, $loginToken);
	if( is_null( $loginRequest ) )
	{
		syslog( LOG_ERR, "Failed to allocate memory for LoginRequest message" );

		// Oops. Severely hosed... Not much can be done to recover. Maybe
		// temporary if a memory issue. Respond with Service Unavailable
		// to indicate (possibly) temporary problem.
		header( "HTTP/1.1 503 Service Unavailable" );

		echo "<html><head><meta charset=\"utf-8\"/></head></html>";
		echo "<h2>HTTP/1.1 503 Service Unavailable</h2>";
		echo "<h4>Please try again later</h4>";

		closelog();
		exit();
	}

	// Send message to WebUIAgent. The MessageHandler will wait for, and return,
	// the response, if any.
	$response = MessageHandler::SendAndReceive( $loginRequest->XmlMsg(), 28);
	if( is_null( $response ) )
	{
		syslog( LOG_ERR, "Failed to send LoginRequest message and/or receive response" );

		// Failed to send and/or receive a response. Serious, but possibly
		// temporary issue. Could mean the WebUIAgent process is not running.
		header( "HTTP/1.1 503 Service Unavailable" );

		echo "<html><head><meta charset=\"utf-8\"/></head></html>";
		echo "<h2>HTTP/1.1 503 Service Unavailable</h2>";
		echo "<h4>Please try again later</h4>";

		closelog();
		exit();
	}

	// Parse the received XML message response.
	$msg = XmlMessageParser::ParseMsg( $response );

	// Validate response.
	if( (is_null( $msg )) || ($msg->OK() !== true) || ($msg->GetMessageType() !== "LoginResponse") )
	{
		// Remove line breaks from response string and print to syslog.
		$msgString = str_replace( "\n", null, $response );
		syslog( LOG_ERR, "Invalid or null LoginResponse message: " . $msgString );

		// Internal error (e.g., protocol error between agent and HTTP server). Not
		// likely to recover.
		header( "HTTP/1.1 500 Internal Server Error" );

		echo "<html><head><meta charset=\"utf-8\"/></head></html>";
		echo "<h2>HTTP/1.1 500 Internal Server Error</h2>";

		closelog();
		exit();
	}


	// Authentication (login) response received from WebUIAgent. Check the result.
	if( $msg->GetStatus() === "Success" )
	{
		// Retrieve session ID generated by the WebUIAgent on successful authentication.
		$sessionId = $msg->GetSessionId();

		// Set a cookie in the client browser with the session ID. Arguments:
		// Cookie name		Any name
		// Cookie value		Session ID from WebUI agent (or internally generated)
		// Expire			Set to 0 to indicate session cookie
		// Path				"/" (any path within the domain)
		// Domain			"" (default)
		// Secure			true (secure https connection required); false in test environment
		// Httponly			true (cookie cannot be accessed via javascript in client)
		setcookie( "FZAPWebUISession", $sessionId, 0, "/", "", true, true );

		// Successful login. Fall through to HTML in order to display login history information.
		$loginSuccess = true;
		$loginReason = "Login successful";
		$typeOfMessage = "success";

		$currentAccount = $username;
		retrieveLoginHistory( $msg );
	}
	else
	{
		// Authentication failed. Check the received failure reason.
		if( $msg->GetFailureReason() === "invalidLogin" )
		{
			// Regular login failure. Fall through to HTML login page for
			// authentication retry (by user).
			$loginReason = "Invalid username or password";
			$typeOfMessage = "error";
		}
		else if( $msg->GetFailureReason() === "blockedByCnum" )
		{
			// The local super user account is blocked because the Login Restriction
			// with CNUM feature is active.
			$loginReason = "Local account login restricted";
			$typeOfMessage = "error";
		}
		else if( $msg->GetFailureReason() === "executionError - A WebUI session exists" )
		{
			// An active user session already exists. This should be a
			// temporary condition.
			header( "HTTP/1.1 503 Service Unavailable" );

			echo "<html><head><meta charset=\"utf-8\"/></head></html>";
			echo "<h2>HTTP/1.1 503 Service Unavailable</h2>";
			echo "<h4>An active WebUI user session exists - Please try again later</h4>";

			closelog();
			exit();
		}
		else if( $msg->GetFailureReason() === "executionError - An SMA/SEM session exists" )
		{
			// An active user session already exists. This should be a
			// temporary condition.
			header( "HTTP/1.1 503 Service Unavailable" );

			echo "<html><head><meta charset=\"utf-8\"/></head></html>";
			echo "<h2>HTTP/1.1 503 Service Unavailable</h2>";
			echo "<h4>An active Site Manager user session exists - Please try again later</h4>";

			closelog();
			exit();
		}
		else if( $msg->GetFailureReason() === "invalidSecurityToken" )
		{
			header( "HTTP/1.1 403 Forbidden" );
			$loginReason = "Page expired";
			$typeOfMessage = "error";
		}
		else if( $msg->GetFailureReason() === "loginDelayActive" )
		{
			// Login retry failures. Fall through to HTML login page for
			// authentication retry (by user).
			$loginReason = "Account temporarily locked";
			$typeOfMessage = "error";
		}
		else
		{
			syslog( LOG_ERR, "Login failure reason: " . $msg->GetFailureReason() );

			// Other execution error has occurred. Don't know the exact reason;
			// treat as temporary.
			header( "HTTP/1.1 503 Service Unavailable" );

			echo "<html><head><meta charset=\"utf-8\"/></head></html>";
			echo "<h2>HTTP/1.1 503 Service Unavailable</h2>";

			closelog();
			exit();
		}
	}
}
else
{
	// No post'ed data. Means we came here as a result of redirection or direct URL
	// access (i.e., user entered <ip address>/login.php in browser address bar).

	if( isset( $_SERVER["QUERY_STRING"] ) )
	{
		switch( $_SERVER[ "QUERY_STRING" ] )
		{
			case "expired":
				$loginReason = "Session expired";
				$typeOfMessage = "info";
				break;

			case "logout":
				$loginReason = "Logout complete";
				$typeOfMessage = "success";
				break;

			case "internal_error":
			case "message_error":
			case "parsing_error":
				// Lump all failures under internal error.
				$loginReason = "Internal error";
				$typeOfMessage = "error";
				break;

			case "pageExpired":
				$loginReason = "Page expired";
				$typeOfMessage = "error";
				break;

			// default: $loginReason remains empty, $typeOfMessage = "none".
		}
	}
}

if( !$loginSuccess )
{
	$tokenRequest = new LoginTokenRequest();
	if( is_null( $tokenRequest ) )
	{
	  syslog( LOG_ERR, "Failed to allocate memory for LoginTokenRequest" );

	  header( "HTTP/1.1 503 Service Unavailable" );

	  echo "<html><head><meta charset=\"utf-8\"/></head></html>";
	  echo "<h2>HTTP/1.1 503 Service Unavailable</h2>";
	  echo "<h4>Please try again later</h4>";

	  closelog();
	  exit();
	}

	$response = MessageHandler::SendAndReceive($tokenRequest->XmlMsg());

	if( is_null( $response ) )
	{
	  syslog( LOG_ERR, "Failed to send LoginTokenRequest message and/or receive response" );

	  // Failed to send and/or receive a response. Serious, but possibly
	  // temporary issue. Could mean the WebUIAgent process is not running.
	  header( "HTTP/1.1 503 Service Unavailable" );

	  echo "<html><head><meta charset=\"utf-8\"/></head></html>";
	  echo "<h2>HTTP/1.1 503 Service Unavailable</h2>";
	  echo "<h4>Please try again later</h4>";

	  closelog();
	  exit();
	}

	// Parse the received XML message response.
	$tokenRsp = XmlMessageParser::ParseMsg( $response );
	// Validate response.
	if( (is_null( $tokenRsp )) || ($tokenRsp->OK() !== true) || ($tokenRsp->GetMessageType() !== "LoginTokenResponse") )
	{
		// Remove line breaks from response string and print to syslog.
		$msgString = str_replace( "\n", null, $response );
		syslog( LOG_ERR, "Invalid or null LoginTokenResponse message: " . $msgString );
		if(is_null( $tokenRsp ))
		{
			syslog( LOG_ERR, "Null LoginTokenResponse message: " . $msgString );
		}
		else if($tokenRsp->OK() !== true)
		{
			syslog( LOG_ERR, "LoginTokenResponse message not OK: " . $msgString );
		}
		else if($tokenRsp->GetMessageType() !== "LoginTokenResponse"){
			// Internal error (e.g., protocol error between agent and HTTP server).
			syslog( LOG_ERR, "LoginTokenResponse message expected: " . $msgString );
		}
			
		// Internal error (e.g., protocol error between agent and HTTP server). Not
		// likely to recover.
		header( "HTTP/1.1 500 Internal Server Error" );

		echo "<html><head><meta charset=\"utf-8\"/></head></html>";
		echo "<h2>HTTP/1.1 500 Internal Server Error</h2>";

		closelog();
		exit();
	}


	if( $tokenRsp->GetStatus() === "Success" )
	{
		$CSRFtoken = $tokenRsp->GetToken();
	}
	else
    {
	   syslog( LOG_ERR, "LoginTokenResponse message failure");
	   // Internal error (e.g., protocol error between agent and HTTP server). Not
	   // likely to recover.
	   header( "HTTP/1.1 500 Internal Server Error" );

	   echo "<html><head><meta charset=\"utf-8\"/></head></html>";
	   echo "<h2>HTTP/1.1 500 Internal Server Error</h2>";
	   closelog();
	   exit();

	}
} // !$loginSuccess

closelog();

// Fall through to HTML login page.
header( "Cache-Control: no-cache, must-revalidate" );

?>

<!DOCTYPE html>
<html lang="en">
	<head>
		<meta charset="utf-8" />
		<meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1" />
		<meta http-equiv="cache-control" content="no-cache" />
		<meta name="viewport" content="width=device-width, minimum-scale=1.0, initial-scale=1.0, user-scalable=yes">
		<title>Flexi Zone Login</title>
		<link rel="stylesheet" type="text/css" href="/OceanTouch/css/styles.css" />
		<style>
			.login-history
			{
				margin-top: 15px;
			}
		</style>
	</head>

	<body class="radial-gradient">

		<div id = "loginDialog" class = "dialog highlight-top modal block-form">

			<form action = "login.php" method = "post">

				<div class="grid span-15 vlead-1">
					<h1 class = "product">Flexi Zone&#8482; Web Access</h1>
					<h2>Account Login</h2>
				</div>

				<div class="grid span-15 vlead-2">
					<div class="span-7 trail-1">
						<input id = "username" name = "username" type="text" class="text required" placeholder="Username" autocomplete = "off" required = "true"></input>
					</div>
					<div class="span-7">
						<input id = "password" name = "password" type="password" class="text required capslock" placeholder="Password" autocomplete = "off" required="true"></input>
						<input type = "hidden" name = "csrfToken" value="<?php echo $CSRFtoken;?>"></input>
					</div>
				</div>

				<div class="grid span-15 vlead-1 ruled">
					<div class="span-10 trail-1 vspan-2">

						<?php
						if( $typeOfMessage === "error" )
						{
						?>
							<span id = "loginMessage" class = "flash error">
								<span class = "icon"></span>
								<?php echo $loginReason; ?>
							</span>
						<?php
						}
						else if( $typeOfMessage === "success" )
						{
						?>
							<span id = "loginMessage" class = "flash success">
								<span class = "icon"></span>
								<?php echo $loginReason; ?>
							</span>
						<?php
						}
						else if( $typeOfMessage === "info" )
						{
						?>
							<span id = "loginMessage" class = "flash info">
								<span class = "icon"></span>
								<?php echo $loginReason; ?>
							</span>
						<?php
						}
						else
						{
						// Empty place holder.
						?>
							<span id = "loginMessage" class = "flash" style = "display: none;">
								<span class = "icon"></span>
							</span>
						<?php
						}
						?>

						<span id = "capslockWarning" class = "flash warning" style = "display: none;">
							<span class = "icon"></span>
							Caps Lock is On
						</span>

					</div>
					<div class="span-4 text-right">
						<button id = "loginDialogButton" class = "call-to-action" type = "submit" value = "Log In">
							Log In
						</button>
					</div>
				</div>

				<div class="grid span-15 legal vpush-1">
You are about to access a private system. This system is for the use of authorized users only. All connections are logged. Any unauthorized access or attempts may be punished to the fullest extent possible under the applicable local legislation.
				</div>

				<div class = "logo"></div>

			</form>

		</div>


		<!-- Login history dialog -->
		<div id = "loginHistoryDialog" class = "dialog modal">

			<h1 class = "vpush-1-half">Login Successful</h1>

			<div class = "vpush-1 ruled">

				<div class = "span-20">
					User: <span id = "userAccount"><?php echo $currentAccount; ?></span>
				</div>

				<div id = "btsLoginHistory" class = "login-history">
					<div class = "span-20 flash info">
						<span class = "icon"></span>
						Last successful login to this network element:&nbsp;
						<span class = "last-successful-login"><?php echo $lastSuccessfulLoginBts; ?></span>
					</div>
					<div class = "span-25 failed-attempt-count flash warning">
						<span class = "icon"></span>
						<span class = "text"><?php echo $failedAttemptCountBts; ?></span>
					</div>
				</div>

				<div id = "networkLoginHistory" class = "login-history">
					<div class = "span-20 flash info">
						<span class = "icon"></span>
						Last successful login to any network element:&nbsp;
						<span class = "last-successful-login"><?php echo $lastSuccessfulLoginNetwork; ?></span>
					</div>
					<div class = "span-25 failed-attempt-count flash warning">
						<span class = "icon"></span>
						<span class = "text"><?php echo $failedAttemptCountNetwork; ?></span>
					</div>
				</div>

				<div id = "loginHistoryNotAvailable" class = "login-history">
					<div class = "span-20 flash warning">
						<span class = "icon"></span>
						<span class = "text">Last login information not available.</span>
					</div>
				</div>

			</div>

			<div class = "text-right buttonbar">
				<button id = "loginHistoryCloseButton">Okay</button>
			</div>
		</div>


		<div class="footer small">
			&copy; 2016 Nokia Networks. All rights reserved.
		</div>


		<script type="text/javascript" src="/OceanTouch/js/jquery.min.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/nsnkit.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/buttons.js"></script>
		<script type="text/javascript" src="/js/webuiDialogs.js"></script>
		<script type="text/javascript" src="/OceanTouch/js/backgrounds.js"></script>
		<script type="text/javascript" src="/js/loginTextinput.js"></script>
		<script type="text/javascript" src="/js/WebuiUtilities.js"></script>
		<script type="text/javascript" src="/js/Login.js"></script>

		<script type="text/javascript">
			NSNReady(function ()
			{
				<?php
				if( $loginSuccess )
				{
				?>
					$( "#loginHistoryDialog" ).trigger( "showLoginHistory" );
				<?php
				}
				else
				{
				?>
					$( "#loginDialog" ).trigger( "showLoginDialog" );
				<?php
				}
				?>
			})
		</script>
	</body>
</html>
