<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2016, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	MessageHandler.php
//
// Description:	The MessageHandler class sends and receives messages to/from
//		the WebUIAgent persistent C++ process. Messages are sent and
//		received over a UDP socket.
//
// ****************************************************************************

class MessageHandler
{
	// Change to "true" to enable message logging
	const MSG_LOGGING_ENABLED = false;

	const ADDRESS = "localhost";
	const PORT = 23456;
	const DEFAULT_MSG_TIMEOUT = 20;

	// The MAX_MSG_SIZE is applicable to messages that do not use fragmentation.
	// When using fragmentation, the total message size can be significantly larger.
	const MAX_MSG_SIZE = 32768;

	// Note: The message fragment (packet) size MUST be set to a value greater than
	// or equal to the corresponding setting in C_Application/SC_OAM/OM_WebUI/WebUIAgent/Include/CGIConnector.h
	const MSG_FRAGMENT_SIZE = 4096;
	const MSG_END_MARKER = "</CGIMessage>";
	const PACKET_TIMEOUT = 2;

	private $socket;

	public function __construct()
	{
		$this->socket = $this->CreateUdpSocket();
	}

	public function __destruct()
	{
		socket_close( $this->socket );
	}

	// ********************************************************************
	// Method: SendAndReceiveLargeMsg
	//
	// Sends passed-in message (string) to the WebUIAgent and waits for a
	// response. This version of the method handles response messages sent
	// as fragmented packages to allow for messages exceeding the UDP max
	// packet size.
	//
	// Returns string containing response if received, or null on failure.
	// ********************************************************************
	public static function SendAndReceiveLargeMsg( $msg, $timeout = self::DEFAULT_MSG_TIMEOUT )
	{
		try
		{
			$agentMessenger = new self();
			$agentMessenger->SendMsg( $msg );

			if( $agentMessenger->ReceiveData( $timeout ) )
			{
				return $agentMessenger->ReadMsg();
			}

			throw new Exception( "Timeout waiting for WebUIAgent response." );
		}

		catch( Exception $e )
		{
			syslog( LOG_ERR, "Message send or receive failure: " . $e->getMessage() );
			return null;
		}
	}


	// ********************************************************************
	// Method:	SendAndReceive
	//
	// Description:	This method sends the passed-in message (string) to the
	//		WebUIAgent C++ process (port 23456). If message send is
	//		successful, it waits for a response to be received and
	//		returns the response to the calling function.
	//
	// Return:	String containing response if received, or null on any
	//		failure.
	//
	// ********************************************************************
	public static function SendAndReceive( $msg, $timeout = self::DEFAULT_MSG_TIMEOUT )
	{

		// Create a UDP socket
		$socket = socket_create( AF_INET, SOCK_DGRAM, SOL_UDP );
		if( $socket === false )
		{
			syslog( LOG_ERR,
				"socket_create() failed; reason: " . socket_strerror( socket_last_error() ) );

			return null;
		}

		self::LogSentMessage( $msg );

		$len = strlen( $msg );
		$retval = socket_sendto( $socket,
					$msg,
					$len,
					0,
					self::ADDRESS,
					self::PORT );

		if( $retval === false )
		{
			// Failed to send message. Return and let client deal with
			// the failure.
			syslog( LOG_ERR, "Failed to send message" );
			return null;
		}


		// Configure arrays for select'ing on socket. Only using read array (expecting
		// response from WebUIAgent).
		$readArray   = array( $socket );
		$writeArray  = null;
		$exceptArray = null;

		// Wait for response with non-null timeout value. Using socket_select before
		// read/receive in order to avoid blocking indefinitely on the read.
		$numSockets = socket_select( $readArray, $writeArray, $exceptArray, $timeout );

		if( $numSockets === false )
		{
			syslog( LOG_ERR, "Socket select failed" );
			return null;
		}

		else if( $numSockets === 0 )
		{
			// Timeout without receiving any data on the socket. Unexpected failure;
			// WebUIAgent did not respond or response did not arrive.
			syslog( LOG_ERR, "Timeout waiting for WebUIAgent response" );
			return null;
		}

		// Else, received data on socket. Read message.
		$response = "";

		// Address and port are passed by reference and populated by the
		// socket_recvfrom function. 
		$address = "";
		$port = 0;
		$retval = socket_recvfrom( $socket,
						$response,
						MessageHandler::MAX_MSG_SIZE,
						0,
						$address,
						$port );

		if( $retval === false )
		{
			syslog( LOG_ERR, "Receive failed" );
			return null;
		}

		self::LogReceivedMessage( $response );

		socket_close( $socket );

		return $response;
	}


	private function CreateUdpSocket()
	{
		$socket = socket_create( AF_INET, SOCK_DGRAM, SOL_UDP );
		if( $socket === false )
		{
			throw new Exception( "Failed to create UDP socket: " . socket_strerror(socket_last_error()) );
		}

		return $socket;
	}


	private function SendMsg( $msg )
	{
		self::LogSentMessage( $msg );

		$len = strlen( $msg );
		$retval = socket_sendto( $this->socket,
					$msg,
					$len,
					0,
					self::ADDRESS,
					self::PORT );

		if( $retval === false )
		{
			throw new Exception( "Failed to send message." );
		}
	}


	private function ReceiveData( $timeout )
	{
		// Wait for messages by select'ing on socket. Configure arrays for select.
		// Only using read array.
		$readArray   = array( $this->socket );
		$writeArray  = null;
		$exceptArray = null;

		$numSockets = socket_select( $readArray, $writeArray, $exceptArray, $timeout );

		if( $numSockets === false )
		{
			throw new Exception( "Socket select failed." );
		}

		if( $numSockets === 0 )
		{
			// Timeout without receiving any data on the socket.
			syslog( LOG_INFO, "Socket select timeout" );
			return false;
		}

		// Num sockets > 0 indicates data available to be read.
		return true;
	}


	private function ReadMsg()
	{
		$response = "";

		do
		{
			$packet = socket_read( $this->socket, self::MSG_FRAGMENT_SIZE );

			if( $packet === false )
			{
				throw new Exception( "Socket receive failure." );
			}

			self::LogPacket( $packet );

			$response .= $packet;
			$endOfMsg = $this->CheckForEndOfMsg( $response );
		}
		while( !$endOfMsg && $this->ReceiveData( self::PACKET_TIMEOUT ) );

		self::LogReceivedMessage( $response );
		return $response;
	}


	private function CheckForEndOfMsg( $msg )
	{
		// Search for end marker. Apply offset to avoid searching the whole (potentially
		// large) message buffer.
		$offset = strlen( $msg ) - strlen( self::MSG_END_MARKER );

		// "Add" some arbitrary margin to allow for newline character, null, etc, at end
		// of message (reduce offset). Note: offset argument tells strpos() where to start
		// searching the string. Must be non-negative.
		$offset -= 3;
		if( $offset < 0 )
		{
			$offset = 0;
		}

		if( strpos( $msg, self::MSG_END_MARKER, $offset ) !== false )
		{
			return true;
		}

		return false;
	}


	private static function LogSentMessage( $msg )
	{
		if( self::MSG_LOGGING_ENABLED )
		{
			// Remove newline characters to allow syslog printing.
			$msgString = str_replace( "\n", null, $msg );

			$len = strlen( $msg );
			syslog( LOG_DEBUG, "Sending message of size " . $len . " bytes: " . $msgString );
		}
	}


	private static function LogReceivedMessage( $msg )
	{
		if( self::MSG_LOGGING_ENABLED )
		{
			// Remove newline characters to allow syslog printing.
			$msgString = str_replace( "\n", null, $msg );

			$len = strlen( $msg );
			syslog( LOG_DEBUG, "Received message of size " . $len . " bytes: " . $msgString );
		}
	}


	private static function LogPacket( $packet )
	{
		if( self::MSG_LOGGING_ENABLED )
		{
			$len = strlen( $packet );
			syslog( LOG_DEBUG, "Received packet of size: " . $len . " bytes" );
		}
	}

}

?>
