<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	XmlMessageParser.php
//
// Description:	Source file for the XmlMessageParser class. Parses XML messages
//		received from the WebUIAgent C++ process.
//
// ****************************************************************************

include_once "messages/LoginResponse.php";
include_once "messages/LogoutResponse.php";
include_once "messages/ValidateSessionResponse.php";
include_once "messages/ConfigureSiteIdResponse.php";
include_once "messages/PlugAndPlayDataResponse.php";
include_once "messages/AlarmsResponse.php";
include_once "messages/SiteInformationResponse.php";
include_once "messages/CellInformationResponse.php";
include_once "messages/OperationalStateResponse.php";
include_once "messages/SnapshotResponse.php";
include_once "messages/SnapshotDataResponse.php";
include_once "messages/SnapshotStopResponse.php";
include_once "messages/BluetoothDataResponse.php";
include_once "messages/IPSecConnectionStatusResponse.php";
include_once "messages/RTTResponse.php";
include_once "messages/BlockingStateChangeResponse.php";
include_once "messages/LinkManagementResponse.php";
include_once "messages/ResetRequestResponse.php";
include_once "messages/ConfigResetRequestResponse.php";
include_once "messages/CertificateInfoResponse.php";
include_once "messages/AntennaInfoResponse.php";
include_once "messages/CmpKeyUpdateResponse.php";
include_once "messages/InitCertsResponse.php";
include_once "messages/ModifyBandClassResponse.php";
include_once "messages/LocationLockUpdateResponse.php";
include_once "messages/LocationLockInfoResponse.php";
include_once "messages/TokenResponse.php";

class XmlMessageParser
{

	// ********************************************************************
	// Method:	ParseMsg
	//
	// Description:	This method parses a received message (from the agent)
	//		in XML format using PHP's built-in SimpleXMLElement
	//		parsing capabilities.
	//
	// Return:	On successful parsing, an instance of AcceptedMessage
	//		containing the received fields. Null on parsing failure.
	// ********************************************************************
	public static function ParseMsg( $xmlMsg )
	{

		// Suppress automatic printing (to standard out) of libxml errors.
		// Save the previous suppression state so it can be restored at the
		// end of the method.
		$xmlErrorState = libxml_use_internal_errors( true );

		$msg = null;

		try
		{

			// Parse the received message
			$sxe = new SimpleXMLElement( $xmlMsg );

			// Validate successful parsing by verifying the element was created.
			// Also, a valid WebUI XML message should have a CGIMessageHdr element.
			if( (!$sxe) || (!$sxe->CGIMessageHdr) )
			{
				syslog( LOG_ERR, "Invalid message; XML parsing failed" );
			}

			else
			{
				// Create appropriate accepted message based on message type.
				if( $sxe->LoginResponse )
				{
					// The message is a LoginResponse
					$msg = new LoginResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->LoginResponse );
					}
				}

				else if( $sxe->ValidateSessionResponse )
				{
					// The message is a ValidateSessionResponse.
					$msg = new ValidateSessionResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->ValidateSessionResponse );
					}
				}

				else if( $sxe->LogoutResponse )
				{
					// The message is a LogoutResponse.
					$msg = new LogoutResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->LogoutResponse );
					}
				}

				else if( $sxe->ConfigureSiteIdResponse )
				{
					// The message is a ConfigureSiteIdResponse.
					$msg = new ConfigureSiteIdResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->ConfigureSiteIdResponse );
					}
				}

				else if( $sxe->PlugAndPlayDataResponse )
				{
					// The message is a PlugAndPlayDataResponse.
					$msg = new PlugAndPlayDataResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->PlugAndPlayDataResponse );
					}
				}

				else if( $sxe->AlarmsResponse )
				{
					// The message is an AlarmsResponse.
					$msg = new AlarmsResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->AlarmsResponse );
					}
				}

				else if( $sxe->OperationalStateResponse )
				{
					$msg = new OperationalStateResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->OperationalStateResponse );
					}
				}

				else if( $sxe->SiteInformationResponse )
				{
					$msg = new SiteInformationResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->SiteInformationResponse );
					}
				}

				else if( $sxe->CellInformationResponse )
				{
					$msg = new CellInformationResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->CellInformationResponse );
					}
				}

				else if( $sxe->SnapshotResponse )
				{
					$msg = new SnapshotResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->SnapshotResponse );
					}
				}

				else if( $sxe->SnapshotDataResponse )
				{
					$msg = new SnapshotDataResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->SnapshotDataResponse );
					}
				}

				else if( $sxe->SnapshotStopResponse )
				{
					$msg = new SnapshotStopResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->SnapshotStopResponse );
					}
				}

				else if( $sxe->BluetoothDataResponse )
				{
					$msg = new BluetoothDataResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->BluetoothDataResponse );
					}
				}

				else if( $sxe->IPSecConnectionStatusResponse )
				{
					$msg = new IPSecConnectionStatusResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->IPSecConnectionStatusResponse );
					}
				}

				else if( $sxe->RTTResponse )
				{
					$msg = new RTTResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->RTTResponse );
					}
				}

				else if( $sxe->BlockingStateChangeResponse )
				{
					$msg = new BlockingStateChangeResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->BlockingStateChangeResponse );
					}
				}

				else if( $sxe->CertificateInfoResponse )
				{
					$msg = new CertificateInfoResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->CertificateInfoResponse );
					}
				}

				else if( $sxe->AntennaInfoResponse )
				{
					$msg = new AntennaInfoResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->AntennaInfoResponse );
					}
				}

				else if( $sxe->CmpKeyUpdateResponse )
				{
					$msg = new CmpKeyUpdateResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->CmpKeyUpdateResponse );
					}
				}

				else if( $sxe->InitCertsResponse )
				{
					$msg = new InitCertsResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->InitCertsResponse );
					}
				}

				else if( $sxe->LinkManagementResponse )
				{
					$msg = new LinkManagementResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->LinkManagementResponse );
					}
				}

				else if( $sxe->ResetRequestResponse )
				{
					$msg = new ResetRequestResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->ResetRequestResponse );
					}
				}

				else if( $sxe->ConfigurationResetRequestResponse )
				{
					$msg = new ConfigurationResetRequestResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->ConfigurationResetRequestResponse );
					}
				}


				else if( $sxe->LocationLockUpdateResponse )
				{
					$msg = new LocationLockUpdateResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->LocationLockUpdateResponse );
					}
				}

				else if( $sxe->LocationLockInfoResponse )
				{
					$msg = new LocationLockInfoResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->LocationLockInfoResponse );
					}
				}

				else if( $sxe->ModifyBandClassResponse )
				{
					$msg = new ModifyBandClassResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->ModifyBandClassResponse );
					}
				}

				else if( $sxe->LoginTokenResponse )
				{
					$msg = new LoginTokenResponse;
					if( $msg )
					{
						// Parse the message body.
						$msg->Parse( $sxe->LoginTokenResponse );
					}
				}

				else
				{
					syslog( LOG_ERR, "Invalid message type" );
					// $msg remains null.
				}

				if( !is_null( $msg ) )
				{
					// Set received session ID from MessageHdr. The session ID should
					// always be present in the message header, but we don't really
					// care if not there (not used).
					if( $sxe->CGIMessageHdr->sessionId )
					{
						$sessionId = $sxe->CGIMessageHdr->sessionId;
						$msg->SetSessionId( (string)$sessionId );
					}
				}
			}
		}

		catch( Exception $e )
		{
			syslog( LOG_ERR, "XML excption: " . $e->getMessage() );
		}

		// Process XML errors, if any
		foreach( libxml_get_errors() as $error )
		{
			syslog( LOG_ERR, XmlMessageParser::FormatXmlError( $error ) );
		}

		// Clear out libxml errors so they don't accummulate...
		libxml_clear_errors();

		// Return libxml error suppression to the original state.
		libxml_use_internal_errors( $xmlErrorState );

		return $msg;
	}


	private static function formatXmlError( $error )
	{
		$output = "";

		switch( $error->level )
		{
			case LIBXML_ERR_WARNING:
				$output .= "XML Warning: ";
				break;

			case LIBXML_ERR_ERROR:
				$output .= "XML Error: ";
				break;

			case LIBXML_ERR_FATAL:
				$output .= "XML Fatal Error: ";
				break;

			default:
				$output .= "Unknown XML Error: ";
		}

		$output .= trim( $error->message );
		$output .= "  Code: " . $error->code;
		$output .= "  Line: " . $error->line;
		$output .= "  Column: " . $error->column;

		if( $error->file )
		{
			$output .= "  File: $error->file";
		}

		return $output;
	}
}

?>
