<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//                      Copyright 2016, Nokia Networks
//                      All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:       phpserver.php
//
// Description: This file is used to simulate HTTP response back from server
//
// Usage:
//		1.Go to /opt/BTSOM/www/phplib/MessageHandler.php
//		Change "const PORT = 23456;" to "const PORT = 23457;" and save
//		And refresh WebUI on browser
//		2.Go to /opt/BTSOM/www/phptest directory
//		Run command "php ./phpserver.php"
//		phpserver.php will generate and send pre-defined respond to
//		WebUI Frontend. Using Ctrl + c to stop phpserver.php
//		3.Revert change in step 1 to let WebUI connect back to server
// ****************************************************************************



define( "ADDRESS", "localhost" );
define( "PORT", 23457 );
define( "TIMEOUT", 5 );
define( "MAX_MSG_SIZE", 4096 );
define( "MSG_FRAGMENT_SIZE", 4096 );
define( "NUM_ALARMS", 12 );

$snapshotEntryCount = 0;
$plugnplayrequests = 0;
$x = 0;

// Create a UDP socket
$socket = socket_create( AF_INET, SOCK_DGRAM, SOL_UDP );
if( $socket === false )
{
	echo "socket_create() failed; reason: " . socket_strerror( socket_last_error() ) . "\n";
	exit();
}


// Bind to port
if( !(socket_bind( $socket, ADDRESS, PORT )) )
{
	echo "Failed to bind to UDP port\n";
	exit();
}


echo "Socket bound to port: " . PORT . "\n";
echo "Waiting for messages\n";

// Continuously listen for messages
while( true )
{
	$readArray   = array( $socket );
	$writeArray  = null;
	$exceptArray = null;
	
	$numSockets = socket_select( $readArray, $writeArray, $exceptArray, TIMEOUT );

	if( $numSockets === false )
	{
		echo "Socket select failed\n";
		exit();
	}

	if( $numSockets === 0 )
	{
		// Timeout without receiving any data on the socket.
		echo "No data; continuing to wait\n";
		continue;
	}

	// Message received on socket.

	$address = "";
	$port = 0;

	// Read message.
	$bytes = socket_recvfrom( $socket, $msg, MAX_MSG_SIZE, 0, $address, $port );
	if( $bytes === false )
	{
		echo "Receive failure\n";
		exit();
	}

	echo "Received " . $bytes . " bytes from remote address $address:$port\n";
	echo "Message: $msg\n\n";

	// Find message type by parsing received XML
	$xml = new SimpleXMLElement( $msg );
	$elements = $xml->children();

	// The message type should be the 2nd element after the MessageHdr
	$msgType = $elements[1]->getName();

	$response  = "<?xml version=\"1.0\"?>\n";
	$response .= "<CGIMessage>\n";
	$response .= "<CGIMessageHdr>\n";
	$response .= "<sessionId>asdfsd56sds432434sdfsdf</sessionId>\n";
	$response .= "</CGIMessageHdr>\n";

	switch( $msgType )
	{
		case "LoginTokenRequest":
			// Send login token response
			$response .= "<LoginTokenResponse>";
			$response .= "<statusInd>Success</statusInd>";
			$response .= "<CGISecurityToken>1xMewJhXU2zJnvLIebm5ishbhc/bsM1Os0v5wK2ERdburbsi0fzd1pCvris9INfyRuVsw+BHZu8IdFkXFhPJ</CGISecurityToken>";
			$response .= "</LoginTokenResponse>\n";
			break;

		case "LoginRequest":
			// Send login response
			$response .= buildLoginResponse( $elements[1] );
			break;

		case "ValidateSessionRequest":
			// Send validation response.
			$response .= "<ValidateSessionResponse>";
			$response .= "<statusInd>Success</statusInd>";
			$response .= "<_userName>Somebody</_userName>";
			$response .= "<CGISecurityToken>1xMewJhXU2zJnvLIebm5ishbhc/bsM1Os0v5wK2ERdburbsi0fzd1pCvris9INfyRuVsw+BHZu8IdFkXFhPJ</CGISecurityToken>";
			$response .= "</ValidateSessionResponse>\n";
			break;

		case "LogoutRequest":
			// Send validation response.
			$response .= "<LogoutResponse>";
			$response .= "<statusInd>Success</statusInd>";
			$response .= "</LogoutResponse>\n";
			break;

		case "ResetRequest":
			// Send validation response.
			$response .= "<ResetRequestResponse>";
			$response .= "<statusInd>Success</statusInd>";
			$response .= "</ResetRequestResponse>\n";
			break;
		case "ConfigureSiteIdRequest":
			// Send configure site id response
			$response .= "<ConfigureSiteIdResponse>";
			$response .= "<statusInd>Success</statusInd>";
			$response .= "</ConfigureSiteIdResponse>\n";
			break;

		case "SiteInformationRequest":
			//Send Site Information response.
			$response .= buildSiteInformationResponse();
			break;

		case "PlugAndPlayDataRequest":
			// Send plug and play data response
			if($plugnplayrequests >= 24)
                        {
			   $response .= buildAltPnPResponse();
			}
			else
			{
                           $response .= buildPnPResponse();
			}	
			$plugnplayrequests++;
			if($plugnplayrequests >= 48)
			{
			  $plugnplayrequests = 0;
			}
			break;

		case "AlarmsRequest":
			// Send alarms response
			$response .= buildAlarmsResponse();
			break;

		case "OperationalStateRequest":
			// Send operational state response
			$response .= buildOperationalStateResponse();
			break;

		case "SnapshotRequest":
			// Send snapshot response
			$response .= "<SnapshotResponse>";
			$response .= "<statusInd>Success</statusInd>";
			$response .= "</SnapshotResponse>\n";
			break;

		case "SnapshotDataRequest":
			// Send snapshot data response
			$response .= buildSnapshotDataResponse();
			break;

		case "SnapshotStopRequest":
			// Send snapshot stop response
			$response .= "<SnapshotStopResponse>";
			$response .= "<statusInd>Success</statusInd>";
			$response .= "</SnapshotStopResponse>\n";
			$snapshotEntryCount = 0;
			break;

		case "BluetoothDataRequest":
			$response .= buildBluetoothDataResponse();
			break;

		case "IPSecConnectionStatusRequest":
			$response .= buildIPSecConnectionStatusResponse();
			break;

		case "RTTRequest":
			$response .= buildRTTResponse();
			break;

		case "BlockingStateChangeRequest":
			$response .= "<BlockingStateChangeResponse>";
			$response .= "<statusInd>Success</statusInd>";
			$response .= "</BlockingStateChangeResponse>\n";
			break;

		case "LinkManagementRequest":
			$response .= "<LinkManagementResponse>";
			$response .= "<statusInd>Success</statusInd>";
			$response .= "<_operationalStatus>OnAir</_operationalStatus>";
			$response .= "<_blockingState>Unblocked</_blockingState>";
			$response .= "<_unitName>FWNC</_unitName>\n";
			$response .= "</LinkManagementResponse>\n";
			break;
		case "CellInformationRequest":
			$response .=buildCellInfoResponse();
			break;		
		case "CertificateInfoRequest":
			$response .= buildCertificateInfoResponse();
			break;
		case "AntennaInfoRequest":
				$x++;
				if ( ($x% 3) == 0 )
					$response .= buildAntennaInfoResponse();
				else
					$response .= buildAntennaInfoResponse2();
				break;
		case "CmpKeyUpdateRequest":
			$response .= "<CmpKeyUpdateResponse>\n";
			$response .= "<statusInd>Success</statusInd>\n";
			$response .= "</CmpKeyUpdateResponse>\n";
			break;
		case "InitCertsRequest":
			$response .="<InitCertsResponse>\n";
                        $response .= "<statusInd>Success</statusInd>\n";
                        $response .="</InitCertsResponse>\n";
			break;
		case "LocationLockInfoRequest":
			$response .="<LocationLockInfoResponse>\n";
                        $response .= "<statusInd>Success</statusInd>\n";
			$response .= "<_actLocationLock>true</_actLocationLock>\n";
                        $response .="</LocationLockInfoResponse>\n";
			break;
		case "LocationLockUpdateRequest":
			$response .="<LocationLockUpdateResponse>\n";
                        $response .= "<statusInd>Failure</statusInd>\n";
			$response .= "<_actLocationLock>true</_actLocationLock>\n";
                        $response .="</LocationLockUpdateResponse>\n";
			break;
               case "ModifyBandClassRequest":
                        $response .="<ModifyBandClassResponse>\n";
                        $response .= "<statusInd>Failure</statusInd>\n";
			$response .= "<_failReason>test</_failReason>\n";
                        $response .="</ModifyBandClassResponse>\n";
                        break;
 		default:
			// Send back empty message body.
	}

	$response .= "</CGIMessage>\n";

	$bytes = 0;

	if( $msgType === "AlarmsRequest" )
	{
		$bytes = sendLargeMsg( $address, $port, $response );
	}
	else
	{
		$len = strlen( $response );
		$bytes = socket_sendto( $socket,
					$response,
					$len,
					0,
					$address,
					$port );
	}

	if( $bytes === false )
	{
		echo "Failed to send response\n";
		continue;
	}

	echo "Sent " . $bytes . " bytes response message\n";

	// Don't print message if enormous; causes link management timeouts...
	if( $bytes < 16384 )
	{
		echo "Message: $response\n";
	}
}

function buildPnPResponse()
{
	static $dataArray = array(
		"AutoConnectionIpState" => "NotStarted",
		"AutoConnectionDhcpState" => "NotStarted",
		"AutoConnectionFzcDiscovery" => "NotStarted",
		"AutoConnectionFzcConnection" => "NotStarted",
		"SWDownload" => "NotStarted",
		"SWActivation" => "NotStarted",
		"ConfFileDownload" => "NotStarted",
		"ConfFileActivation" => "NotStarted" );

	static $entryCount = 0;

	if( $entryCount == 2 )
	{
		$dataArray[ "AutoConnectionIpState" ] = "InProgress";
	}
	else if( $entryCount == 4 )
	{
		$dataArray[ "AutoConnectionIpState" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "AutoConnectionDhcpState" ] = "InProgress";
	}
	else if( $entryCount == 6 )
	{
		$dataArray[ "AutoConnectionDhcpState" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "AutoConnectionFzcDiscovery" ] = "InProgress";
	}
	else if( $entryCount == 8 )
	{
		$dataArray[ "AutoConnectionFzcDiscovery" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "AutoConnectionFzcConnection" ] = "InProgress";
	}
	else if( $entryCount == 9 )
	{
		$dataArray[ "AutoConnectionFzcConnection" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "SWDownload" ] = "InProgress";
	}
	else if( $entryCount == 12 )
	{
		$dataArray[ "SWDownload" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "SWActivation" ] = "InProgress";
	}
	else if( $entryCount == 18 )
	{
		$dataArray[ "SWActivation" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "ConfFileDownload" ] = "InProgress";
	}
	else if( $entryCount == 20 )
	{
		$dataArray[ "ConfFileDownload" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "ConfFileActivation" ] = "InProgress";
	}
	else if( $entryCount == 22 )
	{
		$dataArray[ "ConfFileActivation" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
	}


	// Build message
	$message = "<PlugAndPlayDataResponse>\n";
	$message .= "<statusInd>Success</statusInd>\n";
	foreach( $dataArray as $name => $value )
	{
		$message .= "<" . $name . ">" . $value . "</" . $name . ">\n";
	}
	$message .= "</PlugAndPlayDataResponse>\n";
	
	$entryCount++;
	if( $entryCount >= 24 )
	{
		// Reset to initial state.
		$entryCount = 0;

		$dataArray[ "AutoConnectionIpState" ] = "NotStarted";
		$dataArray[ "AutoConnectionDhcpState" ] = "NotStarted";
		$dataArray[ "AutoConnectionFzcDiscovery" ] = "NotStarted";
		$dataArray[ "AutoConnectionFzcConnection" ] = "NotStarted";
		$dataArray[ "SWDownload" ] = "NotStarted";
		$dataArray[ "SWActivation" ] = "NotStarted";
		$dataArray[ "ConfFileDownload" ] = "NotStarted";
		$dataArray[ "ConfFileActivation" ] = "NotStarted";
	}

	return $message;
}
function buildAltPnPResponse()
{
	static $dataArray = array(
		"AutoConnectionIpState" => "NotStarted",
		"AutoConnectionDhcpState" => "NotStarted",
		"AutoConnectionFzcDiscovery" => "NotStarted",
		"AutoConnectionFzcConnection" => "NotStarted",
		"SWDownload" => "NotStarted",
		"SWPreActivation" => "NotStarted",
		"ConfFileDownload" => "NotStarted",
		"SWAndConfigActivation" => "NotStarted" );

	static $entryCount = 0;

	if( $entryCount == 2 )
	{
		$dataArray[ "AutoConnectionIpState" ] = "InProgress";
	}
	else if( $entryCount == 4 )
	{
		$dataArray[ "AutoConnectionIpState" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "AutoConnectionDhcpState" ] = "InProgress";
	}
	else if( $entryCount == 6 )
	{
		$dataArray[ "AutoConnectionDhcpState" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "AutoConnectionFzcDiscovery" ] = "InProgress";
	}
	else if( $entryCount == 8 )
	{
		$dataArray[ "AutoConnectionFzcDiscovery" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "AutoConnectionFzcConnection" ] = "InProgress";
	}
	else if( $entryCount == 9 )
	{
		$dataArray[ "AutoConnectionFzcConnection" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "SWDownload" ] = "InProgress";
	}
	else if( $entryCount == 12 )
	{
		$dataArray[ "SWDownload" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "SWPreActivation" ] = "InProgress";
	}
	else if( $entryCount == 18 )
	{
		$dataArray[ "SWPreActivation" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "ConfFileDownload" ] = "InProgress";
	}
	else if( $entryCount == 20 )
	{
		$dataArray[ "ConfFileDownload" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
		$dataArray[ "SWAndConfigActivation" ] = "InProgress";
	}
	else if( $entryCount == 22 )
	{
		$dataArray[ "SWAndConfigActivation" ] = (rand( 1, 10 ) < 9) ? "Success" : "Failure";
	}


	// Build message
	$message = "<PlugAndPlayDataResponse>\n";
	$message .= "<statusInd>Success</statusInd>\n";
	foreach( $dataArray as $name => $value )
	{
		$message .= "<" . $name . ">" . $value . "</" . $name . ">\n";
	}
	$message .= "</PlugAndPlayDataResponse>\n";
	
	$entryCount++;
	if( $entryCount >= 24 )
	{
		// Reset to initial state.
		$entryCount = 0;

		$dataArray[ "AutoConnectionIpState" ] = "NotStarted";
		$dataArray[ "AutoConnectionDhcpState" ] = "NotStarted";
		$dataArray[ "AutoConnectionFzcDiscovery" ] = "NotStarted";
		$dataArray[ "AutoConnectionFzcConnection" ] = "NotStarted";
		$dataArray[ "SWDownload" ] = "NotStarted";
		$dataArray[ "SWPreActivation" ] = "NotStarted";
		$dataArray[ "ConfFileDownload" ] = "NotStarted";
		$dataArray[ "SWAndConfigActivation" ] = "NotStarted";
	}

	return $message;
}


function buildSiteInformationResponse()
{
	// Build message
	
	$message = "<SiteInformationResponse>\n";

	$message .= "<statusInd>Success</statusInd>\n";
	$message .= "<ActiveBtsRefSource>PPS</ActiveBtsRefSource>\n";
	$message .= "<AutoconnectionSiteId>WebUI-site-1</AutoconnectionSiteId>\n";
	$message .= "<SIReleaseVersion>TLC16A_ENB_9900_000014_000000</SIReleaseVersion>\n";
	$message .= "<ValidRNWMasterExist>True</ValidRNWMasterExist>\n";
	$message .= "<_serialNro>RY1234567</_serialNro>\n";
//	$message .= "<_apModId>1</_apModId>\n";
	$message .= "<_antModId>2</_antModId>\n";
	$message .= "<sfnApMasterId>10</sfnApMasterId>\n";
	$message .= "<_blockingState>Unblocked</_blockingState>\n";
	$message .= "<_bsAddress>192.168.55.2</_bsAddress>\n";
	$message .= "<_bsId>3</_bsId>\n";
	$message .= "<_distName>MRBTS-3/LNBTS-3</_distName>\n";
	$message .= "<_operationalStatus>OnAir</_operationalStatus>\n";
	$message .= "<_siteName>TDD-Zone-1</_siteName>\n";
	$message .= "<_unitName>FWNC</_unitName>\n";
	$message .= "<btsType>Flexi Zone Indoor SFN FZ Master</btsType>\n";
	$message .= "</SiteInformationResponse>\n";

	return $message;
}

function buildAlarmsResponse()
{

	// Build message
	$message = "<AlarmsResponse>\n";
	$message .= "<statusInd>Success</statusInd>\n";

	for( $i = 0; $i < NUM_ALARMS; $i++ )
	{
		$message .= generateAlarm( $i );
	}

	$message .= "</AlarmsResponse>\n";
	return $message;
}


function generateAlarm( $count )
{
	static $alarmsArray = array(
		array( "severity" => "Minor",
		       "observationTime" => "20141106191216+0000",
		       "faultName" => "Access panel port(s) enabled (0234)",
		       "source" => "BTS:BS 65535/FSM 65535",
		       "alarmId" => "7652" ),

		array( "severity" => "Critical",
		       "observationTime" => "20141106191216+0000",
		       "faultName" => "Commissioning error: Invalid configuration file (4036)",
		       "source" => "BTS:BS 65535/FSM 65535",
		       "alarmId" => "7650" ),

		array( "severity" => "Major",
		       "observationTime" => "20141106191301+0000",
		       "faultName" => "Link OAM link lost on unit 1, interface 1 (61076)",
		       "source" => "TRS:ETHLK-1-1",
		       "alarmId" => "7665" ),

		array( "severity" => "Minor",
		       "observationTime" => "20141106191301+0000",
		       "faultName" => "ssh enabled (61524)",
		       "source" => "TRS:NE-FTM-1",
		       "alarmId" => "7665" ),

		array( "severity" => "Minor",
		       "observationTime" => "20141106191301+0000",
		       "faultName" => "Bluetooth enabled (61520)",
		       "source" => "TRS:BTOOTH-1",
		       "alarmId" => "7665" ),

		array( "severity" => "Minor",
		       "observationTime" => "20141106191301+0000",
		       "faultName" => "BTS time not corrected (0026)",
		       "source" => "BTS:BS 65535/FSPD 1",
		       "alarmId" => "7652" ),

		array( "severity" => "Critical",
		       "observationTime" => "20141106191301+0000",
		       "faultName" => "Difference between BTS master clock and reference frequency (0009)",
		       "source" => "BTS:BS 65535/FSPD 1",
		       "alarmId" => "7652" ),

		array( "severity" => "Notice",
		       "observationTime" => "20141106191404+0000",
		       "faultName" => "SyncE SSM Timed Out on unit 1, interface 2 (61623)",
		       "source" => "TRS:ETHLK-1-2",
		       "alarmId" => "7665" ) );

	$index = $count % count( $alarmsArray );

	$alarm  = "<AlarmInfo>\n";
	$alarm .= "<Severity>" . $alarmsArray[$index]["severity"] . "</Severity>\n";
	$alarm .= "<ObservationTime>" . $alarmsArray[$index]["observationTime"] . "</ObservationTime>\n";
	$alarm .= "<FaultName>" . $alarmsArray[$index]["faultName"] . "</FaultName>\n";
	$alarm .= "<Source>" . $alarmsArray[$index]["source"] . "</Source>\n";

	// Overwrite alarm ID with something that also serves as a unique identifier. Uncomment as necessary
	// to send real alarm IDs.
	// $alarm .= "<AlarmId>" . $alarmsArray[$index]["alarmId"] . "</AlarmId>\n";
	$alarmId = 1001 + $count;
	$alarm .= "<AlarmId>" . $alarmId . "</AlarmId>\n";

	$alarm .= "</AlarmInfo>\n";

	return $alarm;
}


function buildOperationalStateResponse()
{

	// Build message
	$message = "<OperationalStateResponse>\n";
	$message .= "<statusInd>Success</statusInd>\n";
	$message .= "<OperationalState>Configured</OperationalState>\n";

	$message .= "</OperationalStateResponse>\n";
	return $message;
}

function buildSnapshotDataResponse()
{
	static $targetDirectory = "../html/snapshot/";
	static $filename = "test.tgz";
	static $fileExists = false;

	global $snapshotEntryCount;

	echo "Snapshot entry count = " . $snapshotEntryCount . PHP_EOL;

	$filepath = $targetDirectory . $filename;

	if( !file_exists( $targetDirectory ) )
	{
		echo "Creating directory: " . $targetDirectory . PHP_EOL;
		if( !mkdir( $targetDirectory ) )
		{
			echo "Directory creation failed\n";
		}
	}

	if( !file_exists( $filepath ) )
	{
		echo "Copying file: " . $filepath . PHP_EOL;
		if( !copy( $filename, $filepath ) )
		{
			echo "Failed to copy file\n";
		}
	}

	// The file should now exist.
	if( file_exists( $filepath ) )
	{
		$fileExists = true;
	}

	// Build message
	$message = "<SnapshotDataResponse>\n";

	if( $snapshotEntryCount < 6 )
	{
		$message .= "<statusInd>InProgress</statusInd>\n";
	}
	else
	{
		if( $fileExists )
		{
			$message .= "<statusInd>Success</statusInd>\n";
			$message .= "<FileName>snapshot/test.tgz</FileName>\n";
		}
		else
		{
			$message .= "<statusInd>Failure</statusInd>\n";
			$message .= "<_failReason>File copy failed</_failReason>\n";
		}
		$snapshotEntryCount = 0;
	}

	$snapshotEntryCount++;

	$message .= "</SnapshotDataResponse>\n";
	return $message;
}


function buildBluetoothDataResponse()
{

	// Build message
	$message = "<BluetoothDataResponse>\n";
	$message .= "<statusInd>Success</statusInd>\n";

	$message .= "<bluetoothStatus>autoFallback</bluetoothStatus>\n";

	$message .= "<DevicePairing>\n";
	$message .= "<pairingOrderNum>1</pairingOrderNum>\n";
	$message .= "<deviceAddress>a3:ff:00:11:45:bb</deviceAddress>\n";
	$message .= "<deviceStatus>whitelisted</deviceStatus>\n";
	$message .= "</DevicePairing>\n";

	$message .= "<DevicePairing>\n";
	$message .= "<pairingOrderNum>10</pairingOrderNum>\n";
	$message .= "<deviceAddress>a2:ff:00:11:45:bb</deviceAddress>\n";
	$message .= "<deviceStatus>blackListed</deviceStatus>\n";
	$message .= "</DevicePairing>\n";

	$message .= "<DevicePairing>\n";
	$message .= "<pairingOrderNum>3</pairingOrderNum>\n";
	$message .= "<deviceAddress>a3:f0:00:11:45:bb</deviceAddress>\n";
	$message .= "<deviceStatus>neutralEntry</deviceStatus>\n";
	$message .= "</DevicePairing>\n";

	$message .= "<DevicePairing>\n";
	$message .= "<pairingOrderNum>20</pairingOrderNum>\n";
	$message .= "<deviceAddress>a3:ff:01:11:45:bb</deviceAddress>\n";
	$message .= "<deviceStatus>neutralentry</deviceStatus>\n";
	$message .= "</DevicePairing>\n";

	$message .= "</BluetoothDataResponse>\n";
	return $message;
}


function buildIPSecConnectionStatusResponse()
{

	// Build message
	$message = "<IPSecConnectionStatusResponse>\n";
	$message .= "<statusInd>Success</statusInd>\n";

	$message .= "<ipSecEnabled>true</ipSecEnabled>\n";

	$message .= "<IPSecPeerInfo>\n";
	$message .= "<remoteTunnelEndpoint>192.168.255.1</remoteTunnelEndpoint>\n";
	$message .= "<peerState>Enabled</peerState>\n";
	$message .= "</IPSecPeerInfo>\n";

	$message .= "<IPSecPeerInfo>\n";
	$message .= "<remoteTunnelEndpoint>10.52.200.48</remoteTunnelEndpoint>\n";
	$message .= "<peerState>Disabled</peerState>\n";
	$message .= "</IPSecPeerInfo>\n";

	$message .= "<IPSecPeerInfo>\n";
	$message .= "<remoteTunnelEndpoint>20.168.255.1</remoteTunnelEndpoint>\n";
	$message .= "<peerState>Enabled</peerState>\n";
	$message .= "</IPSecPeerInfo>\n";

	$message .= "</IPSecConnectionStatusResponse>\n";
	return $message;
}


function buildRTTResponse()
{

	// Build message
	$message = "<RTTResponse>\n";
	$message .= "<statusInd>Success</statusInd>\n";

	$message .= "<RTTData>\n";
	$message .= "PING 127.0.0.1 (127.0.0.1) 56(84) bytes of data.
64 bytes from 127.0.0.1: icmp_seq=1 ttl=64 time=0.027 ms
64 bytes from 127.0.0.1: icmp_seq=2 ttl=64 time=0.016 ms
64 bytes from 127.0.0.1: icmp_seq=3 ttl=64 time=0.018 ms
64 bytes from 127.0.0.1: icmp_seq=4 ttl=64 time=0.019 ms

--- 127.0.0.1 ping statistics ---
4 packets transmitted, 4 received, 0% packet loss, time 2997ms
rtt min/avg/max/mdev = 0.016/0.020/0.027/0.004 ms\n";

	$message .= "</RTTData>\n";

	$message .= "</RTTResponse>\n";
	return $message;
}


function buildCellInfoResponse()
{
	// Build message
	$message = "<CellInformationResponse>\n";
	$message .= "<statusInd>Success</statusInd>\n";

	$message .= "<Cell>\n";
	$message .= "<DLCarrierFrequency>1234.00 MHz</DLCarrierFrequency>\n";
	$message .= "<ULCarrierFrequency>1234.00 MHz</ULCarrierFrequency>\n";
	$message .= "<_blockingState>unblocked</_blockingState>\n";
	$message .= "<_globalCellId>1</_globalCellId>\n";
	$message .= "<phyCellId>21</phyCellId>\n";
	$message .= "<_id>11</_id>\n";
	$message .= "<_operationalState>enabled</_operationalState>\n";
	$message .= "<dlChBw>10 MHz</dlChBw>\n";
	$message .= "<ulChBw>10 MHz</ulChBw>\n";
	$message .= "<mimoType>2xDL MIMO</mimoType>\n";
	$message .= "</Cell>\n";

	$message .= "<Cell>\n";
	$message .= "<DLCarrierFrequency>1234.00 MHz</DLCarrierFrequency>\n";
	$message .= "<ULCarrierFrequency>1234.00 MHz</ULCarrierFrequency>\n";
	$message .= "<_blockingState>unblocked</_blockingState>\n";
	$message .= "<_globalCellId>2</_globalCellId>\n";
	$message .= "<phyCellId>22</phyCellId>\n";
	$message .= "<_id>22</_id>\n";
	$message .= "<_operationalState>enabled</_operationalState>\n";
	$message .= "<dlChBw>10 MHz</dlChBw>\n";
	$message .= "<ulChBw>10 MHz</ulChBw>\n";
	$message .= "<mimoType>2xDL MIMO</mimoType>\n";
	$message .= "</Cell>\n";

	$message .= "<Cell>\n";
	$message .= "<DLCarrierFrequency>1234.00 MHz</DLCarrierFrequency>\n";
	$message .= "<ULCarrierFrequency>1234.00 MHz</ULCarrierFrequency>\n";
	$message .= "<_blockingState>unblocked</_blockingState>\n";
	$message .= "<_globalCellId>3</_globalCellId>\n";
	$message .= "<phyCellId>23</phyCellId>\n";
	$message .= "<_id>33</_id>\n";
	$message .= "<_operationalState>enabled</_operationalState>\n";
	$message .= "<dlChBw>10 MHz</dlChBw>\n";
	$message .= "<ulChBw>10 MHz</ulChBw>\n";
	$message .= "<mimoType>2xDL MIMO</mimoType>\n";
	$message .= "</Cell>\n";

	$message .= "</CellInformationResponse>\n";

	return $message;
}

function buildCertificateInfoResponse()
{
	//build message
	$message  ="<CertificateInfoResponse>\n";
	$message .="<statusInd>Success</statusInd>\n";

	$message .="<CmpServerStatus>Configured</CmpServerStatus>\n";	

	$message .="<CertificateInfo>\n";
	$message .="<Subject>Root CA</Subject>\n";
	$message .="<Type>TA</Type>\n";
	$message .="<Issuer>Root CA</Issuer>\n";
	$message .="<Index>0</Index>\n";
	$message .="<ValidFrom>11/6/2014, 1:13:01 PM</ValidFrom>\n";
	$message .="<ValidTo>11/6/2014, 1:13:01 PM</ValidTo>\n";
	$message .="<SerialNum>ba8627</SerialNum>\n";
	$message .="<FingerPrint>2B:1D:CC:C0:11:90:2B:5D:DF:9F:E5:83:76:9E:A5:28:2B:20:D1:56</FingerPrint>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.35:X509v3 Authority Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>keyid:F3:0D:63:DB:20:F2:61:3F:2D:24:03:C2:98:32:BF:E9:56:2D:4A:88</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.14:X509v3 Subject Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>F3:0D:63:DB:20:F2:61:3F:2D:24:03:C2:98:32:BF:E9:56:2D:4A:88:4A:88</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.15:X509v3 Key Usage</ExtensionName>\n";
	$message .="<ExtensionValue>Digital Signature, Key Encipherment, Certificate Sign, CRL Sign88</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.19:X509v3 Basic Constraints</ExtensionName>\n";
	$message .="<ExtensionValue>CA:TRUE Signature, Key Encipherment, Certificate Sign, CRL Sign88</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.31:X509v3 CRL Distribution Points</ExtensionName>\n";
	$message .="<ExtensionValue>\n";
	$message .="Full Name:\n";
	$message .="  URI:ldap://10.41.100.254/cn=Root%20CA%2cdc=NSN%20Ulm?certificateRevocationList;binary?sub??ext</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="</CertificateInfo>\n";

	$message .="<CertificateInfo>\n";
	$message .="<Subject>Sub CA_L2a</Subject>\n";
	$message .="<Type>CA</Type>\n";
	$message .="<Issuer>Root CA</Issuer>\n";
	$message .="<Index>1</Index>\n";
        $message .="<ValidFrom>11/6/2014, 1:13:01 PM</ValidFrom>\n";
        $message .="<ValidTo>11/6/2014, 1:13:01 PM</ValidTo>\n";
	$message .="<SerialNum>07ba8bc8</SerialNum>\n";
	$message .="<FingerPrint>DC:6F:F1:AF:20:DE:0F:91:54:3C:3E:30:88:02:5D:6B:7E:0C:7A:DB</FingerPrint>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.35:X509v3 Authority Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>keyid:F3:0D:63:DB:20:F2:61:3F:2D:24:03:C2:98:32:BF:E9:56:2D:4A:88</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.14:X509v3 Subject Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>9C:AB:FD:E2:E2:CF:8B:86:2A:16:1C:B9:95:73:A9:66:BA:1D:2A:95:4A:88</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.15:X509v3 Key Usage</ExtensionName>\n";
	$message .="<ExtensionValue>Digital Signature, Key Encipherment, Certificate Sign, CRL Sign88</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.19:X509v3 Basic Constraints</ExtensionName>\n";
	$message .="<ExtensionValue>CA:TRUE Signature, Key Encipherment, Certificate Sign, CRL Sign88</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.31:X509v3 CRL Distribution Points</ExtensionName>\n";
	$message .="<ExtensionValue>\n";
	$message .="Full Name:\n";
	$message .="  URI:ldap://10.44.35.92:389/CN=Root%20CA,DC=NSN%20Ulm?certificateRevocationList;binary\n";
	$message .="\n";
	$message .="Full Name:\n";
	$message .="  URI:ldap://insta.tre.noklab.net:389/CN=Root%20CA,DC=NSN%20Ulm?certificateRevocationList;binary</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="</CertificateInfo>\n";

	$message .="<CertificateInfo>\n";
	$message .="<Subject>Sub CA_L3a</Subject>\n";
	$message .="<Type>CA</Type>\n";
	$message .="<Issuer>Sub CA_L2a</Issuer>\n";
	$message .="<Index>2</Index>\n";
        $message .="<ValidFrom>11/6/2014, 1:13:01 PM</ValidFrom>\n";
        $message .="<ValidTo>11/6/2014, 1:13:01 PM</ValidTo>\n";
	$message .="<SerialNum>066d9147</SerialNum>\n";
	$message .="<FingerPrint>92:B0:E6:07:02:6B:89:54:B5:8A:F1:7A:01:6E:C4:81:01:57:39:0D</FingerPrint>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.35:X509v3 Authority Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>keyid:9C:AB:FD:E2:E2:CF:8B:86:2A:16:1C:B9:95:73:A9:66:BA:1D:2A:95</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.14:X509v3 Subject Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>40:A4:2E:87:0E:CB:D2:5C:FB:C6:19:4F:87:8B:58:4E:6A:A2:86:D6:2A:95</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.15:X509v3 Key Usage</ExtensionName>\n";
	$message .="<ExtensionValue>Digital Signature, Key Encipherment, Certificate Sign, CRL Sign95</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.19:X509v3 Basic Constraints</ExtensionName>\n";
	$message .="<ExtensionValue>CA:TRUE Signature, Key Encipherment, Certificate Sign, CRL Sign95</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.31:X509v3 CRL Distribution Points</ExtensionName>\n";
	$message .="<ExtensionValue>\n";
	$message .="Full Name:\n";
	$message .="  URI:ldap://10.41.100.254/cn=Sub%20CA_L2a%2cdc=Intermediate%2cdc=NSN%20Ulm?certificateRevocationList;binary?sub??ext</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="</CertificateInfo>\n";

	$message .="<CertificateInfo>\n";
	$message .="<Subject>Sub CA_L4b</Subject>\n";
	$message .="<Type>CA</Type>\n";
	$message .="<Issuer>Sub CA_L3a</Issuer>\n";
	$message .="<Index>3</Index>\n";
        $message .="<ValidFrom>11/6/2014, 1:13:01 PM</ValidFrom>\n";
        $message .="<ValidTo>11/6/2014, 1:13:01 PM</ValidTo>\n";
	$message .="<SerialNum>066d5350</SerialNum>\n";
	$message .="<FingerPrint>57:72:35:74:8A:94:0A:76:CA:90:3F:0A:2F:D9:EA:CE:A4:19:C0:12</FingerPrint>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.35:X509v3 Authority Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>keyid:40:A4:2E:87:0E:CB:D2:5C:FB:C6:19:4F:87:8B:58:4E:6A:A2:86:D6</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.14:X509v3 Subject Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>28:77:DF:9B:EE:2F:4B:FB:AE:08:01:C0:42:77:49:97:1C:A6:D4:40:86:D6</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.15:X509v3 Key Usage</ExtensionName>\n";
	$message .="<ExtensionValue>Certificate Sign, CRL Sign:08:01:C0:42:77:49:97:1C:A6:D4:40:86:D6</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.19:X509v3 Basic Constraints</ExtensionName>\n";
	$message .="<ExtensionValue>CA:TRUEcate Sign, CRL Sign:08:01:C0:42:77:49:97:1C:A6:D4:40:86:D6</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.31:X509v3 CRL Distribution Points</ExtensionName>\n";
	$message .="<ExtensionValue>\n";
	$message .="Full Name:\n";
	$message .="  URI:ldap://10.41.100.254/cn=Sub%20CA_L3a%2cdc=Intermediate%2cdc=NSN%20Ulm?certificateRevocationList;binary?sub??ext</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="</CertificateInfo>\n";

	$message .="<CertificateInfo>\n";
	$message .="<Subject>L123456.nokiasiemensnetworks.com</Subject>\n";
	$message .="<Type>BTS</Type>\n";
	$message .="<Issuer>Sub CA_L4b</Issuer>\n";
	$message .="<Index>4</Index>\n";
        $message .="<ValidFrom>11/6/2014, 1:13:01 PM</ValidFrom>\n";
        $message .="<ValidTo>11/6/2014, 1:13:01 PM</ValidTo>\n";
	$message .="<SerialNum>06755eec</SerialNum>\n";
	$message .="<FingerPrint>48:F2:C3:15:F3:DB:4D:C1:C9:9F:E8:44:DD:58:DC:5D:CE:51:CC:D8</FingerPrint>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.35:X509v3 Authority Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>keyid:28:77:DF:9B:EE:2F:4B:FB:AE:08:01:C0:42:77:49:97:1C:A6:D4:40</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.14:X509v3 Subject Key Identifier</ExtensionName>\n";
	$message .="<ExtensionValue>14:C2:27:DE:69:3A:CA:FC:79:E8:44:97:50:C1:A5:A3:44:03:E3:A4:D4:40</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.15:X509v3 Key Usage</ExtensionName>\n";
	$message .="<ExtensionValue>Digital Signature, Key Encipherment, Data EnciphermentE3:A4:D4:40</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.31:X509v3 CRL Distribution Points</ExtensionName>\n";
	$message .="<ExtensionValue>\n";
	$message .="Full Name:\n";
	$message .="  URI:ldap://10.44.35.92:389/CN=Sub%20CA_L4b,DC=Intermediate,DC=NSN%20Ulm?certificateRevocationList;binary\n";
	$message .="\n";
	$message .="Full Name:\n";
	$message .="  URI:ldap://insta.tre.noklab.net:389/CN=Sub%20CA_L4b,DC=Intermediate,DC=NSN%20Ulm?certificateRevocationList;binary</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="<ExtensionInfo>\n";
	$message .="<ExtensionName>2.5.29.37:X509v3 Extended Key Usage</ExtensionName>\n";
	$message .="<ExtensionValue>TLS Web Server Authentication, TLS Web Client AuthenticationIntermediate,DC=NSN%20Ulm?certificateRevocationList;binary\n";
	$message .="\n";
	$message .="Full Name:\n";
	$message .="  URI:ldap://insta.tre.noklab.net:389/CN=Sub%20CA_L4b,DC=Intermediate,DC=NSN%20Ulm?certificateRevocationList;binary</ExtensionValue>\n";
        $message .="</ExtensionInfo>\n";
	$message .="</CertificateInfo>\n";

	$message .="</CertificateInfoResponse>\n";

	return $message;
}


function buildLoginResponse( SimpleXMLElement $loginRequest )
{
	$failedCountBts = 0;
	$failedCountNetwork = 0;
	$networkAccount = false;
	$includeLoginHistory = true;
	$includeBtsHistory = true;
	$includeBtsLastSuccessfulLogin = true;
	$failReason = "";

	foreach( $loginRequest->children() as $elem )
	{
		if( $elem->getName() === "_userName" )
		{
			$username = (string)$elem;
			if( $username === "Nemuadmin" )
			{
				// Default successful local account login
			}
			else if( $username === "MickeyMouse" )
			{
				$failedCountBts = 1;
			}
			else if( $username === "MinnieMouse" )
			{
				$failedCountBts = 1;
				$networkAccount = true;
			}
			else if( $username === "DonaldDuck" )
			{
				$failedCountBts = 2;
			}
			else if( $username === "DaisyDuck" )
			{
				$failedCountBts = 1;
				$failedCountNetwork = 3;
				$networkAccount = true;
			}
			else if( $username === "Goofy" )
			{
				$networkAccount = true;
			}
			else if( $username === "Pluto" )
			{
				$includeLoginHistory = false;
			}
			else if( $username === "ChipNDale" )
			{
				$networkAccount = true;
				$failedCountNetwork = 1;
				$includeBtsHistory = false;
			}
			else if( $username === "ScroogeMcDuck" )
			{
				$networkAccount = true;
				$failedCountBts = 3;
				$failedCountNetwork = 1;

				// Unexpected special case, local BTS failed count > 0 but last successful
				// login information on local BTS not included.
				$includeBtsLastSuccessfulLogin = false;
			}
			else if( $username === "HueyDeweyAndLouie" )
			{
				$failReason = "blockedByCnum";
			}
			else
			{
				$failReason = "invalidLogin";
			}
		}
	}

	if( $failReason !== "" )
	{
		$message  = "<LoginResponse>\n";
		$message .= "<statusInd>Failure</statusInd>\n";
		$message .= "<_failReason>" . $failReason . "</_failReason>\n";
		$message .= "</LoginResponse>\n";
		return $message;
	}

	$message  = "<LoginResponse>\n";
	$message .= "<statusInd>Success</statusInd>\n";

	if( $includeLoginHistory )
	{
		$message .= "<LoginHistory>\n";

		if( $includeBtsHistory )
		{
			$message .= "<_failedCountSinceLastSuccessOnBts>";
			$message .= $failedCountBts;
			$message .= "</_failedCountSinceLastSuccessOnBts>\n";
			if( $includeBtsLastSuccessfulLogin )
			{
				$message .= "<_lastSuccessfulLoginTimeOnBts>2016-04-15T21:55:45+00:00</_lastSuccessfulLoginTimeOnBts>\n";
			}
		}

		if( $networkAccount )
		{
			$message .= "<_failedCountSinceLastSuccessOnNetwork>";
			$message .= $failedCountNetwork;
			$message .= "</_failedCountSinceLastSuccessOnNetwork>\n";
			$message .= "<_lastSuccessfulLoginTimeOnNetwork>2016-03-10T21:55:45+00:00</_lastSuccessfulLoginTimeOnNetwork>\n";
		}

		$message .= "</LoginHistory>\n";
	}

	$message .= "</LoginResponse>\n";
	return $message;
}


function buildAntennaInfoResponse()
{
	//build message
	$message  ="<AntennaInfoResponse>\n";
	$message .="<statusInd>Success</statusInd>\n";
	$message .="<CellInfo>\n";
	$message .="<CellID>11</CellID>\n";
	$message .="<Pci>1</Pci>\n";
	$message .="<Earfcn>40540</Earfcn>\n";
	$message .="</CellInfo>\n";

	$message .="<CellInfo>\n";
	$message .="<CellID>22</CellID>\n";
	$message .="<Pci>2</Pci>\n";
	$message .="<Earfcn>40740</Earfcn>\n";
	$message .="</CellInfo>\n";

	$message .="<AntennaInfo>\n";
	$message .="<AntennaId>1</AntennaId>\n";
	$message .="<AntennaSiteId>NW Corner Antenna</AntennaSiteId>\n";
	$message .="<AntennaIpAddress>10.52.207.100</AntennaIpAddress>\n";
	$message .="<CellID1>11</CellID1>\n";
	$message .="<CellID2></CellID2>\n";
	$message .="<OpState>On Air</OpState>\n";
	$message .="<SerialNumber>RY135209020</SerialNumber>\n";
	$message .="<HwType>FWNC</HwType>\n";
	$message .="<WorkingState>Working</WorkingState>\n";
	$message .="<BlockState>Unblocked</BlockState>\n";
	$message .="</AntennaInfo>\n";

	$message .="<AntennaInfo>\n";
	$message .="<AntennaId>3</AntennaId>\n";
	$message .="<AntennaSiteId>5th Floor Antenna</AntennaSiteId>\n";
	$message .="<AntennaIpAddress>10.52.207.102</AntennaIpAddress>\n";
	$message .="<CellID1>11</CellID1>\n";
	$message .="<CellID2>22</CellID2>\n";
	$message .="<OpState>On Air</OpState>\n";
	$message .="<SerialNumber>RY135209022</SerialNumber>\n";
	$message .="<HwType>FWNC</HwType>\n";
	$message .="<WorkingState>Working</WorkingState>\n";
	$message .="<BlockState>Blocked</BlockState>\n";
	$message .="</AntennaInfo>\n";

	$message .="<AntennaInfo>\n";
	$message .="<AntennaId>4</AntennaId>\n";
	$message .="<AntennaSiteId>45th Floor Antenna</AntennaSiteId>\n";
	$message .="<AntennaIpAddress>10.52.207.103</AntennaIpAddress>\n";
	$message .="<CellID1></CellID1>\n";
	$message .="<CellID2>22</CellID2>\n";
	$message .="<OpState>Integrated</OpState>\n";
	$message .="<SerialNumber>RY135209023</SerialNumber>\n";
	$message .="<HwType>FWNC</HwType>\n";
	$message .="<WorkingState>Faulty</WorkingState>\n";
	$message .="<BlockState>Unblocked</BlockState>\n";
	$message .="</AntennaInfo>\n";

	$message .="</AntennaInfoResponse>\n";

	return $message;
}

function buildAntennaInfoResponse2()
{
	//build message
	$message  ="<AntennaInfoResponse>\n";
	$message .="<statusInd>Success</statusInd>\n";

	$message .="<CellInfo>\n";
	$message .="<CellID>11</CellID>\n";
	$message .="<Pci>1</Pci>\n";
	$message .="<Earfcn>40540</Earfcn>\n";
	$message .="</CellInfo>\n";

	$message .="<CellInfo>\n";
	$message .="<CellID>22</CellID>\n";
	$message .="<Pci>2</Pci>\n";
	$message .="<Earfcn>40740</Earfcn>\n";
	$message .="</CellInfo>\n";

	$message .="<AntennaInfo>\n";
	$message .="<AntennaId>1</AntennaId>\n";
	$message .="<AntennaSiteId>NW Corner Antenna</AntennaSiteId>\n";
	$message .="<AntennaIpAddress>10.52.207.100</AntennaIpAddress>\n";
	$message .="<CellID1>11</CellID1>\n";
	$message .="<CellID2></CellID2>\n";
	$message .="<OpState>On Air</OpState>\n";
	$message .="<SerialNumber>RY135209020</SerialNumber>\n";
	$message .="<HwType>FWNC</HwType>\n";
	$message .="<WorkingState>working</WorkingState>\n";
	$message .="<BlockState>unblocked</BlockState>\n";
	$message .="</AntennaInfo>\n";

	$message .="<AntennaInfo>\n";
	$message .="<AntennaId>2</AntennaId>\n";
	$message .="<AntennaSiteId>Southwest Corner Antenna Sprint Building Virginia</AntennaSiteId>\n";
	$message .="<AntennaIpAddress>10.52.207.101</AntennaIpAddress>\n";
	$message .="<CellID1>22</CellID1>\n";
	$message .="<CellID2></CellID2>\n";
	$message .="<OpState>On Air</OpState>\n";
	$message .="<SerialNumber>RY135209021</SerialNumber>\n";
	$message .="<HwType>FWNC</HwType>\n";
	$message .="<WorkingState>working</WorkingState>\n";
	$message .="<BlockState>unblocked</BlockState>\n";
	$message .="</AntennaInfo>\n";

	$message .="<AntennaInfo>\n";
	$message .="<AntennaId>3</AntennaId>\n";
	$message .="<AntennaSiteId>5th Floor Antenna</AntennaSiteId>\n";
	$message .="<AntennaIpAddress>10.52.207.102</AntennaIpAddress>\n";
	$message .="<CellID1>11</CellID1>\n";
	$message .="<CellID2>22</CellID2>\n";
	$message .="<OpState>On Air</OpState>\n";
	$message .="<SerialNumber>RY135209022</SerialNumber>\n";
	$message .="<HwType>FWNC</HwType>\n";
	$message .="<WorkingState>working</WorkingState>\n";
	$message .="<BlockState>blocked</BlockState>\n";
	$message .="</AntennaInfo>\n";

	$message .="<AntennaInfo>\n";
	$message .="<AntennaId>4</AntennaId>\n";
	$message .="<AntennaSiteId>45th Floor Antenna</AntennaSiteId>\n";
	$message .="<AntennaIpAddress>10.52.207.103</AntennaIpAddress>\n";
	$message .="<CellID1>11</CellID1>\n";
	$message .="<CellID2>22</CellID2>\n";
	$message .="<OpState>Integrated</OpState>\n";
	$message .="<SerialNumber>RY135209023</SerialNumber>\n";
	$message .="<HwType>FWNC</HwType>\n";
	$message .="<WorkingState>faulty</WorkingState>\n";
	$message .="<BlockState>unblocked</BlockState>\n";
	$message .="</AntennaInfo>\n";

	$message .="</AntennaInfoResponse>\n";

	return $message;
}


function sendLargeMsg( $address, $port, $msg )
{
	$msgLength = strlen( $msg );
	$sentBytes = 0;
	$packetStartPos = 0;

	global $socket;

	do
	{
		$packetBuffer = substr( $msg, $packetStartPos, MSG_FRAGMENT_SIZE );

		$len = strlen( $packetBuffer );
		echo "Sending packet with length of " . $len . " bytes\n";

		$sentBytes += socket_sendto( $socket,
		                         $packetBuffer,
		                         $len,
		                         0,
		                         $address,
		                         $port );

		$packetStartPos += MSG_FRAGMENT_SIZE;
	}
	while( $packetStartPos < $msgLength );

	return $sentBytes;
}

?>
