#!/bin/bash

# Note: If you change anything here please make sure the behavior is the same as in C++ code:
# C_Application/SC_OAM_FZM/OM_PIL/Source/Deployment/DspDeployment.cpp
# C_Application/SC_OAM_FZM/OM_PIL/Source/Deployment/DeploymentSelectorImpl.cpp
# and related files...

LOG_FILE='/ffs/stb/SWDLReport.txt'
SCFC_INACTIVE_DIR='/ffs/run/btsom/config/inactive/asn/'

SCFC_RUN_ACTIVE_DIR='/ffs/run/btsom/config/active'
SCFC_STB_ACTIVE_DIR='/ffs/stb/btsom/config/active'
SCFC_ACTIVE_DIR=$SCFC_RUN_ACTIVE_DIR

RUN_AUTOCONF_FILE='/ffs/run/btsom/config/active/autoconf.xml'
STB_AUTOCONF_FILE='/ffs/stb/btsom/config/active/autoconf.xml'
AUTOCONF_FILE=$RUN_AUTOCONF_FILE

BPF_FILENAME='bpf.xml'
BPF_PATH_IN_PACKAGE='config'
BTSOM_PACKAGE_FILE='/ffs/stb/apps/BTSOM_*.tgz'
UNIT_ID_FILE='/proc/device-tree/board-identity/unit-id'
DEPLOYMENT_TYPE_FILE='/ffs/stb/config/deployment_type'
FPGA_IMAGE_FILE='/ffs/stb/config/fpga_bandwidth'
DSPRTSW_DIR='/ffs/stb/dsp/'
DSPRTSW_SYMLINK='dsprtsw.bin'
DSPRTSW_EXT='.BIN'
P7_EXT='.p7'
ACT_PREP_DEPL_SET_FILE='/ram/act_prep_deployment_set'

DEPL_TDD_6_CORE_SFN_MASTER='6core_SfnMaster'
DEPL_TDD_8_CORE_SFN_ANT='8core_SfnAnt'
DEPL_FDD_6_CORE_SFN_MASTER='6coreFdd_SfnMaster'
DEPL_FDD_4_CORE_SFN_ANT='4coreFdd_SfnAnt'
DEPL_TDD_10_CORE='10core'
DEPL_TDD_8_CORE='8core'
DEPL_FDD_10_CORE_LTE_U='10corelteu'
DEPL_FDD_8_CORE='8core'
DEPL_FDD_8_CORE_MF='8coreFddMf'
DEPL_FDD_10_CORE_FSYMDSP='10corefsymdsp'

PREFIX_TDD_6_CORE_SFN_MASTER='TLF-K2-PSL2DSP'
PREFIX_TDD_8_CORE_SFN_ANT='TLF-K2-L1L1DSP'
PREFIX_FDD_6_CORE_SFN_MASTER='FLF-K2-FSFNMDSP'
PREFIX_FDD_4_CORE_SFN_ANT='FLF-K2-FSFNADSP'
PREFIX_TDD_10_CORE='TLF-K2-L1ExL2DSP'
PREFIX_TDD_8_CORE='TLF-K2-L1L2DSP'
PREFIX_FDD_10_CORE_LTE_U='FLF-K2-ASPSDSP'
PREFIX_FDD_8_CORE='FLF-K2-L1L2DSP'
PREFIX_FDD_8_CORE_MF='FLF-K2-L1L2DSPMF'
PREFIX_FDD_10_CORE_FSYMDSP='FLF-K2-FSYMDSP'

ISOL_CPUS_TDD_6_CORE_SFN_MASTER=12
ISOL_CPUS_TDD_8_CORE_SFN_ANT=8
ISOL_CPUS_FDD_6_CORE_SFN_MASTER=12
ISOL_CPUS_FDD_4_CORE_SFN_ANT=8
ISOL_CPUS_TDD_10_CORE=12
ISOL_CPUS_TDD_8_CORE=0
ISOL_CPUS_FDD_10_CORE_LTE_U=6
ISOL_CPUS_FDD_8_CORE=0
ISOL_CPUS_FDD_8_CORE_MF=0
ISOL_CPUS_FDD_10_CORE_FSYMDSP=6

CELL_PATTERN='LNCEL"'
TECHNOLOGY_CELL_PATTERN='LNCEL_TDD"\|LNCEL_FDD"'
CHANNEL_PATTERN='CHANNEL"'
LCELL_PATTERN='LCELL"'
OPERATING_MODE_PATTERN='<p name="operatingMode">'
NE_TYPE_PATTERN='<p name="neType">'
NUMBER_PATTERN='[0-9]*'
ISOL_CPUS_PATTERN='ic:[0-9]*'
UNIT_TYPE_PATTERN='Unit.[a-zA-Z]*'
PRODUCT_START_PATTERN='<Product code="[a-zA-Z0-9]*">'
PRODUCT_END_PATTERN='</Product>'
MO_END_PATTERN='</managedObject>'
DUPLEX_PATTERN='<Duplex.Type>'
TDD_PATTERN='TDD'
MAX_LCR_PATTERN='<Max.Number.Of.LCR>'
RADIO_PATTERN='<Radio>'
FREQUENCY_PATTERN='<EUTRA.Frequency.Band>'
UNLICENSED_BAND_PATTERN='252|255'
CELL_TECHNOLOGY_PATTERN='<p name="cellTechnology">'
FDD_NB_IOT_CELL_TECHNOLOGY_PATTERN='NB-IoT-FDD'
UL_CH_BW_PATTERN='"ulChBw"'
DL_CH_BW_PATTERN='"dlChBw"'
CH_BW_PATTERN='"chBw"'

NE_TYPE_BTS=21
NE_TYPE_AP=22
NE_TYPE_ANT=24
OPERATING_MODE_SFN='SFN'
OPERATING_MODE_NORMAL='Normal'

function logSeparatorLine()
{
    echo '-------------------------------------------------------------------' >> $LOG_FILE
}

function logSuccess()
{
    echo "$@" >> $LOG_FILE
}

function logError()
{
    echo "Error: $@" >> $LOG_FILE
    errorOccurred=1
}

function parseSCFC()
{
    local scfcFile
    local scfcFileCount
    local lcellsArray
    local eutraFrequencyBandArray
    local compressedScfcFileCount=$(ls $SCFC_INACTIVE_DIR/*.xml.gz | wc -l)

    scfcFileCount=$(grep -l "$CELL_PATTERN" $SCFC_ACTIVE_DIR/bts.xml | wc -l)
    if [ $scfcFileCount -eq 0 ]
    then
        logError "No plan file found $SCFC_ACTIVE_DIR"
    else
        scfcFile=$(grep -l "$CELL_PATTERN" $SCFC_ACTIVE_DIR/bts.xml | head -1)
    fi

    if [ $compressedScfcFileCount -gt 0 ] && [ -z $scfcFile ]
    then
        local scfcTmpDir=$(mktemp -d)
        local latestCompScfcFullPath=$(ls -t $SCFC_INACTIVE_DIR/*.xml.gz | head -1)
        local latestCompScfcBasename=$(basename $latestCompScfcFullPath)
        local tmpScfcFile=$scfcTmpDir/$(echo $latestCompScfcBasename | awk -F".gz" '{print $1}')
        gunzip -c $latestCompScfcFullPath > $tmpScfcFile
        if [ $? -ne 0 ]
        then
            logSuccess "Warning: Failed to gunzip $latestCompScfcFullPath"
        fi
        scfcFileCount=$(grep -l "$CELL_PATTERN" $tmpScfcFile | wc -l)
        if [ $scfcFileCount -eq 1 ]
        then
            logSuccess "Compressed file: $latestCompScfcBasename used as SCFC"
            scfcFile=$tmpScfcFile
        else
            logSuccess "Warning: Compressed file: $latestCompScfcBasename doesn't contain any LNCELs"
        fi
    else
        logSuccess "No compressed SCFC files found, checking uncompressed ones"
    fi

    if [ -z $scfcFile ]
    then
        scfcFileCount=$(grep -l "$CELL_PATTERN" $SCFC_INACTIVE_DIR/*.xml | wc -l)
        if [ $scfcFileCount -eq 0 ]
        then
            logSuccess "No plan file found in $SCFC_INACTIVE_DIR, looking in $SCFC_ACTIVE_DIR"
            return
        fi

        scfcFile=$(ls -t $(grep -l "$CELL_PATTERN" $SCFC_INACTIVE_DIR/*.xml) | head -1)

        if [ $scfcFileCount -gt 1 ]
        then
            logError "Expected 1 plan file in $SCFC_INACTIVE_DIR but $scfcFileCount found." \
                "Taking the latest one: $scfcFile"
        fi
    fi

    local scfcContent=$(xmllint --format --nowarning $scfcFile | awk '{print NR "\t" $0}')
    cellCount=$(echo "$scfcContent" | grep -c "$CELL_PATTERN")
    iotCellCount=$(echo "$scfcContent" | grep "$CELL_TECHNOLOGY_PATTERN" | grep -c "$FDD_NB_IOT_CELL_TECHNOLOGY_PATTERN")
    operatingMode=$(echo "$scfcContent" | grep "$OPERATING_MODE_PATTERN" | cut -d'>' -f2 | cut -d'<' -f1)
    local lcellsContent=$(echo "$scfcContent" | grep "$LCELL_PATTERN" | sed 's/^.*distName="//' | cut -d'"' -f1 | tr '\n' ' ')
    local totalLineCount=$(echo "$scfcContent" | tail -1 | awk '{print $1}')

    read -a lcellsArray <<< "${lcellsContent}"
    read -a eutraFrequencyBandArray <<< $(echo $eutraFrequencyBands | tr '\n' ' ')

    unlicensedCellCount=0
    for lcellDistName in "${lcellsArray[@]}"
    do
        channelBegin=$(echo "$scfcContent" | grep "$CHANNEL_PATTERN" | grep -m 1 "$lcellDistName" | awk '{print $1}')
        channelEnd=$(echo "$scfcContent" | tail -$(($totalLineCount - $channelBegin + 1)) | grep "$MO_END_PATTERN" | head -1 | awk '{print $1}')
        rmodId=$(echo "$scfcContent" | sed -n "$channelBegin,${channelEnd}p" | tr '\n' ' ' | sed 's/^.*RMOD-//' | cut -d'/' -f1)
        rmodPositionInBpf=$(($rmodId - 1))
        echo "${eutraFrequencyBandArray[$rmodPositionInBpf]}" | grep -E "$UNLICENSED_BAND_PATTERN"
        if [ $? -eq 0 ]
        then
            ((unlicensedCellCount++))
        fi
    done

    if [ -z $operatingMode ]
    then
        operatingMode=$OPERATING_MODE_NORMAL
    fi
    if [ $cellCount -eq 0 ]
    then
        logError "No LNCELs found in SCFC, can't set FPGA image."
    else
        local lncelBeginLine=$(echo "$scfcContent" | grep "$TECHNOLOGY_CELL_PATTERN" | head -1 | awk '{print $1}')
        local lncelEndLine=$(echo "$scfcContent" | tail -$(($totalLineCount - $lncelBeginLine + 1)) \
            | grep "$MO_END_PATTERN" | head -1 | awk '{print $1}')
        local lncelContent=$(echo "$scfcContent" | sed -n "$lncelBeginLine,${lncelEndLine}p")
        if [ $isTdd -eq 1 ]
        then
            local chBw=$(echo "$lncelContent" | grep "$CH_BW_PATTERN" | cut -d'>' -f2 | cut -d'<' -f1)
            if [ -z $chBw ]
            then
                logError "chBw not found in SCFC, can't set FPGA image."
            else
                uplinkBandwidth=$chBw
                downlinkBandwidth=$chBw
            fi
        else
            local ulChBw=$(echo "$lncelContent" | grep "$UL_CH_BW_PATTERN" | cut -d'>' -f2 | cut -d'<' -f1)
            local dlChBw=$(echo "$lncelContent" | grep "$DL_CH_BW_PATTERN" | cut -d'>' -f2 | cut -d'<' -f1)
            if [ -z $ulChBw ]
            then
                logError "ulChBw not found in SCFC, can't set FPGA image."
            elif [ -z $dlChBw ]
            then
                logError "dlChBw not found in SCFC, can't set FPGA image."
            else
                uplinkBandwidth=$ulChBw
                downlinkBandwidth=$dlChBw
            fi
        fi
    fi

    rm -rf $scfcTmpDir
}

function parseAutoconf()
{
    local autoconfContent=$(xmllint --format --nowarning $AUTOCONF_FILE)
    neType=$(echo "$autoconfContent" | grep "$NE_TYPE_PATTERN" | grep -o "$NUMBER_PATTERN")
}

function parseBpf()
{
    local bpfTmpDir=$(mktemp -d)
    tar -xzf $BTSOM_PACKAGE_FILE "$BPF_PATH_IN_PACKAGE/$BPF_FILENAME" -C $bpfTmpDir
    if [ $? -ne 0 ]
    then
        logError "Failed to extract $BPF_PATH_IN_PACKAGE/$BPF_FILENAME from BTSOM_*.tgz package."
        return
    fi
    local bpfFile="$bpfTmpDir/$BPF_PATH_IN_PACKAGE/$BPF_FILENAME"

    local boardId=$(cat $UNIT_ID_FILE)
    local bpfContent=$(xmllint --format --nowarning $bpfFile | awk '{print NR "\t" $0}')
    local totalLineCount=$(echo "$bpfContent" | tail -1 | awk '{print $1}')
    local unitTypeLine=$(echo "$bpfContent" | grep "<$UNIT_TYPE_PATTERN>$boardId</$UNIT_TYPE_PATTERN>" \
        | awk '{print $1}')
    local productBeginLine=$(echo "$bpfContent" | head -$(($unitTypeLine - 1)) \
        | grep "$PRODUCT_START_PATTERN" | tail -1 | awk '{print $1}')
    local productEndLine=$(echo "$bpfContent" | tail -$(($totalLineCount - $unitTypeLine + 1)) \
        | grep "$PRODUCT_END_PATTERN" | head -1 | awk '{print $1}')
    local productContent=$(echo "$bpfContent" | sed -n "$productBeginLine,${productEndLine}p")
    isTdd=$(echo "$productContent" | grep "$DUPLEX_PATTERN" | grep "$TDD_PATTERN" > /dev/null \
        && echo 1 || echo 0)
    maxNumberOfLcr=$(echo "$productContent" | grep "$MAX_LCR_PATTERN" | cut -d'>' -f2 | cut -d'<' -f1 \
        | cut -d, -f1)
    radioCount=$(echo "$productContent" | grep -c "$RADIO_PATTERN")
    eutraFrequencyBands=$(echo "$productContent" | grep "$FREQUENCY_PATTERN" | cut -d'>' -f2 | cut -d'<' -f1)

    logSuccess "boardId: $boardId"

    rm -rf $bpfTmpDir
}

function isLowBandwidth()
{
    local bandwidth="$@"
    if [ "$bandwidth" == '1.4 MHz' ] || [ "$bandwidth" == '3 MHz' ]
    then
        echo 1
    else
        echo 0
    fi
}

function determineDeploymentAndFpgaImage()
{
    local cellCount
    local iotCellCount
    local operatingMode
    local uplinkBandwidth
    local downlinkBandwidth
    local neType
    local isTdd
    local maxNumberOfLcr
    local radioCount
    local eutraFrequencyBands
    local isSfnMaster
    local isSfnAntenna
    local isUnlicensedMultiCarrier
    local unlicensedCellCount

    parseBpf
    parseSCFC
    parseAutoconf

    if ([ $neType -eq $NE_TYPE_BTS ] || [ $neType -eq $NE_TYPE_AP ]) \
        && [ $operatingMode == $OPERATING_MODE_SFN ]
    then
        isSfnMaster=1
    else
        isSfnMaster=0
    fi
    if [ $neType -eq $NE_TYPE_ANT ]
    then
        isSfnAntenna=1
    else
        isSfnAntenna=0
    fi
    echo "$eutraFrequencyBands" | grep -E "$UNLICENSED_BAND_PATTERN"
    if [ $? -ne 0 ] || [ $radioCount -lt 2 ]
    then
        isUnlicensedMultiCarrier=0
    else
        isUnlicensedMultiCarrier=1
        isUnlicensedCarrier=1
    fi
    if [ $iotCellCount -ge 1 ]
    then
        ((maxNumberOfLcr+=1))
    fi


    logSuccess "cellCount: $cellCount"
    logSuccess "iotCellCount: $iotCellCount"
    logSuccess "operatingMode: $operatingMode"
    logSuccess "uplinkBandwidth: $uplinkBandwidth"
    logSuccess "downlinkBandwidth: $downlinkBandwidth"
    logSuccess "neType: $neType"
    logSuccess "isTdd: $isTdd"
    logSuccess "maxNumberOfLcr: $maxNumberOfLcr"
    logSuccess "radioCount: $radioCount"
    logSuccess "eutraFrequencyBands: $eutraFrequencyBands"
    logSuccess "isSfnMaster: $isSfnMaster"
    logSuccess "isSfnAntenna: $isSfnAntenna"
    logSuccess "isUnlicensedMultiCarrier: $isUnlicensedMultiCarrier"
    logSuccess "unlicensedCellCount: $unlicensedCellCount"

    if [ $isSfnMaster -eq 1 ]
    then
        if [ $isTdd -eq 1 ]
        then
            deployment=$DEPL_TDD_6_CORE_SFN_MASTER
            imagePrefix=$PREFIX_TDD_6_CORE_SFN_MASTER
            isolCpus=$ISOL_CPUS_TDD_6_CORE_SFN_MASTER
        else
            deployment=$DEPL_FDD_6_CORE_SFN_MASTER
            imagePrefix=$PREFIX_FDD_6_CORE_SFN_MASTER
            isolCpus=$ISOL_CPUS_FDD_6_CORE_SFN_MASTER
        fi
    elif [ $isSfnAntenna -eq 1 ]
    then
        if [ $isTdd -eq 1 ]
        then
            deployment=$DEPL_TDD_8_CORE_SFN_ANT
            imagePrefix=$PREFIX_TDD_8_CORE_SFN_ANT
            isolCpus=$ISOL_CPUS_TDD_8_CORE_SFN_ANT
        else
            deployment=$DEPL_FDD_4_CORE_SFN_ANT
            imagePrefix=$PREFIX_FDD_4_CORE_SFN_ANT
            isolCpus=$ISOL_CPUS_FDD_4_CORE_SFN_ANT
        fi
    else
        if [ $cellCount -gt 1 ] && [ $maxNumberOfLcr -gt 1 ]
        then
            if [ $isTdd -eq 1 ]
            then
                deployment=$DEPL_TDD_10_CORE
                imagePrefix=$PREFIX_TDD_10_CORE
                isolCpus=$ISOL_CPUS_TDD_10_CORE
            else
                if [ $cellCount -eq 3 ] && [ $unlicensedCellCount -eq 2 ]
                then
                    deployment=$DEPL_FDD_10_CORE_LTE_U
                    imagePrefix=$PREFIX_FDD_10_CORE_LTE_U
                    isolCpus=$ISOL_CPUS_FDD_10_CORE_LTE_U
                elif [ $cellCount -ge 2 ]
                then
                    deployment=$DEPL_FDD_10_CORE_FSYMDSP
                    imagePrefix=$PREFIX_FDD_10_CORE_FSYMDSP
                    isolCpus=$ISOL_CPUS_FDD_10_CORE_FSYMDSP
                else
                    deployment=$DEPL_FDD_8_CORE
                    imagePrefix=$PREFIX_FDD_8_CORE
                    isolCpus=$ISOL_CPUS_FDD_8_CORE
                fi
            fi
        else
            if [ $isTdd -eq 1 ]
            then
                deployment=$DEPL_TDD_8_CORE
                imagePrefix=$PREFIX_TDD_8_CORE
                isolCpus=$ISOL_CPUS_TDD_8_CORE
            elif [ $cellCount -eq 1 ] && [ $unlicensedCellCount -eq 1 ]
            then
                deployment=$DEPL_FDD_8_CORE_MF
                imagePrefix=$PREFIX_FDD_8_CORE_MF
                isolCpus=$ISOL_CPUS_FDD_8_CORE_MF
            else
                deployment=$DEPL_FDD_8_CORE
                imagePrefix=$PREFIX_FDD_8_CORE
                isolCpus=$ISOL_CPUS_FDD_8_CORE
            fi
        fi
    fi

    if [ "$uplinkBandwidth" ] && [ "$downlinkBandwidth" ] \
        && [ $(isLowBandwidth "$uplinkBandwidth") -eq 1 ] && [ $(isLowBandwidth "$downlinkBandwidth") -eq 1 ]
    then
        fpgaImage=1
    else
        fpgaImage=0
    fi
}

function setDeploymentAndFpgaImage()
{
    logSuccess "Chosen deployment type is: \"$deployment\"."
    echo $deployment > $DEPLOYMENT_TYPE_FILE \
        && logSuccess "Stored deployment type in file: $DEPLOYMENT_TYPE_FILE" \
        || logError "Failed to store deployment type in file: $DEPLOYMENT_TYPE_FILE"
    cd $DSPRTSW_DIR
    local imageFile=$(ls $imagePrefix'_'*$DSPRTSW_EXT)
    if [ $imageFile ]
    then
        [ -L $DSPRTSW_SYMLINK ] && rm $DSPRTSW_SYMLINK
        [ -L $DSPRTSW_SYMLINK$P7_EXT ] && rm $DSPRTSW_SYMLINK$P7_EXT
        ln -s $imageFile $DSPRTSW_SYMLINK \
            && logSuccess "Created symlink: $DSPRTSW_SYMLINK -> $imageFile" \
            || logError "Failed to create symlink: $DSPRTSW_SYMLINK -> $imageFile"
        ln -s $imageFile$P7_EXT $DSPRTSW_SYMLINK$P7_EXT \
            && logSuccess "Created symlink: $DSPRTSW_SYMLINK$P7_EXT -> $imageFile$P7_EXT" \
            || logError "Failed to create symlink: $DSPRTSW_SYMLINK$P7_EXT -> $imageFile$P7_EXT"
    else
        logError "Image file $imageFile not found."
    fi
    l2u-dtb set_ic $isolCpus 2> /dev/null
    if [ $? -eq 0 ]
    then
        l2u-dtb update 2> /dev/null
        if [ $? -eq 0 ]
        then
            icSet=$(l2u-dtb display 2>&1 | grep -o "$ISOL_CPUS_PATTERN" | grep -o "$NUMBER_PATTERN")
            if [ $icSet -eq $isolCpus ]
            then
                logSuccess "Set isolated cpus to $isolCpus."
            else
                logError "Isolated cpus check failed: currently set value is $icSet but should be $isolCpus."
            fi
        else
            logError "Failed to write isolated cpus to $isolCpus in dtb partition."
        fi
    else
        logError "Failed to set isolated cpus to $isolCpus in ram."
    fi

    logSuccess "Chosen FPGA image is: $fpgaImage."
    echo $fpgaImage > $FPGA_IMAGE_FILE \
        && logSuccess "Stored FPGA image in file: $FPGA_IMAGE_FILE" \
        || logError "Failed to store FPGA image in file: $FPGA_IMAGE_FILE"
}

function parseArguments()
{
    if [ "$1" == "-r" ]
    then
        SCFC_ACTIVE_DIR=$SCFC_STB_ACTIVE_DIR
        AUTOCONF_FILE=$STB_AUTOCONF_FILE
    fi
}

function main()
{
    local deployment
    local imagePrefix
    local isolCpus
    local fpgaImage
    local errorOccurred=0

    logSeparatorLine
    logSuccess "Deployment setup started" $(date)

	parseArguments "$@"
    determineDeploymentAndFpgaImage
    setDeploymentAndFpgaImage

    logSuccess "Deployment setup finished" $(date)
    touch $ACT_PREP_DEPL_SET_FILE
    logSuccess "File indicating that deployment was set created"
    logSeparatorLine

    if [ $errorOccurred -eq 1 ]
    then
        exit 1
    fi
}

main "$@"
