<?php
// *****************************************************************************
// -----------------------------------------------------------------------------
//			Copyright 2017, Nokia Networks
//			All Rights Reserved
// -----------------------------------------------------------------------------
//
// Title:	CbsdInfoHandler.php
//
// Description:	This file handles a request from the client for SAS information.
//		The request is XML encoded and sent to the WebUI Agent for processing.
//		Any response from the agent is JSON encoded and sent as-is to the client.
//		                                                               
//
// ******************************************************************************

include_once "../../phplib/MessageHandler.php";
include_once "../../phplib/XmlMessageParser.php";
include_once "../../phplib/FailureResponse.php";
include_once "../../phplib/messages/CbsdInformationRequest.php";
include_once "../../phplib/messages/CbsdInformationResponse.php";

openlog("WebUI CGI: SAS", LOG_ODELAY, LOG_USER);


header("Content-type: application/json; charset=utf-8");
header("Cache-control: no-store");
header("Pragma: no-cache");


// Read session cookie sent by client. Should always be available unless user
// has disabled cookies (would cause unavailability of WebUI service).
if( !isset( $_COOKIE[ "FZAPWebUISession" ] ) )
{
	// Unexpected. User should not have been able to access WebUI page with
	// cookies disabled. Log error and send failure indication.
	syslog( LOG_ERR, "No session cookie" );

	FailureResponse::SendFailureResponse( "CGI - Client cookie not received" );

	closelog();
	exit();
}

// Session cookie received from client. Retrieve user session ID. Needs to be
// sent to the WebUIAgent for validation (with every request).
$sessionId = $_COOKIE[ "FZAPWebUISession" ];

//create and validate message
$cbsdInfoRequest = new CbsdInformationRequest($sessionId);
if( is_null( $cbsdInfoRequest ) )
{
	syslog( LOG_ERR, "Failed to allocate memory for CbsdInformationRequest message" );

	FailureResponse::SendFailureResponse( "CGI - Memory allocation failure" );

	closelog();
	exit();
}


// Send the message. The MessageHandler will wait for and return the response,
// if any.
$response = MessageHandler::SendAndReceive( $cbsdInfoRequest->XmlMsg() );
if( is_null( $response ) )
{
	syslog( LOG_ERR, "Failed to send CbsdInformationRequest message and/or receive response" );

	FailureResponse::SendFailureResponse( "CGI - Message send or receive failure" );

	closelog();
	exit();
}

//parse and validate
// Parse the received XML message response.
$msg = XmlMessageParser::ParseMsg( $response );

// Validate response.
if( (is_null( $msg )) || ($msg->OK() !== true) || ($msg->GetMessageType() !== "CbsdInformationResponse") )
{
	// Remove line breaks from response string and print to syslog.
	$msgString = str_replace( "\n", null, $response );
	syslog( LOG_ERR, "Invalid or null CbsdInformationResponse message: " . $msgString );

	// Internal error (e.g., protocol error between agent and HTTP server).
	FailureResponse::SendFailureResponse( "CGI - Invalid or null CbsdInformationResponse" );

	closelog();
	exit();
}


//json encode and echo
// No need to check success or failure; will be handled by client. Encode response message
// as JSON and send to client.
$jsonString = json_encode( $msg );

// Check encoding result
if( $jsonString === false )
{
	// JSON encoding failed. Received message must have been invalid.
	syslog( LOG_ERR, "JSON encoding failed" );
	FailureResponse::SendFailureResponse( "CGI - JSON encoding failure" );

	closelog();
	exit();
}


// Output encoded JSON string to client.
echo $jsonString;

closelog();


?>
