<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	ConfigReset.php
//
// Description:	This file handles an asyncronous request from the client for
//		an update to the blocking state. The request is XML encoded and
//		sent to the WebUI Agent for processing. Any response from the
//		agent is JSON encoded and sent as-is to the client (pass-thru).
//
// ****************************************************************************

include_once "../../phplib/MessageHandler.php";
include_once "../../phplib/XmlMessageParser.php";
include_once "../../phplib/FailureResponse.php";
include_once "../../phplib/messages/ConfigResetRequest.php";
include_once "../../phplib/messages/ConfigResetRequestResponse.php";

openlog( "WebUI CGI: ConfigReset", LOG_ODELAY, LOG_USER );

header("Content-type: application/json; charset=utf-8");
header("Cache-control: no-store");
header("Pragma: no-cache");

// Read session cookie sent by client. Should always be available unless user
// has disabled cookies (would cause unavailability of WebUI service).
if( !isset( $_COOKIE[ "FZAPWebUISession" ] ) )
{
	// Unexpected. User should not have been able to access WebUI page with
	// cookies disabled. Log error and send failure indication.
	syslog( LOG_ERR, "No session cookie" );

	FailureResponse::SendFailureResponse( "CGI - Client cookie not received" );

	closelog();
	exit();
}

// Session cookie received from client. Retrieve user session ID. Needs to be
// sent to the WebUIAgent for validation (with every request).
$sessionId = $_COOKIE[ "FZAPWebUISession" ];
$resetSiteId = "false";
$resetSecCred = "false";

if( !(isset( $_POST["ctkn"] )) || ($_POST["ctkn"] === "") )
{

	syslog( LOG_ERR, "Missing CSRF Token" );

	FailureResponse::SendFailureResponse( "CGI - Missing CSRF Token" );

	closelog();
	exit();
}

$csrfToken = $_POST["ctkn"];

// Retrieve query string from URL of page request.
if( (isset( $_POST[ "rsi" ] )) && ($_POST[ "rsi" ] !== "") )
{
	$resetSiteId = $_POST[ "rsi" ];
	if($resetSiteId !== "true")
		{
		$resetSiteId = "false";
	}
}
else
{
	syslog( LOG_ERR, "rsi POST Data not present" );

	FailureResponse::SendFailureResponse( "CGI - rsi POST Data not present" );

	closelog();
	exit();
}

if( (isset( $_POST[ "rsc" ] )) && ($_POST[ "rsc" ] !== "") )
{
	$resetSecCred = $_POST[ "rsc" ];
	//verify params
	if($resetSecCred !== "true")
		{
		$resetSecCred = "false";
	}
}
else
{
	syslog( LOG_ERR, "rsc POST Data not present" );

	FailureResponse::SendFailureResponse( "CGI - rsc POST Data not present" );

	closelog();
	exit();
}

$configResetRequest = new ConfigResetRequest( $sessionId, $csrfToken );
if( is_null( $configResetRequest ) )
{
	syslog( LOG_ERR, "Failed to allocate memory for ConfigResetRequest message" );

	FailureResponse::SendFailureResponse( "CGI - Memory allocation failure" );

	closelog();
	exit();
}

$configResetRequest->SetRemoveSiteId($resetSiteId);
$configResetRequest->SetRemoveSecurityCredentials($resetSecCred);

// Send the message. The MessageHandler will wait for and return the response,
// if any.
$response = MessageHandler::SendAndReceive( $configResetRequest->XmlMsg());
if( is_null( $response ) )
{
	syslog( LOG_ERR, "Failed to send ConfigResetRequest message and/or receive response" );

	FailureResponse::SendFailureResponse( "CGI - Message send or receive failure" );

	closelog();
	exit();
}

// Parse the received XML message response.
$msg = XmlMessageParser::ParseMsg( $response );

// Validate response.
if( (is_null( $msg )) || ($msg->OK() !== true) || ($msg->GetMessageType() !== "ConfigurationResetRequestResponse") )
{
	// Remove line breaks from response string and print to syslog.
	$msgString = str_replace( "\n", null, $response );
	syslog( LOG_ERR, "Invalid or null ConfigReset Response message: " . $msgString );
	if(is_null( $msg ))
	  {
	    FailureResponse::SendFailureResponse( "CGI - Null ConfigReset Response" );
	  }
	else if($msg->OK() !== true)
	  {
	    FailureResponse::SendFailureResponse( "CGI - Invalid ConfigReset Response" );
	  }
	else if($msg->GetMessageType() !== "ConfigurationResetRequestResponse"){
	// Internal error (e.g., protocol error between agent and HTTP server).
	  FailureResponse::SendFailureResponse( "CGI - expected ConfigReset Response but received " . $msg->GetMessageType() );
	}
	closelog();
	exit();
}


// No need to check success or failure; will be handled by client. Encode response message
// as JSON and send to client.
$jsonString = json_encode( $msg );

// Check encoding result
if( $jsonString === false )
{
	// JSON encoding failed. Received message must have been invalid.
	syslog( LOG_ERR, "JSON encoding failed" );
	FailureResponse::SendFailureResponse( "CGI - JSON encoding failure" );

	closelog();
	exit();
}


// Output encoded JSON string to client.
echo $jsonString;

closelog();

?>

