<?php
// *****************************************************************************
// -----------------------------------------------------------------------------
//			Copyright 2016, Nokia Networks
//			All Rights Reserved
// -----------------------------------------------------------------------------
//
// Title:	LinkAndSessionManagementHandler.php
//
// Description:	This file handles an asyncronous link and session management
//		ping from the client. Pings are sent at regular intervals as
//		a mechanism for link and session supervision. Each ping may
//		contain a user activity indicator, which is used to extend the
//		user session. The response from the server includes the current
//		operational state and blocking state, which are used to provide
//		visual clues in the client display.
//
//		On receipt of a periodic ping, a link management request message
//		is XML encoded and sent to the WebUI Agent for processing. Any
//		response from the agent is JSON encoded and sent as-is to the
//		client (pass-thru).
//
// *****************************************************************************

include_once "../../phplib/MessageHandler.php";
include_once "../../phplib/XmlMessageParser.php";
include_once "../../phplib/FailureResponse.php";
include_once "../../phplib/messages/LinkManagementRequest.php";
include_once "../../phplib/messages/LinkManagementResponse.php";

openlog( "WebUI CGI: Link Management", LOG_ODELAY, LOG_USER );

header("Content-type: application/json; charset=utf-8");
header("Cache-control: no-store");
header("Pragma: no-cache");

const MSG_TIMEOUT = 2;

// Read session cookie sent by client. Should always be available unless user
// has disabled cookies (would cause unavailability of WebUI service).
if( !isset( $_COOKIE[ "FZAPWebUISession" ] ) )
{
	// Unexpected. User should not have been able to access WebUI page with
	// cookies disabled. Log error and send failure indication.
	syslog( LOG_ERR, "No session cookie" );

	FailureResponse::SendFailureResponse( "CGI - Client cookie not received" );

	closelog();
	exit();
}

// Session cookie received from client. Retrieve user session ID. Needs to be
// sent to the WebUIAgent for validation (with every request).
$sessionId = $_COOKIE[ "FZAPWebUISession" ];


$linkManagementRequest = new LinkManagementRequest( $sessionId );
if( is_null( $linkManagementRequest ) )
{
	syslog( LOG_ERR, "Failed to allocate memory for LinkManagementRequest message" );

	FailureResponse::SendFailureResponse( "CGI - Memory allocation failure" );

	closelog();
	exit();
}


// Retrieve query string from URL of page request.
if( isset( $_GET["userActivity"] ) )
{
	$linkManagementRequest->SetUserActivityIndicator();
}


// Send the message. The MessageHandler will wait for and return the response,
// if any.
$response = MessageHandler::SendAndReceive( $linkManagementRequest->XmlMsg(), MSG_TIMEOUT );
if( is_null( $response ) )
{
	syslog( LOG_ERR, "Failed to send LinkManagementRequest message and/or receive response" );

	FailureResponse::SendFailureResponse( "CGI - Message send or receive failure" );

	closelog();
	exit();
}


// Parse the received XML message response.
$msg = XmlMessageParser::ParseMsg( $response );

// Validate response.
if( (is_null( $msg )) || ($msg->OK() !== true) || ($msg->GetMessageType() !== "LinkManagementResponse") )
{
	// Remove line breaks from response string and print to syslog.
	$msgString = str_replace( "\n", " ", $response );
	syslog( LOG_ERR, "Invalid or null LinkManagementResponse message: " . $msgString );

	// Internal error (e.g., protocol error between agent and HTTP server).
	FailureResponse::SendFailureResponse( "CGI - Invalid or null LinkManagementResponse" );

	closelog();
	exit();
}


// No need to check success or failure; will be handled by client. Encode response message
// as JSON and send to client.
$jsonString = json_encode( $msg );

// Check encoding result
if( $jsonString === false )
{
	// JSON encoding failed. Received message must have been invalid.
	syslog( LOG_ERR, "JSON encoding failed" );
	FailureResponse::SendFailureResponse( "CGI - JSON encoding failure" );

	closelog();
	exit();
}


// Output encoded JSON string to client.
echo $jsonString;

closelog();

?>

